{******************************************************************************}
{                                                                              }
{ ModLink                                                                      }
{ Copyright (C) 2002 Ing. Ivo Bauer. All Rights Reserved.                      }
{ web: http://www.ozm.cz/ivobauer/modlink/                                     }
{ e-mail: ivo.bauer@tiscali.cz                                                 }
{                                                                              }
{******************************************************************************}

unit ModLinkDemoMainF;

{$I CompilerDefs.inc}

{$B-,H+,J-,T-,X+,Z1}

//------------------------------------------------------------------------------

interface

//------------------------------------------------------------------------------

uses
  { Windows  } Windows, Messages,
  { Delphi   } SysUtils, {$IFDEF COMPILER6_UP} Variants, {$ENDIF} Classes,
               Graphics, Controls, Forms, Dialogs, Menus, StdCtrls, ExtCtrls,
               ComCtrls,
  { CPortLib } CPort,
  { ModLink  } ModLink;

//------------------------------------------------------------------------------

type
  TModLinkDemoMainForm = class(TForm)
    MainMenu: TMainMenu;
    FileMenu: TMenuItem;
    FileExitItem: TMenuItem;
    ToolsMenu: TMenuItem;
    ToolsConnectionOptionsItem: TMenuItem;
    ToolsSlaveOptionsItem: TMenuItem;
    N1: TMenuItem;
    ToolsClearLog: TMenuItem;
    Connection: TModbusConnection;
    Device: TModbusDevice;
    StatusBar: TStatusBar;
    Panel1: TPanel;
    Panel2: TPanel;
    SelectedCmdDescLabel: TLabel;
    Label1: TLabel;
    Label8: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    Label6: TLabel;
    Label7: TLabel;
    Label9: TLabel;
    CommandComboBox: TComboBox;
    StartRegEdit: TEdit;
    RegCountEdit: TEdit;
    RegValuesEdit: TEdit;
    AndMaskEdit: TEdit;
    OrMaskEdit: TEdit;
    StorageFormatRadioGroup: TRadioGroup;
    InitTransactionButton: TButton;
    Panel3: TPanel;
    Label5: TLabel;
    LogMemo: TMemo;
    Label20: TLabel;
    Label10: TLabel;
    Label11: TLabel;
    DecimalsRadioGroup: TRadioGroup;
    procedure FileExitItemClick(Sender: TObject);
    procedure ToolsConnectionOptionsItemClick(Sender: TObject);
    procedure ToolsSlaveOptionsItemClick(Sender: TObject);
    procedure ToolsClearLogClick(Sender: TObject);
    procedure CommandComboBoxClick(Sender: TObject);
    procedure InitTransactionButtonClick(Sender: TObject);
    procedure DeviceLoopbackTestExecute(Sender: TModbusDevice;
      const Analysis: TTransactionAnalysis);
    procedure DeviceHoldingRegistersRead(Sender: TModbusDevice;
      const Analysis: TTransactionAnalysis; StartReg, RegCount: Word;
      const RegValues: TRegValues);
    procedure DeviceInputRegistersRead(Sender: TModbusDevice;
      const Analysis: TTransactionAnalysis; StartReg, RegCount: Word;
      const RegValues: TRegValues);
    procedure DeviceSingleRegisterWrite(Sender: TModbusDevice;
      const Analysis: TTransactionAnalysis; RegAddr, RegValue: Word);
    procedure DeviceMultipleRegistersWrite(Sender: TModbusDevice;
      const Analysis: TTransactionAnalysis; StartReg, RegCount: Word;
      const RegValues: TRegValues);
    procedure DeviceSingleRegisterMaskWrite(Sender: TModbusDevice;
      const Analysis: TTransactionAnalysis; RegAddr, AndMask,
      OrMask: Word);
    procedure StorageFormatRadioGroupClick(Sender: TObject);
  private
    procedure LogExecuted(const TransactionName: string);
    procedure LogMaskWrite(RegAddr: Word; AndMask: Word; OrMask: Word);
    procedure LogProcessedRegs(RegCount: Word);
    procedure LogSingleValue(RegAddr: Word; RegValue: Word; NewContents: Boolean);
    procedure LogStatus(const Analysis: TTransactionAnalysis);
    procedure UpdateConnectionStatus;
    procedure UpdateUI;
  public
    constructor Create(AOwner: TComponent); override;
  end;

//------------------------------------------------------------------------------

var
  ModLinkDemoMainForm: TModLinkDemoMainForm;

//------------------------------------------------------------------------------

implementation

//------------------------------------------------------------------------------

uses ConnectionOptionsF;

{$R *.dfm}

//------------------------------------------------------------------------------

type
  TModbusCommand = (
    mcReadHoldingRegisters,
    mcReadInputRegisters,
    mcWriteSingleRegister,
    mcExecuteLoopbackTest,
    mcWriteMultipleRegisters,
    mcMaskWriteSingleRegister
  );

  TStorageFormat = (sfNative, sfSmallInt, sfFloatingPoint);

//------------------------------------------------------------------------------

const
  ModbusCommands: array [TModbusCommand] of string = (
    'Read Holding Registers (code $03)',
    'Read Input Registers (code $04)',
    'Write Single Register (code $06)',
    'Diagnostics | Return Query Data (code $08)',
    'Write Multiple Registers (code $10)',
    'Mask Write Register (code $16)'
  );

  CommandDescriptions: array [TModbusCommand] of string = (
    'This command is used to read the contents of a contiguous block of ' +
    'holding registers in an underlying slave device. "Starting register" ' +
    'field specifies the register address at which the reading is to be ' +
    'started. "Register count" field specifies the quantity of registers ' +
    'to be read.',

    'This command is used to read the contents of a contiguous block of ' +
    'input registers in an underlying slave device. "Starting register" ' +
    'field specifies the register address at which the reading is to be ' +
    'started. "Register count" field specifies the quantity of registers ' +
    'to be read.',

    'This command is used to write a single holding register in an ' +
    'underlying slave device. "Starting register" field specifies the ' +
    'address of the register to be written. "Value(s) to write" field ' +
    'specifies the requested value to be written.',

    'This command is used to instruct an underlying slave device to perform ' +
    'so-called loopback test; i.e. the data transmitted in the query is ' +
    'the same to be returned (looped back) in the reply. It can be used to get to ' +
    'know whether an underlying slave device is properly connected to the ' +
    'host computer and functioning as expected.',

    'This command is used to write a contiguous block of holding registers ' +
    'in an underlying slave device. "Starting register" field specifies ' +
    'the register address at which the writing is to be started. "Register ' +
    'count" field specifies the quantity of registers to be written. ' +
    '"Value(s) to write" field specifies the requested values to be written.',

    'This command is used to modify the contents of a specified holding ' +
    'register in an underlying slave device using a combination of ' +
    'an AND mask, an OR mask, and the register''s current contents. It can ' +
    'be used to set or clear individual bits in the register. "Starting ' +
    'register" field specifies the address of the register to be modified.'
  );

  FailReasons: array [TFailReason] of string = (
    'No reply has been received (timeout error)',
    'Corrupted (partial) reply has been received',
    'Corrupted (bad CRC) reply has been received',
    'Unexpected (unwanted) reply has been received',
    'Slave device has generated an exception reply',
    'Transaction has succeeded'
  );

  SlaveExceptions: array [TSlaveException] of string = (
    'unknown exception code',
    'illegal command',
    'illegal data address',
    'illegal data value',
    'slave device failure',
    'acknowledge',
    'slave device busy',
    'negative acknowledge',
    'memory parity error'
  );

//------------------------------------------------------------------------------

procedure ParseRegisterValues(const Source: string;
  StorageFormat: TStorageFormat; Decimals: Integer; var Dest: TRegValues);
// Intended to parse semicolon separated values, extracting them into Dest
// array. Values being entered can be of type Word, SmallInt or Extended
// depending on StorageFormat argument.
var
  CurPos, TempPos, EndPos: PChar;
  S: string;
  W: Word;
begin
  // Check for the empty Source.
  if Source = '' then raise Exception.Create('The source string is empty');
  // Source's current position.
  CurPos := PChar(Source);
  // Source's end position (pointer to last character).
  EndPos := @Source[Length(Source)];
  //
  Dest := nil;
  // Loop...
  while CurPos <= EndPos do
  begin
    // Searching for the separator (semicolon).
    TempPos := StrScan(CurPos, ';');
    // Extract the string between the current position and a semicolon,
    // or between the current position and end of the Source.
    if TempPos = nil then
      SetString(S, CurPos, EndPos - CurPos + 1)
    else
      SetString(S, CurPos, TempPos - CurPos);
    // Empty string is considered to be zero.
    if S = '' then S := '0';
    // Treat the value as requested.
    case StorageFormat of
      sfNative:
        W := StrToInt(S);
      sfSmallInt:
        W := Int2Mod(StrToInt(S));
      sfFloatingPoint:
        W := Float2Mod(StrToFloat(S), Decimals);
    else
      raise Exception.Create('Internal ParseRegisterValues error');
    end;
    // Reallocate the array and append a new value to it.
    SetLength(Dest, Length(Dest) + 1);
    Dest[High(Dest)] := W;
    // Advance the current position.
    if TempPos = nil then
      CurPos := EndPos + 1
    else
      CurPos := TempPos + 1;
  end;
end;

//------------------------------------------------------------------------------
// TModLinkDemoMainForm class
//------------------------------------------------------------------------------

constructor TModLinkDemoMainForm.Create(AOwner: TComponent);
var
  I: TModbusCommand;
begin
  inherited;
  Caption := Format('%s - Copyright  2002 Ing. Ivo Bauer. All Rights Reserved.',
    [Application.Title]);
  with LogMemo.Lines do
  begin
    Clear;
    Add(Format('Welcome to %s application!', [Application.Title]));
    Add('');
    Add(Format('Running on ModLink version %s', [ModLinkVersion]));
    Add('');
    Add('Use ''Tools | Modbus Connection Options...'' on the main menu bar ' +
      'to set up the internal Modbus connection.');
    Add('');
    Add('Use ''Tools | Modbus Device Options...'' on the main menu bar ' +
      'to set up your Modbus RTU aware slave device.');
    Add('');
  end;
  with CommandComboBox do
  begin
    Items.Clear;
    for I := Low(I) to High(I) do
      Items.Add(Format(' %s', [ModbusCommands[I]]));
    ItemIndex := 0;
  end;
  UpdateUI;
  Application.HintHidePause := 20000;
  try
    Connection.Connected := True;
  except
    {$IFDEF COMPILER6_UP}
    if Assigned(ApplicationHandleException) then
      ApplicationHandleException(Self);
    {$ELSE}
    Application.HandleException(Self);
    {$ENDIF}
  end;
  UpdateConnectionStatus;
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.LogExecuted(const TransactionName: string);
begin
  LogMemo.Lines.Add(Format('DONE: %s', [TransactionName]));
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.LogMaskWrite(RegAddr: Word; AndMask: Word;
  OrMask: Word);
begin
  LogMemo.Lines.Add(Format('Register address: %d | AND mask: %d | OR mask: %d',
    [RegAddr, AndMask, OrMask]));
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.LogProcessedRegs(RegCount: Word);
begin
  if RegCount > 1 then
    LogMemo.Lines.Add(Format('%d registers were processed.', [RegCount]))
  else
    LogMemo.Lines.Add(Format('%d register was processed.', [RegCount]));
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.LogSingleValue(RegAddr: Word; RegValue: Word;
  NewContents: Boolean);
begin
  case TStorageFormat(StorageFormatRadioGroup.ItemIndex) of
    sfNative:
      if NewContents then
        LogMemo.Lines.Add(Format('Register address: %d | New contents: %d',
          [RegAddr, RegValue]))
      else
        LogMemo.Lines.Add(Format('Register address: %d | Old contents: %d',
          [RegAddr, RegValue]));
    sfSmallInt:
      if NewContents then
        LogMemo.Lines.Add(Format('Register address: %d | New contents: %d',
          [RegAddr, Mod2Int(RegValue)]))
      else
        LogMemo.Lines.Add(Format('Register address: %d | Old contents: %d',
          [RegAddr, Mod2Int(RegValue)]));
    sfFloatingPoint:
      if NewContents then
        LogMemo.Lines.Add(Format('Register address: %d | New contents: %.*f',
          [RegAddr, DecimalsRadioGroup.ItemIndex,
          Mod2Float(RegValue, DecimalsRadioGroup.ItemIndex)]))
      else
        LogMemo.Lines.Add(Format('Register address: %d | Old contents: %.*f',
          [RegAddr, DecimalsRadioGroup.ItemIndex,
          Mod2Float(RegValue, DecimalsRadioGroup.ItemIndex)]));
  end;
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.LogStatus(const Analysis: TTransactionAnalysis);
begin
  if Analysis.FailReason = frSlaveException then
    LogMemo.Lines.Add(Format('%s (%s).', [FailReasons[Analysis.FailReason],
      SlaveExceptions[Analysis.SlaveException]]))
  else
    LogMemo.Lines.Add(Format('%s.', [FailReasons[Analysis.FailReason]]));
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.UpdateConnectionStatus;
begin
  if Connection.Connected then
    StatusBar.Panels[0].Text := Format('Connected to %s', [Connection.Port])
  else
    StatusBar.Panels[0].Text := 'Disconnected';
  StatusBar.Panels[1].Text := Format('Slave device address: %d', [Device.SlaveAddress]);
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.UpdateUI;

  procedure SetEditState(AEdit: TEdit; AEnabled: Boolean);
  begin
    with AEdit do
    begin
      Enabled := AEnabled;
      if AEnabled then
        Color := clWindow
      else
        Color  := clBtnFace;
    end;
  end;

var
  CurrentCmd: TModbusCommand;
begin
  if CommandComboBox.ItemIndex > -1 then
  begin
    CurrentCmd := TModbusCommand(CommandComboBox.ItemIndex);
    SelectedCmdDescLabel.Caption := CommandDescriptions[CurrentCmd];
    // Starting register.
    Label2.Enabled := CurrentCmd <> mcExecuteLoopbackTest;
    SetEditState(StartRegEdit, CurrentCmd <> mcExecuteLoopbackTest);
    // Register count.
    Label3.Enabled := CurrentCmd in [mcReadHoldingRegisters,
      mcReadInputRegisters, mcWriteMultipleRegisters];
    SetEditState(RegCountEdit, CurrentCmd in [mcReadHoldingRegisters,
      mcReadInputRegisters, mcWriteMultipleRegisters]);
    // Register values.
    Label4.Enabled := CurrentCmd in [mcWriteSingleRegister,
      mcWriteMultipleRegisters];
    SetEditState(RegValuesEdit, CurrentCmd in [mcWriteSingleRegister,
      mcWriteMultipleRegisters]);
    // AND mask data.
    Label6.Enabled := CurrentCmd = mcMaskWriteSingleRegister;
    SetEditState(AndMaskEdit, CurrentCmd = mcMaskWriteSingleRegister);
    // OR mask data.
    Label7.Enabled := CurrentCmd = mcMaskWriteSingleRegister;
    SetEditState(OrMaskEdit, CurrentCmd = mcMaskWriteSingleRegister);
    // Storage format.
    StorageFormatRadioGroup.Enabled := not (CurrentCmd in [mcExecuteLoopbackTest,
      mcMaskWriteSingleRegister]);
    // Decimals.
    DecimalsRadioGroup.Enabled := StorageFormatRadioGroup.Enabled and
      (TStorageFormat(StorageFormatRadioGroup.ItemIndex) = sfFloatingPoint);
  end;
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.FileExitItemClick(Sender: TObject);
begin
  Close;
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ToolsConnectionOptionsItemClick(
  Sender: TObject);
begin
  try
    EditConnectionOptions(Connection);
  finally
    UpdateConnectionStatus;
  end;
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ToolsSlaveOptionsItemClick(Sender: TObject);
const
  SCaptionFmt = 'Modbus Device Options';
  SPrompt = 'Enter the address of an underlying slave device ' +
    '(acceptable values are 1 through 247):';
var
  S: string;
begin
  S := IntToStr(Device.SlaveAddress);
  if InputQuery(SCaptionFmt, SPrompt, S) then
  begin
    try
      Device.SlaveAddress := StrToInt(S);
      UpdateConnectionStatus;
    except
      on E: EConvertError do
      begin
        E.Message := Format('''%s'' is not a valid slave address.', [S]);
        raise;
      end
      else raise;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ToolsClearLogClick(Sender: TObject);
begin
  LogMemo.Clear;
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.CommandComboBoxClick(Sender: TObject);
begin
  UpdateUI;
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.InitTransactionButtonClick(Sender: TObject);

  procedure ValidateWord(AEdit: TEdit);
  var
    I: Integer;
    IsInvalid: Boolean;
  begin
    I := 0;
    IsInvalid := False;
    try
      I := StrToInt(AEdit.Text);
    except
      on E: Exception do
        if E is EConvertError then
          IsInvalid := True
        else
          raise;
    end;
    if not IsInvalid and ((I < 0) or (I > High(Word))) then
    begin
      IsInvalid := True;
    end;
    if IsInvalid then
    begin
      MessageDlg('You''ve entered an invalid value.', mtError, [mbOK], 0);
      if AEdit.CanFocus then AEdit.SetFocus;
      Abort;
    end;
  end;

var
  RegValues: TRegValues;
  CurrentCmd: TModbusCommand;
begin
  if CommandComboBox.ItemIndex > -1 then
  begin
    // Perform a simple input data validation.
    if StartRegEdit.Enabled then ValidateWord(StartRegEdit);
    if RegCountEdit.Enabled then ValidateWord(RegCountEdit);
    if AndMaskEdit.Enabled then ValidateWord(AndMaskEdit);
    if OrMaskEdit.Enabled then ValidateWord(OrMaskEdit);
    // Parse the value(s) to be written.
    if RegValuesEdit.Enabled then
      try
        ParseRegisterValues(RegValuesEdit.Text,
          TStorageFormat(StorageFormatRadioGroup.ItemIndex),
          DecimalsRadioGroup.ItemIndex, RegValues);
        if RegCountEdit.Enabled and
          (StrToInt(RegCountEdit.Text) <> Length(RegValues)) then
        begin
          raise Exception.Create('Total number of value(s) to write must be equal to a register count');
        end;
      except
        on E: Exception do
        begin
          MessageDlg(Format('An error occured while parsing value(s) to be ' +
            'written.'#13#10'Reason: %s.', [E.Message]), mtError, [mbOK], 0);
          if RegValuesEdit.CanFocus then RegValuesEdit.SetFocus;
          Abort;
        end;
      end;
    // Determine the currently selected command.
    CurrentCmd := TModbusCommand(CommandComboBox.ItemIndex);
    // Initiate an appropriate Modbus transaction.
    LogMemo.Lines.Add(Format('INIT: %s', [ModbusCommands[CurrentCmd]]));
    LogMemo.Lines.Add('');
    case CurrentCmd of
      mcReadHoldingRegisters:
        Device.ReadHoldingRegisters(StrToInt(StartRegEdit.Text),
          StrToInt(RegCountEdit.Text));
      mcReadInputRegisters:
        Device.ReadInputRegisters(StrToInt(StartRegEdit.Text),
          StrToInt(RegCountEdit.Text));
      mcWriteSingleRegister:
        Device.WriteSingleRegister(StrToInt(StartRegEdit.Text),
          RegValues[0]);
      mcExecuteLoopbackTest:
        Device.ExecuteLoopbackTest;
      mcWriteMultipleRegisters:
        Device.WriteMultipleRegisters(StrToInt(StartRegEdit.Text),
          StrToInt(RegCountEdit.Text), RegValues);
      mcMaskWriteSingleRegister:
        Device.MaskWriteSingleRegister(StrToInt(StartRegEdit.Text),
          StrToInt(AndMaskEdit.Text), StrToInt(OrMaskEdit.Text));
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.DeviceLoopbackTestExecute(
  Sender: TModbusDevice; const Analysis: TTransactionAnalysis);
begin
  LogExecuted(ModbusCommands[mcExecuteLoopbackTest]);
  LogStatus(Analysis);
  if Analysis.FailReason = frSuccess then
    LogMemo.Lines.Add('Underlying slave device appears to be responding.');
  LogMemo.Lines.Add('');
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.DeviceHoldingRegistersRead(
  Sender: TModbusDevice; const Analysis: TTransactionAnalysis; StartReg,
  RegCount: Word; const RegValues: TRegValues);
var
  I: Integer;
begin
  LogExecuted(ModbusCommands[mcReadHoldingRegisters]);
  LogStatus(Analysis);
  if Analysis.FailReason = frSuccess then
  begin
    LogProcessedRegs(RegCount);
    for I := 0 to RegCount - 1 do
      LogSingleValue(StartReg + I, RegValues[I], False);
  end;
  LogMemo.Lines.Add('');
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.DeviceInputRegistersRead(
  Sender: TModbusDevice; const Analysis: TTransactionAnalysis; StartReg,
  RegCount: Word; const RegValues: TRegValues);
var
  I: Integer;
begin
  LogExecuted(ModbusCommands[mcReadInputRegisters]);
  LogStatus(Analysis);
  if Analysis.FailReason = frSuccess then
  begin
    LogProcessedRegs(RegCount);
    for I := 0 to RegCount - 1 do
      LogSingleValue(StartReg + I, RegValues[I], False);
  end;
  LogMemo.Lines.Add('');
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.DeviceSingleRegisterWrite(
  Sender: TModbusDevice; const Analysis: TTransactionAnalysis; RegAddr,
  RegValue: Word);
begin
  LogExecuted(ModbusCommands[mcWriteSingleRegister]);
  LogStatus(Analysis);
  if Analysis.FailReason = frSuccess then
  begin
    LogProcessedRegs(1);
    LogSingleValue(RegAddr, RegValue, True);
  end;
  LogMemo.Lines.Add('');
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.DeviceMultipleRegistersWrite(
  Sender: TModbusDevice; const Analysis: TTransactionAnalysis; StartReg,
  RegCount: Word; const RegValues: TRegValues);
var
  I: Integer;
begin
  LogExecuted(ModbusCommands[mcWriteMultipleRegisters]);
  LogStatus(Analysis);
  if Analysis.FailReason = frSuccess then
  begin
    LogProcessedRegs(RegCount);
    for I := 0 to RegCount - 1 do
      LogSingleValue(StartReg + I, RegValues[I], True);
  end;
  LogMemo.Lines.Add('');
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.DeviceSingleRegisterMaskWrite(
  Sender: TModbusDevice; const Analysis: TTransactionAnalysis; RegAddr,
  AndMask, OrMask: Word);
begin
  LogExecuted(ModbusCommands[mcMaskWriteSingleRegister]);
  LogStatus(Analysis);
  if Analysis.FailReason = frSuccess then
  begin
    LogProcessedRegs(1);
    LogMaskWrite(RegAddr, AndMask, OrMask);
  end;
  LogMemo.Lines.Add('');
end;

//------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.StorageFormatRadioGroupClick(
  Sender: TObject);
begin
  UpdateUI;
end;

//------------------------------------------------------------------------------

end.
