{$INCLUDE ..\cDefines.inc}
unit cWriters;

{                                                                              }
{                              Writers v3.01                                   }
{                                                                              }
{         This unit is copyright  2002 by David Butler (david@e.co.za)        }
{                                                                              }
{                  This unit is part of Delphi Fundamentals.                   }
{                   Its original file name is cWriters.pas                     }
{       The latest version is available from the Fundamentals home page        }
{                     http://fundementals.sourceforge.net/                     }
{                                                                              }
{                I invite you to use this unit, free of charge.                }
{        I invite you to distibute this unit, but it must be for free.         }
{             I also invite you to contribute to its development,              }
{             but do not distribute a modified copy of this file.              }
{                                                                              }
{          A forum is available on SourceForge for general discussion          }
{             http://sourceforge.net/forum/forum.php?forum_id=2117             }
{                                                                              }
{                                                                              }
{ Revision history:                                                            }
{   12/05/2002  3.01  Created cWriters unit from cStreams.                     }
{                     AWriter, TFileWriter.                                    }
{                                                                              }

interface

uses
  { Delphi }
  SysUtils;



{                                                                              }
{ AWriter                                                                      }
{   Writer abstract base class.                                                }
{                                                                              }
type
  AWriter = class
  protected
    function  GetPosition : Int64; virtual; abstract;
    procedure SetPosition(const Position : Int64); virtual; abstract;
    function  GetSize : Int64; virtual; abstract;
    procedure SetSize(const Size : Int64); virtual; abstract;

  public
    function  Write(const Buffer; const Size : Integer) : Integer; virtual; abstract;

    property  Position : Int64 read GetPosition write SetPosition;
    property  Size : Int64 read GetSize write SetSize;
  end;
  EWriter = class(Exception);



{                                                                              }
{ AWriterEx                                                                    }
{   Base class for Writer implementations. AWriteEx extends AWriter with       }
{   commonly used functions.                                                   }
{                                                                              }
{   All methods in AWriterEx is implemented using calls to the abstract        }
{   methods in AWriter. Writer implementations can override the virtual        }
{   methods in AWriterEx with more efficient versions.                         }
{                                                                              }
type
  TWriterNewLineType = (nlCR, nlLF, nlCRLF, nlLFCR);
  AWriterEx = class (AWriter)
  public
    procedure RaiseWriteError;

    procedure Append;
    procedure Truncate; virtual;

    procedure WriteBuffer(const Buffer; const Size : Integer);
    procedure WriteStr(const Buffer : String); virtual;
    procedure SetAsString(const S : String);

    procedure WriteByte(const V : Byte);
    procedure WriteWord(const V : Word);
    procedure WriteLongWord(const V : LongWord);
    procedure WriteLongInt(const V : LongInt);
    procedure WriteInt64(const V : Int64);

    procedure WriteBufLine(const Buffer; const Size : Integer;
              const NewLineType : TWriterNewLineType = nlCRLF);
    procedure WriteLine(const S : String; const NewLineType : TWriterNewLineType = nlCRLF);
  end;



{                                                                              }
{ TFileWriter                                                                  }
{   Writer implementation for a file.                                          }
{                                                                              }
type
  TFileWriterOpenMode = (fwomOpen,              // Open existing
                         fwomTruncate,          // Open existing and truncate
                         fwomCreate,            // Always create
                         fwomCreateIfNotExist); // Create if not exist else open existing
  TFileWriter = class(AWriterEx)
  protected
    FFileName    : String;
    FHandle      : Integer;
    FHandleOwner : Boolean;
    FFileCreated : Boolean;

    function  GetPosition : Int64; override;
    procedure SetPosition(const Position : Int64); override;
    function  GetSize : Int64; override;
    procedure SetSize(const Size : Int64); override;

  public
    Constructor Create(const FileName : String;
                const OpenMode : TFileWriterOpenMode = fwomCreateIfNotExist); overload;
    Constructor Create(const FileHandle : Integer; const HandleOwner : Boolean); overload;
    Destructor Destroy; override;

    property  Handle : Integer read FHandle;
    property  HandleOwner : Boolean read FHandleOwner;
    property  FileCreated : Boolean read FFileCreated;

    function  Write(const Buffer; const Size : Integer) : Integer; override;

    procedure DeleteFile;
  end;
  EFileWriter = class(EWriter);



{                                                                              }
{ TOutputWriter                                                                }
{   Writer implementation for standard system output.                          }
{                                                                              }
type
  TOutputWriter = class(AWriterEx)
  public
    function  Write(const Buffer; const Size : Integer) : Integer; override;
  end;



implementation

uses
  { Delphi }
  Windows;



{                                                                              }
{ AWriterEx                                                                    }
{                                                                              }
procedure AWriterEx.RaiseWriteError;
  begin
    raise EWriter.Create('Write error');
  end;

procedure AWriterEx.Append;
  begin
    Position := Size;
  end;

procedure AWriterEx.Truncate;
  begin
    Size := Position;
  end;

procedure AWriterEx.WriteBuffer(const Buffer; const Size : Integer);
  begin
    if Size <= 0 then
      exit;
    if Write(Buffer, Size) <> Size then
      RaiseWriteError;
  end;

procedure AWriterEx.WriteStr(const Buffer : String);
  begin
    WriteBuffer(Pointer(Buffer)^, Length(Buffer));
  end;

procedure AWriterEx.SetAsString(const S : String);
  begin
    Position := 0;
    WriteStr(S);
    Truncate;
  end;

procedure AWriterEx.WriteByte(const V : Byte);
  begin
    WriteBuffer(V, Sizeof(Byte));
  end;

procedure AWriterEx.WriteWord(const V : Word);
  begin
    WriteBuffer(V, Sizeof(Word));
  end;

procedure AWriterEx.WriteLongWord(const V : LongWord);
  begin
    WriteBuffer(V, Sizeof(LongWord));
  end;

procedure AWriterEx.WriteLongInt(const V : LongInt);
  begin
    WriteBuffer(V, Sizeof(LongInt));
  end;

procedure AWriterEx.WriteInt64(const V : Int64);
  begin
    WriteBuffer(V, Sizeof(Int64));
  end;

procedure AWriterEx.WriteBufLine(const Buffer; const Size : Integer; const NewLineType : TWriterNewLineType);
  begin
    WriteBuffer(Buffer, Size);
    Case NewLineType of
      nlCR   : WriteByte(13);
      nlLF   : WriteByte(10);
      nlCRLF : WriteStr(#13#10);
      nlLFCR : WriteStr(#10#13);
    end;
  end;

procedure AWriterEx.WriteLine(const S : String; const NewLineType : TWriterNewLineType);
  begin
    WriteBufLine(Pointer(S)^, Length(S), NewLineType);
  end;



{                                                                              }
{ TFileWriter                                                                  }
{                                                                              }
Constructor TFileWriter.Create(const FileName : String; const OpenMode : TFileWriterOpenMode);
var CreateFile : Boolean;
  begin
    inherited Create;
    FFileName := FileName;
    Case OpenMode of
      fwomCreate           : CreateFile := True;
      fwomCreateIfNotExist : CreateFile := not FileExists(FileName);
    else
      CreateFile := False;
    end;
    if CreateFile then
      FHandle := FileCreate(FileName) else
      FHandle := FileOpen(FileName, fmOpenReadWrite);
    if FHandle = -1 then {$IFDEF DELPHI6_UP}
      RaiseLastOSError; {$ELSE}
      RaiseLastWin32Error; {$ENDIF}
    FHandleOwner := True;
    FFileCreated := CreateFile;
    if OpenMode = fwomTruncate then
      if not SetEndOfFile(FHandle) then
        raise EFileWriter.Create('File truncate error');
  end;

Constructor TFileWriter.Create(const FileHandle : Integer; const HandleOwner : Boolean);
  begin
    inherited Create;
    FHandle := FileHandle;
    FHandleOwner := HandleOwner;
  end;

Destructor TFileWriter.Destroy;
  begin
    if FHandleOwner and (FHandle <> -1) and (FHandle <> 0) then
      FileClose(FHandle);
    inherited Destroy;
  end;

function TFileWriter.GetPosition : Int64;
  begin
    Result := FileSeek(FHandle, Int64(0), 1);
    if Result = -1 then
      raise EFileWriter.Create('File error');
  end;

procedure TFileWriter.SetPosition(const Position : Int64);
  begin
    if FileSeek(FHandle, Position, 0) = -1 then
      raise EFileWriter.Create('File seek error');
  end;

function TFileWriter.GetSize : Int64;
var I : Int64;
  begin
    I := GetPosition;
    Result := FileSeek(FHandle, Int64(0), 2);
    SetPosition(I);
    if Result = -1 then
      raise EFileWriter.Create('File error');
  end;

procedure TFileWriter.SetSize(const Size : Int64);
  begin
    SetPosition(Size);
    if not SetEndOfFile(FHandle) then
      raise EFileWriter.Create('File resize error');
  end;

function TFileWriter.Write(const Buffer; const Size : Integer) : Integer;
var I : Integer;
  begin
    if Size <= 0 then
      begin
        Result := 0;
        exit;
      end;
    I := FileWrite(FHandle, Buffer, Size);
    if I < 0 then {$IFDEF DELPHI6_UP}
      RaiseLastOSError; {$ELSE}
      RaiseLastWin32Error; {$ENDIF}
    Result := I;
  end;

procedure TFileWriter.DeleteFile;
  begin
    if FFileName = '' then
      raise EFileWriter.Create('No filename');
    if (FHandle <> -1) and (FHandle <> 0) then
      FileClose(FHandle);
    FHandle := -1;
    SysUtils.DeleteFile(FFileName);
  end;



{                                                                              }
{ TOutputWriter                                                                }
{                                                                              }
function TOutputWriter.Write(const Buffer; const Size : Integer) : Integer;
var I : Integer;
    P : PByte;
  begin
    if Size <= 0 then
      begin
        Result := 0;
        exit;
      end;
    P := @Buffer;
    For I := 1 to Size do
      begin
        System.Write(Char(P^));
        Inc(P);
      end;
    Result := Size;
  end;



end.

