{$INCLUDE ..\cDefines.inc}
unit cTypes;

{                                                                              }
{                            Type base class v3.04                             }
{                                                                              }
{      This unit is copyright  1999-2002 by David Butler (david@e.co.za)      }
{                                                                              }
{                  This unit is part of Delphi Fundamentals.                   }
{                    Its original file name is cTypes.pas                      }
{       The latest version is available from the Fundamentals home page        }
{                     http://fundementals.sourceforge.net/                     }
{                                                                              }
{                I invite you to use this unit, free of charge.                }
{        I invite you to distibute this unit, but it must be for free.         }
{             I also invite you to contribute to its development,              }
{             but do not distribute a modified copy of this file.              }
{                                                                              }
{          A forum is available on SourceForge for general discussion          }
{             http://sourceforge.net/forum/forum.php?forum_id=2117             }
{                                                                              }
{                                                                              }
{ Revision history:                                                            }
{   [ cTypes ]                                                                 }
{   1999/11/12  0.01  Split cTypes from cDataStruct and cHolder.               }
{                      Default implementations for Assign, IsEqual             }
{   2001/07/30  1.02  Removed interfaces in AType (each interface adds four    }
{                      bytes to the instance size).                            }
{   [ cDataStructs ]                                                           }
{   2001/08/20  2.03  Merged cTypes and cDataStructs to allow object           }
{                      interface implementation in base classes.               }
{   [ cTypes ]                                                                 }
{   2002/05/15  3.04  Created cTypes from cDataStructs.                        }
{                                                                              }

interface

uses
  { Delphi }
  SysUtils,

  { Fundamentals }
  cUtils;



{                                                                              }
{ Note on class naming convention:                                             }
{   Classes with the A-prefix are abstract base classes. They define the       }
{   interface for the type and must never be instanciated.                     }
{                                                                              }



{                                                                              }
{ AType                                                                        }
{   Abstract base class for data structures.                                   }
{                                                                              }
{   Provides an interface for commonly used data operations such as            }
{   assigning, comparing and duplicating.                                      }
{                                                                              }
{   Duplicate creates a new instance of the object and copies the content.     }
{   Clear sets an instance's content (value) to an empty/zero state.           }
{   IsEqual compares content of an instances.                                  }
{   Compare is the ranking function used by sorting and searching.             }
{   Assign's default implementation calls the protected AssignTo.              }
{                                                                              }
type
  AType = class
  protected
    procedure TypeError(const Msg: String; const Error: Exception = nil;
              const ErrorClass: ExceptClass = nil);
    procedure MethodNotImplementedError(const Method: String);

    procedure Init; virtual;
    procedure AssignTo(const Dest: TObject); virtual;

    function  GetAsString: String; virtual;
    procedure SetAsString(const S: String); virtual;

  public
    constructor Create; virtual;
    class function CreateInstance: AType; virtual;

    function  Duplicate: TObject; virtual;
    procedure Assign(const Source: TObject); virtual;
    procedure Clear; virtual;
    function  IsEmpty: Boolean; virtual;
    function  IsEqual(const V: TObject): Boolean; virtual;
    function  Compare(const V: TObject): TCompareResult; virtual;
    function  HashValue: LongWord; virtual;
    property  AsString: String read GetAsString write SetAsString;
  end;
  EType = class(Exception);
  TypeClass = class of AType;
  ATypeArray = Array of AType;
  TypeClassArray = Array of TypeClass;



{                                                                              }
{ AType helper functions                                                       }
{                                                                              }
function  TypeDuplicate(const V: TObject): TObject;
procedure TypeAssign(const A, B: TObject);
function  TypeIsEqual(const A, B: TObject): Boolean;
function  TypeCompare(const A, B: TObject): TCompareResult;
function  TypeGetAsString(const V: TObject): String;
procedure TypeSetAsString(const V: TObject; const S: String);
function  TypeHashValue(const A: TObject): LongWord;



implementation



{                                                                              }
{ AType                                                                        }
{                                                                              }
constructor AType.Create;
  begin
    inherited Create;
    Init;
  end;

procedure AType.Init;
  begin
  end;

procedure AType.TypeError(const Msg: String; const Error: Exception; const ErrorClass: ExceptClass);
var S: String;
  begin
    S := {$IFDEF DEBUG}ObjectClassName(self) + ': ' + {$ENDIF}
         Msg;
    if Assigned(Error) then
      S := S + ': ' + Error.Message;
    if Assigned(ErrorClass) then
      raise ErrorClass.Create(S) else
      raise EType.Create(S);
  end;

procedure AType.MethodNotImplementedError(const Method: String);
  begin
    TypeError('Method ' + ObjectClassName(self) + '.' + Method + ' not implemented');
  end;

class function AType.CreateInstance: AType;
  begin
    Result := AType(TypeClass(self).Create);
  end;

procedure AType.Clear;
  begin
    MethodNotImplementedError('Clear');
  end;

{$WARNINGS OFF}
function AType.IsEmpty: Boolean;
  begin
    MethodNotImplementedError('IsEmpty');
  end;
{$WARNINGS ON}

function AType.Duplicate: TObject;
  begin
    try
      Result := CreateInstance;
      try
        AType(Result).Assign(self);
      except
        FreeAndNil(Result);
        raise;
      end;
    except
      on E: Exception do
        TypeError('Duplicate failed', E);
    end;
  end;

procedure AType.Assign(const Source: TObject);
var R : Boolean;
  begin
    if Source is AType then
      try
        AType(Source).AssignTo(self);
        R := True;
      except
        R := False;
      end else
      R := False;
    if not R then
      TypeError(ObjectClassName(self) + ' can not assign from ' + ObjectClassName(Source));
  end;

procedure AType.AssignTo(const Dest: TObject);
  begin
    TypeError(ObjectClassName(self) + ' can not assign to ' + ObjectClassName(Dest));
  end;

{$WARNINGS OFF}
function AType.IsEqual(const V: TObject): Boolean;
  begin
    TypeError(ObjectClassName(self) + ' can not compare with ' + ObjectClassName(V));
  end;

function AType.Compare(const V: TObject): TCompareResult;
  begin
    TypeError(ObjectClassName(self) + ' can not compare with ' + ObjectClassName(V));
  end;

function AType.HashValue: LongWord;
  begin
    try
      Result := HashStr(GetAsString, MaxLongWord, True);
    except
      on E : Exception do
        TypeError('Hash error', E);
    end;
  end;
{$WARNINGS ON}

function AType.GetAsString: String;
  begin
    MethodNotImplementedError('GetAsString');
  end;

procedure AType.SetAsString(const S: String);
  begin
    MethodNotImplementedError('SetAsString');
  end;



{                                                                              }
{ AType helper functions                                                       }
{                                                                              }
function TypeGetAsString(const V: TObject): String;
  begin
    if V is AType then
      Result := AType(V).GetAsString else
      raise EType.Create(ObjectClassName(V) + ' can not convert to string');
  end;

procedure TypeSetAsString(const V: TObject; const S: String);
  begin
    if V is AType then
      AType(V).SetAsString(S) else
      raise EType.Create(ObjectClassName(V) + ' can not set as string');
  end;

function TypeDuplicate(const V: TObject): TObject;
  begin
    if V is AType then
      Result := AType(V).Duplicate else
    if not Assigned(V) then
      Result := nil else
      raise EType.Create(ObjectClassName(V) + ' can not duplicate');
  end;

function TypeIsEqual(const A, B: TObject): Boolean;
  begin
    if A = B then
      Result := True else
    if A is AType then
      Result := AType(A).IsEqual(B) else
    if B is AType then
      Result := AType(B).IsEqual(A) else
      raise EType.Create(ObjectClassName(A) + ' and ' + ObjectClassName(B) + ' can not compare');
  end;

function TypeCompare(const A, B: TObject): TCompareResult;
  begin
    if A = B then
      Result := crEqual else
    if A is AType then
      Result := AType(A).Compare(B) else
    if B is AType then
      Result := NegatedCompareResult(AType(B).Compare(A)) else
      Result := crUndefined;
  end;

procedure TypeAssign(const A, B: TObject);
  begin
    if A = B then
      exit else
    if A is AType then
      AType(A).Assign(B) else
    if B is AType then
      AType(B).AssignTo(A) else
      raise EType.Create(ObjectClassName(B) + ' can not assign to ' + ObjectClassName(A));
  end;

{$WARNINGS OFF}
function TypeHashValue(const A: TObject): LongWord;
  begin
    if not Assigned(A) then
      Result := 0 else
    if A is AType then
      Result := AType(A).HashValue else
      raise EType.Create(ObjectClassName(A) + ' can not calculate hash value');
  end;
{$WARNINGS ON}



end.
