{$INCLUDE ..\cDefines.inc}
unit cDictionaries;

{                                                                              }
{                     Data structures: Dictionaries v3.10                      }
{                                                                              }
{      This unit is copyright  1999-2002 by David Butler (david@e.co.za)      }
{                                                                              }
{                  This unit is part of Delphi Fundamentals.                   }
{                 Its original file name is cDictionaries.pas                  }
{                     It was generated 17 Dec 2002 14:44.                      }
{       The latest version is available from the Fundamentals home page        }
{                     http://fundementals.sourceforge.net/                     }
{                                                                              }
{                I invite you to use this unit, free of charge.                }
{        I invite you to distibute this unit, but it must be for free.         }
{             I also invite you to contribute to its development,              }
{             but do not distribute a modified copy of this file.              }
{                                                                              }
{          A forum is available on SourceForge for general discussion          }
{             http://sourceforge.net/forum/forum.php?forum_id=2117             }
{                                                                              }
{                                                                              }
{ Revision history:                                                            }
{   [ cDataStructs ]                                                           }
{   1999/11/12  0.01  Split cTypes from cDataStruct and cHolder.               }
{   2000/06/16  1.02  Added ADictionary.                                       }
{   2000/06/14  1.03  Converted cDataStructs to template.                      }
{   2000/06/16  1.04  Added dictionaries stored in AArrays.                    }
{   2000/07/07  1.05  Added ATypeDictionary.                                   }
{   2001/01/19  1.06  Added THashedStringDictionary.                           }
{   2001/04/13  1.07  Added TObjectDictionary.                                 }
{   2001/08/20  2.08  Merged cTypes and cDataStructs to allow object           }
{                     interface implementation in base classes.                }
{   2002/01/14  2.09  Replaced AllowDuplicates property with DuplicatesAction  }
{                     property.                                                }
{   [ cDictionaries ]                                                          }
{   2002/05/15  3.10  Created cDictionaries unit from cDataStructs.            }
{                     Refactored for Fundamentals 3.                           }
{                                                                              }

interface

uses
  { Delphi }
  SysUtils,

  { Fundamentals }
  cUtils,
  cTypes,
  cArrays;

const
  UnitName      = 'cDictionaries';
  UnitVersion   = '3.10';
  UnitDesc      = 'Data structures: Dictionaries';
  UnitCopyright = '(c) 1999-2002 David Butler';



{                                                                              }
{ DICTIONARY BASE CLASSES                                                      }
{   Classes with the A-prefix are Abstract base classes. They define the       }
{   interface for the type and must never be instanciated.                     }
{   Instead, create one of the implementation classes (T-prefix).              }
{                                                                              }



{                                                                              }
{ ADictionary                                                                  }
{   Base class for a dictionary (key-value pair where the key is a string).    }
{                                                                              }
type
  TDictionaryDuplicatesAction = (ddError,    // raises an exception on duplicate keys
                                 ddAccept,   // allow duplicate keys
                                 ddIgnore);  // silently discard duplicates
  ADictionary = class(AType)
  protected
    procedure DictionaryError(const Msg: String);
    procedure KeyNotFoundError(const Key: String);

    function  GetAddOnSet: Boolean; virtual; abstract;
    procedure SetAddOnSet(const AddOnSet: Boolean); virtual; abstract;
    function  GetDuplicatesAction: TDictionaryDuplicatesAction; virtual; abstract;
    procedure SetDuplicatesAction(const Value: TDictionaryDuplicatesAction); virtual; abstract;
    function  GetKeysCaseSensitive: Boolean; virtual; abstract;

  public
    procedure Delete(const Key: String); virtual; abstract;
    function  HasKey(const Key: String): Boolean; virtual; abstract;
    procedure Rename(const Key, NewKey: String); virtual; abstract;

    function  Count: Integer; virtual; abstract;
    function  GetKeyByIndex(const Idx: Integer): String; virtual; abstract;

    property  AddOnSet: Boolean read GetAddOnSet write SetAddOnSet;
    property  DuplicatesAction: TDictionaryDuplicatesAction
              read GetDuplicatesAction write SetDuplicatesAction;
    property  KeysCaseSensitive: Boolean read GetKeysCaseSensitive;
  end;
  EDictionary = class(EType);



{                                                                              }
{ ALongIntDictionary                                                           }
{   A Dictionary with LongInt values and String keys.                          }
{                                                                              }
type
  ALongIntDictionary = class(ADictionary)
  protected
    function  GetAsString: String; override;

    function  GetItem(const Key: String): LongInt; virtual;
    procedure SetItem(const Key: String; const Value: LongInt); virtual; abstract;

  public
    { AType implementations                                                    }
    procedure Assign(const Source: TObject); override;

    { ALongIntDictionary interface                                            }
    property  Item[const Key: String]: LongInt read GetItem write SetItem; default;
    procedure Add(const Key: String; const Value: LongInt); virtual; abstract;

    function  GetItemByIndex(const Idx: Integer): LongInt; virtual; abstract;
    function  LocateItem(const Key: String; var Value: LongInt): Integer; virtual; abstract;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: LongInt): Integer; virtual; abstract;
  end;
  ELongIntDictionary = class(EDictionary);



{                                                                              }
{ AIntegerDictionary                                                           }
{                                                                              }
type
  AIntegerDictionary = ALongIntDictionary;



{                                                                              }
{ ALongWordDictionary                                                          }
{   A Dictionary with LongWord values and String keys.                         }
{                                                                              }
type
  ALongWordDictionary = class(ADictionary)
  protected
    function  GetAsString: String; override;

    function  GetItem(const Key: String): LongWord; virtual;
    procedure SetItem(const Key: String; const Value: LongWord); virtual; abstract;

  public
    { AType implementations                                                    }
    procedure Assign(const Source: TObject); override;

    { ALongWordDictionary interface                                            }
    property  Item[const Key: String]: LongWord read GetItem write SetItem; default;
    procedure Add(const Key: String; const Value: LongWord); virtual; abstract;

    function  GetItemByIndex(const Idx: Integer): LongWord; virtual; abstract;
    function  LocateItem(const Key: String; var Value: LongWord): Integer; virtual; abstract;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: LongWord): Integer; virtual; abstract;
  end;
  ELongWordDictionary = class(EDictionary);



{                                                                              }
{ ACardinalArray                                                               }
{                                                                              }
type
  ACardinalDictionary = ALongWordDictionary;



{                                                                              }
{ AInt64Dictionary                                                             }
{   A Dictionary with Int64 values and String keys.                            }
{                                                                              }
type
  AInt64Dictionary = class(ADictionary)
  protected
    function  GetAsString: String; override;

    function  GetItem(const Key: String): Int64; virtual;
    procedure SetItem(const Key: String; const Value: Int64); virtual; abstract;

  public
    { AType implementations                                                    }
    procedure Assign(const Source: TObject); override;

    { AInt64Dictionary interface                                            }
    property  Item[const Key: String]: Int64 read GetItem write SetItem; default;
    procedure Add(const Key: String; const Value: Int64); virtual; abstract;

    function  GetItemByIndex(const Idx: Integer): Int64; virtual; abstract;
    function  LocateItem(const Key: String; var Value: Int64): Integer; virtual; abstract;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: Int64): Integer; virtual; abstract;
  end;
  EInt64Dictionary = class(EDictionary);



{                                                                              }
{ ASingleDictionary                                                            }
{   A Dictionary with Single values and String keys.                           }
{                                                                              }
type
  ASingleDictionary = class(ADictionary)
  protected
    function  GetAsString: String; override;

    function  GetItem(const Key: String): Single; virtual;
    procedure SetItem(const Key: String; const Value: Single); virtual; abstract;

  public
    { AType implementations                                                    }
    procedure Assign(const Source: TObject); override;

    { ASingleDictionary interface                                            }
    property  Item[const Key: String]: Single read GetItem write SetItem; default;
    procedure Add(const Key: String; const Value: Single); virtual; abstract;

    function  GetItemByIndex(const Idx: Integer): Single; virtual; abstract;
    function  LocateItem(const Key: String; var Value: Single): Integer; virtual; abstract;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: Single): Integer; virtual; abstract;
  end;
  ESingleDictionary = class(EDictionary);



{                                                                              }
{ ADoubleDictionary                                                            }
{   A Dictionary with Double values and String keys.                           }
{                                                                              }
type
  ADoubleDictionary = class(ADictionary)
  protected
    function  GetAsString: String; override;

    function  GetItem(const Key: String): Double; virtual;
    procedure SetItem(const Key: String; const Value: Double); virtual; abstract;

  public
    { AType implementations                                                    }
    procedure Assign(const Source: TObject); override;

    { ADoubleDictionary interface                                            }
    property  Item[const Key: String]: Double read GetItem write SetItem; default;
    procedure Add(const Key: String; const Value: Double); virtual; abstract;

    function  GetItemByIndex(const Idx: Integer): Double; virtual; abstract;
    function  LocateItem(const Key: String; var Value: Double): Integer; virtual; abstract;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: Double): Integer; virtual; abstract;
  end;
  EDoubleDictionary = class(EDictionary);



{                                                                              }
{ AExtendedDictionary                                                          }
{   A Dictionary with Extended values and String keys.                         }
{                                                                              }
type
  AExtendedDictionary = class(ADictionary)
  protected
    function  GetAsString: String; override;

    function  GetItem(const Key: String): Extended; virtual;
    procedure SetItem(const Key: String; const Value: Extended); virtual; abstract;

  public
    { AType implementations                                                    }
    procedure Assign(const Source: TObject); override;

    { AExtendedDictionary interface                                            }
    property  Item[const Key: String]: Extended read GetItem write SetItem; default;
    procedure Add(const Key: String; const Value: Extended); virtual; abstract;

    function  GetItemByIndex(const Idx: Integer): Extended; virtual; abstract;
    function  LocateItem(const Key: String; var Value: Extended): Integer; virtual; abstract;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: Extended): Integer; virtual; abstract;
  end;
  EExtendedDictionary = class(EDictionary);



{                                                                              }
{ APointerDictionary                                                           }
{   A Dictionary with Pointer values and String keys.                          }
{                                                                              }
type
  APointerDictionary = class(ADictionary)
  protected
    function  GetAsString: String; override;

    function  GetItem(const Key: String): Pointer; virtual;
    procedure SetItem(const Key: String; const Value: Pointer); virtual; abstract;

  public
    { AType implementations                                                    }
    procedure Assign(const Source: TObject); override;

    { APointerDictionary interface                                            }
    property  Item[const Key: String]: Pointer read GetItem write SetItem; default;
    procedure Add(const Key: String; const Value: Pointer); virtual; abstract;

    function  GetItemByIndex(const Idx: Integer): Pointer; virtual; abstract;
    function  LocateItem(const Key: String; var Value: Pointer): Integer; virtual; abstract;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: Pointer): Integer; virtual; abstract;
  end;
  EPointerDictionary = class(EDictionary);



{                                                                              }
{ AStringDictionary                                                            }
{   A Dictionary with String values and String keys.                           }
{                                                                              }
type
  AStringDictionary = class(ADictionary)
  protected
    function  GetAsString: String; override;

    function  GetItem(const Key: String): String; virtual;
    procedure SetItem(const Key: String; const Value: String); virtual; abstract;

  public
    { AType implementations                                                    }
    procedure Assign(const Source: TObject); override;

    { AStringDictionary interface                                            }
    property  Item[const Key: String]: String read GetItem write SetItem; default;
    procedure Add(const Key: String; const Value: String); virtual; abstract;

    function  GetItemByIndex(const Idx: Integer): String; virtual; abstract;
    function  LocateItem(const Key: String; var Value: String): Integer; virtual; abstract;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: String): Integer; virtual; abstract;

    function  GetItemLength(const Key: String): Integer; virtual;
    function  GetTotalLength: Int64; virtual;
  end;
  EStringDictionary = class(EDictionary);






{                                                                              }
{ AObjectDictionary                                                            }
{                                                                              }
type
  AObjectDictionary = class(ADictionary)
  protected
    function  GetItem(const Key: String): TObject; virtual;
    procedure SetItem(const Key: String; const Value: TObject); virtual; abstract;
    function  GetIsItemOwner: Boolean; virtual; abstract;
    procedure SetIsItemOwner(const IsItemOwner: Boolean); virtual; abstract;

  public
    { AType implementation                                                     }
    function  GetAsString: String; override;
    procedure Clear; override;
    procedure Assign(const Source: TObject); reintroduce; overload; override;

    { AObjectDictionary interface                                              }
    procedure Add(const Key: String; const Value: TObject); virtual; abstract;
    property  Item[const Key: String]: TObject read GetItem write SetItem; default;

    function  GetItemByIndex(const Idx: Integer): TObject; virtual; abstract;
    function  LocateItem(const Key: String; var Value: TObject): Integer; virtual; abstract;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: TObject): Integer; virtual; abstract;

    property  IsItemOwner: Boolean read GetIsItemOwner write SetIsItemOwner;
    function  ReleaseItem(const Key: String): TObject; virtual; abstract;
    procedure ReleaseItems; virtual; abstract;
    procedure FreeItems; virtual; abstract;
  end;
  EObjectDictionary = class(EDictionary);



{                                                                              }
{ DICTIONARY IMPLEMENTATIONS                                                   }
{                                                                              }



{                                                                              }
{ TLongIntDictionary                                                           }
{   Implements ALongIntDictionary using arrays.                                }
{   A 'chained-hash' lookup table is used for quick access.                    }
{                                                                              }
type
  TLongIntDictionary = class(ALongIntDictionary)
  protected
    FKeys             : AStringArray;
    FValues           : ALongIntArray;
    FLookup           : Array of IntegerArray;
    FCaseSensitive    : Boolean;
    FAddOnSet         : Boolean;
    FDuplicatesAction : TDictionaryDuplicatesAction;

    function  LocateKey(const Key: String; var LookupIdx: Integer;
              const ErrorIfNotFound: Boolean): Integer;
    function  KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
    procedure DeleteByIndex(const Idx: Integer; const Hash: Integer = -1);
    procedure Rehash;
    function  GetHashTableSize: Integer;
    procedure IndexError;

    { ADictionary implementations                                              }
    function  GetKeysCaseSensitive: Boolean; override;
    function  GetAddOnSet: Boolean; override;
    procedure SetAddOnSet(const AddOnSet: Boolean); override;
    function  GetDuplicatesAction: TDictionaryDuplicatesAction; override;
    procedure SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction); override;

    { ALongIntDictionary implementations                                    }
    procedure SetItem(const Key: String; const Value: LongInt); override;

  public
    { TLongIntDictionary interface                                            }
    constructor Create; override;
    constructor CreateEx(const Keys: AStringArray = nil; const Values: ALongIntArray = nil;
                const KeysCaseSensitive: Boolean = True;
                const AddOnSet: Boolean = True;
                const DuplicatesAction: TDictionaryDuplicatesAction = ddAccept);
    destructor Destroy; override;

    { AType implementations                                                    }
    procedure Clear; override;

    { ADictionary implementations                                              }
    procedure Delete(const Key: String); override;
    function  HasKey(const Key: String): Boolean; override;
    procedure Rename(const Key: String; const NewKey: String); override;
    function  Count: Integer; override;
    function  GetKeyByIndex(const Idx: Integer): String; override;

    { ALongIntDictionary implementations                                    }
    procedure Add(const Key: String; const Value: LongInt); override;
    function  GetItemByIndex(const Idx: Integer): LongInt; override;
    procedure SetItemByIndex(const Idx: Integer; const Value: LongInt);
    function  LocateItem(const Key: String; var Value: LongInt): Integer; override;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: LongInt): Integer; override;

    { TLongIntDictionary interface                                            }
    property  HashTableSize: Integer read GetHashTableSize;
    procedure DeleteItemByIndex(const Idx: Integer);
  end;



{                                                                              }
{ TIntegerDictionary                                                           }
{                                                                              }
type
  TIntegerDictionary = TLongIntDictionary;



{                                                                              }
{ TLongWordDictionary                                                          }
{   Implements ALongWordDictionary using arrays.                               }
{   A 'chained-hash' lookup table is used for quick access.                    }
{                                                                              }
type
  TLongWordDictionary = class(ALongWordDictionary)
  protected
    FKeys             : AStringArray;
    FValues           : ALongWordArray;
    FLookup           : Array of IntegerArray;
    FCaseSensitive    : Boolean;
    FAddOnSet         : Boolean;
    FDuplicatesAction : TDictionaryDuplicatesAction;

    function  LocateKey(const Key: String; var LookupIdx: Integer;
              const ErrorIfNotFound: Boolean): Integer;
    function  KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
    procedure DeleteByIndex(const Idx: Integer; const Hash: Integer = -1);
    procedure Rehash;
    function  GetHashTableSize: Integer;
    procedure IndexError;

    { ADictionary implementations                                              }
    function  GetKeysCaseSensitive: Boolean; override;
    function  GetAddOnSet: Boolean; override;
    procedure SetAddOnSet(const AddOnSet: Boolean); override;
    function  GetDuplicatesAction: TDictionaryDuplicatesAction; override;
    procedure SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction); override;

    { ALongWordDictionary implementations                                    }
    procedure SetItem(const Key: String; const Value: LongWord); override;

  public
    { TLongWordDictionary interface                                            }
    constructor Create; override;
    constructor CreateEx(const Keys: AStringArray = nil; const Values: ALongWordArray = nil;
                const KeysCaseSensitive: Boolean = True;
                const AddOnSet: Boolean = True;
                const DuplicatesAction: TDictionaryDuplicatesAction = ddAccept);
    destructor Destroy; override;

    { AType implementations                                                    }
    procedure Clear; override;

    { ADictionary implementations                                              }
    procedure Delete(const Key: String); override;
    function  HasKey(const Key: String): Boolean; override;
    procedure Rename(const Key: String; const NewKey: String); override;
    function  Count: Integer; override;
    function  GetKeyByIndex(const Idx: Integer): String; override;

    { ALongWordDictionary implementations                                    }
    procedure Add(const Key: String; const Value: LongWord); override;
    function  GetItemByIndex(const Idx: Integer): LongWord; override;
    procedure SetItemByIndex(const Idx: Integer; const Value: LongWord);
    function  LocateItem(const Key: String; var Value: LongWord): Integer; override;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: LongWord): Integer; override;

    { TLongWordDictionary interface                                            }
    property  HashTableSize: Integer read GetHashTableSize;
    procedure DeleteItemByIndex(const Idx: Integer);
  end;



{                                                                              }
{ TCardinalDictionary                                                          }
{                                                                              }
type
  TCardinalDictionary = TLongWordDictionary;



{                                                                              }
{ TInt64Dictionary                                                             }
{   Implements AInt64Dictionary using arrays.                                  }
{   A 'chained-hash' lookup table is used for quick access.                    }
{                                                                              }
type
  TInt64Dictionary = class(AInt64Dictionary)
  protected
    FKeys             : AStringArray;
    FValues           : AInt64Array;
    FLookup           : Array of IntegerArray;
    FCaseSensitive    : Boolean;
    FAddOnSet         : Boolean;
    FDuplicatesAction : TDictionaryDuplicatesAction;

    function  LocateKey(const Key: String; var LookupIdx: Integer;
              const ErrorIfNotFound: Boolean): Integer;
    function  KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
    procedure DeleteByIndex(const Idx: Integer; const Hash: Integer = -1);
    procedure Rehash;
    function  GetHashTableSize: Integer;
    procedure IndexError;

    { ADictionary implementations                                              }
    function  GetKeysCaseSensitive: Boolean; override;
    function  GetAddOnSet: Boolean; override;
    procedure SetAddOnSet(const AddOnSet: Boolean); override;
    function  GetDuplicatesAction: TDictionaryDuplicatesAction; override;
    procedure SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction); override;

    { AInt64Dictionary implementations                                    }
    procedure SetItem(const Key: String; const Value: Int64); override;

  public
    { TInt64Dictionary interface                                            }
    constructor Create; override;
    constructor CreateEx(const Keys: AStringArray = nil; const Values: AInt64Array = nil;
                const KeysCaseSensitive: Boolean = True;
                const AddOnSet: Boolean = True;
                const DuplicatesAction: TDictionaryDuplicatesAction = ddAccept);
    destructor Destroy; override;

    { AType implementations                                                    }
    procedure Clear; override;

    { ADictionary implementations                                              }
    procedure Delete(const Key: String); override;
    function  HasKey(const Key: String): Boolean; override;
    procedure Rename(const Key: String; const NewKey: String); override;
    function  Count: Integer; override;
    function  GetKeyByIndex(const Idx: Integer): String; override;

    { AInt64Dictionary implementations                                    }
    procedure Add(const Key: String; const Value: Int64); override;
    function  GetItemByIndex(const Idx: Integer): Int64; override;
    procedure SetItemByIndex(const Idx: Integer; const Value: Int64);
    function  LocateItem(const Key: String; var Value: Int64): Integer; override;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: Int64): Integer; override;

    { TInt64Dictionary interface                                            }
    property  HashTableSize: Integer read GetHashTableSize;
    procedure DeleteItemByIndex(const Idx: Integer);
  end;



{                                                                              }
{ TSingleDictionary                                                            }
{   Implements ASingleDictionary using arrays.                                 }
{   A 'chained-hash' lookup table is used for quick access.                    }
{                                                                              }
type
  TSingleDictionary = class(ASingleDictionary)
  protected
    FKeys             : AStringArray;
    FValues           : ASingleArray;
    FLookup           : Array of IntegerArray;
    FCaseSensitive    : Boolean;
    FAddOnSet         : Boolean;
    FDuplicatesAction : TDictionaryDuplicatesAction;

    function  LocateKey(const Key: String; var LookupIdx: Integer;
              const ErrorIfNotFound: Boolean): Integer;
    function  KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
    procedure DeleteByIndex(const Idx: Integer; const Hash: Integer = -1);
    procedure Rehash;
    function  GetHashTableSize: Integer;
    procedure IndexError;

    { ADictionary implementations                                              }
    function  GetKeysCaseSensitive: Boolean; override;
    function  GetAddOnSet: Boolean; override;
    procedure SetAddOnSet(const AddOnSet: Boolean); override;
    function  GetDuplicatesAction: TDictionaryDuplicatesAction; override;
    procedure SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction); override;

    { ASingleDictionary implementations                                    }
    procedure SetItem(const Key: String; const Value: Single); override;

  public
    { TSingleDictionary interface                                            }
    constructor Create; override;
    constructor CreateEx(const Keys: AStringArray = nil; const Values: ASingleArray = nil;
                const KeysCaseSensitive: Boolean = True;
                const AddOnSet: Boolean = True;
                const DuplicatesAction: TDictionaryDuplicatesAction = ddAccept);
    destructor Destroy; override;

    { AType implementations                                                    }
    procedure Clear; override;

    { ADictionary implementations                                              }
    procedure Delete(const Key: String); override;
    function  HasKey(const Key: String): Boolean; override;
    procedure Rename(const Key: String; const NewKey: String); override;
    function  Count: Integer; override;
    function  GetKeyByIndex(const Idx: Integer): String; override;

    { ASingleDictionary implementations                                    }
    procedure Add(const Key: String; const Value: Single); override;
    function  GetItemByIndex(const Idx: Integer): Single; override;
    procedure SetItemByIndex(const Idx: Integer; const Value: Single);
    function  LocateItem(const Key: String; var Value: Single): Integer; override;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: Single): Integer; override;

    { TSingleDictionary interface                                            }
    property  HashTableSize: Integer read GetHashTableSize;
    procedure DeleteItemByIndex(const Idx: Integer);
  end;



{                                                                              }
{ TDoubleDictionary                                                            }
{   Implements ADoubleDictionary using arrays.                                 }
{   A 'chained-hash' lookup table is used for quick access.                    }
{                                                                              }
type
  TDoubleDictionary = class(ADoubleDictionary)
  protected
    FKeys             : AStringArray;
    FValues           : ADoubleArray;
    FLookup           : Array of IntegerArray;
    FCaseSensitive    : Boolean;
    FAddOnSet         : Boolean;
    FDuplicatesAction : TDictionaryDuplicatesAction;

    function  LocateKey(const Key: String; var LookupIdx: Integer;
              const ErrorIfNotFound: Boolean): Integer;
    function  KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
    procedure DeleteByIndex(const Idx: Integer; const Hash: Integer = -1);
    procedure Rehash;
    function  GetHashTableSize: Integer;
    procedure IndexError;

    { ADictionary implementations                                              }
    function  GetKeysCaseSensitive: Boolean; override;
    function  GetAddOnSet: Boolean; override;
    procedure SetAddOnSet(const AddOnSet: Boolean); override;
    function  GetDuplicatesAction: TDictionaryDuplicatesAction; override;
    procedure SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction); override;

    { ADoubleDictionary implementations                                    }
    procedure SetItem(const Key: String; const Value: Double); override;

  public
    { TDoubleDictionary interface                                            }
    constructor Create; override;
    constructor CreateEx(const Keys: AStringArray = nil; const Values: ADoubleArray = nil;
                const KeysCaseSensitive: Boolean = True;
                const AddOnSet: Boolean = True;
                const DuplicatesAction: TDictionaryDuplicatesAction = ddAccept);
    destructor Destroy; override;

    { AType implementations                                                    }
    procedure Clear; override;

    { ADictionary implementations                                              }
    procedure Delete(const Key: String); override;
    function  HasKey(const Key: String): Boolean; override;
    procedure Rename(const Key: String; const NewKey: String); override;
    function  Count: Integer; override;
    function  GetKeyByIndex(const Idx: Integer): String; override;

    { ADoubleDictionary implementations                                    }
    procedure Add(const Key: String; const Value: Double); override;
    function  GetItemByIndex(const Idx: Integer): Double; override;
    procedure SetItemByIndex(const Idx: Integer; const Value: Double);
    function  LocateItem(const Key: String; var Value: Double): Integer; override;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: Double): Integer; override;

    { TDoubleDictionary interface                                            }
    property  HashTableSize: Integer read GetHashTableSize;
    procedure DeleteItemByIndex(const Idx: Integer);
  end;



{                                                                              }
{ TExtendedDictionary                                                          }
{   Implements AExtendedDictionary using arrays.                               }
{   A 'chained-hash' lookup table is used for quick access.                    }
{                                                                              }
type
  TExtendedDictionary = class(AExtendedDictionary)
  protected
    FKeys             : AStringArray;
    FValues           : AExtendedArray;
    FLookup           : Array of IntegerArray;
    FCaseSensitive    : Boolean;
    FAddOnSet         : Boolean;
    FDuplicatesAction : TDictionaryDuplicatesAction;

    function  LocateKey(const Key: String; var LookupIdx: Integer;
              const ErrorIfNotFound: Boolean): Integer;
    function  KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
    procedure DeleteByIndex(const Idx: Integer; const Hash: Integer = -1);
    procedure Rehash;
    function  GetHashTableSize: Integer;
    procedure IndexError;

    { ADictionary implementations                                              }
    function  GetKeysCaseSensitive: Boolean; override;
    function  GetAddOnSet: Boolean; override;
    procedure SetAddOnSet(const AddOnSet: Boolean); override;
    function  GetDuplicatesAction: TDictionaryDuplicatesAction; override;
    procedure SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction); override;

    { AExtendedDictionary implementations                                    }
    procedure SetItem(const Key: String; const Value: Extended); override;

  public
    { TExtendedDictionary interface                                            }
    constructor Create; override;
    constructor CreateEx(const Keys: AStringArray = nil; const Values: AExtendedArray = nil;
                const KeysCaseSensitive: Boolean = True;
                const AddOnSet: Boolean = True;
                const DuplicatesAction: TDictionaryDuplicatesAction = ddAccept);
    destructor Destroy; override;

    { AType implementations                                                    }
    procedure Clear; override;

    { ADictionary implementations                                              }
    procedure Delete(const Key: String); override;
    function  HasKey(const Key: String): Boolean; override;
    procedure Rename(const Key: String; const NewKey: String); override;
    function  Count: Integer; override;
    function  GetKeyByIndex(const Idx: Integer): String; override;

    { AExtendedDictionary implementations                                    }
    procedure Add(const Key: String; const Value: Extended); override;
    function  GetItemByIndex(const Idx: Integer): Extended; override;
    procedure SetItemByIndex(const Idx: Integer; const Value: Extended);
    function  LocateItem(const Key: String; var Value: Extended): Integer; override;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: Extended): Integer; override;

    { TExtendedDictionary interface                                            }
    property  HashTableSize: Integer read GetHashTableSize;
    procedure DeleteItemByIndex(const Idx: Integer);
  end;



{                                                                              }
{ TStringDictionary                                                            }
{   Implements AStringDictionary using arrays.                                 }
{   A 'chained-hash' lookup table is used for quick access.                    }
{                                                                              }
type
  TStringDictionary = class(AStringDictionary)
  protected
    FKeys             : AStringArray;
    FValues           : AStringArray;
    FLookup           : Array of IntegerArray;
    FCaseSensitive    : Boolean;
    FAddOnSet         : Boolean;
    FDuplicatesAction : TDictionaryDuplicatesAction;

    function  LocateKey(const Key: String; var LookupIdx: Integer;
              const ErrorIfNotFound: Boolean): Integer;
    function  KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
    procedure DeleteByIndex(const Idx: Integer; const Hash: Integer = -1);
    procedure Rehash;
    function  GetHashTableSize: Integer;
    procedure IndexError;

    { ADictionary implementations                                              }
    function  GetKeysCaseSensitive: Boolean; override;
    function  GetAddOnSet: Boolean; override;
    procedure SetAddOnSet(const AddOnSet: Boolean); override;
    function  GetDuplicatesAction: TDictionaryDuplicatesAction; override;
    procedure SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction); override;

    { AStringDictionary implementations                                    }
    procedure SetItem(const Key: String; const Value: String); override;

  public
    { TStringDictionary interface                                            }
    constructor Create; override;
    constructor CreateEx(const Keys: AStringArray = nil; const Values: AStringArray = nil;
                const KeysCaseSensitive: Boolean = True;
                const AddOnSet: Boolean = True;
                const DuplicatesAction: TDictionaryDuplicatesAction = ddAccept);
    destructor Destroy; override;

    { AType implementations                                                    }
    procedure Clear; override;

    { ADictionary implementations                                              }
    procedure Delete(const Key: String); override;
    function  HasKey(const Key: String): Boolean; override;
    procedure Rename(const Key: String; const NewKey: String); override;
    function  Count: Integer; override;
    function  GetKeyByIndex(const Idx: Integer): String; override;

    { AStringDictionary implementations                                    }
    procedure Add(const Key: String; const Value: String); override;
    function  GetItemByIndex(const Idx: Integer): String; override;
    procedure SetItemByIndex(const Idx: Integer; const Value: String);
    function  LocateItem(const Key: String; var Value: String): Integer; override;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: String): Integer; override;

    { TStringDictionary interface                                            }
    property  HashTableSize: Integer read GetHashTableSize;
    procedure DeleteItemByIndex(const Idx: Integer);
  end;



{                                                                              }
{ TObjectDictionary                                                            }
{   Implements AObjectDictionary using arrays.                                 }
{   A 'chained-hash' lookup table is used for quick access.                    }
{                                                                              }
type
  TObjectDictionary = class(AObjectDictionary)
  protected
    FKeys             : AStringArray;
    FValues           : AObjectArray;
    FLookup           : Array of IntegerArray;
    FCaseSensitive    : Boolean;
    FAddOnSet         : Boolean;
    FDuplicatesAction : TDictionaryDuplicatesAction;

    function  LocateKey(const Key: String; var LookupIdx: Integer;
              const ErrorIfNotFound: Boolean): Integer;
    function  KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
    procedure DeleteByIndex(const Idx: Integer; const Hash: Integer = -1);
    procedure Rehash;
    function  GetHashTableSize: Integer;
    procedure IndexError;

    { ADictionary implementations                                              }
    function  GetKeysCaseSensitive: Boolean; override;
    function  GetAddOnSet: Boolean; override;
    procedure SetAddOnSet(const AddOnSet: Boolean); override;
    function  GetDuplicatesAction: TDictionaryDuplicatesAction; override;
    procedure SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction); override;

    { AObjectDictionary implementations                                        }
    function  GetIsItemOwner: Boolean; override;
    procedure SetIsItemOwner(const IsItemOwner: Boolean); override;
    
    procedure SetItem(const Key: String; const Value: TObject); override;

  public
    { TObjectDictionary interface                                            }
    constructor Create; override;
    constructor CreateEx(const Keys: AStringArray = nil; const Values: AObjectArray = nil;
                const IsItemOwner: Boolean = False;
                const KeysCaseSensitive: Boolean = True;
                const AddOnSet: Boolean = True;
                const DuplicatesAction: TDictionaryDuplicatesAction = ddAccept);
    destructor Destroy; override;

    { AType implementations                                                    }
    procedure Clear; override;

    { ADictionary implementations                                              }
    procedure Delete(const Key: String); override;
    function  HasKey(const Key: String): Boolean; override;
    procedure Rename(const Key: String; const NewKey: String); override;
    function  Count: Integer; override;
    function  GetKeyByIndex(const Idx: Integer): String; override;

    { AObjectDictionary implementations                                        }
    procedure Add(const Key: String; const Value: TObject); override;
    function  GetItemByIndex(const Idx: Integer): TObject; override;
    procedure SetItemByIndex(const Idx: Integer; const Value: TObject);
    function  LocateItem(const Key: String; var Value: TObject): Integer; override;
    function  LocateNext(const Key: String; const Idx: Integer;
              var Value: TObject): Integer; override;

    function  ReleaseItem(const Key: String): TObject; override;
    procedure ReleaseItems; override;
    procedure FreeItems; override;

    { TObjectDictionary interface                                            }
    property  HashTableSize: Integer read GetHashTableSize;
    procedure DeleteItemByIndex(const Idx: Integer);
  end;



{                                                                              }
{ Dictionary functions                                                         }
{                                                                              }
const
  AverageHashChainSize = 4;

function  DictionaryRehashSize(const Count: Integer): Integer;



{                                                                              }
{ Self testing code                                                            }
{                                                                              }
procedure SelfTest;



implementation

uses
  { Fundamentals }
  cStrings;



{                                                                              }
{ DICTIONARY BASE CLASSES                                                      }
{                                                                              }



{                                                                              }
{ ADictionary                                                                  }
{                                                                              }
procedure ADictionary.DictionaryError(const Msg: String);
  begin
    TypeError(Msg, nil, EDictionary);
  end;

procedure ADictionary.KeyNotFoundError(const Key: String);
  begin
    DictionaryError('Key not found: ' + Key);
  end;



{                                                                              }
{ ALongIntDictionary                                                           }
{                                                                              }
function ALongIntDictionary.GetItem(const Key: String): LongInt;
  begin
    if LocateItem(Key, Result) < 0 then
      KeyNotFoundError(Key);
  end;

procedure ALongIntDictionary.Assign(const Source: TObject);
var I : Integer;
  begin
    if Source is ALongIntDictionary then
      begin
        Clear;
        For I := 0 to ALongIntDictionary(Source).Count - 1 do
          Add(ALongIntDictionary(Source).GetKeyByIndex(I),
               ALongIntDictionary(Source).GetItemByIndex(I));
      end else
      inherited Assign(Source);
  end;

function ALongIntDictionary.GetAsString: String;
var I, L : Integer;
  begin
    L := Count - 1;
    For I := 0 to L do
      begin
        Result := Result + GetKeyByIndex(I) + ':' + IntToStr(GetItemByIndex(I));
        if I < L then
          Result := Result + ',';
      end;
  end;



{                                                                              }
{ ALongWordDictionary                                                          }
{                                                                              }
function ALongWordDictionary.GetItem(const Key: String): LongWord;
  begin
    if LocateItem(Key, Result) < 0 then
      KeyNotFoundError(Key);
  end;

procedure ALongWordDictionary.Assign(const Source: TObject);
var I : Integer;
  begin
    if Source is ALongWordDictionary then
      begin
        Clear;
        For I := 0 to ALongWordDictionary(Source).Count - 1 do
          Add(ALongWordDictionary(Source).GetKeyByIndex(I),
               ALongWordDictionary(Source).GetItemByIndex(I));
      end else
      inherited Assign(Source);
  end;

function ALongWordDictionary.GetAsString: String;
var I, L : Integer;
  begin
    L := Count - 1;
    For I := 0 to L do
      begin
        Result := Result + GetKeyByIndex(I) + ':' + IntToStr(GetItemByIndex(I));
        if I < L then
          Result := Result + ',';
      end;
  end;



{                                                                              }
{ AInt64Dictionary                                                             }
{                                                                              }
function AInt64Dictionary.GetItem(const Key: String): Int64;
  begin
    if LocateItem(Key, Result) < 0 then
      KeyNotFoundError(Key);
  end;

procedure AInt64Dictionary.Assign(const Source: TObject);
var I : Integer;
  begin
    if Source is AInt64Dictionary then
      begin
        Clear;
        For I := 0 to AInt64Dictionary(Source).Count - 1 do
          Add(AInt64Dictionary(Source).GetKeyByIndex(I),
               AInt64Dictionary(Source).GetItemByIndex(I));
      end else
      inherited Assign(Source);
  end;

function AInt64Dictionary.GetAsString: String;
var I, L : Integer;
  begin
    L := Count - 1;
    For I := 0 to L do
      begin
        Result := Result + GetKeyByIndex(I) + ':' + IntToStr(GetItemByIndex(I));
        if I < L then
          Result := Result + ',';
      end;
  end;



{                                                                              }
{ ASingleDictionary                                                            }
{                                                                              }
function ASingleDictionary.GetItem(const Key: String): Single;
  begin
    if LocateItem(Key, Result) < 0 then
      KeyNotFoundError(Key);
  end;

procedure ASingleDictionary.Assign(const Source: TObject);
var I : Integer;
  begin
    if Source is ASingleDictionary then
      begin
        Clear;
        For I := 0 to ASingleDictionary(Source).Count - 1 do
          Add(ASingleDictionary(Source).GetKeyByIndex(I),
               ASingleDictionary(Source).GetItemByIndex(I));
      end else
      inherited Assign(Source);
  end;

function ASingleDictionary.GetAsString: String;
var I, L : Integer;
  begin
    L := Count - 1;
    For I := 0 to L do
      begin
        Result := Result + GetKeyByIndex(I) + ':' + FloatToStr(GetItemByIndex(I));
        if I < L then
          Result := Result + ',';
      end;
  end;



{                                                                              }
{ ADoubleDictionary                                                            }
{                                                                              }
function ADoubleDictionary.GetItem(const Key: String): Double;
  begin
    if LocateItem(Key, Result) < 0 then
      KeyNotFoundError(Key);
  end;

procedure ADoubleDictionary.Assign(const Source: TObject);
var I : Integer;
  begin
    if Source is ADoubleDictionary then
      begin
        Clear;
        For I := 0 to ADoubleDictionary(Source).Count - 1 do
          Add(ADoubleDictionary(Source).GetKeyByIndex(I),
               ADoubleDictionary(Source).GetItemByIndex(I));
      end else
      inherited Assign(Source);
  end;

function ADoubleDictionary.GetAsString: String;
var I, L : Integer;
  begin
    L := Count - 1;
    For I := 0 to L do
      begin
        Result := Result + GetKeyByIndex(I) + ':' + FloatToStr(GetItemByIndex(I));
        if I < L then
          Result := Result + ',';
      end;
  end;



{                                                                              }
{ AExtendedDictionary                                                          }
{                                                                              }
function AExtendedDictionary.GetItem(const Key: String): Extended;
  begin
    if LocateItem(Key, Result) < 0 then
      KeyNotFoundError(Key);
  end;

procedure AExtendedDictionary.Assign(const Source: TObject);
var I : Integer;
  begin
    if Source is AExtendedDictionary then
      begin
        Clear;
        For I := 0 to AExtendedDictionary(Source).Count - 1 do
          Add(AExtendedDictionary(Source).GetKeyByIndex(I),
               AExtendedDictionary(Source).GetItemByIndex(I));
      end else
      inherited Assign(Source);
  end;

function AExtendedDictionary.GetAsString: String;
var I, L : Integer;
  begin
    L := Count - 1;
    For I := 0 to L do
      begin
        Result := Result + GetKeyByIndex(I) + ':' + FloatToStr(GetItemByIndex(I));
        if I < L then
          Result := Result + ',';
      end;
  end;



{                                                                              }
{ AStringDictionary                                                            }
{                                                                              }
function AStringDictionary.GetItem(const Key: String): String;
  begin
    if LocateItem(Key, Result) < 0 then
      KeyNotFoundError(Key);
  end;

procedure AStringDictionary.Assign(const Source: TObject);
var I : Integer;
  begin
    if Source is AStringDictionary then
      begin
        Clear;
        For I := 0 to AStringDictionary(Source).Count - 1 do
          Add(AStringDictionary(Source).GetKeyByIndex(I),
               AStringDictionary(Source).GetItemByIndex(I));
      end else
      inherited Assign(Source);
  end;

function AStringDictionary.GetAsString: String;
var I, L : Integer;
  begin
    L := Count - 1;
    For I := 0 to L do
      begin
        Result := Result + GetKeyByIndex(I) + ':' + StrQuote(GetItemByIndex(I));
        if I < L then
          Result := Result + ',';
      end;
  end;

function AStringDictionary.GetItemLength(const Key: String): Integer;
  begin
    Result := Length(GetItem(Key));
  end;

function AStringDictionary.GetTotalLength: Int64;
var I : Integer;
  begin
    Result := 0;
    For I := 0 to Count - 1 do
      Inc(Result, Length(GetItemByIndex(I)));
  end;



{                                                                              }
{ APointerDictionary                                                           }
{                                                                              }
function APointerDictionary.GetItem(const Key: String): Pointer;
  begin
    if LocateItem(Key, Result) < 0 then
      KeyNotFoundError(Key);
  end;

procedure APointerDictionary.Assign(const Source: TObject);
var I : Integer;
  begin
    if Source is APointerDictionary then
      begin
        Clear;
        For I := 0 to APointerDictionary(Source).Count - 1 do
          Add(APointerDictionary(Source).GetKeyByIndex(I),
               APointerDictionary(Source).GetItemByIndex(I));
      end else
      inherited Assign(Source);
  end;

function APointerDictionary.GetAsString: String;
var I, L : Integer;
  begin
    L := Count - 1;
    For I := 0 to L do
      begin
        Result := Result + GetKeyByIndex(I) + ':' + PointerToStr(GetItemByIndex(I));
        if I < L then
          Result := Result + ',';
      end;
  end;



{                                                                              }
{ AObjectDictionary                                                            }
{                                                                              }
function AObjectDictionary.GetItem(const Key: String): TObject;
  begin
    if LocateItem(Key, Result) < 0 then
      KeyNotFoundError(Key);
  end;

function AObjectDictionary.GetAsString: String;
var I, L : Integer;
  begin
    L := Count - 1;
    For I := 0 to L do
      begin
        Result := Result + GetKeyByIndex(I) + ':' + ObjectClassName(GetItemByIndex(I));
        if I < L then
          Result := Result + ',';
      end;
  end;

procedure AObjectDictionary.Clear;
  begin
    if IsItemOwner then
      FreeItems else
      ReleaseItems;
  end;

procedure AObjectDictionary.Assign(const Source: TObject);
var I : Integer;
  begin
    if Source is AObjectDictionary then
      begin
        Clear;
        For I := 0 to AObjectDictionary(Source).Count - 1 do
          Add(AObjectDictionary(Source).GetKeyByIndex(I),
               AObjectDictionary(Source).GetItemByIndex(I));
      end else
      inherited Assign(Source);
  end;



{                                                                              }
{ DICTIONARY IMPLEMENTATIONS                                                   }
{                                                                              }



{ Dictionary helper functions                                                  }
function DictionaryRehashSize(const Count: Integer): Integer;
var L : Integer;
  begin
    L := Count div AverageHashChainSize; // Number of slots
    if L <= 16 then                      // Rehash in powers of 16
      Result := 16 else
    if L <= 256 then
      Result := 256 else
    if L <= 4096 then
      Result := 4096 else
    if L <= 65536 then
      Result := 65536 else
    if L <= 1048576 then
      Result := 1048576 else
    if L <= 16777216 then
      Result := 16777216 else
      Result := 268435456;
  end;

{                                                                              }
{ TLongIntDictionary                                                           }
{                                                                              }
constructor TLongIntDictionary.Create;
  begin
    inherited Create;
    FCaseSensitive := True;
    FDuplicatesAction := ddAccept;
    FAddOnSet := True;
    FKeys := TStringArray.Create;
    FValues := TLongIntArray.Create;
  end;

constructor TLongIntDictionary.CreateEx(const Keys: AStringArray; const Values: ALongIntArray; const KeysCaseSensitive: Boolean; const AddOnSet: Boolean; const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    inherited Create;
    if Assigned(Keys) then
      FKeys := Keys else
      FKeys := TStringArray.Create;
    if Assigned(Values) then
      FValues := Values else
      FValues := TLongIntArray.Create;
    Assert(FKeys.Count = FValues.Count, 'Keys and Values must be equal length');
    FAddOnSet := AddOnSet;
    FDuplicatesAction := DuplicatesAction;
    Rehash;
  end;

destructor TLongIntDictionary.Destroy;
  begin
    FreeAndNil(FValues);
    FreeAndNil(FKeys);
    inherited Destroy;
  end;

function TLongIntDictionary.GetKeysCaseSensitive: Boolean;
  begin
    Result := FCaseSensitive;
  end;

function TLongIntDictionary.GetAddOnSet: Boolean;
  begin
    Result := FAddOnSet;
  end;

procedure TLongIntDictionary.SetAddOnSet(const AddOnSet: Boolean);
  begin
    FAddOnSet := AddOnSet;
  end;

function TLongIntDictionary.GetHashTableSize: Integer;
  begin
    Result := Length(FLookup);
  end;

procedure TLongIntDictionary.Rehash;
var I, C, L : Integer;
  begin
    C := FKeys.Count;
    L := DictionaryRehashSize(C);
    FLookup := nil;
    SetLength(FLookup, L);
    For I := 0 to C - 1 do
      Append(FLookup[HashStr(FKeys[I], L, FCaseSensitive)], I);
  end;

function TLongIntDictionary.LocateKey(const Key: String; var LookupIdx: Integer; const ErrorIfNotFound: Boolean): Integer;
var H, I, J, L : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L > 0 then
      begin
        H := HashStr(Key, L, FCaseSensitive);
        LookupIdx := H;
        For I := 0 to Length(FLookup[H]) - 1 do
          begin
            J := FLookup[H, I];
            if StrEqual(Key, FKeys[J], FCaseSensitive) then
              begin
                Result := J;
                break;
              end;
          end;
      end;
    if ErrorIfNotFound and (Result = -1) then
      KeyNotFoundError(Key);
  end;

function TLongIntDictionary.KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
var H : Integer;
  begin
    Result := LocateKey(Key, H, ErrorIfNotFound);
  end;

procedure TLongIntDictionary.Add(const Key: String; const Value: LongInt);
var H, L, I : Integer;
  begin
    if FDuplicatesAction in [ddIgnore, ddError] then
      if LocateKey(Key, H, False) >= 0 then
        if FDuplicatesAction = ddIgnore then
          exit else
          DictionaryError('Duplicate key: ' + StrQuote(Key));
    L := Length(FLookup);
    if L = 0 then
      begin
        Rehash;
        L := Length(FLookup);
      end;
    H := Integer(HashStr(Key, LongWord(L), FCaseSensitive));
    I := FKeys.AddItem(Key);
    Append(FLookup[H], I);
    FValues.AddItem(Value);

    if (I + 1) div AverageHashChainSize > L then
      Rehash;
  end;

procedure TLongIntDictionary.DeleteByIndex(const Idx: Integer; const Hash: Integer);
var I, J, H : Integer;
  begin
    if Hash = -1 then
      H := HashStr(FKeys[Idx], Length(FLookup), FCaseSensitive) else
      H := Hash;
    FKeys.Delete(Idx);
    FValues.Delete(Idx);
    J := PosNext(Idx, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);

    For I := 0 to Length(FLookup) - 1 do
      For J := 0 to Length(FLookup[I]) - 1 do
        if FLookup[I][J] > Idx then
          Dec(FLookup[I][J]);
  end;

procedure TLongIntDictionary.Delete(const Key: String);
var I, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    DeleteByIndex(I, H);
  end;

function TLongIntDictionary.HasKey(const Key: String): Boolean;
  begin
    Result := KeyIndex(Key, False) >= 0;
  end;

procedure TLongIntDictionary.Rename(const Key, NewKey: String);
var I, J, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    FKeys[I] := NewKey;
    J := PosNext(I, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);
    Append(FLookup[HashStr(NewKey, Length(FLookup), FCaseSensitive)], I);
  end;

function TLongIntDictionary.GetDuplicatesAction: TDictionaryDuplicatesAction;
  begin
    Result := FDuplicatesAction;
  end;

procedure TLongIntDictionary.SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    FDuplicatesAction := DuplicatesAction;
  end;

function TLongIntDictionary.LocateItem(const Key: String; var Value: LongInt): Integer;
  begin
    Result := KeyIndex(Key, False);
    if Result >= 0 then
      Value := FValues[Result] else
      Value := 0;
  end;

function TLongIntDictionary.LocateNext(const Key: String; const Idx: Integer; var Value: LongInt): Integer;
var L, H, I, J, K : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L = 0 then
      DictionaryError('Item not found');
    H := HashStr(Key, L, FCaseSensitive);
    For I := 0 to Length(FLookup[H]) - 1 do
      begin
        J := FLookup[H, I];
        if J = Idx then
          begin
            if not StrEqual(Key, FKeys[J], FCaseSensitive) then
              DictionaryError('Item not found');
            For K := I + 1 to Length(FLookup[H]) - 1 do
              begin
                J := FLookup[H, K];
                if StrEqual(Key, FKeys[J], FCaseSensitive) then
                  begin
                    Value := FValues[J];
                    Result := J;
                    exit;
                  end;
              end;
            Result := -1;
            exit;
          end;
      end;
    DictionaryError('Item not found');
  end;

procedure TLongIntDictionary.SetItem(const Key: String; const Value: LongInt);
var I : Integer;
  begin
    I := KeyIndex(Key, False);
    if I >= 0 then
      FValues[I] := Value else
      if AddOnSet then
        Add(Key, Value) else
        KeyNotFoundError(Key);
  end;

procedure TLongIntDictionary.IndexError;
  begin
    DictionaryError('Index out of range');
  end;

function TLongIntDictionary.Count: Integer;
  begin
    Result := FKeys.Count;
    Assert(FValues.Count = Result, 'Key/Value count mismatch');
  end;

function TLongIntDictionary.GetKeyByIndex(const Idx: Integer): String;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FKeys.Count) then
      IndexError;
    {$ENDIF}
    Result := FKeys[Idx];
  end;

procedure TLongIntDictionary.DeleteItemByIndex(const Idx: Integer);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    DeleteByIndex(Idx, -1);
  end;

function TLongIntDictionary.GetItemByIndex(const Idx: Integer): LongInt;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    Result := FValues[Idx];
  end;

procedure TLongIntDictionary.SetItemByIndex(const Idx: Integer; const Value: LongInt);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    FValues[Idx] := Value;
  end;

procedure TLongIntDictionary.Clear;
  begin
    FKeys.Clear;
    FValues.Clear;
    Rehash;
  end;



{                                                                              }
{ TLongWordDictionary                                                          }
{                                                                              }
constructor TLongWordDictionary.Create;
  begin
    inherited Create;
    FCaseSensitive := True;
    FDuplicatesAction := ddAccept;
    FAddOnSet := True;
    FKeys := TStringArray.Create;
    FValues := TLongWordArray.Create;
  end;

constructor TLongWordDictionary.CreateEx(const Keys: AStringArray; const Values: ALongWordArray; const KeysCaseSensitive: Boolean; const AddOnSet: Boolean; const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    inherited Create;
    if Assigned(Keys) then
      FKeys := Keys else
      FKeys := TStringArray.Create;
    if Assigned(Values) then
      FValues := Values else
      FValues := TLongWordArray.Create;
    Assert(FKeys.Count = FValues.Count, 'Keys and Values must be equal length');
    FAddOnSet := AddOnSet;
    FDuplicatesAction := DuplicatesAction;
    Rehash;
  end;

destructor TLongWordDictionary.Destroy;
  begin
    FreeAndNil(FValues);
    FreeAndNil(FKeys);
    inherited Destroy;
  end;

function TLongWordDictionary.GetKeysCaseSensitive: Boolean;
  begin
    Result := FCaseSensitive;
  end;

function TLongWordDictionary.GetAddOnSet: Boolean;
  begin
    Result := FAddOnSet;
  end;

procedure TLongWordDictionary.SetAddOnSet(const AddOnSet: Boolean);
  begin
    FAddOnSet := AddOnSet;
  end;

function TLongWordDictionary.GetHashTableSize: Integer;
  begin
    Result := Length(FLookup);
  end;

procedure TLongWordDictionary.Rehash;
var I, C, L : Integer;
  begin
    C := FKeys.Count;
    L := DictionaryRehashSize(C);
    FLookup := nil;
    SetLength(FLookup, L);
    For I := 0 to C - 1 do
      Append(FLookup[HashStr(FKeys[I], L, FCaseSensitive)], I);
  end;

function TLongWordDictionary.LocateKey(const Key: String; var LookupIdx: Integer; const ErrorIfNotFound: Boolean): Integer;
var H, I, J, L : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L > 0 then
      begin
        H := HashStr(Key, L, FCaseSensitive);
        LookupIdx := H;
        For I := 0 to Length(FLookup[H]) - 1 do
          begin
            J := FLookup[H, I];
            if StrEqual(Key, FKeys[J], FCaseSensitive) then
              begin
                Result := J;
                break;
              end;
          end;
      end;
    if ErrorIfNotFound and (Result = -1) then
      KeyNotFoundError(Key);
  end;

function TLongWordDictionary.KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
var H : Integer;
  begin
    Result := LocateKey(Key, H, ErrorIfNotFound);
  end;

procedure TLongWordDictionary.Add(const Key: String; const Value: LongWord);
var H, L, I : Integer;
  begin
    if FDuplicatesAction in [ddIgnore, ddError] then
      if LocateKey(Key, H, False) >= 0 then
        if FDuplicatesAction = ddIgnore then
          exit else
          DictionaryError('Duplicate key: ' + StrQuote(Key));
    L := Length(FLookup);
    if L = 0 then
      begin
        Rehash;
        L := Length(FLookup);
      end;
    H := Integer(HashStr(Key, LongWord(L), FCaseSensitive));
    I := FKeys.AddItem(Key);
    Append(FLookup[H], I);
    FValues.AddItem(Value);

    if (I + 1) div AverageHashChainSize > L then
      Rehash;
  end;

procedure TLongWordDictionary.DeleteByIndex(const Idx: Integer; const Hash: Integer);
var I, J, H : Integer;
  begin
    if Hash = -1 then
      H := HashStr(FKeys[Idx], Length(FLookup), FCaseSensitive) else
      H := Hash;
    FKeys.Delete(Idx);
    FValues.Delete(Idx);
    J := PosNext(Idx, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);

    For I := 0 to Length(FLookup) - 1 do
      For J := 0 to Length(FLookup[I]) - 1 do
        if FLookup[I][J] > Idx then
          Dec(FLookup[I][J]);
  end;

procedure TLongWordDictionary.Delete(const Key: String);
var I, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    DeleteByIndex(I, H);
  end;

function TLongWordDictionary.HasKey(const Key: String): Boolean;
  begin
    Result := KeyIndex(Key, False) >= 0;
  end;

procedure TLongWordDictionary.Rename(const Key, NewKey: String);
var I, J, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    FKeys[I] := NewKey;
    J := PosNext(I, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);
    Append(FLookup[HashStr(NewKey, Length(FLookup), FCaseSensitive)], I);
  end;

function TLongWordDictionary.GetDuplicatesAction: TDictionaryDuplicatesAction;
  begin
    Result := FDuplicatesAction;
  end;

procedure TLongWordDictionary.SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    FDuplicatesAction := DuplicatesAction;
  end;

function TLongWordDictionary.LocateItem(const Key: String; var Value: LongWord): Integer;
  begin
    Result := KeyIndex(Key, False);
    if Result >= 0 then
      Value := FValues[Result] else
      Value := 0;
  end;

function TLongWordDictionary.LocateNext(const Key: String; const Idx: Integer; var Value: LongWord): Integer;
var L, H, I, J, K : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L = 0 then
      DictionaryError('Item not found');
    H := HashStr(Key, L, FCaseSensitive);
    For I := 0 to Length(FLookup[H]) - 1 do
      begin
        J := FLookup[H, I];
        if J = Idx then
          begin
            if not StrEqual(Key, FKeys[J], FCaseSensitive) then
              DictionaryError('Item not found');
            For K := I + 1 to Length(FLookup[H]) - 1 do
              begin
                J := FLookup[H, K];
                if StrEqual(Key, FKeys[J], FCaseSensitive) then
                  begin
                    Value := FValues[J];
                    Result := J;
                    exit;
                  end;
              end;
            Result := -1;
            exit;
          end;
      end;
    DictionaryError('Item not found');
  end;

procedure TLongWordDictionary.SetItem(const Key: String; const Value: LongWord);
var I : Integer;
  begin
    I := KeyIndex(Key, False);
    if I >= 0 then
      FValues[I] := Value else
      if AddOnSet then
        Add(Key, Value) else
        KeyNotFoundError(Key);
  end;

procedure TLongWordDictionary.IndexError;
  begin
    DictionaryError('Index out of range');
  end;

function TLongWordDictionary.Count: Integer;
  begin
    Result := FKeys.Count;
    Assert(FValues.Count = Result, 'Key/Value count mismatch');
  end;

function TLongWordDictionary.GetKeyByIndex(const Idx: Integer): String;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FKeys.Count) then
      IndexError;
    {$ENDIF}
    Result := FKeys[Idx];
  end;

procedure TLongWordDictionary.DeleteItemByIndex(const Idx: Integer);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    DeleteByIndex(Idx, -1);
  end;

function TLongWordDictionary.GetItemByIndex(const Idx: Integer): LongWord;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    Result := FValues[Idx];
  end;

procedure TLongWordDictionary.SetItemByIndex(const Idx: Integer; const Value: LongWord);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    FValues[Idx] := Value;
  end;

procedure TLongWordDictionary.Clear;
  begin
    FKeys.Clear;
    FValues.Clear;
    Rehash;
  end;



{                                                                              }
{ TInt64Dictionary                                                             }
{                                                                              }
constructor TInt64Dictionary.Create;
  begin
    inherited Create;
    FCaseSensitive := True;
    FDuplicatesAction := ddAccept;
    FAddOnSet := True;
    FKeys := TStringArray.Create;
    FValues := TInt64Array.Create;
  end;

constructor TInt64Dictionary.CreateEx(const Keys: AStringArray; const Values: AInt64Array; const KeysCaseSensitive: Boolean; const AddOnSet: Boolean; const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    inherited Create;
    if Assigned(Keys) then
      FKeys := Keys else
      FKeys := TStringArray.Create;
    if Assigned(Values) then
      FValues := Values else
      FValues := TInt64Array.Create;
    Assert(FKeys.Count = FValues.Count, 'Keys and Values must be equal length');
    FAddOnSet := AddOnSet;
    FDuplicatesAction := DuplicatesAction;
    Rehash;
  end;

destructor TInt64Dictionary.Destroy;
  begin
    FreeAndNil(FValues);
    FreeAndNil(FKeys);
    inherited Destroy;
  end;

function TInt64Dictionary.GetKeysCaseSensitive: Boolean;
  begin
    Result := FCaseSensitive;
  end;

function TInt64Dictionary.GetAddOnSet: Boolean;
  begin
    Result := FAddOnSet;
  end;

procedure TInt64Dictionary.SetAddOnSet(const AddOnSet: Boolean);
  begin
    FAddOnSet := AddOnSet;
  end;

function TInt64Dictionary.GetHashTableSize: Integer;
  begin
    Result := Length(FLookup);
  end;

procedure TInt64Dictionary.Rehash;
var I, C, L : Integer;
  begin
    C := FKeys.Count;
    L := DictionaryRehashSize(C);
    FLookup := nil;
    SetLength(FLookup, L);
    For I := 0 to C - 1 do
      Append(FLookup[HashStr(FKeys[I], L, FCaseSensitive)], I);
  end;

function TInt64Dictionary.LocateKey(const Key: String; var LookupIdx: Integer; const ErrorIfNotFound: Boolean): Integer;
var H, I, J, L : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L > 0 then
      begin
        H := HashStr(Key, L, FCaseSensitive);
        LookupIdx := H;
        For I := 0 to Length(FLookup[H]) - 1 do
          begin
            J := FLookup[H, I];
            if StrEqual(Key, FKeys[J], FCaseSensitive) then
              begin
                Result := J;
                break;
              end;
          end;
      end;
    if ErrorIfNotFound and (Result = -1) then
      KeyNotFoundError(Key);
  end;

function TInt64Dictionary.KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
var H : Integer;
  begin
    Result := LocateKey(Key, H, ErrorIfNotFound);
  end;

procedure TInt64Dictionary.Add(const Key: String; const Value: Int64);
var H, L, I : Integer;
  begin
    if FDuplicatesAction in [ddIgnore, ddError] then
      if LocateKey(Key, H, False) >= 0 then
        if FDuplicatesAction = ddIgnore then
          exit else
          DictionaryError('Duplicate key: ' + StrQuote(Key));
    L := Length(FLookup);
    if L = 0 then
      begin
        Rehash;
        L := Length(FLookup);
      end;
    H := Integer(HashStr(Key, LongWord(L), FCaseSensitive));
    I := FKeys.AddItem(Key);
    Append(FLookup[H], I);
    FValues.AddItem(Value);

    if (I + 1) div AverageHashChainSize > L then
      Rehash;
  end;

procedure TInt64Dictionary.DeleteByIndex(const Idx: Integer; const Hash: Integer);
var I, J, H : Integer;
  begin
    if Hash = -1 then
      H := HashStr(FKeys[Idx], Length(FLookup), FCaseSensitive) else
      H := Hash;
    FKeys.Delete(Idx);
    FValues.Delete(Idx);
    J := PosNext(Idx, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);

    For I := 0 to Length(FLookup) - 1 do
      For J := 0 to Length(FLookup[I]) - 1 do
        if FLookup[I][J] > Idx then
          Dec(FLookup[I][J]);
  end;

procedure TInt64Dictionary.Delete(const Key: String);
var I, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    DeleteByIndex(I, H);
  end;

function TInt64Dictionary.HasKey(const Key: String): Boolean;
  begin
    Result := KeyIndex(Key, False) >= 0;
  end;

procedure TInt64Dictionary.Rename(const Key, NewKey: String);
var I, J, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    FKeys[I] := NewKey;
    J := PosNext(I, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);
    Append(FLookup[HashStr(NewKey, Length(FLookup), FCaseSensitive)], I);
  end;

function TInt64Dictionary.GetDuplicatesAction: TDictionaryDuplicatesAction;
  begin
    Result := FDuplicatesAction;
  end;

procedure TInt64Dictionary.SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    FDuplicatesAction := DuplicatesAction;
  end;

function TInt64Dictionary.LocateItem(const Key: String; var Value: Int64): Integer;
  begin
    Result := KeyIndex(Key, False);
    if Result >= 0 then
      Value := FValues[Result] else
      Value := 0;
  end;

function TInt64Dictionary.LocateNext(const Key: String; const Idx: Integer; var Value: Int64): Integer;
var L, H, I, J, K : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L = 0 then
      DictionaryError('Item not found');
    H := HashStr(Key, L, FCaseSensitive);
    For I := 0 to Length(FLookup[H]) - 1 do
      begin
        J := FLookup[H, I];
        if J = Idx then
          begin
            if not StrEqual(Key, FKeys[J], FCaseSensitive) then
              DictionaryError('Item not found');
            For K := I + 1 to Length(FLookup[H]) - 1 do
              begin
                J := FLookup[H, K];
                if StrEqual(Key, FKeys[J], FCaseSensitive) then
                  begin
                    Value := FValues[J];
                    Result := J;
                    exit;
                  end;
              end;
            Result := -1;
            exit;
          end;
      end;
    DictionaryError('Item not found');
  end;

procedure TInt64Dictionary.SetItem(const Key: String; const Value: Int64);
var I : Integer;
  begin
    I := KeyIndex(Key, False);
    if I >= 0 then
      FValues[I] := Value else
      if AddOnSet then
        Add(Key, Value) else
        KeyNotFoundError(Key);
  end;

procedure TInt64Dictionary.IndexError;
  begin
    DictionaryError('Index out of range');
  end;

function TInt64Dictionary.Count: Integer;
  begin
    Result := FKeys.Count;
    Assert(FValues.Count = Result, 'Key/Value count mismatch');
  end;

function TInt64Dictionary.GetKeyByIndex(const Idx: Integer): String;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FKeys.Count) then
      IndexError;
    {$ENDIF}
    Result := FKeys[Idx];
  end;

procedure TInt64Dictionary.DeleteItemByIndex(const Idx: Integer);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    DeleteByIndex(Idx, -1);
  end;

function TInt64Dictionary.GetItemByIndex(const Idx: Integer): Int64;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    Result := FValues[Idx];
  end;

procedure TInt64Dictionary.SetItemByIndex(const Idx: Integer; const Value: Int64);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    FValues[Idx] := Value;
  end;

procedure TInt64Dictionary.Clear;
  begin
    FKeys.Clear;
    FValues.Clear;
    Rehash;
  end;



{                                                                              }
{ TSingleDictionary                                                            }
{                                                                              }
constructor TSingleDictionary.Create;
  begin
    inherited Create;
    FCaseSensitive := True;
    FDuplicatesAction := ddAccept;
    FAddOnSet := True;
    FKeys := TStringArray.Create;
    FValues := TSingleArray.Create;
  end;

constructor TSingleDictionary.CreateEx(const Keys: AStringArray; const Values: ASingleArray; const KeysCaseSensitive: Boolean; const AddOnSet: Boolean; const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    inherited Create;
    if Assigned(Keys) then
      FKeys := Keys else
      FKeys := TStringArray.Create;
    if Assigned(Values) then
      FValues := Values else
      FValues := TSingleArray.Create;
    Assert(FKeys.Count = FValues.Count, 'Keys and Values must be equal length');
    FAddOnSet := AddOnSet;
    FDuplicatesAction := DuplicatesAction;
    Rehash;
  end;

destructor TSingleDictionary.Destroy;
  begin
    FreeAndNil(FValues);
    FreeAndNil(FKeys);
    inherited Destroy;
  end;

function TSingleDictionary.GetKeysCaseSensitive: Boolean;
  begin
    Result := FCaseSensitive;
  end;

function TSingleDictionary.GetAddOnSet: Boolean;
  begin
    Result := FAddOnSet;
  end;

procedure TSingleDictionary.SetAddOnSet(const AddOnSet: Boolean);
  begin
    FAddOnSet := AddOnSet;
  end;

function TSingleDictionary.GetHashTableSize: Integer;
  begin
    Result := Length(FLookup);
  end;

procedure TSingleDictionary.Rehash;
var I, C, L : Integer;
  begin
    C := FKeys.Count;
    L := DictionaryRehashSize(C);
    FLookup := nil;
    SetLength(FLookup, L);
    For I := 0 to C - 1 do
      Append(FLookup[HashStr(FKeys[I], L, FCaseSensitive)], I);
  end;

function TSingleDictionary.LocateKey(const Key: String; var LookupIdx: Integer; const ErrorIfNotFound: Boolean): Integer;
var H, I, J, L : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L > 0 then
      begin
        H := HashStr(Key, L, FCaseSensitive);
        LookupIdx := H;
        For I := 0 to Length(FLookup[H]) - 1 do
          begin
            J := FLookup[H, I];
            if StrEqual(Key, FKeys[J], FCaseSensitive) then
              begin
                Result := J;
                break;
              end;
          end;
      end;
    if ErrorIfNotFound and (Result = -1) then
      KeyNotFoundError(Key);
  end;

function TSingleDictionary.KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
var H : Integer;
  begin
    Result := LocateKey(Key, H, ErrorIfNotFound);
  end;

procedure TSingleDictionary.Add(const Key: String; const Value: Single);
var H, L, I : Integer;
  begin
    if FDuplicatesAction in [ddIgnore, ddError] then
      if LocateKey(Key, H, False) >= 0 then
        if FDuplicatesAction = ddIgnore then
          exit else
          DictionaryError('Duplicate key: ' + StrQuote(Key));
    L := Length(FLookup);
    if L = 0 then
      begin
        Rehash;
        L := Length(FLookup);
      end;
    H := Integer(HashStr(Key, LongWord(L), FCaseSensitive));
    I := FKeys.AddItem(Key);
    Append(FLookup[H], I);
    FValues.AddItem(Value);

    if (I + 1) div AverageHashChainSize > L then
      Rehash;
  end;

procedure TSingleDictionary.DeleteByIndex(const Idx: Integer; const Hash: Integer);
var I, J, H : Integer;
  begin
    if Hash = -1 then
      H := HashStr(FKeys[Idx], Length(FLookup), FCaseSensitive) else
      H := Hash;
    FKeys.Delete(Idx);
    FValues.Delete(Idx);
    J := PosNext(Idx, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);

    For I := 0 to Length(FLookup) - 1 do
      For J := 0 to Length(FLookup[I]) - 1 do
        if FLookup[I][J] > Idx then
          Dec(FLookup[I][J]);
  end;

procedure TSingleDictionary.Delete(const Key: String);
var I, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    DeleteByIndex(I, H);
  end;

function TSingleDictionary.HasKey(const Key: String): Boolean;
  begin
    Result := KeyIndex(Key, False) >= 0;
  end;

procedure TSingleDictionary.Rename(const Key, NewKey: String);
var I, J, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    FKeys[I] := NewKey;
    J := PosNext(I, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);
    Append(FLookup[HashStr(NewKey, Length(FLookup), FCaseSensitive)], I);
  end;

function TSingleDictionary.GetDuplicatesAction: TDictionaryDuplicatesAction;
  begin
    Result := FDuplicatesAction;
  end;

procedure TSingleDictionary.SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    FDuplicatesAction := DuplicatesAction;
  end;

function TSingleDictionary.LocateItem(const Key: String; var Value: Single): Integer;
  begin
    Result := KeyIndex(Key, False);
    if Result >= 0 then
      Value := FValues[Result] else
      Value := 0.0;
  end;

function TSingleDictionary.LocateNext(const Key: String; const Idx: Integer; var Value: Single): Integer;
var L, H, I, J, K : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L = 0 then
      DictionaryError('Item not found');
    H := HashStr(Key, L, FCaseSensitive);
    For I := 0 to Length(FLookup[H]) - 1 do
      begin
        J := FLookup[H, I];
        if J = Idx then
          begin
            if not StrEqual(Key, FKeys[J], FCaseSensitive) then
              DictionaryError('Item not found');
            For K := I + 1 to Length(FLookup[H]) - 1 do
              begin
                J := FLookup[H, K];
                if StrEqual(Key, FKeys[J], FCaseSensitive) then
                  begin
                    Value := FValues[J];
                    Result := J;
                    exit;
                  end;
              end;
            Result := -1;
            exit;
          end;
      end;
    DictionaryError('Item not found');
  end;

procedure TSingleDictionary.SetItem(const Key: String; const Value: Single);
var I : Integer;
  begin
    I := KeyIndex(Key, False);
    if I >= 0 then
      FValues[I] := Value else
      if AddOnSet then
        Add(Key, Value) else
        KeyNotFoundError(Key);
  end;

procedure TSingleDictionary.IndexError;
  begin
    DictionaryError('Index out of range');
  end;

function TSingleDictionary.Count: Integer;
  begin
    Result := FKeys.Count;
    Assert(FValues.Count = Result, 'Key/Value count mismatch');
  end;

function TSingleDictionary.GetKeyByIndex(const Idx: Integer): String;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FKeys.Count) then
      IndexError;
    {$ENDIF}
    Result := FKeys[Idx];
  end;

procedure TSingleDictionary.DeleteItemByIndex(const Idx: Integer);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    DeleteByIndex(Idx, -1);
  end;

function TSingleDictionary.GetItemByIndex(const Idx: Integer): Single;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    Result := FValues[Idx];
  end;

procedure TSingleDictionary.SetItemByIndex(const Idx: Integer; const Value: Single);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    FValues[Idx] := Value;
  end;

procedure TSingleDictionary.Clear;
  begin
    FKeys.Clear;
    FValues.Clear;
    Rehash;
  end;



{                                                                              }
{ TDoubleDictionary                                                            }
{                                                                              }
constructor TDoubleDictionary.Create;
  begin
    inherited Create;
    FCaseSensitive := True;
    FDuplicatesAction := ddAccept;
    FAddOnSet := True;
    FKeys := TStringArray.Create;
    FValues := TDoubleArray.Create;
  end;

constructor TDoubleDictionary.CreateEx(const Keys: AStringArray; const Values: ADoubleArray; const KeysCaseSensitive: Boolean; const AddOnSet: Boolean; const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    inherited Create;
    if Assigned(Keys) then
      FKeys := Keys else
      FKeys := TStringArray.Create;
    if Assigned(Values) then
      FValues := Values else
      FValues := TDoubleArray.Create;
    Assert(FKeys.Count = FValues.Count, 'Keys and Values must be equal length');
    FAddOnSet := AddOnSet;
    FDuplicatesAction := DuplicatesAction;
    Rehash;
  end;

destructor TDoubleDictionary.Destroy;
  begin
    FreeAndNil(FValues);
    FreeAndNil(FKeys);
    inherited Destroy;
  end;

function TDoubleDictionary.GetKeysCaseSensitive: Boolean;
  begin
    Result := FCaseSensitive;
  end;

function TDoubleDictionary.GetAddOnSet: Boolean;
  begin
    Result := FAddOnSet;
  end;

procedure TDoubleDictionary.SetAddOnSet(const AddOnSet: Boolean);
  begin
    FAddOnSet := AddOnSet;
  end;

function TDoubleDictionary.GetHashTableSize: Integer;
  begin
    Result := Length(FLookup);
  end;

procedure TDoubleDictionary.Rehash;
var I, C, L : Integer;
  begin
    C := FKeys.Count;
    L := DictionaryRehashSize(C);
    FLookup := nil;
    SetLength(FLookup, L);
    For I := 0 to C - 1 do
      Append(FLookup[HashStr(FKeys[I], L, FCaseSensitive)], I);
  end;

function TDoubleDictionary.LocateKey(const Key: String; var LookupIdx: Integer; const ErrorIfNotFound: Boolean): Integer;
var H, I, J, L : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L > 0 then
      begin
        H := HashStr(Key, L, FCaseSensitive);
        LookupIdx := H;
        For I := 0 to Length(FLookup[H]) - 1 do
          begin
            J := FLookup[H, I];
            if StrEqual(Key, FKeys[J], FCaseSensitive) then
              begin
                Result := J;
                break;
              end;
          end;
      end;
    if ErrorIfNotFound and (Result = -1) then
      KeyNotFoundError(Key);
  end;

function TDoubleDictionary.KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
var H : Integer;
  begin
    Result := LocateKey(Key, H, ErrorIfNotFound);
  end;

procedure TDoubleDictionary.Add(const Key: String; const Value: Double);
var H, L, I : Integer;
  begin
    if FDuplicatesAction in [ddIgnore, ddError] then
      if LocateKey(Key, H, False) >= 0 then
        if FDuplicatesAction = ddIgnore then
          exit else
          DictionaryError('Duplicate key: ' + StrQuote(Key));
    L := Length(FLookup);
    if L = 0 then
      begin
        Rehash;
        L := Length(FLookup);
      end;
    H := Integer(HashStr(Key, LongWord(L), FCaseSensitive));
    I := FKeys.AddItem(Key);
    Append(FLookup[H], I);
    FValues.AddItem(Value);

    if (I + 1) div AverageHashChainSize > L then
      Rehash;
  end;

procedure TDoubleDictionary.DeleteByIndex(const Idx: Integer; const Hash: Integer);
var I, J, H : Integer;
  begin
    if Hash = -1 then
      H := HashStr(FKeys[Idx], Length(FLookup), FCaseSensitive) else
      H := Hash;
    FKeys.Delete(Idx);
    FValues.Delete(Idx);
    J := PosNext(Idx, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);

    For I := 0 to Length(FLookup) - 1 do
      For J := 0 to Length(FLookup[I]) - 1 do
        if FLookup[I][J] > Idx then
          Dec(FLookup[I][J]);
  end;

procedure TDoubleDictionary.Delete(const Key: String);
var I, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    DeleteByIndex(I, H);
  end;

function TDoubleDictionary.HasKey(const Key: String): Boolean;
  begin
    Result := KeyIndex(Key, False) >= 0;
  end;

procedure TDoubleDictionary.Rename(const Key, NewKey: String);
var I, J, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    FKeys[I] := NewKey;
    J := PosNext(I, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);
    Append(FLookup[HashStr(NewKey, Length(FLookup), FCaseSensitive)], I);
  end;

function TDoubleDictionary.GetDuplicatesAction: TDictionaryDuplicatesAction;
  begin
    Result := FDuplicatesAction;
  end;

procedure TDoubleDictionary.SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    FDuplicatesAction := DuplicatesAction;
  end;

function TDoubleDictionary.LocateItem(const Key: String; var Value: Double): Integer;
  begin
    Result := KeyIndex(Key, False);
    if Result >= 0 then
      Value := FValues[Result] else
      Value := 0.0;
  end;

function TDoubleDictionary.LocateNext(const Key: String; const Idx: Integer; var Value: Double): Integer;
var L, H, I, J, K : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L = 0 then
      DictionaryError('Item not found');
    H := HashStr(Key, L, FCaseSensitive);
    For I := 0 to Length(FLookup[H]) - 1 do
      begin
        J := FLookup[H, I];
        if J = Idx then
          begin
            if not StrEqual(Key, FKeys[J], FCaseSensitive) then
              DictionaryError('Item not found');
            For K := I + 1 to Length(FLookup[H]) - 1 do
              begin
                J := FLookup[H, K];
                if StrEqual(Key, FKeys[J], FCaseSensitive) then
                  begin
                    Value := FValues[J];
                    Result := J;
                    exit;
                  end;
              end;
            Result := -1;
            exit;
          end;
      end;
    DictionaryError('Item not found');
  end;

procedure TDoubleDictionary.SetItem(const Key: String; const Value: Double);
var I : Integer;
  begin
    I := KeyIndex(Key, False);
    if I >= 0 then
      FValues[I] := Value else
      if AddOnSet then
        Add(Key, Value) else
        KeyNotFoundError(Key);
  end;

procedure TDoubleDictionary.IndexError;
  begin
    DictionaryError('Index out of range');
  end;

function TDoubleDictionary.Count: Integer;
  begin
    Result := FKeys.Count;
    Assert(FValues.Count = Result, 'Key/Value count mismatch');
  end;

function TDoubleDictionary.GetKeyByIndex(const Idx: Integer): String;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FKeys.Count) then
      IndexError;
    {$ENDIF}
    Result := FKeys[Idx];
  end;

procedure TDoubleDictionary.DeleteItemByIndex(const Idx: Integer);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    DeleteByIndex(Idx, -1);
  end;

function TDoubleDictionary.GetItemByIndex(const Idx: Integer): Double;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    Result := FValues[Idx];
  end;

procedure TDoubleDictionary.SetItemByIndex(const Idx: Integer; const Value: Double);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    FValues[Idx] := Value;
  end;

procedure TDoubleDictionary.Clear;
  begin
    FKeys.Clear;
    FValues.Clear;
    Rehash;
  end;



{                                                                              }
{ TExtendedDictionary                                                          }
{                                                                              }
constructor TExtendedDictionary.Create;
  begin
    inherited Create;
    FCaseSensitive := True;
    FDuplicatesAction := ddAccept;
    FAddOnSet := True;
    FKeys := TStringArray.Create;
    FValues := TExtendedArray.Create;
  end;

constructor TExtendedDictionary.CreateEx(const Keys: AStringArray; const Values: AExtendedArray; const KeysCaseSensitive: Boolean; const AddOnSet: Boolean; const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    inherited Create;
    if Assigned(Keys) then
      FKeys := Keys else
      FKeys := TStringArray.Create;
    if Assigned(Values) then
      FValues := Values else
      FValues := TExtendedArray.Create;
    Assert(FKeys.Count = FValues.Count, 'Keys and Values must be equal length');
    FAddOnSet := AddOnSet;
    FDuplicatesAction := DuplicatesAction;
    Rehash;
  end;

destructor TExtendedDictionary.Destroy;
  begin
    FreeAndNil(FValues);
    FreeAndNil(FKeys);
    inherited Destroy;
  end;

function TExtendedDictionary.GetKeysCaseSensitive: Boolean;
  begin
    Result := FCaseSensitive;
  end;

function TExtendedDictionary.GetAddOnSet: Boolean;
  begin
    Result := FAddOnSet;
  end;

procedure TExtendedDictionary.SetAddOnSet(const AddOnSet: Boolean);
  begin
    FAddOnSet := AddOnSet;
  end;

function TExtendedDictionary.GetHashTableSize: Integer;
  begin
    Result := Length(FLookup);
  end;

procedure TExtendedDictionary.Rehash;
var I, C, L : Integer;
  begin
    C := FKeys.Count;
    L := DictionaryRehashSize(C);
    FLookup := nil;
    SetLength(FLookup, L);
    For I := 0 to C - 1 do
      Append(FLookup[HashStr(FKeys[I], L, FCaseSensitive)], I);
  end;

function TExtendedDictionary.LocateKey(const Key: String; var LookupIdx: Integer; const ErrorIfNotFound: Boolean): Integer;
var H, I, J, L : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L > 0 then
      begin
        H := HashStr(Key, L, FCaseSensitive);
        LookupIdx := H;
        For I := 0 to Length(FLookup[H]) - 1 do
          begin
            J := FLookup[H, I];
            if StrEqual(Key, FKeys[J], FCaseSensitive) then
              begin
                Result := J;
                break;
              end;
          end;
      end;
    if ErrorIfNotFound and (Result = -1) then
      KeyNotFoundError(Key);
  end;

function TExtendedDictionary.KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
var H : Integer;
  begin
    Result := LocateKey(Key, H, ErrorIfNotFound);
  end;

procedure TExtendedDictionary.Add(const Key: String; const Value: Extended);
var H, L, I : Integer;
  begin
    if FDuplicatesAction in [ddIgnore, ddError] then
      if LocateKey(Key, H, False) >= 0 then
        if FDuplicatesAction = ddIgnore then
          exit else
          DictionaryError('Duplicate key: ' + StrQuote(Key));
    L := Length(FLookup);
    if L = 0 then
      begin
        Rehash;
        L := Length(FLookup);
      end;
    H := Integer(HashStr(Key, LongWord(L), FCaseSensitive));
    I := FKeys.AddItem(Key);
    Append(FLookup[H], I);
    FValues.AddItem(Value);

    if (I + 1) div AverageHashChainSize > L then
      Rehash;
  end;

procedure TExtendedDictionary.DeleteByIndex(const Idx: Integer; const Hash: Integer);
var I, J, H : Integer;
  begin
    if Hash = -1 then
      H := HashStr(FKeys[Idx], Length(FLookup), FCaseSensitive) else
      H := Hash;
    FKeys.Delete(Idx);
    FValues.Delete(Idx);
    J := PosNext(Idx, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);

    For I := 0 to Length(FLookup) - 1 do
      For J := 0 to Length(FLookup[I]) - 1 do
        if FLookup[I][J] > Idx then
          Dec(FLookup[I][J]);
  end;

procedure TExtendedDictionary.Delete(const Key: String);
var I, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    DeleteByIndex(I, H);
  end;

function TExtendedDictionary.HasKey(const Key: String): Boolean;
  begin
    Result := KeyIndex(Key, False) >= 0;
  end;

procedure TExtendedDictionary.Rename(const Key, NewKey: String);
var I, J, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    FKeys[I] := NewKey;
    J := PosNext(I, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);
    Append(FLookup[HashStr(NewKey, Length(FLookup), FCaseSensitive)], I);
  end;

function TExtendedDictionary.GetDuplicatesAction: TDictionaryDuplicatesAction;
  begin
    Result := FDuplicatesAction;
  end;

procedure TExtendedDictionary.SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    FDuplicatesAction := DuplicatesAction;
  end;

function TExtendedDictionary.LocateItem(const Key: String; var Value: Extended): Integer;
  begin
    Result := KeyIndex(Key, False);
    if Result >= 0 then
      Value := FValues[Result] else
      Value := 0.0;
  end;

function TExtendedDictionary.LocateNext(const Key: String; const Idx: Integer; var Value: Extended): Integer;
var L, H, I, J, K : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L = 0 then
      DictionaryError('Item not found');
    H := HashStr(Key, L, FCaseSensitive);
    For I := 0 to Length(FLookup[H]) - 1 do
      begin
        J := FLookup[H, I];
        if J = Idx then
          begin
            if not StrEqual(Key, FKeys[J], FCaseSensitive) then
              DictionaryError('Item not found');
            For K := I + 1 to Length(FLookup[H]) - 1 do
              begin
                J := FLookup[H, K];
                if StrEqual(Key, FKeys[J], FCaseSensitive) then
                  begin
                    Value := FValues[J];
                    Result := J;
                    exit;
                  end;
              end;
            Result := -1;
            exit;
          end;
      end;
    DictionaryError('Item not found');
  end;

procedure TExtendedDictionary.SetItem(const Key: String; const Value: Extended);
var I : Integer;
  begin
    I := KeyIndex(Key, False);
    if I >= 0 then
      FValues[I] := Value else
      if AddOnSet then
        Add(Key, Value) else
        KeyNotFoundError(Key);
  end;

procedure TExtendedDictionary.IndexError;
  begin
    DictionaryError('Index out of range');
  end;

function TExtendedDictionary.Count: Integer;
  begin
    Result := FKeys.Count;
    Assert(FValues.Count = Result, 'Key/Value count mismatch');
  end;

function TExtendedDictionary.GetKeyByIndex(const Idx: Integer): String;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FKeys.Count) then
      IndexError;
    {$ENDIF}
    Result := FKeys[Idx];
  end;

procedure TExtendedDictionary.DeleteItemByIndex(const Idx: Integer);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    DeleteByIndex(Idx, -1);
  end;

function TExtendedDictionary.GetItemByIndex(const Idx: Integer): Extended;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    Result := FValues[Idx];
  end;

procedure TExtendedDictionary.SetItemByIndex(const Idx: Integer; const Value: Extended);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    FValues[Idx] := Value;
  end;

procedure TExtendedDictionary.Clear;
  begin
    FKeys.Clear;
    FValues.Clear;
    Rehash;
  end;



{                                                                              }
{ TStringDictionary                                                            }
{                                                                              }
constructor TStringDictionary.Create;
  begin
    inherited Create;
    FCaseSensitive := True;
    FDuplicatesAction := ddAccept;
    FAddOnSet := True;
    FKeys := TStringArray.Create;
    FValues := TStringArray.Create;
  end;

constructor TStringDictionary.CreateEx(const Keys: AStringArray; const Values: AStringArray; const KeysCaseSensitive: Boolean; const AddOnSet: Boolean; const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    inherited Create;
    if Assigned(Keys) then
      FKeys := Keys else
      FKeys := TStringArray.Create;
    if Assigned(Values) then
      FValues := Values else
      FValues := TStringArray.Create;
    Assert(FKeys.Count = FValues.Count, 'Keys and Values must be equal length');
    FAddOnSet := AddOnSet;
    FDuplicatesAction := DuplicatesAction;
    Rehash;
  end;

destructor TStringDictionary.Destroy;
  begin
    FreeAndNil(FValues);
    FreeAndNil(FKeys);
    inherited Destroy;
  end;

function TStringDictionary.GetKeysCaseSensitive: Boolean;
  begin
    Result := FCaseSensitive;
  end;

function TStringDictionary.GetAddOnSet: Boolean;
  begin
    Result := FAddOnSet;
  end;

procedure TStringDictionary.SetAddOnSet(const AddOnSet: Boolean);
  begin
    FAddOnSet := AddOnSet;
  end;

function TStringDictionary.GetHashTableSize: Integer;
  begin
    Result := Length(FLookup);
  end;

procedure TStringDictionary.Rehash;
var I, C, L : Integer;
  begin
    C := FKeys.Count;
    L := DictionaryRehashSize(C);
    FLookup := nil;
    SetLength(FLookup, L);
    For I := 0 to C - 1 do
      Append(FLookup[HashStr(FKeys[I], L, FCaseSensitive)], I);
  end;

function TStringDictionary.LocateKey(const Key: String; var LookupIdx: Integer; const ErrorIfNotFound: Boolean): Integer;
var H, I, J, L : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L > 0 then
      begin
        H := HashStr(Key, L, FCaseSensitive);
        LookupIdx := H;
        For I := 0 to Length(FLookup[H]) - 1 do
          begin
            J := FLookup[H, I];
            if StrEqual(Key, FKeys[J], FCaseSensitive) then
              begin
                Result := J;
                break;
              end;
          end;
      end;
    if ErrorIfNotFound and (Result = -1) then
      KeyNotFoundError(Key);
  end;

function TStringDictionary.KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
var H : Integer;
  begin
    Result := LocateKey(Key, H, ErrorIfNotFound);
  end;

procedure TStringDictionary.Add(const Key: String; const Value: String);
var H, L, I : Integer;
  begin
    if FDuplicatesAction in [ddIgnore, ddError] then
      if LocateKey(Key, H, False) >= 0 then
        if FDuplicatesAction = ddIgnore then
          exit else
          DictionaryError('Duplicate key: ' + StrQuote(Key));
    L := Length(FLookup);
    if L = 0 then
      begin
        Rehash;
        L := Length(FLookup);
      end;
    H := Integer(HashStr(Key, LongWord(L), FCaseSensitive));
    I := FKeys.AddItem(Key);
    Append(FLookup[H], I);
    FValues.AddItem(Value);

    if (I + 1) div AverageHashChainSize > L then
      Rehash;
  end;

procedure TStringDictionary.DeleteByIndex(const Idx: Integer; const Hash: Integer);
var I, J, H : Integer;
  begin
    if Hash = -1 then
      H := HashStr(FKeys[Idx], Length(FLookup), FCaseSensitive) else
      H := Hash;
    FKeys.Delete(Idx);
    FValues.Delete(Idx);
    J := PosNext(Idx, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);

    For I := 0 to Length(FLookup) - 1 do
      For J := 0 to Length(FLookup[I]) - 1 do
        if FLookup[I][J] > Idx then
          Dec(FLookup[I][J]);
  end;

procedure TStringDictionary.Delete(const Key: String);
var I, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    DeleteByIndex(I, H);
  end;

function TStringDictionary.HasKey(const Key: String): Boolean;
  begin
    Result := KeyIndex(Key, False) >= 0;
  end;

procedure TStringDictionary.Rename(const Key, NewKey: String);
var I, J, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    FKeys[I] := NewKey;
    J := PosNext(I, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);
    Append(FLookup[HashStr(NewKey, Length(FLookup), FCaseSensitive)], I);
  end;

function TStringDictionary.GetDuplicatesAction: TDictionaryDuplicatesAction;
  begin
    Result := FDuplicatesAction;
  end;

procedure TStringDictionary.SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    FDuplicatesAction := DuplicatesAction;
  end;

function TStringDictionary.LocateItem(const Key: String; var Value: String): Integer;
  begin
    Result := KeyIndex(Key, False);
    if Result >= 0 then
      Value := FValues[Result] else
      Value := '';
  end;

function TStringDictionary.LocateNext(const Key: String; const Idx: Integer; var Value: String): Integer;
var L, H, I, J, K : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L = 0 then
      DictionaryError('Item not found');
    H := HashStr(Key, L, FCaseSensitive);
    For I := 0 to Length(FLookup[H]) - 1 do
      begin
        J := FLookup[H, I];
        if J = Idx then
          begin
            if not StrEqual(Key, FKeys[J], FCaseSensitive) then
              DictionaryError('Item not found');
            For K := I + 1 to Length(FLookup[H]) - 1 do
              begin
                J := FLookup[H, K];
                if StrEqual(Key, FKeys[J], FCaseSensitive) then
                  begin
                    Value := FValues[J];
                    Result := J;
                    exit;
                  end;
              end;
            Result := -1;
            exit;
          end;
      end;
    DictionaryError('Item not found');
  end;

procedure TStringDictionary.SetItem(const Key: String; const Value: String);
var I : Integer;
  begin
    I := KeyIndex(Key, False);
    if I >= 0 then
      FValues[I] := Value else
      if AddOnSet then
        Add(Key, Value) else
        KeyNotFoundError(Key);
  end;

procedure TStringDictionary.IndexError;
  begin
    DictionaryError('Index out of range');
  end;

function TStringDictionary.Count: Integer;
  begin
    Result := FKeys.Count;
    Assert(FValues.Count = Result, 'Key/Value count mismatch');
  end;

function TStringDictionary.GetKeyByIndex(const Idx: Integer): String;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FKeys.Count) then
      IndexError;
    {$ENDIF}
    Result := FKeys[Idx];
  end;

procedure TStringDictionary.DeleteItemByIndex(const Idx: Integer);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    DeleteByIndex(Idx, -1);
  end;

function TStringDictionary.GetItemByIndex(const Idx: Integer): String;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    Result := FValues[Idx];
  end;

procedure TStringDictionary.SetItemByIndex(const Idx: Integer; const Value: String);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    FValues[Idx] := Value;
  end;

procedure TStringDictionary.Clear;
  begin
    FKeys.Clear;
    FValues.Clear;
    Rehash;
  end;



{                                                                              }
{ TObjectDictionary                                                            }
{                                                                              }
constructor TObjectDictionary.Create;
  begin
    inherited Create;
    FCaseSensitive := True;
    FDuplicatesAction := ddAccept;
    FAddOnSet := True;
    FKeys := TStringArray.Create;
    FValues := TObjectArray.Create;
  end;

constructor TObjectDictionary.CreateEx(const Keys: AStringArray; const Values: AObjectArray; const IsItemOwner: Boolean; const KeysCaseSensitive: Boolean; const AddOnSet: Boolean; const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    inherited Create;
    if Assigned(Keys) then
      FKeys := Keys else
      FKeys := TStringArray.Create;
    if Assigned(Values) then
      FValues := Values else
      FValues := TObjectArray.Create;
    Assert(FKeys.Count = FValues.Count, 'Keys and Values must be equal length');
    FAddOnSet := AddOnSet;
    FValues.IsItemOwner := IsItemOwner;
    FCaseSensitive := KeysCaseSensitive;
    FDuplicatesAction := DuplicatesAction;
    Rehash;
  end;

destructor TObjectDictionary.Destroy;
  begin
    FreeAndNil(FValues);
    FreeAndNil(FKeys);
    inherited Destroy;
  end;

function TObjectDictionary.GetKeysCaseSensitive: Boolean;
  begin
    Result := FCaseSensitive;
  end;

function TObjectDictionary.GetAddOnSet: Boolean;
  begin
    Result := FAddOnSet;
  end;

procedure TObjectDictionary.SetAddOnSet(const AddOnSet: Boolean);
  begin
    FAddOnSet := AddOnSet;
  end;

function TObjectDictionary.GetHashTableSize: Integer;
  begin
    Result := Length(FLookup);
  end;

function TObjectDictionary.GetIsItemOwner: Boolean;
  begin
    Result := FValues.IsItemOwner;
  end;

procedure TObjectDictionary.SetIsItemOwner(const IsItemOwner: Boolean);
  begin
    FValues.IsItemOwner := IsItemOwner;
  end;

procedure TObjectDictionary.Rehash;
var I, C, L : Integer;
  begin
    C := FKeys.Count;
    L := DictionaryRehashSize(C);
    FLookup := nil;
    SetLength(FLookup, L);
    For I := 0 to C - 1 do
      Append(FLookup[HashStr(FKeys[I], L, FCaseSensitive)], I);
  end;

function TObjectDictionary.LocateKey(const Key: String; var LookupIdx: Integer; const ErrorIfNotFound: Boolean): Integer;
var H, I, J, L : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L > 0 then
      begin
        H := HashStr(Key, L, FCaseSensitive);
        LookupIdx := H;
        For I := 0 to Length(FLookup[H]) - 1 do
          begin
            J := FLookup[H, I];
            if StrEqual(Key, FKeys[J], FCaseSensitive) then
              begin
                Result := J;
                break;
              end;
          end;
      end;
    if ErrorIfNotFound and (Result = -1) then
      KeyNotFoundError(Key);
  end;

function TObjectDictionary.KeyIndex(const Key: String; const ErrorIfNotFound: Boolean): Integer;
var H : Integer;
  begin
    Result := LocateKey(Key, H, ErrorIfNotFound);
  end;

procedure TObjectDictionary.Add(const Key: String; const Value: TObject);
var H, L, I : Integer;
  begin
    if FDuplicatesAction in [ddIgnore, ddError] then
      if LocateKey(Key, H, False) >= 0 then
        if FDuplicatesAction = ddIgnore then
          exit else
          DictionaryError('Duplicate key: ' + StrQuote(Key));
    L := Length(FLookup);
    if L = 0 then
      begin
        Rehash;
        L := Length(FLookup);
      end;
    H := Integer(HashStr(Key, LongWord(L), FCaseSensitive));
    I := FKeys.AddItem(Key);
    Append(FLookup[H], I);
    FValues.AddItem(Value);

    if (I + 1) div AverageHashChainSize > L then
      Rehash;
  end;

procedure TObjectDictionary.DeleteByIndex(const Idx: Integer; const Hash: Integer);
var I, J, H : Integer;
  begin
    if Hash = -1 then
      H := HashStr(FKeys[Idx], Length(FLookup), FCaseSensitive) else
      H := Hash;
    FKeys.Delete(Idx);
    FValues.Delete(Idx);
    J := PosNext(Idx, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);

    For I := 0 to Length(FLookup) - 1 do
      For J := 0 to Length(FLookup[I]) - 1 do
        if FLookup[I][J] > Idx then
          Dec(FLookup[I][J]);
  end;

procedure TObjectDictionary.Delete(const Key: String);
var I, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    DeleteByIndex(I, H);
  end;

function TObjectDictionary.HasKey(const Key: String): Boolean;
  begin
    Result := KeyIndex(Key, False) >= 0;
  end;

procedure TObjectDictionary.Rename(const Key, NewKey: String);
var I, J, H : Integer;
  begin
    I := LocateKey(Key, H, True);
    FKeys[I] := NewKey;
    J := PosNext(I, FLookup[H]);
    Assert(J >= 0, 'Invalid hash value/lookup table');
    Remove(FLookup[H], J, 1);
    Append(FLookup[HashStr(NewKey, Length(FLookup), FCaseSensitive)], I);
  end;

function TObjectDictionary.GetDuplicatesAction: TDictionaryDuplicatesAction;
  begin
    Result := FDuplicatesAction;
  end;

procedure TObjectDictionary.SetDuplicatesAction(const DuplicatesAction: TDictionaryDuplicatesAction);
  begin
    FDuplicatesAction := DuplicatesAction;
  end;

function TObjectDictionary.LocateItem(const Key: String; var Value: TObject): Integer;
  begin
    Result := KeyIndex(Key, False);
    if Result >= 0 then
      Value := FValues[Result] else
      Value := nil;
  end;

function TObjectDictionary.LocateNext(const Key: String; const Idx: Integer; var Value: TObject): Integer;
var L, H, I, J, K : Integer;
  begin
    Result := -1;
    L := Length(FLookup);
    if L = 0 then
      DictionaryError('Item not found');
    H := HashStr(Key, L, FCaseSensitive);
    For I := 0 to Length(FLookup[H]) - 1 do
      begin
        J := FLookup[H, I];
        if J = Idx then
          begin
            if not StrEqual(Key, FKeys[J], FCaseSensitive) then
              DictionaryError('Item not found');
            For K := I + 1 to Length(FLookup[H]) - 1 do
              begin
                J := FLookup[H, K];
                if StrEqual(Key, FKeys[J], FCaseSensitive) then
                  begin
                    Value := FValues[J];
                    Result := J;
                    exit;
                  end;
              end;
            Result := -1;
            exit;
          end;
      end;
    DictionaryError('Item not found');
  end;

procedure TObjectDictionary.SetItem(const Key: String; const Value: TObject);
var I : Integer;
  begin
    I := KeyIndex(Key, False);
    if I >= 0 then
      FValues[I] := Value else
      if AddOnSet then
        Add(Key, Value) else
        KeyNotFoundError(Key);
  end;

procedure TObjectDictionary.IndexError;
  begin
    DictionaryError('Index out of range');
  end;

function TObjectDictionary.Count: Integer;
  begin
    Result := FKeys.Count;
    Assert(FValues.Count = Result, 'Key/Value count mismatch');
  end;

function TObjectDictionary.GetKeyByIndex(const Idx: Integer): String;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FKeys.Count) then
      IndexError;
    {$ENDIF}
    Result := FKeys[Idx];
  end;

procedure TObjectDictionary.DeleteItemByIndex(const Idx: Integer);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    DeleteByIndex(Idx, -1);
  end;

function TObjectDictionary.GetItemByIndex(const Idx: Integer): TObject;
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    Result := FValues[Idx];
  end;

procedure TObjectDictionary.SetItemByIndex(const Idx: Integer; const Value: TObject);
  begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FValues.Count) then
      IndexError;
    {$ENDIF}
    FValues[Idx] := Value;
  end;

function TObjectDictionary.ReleaseItem(const Key: String): TObject;
var I : Integer;
  begin
    I := KeyIndex(Key, True);
    Result := FValues.ReleaseItem(I);
  end;

procedure TObjectDictionary.ReleaseItems;
  begin
    FKeys.Clear;
    FValues.ReleaseItems;
    Rehash;
  end;

procedure TObjectDictionary.FreeItems;
  begin
    FKeys.Clear;
    FValues.FreeItems;
    Rehash;
  end;

procedure TObjectDictionary.Clear;
  begin
    FKeys.Clear;
    FValues.Clear;
    Rehash;
  end;



{                                                                              }
{ Self testing code                                                            }
{                                                                              }
procedure SelfTest;
var F : TIntegerDictionary;
    G : TStringDictionary;
    I : Integer;
  begin
    F := TIntegerDictionary.Create;
    For I := 0 to 16384 do
      F.Add(IntToStr(I), I);
    Assert(F.Count = 16385, 'Dictionary.Count');
    For I := 0 to 16384 do
      Assert(F.GetKeyByIndex(I) = IntToStr(I), 'Dictionary.GetKeyByIndex');
    Assert(F['0'] = 0, 'Dictionary.GetItem');
    Assert(F['5'] = 5, 'Dictionary.GetItem');
    Assert(F['16384'] = 16384, 'Dictionary.GetItem');
    For I := 0 to 16384 do
      Assert(F.GetItemByIndex(I) = I, 'Dictionary.GetItemByIndex');
    Assert(F.HasKey('5'), 'Dictionary.HasKey');
    Assert(not F.HasKey('X'), 'Dictionary.HasKey');
    F.Rename('5', 'X');
    Assert(not F.HasKey('5'), 'Dictionary.Rename');
    Assert(F.HasKey('X'), 'Dictionary.Rename');
    Assert(F['X'] = 5, 'Dictionary.Rename');
    F.Delete('X');
    Assert(not F.HasKey('X'), 'Dictionary.Delete');
    Assert(F.Count = 16384, 'Dictionary.Delete');
    F.Delete('0');
    Assert(not F.HasKey('0'), 'Dictionary.Delete');
    Assert(F.Count = 16383, 'Dictionary.Delete');
    F.DeleteItemByIndex(0);
    Assert(not F.HasKey('1'), 'Dictionary.DeleteItemByIndex');
    Assert(F.Count = 16382, 'Dictionary.DeleteItemByIndex');
    F.Free;

    G := TStringDictionary.Create;
    For I := 0 to 16384 do
      G.Add(IntToStr(I), IntToStr(I));
    Assert(G.Count = 16385, 'Dictionary.Count');
    For I := 0 to 16384 do
      Assert(G.GetKeyByIndex(I) = IntToStr(I), 'Dictionary.GetKeyByIndex');
    Assert(G['0'] = '0', 'Dictionary.GetItem');
    Assert(G['5'] = '5', 'Dictionary.GetItem');
    Assert(G['16384'] = '16384', 'Dictionary.GetItem');
    For I := 0 to 16384 do
      Assert(G.GetItemByIndex(I) = IntToStr(I), 'Dictionary.GetItemByIndex');
    Assert(G.HasKey('5'), 'Dictionary.HasKey');
    Assert(not G.HasKey('X'), 'Dictionary.HasKey');
    G.Rename('5', 'X');
    Assert(not G.HasKey('5'), 'Dictionary.Rename');
    Assert(G.HasKey('X'), 'Dictionary.Rename');
    Assert(G['X'] = '5', 'Dictionary.Rename');
    G.Delete('X');
    Assert(not G.HasKey('X'), 'Dictionary.Delete');
    Assert(G.Count = 16384, 'Dictionary.Delete');
    G.Delete('0');
    Assert(not G.HasKey('0'), 'Dictionary.Delete');
    Assert(G.Count = 16383, 'Dictionary.Delete');
    G.DeleteItemByIndex(0);
    Assert(not G.HasKey('1'), 'Dictionary.DeleteItemByIndex');
    Assert(G.Count = 16382, 'Dictionary.DeleteItemByIndex');
    G.Free;
  end;



end.

