unit DEditors;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, DB,
  DFilters, ADOBase;

type
  TDConnectionEditor = class(TComponent)
  private
    { Private declarations }
    FConnect  : String;
    FDefTable : String;
  protected
    { Protected declarations }
  public
    { Public declarations }
    function Execute: Boolean;
  published
    { Published declarations }
    property Connection   : String read FConnect  write FConnect;
    property DefaultTable : String read FDefTable write FDefTable;
  end;

{******************************************************************************}

  TDFilterEditor = class(TComponent)
  private
    { Private declarations }
    FField    : TField;
    FDataSet  : TDataSet;
    FMarker   : String;
    FFilter   : TDSubFilters;

    // Handle properties
    procedure SetFilter(Value: TDSubFilters);
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Execute: Boolean;
  published
    { Published declarations }
    property Field     : TField       read FField   write FField;
    property DataSet   : TDataSet     read FDataSet write FDataSet;
    property Marker    : String       read FMarker  write FMarker;
    property SubFilter : TDSubFilters read FFilter  write SetFilter;
  end;

{******************************************************************************}

  TDColorEditor = class(TComponent)
  private
    { Private declarations }
    FDataSet  : TDataSet;
    FColors   : TDFilters;
    FMarker   : String;

  protected
    { Protected declarations }
  public
    { Public declarations }
    function Execute: Boolean;
  published
    { Published declarations }
    property DataSet : TDataSet  read FDataSet write FDataSet;
    property Colors  : TDFilters read FColors  write FColors;
    property Marker  : String    read FMarker  write FMarker;
  end;

{******************************************************************************}

  TDExportEditor = class(TComponent)
  private
    { Private declarations }
    FDataSet : TDataSet;
  protected
    { Protected declarations }
  public
    { Public declarations }
    procedure Execute;
  published
    { Published declarations }
    property DataSet : TDataSet read FDataSet write FDataSet;
  end;

{******************************************************************************}

  TDFillUpMode = (fmNone, fmConst, fmLower, fmUpper, fmSQL);
  TDFillUpEditor = class(TComponent)
  private
    { Private declarations }
    FField    : TField;
    FDataSet  : TDataSet;
    FMode     : TDFillUpMode;
    FText     : String;
    FLines    : TStrings;

  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Execute: Boolean;
  published
    { Published declarations }
    property Field     : TField       read FField   write FField;
    property DataSet   : TDataSet     read FDataSet write FDataSet;
    property Mode      : TDFillUpMode read FMode    write FMode default fmNone;
    property Lines     : TStrings     read FLines;
    property Text      : String       read FText;

  end;

implementation

uses DlgConnectEditor,
     DlgFilter,
     DlgExport,
     DlgFillUp,
     DlgColor;

{ TDConnection Editor }

function TDConnectionEditor.Execute: Boolean;
var
   oEditor : TConnectEditor;
   oConn   : TDConnection;
begin
     oEditor := TConnectEditor.Create(Application);
     try
     begin
          oConn := TDConnection.Create(True);
          try
          begin
               oConn.Connection   := FConnect;
               oEditor.Connection := FConnect;
               oEditor.OdbcFilter := oConn.GetOdbcDriverFilter;
               oConn.GetOdbcDSNList(oEditor.cbxSystem.Items);
               oConn.GetOdbcDriverList(oEditor.cbxDriver.Items);
               oConn.GetADOProviderList(oEditor.cbxProvider.Items, oEditor.Provider);
               Result := (oEditor.ShowModal = mrOk);
               if Result then
               begin
                    FConnect  := oEditor.Connection;
                    FDefTable := oEditor.TableName;
               end
               else FDefTable := '';
          end;
          finally
               oConn.Free;
          end;
     end;
     finally
          oEditor.Free;
     end;
end;


{ TDFilterEditor }

constructor TDFilterEditor.Create(AOwner: TComponent);
begin
     inherited Create(AOwner);
     FFilter := TDSubFilters.Create(Self);
end;


destructor TDFilterEditor.Destroy;
begin
     FFilter.Free;
     inherited Destroy;
end;


function TDFilterEditor.Execute: Boolean;
var
   oDialog : TFilterDialog;
   i       : Integer;
begin
     Result := False;
     if not Assigned(Field) or not Assigned(DataSet) then Exit;
     oDialog := TFilterDialog.Create(Application);
     try
     begin
          oDialog.Field     := FField;
          oDialog.Marker    := FMarker;
          oDialog.DataSet   := FDataSet;
          oDialog.UpperCase := (foCaseInsensitive in FDataSet.FilterOptions);
          oDialog.Fields.Clear;
          oDialog.cbxFields.Items.Clear;
          for i := 0 to FDataSet.Fields.Count-1 do
          begin
               oDialog.Fields.Add(FDataSet.Fields[i].FieldName);
               oDialog.cbxFields.Items.Add(Trim(FDataSet.Fields[i].DisplayLabel));
          end;
          Result := (oDialog.ShowModal = mrOk);
          if Result then SubFilter := oDialog.Filters;
     end;
     finally
          oDialog.Free;
     end;
end;


procedure TDFilterEditor.SetFilter(Value: TDSubFilters);
begin
     FFilter.Assign(Value);
end;


{ TDColorEditor }


function TDColorEditor.Execute: Boolean;
var
  oColFlt : TColorFltDialog;
begin
     oColFlt := TColorFltDialog.Create(Application);
     try
     begin
          oColFlt.DataSet     := FDataSet;
          oColFlt.Colors      := FColors;
          oColFlt.MarkerField := FMarker;
          Result := (oColFlt.ShowModal = mrOk);
     end;
     finally
          oColFlt.Free;
     end;
end;


{ TDExportEditor }

procedure TDExportEditor.Execute;
var
   oExport: TExportDialog;
begin
     oExport := TExportDialog.Create(Application);
     try
     begin
          oExport.DataSet := FDataSet;
          oExport.ShowModal;
     end;
     finally
          oExport.Free;
     end;
end;


{ TDFillUpEditor }

constructor TDFillUpEditor.Create(AOwner: TComponent);
begin
     inherited Create(AOwner);
     FLines := TStringList.Create;
end;


destructor TDFillUpEditor.Destroy;
begin
     FLines.Free;
     FLines := nil;
     inherited Destroy;
end;


function TDFillUpEditor.Execute: Boolean;
var
   oFillUp: TFillUpDialog;
begin
     Result := False;
     if Assigned(FField) then
     begin
          if FField.FieldKind = fkData then
          begin
               oFillUp := TFillUpDialog.Create(Application);
               try
               begin
                    oFillUp.Field := FField;
                    FMode  := fmNone;
                    Result := (oFillUp.ShowModal = mrOk);
                    if Result then
                    begin
                         FText := oFillUp.edTolt.Text;
                         FLines.Assign(oFillUp.mSQL.Lines);
                         case oFillUp.rgpKonvert.ItemIndex of
                              cniToUpper    : FMode := fmUpper;
                              cniToLower    : FMode := fmLower;
                              cniFillConst  : FMode := fmConst;
                              cniSQL        : FMode := fmSQL;
                         end;
                    end;
               end;
               finally
                    oFillUp.Free;
               end;
          end;
     end;
end;


end.
