unit gcMultiPolyline;
{
Version 2.8.07
See VersionInfo.doc for details

Acknowledgements below:

//Delphi Component Design by Danny Thorpe

////////////////////////////////////////////////////////////////////////////////
// PolylineClass is a unit designed to store, manipulate and draw polylines.  //
// The TPolylineList type (derived from TObjectList) provides a list container//
// within which TPolyline objects can be stored.                              //
// The TPolyline type (derived from TObject) contains a complete description  //
// of an individual polyline object and how to draw it.                       //
////////////////////////////////////////////////////////////////////////////////
// Written by Dr Steve Evans (steve@lociuk.com)                               //
// You can use this code however you like providing the above credits remain  //
////////////////////////////////////////////////////////////////////////////////
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, Math, stdctrls, TypInfo, Contnrs, Types, Menus, Globals
  ;

type
  TmpOutputEvents = class;
  TgcMultiPolyline = class;
  TPolyLine = class;
  TLogicLine = class;

  TDrawingMode = (dmNone, dmDraw, dmSelect);  
  TbldGridSize = 0..10;
  TLineWidth = 1..5;
  TllEndPoint = -1..0;
  TLinkType = (ltInput,ltOutput);
  TDragPoint = (dpFirst,dpLast);

  TGeoPt = record 
    X,Y: Integer;
    Selected: Boolean;
  end;

  TmpOutputEvent = class(TCollectionItem) 
  private
    FOutputControl: TComponent;
    FOutputEvent: TLogicChangeEvent;
    procedure SetOutputControl(Value: TComponent);
    procedure SetOutputEvent(Value: TLogicChangeEvent);
  protected
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
  //published
    property OutputControl:TComponent read FOutputControl
      write SetOutputControl;
    property OutputEvent:TLogicChangeEvent read FOutputEvent
      write SetOutputEvent stored true default nil;
  end;

  TmpOutputEvents = class(TCollection)  
  private
    FOwner: TLogicLine;
    function GetItem(Index: Integer): TmpOutputEvent;
    procedure SetItem(Index: Integer; Value: TmpOutputEvent);
  protected
    function GetOwner: TPersistent; override;
  public
    constructor Create(AOwner: TLogicLine);
    function Add: TmpOutputEvent;
    function Owner: TLogicLine;
    {If a class has a default property, you can access that property with the
    abbreviation object[index], which is equivalent to object.property[index].}
    property Items[Index: Integer]: TmpOutputEvent read GetItem
      write SetItem; default;
  end;

  TPolyPoint = class(TCollectionItem)
  private
    FSelected: Boolean;
    FX: Integer;
    FY: Integer;
    procedure SetSelected(Value: Boolean);
    procedure SetX(Value: Integer);
    procedure SetY(Value: Integer);
  protected
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    property Selected: Boolean read FSelected write SetSelected;
  published
    property X: Integer read FX write SetX;
    property Y: Integer read FY write SetY;
  end;

  TPolyPoints = class(TCollection)
  private
    FOwner: TComponent;
    function GetPolyPoint(Index: Integer): TPolyPoint;
    procedure SetPolyPoint(Index: Integer; Value: TPolyPoint);
  protected
    function GetOwner: TPersistent; override;
    procedure Update(Item: TCollectionItem); override;
  public
    constructor Create(AOwner: TComponent);
    function Add: TPolyPoint;
    procedure DeleteNode(ListIndex: Integer);
    {If a class has a default property, you can access that property with the
    abbreviation object[index], which is equivalent to object.property[index].}
    property Items[Index: Integer]: TPolyPoint read GetPolyPoint
      write SetPolyPoint; default;
  end;

  TPolyLine = class(TComponent)
  private
    DraggingPoint: Boolean;
    DraggingPolyLine: Boolean;
    DraggingSegment: Boolean;
    HotSize:Integer;
    FDragPoint : TDragPoint;
    FHost : TgcMultiPolyline;
    FLineColor : TColor;
    FLineStyle : TPenStyle;
    FLineWidth : TLineWidth;
    FSelected : boolean;
    FSelectedSegment: Integer;
    FHeight : integer;
    FPolyPoints : TPolyPoints;
    FWidth : integer;
    OX,OY : integer;
    procedure AddPoint(APoint: TGeoPt);
    function  CursorIsOverPolyPoint(x,y: Integer): integer;
    procedure DeleteAllPoint;     //made private to avoid use by others
    function GetLineColor: TColor;
    function GetLineStyle: TPenStyle;
    function GetLineWidth: TLineWidth;
    function GetPointSelRect(Index: Integer): TRect;
    function IsPointOnLine(XA, YA, XB, YB, XC, YC,
      Tolerance: Double): Boolean;
    procedure SelectPolySegment(x,y: Integer);  //Identifies & Selects Poly containing X,Y
    procedure SetDragPoint(const Value: TDragPoint);
    procedure SetLineColor(const Value: TColor);
    procedure SetLineWidth(const Value: TLineWidth);
    {LineStyles: psSolid, psDash, psDot, psDashDot, psDashDotDot, psClear, psInsideFrame}
    procedure SetLineStyle(const Value: TPenStyle);
    procedure SetPolyPoints(Value: TPolyPoints);
    procedure SetSelected(const Value: boolean);
    procedure SetWidth(Value: integer);
    procedure SetHeight(Value: integer);
    procedure SetHost(Value: TgcMultiPolyline);   
    function GetSelectedPointIdx: Integer;
  protected
    procedure MouseDownHandler(Button: TMouseButton; Shift: TShiftState; X,
      Y: Integer); virtual;
    procedure MouseMoveHandler(Shift: TShiftState; X, Y: Integer);
    procedure MouseUpHandler(Button: TMouseButton; Shift: TShiftState; X,
      Y: Integer);
    procedure MovePolyLineSegment(dx, dy, I: Integer);
    procedure MoveSelectedPolyline(dx, dy: Integer);
    procedure Loaded; override;
  public
    function AddPointToPolyline(pDPoint: TGeoPt): Integer;
    procedure AddPolylinePoint(x,y: Integer; Selected: Boolean);
    constructor Create(AOwner: TComponent); override;
    function  DeleteAPoint(PointId:integer):Boolean; overload;
    function  DeleteAPoint(APoint:TPoint):Boolean;   overload;
    procedure DeleteSelectedPolyPoint;
    procedure DeselectPolyline;
    destructor Destroy; override;
    procedure Draw(pCanvas: TCanvas); virtual;
    function GetPointsCount: Integer;
    procedure InitTwoPoints(APoint1,Apoint2:TPoint);
    function  InsertAPointAfter(PointId: integer;APoint: TGeoPt): boolean; overload;
    function  InsertAPointAfter(NewPoint, APoint: TGeoPt): boolean; overload;
    procedure InsertPointInPolyline(Index: Integer; pDPoint: TGeoPt);
    function InsertPolylinePoint(x, y: Integer; Selected: Boolean;
      PointID: Integer): boolean;
    property DragPoint:TDragPoint read FDragPoint write SetDragPoint;
    property LineColor: TColor read GetLineColor write SetLineColor;
    property LineStyle:TPenStyle read GetLineStyle write SetLineStyle;
    property LineWidth: TLineWidth read GetLineWidth write SetLineWidth;
    procedure MovePolyline(dx, dy: Integer);
    procedure MoveSelectedPolyPoint(dx, dy: Integer);// default 20;
    procedure PlacePolyPoint(PointId, X, Y: Integer);
    procedure SetPointsArray(PolyPtList: array of TPoint);
    property Height : integer read FHeight write SetHeight;
    property Selected : boolean read FSelected write SetSelected;
    property SelectedSegment:Integer read FSelectedSegment write FSelectedSegment;
    property Width : integer read FWidth write SetWidth;
    function GetParentComponent: TComponent; override;
  published
    property Host:TgcMultiPolyline read FHost write SetHost;
    property PolyPoints: TPolyPoints read FPolyPoints write SetPolyPoints;
  end;

  TgcMultiPolyLine = class(TGraphicControl)
  private
  { Private declarations }
    FtX,FtY: Integer;   {Origin and Temp coords}
    PointCount : Integer;
    FGridEnabled : Boolean;
    FbldGridSize : TbldGridSize;
    FLogicMode : TLogicMode;
    FPolyLines : TComponentList;
    FPopupMenu : TPopupMenu;
    FPopupItem : array of TMenuItem;
    FUnique : Integer;
    TempPolyline : array of TPoint;
    procedure LogicLineStartDrag(Sender: TObject; var DragObject: TDragObject);
    function GetPositionOnGrid(APos: Integer): Integer;
    procedure puiLineAddPoint(Sender: TObject);
    procedure puiLineDeletePoint(Sender: TObject);
    procedure puiLineDisconnectInput(Sender: TObject);
    procedure puiLineInvertOutput(Sender: TObject);
    procedure puiLineFreeLogicElement(Sender: TObject);
    procedure puiLineSetPriority(Sender: TObject); //V2.26
    procedure puMenuHandler(Sender: TObject; MousePos: TPoint; var Handled: Boolean);
    procedure SetGridEnabled(Value:Boolean);
    procedure SetbldGridSize(Value:TbldGridSize);
    procedure SetLogicMode(Value:TLogicMode);
    procedure SetTX(Value: Integer);
    procedure SetTY(Value: Integer);
    procedure SetUnique(Value: integer);
    property TX: integer read FtX write SetTX;
    property TY: integer read FtY write SetTY;
    function GetSelectedPolyLine: TLogicLine;
  protected
  { Protected declarations }
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X,
      Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X,
      Y: Integer); override;
    procedure Paint; override;
    procedure DrawAll;
    function CreateNewPolyline(AOwner:TComponent): TLogicLine;
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
    procedure EraseCanvas;
  public
    { Public declarations }
    DrawMode: TDrawingMode;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function  ArrayToNewPolyline(PolyPtList: array of TPoint):TLogicLine;
    procedure EndAPolyline;
    procedure StartAPolyline;
    procedure DeleteSelectedPolyline;
    procedure DeletePolyline(ListIndex: Integer);
    function GetExtents: TPoint;
    property GridEnabled : boolean read FGridEnabled write SetGridEnabled;
    property bldGridSize:TbldGridSize read FbldGridSize write SetbldGridSize;
    property LogicMode:TLogicMode read FLogicMode write SetLogicMode;
    property PopupMenu;
  published
  { Published declarations }
    property Unique: integer read FUnique write SetUnique;
    property Align;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Hint;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
    property ParentShowHint;
    property ShowHint;
    property Visible;
  end;

  TLogicLine = class(TPolyLine)
  private
    FColorOutput : TColor;
    FInitLogic : Boolean;
    FInputRecursionCounter : integer; ///V2.27
    FInvertOut : Boolean;
    FLogicInputSource : TComponent;
    FLogicState : Boolean;
    FOnLogicChange : TLogicChangeEvent;
    FOutputEvents : TmpOutputEvents;
    FOutputState : Boolean;
    FPosX, FPosY : integer;
    FPriority: Integer;   //V2.26
    procedure SetPriority(const Value: Integer);     //V2.26
    procedure AddOutput(ALogicElement:TComponent;
      ALogicChangeMethod:TLogicChangeEvent);
    procedure DeleteOutput(ALogicElement:TComponent);
    procedure SetInitLogic(Init:Boolean);
    procedure SetInvertOut(Value:Boolean);
    procedure SetLogicInputSource(ALogicElement: TComponent);
    procedure SetLogicState(State:Boolean);
    procedure SetOutputEvents(Value: TmpOutputEvents);
    procedure SetOutputState(Value: Boolean);
    function GetNotBounds: TRect;
  protected
    function DetermineLogicalState(ALogicInput:Boolean):Boolean; virtual;
    procedure MouseDownHandler(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure ReceiveLogicChange(ALogicState,Init:Boolean);
    procedure RequestNewPriority(APriority: integer); //V2.26
    property InitLogic:Boolean read FInitLogic write SetInitLogic;
    property  OutputState:Boolean read FOutputState write SetOutputState;
    property OnLogicChange: TLogicChangeEvent
             read FOnLogicChange write FOnLogicChange;
  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy;override;
    procedure AssignOutput(ALogicElement:TComponent;
              ALogicChangeMethod:TLogicChangeEvent;
              AnOutputAction:TOutputActionType);
    procedure Draw(pCanvas: TCanvas); override;
    procedure MoveEndPoint(PointID:TllEndPoint;Xoffset:integer);
    procedure Notification(AComponent:TComponent; Operation:TOperation); override;
    property LogicState:Boolean read FLogicState write SetLogicState;
    property OutputEvents: TmpOutputEvents read FOutputEvents write SetOutputEvents;
    procedure PlacePolyPointOrMoveBothPoints(PointId, x, y: Integer);
  published
    property InvertOut:Boolean read FInvertOut write SetInvertOut stored true default false;
    property LogicInputSource:TComponent
             read FLogicInputSource write SetLogicInputSource
             stored true default nil;
    property Priority:Integer read FPriority Write SetPriority default -1;  //V2.26
  end;

  TLogicLineDragObject = class(TDragObjectEx)
  private
    FLinkType : TLinkType;
    FLogicLineID : TLogicLine;
  protected
  public
    property LinkType:TLinkType read FLinkType write FLinkType;
    property LogicLineID:TLogicLine read FLogicLineID write FLogicLineID;
  end;

implementation

uses LogicUnaryOp, LogicBinaryOp, LogicConnect, LogicMultiOp;

procedure TPolyLine.AddPoint(APoint: TGeoPt);
var
  APolyPoint : TPolyPoint;
begin
  PolyPoints.BeginUpdate;
  APolyPoint := PolyPoints.Add;
  APolyPoint.X := APoint.X;
  APolyPoint.Y := APoint.Y;
  APolyPoint.Selected := APoint.Selected;
  PolyPoints.EndUpdate;
  TgcMultiPolyline(Host).Invalidate;
end;

procedure TPolyLine.AddPolylinePoint(x,y: Integer; Selected: Boolean);
var
   aNewPoint: TGeoPt;
begin  
   aNewPoint.X:=x;
   aNewPoint.Y:=y;
   aNewPoint.Selected:=Selected;
   AddPoint(aNewPoint);
end;

function TgcMultiPolyLine.ArrayToNewPolyline(PolyPtList: array of TPoint):TLogicLine;
begin
  if High(PolyPtList) > 0 then {at least two elements}
    begin
    Result := CreateNewPolyline(Owner);
    Result.SetPointsArray(PolyPtList);
    end
  else
    Result := nil;
end;

constructor TgcMultiPolyLine.Create(AOwner: TComponent);
var
  i : integer;
begin
  inherited Create(AOwner);
  FPolyLines := TComponentList.Create(false);   
  FtX := 0;
  FtY := 0;
  FLogicMode := lmDesign;
  ShowHint := false;
  //size is fixed or automatically adjusted, and scrollbars handle oversize
  Align := alNone;
  Anchors := [akTop,akLeft];
  Top := 0;
  Left := 0;
  Width := 80;
  Height := 40;
  FUnique := 0;
  FGridEnabled := false;
  FbldGridSize := 0;
  bldGridSize := 5;
  GridEnabled := True;
  PointCount := 0;

  //DragDrop init
  DragKind := dkDrag;
  DragMode := dmManual;
  OnStartDrag := LogicLineStartDrag;

  //Popup Menu
  FPopupMenu := TPopUpMenu.Create(self);
  SetLength(FPopupItem,6);   //V2.26
  for i := 0 to 5 do begin   //V2.26
    FPopupItem[i] := TMenuItem.Create(Self);
    FPopupMenu.Items.Add(FPopupItem[i]);
  end;
  FPopupItem[0].Caption := 'Add Point';
  FPopupItem[0].OnClick := puiLineAddPoint;
  FPopupItem[1].Caption := 'Delete Point';
  FPopupItem[1].OnClick := puiLineDeletePoint;
  FPopupItem[2].Caption := 'Invert Output';
  FPopupItem[2].OnClick := puiLineInvertOutput;
  FPopupItem[3].Caption := 'Set Output Priority (-1)'; //V2.26
  FPopupItem[3].OnClick := puiLineSetPriority;    //V2.26
  FPopupItem[4].Caption := 'Disconnect Input';
  FPopupItem[4].OnClick := puiLineDisconnectInput;
  FPopupItem[5].Caption := 'Delete LogicElement';
  FPopupItem[5].OnClick := puiLineFreeLogicElement;
  {In autopopup mode, the menu appears even if you right click far off the
   polyline.}
  FPopupMenu.AutoPopup := false;
  OnContextPopup := puMenuHandler;
  with Canvas do
  begin
    Pen.Style := psSolid;
    Pen.Color := clBlack;
    Brush.Style := bsClear;
    Brush.Color := clWhite;
  end;
end;

function TgcMultiPolyLine.CreateNewPolyline(AOwner:TComponent): TLogicLine;
begin
  //AOwner is not used.  TgcMPL needs to own all TPLs so use Self
  Result := TLogicLine.Create(AOwner);
  Result.Name := 'ALogicLine' + IntToStr(Unique);
  Unique := Unique + 1;
  Result.Host := Self;
end;

function TPolyLine.CursorIsOverPolyPoint(x,y: Integer): integer;
var
  i: Integer;
  aRect: TRect;
  aPoint: TPoint;
begin  
  aPoint.X:=x;
  aPoint.Y:=y;
  Result:= -1;

  for i:=0 to GetPointsCount-1 do
  begin
    arect:=GetPointSelRect(i);
    if PtInRect(arect,aPoint) then
    begin
      Result:= i;
      PolyPoints[i].Selected:=True;
    end else
      begin
      PolyPoints[i].Selected:= false;
      end;
  end;
end;

procedure TPolyLine.DeleteAllPoint;
begin
  //Dangerous, don't want invisible components.  Destroy instead?
  //Temporarily made this a private procedure to limit exposure
  PolyPoints.Clear;
  TgcMultiPolyLine(Host).Invalidate;
end;

function TPolyLine.DeleteAPoint(APoint: TPoint): Boolean;
var
  i:integer;
begin
 Result:=false;
 for i:=GetPointsCount -1 downto 0 do
 begin
   if (APoint.x = PolyPoints[i].x) and (APoint.y = PolyPoints[i].y) then
     Result:= DeleteAPoint(i);
 end;
end;

function TPolyLine.DeleteAPoint(PointId: integer): Boolean;
begin
  Result:=false;
  if (PointId >= 0) and (PointId < GetPointsCount) and
     (GetPointsCount > 2) then
  begin
    PolyPoints.Delete(PointId);
    Result := true;
    TgcMultiPolyLine(Host).Invalidate;
  end;
end;

procedure TgcMultiPolyLine.DeletePolyline(ListIndex: Integer);
begin
  if FPolyLines.Count >= ListIndex then
    begin
    TLogicLine(FPolyLines[ListIndex]).Free;
    Invalidate;
    end;
end;

procedure TgcMultiPolyLine.DeleteSelectedPolyline;
var
  MyPoly : TLogicLine;
begin
  MyPoly := GetSelectedPolyLine;
  if Assigned(MyPoly) then
    begin
    MyPoly.Free;
    Invalidate;
    end;
end;

destructor TgcMultiPolyLine.Destroy;
var
  i : integer;
begin
  for i := FPolyLines.Count - 1 downto 0 do
    TLogicLine(FPolyLines[i]).Free;

  try
    FPopupMenu.Free;
    //also frees each instance of FpuiLogicDelay
  except
  end;
  try
    SetLength(FPopupItem,0);
  except
  end;
   inherited Destroy;
end;

procedure TgcMultiPolyLine.DrawAll;
var
  i : integer;
begin
  for i := 0 to FPolyLines.Count - 1 do
    TLogicLine(FPolyLines[i]).Draw(Canvas);
end;

function TgcMultiPolyLine.GetExtents:TPoint;
var
  i,j,h,w : integer;
  ALogicLine : TLogicLine;
begin
  h := 0;
  w := 0;
  for i := 0 to FPolyLines.Count - 1 do
  begin
    ALogicLine := TLogicLine(FPolyLines[i]);
    for j:=0 to ALogicLine.PolyPoints.Count-1 do
    begin
      h := max(h,ALogicLine.PolyPoints[j].X);  //V2.8.07
      w := max(w,ALogicLine.PolyPoints[j].Y);  //V2.8.07
    end;
  end;
  GetExtents := Point(h,w);
end;

procedure TgcMultiPolyLine.EndAPolyline;
var
  APolyLine : TLogicLine;
begin
  DrawMode:=dmSelect;
  if PointCount > 1 then
    begin
    APolyLine := CreateNewPolyline(Owner);
    APolyLine.SetPointsArray(TempPolyLine);
    end;
  PointCount := 0;
  SetLength(TempPolyline,0);
  Screen.Cursor:=crDefault;
end;

procedure TgcMultiPolyLine.GetChildren(Proc: TGetChildProc;
  Root: TComponent);
var
  I: Integer;
begin
  for I := 0 to FPolyLines.Count - 1 do
  begin
    if TLogicLine(FPolyLines[I]).GetParentComponent = Self then
      Proc(FPolyLines[I]);
  end;
end;

function TPolyLine.GetLineColor: TColor;
begin
  Result := FLineColor;
end;

function TPolyLine.GetLineStyle: TPenStyle;
begin
  Result := FLineStyle;
end;

function TPolyLine.GetLineWidth: TLineWidth;
begin
  Result := FLineWidth;
end;

function TgcMultiPolyLine.GetPositionOnGrid(APos:Integer):Integer;
var
  Delta : integer;
begin
  if GridEnabled and (bldGridSize > 0) then
    begin
    Delta := APos mod bldGridSize;
    if (Delta / bldGridSize) >= 0.5 then
      Result := APos + bldGridSize - Delta
    else
      Result := APos - Delta;
    end
  else
    Result := APos;
end;

function  TPolyLine.GetSelectedPointIdx:Integer;
var
  i:integer;
begin
  Result := -1;
  for i := 0 to GetPointsCount - 1 do
    begin
    if PolyPoints[i].Selected then Result := i;
    end;
end;

procedure TPolyLine.InitTwoPoints(APoint1,Apoint2:TPoint);
begin
  AddPolylinePoint(APoint1.X, APoint1.Y, True);
  AddPolyLinePoint(APoint2.X, APoint2.Y, True);
  Selected := true;
end;

function TPolyLine.InsertAPointAfter(PointId: integer;APoint: TGeoPt): boolean;
var
 APolyPoint : TPolyPoint;
begin
  // after PointID insert APoint (Can't insert before first point, can after last
  Result:=false;
  if (PointID >= 0) and (PointID < GetPointsCount) then
  begin
    PolyPoints.BeginUpdate;
    APolyPoint := PolyPoints.Insert(PointID+1) as TPolyPoint;
    APolyPoint.X := APoint.X;
    APolyPoint.Y := APoint.Y;
    APolyPoint.Selected := APoint.Selected;
    PolyPoints.EndUpdate;
    Result:=true;
  end;
end;

function TPolyLine.InsertAPointAfter(NewPoint, APoint: TGeoPt): boolean;
var
  i:integer;
begin
  Result:=false;
  for i:=0 to GetPointsCount -1 do
  begin
    if (APoint.x = PolyPoints[i].x) and (APoint.y = PolyPoints[i].y) then
    begin
      Result:= InsertAPointAfter(i,NewPoint);
      break;
    end;
 end;
end;

function TPolyLine.InsertPolylinePoint(x,y: Integer; Selected: Boolean;
  PointID: Integer):boolean;
var
   aNewPoint: TGeoPt;
begin  
   aNewPoint.X:=x;
   aNewPoint.Y:=y;
   aNewPoint.Selected:=Selected;
   Result := InsertAPointAfter(PointID,aNewPoint);
end;

function TPolyLine.IsPointOnLine(XA,YA,XB,YB,XC,YC, Tolerance: Double): Boolean;
var

  L,R,S: Double;

begin

  Result:=False;

  L:=SQRT(((XB-XA)*(XB-XA)+(YB-YA)*(YB-YA)));

  if l<>0 then
  begin
    R:= ((YA-YC)*(YA-YB)-(XA-XC)*(XB-XA))/(L*L);
    S:= ((YA-YC)*(XB-XA)-(XA-XC)*(YB-YA))/(L*L);
    //if (r>0) and (r<1) then if Abs(S*L)<=Tolerance then Result:=True;  //s*l=distance
    //Note that R is the % distance along the axis of line and S*L is the
    //perpendicular distance from the axis of the line
    //Modified original formula to hit before start and beyond end of line
    if (R>=0-Tolerance/100.0) and (R<=1+Tolerance/100.0) and
      (ABS(S*L)<=Tolerance) then Result:=True;
  end;
end;

procedure TPolyLine.SelectPolySegment(x,y: Integer);
var
  i: Integer;
  Found: Boolean;
begin
  //Selected property setter will invalidate graphics as needed.
  SelectedSegment:=-1;
  Found := false;
  //select line, segment, and points if clicked nearby
  for i:=0 to GetPointsCount-2 do
    if IsPointOnLine(PolyPoints[i].X,   PolyPoints[i].Y,
                     PolyPoints[i+1].X, PolyPoints[i+1].Y, X,Y, HotSize) then
    begin
      Found := true;
      SelectedSegment := i;
      Break;
    end;
  if Found <> Selected then Selected := found;
end;

procedure TPolyLine.SetDragPoint(const Value:TDragPoint);
begin
  if Value <> FDragPoint then
    FDragPoint := Value;
end;

procedure TgcMultiPolyLine.SetGridEnabled(Value:Boolean);
var
  i,j : integer;
begin
  if Value <> FGridEnabled then
    if not FGridEnabled then  //PolyLine may not have been on grid
      begin
      FGridEnabled := Value;
      for i := 0 to FPolyLines.Count - 1 do
        for j := 0 to TPolyLine(FPolyLines[i]).PolyPoints.Count-1 do
          with TPolyLine(FPolyLines[i]).PolyPoints do
            begin
            Items[j].X:=Items[j].X;  
            Items[j].Y:=Items[j].Y;
            end;
      Invalidate; 
      end
  else
      FGridEnabled := Value;
end;

procedure TgcMultiPolyLine.SetbldGridSize(Value:TbldGridSize);
begin
  if Value <> FbldGridSize then
  begin
    FbldGridSize := Value;
  end;
end;

procedure TgcMultiPolyLine.SetLogicMode(Value:TLogicMode);
var
  i : integer;
  Designing : boolean; //V2.26
begin
  if Value <> FLogicMode then
  begin
    if Value = lmRun then
      for i:= 0 to FPolyLines.Count -1 do
        TLogicLine(FPolyLines[i]).Selected := false;
    FLogicMode := Value;
    If FLogicMode = lmDesign then Designing := true else Designing := false;
    {
    FPopupItem[0].Caption := 'Add Point';
    FPopupItem[1].Caption := 'Delete Point';
    FPopupItem[2].Caption := 'Invert Output';
    FPopupItem[3].Caption := 'Set Output Priority'; //V2.26
    FPopupItem[4].Caption := 'Disconnect Input';
    FPopupItem[5].Caption := 'Delete LogicElement';
    }
    FPopupItem[0].Enabled := Designing;
    FPopupItem[1].Enabled := Designing;
    //FPopupItem[2].Enabled := Designing;
    //FPopupItem[3].Enabled := Designing;
    FPopupItem[4].Enabled := Designing;
    FPopupItem[5].Enabled := Designing;
  end;
end;

procedure TPolyLine.SetLineColor(const Value: TColor);
begin
  if Value <> FLineColor then
  begin
    FLineColor := Value;
    TgcMultiPolyLine(Host).Invalidate;
  end;
end;

procedure TPolyLine.SetLineStyle(const Value: TPenStyle);
begin
  if Value <> FLineStyle then
  begin
    FlineStyle := Value;
    TgcMultiPolyLine(Host).Invalidate;
  end;
end;

procedure TPolyLine.SetLineWidth(const Value: TLineWidth);
begin
  if Value <> FLineWidth then
  begin
    FlineWidth := Value;
    TgcMultiPolyLine(Host).Invalidate;
  end;
end;

procedure TPolyLine.SetSelected(const Value: Boolean);
begin
  if Value <> FSelected then
  begin
    FSelected := Value;
    TgcMultiPolyLine(Host).Invalidate;
  end;
end;

procedure TgcMultiPolyLine.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);  {override}
var
  I : Integer;
begin
  //Pass the click on to each polyline
  for i := 0 to FPolyLines.Count - 1 do
    TLogicLine(FPolyLines[i]).MouseDownHandler(Button,Shift,X,Y);

  if DrawMode=dmDraw then
  begin
    if (Button=mbLeft) then
    begin
       PointCount:=PointCount+1;
       SetLength(TempPolyline,PointCount);
       TempPolyline[PointCount-1].X:=X; 
       TempPolyline[PointCount-1].Y:=Y;
       TX := X;
       TY := Y;
       Invalidate;
    end;
    if (Button=mbRight) then
    begin
       EndAPolyline;
    end;
  end;

  inherited; //calls TGraphicControl's MouseDown
end;

procedure TPolyLine.MouseDownHandler( Button: TMouseButton; Shift: TShiftState;
  X,Y: Integer);
var
  J:Integer;
begin
  SelectPolySegment(X,Y);
  if (SelectedSegment > -1) then
    if CursorIsOverPolyPoint(X,Y) > -1 then
    begin
      if (Button=mbLeft) then
      begin
        J := GetSelectedPointIdx;
        {if either end then DragObject}
        if (ssCtrl in Shift) and ((J = 0) or (J = GetPointsCount-1))then
        begin
          if J = 0 then DragPoint := dpFirst else DragPoint := dpLast;
          TgcMultiPolyLine(Host).BeginDrag(false,4);
          TgcMultiPolyLine(Host).SendToBack;
        end
        else {Move the point}
          DraggingPoint := True;
      end;
      if (Button=mbRight) and (ssCtrl in Shift) then
        DeleteSelectedPolyPoint;
    end
    else
    //If ctrl right clicked on a line, then add a point to this segment
    //Note I is the segment ID
    if (Button=mbRight) and (ssCtrl in Shift) then
      InsertPolylinePoint(X,Y,True,SelectedSegment)
    else
    if (ssShift in Shift) then
      DraggingPolyLine := True
    else
      DraggingSegment := True;

  OX:=TgcMultiPolyLine(Host).GetPositionOnGrid(X);
  OY:=TgcMultiPolyLine(Host).GetPositionOnGrid(Y);
end;

procedure TgcMultiPolyLine.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  i: Integer;
begin
  //Pass the click on to each polyline
  for i := 0 to FPolyLines.Count - 1 do
    TLogicLine(FPolyLines[i]).MouseMoveHandler(Shift,X,Y);

  if (DrawMode=dmDraw) then
  begin
    TX := GetPositionOnGrid(X);
    TY := GetPositionOnGrid(Y);
    Invalidate;
  end;
  inherited;
end;

procedure TPolyLine.MouseMoveHandler(Shift: TShiftState; X, Y: Integer);
var
  dx,dy: Integer;
begin
  if Selected and (ssLeft in Shift) then
  begin
    dx := OX-X;
    dy := OY-Y;

    if DraggingPoint then
      MoveSelectedPolyPoint(dx,dy)
    else
    if DraggingSegment then
      MovePolyLineSegment(dx,dy,SelectedSegment)
    else
    if DraggingPolyLine then
      MoveSelectedPolyline(dx,dy);

    OX:=TgcMultiPolyLine(Host).GetPositionOnGrid(X);
    OY:=TgcMultiPolyLine(Host).GetPositionOnGrid(Y);
  end;
end;

procedure TgcMultiPolyLine.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
var
  i : integer;
begin
  //Pass the click on to each polyline
  for i := 0 to FPolyLines.Count - 1 do
    TLogicLine(FPolyLines[i]).MouseUpHandler(Button,Shift,X,Y); 

  inherited;
end;

procedure TPolyLine.MouseUpHandler(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  DraggingPoint:=False;
  DraggingPolyLine:=False;
  DraggingSegment:=False;
end;

procedure TPolyLine.MovePolyLineSegment(dx,dy, I: Integer);
begin
  if Selected and (I > -1) and (PolyPoints.Count > 1) then
    begin
    //Segment I=0 is surrounded by PolyPoints 0 and 1
    PolyPoints.BeginUpdate;
    with PolyPoints[I] do
      begin
      X:=X-dx;
      Y:=Y-dy;
      end;
    with PolyPoints[I+1] do
      begin
      X:=X-dx;
      Y:=Y-dy;
      end;
    PolyPoints.EndUpdate;
    end;
end;

procedure TPolyLine.MoveSelectedPolyline(dx,dy: Integer);
var
i: Integer;
begin  
  if Selected then
  begin
    PolyPoints.BeginUpdate;
    for i:=0 to GetPointsCount-1 do
    begin
      PolyPoints[i].X:=PolyPoints[i].X-dx;
      PolyPoints[i].Y:=PolyPoints[i].Y-dy;
    end;
    PolyPoints.EndUpdate;
  end;
end;


procedure TgcMultiPolyLine.Paint;
var
  i: Integer;
begin
  inherited;
  EraseCanvas;
  Canvas.Rectangle(GetClientRect);
  DrawAll;
  if PointCount > 0 then
    begin
    Canvas.MoveTo(TempPolyLine[0].X,TempPolyLine[0].Y);
    for i:=1 to PointCount-1 do
       Canvas.LineTo(TempPolyLine[i].X,TempPolyLine[i].Y);
    if DrawMode = dmDraw then
      begin
        Canvas.MoveTo(TempPolyLine[PointCount-1].X,
          TempPolyLine[PointCount-1].Y);
        Canvas.LineTo(TX,TY);
      end;
    end;
end;

procedure TgcMultiPolyLine.EraseCanvas;
var
  MyBrush : TBrush;
begin
  MyBrush := Canvas.Brush;
  with Canvas.Brush do
    begin
    Style := bsSolid;
    Color := clWhite;
    end;
  Canvas.FillRect(Canvas.ClipRect);  
  Canvas.Brush := MyBrush;
end;

procedure TgcMultiPolyLine.SetUnique(Value: integer);
begin
  if FUnique <> Value then
    FUnique := Value;
end;

procedure TgcMultiPolyLine.SetTX(Value: Integer);
begin
  if Value <> FtX then
    begin
    FtX := GetPositionOnGrid(Value);
    end;
end;

procedure TgcMultiPolyLine.SetTY(Value: Integer);
begin
  if Value <> FtY then
    begin
    FtY := GetPositionOnGrid(Value);
    end;
end;

procedure TgcMultiPolyLine.StartAPolyline;
begin
  DrawMode:=dmDraw;
  Screen.Cursor:=crCross;
end;

{TPolyLine}

constructor TPolyLine.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  //Polyline init
  FPolyPoints := TPolyPoints.Create(Self);
  FHeight := 0;
  FWidth := 0;
  FSelected := false;
  FHost := nil;
  DraggingPoint := false;
  DraggingPolyLine := false;
  DraggingSegment := false;
  HotSize:=4;
  FLineColor:=clBlack; {clBlack,clRed,clGray,clGreen}
  FLineStyle:=psDash; //dashed until linked in and out
  FLineWidth:=1;
  FDragPoint := dpFirst;
end;

destructor TPolyLine.Destroy;
begin
  FPolyPoints.Free;
  if FHost <> nil then TgcMultiPolyline(FHost).FPolyLines.Remove(Self);
  inherited Destroy;
end;

function TPolyLine.GetPointsCount: Integer;
begin
 Result:=PolyPoints.Count;
end;

procedure TPolyLine.SetHeight(Value: integer);
begin
  if Value <> FHeight then
    begin
    FHeight := Value;
    end;
end;

procedure TPolyLine.SetPointsArray(PolyPtList: array of TPoint);
var
i: Integer;
aPolyPoint: TPolyPoint;
begin
  Selected := false;
  PolyPoints.BeginUpdate;
  PolyPoints.Clear;
  for i:=0 to High(PolyPtList) do
    begin
    aPolyPoint := TPolyPoint(PolyPoints.Add);
    aPolyPoint.X := PolyPtList[i].X;
    aPolyPoint.Y := PolyPtList[i].Y;
    aPolyPoint.Selected := false;
    end;
  PolyPoints.EndUpdate;
end;

procedure TPolyLine.SetPolyPoints(Value: TPolyPoints);
begin
  if assigned(FPolyPoints) and (FPolyPoints <> Value) then
  begin
    FPolyPoints.BeginUpdate;
    FPolyPoints.Free;
    FPolyPoints.Assign(Value);
    FPolyPoints.EndUpdate;
  end;
end;

procedure TPolyLine.SetWidth(Value: integer);
begin
  if Value <> FWidth then
    begin
    FWidth := Value;
    end;
end;

{ TPolyPoint based on TListColumn }

constructor TPolyPoint.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FX := 0;
  FY := 0;
  FSelected := false;
end;

destructor TPolyPoint.Destroy;
begin
  inherited Destroy;
end;

procedure TPolyPoint.SetSelected(Value: Boolean);
begin
  if FSelected <> Value then
  begin
    FSelected := Value;
  end;
end;

procedure TPolyPoint.SetX(Value: Integer);
begin
  if FX <> Value then
  begin
    if not (csReading in TPolyLine(Collection.Owner).ComponentState) then
      begin
      if Value > 0 then
        FX := TgcMultiPolyline(TPolyLine(Collection.Owner).Host).GetPositionOnGrid(Value)
      else FX := 0;
      end
    else
      FX := Value;
    Changed(False);
  end;
end;

procedure TPolyPoint.SetY(Value: Integer);
begin
  if FY <> Value then
  begin
    if not (csReading in TPolyLine(Collection.Owner).ComponentState) then
      begin
      if Value > 0 then
        FY := TgcMultiPolyline(TPolyLine(Collection.Owner).Host).GetPositionOnGrid(Value)
      else FY :=0;
      end
    else
      FY := Value;
    Changed(False);
  end;
end;

procedure TPolyPoint.Assign(Source: TPersistent);
var
  APolyPoint: TPolyPoint;
begin
  if Source is TPolyPoint then
  begin
    APolyPoint := TPolyPoint(Source);
    X := APolyPoint.X;
    Y := APolyPoint.Y;
    Selected := APolyPoint.Selected;
  end
  else inherited Assign(Source);
end;

{ TPolyPoints based on TListColumns }

function TPolyPoints.Add: TPolyPoint;
begin
  Result := TPolyPoint(inherited Add);
end;

function TPolyLine.AddPointToPolyline(pDPoint: TGeoPt): Integer;
var
  aPolyPoint : TPolyPoint;
begin
  PolyPoints.BeginUpdate;
  aPolyPoint := TPolyPoint(PolyPoints.Add);
  with aPolyPoint do
    begin
    X := pDPoint.X;
    Y := pDPoint.Y;
    Selected := pDPoint.Selected;
    end;
  PolyPoints.EndUpdate;
  Result := aPolyPoint.Index;
end;

constructor TPolyPoints.Create(AOwner: TComponent);
begin
  inherited Create(TPolyPoint);
  FOwner := AOwner;
end;

procedure TPolyPoints.DeleteNode(ListIndex: Integer);
begin
  Self.Delete(ListIndex);
end;

procedure TPolyLine.DeleteSelectedPolyPoint;
var
i: Integer;
begin
  for i:=0 to PolyPoints.Count-1 do
  begin
    if PolyPoints[i].Selected then
      begin
      DeleteAPoint(i);
      break;
      end;
  end;
end;

procedure TPolyLine.DeselectPolyline;
var
i: Integer;
begin
  Selected := false;
  for i:=0 to PolyPoints.Count-1 do
    PolyPoints[i].Selected := false;
end;

procedure TPolyLine.Draw(pCanvas: TCanvas);
var
i: Integer;
OrigColor: TColor;
OrigStyle: TPenStyle;
OrigWidth: TLineWidth;
begin
  OrigColor := pCanvas.Pen.Color;
  OrigStyle := pCanvas.Pen.Style;
  OrigWidth := pCanvas.Pen.Width;

  if Selected then pCanvas.Pen.Color:= clRed else pCanvas.Pen.Color:= LineColor;
  pCanvas.Pen.Style := LineStyle;
  pCanvas.Pen.Width := LineWidth;

  pCanvas.MoveTo(PolyPoints[0].X,PolyPoints[0].Y);
  for i:=1 to PolyPoints.Count-1 do
     pCanvas.LineTo(PolyPoints[i].X,PolyPoints[i].Y);

  if Selected then
    begin
    pCanvas.Pen.Style := psSolid;
    for i:=0 to PolyPoints.Count-1 do
      pCanvas.Rectangle(PolyPoints[i].X-HotSize,PolyPoints[i].Y-HotSize,
                        PolyPoints[i].X+HotSize,PolyPoints[i].Y+HotSize);
    end;

  pCanvas.Pen.Color := OrigColor;
  pCanvas.Pen.Style := OrigStyle;
  pCanvas.Pen.Width := OrigWidth;
end;

function TPolyPoints.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

function TPolyLine.GetPointSelRect(Index: Integer): TRect;
var
  MySize : integer;
begin
  MySize := HotSize;
  Result.Left:=PolyPoints[Index].X-MySize;
  Result.Top:=PolyPoints[Index].Y-MySize;
  Result.Right:=Result.Left+2*MySize+1;
  Result.Bottom:=Result.Top+2*MySize+1;
end;

function TPolyPoints.GetPolyPoint(Index: Integer): TPolyPoint;
begin
  Result := TPolyPoint(inherited GetItem(Index));
end;

procedure TPolyLine.InsertPointInPolyline(Index: Integer;
  pDPoint: TGeoPt);
var
  APolyPoint : TPolyPoint;
begin
  PolyPoints.BeginUpdate;
  APolyPoint := TPolyPoint(PolyPoints.Insert(Index));
  with APolyPoint do
    begin
    X := pDPoint.X;
    Y := pDPoint.Y;
    Selected := pDPoint.Selected;
    end;
  PolyPoints.EndUpdate;
end;

procedure TPolyLine.MovePolyline(dx, dy: Integer);
var
i: Integer;
begin
  PolyPoints.BeginUpdate;
  for i:=0 to PolyPoints.Count-1 do
  begin
    PolyPoints[i].X:=PolyPoints[i].X-dx;
    PolyPoints[i].Y:=PolyPoints[i].Y-dy;
  end;
  PolyPoints.EndUpdate;
end;

procedure TPolyLine.MoveSelectedPolyPoint(dx, dy: Integer);
var
i: Integer;
begin
  PolyPoints.BeginUpdate;
  for i:=0 to PolyPoints.Count-1 do
  begin
     if PolyPoints[i].Selected then
     begin
     PolyPoints[i].X:=PolyPoints[i].X-dx;
     PolyPoints[i].Y:=PolyPoints[i].Y-dy;
     end;
  end;
  PolyPoints.EndUpdate;
end;

procedure TPolyLine.PlacePolyPoint(PointId, X, Y: Integer);
var
  APoint : integer;
begin
  APoint := PointID;
  //PointID = -1 refers to the last point
  if APoint = -1 then APoint := GetPointsCount-1;
  if (APoint < GetPointsCount) and (APoint >= 0) then
    if csReading in FHost.ComponentState then  
    begin
      PolyPoints.BeginUpdate;
      PolyPoints[APoint].X := X;
      PolyPoints[APoint].Y := Y;
      PolyPoints.EndUpdate;
    end
    else
    begin
      PolyPoints.BeginUpdate;
      PolyPoints[APoint].X := X-FHost.Left;
      PolyPoints[APoint].Y := Y-FHost.Top;
      PolyPoints.EndUpdate;
    end;
end;

procedure TPolyPoints.SetPolyPoint(Index: Integer; Value: TPolyPoint);
begin
  inherited SetItem(Index, Value);
end;

procedure TPolyPoints.Update(Item: TCollectionItem);
{ TCollection.Update is called by TCollectionItems.Changed() if you want to
  react to a change is made to any of the collection items. This is
  initially an abstract method. It must be overridden to contain
  whatever logic is necessary when a TCollectionItem has changed.
  I use it to set the outer dimensions of the TPolyLine.
  If Item is nil, then the update affects the whole collection, otherwise
  it affects just that Item}
begin
  if not (csReading in TPolyLine(Owner).ComponentState) then
    begin
    TgcMultiPolyLine(TPolyLine(FOwner).Host).Invalidate;
    end;
end;

{ TOutputEvent based on TListColumn }

constructor TmpOutputEvent.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FOutputEvent := nil;
end;

destructor TmpOutputEvent.Destroy;
begin
  inherited Destroy;
end;

procedure TmpOutputEvent.SetOutputControl(Value: TComponent);
begin
  if FOutputControl <> Value then
  begin
    FOutputControl := Value;
  end;
end;

procedure TmpOutputEvent.SetOutputEvent(Value: TLogicChangeEvent);
begin
  if @FOutputEvent <> @Value then
  begin
    FOutputEvent := Value;
  end;
end;

procedure TmpOutputEvent.Assign(Source: TPersistent);
var
  AOutputEvent: TmpOutputEvent;
begin
  if Source is TmpOutputEvent then
  begin
    AOutputEvent := TmpOutputEvent(Source);
    OutputControl := AOutputEvent.OutputControl;
    OutputEvent   := AOutputEvent.OutputEvent;
  end
  else inherited Assign(Source);
end;

{ TOutputEvents based on TListColumns }

constructor TmpOutputEvents.Create(AOwner: TLogicLine);
begin
  inherited Create(TmpOutputEvent);
  FOwner := AOwner;
end;

function TmpOutputEvents.GetItem(Index: Integer): TmpOutputEvent;
begin
  Result := TmpOutputEvent(inherited GetItem(Index));
end;

procedure TmpOutputEvents.SetItem(Index: Integer; Value: TmpOutputEvent);
begin
  inherited SetItem(Index, Value);
end;

function TmpOutputEvents.Add: TmpOutputEvent;
begin
  Result := TmpOutputEvent(inherited Add);
end;

function TmpOutputEvents.Owner: TLogicLine;
begin
  Result := FOwner;
end;

function TmpOutputEvents.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

procedure TPolyLine.SetHost(Value: TgcMultiPolyline);   
begin
  if Value <> FHost then
    begin
    if Value <> nil then
      TgcMultiPolyline(Value).FPolyLines.Add(Self)
    else
      begin
      TgcMultiPolyline(FHost).FPolyLines.Remove(Self)
      end;
    FHost := Value;
    end;
end;

function TPolyLine.GetParentComponent: TComponent;
begin
  Result := Host;
end;

procedure TPolyLine.Loaded;
begin
  PolyPoints.Update(nil);
end;

{ TLogicLine }
procedure TLogicLine.SetOutputEvents(Value: TmpOutputEvents);
begin
  FOutputEvents.Assign(Value);
end;

constructor TLogicLine.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FLogicState := false;
  FInitLogic := false;
  FInputRecursionCounter := 0; //V2.27
  FColorOutput := clBlack;
  FPriority := -1;  //V2.26
  FLogicInputSource := nil;
  FOutputEvents := TmpOutputEvents.Create(Self);
end;

destructor TLogicLine.Destroy;
begin
  try
  if Assigned(FLogicInputSource) then
  begin
    //Before leaving, delete input links
    LogicInputSource := nil;
  end;
  except
  end;
  inherited Destroy;
end;

procedure TLogicLine.MoveEndPoint(PointID:TllEndPoint;Xoffset:integer);
var
  APoint : integer;
begin
  APoint := PointID;
  //PointID =  0 refers to the first point
  //PointID = -1 refers to the last point
  if APoint = -1 then APoint := GetPointsCount-1;
  //APoint is now either 0 or the value of the last point or -1 if there are
  //no PolyPoints (as when the LogicLine is being deleted and this procedure
  //is called by the connected LogicElements' SetLogicInputSource procedure).
  if APoint > -1 then
  begin
    PolyPoints[APoint].X := PolyPoints[APoint].X + Xoffset;
    PolyPoints[APoint].Selected := true;
    Selected := true;
  end;
end;

procedure TLogicLine.PlacePolyPointOrMoveBothPoints(PointId,x,y: Integer);
var
  dx,dy : integer;
begin
  //Move the PolyLine if PointId is 0 and only two points and
  //2nd is not connected, otherwise place the specified point.
  if (PointId = 0) and (GetPointsCount = 2) and (OutputEvents.Count = 0) then
    begin
    dx := x - PolyPoints[0].X;
    dy := y - PolyPoints[0].Y;
    PlacePolyPoint(0,x,y);
    PlacePolyPoint(1,PolyPoints[1].X+dx,PolyPoints[1].Y+dy);
    end
  else
    PlacePolyPoint(PointId,x,y);
end;

//This is an important tool for removing dependencies to linked components
procedure TLogicLine.Notification(AComponent:TComponent ; Operation:TOperation);
begin
  if (Operation=opRemove) and (AComponent=LogicInputSource) then
    begin
    //Unlink myself from the component being destroyed
    LogicInputSource := nil;
    end;
  inherited Notification(Acomponent,Operation);
end;

procedure TgcMultiPolyline.LogicLineStartDrag(Sender: TObject;
  var DragObject: TDragObject);
var
  MyDragObj : TLogicLineDragObject;
  MyPoly : TLogicLine;
begin
  MyPoly := GetSelectedPolyline;
  if assigned(MyPoly) then
    begin
    MyDragObj := TLogicLineDragObject.Create;
    if MyPoly.DragPoint = dpFirst then
      MyDragObj.LinkType := ltOutput
    else
      MyDragObj.LinkType := ltInput;
    MyDragObj.LogicLineID := MyPoly;
    DragObject := MyDragObj;
    end;
end;

//Store my input source object, and add myself to its output list
procedure TLogicLine.SetLogicInputSource(ALogicElement: TComponent); {TLogicLine}
begin
  //Before assigning new InputSource, delete old InputSource events
  if Assigned(FLogicInputSource) then
  begin
    if FLogicInputSource is TLogicLine then
    begin
      TLogicLine(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,oaDelete);
      TLogicLine(FLogicInputSource).MoveEndPoint(-1,-10);
    end;
    if FLogicInputSource is TLogicUnaryOp then
    begin
      TLogicUnaryOp(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,
        PlacePolyPointOrMoveBothPoints,oaDelete);
      MoveEndPoint(0,10);
    end;
    if FLogicInputSource is TLogicBinaryOp then
    begin
      TLogicBinaryOp(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,
          PlacePolyPointOrMoveBothPoints,oaDelete);
      MoveEndPoint(0,10);
    end;
    if FLogicInputSource is TLogicConnect then
    begin
      TLogicConnect(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,
          PlacePolyPointOrMoveBothPoints,oaDelete);
      MoveEndPoint(0,10);
    end;
    if FLogicInputSource is TLogicMultiOp then
    begin
      TLogicMultiOp(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,
          PlacePolyPointOrMoveBothPoints,oaDelete);
      MoveEndPoint(0,10);
    end;
    //Force input to false on disconnecting input
    ReceiveLogicChange(false,false);
  end;

  try
    FLogicInputSource := ALogicElement;
  except
    FLogicInputSource := nil;
  end;
  if Assigned(FLogicInputSource) then
  begin
    if FLogicInputSource is TLogicLine then
      TLogicLine(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,oaAdd);
    if FLogicInputSource is TLogicUnaryOp then
      TLogicUnaryOp(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,
        PlacePolyPointOrMoveBothPoints,oaAdd);
    if FLogicInputSource is TLogicBinaryOp then
      TLogicBinaryOp(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,
        PlacePolyPointOrMoveBothPoints, oaAdd); 
    if FLogicInputSource is TLogicConnect then
      TLogicConnect(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,
        PlacePolyPointOrMoveBothPoints,oaAdd);
    if FLogicInputSource is TLogicMultiOp then
      TLogicMultiOp(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,
        PlacePolyPointOrMoveBothPoints,oaAdd);
    if OutputEvents.Count > 0 then LineStyle := psSolid;
  end else
    LineStyle := psDash;
  //Refresh screen not needed since AssignOutput will trigger invalidate
  //when new XY positions set.
end;

//Add/delete methods in my output list
procedure TLogicLine.AssignOutput(ALogicElement:TComponent;
          ALogicChangeMethod:TLogicChangeEvent;
          AnOutputAction:TOutputActionType);
begin 
  case AnOutputAction of
  oaAdd:
    //add ALogicChange (pointer to an OnLogicChange event)
    // to a list of TLogicChangeEvents
    AddOutput(ALogicElement,ALogicChangeMethod);
  oaDelete:
    //Lookup item = ALogicChangeMethod and delete it
    DeleteOutput(ALogicElement);
  end; //case
  //see SetLogicInputSource for companion logic
  if Assigned(FLogicInputSource) and (OutputEvents.Count > 0) then
    LineStyle := psSolid
  else
    LineStyle := psDash;
end;

//Override this function to perform custom logic
function TLogicLine.DetermineLogicalState(ALogicInput:Boolean):Boolean;
begin
  Result := ALogicInput;
end;

procedure TLogicLine.ReceiveLogicChange(ALogicState,Init:Boolean);
begin
  inc(FInputRecursionCounter);  //V2.27
  if FInputRecursionCounter < 100 then  //V2.27
    begin
    InitLogic := Init;
    LogicState := DetermineLogicalState(ALogicState);
    if FInputRecursionCounter > 0 then dec(FInputRecursionCounter); //V2.27
    end
  else
    begin
    InitLogic := Init;
    LogicState := DetermineLogicalState(ALogicState);
    if FInputRecursionCounter > 0 then dec(FInputRecursionCounter); //V2.27
    FLineColor := clYellow;  //V2.27
    TgcMultiPolyLine(Host).Invalidate;  //V2.27
    end;
{
  if FInputRecursionCounter < 100 then  //V2.27
    begin
    InitLogic := Init;
    LogicState := DetermineLogicalState(ALogicState);
    if FInputRecursionCounter > 0 then dec(FInputRecursionCounter); //V2.27
    end
  else
    begin
    if FInputRecursionCounter > 0 then dec(FInputRecursionCounter); //V2.27
    FLineColor := clYellow;  //V2.27
    TgcMultiPolyLine(Host).Invalidate;  //V2.27
    end;
}
end;

procedure TLogicLine.SetInitLogic(Init:Boolean);
begin
  FInitLogic := Init;
end;

procedure TLogicLine.SetInvertOut(Value:Boolean);
begin
  if Value <> FInvertOut then
  begin
    FInvertOut := Value;
    InitLogic := True;
    if not (csReading in ComponentState) then
      LogicState := LogicState;
  end;
end;

procedure TLogicLine.SetLogicState(State:Boolean);
var
  i : integer;
  AnOutputEvent : TmpOutputEvent;
begin
  if (State <> FLogicState) or InitLogic then
  begin
    FLogicState := State;
    OutputState := FLogicState xor InvertOut;
    for i := 0 to OutputEvents.Count-1 do
    begin
      try
      AnOutputEvent := OutputEvents.Items[i];
      AnOutputEvent.OutputEvent(OutputState,false);
      except
      end;
    end;
    If InitLogic then InitLogic := false;
    If FLogicState then LineColor := clLime else LineColor := clRed;
    if Assigned(OnLogicChange) then OnLogicChange(OutputState,FInitLogic);
  end;
end;

procedure TLogicLine.SetOutputState(Value:Boolean);
begin
  if Value <> FOutputState then
  begin
    FOutputState := Value;
    if FOutputState then FColorOutput := clLime else FColorOutput := clRed;
    TgcMultiPolyLine(Host).Invalidate;
  end;
end;

procedure TLogicLine.AddOutput(ALogicElement:TComponent;
    ALogicChangeMethod:TLogicChangeEvent);
var
  AnOutputEvent : TmpOutputEvent;
begin
  //add ALogicChangeMethod (pointer to an LogicChange event)
  // to a list of TLogicChangeEvents
  AnOutputEvent := OutputEvents.Add;
  AnOutputEvent.OutputControl := ALogicElement;
  AnOutputEvent.OutputEvent   := ALogicChangeMethod;
  //Initialize the output
  InitLogic := true;
  LogicState := LogicState;
end;

procedure TLogicLine.DeleteOutput(ALogicElement:TComponent);
var
  i : integer;
begin
  //Lookup item = ALogicChangeMethod and delete it
  for i:=OutputEvents.Count-1 downto 0 do
  begin
    if OutputEvents[i].OutputControl = ALogicElement then
      OutputEvents.Delete(i);
  end;
end;

procedure TgcMultiPolyLine.puiLineInvertOutput(Sender: TObject);
var
  MyPoly : TLogicLine;
begin
  MyPoly := GetSelectedPolyline;
  if Assigned(MyPoly) then
  begin
    MyPoly.InvertOut := not MyPoly.InvertOut;
    if FLogicMode = lmRun then MyPoly.Selected := false;  //V2.26
  end;
end;

procedure TgcMultiPolyLine.puiLineDisconnectInput(Sender: TObject);
var
  MyPoly : TLogicLine;
begin
  MyPoly := GetSelectedPolyline;
  if Assigned(MyPoly) then
  begin
    if MyPoly.Priority <> -1 then MyPoly.RequestNewPriority(-1); //V2.26
    MyPoly.LogicInputSource := nil;
  end;
end;

procedure TgcMultiPolyLine.puiLineFreeLogicElement(Sender: TObject);
var
  MyPoly : TLogicLine;
begin
  MyPoly := GetSelectedPolyline;
  if Assigned(MyPoly) then
    begin
    if MyPoly.Priority <> -1 then MyPoly.RequestNewPriority(-1);  //V2.26
    MyPoly.Free;
    Invalidate;
    end;
end;

procedure TgcMultiPolyLine.puiLineAddPoint(Sender: TObject);
var
  MyPoly : TLogicLine;
begin
  MyPoly := GetSelectedPolyline;
  if Assigned(MyPoly) then
  if (MyPoly.SelectedSegment > -1) then
  with MyPoly do
    InsertPolylinePoint(FPosX,FPosY,True,SelectedSegment);
end;

procedure TgcMultiPolyLine.puiLineDeletePoint(Sender: TObject);
var
  MyPoly : TLogicLine;
begin
  MyPoly := GetSelectedPolyline;
  if Assigned(MyPoly) then
  if (MyPoly.SelectedSegment > -1) then
    MyPoly.DeleteSelectedPolyPoint;
end;

procedure TgcMultiPolyLine.puiLineSetPriority(Sender: TObject); //V2.26
var
  MyPoly : TLogicLine;
  NewPriority : integer;
begin
  MyPoly := GetSelectedPolyline;
  if Assigned(MyPoly) then
  begin
  //Call a dialog with existing priority default, show range of priority
  NewPriority := EditOutputPriority(MyPoly,MyPoly.Priority);
  if MyPoly.Priority <> NewPriority then
    MyPoly.RequestNewPriority(NewPriority);
  if FLogicMode = lmRun then MyPoly.Selected := false;
  end;
end;

procedure TLogicLine.MouseDownHandler(Button: TMouseButton; Shift: TShiftState;
    X, Y: Integer);
var
  MyPos : TPoint;
begin
  if Host.LogicMode = lmDesign then
  begin
    inherited;
    if Selected and (Button = mbRight) and (not (ssCtrl in Shift)) and
      (not (ssDouble in Shift)) then
    begin
      FPosX := X;
      FPosY := Y;
      MyPos := TgcMultiPolyLine(Host).ClientToScreen(Point(X,Y));
      //Don't allow point delete if there are 2 or less PolyPoints
      TgcMultiPolyLine(Host).FPopupItem[1].Enabled := (GetPointsCount > 2);
      TgcMultiPolyLine(Host).FPopupItem[3].Caption :=
        'Set Output Priority (' + IntToStr(Priority) + ')'; //V2.26
      TgcMultiPolyLine(Host).FPopupMenu.Popup(MyPos.X,MyPos.Y);
    end;
  end
  else   //V2.26
  begin
    if (Button = mbRight) then
    begin
      SelectPolySegment(X,Y);
      if Selected then
      begin
        MyPos := TgcMultiPolyLine(Host).ClientToScreen(Point(X,Y));
        TgcMultiPolyLine(Host).FPopupItem[3].Caption :=
          'Set Output Priority (' + IntToStr(Priority) + ')'; //V2.26
        TgcMultiPolyLine(Host).FPopupMenu.Popup(MyPos.X,MyPos.Y);
        //Selected := false;
      end;
    end;
  end;
end;

procedure TgcMultiPolyLine.puMenuHandler(Sender: TObject; MousePos: TPoint;
  var Handled: Boolean);
begin
  Handled := true;
end;

procedure TLogicLine.Draw(pCanvas: TCanvas);
var
  OrigColor : TColor;
begin
  inherited;
  if  (InvertOut) and (GetPointsCount > 1) and (not Selected) then
  begin
    OrigColor := pCanvas.Pen.Color;
    pCanvas.Pen.Color := FColorOutput;
    pCanvas.Ellipse(GetNotBounds);
    pCanvas.Pen.Color := OrigColor;
  end;
end;

function TgcMultiPolyLine.GetSelectedPolyLine: TLogicLine;
var
  i : integer;
begin
  Result := nil;
  for i := 0 to FPolyLines.Count - 1 do
    if TLogicLine(FPolyLines[i]).Selected then
        begin
        Result := TLogicLine(FPolyLines[i]);
        break;
        end;
end;

function TLogicLine.GetNotBounds: TRect;
var
  NotRad : extended;
  Theta : extended;
  x1,x2,xa,y1,y2,ya : extended;
  i,j : integer;
begin
  NotRad := 6;
  Result := rect(0,0,10,10);

  i := PolyPoints.Count-1;
  j := PolyPoints.Count-2;

  x1 := PolyPoints[j].X;
  y1 := PolyPoints[j].Y;
  x2 := PolyPoints[i].X;
  y2 := PolyPoints[i].Y;
  if x2-x1 > 0.1 then
    begin
    Theta := arctan((y2-y1)/(x2-x1));
    xa := x2 - NotRad * cos(Theta);
    ya := (y2-y1)*(xa-x1)/(x2-x1) + y1;
    end
  else if abs(x2-x1) <= 0.1 then
    begin
    xa := x2;
    if y1 < y2 then ya := y2 - NotRad else ya := y2 + NotRad;
    end
  else
    begin
    Theta := arctan((y2-y1)/(x2-x1));
    xa := x2 + NotRad * cos(Theta);
    ya := (y2-y1)*(xa-x1)/(x2-x1) + y1;
    end;
  Result := rect(round(xa-NotRad),round(ya-NotRad),
      round(xa+NotRad),round(ya+NotRad));
end;

procedure TLogicLine.SetPriority(const Value: Integer);  //V2.26
begin
  if FPriority <> Value then
  begin
    FPriority := Value;
  end;
end;

procedure TLogicLine.RequestNewPriority(APriority: integer);    //V2.26
begin
  //Change the published priority of this LogicLine
  Priority := APriority;
  //Update priority in the LogicInputSource output event.
  if LogicInputSource is TLogicMultiOp then
    TLogicMultiOp(LogicInputSource).ChangePriority(Self,Priority);
  if LogicInputSource is TLogicUnaryOp then
    TLogicUnaryOp(LogicInputSource).ChangePriority(Self,Priority);
  if LogicInputSource is TLogicBinaryOp then
    TLogicBinaryOp(LogicInputSource).ChangePriority(Self,Priority);
  if LogicInputSource is TLogicConnSink then
    TLogicConnSink(LogicInputSource).ChangePriority(Self,Priority);
end;

initialization
RegisterClasses([TGraphicControl, TgcMultiPolyline, TPolyLine,
  TPolyPoints, TPolyPoint, TLogicLine]);
finalization

end.
