unit LogicUnaryOp;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, Math, stdctrls, TypInfo, Types, Menus, Globals, gcMultiPolyline
  ;

const
  rNot :integer = 5;   //radius of LogicInput inverted input/output symbol
  //clPaleGreen :TColor = RGB(116,207,103);
  //clPaleRed   :TColor = RGB(231,113,109);
  
type
  TuoOutputEvents = class;
  TLogicUnaryOp = class;

  TuoOutputEvent = class(TCollectionItem)
  private
    FPriority : integer;  //V2.26
    FOutputControl: TLogicLine;
    FOutputEvent: TLogicChangeEvent;
    FOutputMove: TLogicMoveOutputPos;
    procedure SetOutputControl(Value: TLogicLine);
    procedure SetOutputEvent(Value: TLogicChangeEvent);
    procedure SetOutputMove(Value: TLogicMoveOutputPos);
    procedure SetPriority(Value: Integer);  //V2.26
  protected
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
  //published
    property OutputControl:TLogicLine read FOutputControl
      write SetOutputControl;
    property OutputEvent:TLogicChangeEvent read FOutputEvent
      write SetOutputEvent stored true default nil;
    property OutputMove:TLogicMoveOutputPos read FOutputMove
      write SetOutputMove default nil;
    property Priority:Integer read FPriority write SetPriority; //V2.26
  end;

  TuoOutputEvents = class(TCollection)
  private
    FOwner: TLogicUnaryOp;
    function GetItem(Index: Integer): TuoOutputEvent;
    procedure QuickSort(L, R: Integer; SCompare: TCollectionSortCompare);  //V2.26
    procedure SetItem(Index: Integer; Value: TuoOutputEvent);
    procedure UpdatePriorities;
  protected
    function GetOwner: TPersistent; override;
    procedure PrioritizeOutputs; //V2.26
  public
    constructor Create(AOwner: TLogicUnaryOp);
    function Add: TuoOutputEvent;
    function Owner: TLogicUnaryOp;
    property Items[Index: Integer]: TuoOutputEvent read GetItem
      write SetItem; default;
  //published
  end;

TLogicUnaryOp = class(TCustomControl)
private
  clPaleRed : TColor;
  clPaleGreen : TColor;
  FColorInput : TColor;
  FColorOutput : TColor;
  FColorState  : TColor;
  FColorStateDk : TColor;
  FColorStateBk : TColor;
  FColorBrush : TColor;
  FGridSize: TGridSize;
  FGridEnabled: Boolean;
  FInitLogic : Boolean;
  FInput : Boolean;
  FInputRecursionCounter : Integer;  //V2.27
  FInvert : Boolean;
  FInvertOut : Boolean;
  FLogicInputSource : TLogicLine;
  FLogicMode : TLogicMode;
  FLogicState : Boolean;
  FOutputEvents : TuoOutputEvents;
  FOutputState : Boolean;
  FSelected   : Boolean;
  FOnLogicChange : TLogicChangeEvent;
  StartXY : TPoint;
  LastRect, ParentExtent: TRect;
  procedure OnLogicModeChange; dynamic;
  procedure AddOutput(ALogicLine:TLogicLine;
            ALogicChangeMethod:TLogicChangeEvent;
            ALogicMoveOutputMethod:TLogicMoveOutputPos);
  procedure DeleteOutput(ALogicLine:TLogicLine);
  function  GetPositionOnGrid(APos: Integer): Integer;
  function  GetTextBounds:TRect;
  procedure MoveConnectedIO;
  procedure MoveConnectedInputs;
  procedure MoveConnectedOutputs;
  procedure SetGridEnabled(Value: Boolean);
  procedure SetGridSize(Value: TGridSize);
  procedure SetInitLogic(Init:Boolean);
  procedure SetInput(Value: Boolean);
  procedure SetInvert(Value:Boolean);
  procedure SetInvertOut(Value:Boolean);
  procedure SetLogicInputSource(ALogicElement: TLogicLine);
  procedure SetLogicState(State:Boolean);
  procedure SetOutputEvents(Value: TuoOutputEvents);
  procedure SetOutputState(Value:Boolean);
  procedure puiUnOpChangeHint(Sender: TObject);
  procedure puiUnOpDisconnectInput(Sender: TObject);
  procedure puiUnOpFreeLogicElement(Sender: TObject);
  procedure puiUnOpNotImplemented(Sender: TObject);
    procedure SetLogicMode(Value: TLogicMode);
    procedure puiUnOpAddOutputLogicLine(Sender: TObject);
protected
  function  DetermineLogicalState(ALogicInput:Boolean):Boolean; dynamic;
  function  GetBoundsRect:TRect; dynamic;
  procedure Loaded; override; //v2.26
  procedure LogicLineDragOver(Sender, Source: TObject; X,
            Y: Integer; State: TDragState; var Accept: Boolean);
  procedure Paint; override;
  procedure SetSelected(Value:Boolean);
  procedure uoMouseDown(Sender: TObject; Button: TMouseButton;
    Shift: TShiftState; X, Y: Integer);
  procedure uoMouseUp(Sender: TObject; Button: TMouseButton;
    Shift: TShiftState; X, Y: Integer);
  procedure uoMouseMove(Sender: TObject; Shift: TShiftState;
    X, Y: Integer);
  procedure ReceiveLogicChange(ALogicState,Init:Boolean);
  property  InitLogic:Boolean read FInitLogic write SetInitLogic;
  property  OnLogicChange: TLogicChangeEvent
            read FOnLogicChange write FOnLogicChange;
  property  OutputState:Boolean read FOutputState write SetOutputState;
  property  Canvas;
  property Invert:Boolean read FInvert write SetInvert stored true default false;
  property InvertOut:Boolean read FInvertOut write SetInvertOut default false;
public
  StartX, StartY : integer;
  constructor Create(AOwner: TComponent); override;
  destructor  Destroy;override;
  procedure AssignOutput(ALogicLine:TLogicLine;
            ALogicChangeMethod:TLogicChangeEvent;
            ALogicMoveOutputMethod:TLogicMoveOutputPos;
            AnOutputAction:TOutputActionType);
  procedure ChangePriority(ALogicLine:TLogicLine;ALogicPriority:Integer);  //V2.26
  procedure Notification(AComponent:TComponent; Operation:TOperation); override;
  property  GridEnabled : boolean read FGridEnabled write SetGridEnabled;
  property  GridSize:TGridSize read FGridSize write SetGridSize;
  property  Input:Boolean read FInput write SetInput;
  property  LogicMode:TLogicMode read FLogicMode write SetLogicMode;
  property  LogicState:Boolean read FLogicState write SetLogicState default false; //v2.28
  property  OutputEvents: TuoOutputEvents read FOutputEvents write SetOutputEvents;
  property  Selected:Boolean read FSelected write SetSelected;
  property  OnDragDrop;
  property  OnDragOver;
  property  PopupMenu;
published
  property LogicInputSource:TLogicLine
           read FLogicInputSource write SetLogicInputSource
           stored true default nil;
end;

TLogicNot = class(TLogicUnaryOp)
private
  { Private declarations }
  FPopupMenu : TPopupMenu;
  FPopupItem : array of TMenuItem;
  procedure OnLogicModeChange; override;
protected
  { Protected declarations }
  function DetermineLogicalState(ALogicInput:Boolean):Boolean; override;
  procedure LogicLineDragDrop(Sender, Source: TObject; X,
            Y: Integer);
  procedure Paint; override;
public
  { Public declarations }
  constructor Create(AOwner: TComponent); override;
  destructor  Destroy;override;
published
  { Published declarations }
end;

TLogicInput = class(TLogicUnaryOp)
private
  { Private declarations }
  FMinWidth : integer;
  FMinHeight : integer;
  FBevelWidth : integer;
  FMouseDown : boolean;
  FOneShot : boolean;
  FPopupMenu: TPopupMenu;
  FPopupItem: array of TMenuItem;
  procedure OnLogicModeChange; override;
  procedure SetNewSize;
  procedure SetOneShot(Value:boolean);
  procedure puiInputChangeDescription(Sender: TObject);
  procedure puiInputInvertOutput(Sender: TObject);
  procedure puiInputMaint_OneShot(Sender: TObject);
  function DrawButtonFace(Canvas: TCanvas; const Client: TRect;
    BevelWidth: Integer; LightColor, DarkColor, BackColor: TColor;
    IsDown: Boolean): TRect;
protected
  { Protected declarations }
  procedure InputDblClick(Sender:TObject);
  function DetermineLogicalState(ALogicInput:Boolean):Boolean; override;
  procedure LogicLineDragDrop(Sender, Source: TObject; X,
            Y: Integer);
  procedure Paint; override; 
  procedure liMouseDown(Sender: TObject; Button: TMouseButton;
            Shift: TShiftState; X, Y: Integer);
  procedure liMouseUp(Sender: TObject; Button: TMouseButton;
            Shift: TShiftState; X, Y: Integer);
public
  { Public declarations }
  constructor Create(AOwner: TComponent); override;
  destructor  Destroy;override;
published
  { Published declarations }
  Property Caption;
  Property InvertOut;
  Property LogicState;
  Property OneShot : boolean read FOneShot write SetOneShot stored true default false;
  property OnDblClick;
end;

TDelayType = (dtOn,dtOff);

TLogicDelay = class(TLogicUnaryOp)
private
  { Private declarations }
  FMinWidth : Integer;
  FMinHeight : Integer;
  FDelayType: TDelayType;
  FTimer: TTimer;
  FTiming: boolean;
  FpuiLogicDelay: array of TMenuItem;
  FpumLogicDelay: TPopupMenu;
  procedure OnLogicModeChange; override;
  function GetInterval:Cardinal;
  procedure SetDelayType(Value:TDelayType);
  procedure SetInterval(Value:Cardinal);
  procedure puiLogDelInvertInput(Sender: TObject);
  procedure puiLogDelInvertOutput(Sender: TObject);
  procedure puiLogDelToggleDelayType(Sender: TObject);
  procedure puiLogDelSetDelay(Sender: TObject);
  procedure SetCaption;
  procedure SetNewSize;
protected
  { Protected declarations }
  function DetermineLogicalState(ALogicInput:Boolean):Boolean; override;
  procedure LogicLineDragDrop(Sender, Source: TObject; X,
            Y: Integer);
  procedure Paint; override;
  procedure TimedOut(Sender:TObject); virtual;
public
  { Public declarations }
  constructor Create(AOwner: TComponent); override;
  destructor  Destroy;override;
published
  { Published declarations }
  property DelayType:TDelayType read FDelayType write SetDelayType stored true default dtOff;
  property Interval: Cardinal read GetInterval write SetInterval stored true default 3000;
  property Invert;
  property InvertOut;
end;

TLogicOneShot = class(TLogicUnaryOp)
private
  { Private declarations }
  FMinWidth : Integer;
  FMinHeight : Integer;
  FpuiOneShot: array of TMenuItem;
  FpumOneShot: TPopupMenu;
  procedure OnLogicModeChange; override;
  procedure puiLogOneInvertInput(Sender: TObject);
  procedure puiLogOneInvertOutput(Sender: TObject);
protected
  { Protected declarations }
    function DetermineLogicalState(ALogicInput: Boolean): Boolean;
      override;
    procedure LogicLineDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure Paint; override;
public
  { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
published
  { Published declarations }
  property Invert;
  property InvertOut;
end;


implementation

uses LogicBinaryOp, LogicConnect, LogicMultiOp, LogicWindow;

{ TOutputEvent based on TListColumn }

constructor TuoOutputEvent.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FOutputControl := nil;
  FOutputEvent := nil;
  FOutputMove := nil;
  FPriority := -1; //V2.26
end;

destructor TuoOutputEvent.Destroy;
begin
  inherited Destroy;
end;

procedure TuoOutputEvent.SetOutputControl(Value: TLogicLine);
begin
  if FOutputControl <> Value then
  begin
    FOutputControl := Value;
  end;
end;

procedure TuoOutputEvent.SetOutputEvent(Value: TLogicChangeEvent);
begin
  if Addr(FOutputEvent) <> Addr(Value) then
  begin
    FOutputEvent := Value;
  end;
end;

procedure TuoOutputEvent.SetOutputMove(Value: TLogicMoveOutputPos);
begin
  if Addr(FOutputMove) <> Addr(Value) then
  begin
    FOutputMove := Value;
  end;
end;

procedure TuoOutputEvent.Assign(Source: TPersistent);
var
  AOutputEvent: TuoOutputEvent;
begin
  if Source is TuoOutputEvent then
  begin
    AOutputEvent := TuoOutputEvent(Source);
    OutputControl := AOutputEvent.OutputControl;
    OutputEvent   := AOutputEvent.OutputEvent;
    OutputMove    := AOutputEvent.OutputMove;
  end
  else inherited Assign(Source);
end;

{ TOutputEvents based on TListColumns }

constructor TuoOutputEvents.Create(AOwner: TLogicUnaryOp);
begin
  inherited Create(TuoOutputEvent);
  FOwner := AOwner;
end;

function TuoOutputEvents.GetItem(Index: Integer): TuoOutputEvent;
begin
  Result := TuoOutputEvent(inherited GetItem(Index));
end;

procedure TuoOutputEvents.SetItem(Index: Integer; Value: TuoOutputEvent);
begin
  inherited SetItem(Index, Value);
end;

function TuoOutputEvents.Add: TuoOutputEvent;
begin
  Result := TuoOutputEvent(inherited Add);
end;

function TuoOutputEvents.Owner: TLogicUnaryOp;
begin
  Result := FOwner;
end;

function TuoOutputEvents.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

{ TLogicUnaryOp }
procedure TLogicUnaryOp.SetOutputEvents(Value: TuoOutputEvents);
begin
  FOutputEvents.Assign(Value);
end;

constructor TLogicUnaryOp.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  clPaleGreen := TColor(RGB(116,207,103));
  clPaleRed  := TColor(RGB(231,113,109));
  FColorInput := clBlack;
  FColorOutput := clBlack;
  FColorState  := clBlack;
  FColorStateDk := clBlack;
  FColorStateBk := clBlack;
  FColorBrush := Canvas.Brush.Color;
  FGridEnabled := false;
  FGridSize    := 0;
  GridSize     := 5;
  GridEnabled  := true;
  FLogicInputSource := nil;
  FOutputEvents := TuoOutputEvents.Create(Self);
  FInput := true;
  FInputRecursionCounter := 0; //V2.27
  FInvert := false;
  Input := false; //force input update
  FInvertOut := false;
  FOutputState := true;
  FLogicMode := lmDesign;
  FLogicState := false;
  FInitLogic := true;
  LogicState := false; //force logic/output update
  OnDragOver := LogicLineDragOver;
  OnMouseDown := uoMouseDown;
  OnMouseMove := uoMouseMove;
  OnMouseUp   := uoMouseUp;
  with Canvas do
  begin
    Pen.Style := psSolid;
    Pen.Color := clBlack;
    Brush.Style := bsClear;
  end;
  Color := clWhite;
end;

destructor TLogicUnaryOp.Destroy;
begin
  try
  if Assigned(FLogicInputSource) then
  begin
    //Before leaving, delete input links
    LogicInputSource := nil;
  end;
  except
    on E: EOutOfResources do
      begin
      MessageDlg('Trapped exception: '+E.Message, mtInformation, [mbOk],0);
      end;
  end;
  inherited Destroy;
end;

//This is an important tool for removing dependencies to linked components
procedure TLogicUnaryOp.Notification(AComponent:TComponent ; Operation:TOperation);
begin
  if (Operation=opRemove) and (AComponent=LogicInputSource) then
    begin
    //Unlink myself from the component being destroyed
    LogicInputSource := nil;
    end;
  inherited Notification(Acomponent,Operation);
end;

procedure TLogicUnaryOp.LogicLineDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
  if IsDragObject(Source) and (Source is TLogicLineDragObject)
    then Accept := True else Accept := False;
end;

procedure TLogicUnaryOp.SetInput(Value: Boolean);
begin
  if (Value xor FInvert) <> FInput then
  begin
    FInput := Value xor FInvert;
    if FInput then FColorInput := clLime else FColorInput := clRed;
    invalidate;
  end;
end;

procedure TLogicUnaryOp.SetInvert(Value:Boolean);
var
  RealInput : boolean;
begin
  if Value <> FInvert then
  begin
    if FInvert then RealInput := not Input else RealInput := Input;
    FInvert := Value;
    ReceiveLogicChange(RealInput,True);
  end;
end;

procedure TLogicUnaryOp.SetInvertOut(Value:Boolean);
begin
  if Value <> FInvertOut then
  begin
    FInvertOut := Value;
    InitLogic := True;
    LogicState := LogicState;
  end;
end;

//Store my input source object, and add myself to its output list
procedure TLogicUnaryOp.SetLogicInputSource(ALogicElement: TLogicLine);
begin
  //Before assigning new InputSource, delete old InputSource events
  if Assigned(FLogicInputSource) then
  begin
    if FLogicInputSource is TLogicLine then
    begin
        TLogicLine(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,oaDelete);
        TLogicLine(FLogicInputSource).MoveEndPoint(-1,-10);
    end;
    //Force input to false on disconnecting input
    ReceiveLogicChange(false,false);
  end;

  try
    FLogicInputSource := TLogicLine(ALogicElement); 
  except
    FLogicInputSource := nil;
  end;
  
  if Assigned(FLogicInputSource) then
  begin
    if FLogicInputSource is TLogicLine then
    begin
      TLogicLine(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,oaAdd);
      MoveConnectedInputs;
    end;
  end;
end;

procedure TLogicUnaryOp.AssignOutput(ALogicLine:TLogicLine;
          ALogicChangeMethod:TLogicChangeEvent;
          ALogicMoveOutputMethod:TLogicMoveOutputPos;
          AnOutputAction:TOutputActionType);
begin 
  case AnOutputAction of
  oaAdd:
    //add ALogicChange (pointer to an OnLogicChange event)
    // to a list of TLogicChangeEvents
    AddOutput(ALogicLine,ALogicChangeMethod,ALogicMoveOutputMethod);
  oaDelete:
    //Lookup item = ALogicChangeMethod and delete it
    DeleteOutput(ALogicLine);
  end; //case
end;

//Override this function to perform custom logic
function TLogicUnaryOp.DetermineLogicalState(ALogicInput:Boolean):Boolean;
begin
  Result := ALogicInput;
end;

procedure TLogicUnaryOp.ReceiveLogicChange(ALogicState,Init:Boolean);
begin
  inc(FInputRecursionCounter);  //V2.27
  if FInputRecursionCounter < 100 then  //V2.27
    begin
    InitLogic := Init;
    LogicState := DetermineLogicalState(ALogicState);
    if FInputRecursionCounter > 0 then dec(FInputRecursionCounter); //V2.27
    end
  else
    begin
    if FInputRecursionCounter > 0 then dec(FInputRecursionCounter); //V2.27
    FColorInput := clYellow;  //V2.27
    Invalidate;  //V2.27
    end;
end;

function TLogicUnaryOp.GetPositionOnGrid(APos:Integer):Integer;
var
  Delta : integer;
begin
  if GridEnabled and (GridSize > 0) then
    begin
    Delta := APos mod GridSize;
    if (Delta / GridSize) >= 0.5 then
      Result := APos + GridSize - Delta
    else
      Result := APos - Delta;
    end
  else
    Result := APos;
end;

function TLogicUnaryOp.GetTextBounds:TRect;
begin
  Result := Rect(0,0,0,0);
  Windows.DrawText(Self.Canvas.Handle, PChar(Self.Caption), -1,
    Result, DT_CALCRECT);
end;

procedure TLogicUnaryOp.SetGridEnabled(Value:Boolean);
begin
  if Value <> FGridEnabled then
    if not FGridEnabled then  //object may not have been on grid
      begin
      FGridEnabled := Value;
      Left := GetPositionOnGrid(Left);
      Top  := GetPositionOnGrid(Top);
      end
  else
      FGridEnabled := Value;
end;

procedure TLogicUnaryOp.SetGridSize(Value:TGridSize);
begin
  if Value <> FGridSize then
  begin
    FGridSize := Value;
  end;
end;

procedure TLogicUnaryOp.SetInitLogic(Init:Boolean);
begin
  FInitLogic := Init;
end;

procedure TLogicUnaryOp.SetOutputState(Value:Boolean);
begin
  if Value <> FOutputState then
  begin
    FOutputState := Value;
    if FOutputState then FColorOutput := clLime else FColorOutput := clRed;
    invalidate;
  end;
end;

procedure TLogicUnaryOp.SetLogicState(State:Boolean);
var
  i : integer;
  AnOutputEvent : TuoOutputEvent;
begin
  if (State <> FLogicState) or InitLogic then
  begin
    FLogicState := State;
    OutputState := FLogicState xor InvertOut;
    for i := 0 to OutputEvents.Count-1 do
    begin
      try
      AnOutputEvent := OutputEvents.Items[i];
      AnOutputEvent.OutputEvent(OutputState,false);
      except
      end;
    end;
    If InitLogic then InitLogic := false;
    If FLogicState then
    begin
      FColorState := clLime;
      FColorStateDk := clGreen;
      FColorStateBk := clPaleGreen;
    end
    else
    begin
      FColorState := clRed;
      FColorStateDk := clMaroon;
      FColorStateBk := clPaleRed;
    end;
    Invalidate;
    if Assigned(OnLogicChange) then OnLogicChange(OutputState,FInitLogic);
  end;
end;

procedure TLogicUnaryOp.AddOutput(ALogicLine:TLogicLine;
          ALogicChangeMethod:TLogicChangeEvent;
          ALogicMoveOutputMethod:TLogicMoveOutputPos);
var
  AnOutputEvent : TuoOutputEvent;
begin
  //add ALogicChangeMethod (pointer to an LogicChange event)
  // to a list of TLogicChangeEvents
  AnOutputEvent := OutputEvents.Add;
  AnOutputEvent.OutputControl := ALogicLine;
  AnOutputEvent.OutputEvent := ALogicChangeMethod;
  AnOutputEvent.OutputMove  := ALogicMoveOutputMethod;
  //Connect the output
  MoveConnectedOutputs;
  //Initialize the output
  InitLogic := True;
  LogicState := LogicState;
end;

procedure TLogicUnaryOp.DeleteOutput(ALogicLine:TLogicLine);
var
  i : integer;
begin
  //Lookup item = ALogicChangeMethod and delete it
  for i:=OutputEvents.Count-1 downto 0 do
  begin
    if OutputEvents[i].OutputControl = ALogicLine then
      OutputEvents.Delete(i);
  end;
end;

procedure TLogicUnaryOp.Paint;
begin
  inherited Paint;
  if Selected then Canvas.Rectangle(0,0,Width,Height);
end;

function TLogicUnaryOp.GetBoundsRect:TRect;
begin
  Result := BoundsRect;
end;

procedure TLogicUnaryOp.uoMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if LogicMode = lmDesign then
  begin
    if Button = mbLeft then
      begin
      Selected := true;
      StartXY := ClientToParent(Point(X,Y),TPanel(Parent));
      ParentExtent := GetDeltaRect(TScrollBox(Parent.Parent).ClientRect,
        Point(-TPanel(Parent).Left,-TPanel(Parent).Top));
      LastRect:= GetBoundsRect;
      end;
  end;
end;

procedure TLogicUnaryOp.uoMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if (Button = mbLeft) and (LogicMode = lmDesign) then
  begin
    Selected := false;
  end;
end;

procedure TLogicUnaryOp.uoMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
var
  AbsXY, DeltaXY : TPoint;
  NewExtent : TRect;
begin
  if (ssLeft in Shift) and (LogicMode = lmDesign) then { make sure button is down }
  begin
    AbsXY := ClientToParent(Point(X,Y),TPanel(Parent));
    DeltaXY.X := GetPositionOnGrid(AbsXY.X - StartXY.X);
    DeltaXY.Y := GetPositionOnGrid(AbsXY.Y - StartXY.Y);
    NewExtent := GetDeltaRect(LastRect,DeltaXY);
    if ( PtInRect(ParentExtent, NewExtent.TopLeft) and
         PtInRect(ParentExtent, NewExtent.BottomRight) ) then
      begin
      Inc(StartXY.X,DeltaXY.X);
      Inc(StartXY.Y,DeltaXY.Y);
      end
    else
      begin
      if NewExtent.Left < ParentExtent.Left then
        DeltaXY.X := GetPositionOnGrid(ParentExtent.Left - LastRect.Left);
      if NewExtent.Right > ParentExtent.Right then
        DeltaXY.X := GetPositionOnGrid(ParentExtent.Right - LastRect.Right);
      if NewExtent.Top < ParentExtent.Top then
        DeltaXY.Y := GetPositionOnGrid(ParentExtent.Top - LastRect.Top);
      if NewExtent.Bottom > ParentExtent.Bottom then
        DeltaXY.Y := GetPositionOnGrid(ParentExtent.Bottom - LastRect.Bottom);
      NewExtent := GetDeltaRect(LastRect,DeltaXY);
      Inc(StartXY.X,DeltaXY.X);
      Inc(StartXY.Y,DeltaXY.Y);
      end;
    Self.Left := NewExtent.Left;
    Self.Top  := NewExtent.Top;
    LastRect := NewExtent;
    MoveConnectedIO;
  end;
end;

procedure TLogicUnaryOp.MoveConnectedIO;
begin
  MoveConnectedInputs;
  MoveConnectedOutputs;
end;

procedure TLogicUnaryOp.MoveConnectedInputs;
var
  X,Y : integer;
begin
  X := Left;
  Y := Top + (Height div 2);
  if Assigned(LogicInputSource) then
    if LogicInputSource is TLogicLine then
      TLogicLine(LogicInputSource).PlacePolyPoint(-1,X,Y);
end;

procedure TLogicUnaryOp.MoveConnectedOutputs;
var
  i : integer;
  X,Y : integer;
  AnOutputMove : TuoOutputEvent;
begin
  X := Left + Width;
  Y := Top + (Height div 2);
  for i := 0 to OutputEvents.Count-1 do
  begin
    try
    AnOutputMove := OutputEvents.Items[i];
    AnOutputMove.OutputMove(0,X,Y);
    except
    end;
  end;
end;

procedure TLogicUnaryOp.SetSelected(Value:Boolean);
begin
  if Value <> FSelected then
  begin
    FSelected := Value;
    Invalidate;
  end;
end;

procedure TLogicUnaryOp.puiUnOpAddOutputLogicLine(Sender: TObject); //v2.28
var
  Xo,Yo : integer;
  ALogicLine : TLogicLine;
begin
  Xo := Left + Width;
  Yo := Top  + (Height div 2);
  ALogicLine := MultiPolyline.ArrayToNewPolyline([point(Xo,Yo),
    point(Xo+30,Yo+30)]);
  ALogicLine.Selected := true;
  LogicCount.Count:=LogicCount.Count+1;
  ALogicLine.Name := 'LogicLine' + IntToStr(LogicCount.Count);
  ALogicLine.LogicInputSource := Self as TLogicUnaryOp;
end;

procedure TLogicUnaryOp.puiUnOpChangeHint(Sender: TObject);
begin
  if Self.Hint <> '' then
    Self.Hint := EditControlHint(Self,Self.Hint)
  else
    Self.Hint := EditControlHint(Self,Self.Caption);
  if Self.Hint <> '' then
    Self.ShowHint := true
  else
    Self.ShowHint := false;
  Invalidate;
end;

procedure TLogicUnaryOp.puiUnOpDisconnectInput(Sender: TObject);
begin
  with Self as TLogicUnaryOp do begin
    LogicInputSource := nil;
    Invalidate;
  end;
end;

procedure TLogicUnaryOp.puiUnOpFreeLogicElement(Sender: TObject);
var
  i : integer;
begin
  //Set priority of output logiclines to -1  //V2.26
  for i := FOutputEvents.Count-1 downto 0 do
  begin
    FOutputEvents[i].Priority := -1;
    if FOutputEvents[i].FOutputControl.OutputEvents.Count = 0 then  //v2.28
      FOutputEvents[i].FOutputControl.Free;
  end;
  FOutputEvents.UpdatePriorities;

  TLogicUnaryOp(Self).Free;
end;

procedure TLogicUnaryOp.puiUnOpNotImplemented(Sender: TObject);
begin
  ShowMessage('Feature Not Implemented.');
end;

procedure TLogicUnaryOp.SetLogicMode(Value: TLogicMode);
begin
  If FLogicMode <> Value then
  begin
    FLogicMode := Value;
    OnLogicModeChange;
    Invalidate; //Updates TLogicInput button face
  end;
end;

procedure TLogicUnaryOp.OnLogicModeChange;
begin
  //Override in descendants to update popup menus
end;

procedure TLogicUnaryOp.Loaded;   //V2.26
begin
  inherited;
  FOutputEvents.PrioritizeOutputs;
end;

procedure TLogicUnaryOp.ChangePriority(ALogicLine: TLogicLine;
  ALogicPriority: Integer);   //V2.26
var
  i : integer;
begin
  //Lookup ALogicLine and change the priority of its output event
  for i:=0 to OutputEvents.Count-1 do
  begin
    if OutputEvents[i].OutputControl = ALogicLine then
    begin
      OutputEvents[i].Priority := ALogicPriority;
      OutputEvents.PrioritizeOutputs;
    end;
  end;
end;

{TLogicNot}

constructor TLogicNot.Create(AOwner: TComponent);
var
  i : integer;
begin
  inherited Create(AOwner);
  Top := 50;
  Left := 50;
  Width := 10;
  Height := 10;
  Visible := true;
  OnDragDrop := LogicLineDragDrop;

  //Popup Menu
  FPopupMenu := TPopUpMenu.Create(self);
  FPopupMenu.AutoPopup := true;
  SetLength(FPopupItem,3);
  for i := 0 to 2 do begin
    FPopupItem[i] := TMenuItem.Create(Self);
    FPopupMenu.Items.Add(FPopupItem[i]);
  end;
  FPopupItem[0].Caption := 'Disconnect Input';
  FPopupItem[0].OnClick := puiUnOpDisconnectInput;
  FPopupItem[1].Caption := 'Add Output LogicLine'; //v2.28
  FPopupItem[1].OnClick := puiUnOpAddOutputLogicLine;
  FPopupItem[2].Caption := 'Delete LogicElement';
  FPopupItem[2].OnClick := puiUnOpFreeLogicElement;
  PopupMenu := FPopupMenu;

  with Canvas do
  begin
    Pen.Style := psSolid;
    Pen.Color := clBlack;
    Brush.Style := bsClear;
  end;
end;

destructor TLogicNot.Destroy;
begin
  try
    FPopupMenu.Free;
    //also frees each instance of FpuiLogicDelay
  except
  end;
  try
    SetLength(FPopupItem,0);
  except
  end;
  inherited Destroy;
end;

procedure TLogicNot.LogicLineDragDrop(Sender, Source: TObject; X,
  Y: Integer);
begin
  if IsDragObject(Source) and (Source is TLogicLineDragObject) then
    with (Source as TLogicLineDragObject) do
    begin
      if LinkType = ltInput then
      begin
        LogicInputSource := LogicLineID as TLogicLine;
      end else begin
        LogicLineID.LogicInputSource := Sender as TLogicNot;
      end;
    end;
end;

procedure TLogicNot.Paint;
begin
  inherited Paint;
  Canvas.Pen.Color := FColorState;
  Canvas.Ellipse(0,0,Width,Height);
end;

function TLogicNot.DetermineLogicalState(ALogicInput:Boolean):Boolean;
begin
  Result := not ALogicInput;
end;


procedure TLogicNot.OnLogicModeChange;
begin
{
  FPopupItem[0].Caption := 'Disconnect Input';
  FPopupItem[1].Caption := 'Add Output LogicLine'; //v2.28
  FPopupItem[2].Caption := 'Delete LogicElement';
}
  FPopupItem[0].Enabled := LogicMode = lmDesign;
  FPopupItem[1].Enabled := LogicMode = lmDesign;
  FPopupItem[2].Enabled := LogicMode = lmDesign;
end;

{TLogicInput}

constructor TLogicInput.Create(AOwner: TComponent);
var
  i : integer;
begin
  inherited Create(AOwner);
  FMinWidth := 40;
  FMinHeight := 20;
  FMouseDown := false;
  FBevelWidth := 2;
  Caption := 'Input';
  FOneShot := false;
  FInvertOut := false;
  Top := 50;
  Left := 50;
  Width := FMinWidth;
  Height := FMinHeight;
  Visible := true;
  OnMouseDown := liMouseDown;
  OnMouseUp   := liMouseUp;
  OnDblClick := InputDblClick;
  OnDragDrop := LogicLineDragDrop;

  //Popup Menu
  FPopupMenu := TPopUpMenu.Create(self);
  FPopupMenu.AutoPopup := true;
  SetLength(FPopupItem,6);
  for i := 0 to 5 do begin
    FPopupItem[i] := TMenuItem.Create(Self);
    FPopupMenu.Items.Add(FPopupItem[i]);
  end;
  FPopupItem[0].Caption := 'Change description';
  FPopupItem[0].OnClick := puiInputChangeDescription;
  FPopupItem[1].Caption := 'Change hint';
  FPopupItem[1].OnClick := puiUnOpChangeHint;
  FPopupItem[2].Caption := 'Invert Output';
  FPopupItem[2].OnClick := puiInputInvertOutput;
  FPopupItem[3].Caption := 'Toggle Maintained/OneShot';
  FPopupItem[3].OnClick := puiInputMaint_OneShot;
  FPopupItem[4].Caption := 'Add Output LogicLine';  //v2.28
  FPopupItem[4].OnClick := puiUnOpAddOutputLogicLine;
  FPopupItem[5].Caption := 'Delete LogicElement';
  FPopupItem[5].OnClick := puiUnOpFreeLogicElement;
  PopupMenu := FPopupMenu;

  with Canvas do
  begin
    Pen.Style := psSolid;
    Pen.Color := clBlack;
    Brush.Style := bsClear;
  end;
end;

destructor TLogicInput.Destroy;
begin
  try
    FPopupMenu.Free;
    //also frees each instance of FpuiLogicDelay
  except
  end;
  try
    SetLength(FPopupItem,0);
  except
  end;
  inherited Destroy;
end;

procedure TLogicInput.LogicLineDragDrop(Sender, Source: TObject; X,
  Y: Integer);
begin
  if IsDragObject(Source) and (Source is TLogicLineDragObject) then
    with (Source as TLogicLineDragObject) do
    begin
      if LinkType = ltOutput then
      begin
        LogicLineID.LogicInputSource := Sender as TLogicInput;
      end;
    end;
end;

procedure TLogicInput.Paint;
var
  x,y : integer;
  DrawFlags : integer;
  BtnRect : TRect;
begin
  inherited Paint;
  x := Width-rNot*2;
  y := Height div 2;
  BtnRect := Rect(0,0,x,Height);
  //Draw Button
  if LogicMode = lmDesign then
  begin
    Canvas.Brush.Color := FColorBrush;
    Canvas.Pen.Color := FColorState;
    Canvas.Rectangle(BtnRect);
    InflateRect(BtnRect,-1,-2); //V2.28
  end
  else
  begin
    if FOneShot = true then //v2.28
    BtnRect := DrawButtonFace(Canvas,BtnRect,FBevelWidth, {clBtnHighlight=white}
                cl3DLight,clBtnShadow,clBtnFace,FMouseDown)
    else
    BtnRect := DrawButtonFace(Canvas,BtnRect,FBevelWidth,
                FColorState,FColorStateDk,FColorStateBk,FMouseDown);
  end;
  Canvas.TextOut(BtnRect.Left+1,BtnRect.Top+1,Self.Caption); //V2.28

  //Draw Output
  Canvas.Pen.Color := FColorOutput;
  Canvas.Brush.Color := FColorBrush; //v2.28
  if FInvertOut then
  begin
    Canvas.Ellipse(x,y-rNot,x+rNot*2,y+rNot);
  end else begin
    Canvas.Polyline([Point(x,y),Point(Width,y)]);
  end;
end;

{ DrawButtonFace
  - draws a 3D button with colored borders
  - returns the remaining usable area inside the Client rect.
 - based on DrawButtonFace in Buttons.pas }
function TLogicInput.DrawButtonFace(Canvas: TCanvas; const Client: TRect;
  BevelWidth: Integer; LightColor, DarkColor, BackColor: TColor;
  IsDown: Boolean): TRect;
var
  R: TRect;
  DC: THandle;
  MyPen : TPen;
  MyBrush : TBrush;
begin

  R := Client;
  with Canvas do
  begin
    //Save original Pen/Brush
    MyPen := Pen;
    MyBrush := Brush;
    //Rectangle() fills the button in Brush Color.  Outer rectangle is
    //replaced by Frame3D
    Pen.Color := clWindowFrame; 
    Brush.Color := BackColor; 
    Brush.Style := bsSolid;
    Rectangle(R.Left, R.Top, R.Right, R.Bottom);
    //Draw a 3D button
    if IsDown then
      Frame3D(Canvas, R, DarkColor, LightColor, BevelWidth)
    else
      Frame3D(Canvas, R, LightColor, DarkColor, BevelWidth);
    //Restore original Pen/Brush
    Brush := MyBrush;
    Pen := MyPen;
  end;

  Result := R;
  if IsDown then OffsetRect(Result, 1, 1);
end;

procedure TLogicInput.SetNewSize;
var
  TextRect : TRect;
begin
    if assigned(Parent) then
      TextRect := GetTextBounds
    else
      TextRect := Rect(0,0,0,0);
    Width  := GetGridPosition(Max(FMinWidth,TextRect.Right+6+rNot*2+FBevelWidth*2));
    Height := GetGridPosition(Max(FMinHeight,TextRect.Bottom+6));
    Invalidate;
    MoveConnectedOutputs;
end;

function TLogicInput.DetermineLogicalState(ALogicInput:Boolean):Boolean;
begin
  Result := not ALogicInput;
end;

procedure TLogicInput.InputDblClick(Sender:TObject);
begin
  LogicState := not LogicState;
  if (FOneShot and LogicState) then
    LogicState := false;
end;

procedure TLogicInput.liMouseDown(Sender: TObject; Button: TMouseButton;
          Shift: TShiftState; X, Y: Integer);
begin
  uoMouseDown(Sender, Button, Shift, X, Y);
  if (Button = mbLeft) and (LogicMode = lmRun) then
  begin
    FMouseDown := true;
    Invalidate;
    InputDblClick(Sender);
  end;
end;

procedure TLogicInput.liMouseUp(Sender: TObject; Button: TMouseButton;
          Shift: TShiftState; X, Y: Integer);
begin
  uoMouseUp(Sender, Button, Shift, X, Y);
  if (Button = mbLeft) and (LogicMode = lmRun) then
  begin
    FMouseDown := false;
    Invalidate; 
  end;
end;

procedure TLogicInput.SetOneShot(Value:boolean);
begin
  if Value <> FOneShot then
    begin
    FOneShot := Value;
    if FOneShot then
      begin
      if Self.Caption = 'Input' then Self.Caption := 'PB';
      if LogicState then InputDblClick(Self);
      end
    else
      if Self.Caption = 'PB' then Self.Caption := 'Input';
    Invalidate;
    end;
end;

procedure TLogicInput.puiInputMaint_OneShot(Sender: TObject);
begin
  OneShot := not OneShot;
end;

procedure TLogicInput.puiInputChangeDescription(Sender: TObject);
begin
  Caption := EditControlDescription(Self,Self.Caption);
  SetNewSize;
end;

procedure TLogicInput.puiInputInvertOutput(Sender: TObject);
begin
  with Self as TLogicInput do begin
    InvertOut := not InvertOut;
  end;
end;


procedure TLogicInput.OnLogicModeChange;
begin
{
  FPopupItem[0].Caption := 'Change description';
  FPopupItem[1].Caption := 'Change hint';
  FPopupItem[2].Caption := 'Invert Output';
  FPopupItem[3].Caption := 'Toggle Maintained/OneShot';
  FPopupItem[4].Caption := 'Add Output LogicLine';  //v2.28
  FPopupItem[5].Caption := 'Delete LogicElement';
}
  FPopupItem[4].Enabled := LogicMode = lmDesign; //v2.28
  FPopupItem[5].Enabled := LogicMode = lmDesign;
end;

{TLogicDelay}

constructor TLogicDelay.Create(AOwner: TComponent);
var
  i : integer;
begin
  inherited Create(AOwner);
  FMinWidth := 50;
  FMinHeight := 20;
  FDelayType := dtOff;
  FTiming := false;
  FTimer := TTimer.Create(self);
  FTimer.Name := 'LogicDelayTimer'; //v2.28
  FTimer.OnTimer := TimedOut;
  FTimer.Enabled := false;
  FTimer.Interval := 3000;
  SetCaption;
  Top := 50;
  Left := 50;
  Width := FMinWidth;
  Height := FMinHeight;
  Visible := true;

  //Popup Menu
  FpumLogicDelay := TPopUpMenu.Create(self);
  FpumLogicDelay.AutoPopup := true;
  SetLength(FpuiLogicDelay,8);
  for i := 0 to 7 do begin
    FpuiLogicDelay[i] := TMenuItem.Create(Self);
    FpumLogicDelay.Items.Add(FpuiLogicDelay[i]);
  end;
  FpuiLogicDelay[0].Caption := 'Toggle TOFF-TON Delay';
  FpuiLogicDelay[0].OnClick := puiLogDelToggleDelayType;
  FpuiLogicDelay[1].Caption := 'Change Hint';
  FpuiLogicDelay[1].OnClick := puiUnOpChangeHint;
  FpuiLogicDelay[2].Caption := 'Change Delay Time';
  FpuiLogicDelay[2].OnClick := puiLogDelSetDelay;
  FpuiLogicDelay[3].Caption := 'Invert Input';
  FpuiLogicDelay[3].OnClick := puiLogDelInvertInput;
  FpuiLogicDelay[4].Caption := 'Invert Output';
  FpuiLogicDelay[4].OnClick := puiLogDelInvertOutput;
  FpuiLogicDelay[5].Caption := 'Add Output LogicLine';  //v2.28
  FpuiLogicDelay[5].OnClick := puiUnOpAddOutputLogicLine;
  FpuiLogicDelay[6].Caption := 'Disconnect Input';
  FpuiLogicDelay[6].OnClick := puiUnOpDisconnectInput;
  FpuiLogicDelay[7].Caption := 'Delete LogicElement';
  FpuiLogicDelay[7].OnClick := puiUnOpFreeLogicElement;
  PopupMenu := FpumLogicDelay;

  OnDragDrop := LogicLineDragDrop;
  with Canvas do
  begin
    Pen.Style := psSolid;
    Pen.Color := clBlack;
    Brush.Style := bsClear;
  end;
end;

destructor TLogicDelay.Destroy;
begin
  try
    FTimer.Free;
    FTimer := nil; //V2.27
  except
  end;
  try
    FpumLogicDelay.Free;
    //also frees each instance of FpuiLogicDelay
  except
  end;
  try
    SetLength(FpuiLogicDelay,0);
  except
  end;
  inherited Destroy;
end;

procedure TLogicDelay.LogicLineDragDrop(Sender, Source: TObject; X,
  Y: Integer);
begin
  if IsDragObject(Source) and (Source is TLogicLineDragObject) then
    with (Source as TLogicLineDragObject) do
    begin
      if LinkType = ltInput then
      begin
        LogicInputSource := LogicLineID as TLogicLine;
      end else begin
        LogicLineID.LogicInputSource := Sender as TLogicDelay;
      end;
    end;
end;

procedure TLogicDelay.Paint;
var
  x0,x1,y : integer;
begin
  inherited Paint;
  y := Height div 2;
  //draw input
  x0 := 0;
  x1 := x0 + rNot*2;
  Canvas.Pen.Color := FColorInput;
  if FInvert then
  begin
    Canvas.Ellipse(x0,y-rNot,x1,y+rNot);
  end else begin
    Canvas.Polyline([Point(x0,y),Point(x1,y)]);
  end;
  //draw body
  x0 := x1;
  x1 := Width - rNot*2;
  Canvas.Pen.Color := FColorState;
  Canvas.Rectangle(x0,0,x1,Height);
  Canvas.TextOut(x0+1,3,Self.Caption); //v2.28
  //Draw Output
  x0 := x1;
  x1 := Width;
  Canvas.Pen.Color := FColorOutput;
  if FInvertOut then
  begin
    Canvas.Ellipse(x0,y-rNot,x1,y+rNot);
  end else begin
    Canvas.Polyline([Point(x0,y),Point(x1,y)]);
  end;
end;

function TLogicDelay.DetermineLogicalState(ALogicInput:Boolean):Boolean;
begin
  {
  Off Delay
  State
  -----
    Input	Timer	Logic
    ----- ----- -----
  Init (off)
    off	off	off
  Start
    on	off	on
  Stop (Begin Timing)
    off	on	on
  Timed Out
    off	off	off
  }
  Input := ALogicInput;
  if not Assigned(FTimer) then Result := false else   //V2.27
  case FDelayType of
  dtOff: begin
    if FInput then
      begin
      FTimer.Enabled := false;
      Result := true;
      end
    else
      begin
      FTimer.Enabled := true;
      Result := true;
      end;
    end;
  dtOn: begin
    if FInput then
      begin
      FTimer.Enabled := true;
      Result := false;
      end
    else
      begin
      FTimer.Enabled := false;
      Result := false;
      end;
    end;
  else
    Result := false; //unexpected state
  end; //case...
end;

procedure TLogicDelay.TimedOut(Sender:TObject);
begin
  FTimer.Enabled := false;
  Case FDelayType of
  dtOff: begin
    LogicState := false;
    end;
  dtOn: begin
    LogicState := true;
    end;
  end;
end;

procedure TLogicDelay.SetDelayType(Value:TDelayType);
begin
  if Value <> FDelayType then
    begin
    FDelayType := Value; {dtOn, dtOff}
    SetCaption;
  end;
end;

function TLogicDelay.GetInterval:Cardinal;
begin
  Result := FTimer.Interval;
end;

procedure TLogicDelay.SetInterval(Value:Cardinal);
begin
  If FTimer.Interval <> Value then
  begin
    //No need to check for negative timer value since cardinal prevents it
    FTimer.Interval := Value;
    SetCaption;
  end;
end;

procedure TLogicDelay.SetNewSize;
var
  TextRect : TRect;
begin
  if assigned(Parent) then
    TextRect := GetTextBounds
  else
    TextRect := Rect(0,0,0,0);
  Width  := GetGridPosition(Max(FMinWidth,TextRect.Right+6+rNot*4));
  Height := GetGridPosition(Max(FMinHeight,TextRect.Bottom+6));
  Invalidate;
  MoveConnectedOutputs;
end;

procedure TLogicDelay.SetCaption;
begin
    case FDelayType of
    dtOff: begin
      //Self.Caption := 'TOF ' + IntToStr(Interval div 1000);
      Self.Caption := 'TOF ' + format('%.1f', [Interval/1000.0]);
      end;
    dtOn: begin
      //Self.Caption := 'TON ' + IntToStr(Interval div 1000);
      Self.Caption := 'TON ' + format('%.1f', [Interval/1000.0]);
      end;
    end;
    SetNewSize;
end;

procedure TLogicDelay.puiLogDelToggleDelayType(Sender: TObject);
begin
  Case DelayType of
  dtOn: DelayType := dtOff;
  dtOff: DelayType := dtOn;
  end;
end;

procedure TLogicDelay.puiLogDelInvertInput(Sender: TObject);
begin
  with Self as TLogicDelay do begin
    Invert := not Invert;
  end;
end;

procedure TLogicDelay.puiLogDelInvertOutput(Sender: TObject);
begin
  with Self as TLogicDelay do begin
    InvertOut := not InvertOut;
  end;
end;

procedure TLogicDelay.puiLogDelSetDelay(Sender: TObject);
var
  sInterval : string;
  aCaption, aPrompt : string;
  iInterval : integer;
  OrigInterval : integer;
  rInterval : single;
  vCode : integer;
begin
  OrigInterval := Interval;
  //call dialog InputBox to get a new value for property NumReqd
  aCaption := 'Delay Timer Editor';
  aPrompt  := 'Enter the interval for ' + Self.Name + ' (in seconds > 0):';
  rInterval := Interval / 1000.0;
  sInterval := format('%.1f', [rInterval]);
  sInterval := InputBox(ACaption,APrompt,sInterval);
  try
    val(sInterval,rInterval,vCode);
    if vCode = 0 then
    begin
    iInterval := Round(rInterval*10)*100; //round to 1 decimal
    if iInterval < 100 then iInterval := 100;
    end else raise EMathError.Create('Invalid number');
  except
    on E : Exception do
      begin
      ShowMessage(E.ClassName + ': ' + E.Message + CRLF +
        sInterval + ' is not a good delay');
      iInterval := OrigInterval;
      end;
  end;
  Interval := iInterval;
{
  //call dialog InputBox to get a new value for property NumReqd
  aCaption := 'Delay Timer Editor';
  aPrompt  := 'Enter the interval for ' + Self.Name + ' (in seconds > 0):';
  sInterval := IntToStr(Interval div 1000);
  sInterval := InputBox(ACaption,APrompt,sInterval);
  try
    M := StrToInt(sInterval);
  except
    on E: EConvertError do
      begin
      ShowMessage(sInterval + ' is not a valid number' + CRLF +
        E.ClassName + CRLF + E.Message);
      M := 3;
      end;
    else
      M := 3;
  end;
  Interval := M*1000;
}
end;

procedure TLogicDelay.OnLogicModeChange;
begin
{
  FpuiLogicDelay[0].Caption := 'Toggle TOFF-TON Delay';
  FpuiLogicDelay[1].Caption := 'Change Hint';
  FpuiLogicDelay[2].Caption := 'Change Delay Time';
  FpuiLogicDelay[3].Caption := 'Invert Input';
  FpuiLogicDelay[4].Caption := 'Invert Output';
  FpuiLogicDelay[5].Caption := 'Add Output LogicLine'; //v2.28
  FpuiLogicDelay[6].Caption := 'Disconnect Input';
  FpuiLogicDelay[7].Caption := 'Delete LogicElement';
}
  FpuiLogicDelay[5].Enabled := LogicMode = lmDesign; //v2.28
  FpuiLogicDelay[6].Enabled := LogicMode = lmDesign;
  FpuiLogicDelay[7].Enabled := LogicMode = lmDesign;
end;

{TLogicOneShot}
constructor TLogicOneShot.Create(AOwner: TComponent);
var
  i : integer;
begin
  inherited Create(AOwner);
  FMinWidth := 30;
  FMinHeight := 20;
  Self.Caption := '1';
  Top := 50;
  Left := 50;
  Width := FMinWidth;
  Height := FMinHeight;
  Visible := true;

  //Popup Menu
  FpumOneShot := TPopUpMenu.Create(self);
  FpumOneShot.AutoPopup := true;
  SetLength(FpuiOneShot,5);
  for i := 0 to 4 do begin
    FpuiOneShot[i] := TMenuItem.Create(Self);
    FpumOneShot.Items.Add(FpuiOneShot[i]);
  end;
  FpuiOneShot[0].Caption := 'Invert Input';
  FpuiOneShot[0].OnClick := puiLogOneInvertInput;
  FpuiOneShot[1].Caption := 'Invert Output';
  FpuiOneShot[1].OnClick := puiLogOneInvertOutput;
  FpuiOneShot[2].Caption := 'Disconnect Input';
  FpuiOneShot[2].OnClick := puiUnOpDisconnectInput;
  FpuiOneShot[3].Caption := 'Add Output LogicLine'; //v2.28
  FpuiOneShot[3].OnClick := puiUnOpAddOutputLogicLine;
  FpuiOneShot[4].Caption := 'Delete LogicElement';
  FpuiOneShot[4].OnClick := puiUnOpFreeLogicElement;
  PopupMenu := FpumOneShot;

  OnDragDrop := LogicLineDragDrop;
  with Canvas do
  begin
    Pen.Style := psSolid;
    Pen.Color := clBlack;
    Brush.Style := bsClear;
  end;
end;

destructor TLogicOneShot.Destroy;
begin
  try
    FpumOneShot.Free;
    //also frees each instance menu items
  except
  end;
  try
    SetLength(FpuiOneShot,0);
  except
  end;
  inherited Destroy;
end;

function TLogicOneShot.DetermineLogicalState(
  ALogicInput: Boolean): Boolean;
begin
  Input := ALogicInput;
  if FInput and not LogicState then
    begin
    LogicState := true;
    end;
  Result := false;
end;

procedure TLogicOneShot.puiLogOneInvertInput(Sender: TObject);
begin
  with Self as TLogicOneShot do begin
    Invert := not Invert;
  end;
end;

procedure TLogicOneShot.puiLogOneInvertOutput(Sender: TObject);
begin
  with Self as TLogicOneShot do begin
    InvertOut := not InvertOut;
  end;
end;

procedure TLogicOneShot.LogicLineDragDrop(Sender, Source: TObject; X,
  Y: Integer);
begin
  if IsDragObject(Source) and (Source is TLogicLineDragObject) then
    with (Source as TLogicLineDragObject) do
    begin
      if LinkType = ltInput then
      begin
        LogicInputSource := LogicLineID as TLogicLine;
      end else begin
        LogicLineID.LogicInputSource := Sender as TLogicOneShot;
      end;
    end;
end;

procedure TLogicOneShot.Paint;
var
  x0,x1,y : integer;
begin
  inherited Paint;
  y := Height div 2;
  //draw input
  x0 := 0;
  x1 := x0 + rNot*2;
  Canvas.Pen.Color := FColorInput;
  if FInvert then
  begin
    Canvas.Ellipse(x0,y-rNot,x1,y+rNot);
  end else begin
    Canvas.Polyline([Point(x0,y),Point(x1,y)]);
  end;
  //draw body
  x0 := x1;
  x1 := Width - rNot*2;
  Canvas.Pen.Color := FColorState;
  Canvas.Rectangle(x0,0,x1,Height);
  Canvas.TextOut(x0+1,3,Self.Caption); //v2.28
  //Draw Output
  x0 := x1;
  x1 := Width;
  Canvas.Pen.Color := FColorOutput;
  if FInvertOut then
  begin
    Canvas.Ellipse(x0,y-rNot,x1,y+rNot);
  end else begin
    Canvas.Polyline([Point(x0,y),Point(x1,y)]);
  end;
end;

procedure TLogicOneShot.OnLogicModeChange;
begin
{
  FpuiOneShot[0].Caption := 'Invert Input';
  FpuiOneShot[1].Caption := 'Invert Output';
  FpuiOneShot[2].Caption := 'Disconnect Input';
  FpuiOneShot[3].Caption := 'Add Output LogicLine'; //v2.28
  FpuiOneShot[4].Caption := 'Delete LogicElement';
}
  FpuiOneShot[2].Enabled := LogicMode = lmDesign;
  FpuiOneShot[3].Enabled := LogicMode = lmDesign;
  FpuiOneShot[4].Enabled := LogicMode = lmDesign;
end;

procedure TuoOutputEvents.PrioritizeOutputs; //V2.26
var
  i, j : integer;
begin
  //set Priority from LogicLines
  for i:= 0 to Self.Count-1 do
  begin
    Items[i].Priority := Items[i].OutputControl.Priority;
  end;
  //Change Output order to match Self.Priority
  QuickSort(0,Count-1, ComparePriorities);
  //Synch Output receivers to output events
  UpdatePriorities;
end;

procedure TuoOutputEvents.QuickSort(L, R: Integer;
  SCompare: TCollectionSortCompare);  //V2.26
var
  I, J: Integer;
  P: Integer; //Priority Value, not position
  T: TuoOutputEvent;
begin
  if R < L then exit; //true if no OutputEvents defined
  repeat
    I := L;
    J := R;
    P := Items[(L + R) shr 1].FPriority;
    repeat
      while SCompare(Items[I].FPriority, P) < 0 do
        Inc(I);
      while SCompare(Items[J].FPriority, P) > 0 do
        Dec(J);
      if I <= J then
      begin
        Items[J].Index := I;
        if I+1 < R then Items[I+1].Index := J;
        {T := Items[I];
        Items[I] := Items[J];
        Items[J] := T; }
        Inc(I);
        Dec(J);
      end;
    until I > J;
    if L < J then
      QuickSort(L, J, SCompare);
    L := I;
  until I >= R;
end;

procedure TuoOutputEvents.UpdatePriorities; //V2.26
var
  i : integer;
begin
  for i := 0 to Self.Count-1 do
  begin
    //set FPriority for each output if not default
    if Items[i].FPriority <> -1 then
      Items[i].FPriority := i;
    //feed back priority to LogicLine
    TLogicLine(Items[i].OutputControl).Priority := Items[i].FPriority;
  end;
end;

procedure TuoOutputEvent.SetPriority(Value: Integer); //V2.26
begin
  if Value <> FPriority then
  begin
    FPriority := Value;
  end;
end;

initialization
RegisterClasses([TLogicUnaryOp, TLogicNot, TLogicInput, TLogicDelay, TLogicOneShot]);
finalization

end.
