unit LogicMultiOp;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, Math, stdctrls, TypInfo, Types, Menus, Globals, gcMultiPolyline
  ;

const
  //LogicMultiOp (lmo) dimensions: 50x50 Outer Dimension initially, but
  //height can change dynamically.
  //Horizontal positions
  lmoXi : Integer = 15; //End of Input, Start of Main
  lmoXo : Integer = 35; //End of Main, Start of Output
  //Vertical positions
  lmoYmin : Integer = 50; //minumum total height
  lmoYfst : Integer = 10; //first input
  lmoYint : Integer = 10; //typical height between connections
  //Circle radius
  lmoR  : Integer =  5; //Radius of "not"

type
  TmoOutputEvents = class;
  TLogicMultiOp = class;

  TiLine = Record    //logic input line
    P1 : TPoint;  //start point
    dX : integer; //length
    Selected : boolean; //if selected then draw highlighted
    end;

  TmoOutputEvent = class(TCollectionItem)
  private
    FOutputControl: TLogicLine;
    FOutputMove: TLogicMoveOutputPos;
    FOutputEvent: TLogicChangeEvent;
    FPriority: Integer;  //V2.26
    procedure SetOutputControl(Value: TLogicLine);
    procedure SetOutputEvent(Value: TLogicChangeEvent);
    procedure SetOutputMove(Value: TLogicMoveOutputPos);
    procedure SetPriority(Value: Integer);  //V2.26
  protected
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
  //published
    property OutputControl:TLogicLine read FOutputControl
      write SetOutputControl;
    property OutputEvent:TLogicChangeEvent read FOutputEvent
      write SetOutputEvent stored true default nil;
    property OutputMove:TLogicMoveOutputPos read FOutputMove
      write SetOutputMove default nil;
    property Priority:Integer read FPriority write SetPriority; //V2.26
  end;

  TmoOutputEvents = class(TCollection)
  private
    FOwner: TLogicMultiOp;
    function GetItem(Index: Integer): TmoOutputEvent;
    procedure SetItem(Index: Integer; Value: TmoOutputEvent);
    procedure UpdatePriorities;
    procedure QuickSort(L, R: Integer;
      SCompare: TCollectionSortCompare);  //V2.26
  protected
    function GetOwner: TPersistent; override;
    procedure PrioritizeOutputs; //V2.26
  public
    constructor Create(AOwner: TLogicMultiOp);
    function Add: TmoOutputEvent;
    function Owner: TLogicMultiOp;
  //published
    {If a class has a default property, you can access that property with the
    abbreviation object[index], which is equivalent to object.property[index].}
    property Items[Index: Integer]: TmoOutputEvent read GetItem
      write SetItem; default;
  end;

  TmoLogicInput = class(TCollectionItem)
  private
    FColorInput: TColor;
    FiLine: TiLine;
    FInput: Boolean;
    FInputRecursionCounter: integer; //V2.27
    FInvert: Boolean;
    FLogicInputSource: TLogicLine;
    procedure SetInput(Value: Boolean);
    procedure SetInvert(Value: Boolean);
    procedure SetLogicInputSource(Value: TLogicLine);
  protected
  public
    procedure Assign(Source: TPersistent); override;
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure ReceiveLogicChange(ALogicState, Init: Boolean);
    property Input: Boolean read FInput write SetInput;
  published
    property Invert:Boolean read FInvert
      write SetInvert default false;
    property LogicInputSource:TLogicLine read FLogicInputSource
      write SetLogicInputSource default nil;
  end;

  TmoLogicInputs = class(TCollection)
  private
    FOwner: TLogicMultiOp;
    function GetItem(Index: Integer): TmoLogicInput;
    procedure SetItem(Index: Integer; Value: TmoLogicInput);
  protected
    function GetOwner: TPersistent; override;
    procedure Update(Item: TCollectionItem); override;
  public
    constructor Create(AOwner: TLogicMultiOp);
    function Add: TmoLogicInput;
    function Owner: TLogicMultiOp;
    property Items[Index: Integer]: TmoLogicInput read GetItem
      write SetItem; default;
  end;

TLogicMultiOp = class(TCustomControl)
private
  // Dimensions for painting self
  DimBody : TRect;
  DimOutputLine : array[0..1] of TPoint;
  DimInputLines : array of array[0..1] of TPoint;

  FColorOutput : TColor;
  FColorState  : TColor;
  FGridSize: TGridSize;
  FGridEnabled: Boolean;
  FInitLogic : Boolean;
  FInvertOut : Boolean;
  FLogicInputs : TmoLogicInputs;
  FLogicMode : TLogicMode;
  FLogicState : Boolean;
  FOnLogicChange : TLogicChangeEvent;
  FOutputEvents : TmoOutputEvents;
  FOutputState : Boolean;
  FPopupInputMenu : TPopupMenu;
  FPopupInputItem : array of TMenuItem;
  FPopupMenu : TPopupMenu;
  FPopupItem : array of TMenuItem;
  FSelected   : Boolean;
  StartXY : TPoint;
  LastRect, ParentExtent: TRect;
  procedure OnLogicModeChange; dynamic;
  procedure AddOutput(ALogicLine:TLogicLine;
            ALogicChangeMethod:TLogicChangeEvent;
            ALogicMoveOutputMethod:TLogicMoveOutputPos);
  procedure DeleteOutput(ALogicLine:TLogicLine);
  procedure DeselectInput;
  function  GetPositionOnGrid(APos: Integer): Integer;
  procedure MoveConnectedIO;
  procedure MoveConnectedInputs;
  procedure MoveConnectedOutputs;
  procedure puiMulOpInvertInput(Sender: TObject);
  procedure puiMulOpMoveInputDown(Sender: TObject);
  procedure puiMulOpMoveInputUp(Sender: TObject);
  procedure puiMulOpInvertOutput(Sender: TObject);
  procedure puiMulOpDisconnectInput(Sender: TObject);
  procedure puiMulOpDeleteLogicElement(Sender: TObject);
  procedure puMenuHandler(Sender: TObject; MousePos: TPoint; var Handled: Boolean);
  function  SelectInput(X,Y:Integer):boolean;
  procedure SetGridEnabled(Value: Boolean);
  procedure SetGridSize(Value: TGridSize);
  procedure SetInitLogic(Init:Boolean);
  procedure SetInvertOut(Value:Boolean);
  procedure SetLogicInputs(Value: TmoLogicInputs);
  procedure SetLogicMode(Value:TLogicMode);
  procedure SetLogicState(State:Boolean);
  procedure SetOutputState(Value:Boolean);
  procedure SetOutputEvents(Value: TmoOutputEvents);
    procedure puiMulOpAddOutputLogicLine(Sender: TObject);

protected
  procedure DetermineLogicalState; dynamic;
  function  GetBoundsRect:TRect; dynamic;
  procedure Loaded; override; //v2.26
  procedure LogicLineDragOver(Sender, Source: TObject; X,
            Y: Integer; State: TDragState; var Accept: Boolean);
  procedure Paint; override; //Implements TCustomControl virtual method
  procedure PaintDimensionsCalc;
  procedure PaintDimensionsCalc2;
  procedure PaintNew;
  procedure PaintNew2;
  procedure SetSelected(Value:Boolean);
  procedure uoMouseDown(Sender: TObject; Button: TMouseButton;
    Shift: TShiftState; X, Y: Integer);
  procedure uoMouseUp(Sender: TObject; Button: TMouseButton;
    Shift: TShiftState; X, Y: Integer);
  procedure uoMouseMove(Sender: TObject; Shift: TShiftState;
    X, Y: Integer);
  property  InitLogic:Boolean read FInitLogic write SetInitLogic;
  property  OutputState:Boolean read FOutputState write SetOutputState;
  property  OnLogicChange: TLogicChangeEvent
            read FOnLogicChange write FOnLogicChange;
  property  Canvas;
  procedure LogicLineDragDrop(Sender, Source: TObject; X, Y: Integer);
public
  constructor Create(AOwner: TComponent); override;
  destructor  Destroy;override;
  procedure AssignOutput(ALogicLine:TLogicLine;
          ALogicChangeMethod:TLogicChangeEvent;
          ALogicMoveOutputMethod:TLogicMoveOutputPos;
          AnOutputAction:TOutputActionType);
  procedure ChangePriority(ALogicLine:TLogicLine;ALogicPriority:Integer);  //V2.26
  procedure Notification(AComponent:TComponent; Operation:TOperation); override;
  property  GridEnabled : boolean read FGridEnabled write SetGridEnabled;
  property  GridSize:TGridSize read FGridSize write SetGridSize;
  property  LogicMode:TLogicMode read FLogicMode write SetLogicMode;
  property  LogicState:Boolean read FLogicState write SetLogicState;
  property  OutputEvents: TmoOutputEvents read FOutputEvents write SetOutputEvents;
  property  Selected:Boolean read FSelected write SetSelected;
  property  OnDragDrop;
  property  OnDragOver;
  property  PopupMenu;
published
  property InvertOut:Boolean read FInvertOut write SetInvertOut stored true default false;
  property LogicInputs: TmoLogicInputs read FLogicInputs
            write SetLogicInputs;
end;

TLogicMultiAnd = class(TLogicMultiOp)
private
  { Private declarations }
protected
    procedure OnLogicModeChange; override;
  { Protected declarations }
  procedure  DetermineLogicalState; override;
  procedure Paint; override;
public
  { Public declarations }
  constructor Create(AOwner: TComponent); override;
  destructor  Destroy;override;
published
  { Published declarations }
end;

TLogicMultiOr = class(TLogicMultiOp)
private
  { Private declarations }
protected
  procedure OnLogicModeChange; override;
  { Protected declarations }
  procedure  DetermineLogicalState; override;
  procedure Paint; override; 
public
  { Public declarations }
  constructor Create(AOwner: TComponent); override;
  destructor  Destroy;override;
published
  { Published declarations }
end;

TLogicMultiMooN = class(TLogicMultiOp)
private
  { Private declarations }
  FNumReqd : integer;
  procedure OnLogicModeChange; override;
  function GetTextBounds(aString:string):TRect;
  procedure SetNumReqd(Value : integer);
  procedure puiMooNSetNumReqd(Sender: TObject);
protected
  { Protected declarations }
  procedure  DetermineLogicalState; override;
  procedure Paint; override; 
public
  { Public declarations }
  constructor Create(AOwner: TComponent); override;
  destructor  Destroy;override;
published
  { Published declarations }
  property NumReqd : integer read FNumReqd write SetNumReqd stored true default 2;
end;

implementation

uses LogicUnaryOp, LogicConnect, LogicWindow;

{ TOutputEvent based on TListColumn }

procedure TmoOutputEvent.Assign(Source: TPersistent);
var
  AOutputEvent: TmoOutputEvent;
begin
  if Source is TmoOutputEvent then
  begin
    AOutputEvent := TmoOutputEvent(Source);
    OutputControl := AOutputEvent.OutputControl;
    OutputEvent   := AOutputEvent.OutputEvent;
    OutputMove    := AOutputEvent.OutputMove;
  end
  else inherited Assign(Source);
end;

constructor TmoOutputEvent.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FOutputControl := nil; //V2.26
  FOutputEvent := nil;
  FOutputMove := nil;  //V2.26
  FPriority := -1; //V2.26
end;

destructor TmoOutputEvent.Destroy;
begin
  inherited Destroy;
end;

procedure TmoOutputEvent.SetOutputControl(Value: TLogicLine);
begin
  if FOutputControl <> Value then
  begin
    FOutputControl := Value;
  end;
end;

procedure TmoOutputEvent.SetOutputEvent(Value: TLogicChangeEvent);
begin
  if @FOutputEvent <> @Value then
  begin
    FOutputEvent := Value;
  end;
end;

procedure TmoOutputEvent.SetOutputMove(Value: TLogicMoveOutputPos);
begin
  if @FOutputMove <> @Value then
  begin
    FOutputMove := Value;
  end;
end;

{ TOutputEvents based on TListColumns }

constructor TmoOutputEvents.Create(AOwner: TLogicMultiOp);
begin
  inherited Create(TmoOutputEvent);
  FOwner := AOwner;
end;

function TmoOutputEvents.GetItem(Index: Integer): TmoOutputEvent;
begin
  Result := TmoOutputEvent(inherited GetItem(Index));
end;

procedure TmoOutputEvents.SetItem(Index: Integer; Value: TmoOutputEvent);
begin
  inherited SetItem(Index, Value);
end;

function TmoOutputEvents.Add: TmoOutputEvent;
begin
  Result := TmoOutputEvent(inherited Add);
end;

function TmoOutputEvents.Owner: TLogicMultiOp;
begin
  Result := FOwner;
end;

function TmoOutputEvents.GetOwner: TPersistent;
begin
  Result := FOwner;
end;


{ TLogicMultiOp }
procedure TLogicMultiOp.SetOutputEvents(Value: TmoOutputEvents);
begin
  FOutputEvents.Assign(Value);
end;

constructor TLogicMultiOp.Create(AOwner: TComponent);
var
  i : integer;
begin
  inherited Create(AOwner);
  FOutputEvents := TmoOutputEvents.Create(Self);
  FLogicInputs := TmoLogicInputs.Create(Self);
  FColorOutput := clBlack;
  FColorState  := clBlack;
  FGridEnabled := false;
  FGridSize    := 0;
  GridSize     := 5;
  GridEnabled  := true;
  Height := 50;
  Width  := 50;
  Color := clWhite;
  FInvertOut := false;
  FOutputState := true;
  FLogicState := false;
  FInitLogic := true;
  LogicState := false; //force logic/output update
  FLogicMode := lmDesign;
  with Canvas do
  begin
    Pen.Style := psSolid;
    Pen.Color := clBlack;
    Brush.Style := bsClear;
  end;

  //Popup Menu
  FPopupMenu := TPopUpMenu.Create(self);
  FPopupMenu.AutoPopup := false; 
  SetLength(FPopupItem,3);
  for i := 0 to 2 do begin
    FPopupItem[i] := TMenuItem.Create(Self);
    FPopupMenu.Items.Add(FPopupItem[i]);
  end;
  FPopupItem[0].Caption := 'Invert Output';
  FPopupItem[0].OnClick := puiMulOpInvertOutput;
  FPopupItem[1].Caption := 'Add Output LogicLine'; //v2.28
  FPopupItem[1].OnClick := puiMulOpAddOutputLogicLine;
  FPopupItem[2].Caption := 'Delete LogicElement';
  FPopupItem[2].OnClick := puiMulOpDeleteLogicElement;
  PopupMenu := FPopupMenu;
  OnContextPopup := puMenuHandler;

  //PopupInputMenu - specific to the input nearest X,Y click
  FPopupInputMenu := TPopupMenu.Create(self);
  FPopupInputMenu.AutoPopup := false;
  SetLength(FPopupInputItem,4);
  for i := 0 to 3 do begin
    FPopupInputItem[i] := TMenuItem.Create(Self);
    FPopupInputMenu.Items.Add(FPopupInputItem[i]);
  end;
  FPopupInputItem[0].Caption := 'Invert Input';
  FPopupInputItem[0].OnClick := puiMulOpInvertInput;
  FPopupInputItem[1].Caption := 'Move Input Up';
  FPopupInputItem[1].OnClick := puiMulOpMoveInputUp;
  FPopupInputItem[2].Caption := 'Move Input Down';
  FPopupInputItem[2].OnClick := puiMulOpMoveInputDown;
  FPopupInputItem[3].Caption := 'Disconnect Input';
  FPopupInputItem[3].OnClick := puiMulOpDisconnectInput;

  OnDragOver := LogicLineDragOver;
  OnMouseDown := uoMouseDown;
  OnMouseMove := uoMouseMove;
  OnMouseUp   := uoMouseUp;

  PaintDimensionsCalc2;
end;

destructor TLogicMultiOp.Destroy;
begin
  try
    FOutputEvents.Free;
  except
  end;
  try
    FLogicInputs.Free;
  except
  end;
  try
    FPopupMenu.Free;
    //also frees each instance of FpuiLogicDelay
  except
  end;
  try
    SetLength(FPopupItem,0);
  except
  end;
  try
    FPopupInputMenu.Free;
    //also frees each instance of FpuiLogicDelay
  except
  end;
  try
    SetLength(FPopupInputItem,0);
  except
  end;
  inherited Destroy;
end;

//This is an important tool for removing dependencies to linked components
procedure TLogicMultiOp.Notification(AComponent:TComponent ; Operation:TOperation);
var
  i : integer;
begin
  if (Operation=opRemove) then
    for i := 0 to LogicInputs.Count-1 do
      if (AComponent=LogicInputs[i].LogicInputSource) then
        begin
        //Unlink myself from the component being destroyed
        LogicInputs[i].LogicInputSource := nil;
        //Delete the TmoLogicInput and redraw  //V2.27
        LogicInputs[i].Free;
        PaintDimensionsCalc;
        Invalidate;
        MoveConnectedIO;
        break;
        end;
  inherited Notification(Acomponent,Operation);
end;

procedure TLogicMultiOp.LogicLineDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
end;

procedure TLogicMultiOp.SetInvertOut(Value:Boolean);
begin
  if Value <> FInvertOut then
  begin
    FInvertOut := Value;
    InitLogic := True;
    LogicState := LogicState;
  end;
end;

procedure TLogicMultiOp.AssignOutput(ALogicLine:TLogicLine;
          ALogicChangeMethod:TLogicChangeEvent;
          ALogicMoveOutputMethod:TLogicMoveOutputPos;
          AnOutputAction:TOutputActionType);
begin 
  case AnOutputAction of
  oaAdd:
    //add ALogicChange (pointer to an OnLogicChange event)
    // to a list of TLogicChangeEvents
    AddOutput(ALogicLine,ALogicChangeMethod,ALogicMoveOutputMethod);
  oaDelete:
    //Lookup item = ALogicChangeMethod and delete it
    DeleteOutput(ALogicLine);
  end; //case
end;

//Override this function to perform custom logic
procedure TLogicMultiOp.DetermineLogicalState;
begin
  ;
end;

function TLogicMultiOp.GetPositionOnGrid(APos:Integer):Integer;
var
  Delta : integer;
begin
  if GridEnabled and (GridSize > 0) then
    begin
    Delta := APos mod GridSize;
    if (Delta / GridSize) >= 0.5 then
      Result := APos + GridSize - Delta
    else
      Result := APos - Delta;
    end
  else
    Result := APos;
end;

procedure TLogicMultiOp.SetGridEnabled(Value:Boolean);
begin
  if Value <> FGridEnabled then
    if not FGridEnabled then  //object may not have been on grid
      begin
      FGridEnabled := Value;
      Left := GetPositionOnGrid(Left);
      Top  := GetPositionOnGrid(Top);
      end
  else
      FGridEnabled := Value;
end;

procedure TLogicMultiOp.SetGridSize(Value:TGridSize);
begin
  if Value <> FGridSize then
  begin
    FGridSize := Value;
  end;
end;

procedure TLogicMultiOp.SetInitLogic(Init:Boolean);
begin
  FInitLogic := Init;
end;

procedure TLogicMultiOp.SetLogicState(State:Boolean);
var
  i : integer;
  AnOutputEvent : TmoOutputEvent;
begin
  if (State <> FLogicState) or InitLogic then
  begin
    FLogicState := State;
    OutputState := FLogicState xor InvertOut;
    for i := 0 to OutputEvents.Count-1 do
    begin
      try
      AnOutputEvent := OutputEvents.Items[i];
      AnOutputEvent.OutputEvent(OutputState,false);
      except
      end;
    end;
    If InitLogic then InitLogic := false;
    If FLogicState then FColorState := clLime else FColorState := clRed;
    Invalidate;
    if Assigned(OnLogicChange) then OnLogicChange(OutputState,FInitLogic);
  end;
end;

procedure TLogicMultiOp.SetOutputState(Value:Boolean);
begin
  if Value <> FOutputState then
  begin
    FOutputState := Value;
    if FOutputState then FColorOutput := clLime else FColorOutput := clRed;
    invalidate;
  end;
end;

procedure TLogicMultiOp.AddOutput(ALogicLine:TLogicLine;
          ALogicChangeMethod:TLogicChangeEvent;
          ALogicMoveOutputMethod:TLogicMoveOutputPos);
var
  AnOutputEvent : TmoOutputEvent;
begin
  //add ALogicChangeMethod (pointer to an LogicChange event)
  // to a list of TLogicChangeEvents
  AnOutputEvent := OutputEvents.Add;
  AnOutputEvent.OutputControl := ALogicLine;
  AnOutputEvent.OutputEvent := ALogicChangeMethod;
  AnOutputEvent.OutputMove  := ALogicMoveOutputMethod;
  //Connect the output
  MoveConnectedOutputs;
  //Initialize the output
  InitLogic := True;
  LogicState := LogicState;
end;

procedure TLogicMultiOp.DeleteOutput(ALogicLine:TLogicLine);
var
  i : integer;
begin
  //Lookup item = ALogicChangeMethod and delete it
  for i:=OutputEvents.Count-1 downto 0 do
  begin
    if OutputEvents[i].OutputControl = ALogicLine then
      OutputEvents.Delete(i);
  end;
end;

procedure TLogicMultiOp.ChangePriority(ALogicLine: TLogicLine;
  ALogicPriority:Integer);  //V2.26
var
  i : integer;
begin
  //Lookup ALogicLine and change the priority of its output event
  for i:=0 to OutputEvents.Count-1 do
  begin
    if OutputEvents[i].OutputControl = ALogicLine then
    begin
      OutputEvents[i].Priority := ALogicPriority;
      OutputEvents.PrioritizeOutputs;
    end;
  end;
end;

function TLogicMultiOp.SelectInput(X,Y:Integer):boolean;
var
  i : integer;
  Xa,Ya,Xb,Yb : integer;
  Hit : boolean;
begin
  //iterate thru LogicInputs and select one if IsXYOnLine()
  Hit := false;
  for i := 0 to LogicInputs.Count-1 do
  begin
    if not Hit then
    begin
      with LogicInputs[i].FiLine do
      begin
        Xa := P1.X;
        Ya := P1.Y;
        Xb := dX;
        Yb := Ya;
      end;
      if IsXYOnLine(X,Y,Xa,Ya,Xb,Yb,4) then
      begin
        Hit := true;
        LogicInputs[i].FiLine.Selected := true;
      end;
    end
    else
      LogicInputs[i].FiLine.Selected := false;
  end;
  Result := Hit;
end;

procedure TLogicMultiOp.DeselectInput;
var
  i : integer;
begin
  //iterate thru LogicInputs and deselect them
  for i := 0 to LogicInputs.Count-1 do
    LogicInputs[i].FiLine.Selected := false;
end;

procedure TLogicMultiOp.Paint;
var
  i : integer;
begin
  inherited Paint;
  if Selected then Canvas.Rectangle(0,0,Width,Height);
  PaintDimensionsCalc2;
  PaintNew2;
end;

function TLogicMultiOp.GetBoundsRect:TRect;
var
  i,n,Xi,Yi,Xo,Yo : integer;
  ll : TLogicLine;
begin
  Result := BoundsRect;
end;

procedure TLogicMultiOp.uoMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  MyPos : TPoint;
begin
  if LogicMode = lmDesign then
  begin
    if Button = mbLeft then
      begin
      Selected := true;
      StartXY := ClientToParent(Point(X,Y),TPanel(Parent));
      ParentExtent := GetDeltaRect(TScrollBox(Parent.Parent).ClientRect,
        Point(-TPanel(Parent).Left,-TPanel(Parent).Top));
      LastRect:= GetBoundsRect;
      end;
  end;
  //Call popup menu if simple right click
  if (Button = mbRight) and (not (ssCtrl in Shift)) and
    (not (ssDouble in Shift)) then
  begin
    MyPos := ClientToScreen(Point(X,Y));
    //Don't allow move up or down if input is first or last
    //Call FPopupMenu if X is on right side of LogicElement, otherwise
    //call FPopupInputMenu.
    if X > 15 then
      FPopupMenu.Popup(MyPos.X,MyPos.Y)
    else
      //Identify the selected input and call the Popup then deselect
      if SelectInput(X,Y) then
      begin
        Repaint;
        FPopupInputMenu.Popup(MyPos.X,MyPos.Y);
        Application.ProcessMessages;
        DeselectInput;
        Repaint;
      end;
  end;
end;

procedure TLogicMultiOp.uoMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if (Button = mbLeft) and (LogicMode = lmDesign) then
  begin
    Selected := false;
  end;
end;

procedure TLogicMultiOp.uoMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
var
  AbsXY, DeltaXY : TPoint;
  NewExtent : TRect;
begin
  if (ssLeft in Shift) and (LogicMode = lmDesign) then { make sure button is down }
  begin
    AbsXY := ClientToParent(Point(X,Y),TPanel(Parent));
    DeltaXY.X := GetPositionOnGrid(AbsXY.X - StartXY.X);
    DeltaXY.Y := GetPositionOnGrid(AbsXY.Y - StartXY.Y);
    NewExtent := GetDeltaRect(LastRect,DeltaXY);
    if ( PtInRect(ParentExtent, NewExtent.TopLeft) and
         PtInRect(ParentExtent, NewExtent.BottomRight) ) then
      begin
      Inc(StartXY.X,DeltaXY.X);
      Inc(StartXY.Y,DeltaXY.Y);
      end
    else
      begin
      if NewExtent.Left < ParentExtent.Left then
        DeltaXY.X := GetPositionOnGrid(ParentExtent.Left - LastRect.Left);
      if NewExtent.Right > ParentExtent.Right then
        DeltaXY.X := GetPositionOnGrid(ParentExtent.Right - LastRect.Right);
      if NewExtent.Top < ParentExtent.Top then
        DeltaXY.Y := GetPositionOnGrid(ParentExtent.Top - LastRect.Top);
      if NewExtent.Bottom > ParentExtent.Bottom then
        DeltaXY.Y := GetPositionOnGrid(ParentExtent.Bottom - LastRect.Bottom);
      NewExtent := GetDeltaRect(LastRect,DeltaXY);
      Inc(StartXY.X,DeltaXY.X);
      Inc(StartXY.Y,DeltaXY.Y);
      end;
    Self.Left := NewExtent.Left;
    Self.Top  := NewExtent.Top;
    LastRect := NewExtent;
    MoveConnectedIO;
  end;
end;

procedure TLogicMultiOp.MoveConnectedIO;
begin
  MoveConnectedInputs;
  MoveConnectedOutputs;
end;

procedure TLogicMultiOp.MoveConnectedInputs;
var
  X,Y : integer;
  i : integer;
begin
  X := Left;
  Y := Top;
  for i := 0 to LogicInputs.Count-1 do
    if Assigned(LogicInputs[i].LogicInputSource) then
      if LogicInputs[i].LogicInputSource is TLogicLine then
        TLogicLine(LogicInputs[i].LogicInputSource).PlacePolyPoint(-1,X,
          Y+DimInputLines[i][0].Y);
end;

procedure TLogicMultiOp.MoveConnectedOutputs;
var
  i : integer;
  Xo,Yo : integer;
  AnOutputMove : TmoOutputEvent;
begin
  Xo := Left + Width;
  Yo := Top  + Height div 2;
  LogicForm.StatusBar.Panels[0].Text := 'Output position X = ' +
    IntToStr(Xo) + ', Y = ' + IntToStr(Yo);
  for i := 0 to OutputEvents.Count-1 do
  begin
    try
    AnOutputMove := OutputEvents.Items[i];
    AnOutputMove.OutputMove(0,Xo,Yo);
    except
    end;
  end;
end;

procedure TLogicMultiOp.SetSelected(Value:Boolean);
begin
  if Value <> FSelected then
  begin
    FSelected := Value;
    Invalidate;
  end;
end;

procedure TLogicMultiOp.puiMulOpDisconnectInput(Sender: TObject);
var
  i : integer;
begin
  for i := 0 to LogicInputs.Count-1 do
    if LogicInputs[i].FiLine.Selected then
    begin
      LogicInputs[i].LogicInputSource := nil;
      LogicInputs[i].Free;
      PaintDimensionsCalc;
      Invalidate;
      MoveConnectedIO;
      break;
    end;
end;

procedure TLogicMultiOp.puiMulOpInvertInput(Sender: TObject);
var
  i : integer;
begin
  for i := 0 to LogicInputs.Count-1 do
    if LogicInputs[i].FiLine.Selected then
      LogicInputs[i].Invert := not LogicInputs[i].Invert;
end;

procedure TLogicMultiOp.puiMulOpMoveInputDown(Sender: TObject);
var
  i : integer;
begin
  for i := 0 to LogicInputs.Count-1 do
    if LogicInputs[i].FiLine.Selected then
    begin
      if i < LogicInputs.Count-1 then
      begin
        LogicInputs[i].Index := i+1;
        break;
      end;
    end;
  PaintDimensionsCalc2;
  MoveConnectedIO;
end;

procedure TLogicMultiOp.puiMulOpMoveInputUp(Sender: TObject);
var
  i : integer;
begin
  for i := 0 to LogicInputs.Count-1 do
    if LogicInputs[i].FiLine.Selected then
    begin
      if i > 0 then
      begin
        LogicInputs[i].Index := i-1;
        break;
      end;
    end;
  PaintDimensionsCalc2;
  MoveConnectedIO;
end;

procedure TLogicMultiOp.puiMulOpInvertOutput(Sender: TObject);
begin
  with Self as TLogicMultiOp do begin
    InvertOut := not InvertOut;
  end;
end;

procedure TLogicMultiOp.puiMulOpAddOutputLogicLine(Sender: TObject); //v2.28
var
  Xo,Yo : integer;
  ALogicLine : TLogicLine;
begin
  Xo := Left + Width;
  Yo := Top  + (Height div 2);
  ALogicLine := MultiPolyline.ArrayToNewPolyline([point(Xo,Yo),
    point(Xo+30,Yo+30)]);
  ALogicLine.Selected := true;
  LogicCount.Count:=LogicCount.Count+1;
  ALogicLine.Name := 'LogicLine' + IntToStr(LogicCount.Count);
  ALogicLine.LogicInputSource := Self as TLogicMultiOp;
end;

procedure TLogicMultiOp.puiMulOpDeleteLogicElement(Sender: TObject);
var
  i : integer;
begin
  //Set priority of output logiclines to -1  //V2.26
  for i := FOutputEvents.Count-1 downto 0 do
  begin
    FOutputEvents[i].Priority := -1;
    if FOutputEvents[i].FOutputControl.OutputEvents.Count = 0 then  //v2.28
      FOutputEvents[i].FOutputControl.Free;
  end;
  FOutputEvents.UpdatePriorities;

  TLogicMultiOp(Self).Free;
end;

procedure TLogicMultiOp.puMenuHandler(Sender: TObject; MousePos: TPoint;
  var Handled: Boolean);
begin
  Handled := true;
end;

procedure TLogicMultiOp.SetLogicInputs(Value: TmoLogicInputs);
begin
  FLogicInputs.Assign(Value);
end;

procedure TLogicMultiOp.LogicLineDragDrop(Sender, Source: TObject; X,
  Y: Integer);
var
  MyInput : TmoLogicInput;
begin
  if IsDragObject(Source) and (Source is TLogicLineDragObject) then
    with (Source as TLogicLineDragObject) do
    begin
      if LinkType = ltInput then
      begin
        MyInput := LogicInputs.Add;
        MyInput.LogicInputSource := LogicLineID as TLogicLine
      end else begin
        LogicLineID.LogicInputSource := Sender as TLogicMultiOp;
      end;
    end;
end;

procedure TLogicMultiOp.PaintDimensionsCalc;
var
  i : integer;
  Y : integer;
begin

  case LogicInputs.Count of
  0..2:
  begin
    Height := lmoYmin;
    DimBody := Rect(lmoXi,0,lmoXo,Height);
    SetLength(DimInputLines,2);
    if (LogicInputs.Count > 0) and LogicInputs[0].FInvert then
    begin
      DimInputLines[0][0] := Point(lmoR,lmoYfst);
      DimInputLines[0][1] := Point(lmoXi,lmoYfst);
    end
    else
    begin
      DimInputLines[0][0] := Point(0,lmoYfst);
      DimInputLines[0][1] := Point(lmoXi,lmoYfst);
    end;
    if (LogicInputs.Count > 1) then if LogicInputs[1].FInvert then
    begin
      DimInputLines[1][0] := Point(lmoR,lmoYmin-lmoYfst);
      DimInputLines[1][1] := Point(lmoXi,lmoYmin-lmoYfst);
    end
    else
    begin
      DimInputLines[1][0] := Point(0,lmoYmin-lmoYfst);
      DimInputLines[1][1] := Point(lmoXi,lmoYmin-lmoYfst);
    end;
  end;
  3:
  begin
    Height := lmoYmin;
    DimBody := Rect(lmoXi,0,lmoXo,Height);
    SetLength(DimInputLines,3);
    for i := 0 to 2 do
    begin
      case i of
      0: Y:=lmoYfst;
      1: Y:=lmoYmin div 2;
      2: Y:=lmoYmin-lmoYfst;
      end;
      if LogicInputs[i].FInvert then
      begin
        DimInputLines[i][0] := Point(lmoR,Y);
        DimInputLines[i][1] := Point(lmoXi,Y);
      end
      else
      begin
        DimInputLines[i][0] := Point(0,Y);
        DimInputLines[i][1] := Point(lmoXi,Y);
      end;
    end;
  end;
  4:
  begin
    Height := lmoYmin;
    DimBody := Rect(lmoXi,0,lmoXo,Height);
    SetLength(DimInputLines,4);
    for i := 0 to 3 do
    begin
      Y := lmoYfst + lmoYint*i;
      if LogicInputs[i].FInvert then
      begin
        DimInputLines[i][0] := Point(lmoR,Y);
        DimInputLines[i][1] := Point(lmoXi,Y);
      end
      else
      begin
        DimInputLines[i][0] := Point(0,Y);
        DimInputLines[i][1] := Point(lmoXi,Y);
      end;
    end;
  end;
  else //all TLogicMultiOp logic elements with more than 4 inputs
    Height := lmoYfst+LogicInputs.Count*lmoYint;
    DimBody := Rect(lmoXi,0,lmoXo,Height);
    SetLength(DimInputLines,LogicInputs.Count);
    for i := 0 to LogicInputs.Count-1 do
    begin
      Y := 10 + 10*i;
      if LogicInputs[i].FInvert then
      begin
        DimInputLines[i][0] := Point(lmoR,Y);
        DimInputLines[i][1] := Point(lmoXi,Y);
      end
      else
      begin
        DimInputLines[i][0] := Point(0,Y);
        DimInputLines[i][1] := Point(lmoXi,Y);
      end;
    end;
  end;

  Y := Height div 2;
  DimOutputLine[0] := Point(lmoXo,Y);
  DimOutputLine[1] := Point(Width,Y);
end;

procedure TLogicMultiOp.PaintDimensionsCalc2;
var
  i : integer;
  Y : integer;
begin
  case LogicInputs.Count of
  0..2:
  begin
    Height := lmoYmin;
    DimBody := Rect(lmoXi,0,lmoXo,Height);
    for i := 0 to LogicInputs.Count-1 do
    begin
      case i of
        0: Y:=lmoYfst;
        1: Y:=lmoYmin-lmoYfst;
      end;
      LogicInputs[i].FiLine.P1 := Point(0,Y);
      LogicInputs[i].FiLine.dX := lmoXi;
    end;
  end;
  3:
  begin
    Height := lmoYmin;
    DimBody := Rect(lmoXi,0,lmoXo,Height);
    for i := 0 to 2 do
    begin
      case i of
      0: Y:=lmoYfst;
      1: Y:=lmoYmin div 2;
      2: Y:=lmoYmin-lmoYfst;
      end;
      LogicInputs[i].FiLine.P1 := Point(0,Y);
      LogicInputs[i].FiLine.dX := lmoXi;
    end;
  end;
  4:
  begin
    Height := lmoYmin;
    DimBody := Rect(lmoXi,0,lmoXo,Height);
    for i := 0 to 3 do
    begin
      Y := lmoYfst + lmoYint*i;
      LogicInputs[i].FiLine.P1 := Point(0,Y);
      LogicInputs[i].FiLine.dX := lmoXi;
    end;
  end;
  else //all TLogicMultiOp logic elements with more than 4 inputs
    Height := lmoYfst+LogicInputs.Count*lmoYint;
    DimBody := Rect(lmoXi,0,lmoXo,Height);
    for i := 0 to LogicInputs.Count-1 do
    begin
      Y := 10 + 10*i;
      LogicInputs[i].FiLine.P1 := Point(0,Y);
      LogicInputs[i].FiLine.dX := lmoXi;
    end;
  end;

  Y := Height div 2;
  DimOutputLine[0] := Point(lmoXo,Y);
  DimOutputLine[1] := Point(Width,Y);
end;

procedure TLogicMultiOp.PaintNew;
var
  i,x,y : integer;
begin

  //Draw main body in LogicState color
  Canvas.Pen.Color := FColorState;
  Canvas.Rectangle(DimBody);

  //Draw each Input
  for i := 0 to LogicInputs.Count-1 do
  begin
    Canvas.Pen.Color := LogicInputs[i].FColorInput;
    Canvas.Polyline(DimInputLines[i]);
    if LogicInputs[i].FInvert then
    begin
      x := DimInputLines[i][0].X;
      y := DimInputLines[i][0].Y;
      Canvas.Ellipse(x-lmoR,y-lmoR,x+lmoR,y+lmoR);
    end;
  end;
  //Draw Output
  Canvas.Pen.Color := FColorOutput;
  Canvas.Polyline(DimOutputLine);
  if FInvertOut then
  begin
      x := DimOutputLine[0].X;
      y := DimOutputLine[0].Y;
      Canvas.Ellipse(x-lmoR,y-lmoR,x+lmoR,y+lmoR);
  end;
end;

procedure TLogicMultiOp.PaintNew2;
var
  i,x,y : integer;
  P1i,P2 : TPoint;
  O1,O2 : TPoint;
begin

  //Draw main body in LogicState color
  Canvas.Pen.Color := FColorState;
  Canvas.Rectangle(DimBody);

  //Draw each Input
  for i := 0 to LogicInputs.Count-1 do
  begin
    Canvas.Pen.Color := LogicInputs[i].FColorInput;
    if LogicInputs[i].FiLine.Selected then
      Canvas.Pen.Width := 2
    else
      Canvas.Pen.Width := 1;
    if LogicInputs[i].FInvert then
    begin
      with LogicInputs[i].FiLine do
      begin
        P1i := Point(P1.X+2*lmoR,P1.Y);
        P2 := Point(P1.X+dX,P1.Y);
        Canvas.Polyline([P1i,P2]);
        x := P1.X;
        y := P1.Y;
        Canvas.Ellipse(x,y-lmoR,x+2*lmoR,y+lmoR);
      end;
    end else begin
      with LogicInputs[i].FiLine do
        P2 := Point(P1.X+dX,P1.Y);
      Canvas.Polyline([LogicInputs[i].FiLine.P1,P2]);
    end;
  end;
  Canvas.Pen.Width := 1;

  //Draw Output
  Canvas.Pen.Color := FColorOutput;
  if FInvertOut then
  begin
    O1 := DimOutputLine[0];
    O2 := DimOutputLine[1];
    O2.X := O2.X-2*lmoR;
    Canvas.Polyline([O1,O2]);
    x := O2.X;
    y := O2.Y;
    Canvas.Ellipse(x,y-lmoR,x+2*lmoR,y+lmoR);
  end else begin
    Canvas.Polyline(DimOutputLine);
  end;
end;

procedure TLogicMultiOp.SetLogicMode(Value:TLogicMode);
begin
  if Value <> FLogicMode then
  begin
    FLogicMode := Value;
    OnLogicModeChange;
  end;
end;

procedure TLogicMultiOp.OnLogicModeChange;
begin
  //Override in descendants to update popup menus
end;

procedure TLogicMultiOp.Loaded;  //v2.26
begin
  inherited;
  FOutputEvents.PrioritizeOutputs;
end;

{TLogicMultiAnd}

constructor TLogicMultiAnd.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Top := 50;
  Left := 50;
  Visible := true;
  OnDragDrop := LogicLineDragDrop;
end;

destructor TLogicMultiAnd.Destroy;
begin
  //Inherited TLogicMultiOp unlinks all inputs
  inherited Destroy;
end;

{
procedure TLogicMultiAnd.LogicLineDragDrop(Sender, Source: TObject; X,
  Y: Integer);
begin
  if IsDragObject(Source) and (Source is TLogicLineDragObject) then
    with (Source as TLogicLineDragObject) do
    begin
      if LinkType = ltInput then
      begin
        //X and Y are relative to the Control's upperleft corner
        if ( Y < (Height div 2) ) then
        begin
          LogicInputSource1 := LogicLineID as TLogicLine;
        end else begin
          LogicInputSource2 := LogicLineID as TLogicLine;
        end;
      end else begin
        LogicLineID.LogicInputSource := Sender as TLogicMultiAnd;
      end;
    end;
end;
}

procedure TLogicMultiAnd.Paint;
var
  x,y : integer;
begin
  inherited Paint;
  Canvas.TextOut((Width div 2)-3, (Height div 2)-6,'A');
end;

procedure TLogicMultiAnd.DetermineLogicalState;
var
  i : integer;
  CalcState : boolean;
begin
  CalcState := true;
  for i := 0 to LogicInputs.Count-1 do
    CalcState := CalcState and LogicInputs[i].Input;
  LogicState := CalcState;
end;

procedure TLogicMultiAnd.OnLogicModeChange;
begin
{
  FPopupItem[0].Caption := 'Invert Output';
  FPopupItem[1].Caption := 'Add Output LogicLine'; //v2.28
  FPopupItem[2].Caption := 'Delete LogicElement';
  FPopupInputItem[0].Caption := 'Invert Input';
  FPopupInputItem[1].Caption := 'Move Input Up';
  FPopupInputItem[2].Caption := 'Move Input Down';
  FPopupInputItem[3].Caption := 'Disconnect Input';
}
  FPopupItem[1].Enabled := LogicMode = lmDesign;
  FPopupItem[2].Enabled := LogicMode = lmDesign; //v2.28
  FPopupInputItem[1].Enabled := LogicMode = lmDesign;
  FPopupInputItem[2].Enabled := LogicMode = lmDesign;
  FPopupInputItem[3].Enabled := LogicMode = lmDesign;
end;

{TLogicMultiOr}

constructor TLogicMultiOr.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Top := 50;
  Left := 50;
  Visible := true;
  OnDragDrop := LogicLineDragDrop;
end;

destructor TLogicMultiOr.Destroy;
begin
  //Inherited TLogicMultiOp unlinks all inputs
  inherited Destroy;
end;

procedure TLogicMultiOr.Paint;
var
  x,y : integer;
begin
  inherited Paint;
  Canvas.TextOut((Width div 2)-3, (Height div 2)-6,'O');
end;

procedure TLogicMultiOr.DetermineLogicalState;
var
  i : integer;
  CalcState : boolean;
begin
  CalcState := false;
  for i := 0 to LogicInputs.Count-1 do
    CalcState := CalcState or LogicInputs[i].Input;
  LogicState := CalcState;
end;

{TLogicMultiMooN (M inputs out of N)}

constructor TLogicMultiMooN.Create(AOwner: TComponent);
var
  i : integer;
begin
  inherited Create(AOwner);
  FNumReqd := 2;
  Top := 50;
  Left := 50;
  Visible := true;
  OnDragDrop := LogicLineDragDrop;

  //Extend the Popup Menu by 1
  SetLength(FPopupItem,4);  //v2.28
  for i := 3 to 3 do begin
    FPopupItem[i] := TMenuItem.Create(Self);
    FPopupMenu.Items.Add(FPopupItem[i]);
  end;
  FPopupItem[3].Caption := 'Set number of required inputs';
  FPopupItem[3].OnClick := puiMooNSetNumReqd;

end;

destructor TLogicMultiMooN.Destroy;
begin
  //Inherited TLogicMultiOp unlinks all inputs
  inherited Destroy;
end;

procedure TLogicMultiMooN.Paint;
var
  x,y : integer;
  rCapt : TRect;
  cTop,cLeft : integer;
begin
  inherited Paint;
  //Draw main body in LogicState color
  cTop := 1;
  Canvas.TextOut((Width div 2)-3, cTop, 'A');
  rCapt := GetTextBounds('A');
  inc(cTop,rCapt.Bottom * 2 div 3);
  Canvas.TextOut((Width div 2)-3, cTop, 'n');
  rCapt := GetTextBounds('n');
  inc(cTop,rCapt.Bottom * 2 div 3);
  Canvas.TextOut((Width div 2)-3, cTop, 'y');
  rCapt := GetTextBounds('y');
  inc(cTop,rCapt.Bottom + 1);
  Canvas.TextOut((Width div 2)-3, cTop, IntToStr(FNumReqd));
end;

procedure TLogicMultiMooN.DetermineLogicalState;
var
  i : integer;
  CalcState : boolean;
  NumTripped : integer;
begin
  NumTripped := 0;
  CalcState := false;
  for i := 0 to LogicInputs.Count-1 do
    if LogicInputs[i].Input then inc(NumTripped);
  if NumTripped >= FNumReqd then CalcState := true;
  LogicState := CalcState;
end;

function TLogicMultiMooN.GetTextBounds(aString:string):TRect;
begin
  Result := Rect(0,0,0,0);
  Windows.DrawText(Self.Canvas.Handle, PChar(aString), -1,
    Result, DT_CALCRECT);
end;

procedure TLogicMultiMooN.SetNumReqd(Value:integer);
begin
  if Value <> FNumReqd then
  begin
    FNumReqd := Value;
    if FNumReqd > 9 then
      FNumReqd := 9
    else if FNumReqd < 1 then
      FNumReqd := 1;
    InitLogic := True;
    DetermineLogicalState;
  end;
end;


procedure TLogicMultiOr.OnLogicModeChange;
begin
{
  FPopupItem[0].Caption := 'Invert Output';
  FPopupItem[1].Caption := 'Add Output LogicLine'; //v2.28
  FPopupItem[1].Caption := 'Delete LogicElement';
  FPopupInputItem[0].Caption := 'Invert Input';
  FPopupInputItem[1].Caption := 'Move Input Up';
  FPopupInputItem[2].Caption := 'Move Input Down';
  FPopupInputItem[3].Caption := 'Disconnect Input';
}
  FPopupItem[1].Enabled := LogicMode = lmDesign;
  FPopupItem[2].Enabled := LogicMode = lmDesign;
  FPopupInputItem[1].Enabled := LogicMode = lmDesign;
  FPopupInputItem[2].Enabled := LogicMode = lmDesign;
  FPopupInputItem[3].Enabled := LogicMode = lmDesign;
end;

{TmoLogicInput}

procedure TmoLogicInput.Assign(Source: TPersistent);
var
  ALogicInputSource: TmoLogicInput;
begin
  if Source is TmoLogicInput then
  begin
    //?make assignment as in SetLogicInputSource1/2
  end
  else inherited Assign(Source);
end;

constructor TmoLogicInput.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FLogicInputSource := nil;
  FColorInput := clBlack;
  FiLine.P1 := point(0,0);
  FiLine.dX := lmoXi;
  FiLine.Selected := false;
  FInvert := false;
  FInput := true;
  FInputRecursionCounter := 0;  //V2.27
  Input := false; //force Input update
end;

destructor TmoLogicInput.Destroy;
begin
  if Assigned(FLogicInputSource) then
  try
    //Before leaving, delete input links
    LogicInputSource := nil;
  except
  end;
  inherited Destroy;
end;

procedure TmoLogicInput.ReceiveLogicChange(ALogicState, Init: Boolean);
begin
  inc(FInputRecursionCounter);  //V2.27
  if FInputRecursionCounter < 100 then  //V2.27
    begin
    TLogicMultiOp(Collection.Owner).InitLogic := Init;
    Input := ALogicState;
    Changed(False); //calls Update
    if FInputRecursionCounter > 0 then dec(FInputRecursionCounter);
    end
  else
    begin
    if FInputRecursionCounter > 0 then dec(FInputRecursionCounter);
    FColorInput := clYellow;
    TLogicMultiOp(Collection.Owner).Invalidate;
    end;
end;

procedure TmoLogicInput.SetInput(Value: Boolean);
begin
  if ((Value xor FInvert) <> FInput) or
    TLogicMultiOp(Collection.Owner).InitLogic then
  begin
    TLogicMultiOp(Collection.Owner).InitLogic := false;
    FInput := Value xor FInvert;
    if FInput then FColorInput := clLime else FColorInput := clRed;
    TLogicMultiOp(Collection.Owner).Invalidate;
  end;
end;

procedure TmoLogicInput.SetInvert(Value: Boolean);
var
  RealInput : boolean;
begin
  if Value <> FInvert then
  begin
    if FInvert then RealInput := not Input else RealInput := Input;
    FInvert := Value;
    ReceiveLogicChange(RealInput,True);
  end;
end;

procedure TmoLogicInput.SetLogicInputSource(Value: TLogicLine);
var
  i : integer;
begin
  //Before assigning new InputSource, delete old InputSource events
  if Assigned(FLogicInputSource) then
  begin
    if FLogicInputSource is TLogicLine then
    begin
        TLogicLine(FLogicInputSource).AssignOutput(
          TLogicMultiOp(Collection.Owner),
          ReceiveLogicChange,oaDelete);
        TLogicLine(FLogicInputSource).MoveEndPoint(-1,-10);
    end;
    //Force input to false on disconnecting input
    ReceiveLogicChange(false,false);
  end;

  try
    FLogicInputSource := TLogicLine(Value);
  except
    FLogicInputSource := nil;
  end;

  TLogicMultiOp(Collection.Owner).PaintDimensionsCalc;
  if Assigned(FLogicInputSource) then
  begin
    if FLogicInputSource is TLogicLine then
    begin
      TLogicLine(FLogicInputSource).AssignOutput(
          TLogicMultiOp(Collection.Owner),
          ReceiveLogicChange,oaAdd);
      TLogicMultiOp(Collection.Owner).MoveConnectedInputs;
    end;
  end;
end;


{TmoLogicInputs}

function TmoLogicInputs.GetItem(Index: Integer): TmoLogicInput;
begin
  Result := TmoLogicInput(inherited GetItem(Index));
end;

function TmoLogicInputs.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

procedure TmoLogicInputs.SetItem(Index: Integer; Value: TmoLogicInput);
begin
  inherited SetItem(Index, Value);
end;

function TmoLogicInputs.Add: TmoLogicInput;
begin
  Result := TmoLogicInput(inherited Add);
end;

constructor TmoLogicInputs.Create(AOwner: TLogicMultiOp);
begin
  inherited Create(TmoLogicInput);
  FOwner := AOwner;
end;

function TmoLogicInputs.Owner: TLogicMultiOp;
begin
  Result := FOwner;
end;

procedure TmoLogicInputs.Update(Item: TCollectionItem);
begin
  TLogicMultiOp(Owner).DetermineLogicalState;
end;

procedure TLogicMultiMooN.puiMooNSetNumReqd(Sender: TObject);
var
  sNumReqd : string;
  aCaption, aPrompt : string;
  M : integer;
begin
  //call dialog InputBox to get a new value for property NumReqd
  aCaption := 'Value Editor';
  aPrompt  := 'Enter a new M value (1..9) for ' + Self.Name;
  sNumReqd := IntToStr(NumReqd);
  sNumReqd := InputBox(ACaption,APrompt,sNumReqd);
  try
    M := StrToInt(sNumReqd);
  except
    on E: EConvertError do
      begin
      ShowMessage(sNumReqd + ' is not a valid number' + CRLF +
        E.ClassName + CRLF + E.Message);
      M := 2;
      end;
    else
      M := 2;
  end;
  NumReqd := M;
end;

procedure TLogicMultiMooN.OnLogicModeChange;
begin
{
  FPopupItem[0].Caption := 'Invert Output';
  FPopupItem[1].Caption := 'Add Output LogicLine'; //v2.28
  FPopupItem[2].Caption := 'Delete LogicElement';
  FPopupItem[3].Caption := 'Set number of required inputs';
  FPopupInputItem[0].Caption := 'Invert Input';
  FPopupInputItem[1].Caption := 'Move Input Up';
  FPopupInputItem[2].Caption := 'Move Input Down';
  FPopupInputItem[3].Caption := 'Disconnect Input';
}
  FPopupItem[1].Enabled := LogicMode = lmDesign;
  FPopupItem[2].Enabled := LogicMode = lmDesign;
  FPopupItem[3].Enabled := LogicMode = lmDesign;
  FPopupInputItem[1].Enabled := LogicMode = lmDesign;
  FPopupInputItem[2].Enabled := LogicMode = lmDesign;
  FPopupInputItem[3].Enabled := LogicMode = lmDesign;
end;

procedure TmoOutputEvent.SetPriority(Value: Integer);  //V2.26
begin
  if Value <> FPriority then
  begin
    FPriority := Value;
  end;
end;

procedure TmoOutputEvents.PrioritizeOutputs; //V2.26
var
  i, j : integer;
begin
  //set Priority from LogicLines
  for i:= 0 to Self.Count-1 do
  begin
    Items[i].Priority := Items[i].OutputControl.Priority;
  end;
  //Change Output order to match Self.Priority
  QuickSort(0,Count-1, ComparePriorities);
  //Synch Output receivers to output events
  UpdatePriorities;
end;

procedure TmoOutputEvents.QuickSort(L, R: Integer;
  SCompare: TCollectionSortCompare);  //V2.26
var
  I, J: Integer;
  P: Integer; //Priority Value, not position
  T: TmoOutputEvent;
begin
  if R < L then exit; //true if no OutputEvents defined
  repeat
    I := L;
    J := R;
    P := Items[(L + R) shr 1].FPriority;
    repeat
      while SCompare(Items[I].FPriority, P) < 0 do
        Inc(I);
      while SCompare(Items[J].FPriority, P) > 0 do
        Dec(J);
      if I <= J then
      begin
        Items[J].Index := I;
        if I+1 < R then Items[I+1].Index := J;
        {T := Items[I];
        Items[I] := Items[J];
        Items[J] := T; }
        Inc(I);
        Dec(J);
      end;
    until I > J;
    if L < J then
      QuickSort(L, J, SCompare);
    L := I;
  until I >= R;
end;

procedure TmoOutputEvents.UpdatePriorities;  //V2.26
var
  i : integer;
begin
  for i := 0 to Self.Count-1 do
  begin
    //set FPriority for each output if not default
    if Items[i].FPriority <> -1 then
      Items[i].FPriority := i;
    //feed back priority to LogicLine
    TLogicLine(Items[i].OutputControl).Priority := Items[i].FPriority;
  end;
end;




initialization
RegisterClasses([TLogicMultiOp, TLogicMultiAnd, TLogicMultiOr, TLogicMultiMooN]);
finalization

end.
