unit LogicConnect;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, Math, StdCtrls, TypInfo, Types, Menus, Globals, gcMultiPolyline
  ;

const
  rNot :integer = 5;   //radius of LogicInput inverted input/output symbol

type
  TlcOutputEvents = class;
  TLogicConnect = class;

  TlcOutputEvent = class(TCollectionItem)
  private
    FOutputControl: TComponent;
    FOutputMove: TLogicMoveOutputPos;
    FOutputEvent: TLogicChangeEvent;
    FPriority: Integer;  //V2.26
    procedure SetOutputControl(Value: TComponent);
    procedure SetOutputEvent(Value: TLogicChangeEvent);
    procedure SetOutputMove(Value: TLogicMoveOutputPos);
    procedure SetPriority(Value: Integer);  //V2.26
  protected
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
  //published
    property OutputControl:TComponent read FOutputControl
      write SetOutputControl;
    property OutputEvent:TLogicChangeEvent read FOutputEvent
      write SetOutputEvent stored true default nil;
    property OutputMove:TLogicMoveOutputPos read FOutputMove
      write SetOutputMove default nil;
    property Priority:Integer read FPriority write SetPriority; //V2.26
  end;

  TlcOutputEvents = class(TCollection)
  private
    FOwner: TLogicConnect;
    function GetItem(Index: Integer): TlcOutputEvent;
    procedure QuickSort(L, R: Integer; SCompare: TCollectionSortCompare);  //V2.26
    procedure SetItem(Index: Integer; Value: TlcOutputEvent);
    procedure UpdatePriorities;
  protected
    function GetOwner: TPersistent; override;
    procedure PrioritizeOutputs; //V2.26
  public
    constructor Create(AOwner: TLogicConnect);
    function Add: TlcOutputEvent;
    function Owner: TLogicConnect;
  //published
    property Items[Index: Integer]: TlcOutputEvent read GetItem
      write SetItem; default;
  end;

TLogicConnect = class(TCustomControl)
private
  FColorInput : TColor;
  FColorOutput : TColor;
  FColorState  : TColor;
  FGridSize: TGridSize;
  FGridEnabled: Boolean;
  FInitLogic : Boolean;
  FInput : Boolean;
  FInputRecursionCounter : integer;  //V2.27
  FInvert : Boolean;
  FInvertOut : Boolean;
  FLogicInputSource : TComponent;
  FLogicMode : TLogicMode;
  FLogicState : Boolean;
  FMinHeight : integer;
  FMinWidth  : integer;
  FOutputEvents : TlcOutputEvents;
  FOutputState : Boolean;
  FSelected   : Boolean;
  FOnLogicChange : TLogicChangeEvent;
  StartXY : TPoint;
  LastRect, ParentExtent: TRect;
    procedure OnLogicModeChange; dynamic;
  procedure AddOutput(ALogicElement:TComponent;
            ALogicChangeMethod:TLogicChangeEvent;
            ALogicMoveOutputMethod:TLogicMoveOutputPos);
  procedure DeleteOutput(ALogicElement:TComponent);
  function  GetPositionOnGrid(APos: Integer): Integer;
  function  GetTextBounds:TRect;
  procedure MoveConnectedIO;
  procedure MoveConnectedInputs;
  procedure MoveConnectedOutputs;
  procedure SetGridEnabled(Value: Boolean);
  procedure SetGridSize(Value:TGridSize);
  procedure SetInitLogic(Init:Boolean);
  procedure SetInput(Value: Boolean);
  procedure SetInvert(Value:Boolean);
  procedure SetInvertOut(Value:Boolean);
  procedure SetLogicInputSource(ALogicElement: TComponent);
  procedure SetLogicState(State:Boolean);
  procedure SetOutputEvents(Value: TlcOutputEvents);
  procedure SetOutputState(Value:Boolean);
  procedure puiConnChangeHint(Sender: TObject);
  procedure puiConnDisconnectInput(Sender: TObject);
  procedure puiConnFreeLogicElement(Sender: TObject);
  procedure SetLogicMode(Value: TLogicMode);
  procedure puiLogConnInvertInput(Sender: TObject);
  procedure puiLogConnInvertOutput(Sender: TObject);
  procedure SetNewSize;
protected
  function  DetermineLogicalState(ALogicInput:Boolean):Boolean; dynamic;
  function  GetBoundsRect:TRect; dynamic;
  procedure Loaded; override; //v2.26
  procedure Paint; override; //Implements TCustomControl virtual method
  procedure SetSelected(Value:Boolean);
  procedure uoMouseDown(Sender: TObject; Button: TMouseButton;
    Shift: TShiftState; X, Y: Integer);
  procedure uoMouseUp(Sender: TObject; Button: TMouseButton;
    Shift: TShiftState; X, Y: Integer);
  procedure uoMouseMove(Sender: TObject; Shift: TShiftState;
    X, Y: Integer);
  procedure ReceiveLogicChange(ALogicState,Init:Boolean);
  property  InitLogic:Boolean read FInitLogic write SetInitLogic;
  property  OnLogicChange: TLogicChangeEvent
            read FOnLogicChange write FOnLogicChange;
  property  OutputState:Boolean read FOutputState write SetOutputState;
  property  Canvas;
  property Invert:Boolean read FInvert write SetInvert stored true default false;
  property InvertOut:Boolean read FInvertOut write SetInvertOut stored true default false;
public
  StartX, StartY : integer;
  constructor Create(AOwner: TComponent); override;
  destructor  Destroy;override;
  procedure AssignOutput(ALogicElement:TComponent;
          ALogicChangeMethod:TLogicChangeEvent;
          ALogicMoveOutputMethod:TLogicMoveOutputPos;
          AnOutputAction:TOutputActionType);
  procedure Notification(AComponent:TComponent; Operation:TOperation); override;
  property  GridEnabled : boolean read FGridEnabled write SetGridEnabled;
  property  GridSize:TGridSize read FGridSize write SetGridSize;
  property  Input:Boolean read FInput write SetInput;
  property  LogicMode:TLogicMode read FLogicMode write SetLogicMode;
  property  LogicState:Boolean read FLogicState write SetLogicState;
  property  OutputEvents: TlcOutputEvents read FOutputEvents write SetOutputEvents;
  property  Selected:Boolean read FSelected write SetSelected;
  property  OnDragDrop;
  property  OnDragOver;
  property  PopupMenu;
published
  property LogicInputSource:TComponent
           read FLogicInputSource write SetLogicInputSource
           stored true default nil;
end;

TLogicConnSink = class(TLogicConnect)
//LogicConnSink is based on LogicInput
private
  { Private declarations }
  FpumConnSink: TPopupMenu;
  FpuiConnSink: array of TMenuItem;
  FPriority: Integer;   //V2.26
  procedure SetPriority(const Value: Integer);     //V2.26
  procedure OnLogicModeChange; override;
  procedure puiConnSinkChangeDescription(Sender: TObject);
  procedure puiConnSinkFreeLogicElement(Sender: TObject); //V2.26
  procedure puiConnSinkLogicInputSource(Sender: TObject);
  procedure puiConnSinkShowConnection(Sender: TObject);
  procedure puiConnSinkDisconnectSource(Sender: TObject);
  procedure puiConnSinkSetPriority(Sender: TObject);
    procedure puiConnSinkAddOutputLogicLine(Sender: TObject);
protected
  { Protected declarations }
  function DetermineLogicalState(ALogicInput:Boolean):Boolean; override;
  procedure LogicLineDragDrop(Sender, Source: TObject; X, Y: Integer);
  procedure LogicLineDragOver(Sender, Source: TObject; X, Y: Integer;
    State: TDragState; var Accept: Boolean);
  procedure Paint; override; //Implements TCustomControl virtual method
  procedure RequestNewPriority(APriority: integer); //V2.26
public
  { Public declarations }
  constructor Create(AOwner: TComponent); override;
  destructor  Destroy;override;
  procedure ChangePriority(ALogicReceiver: TLogicLine;
      ALogicPriority: Integer);  //V2.26
published
  { Published declarations }
  Property Caption;
  Property InvertOut;
  property Priority:Integer read FPriority Write SetPriority default -1;  //V2.26
end;

TLogicConnSource = class(TLogicConnect)
//LogicConnSource is based on LogicNot
private
  { Private declarations }
  FpuiConnSource: array of TMenuItem;
  FpumConnSource: TPopupMenu;
  procedure OnLogicModeChange; override;
  procedure puiConnSourceChangeDescription(Sender: TObject);
protected
  { Protected declarations }
  function DetermineLogicalState(ALogicInput: Boolean): Boolean;
    override;
  procedure LogicLineDragDrop(Sender, Source: TObject; X, Y: Integer);
  procedure LogicLineDragOver(Sender, Source: TObject; X, Y: Integer;
    State: TDragState; var Accept: Boolean);
  procedure Paint; override;
public
  { Public declarations }
  constructor Create(AOwner: TComponent); override;
  destructor Destroy; override;
  procedure ChangePriority(ALogicReceiver: TLogicConnSink;
      ALogicPriority: Integer);
published
  { Published declarations }
  Property Caption;
  Property Invert;
end;

implementation

uses LogicBinaryOp, LogicUnaryOp, LogicWindow;

{ TOutputEvent based on TListColumn }

constructor TlcOutputEvent.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FOutputControl := nil; //V2.26
  FOutputEvent := nil;
  FOutputMove  := nil;
  FPriority := -1; //V2.26
end;

destructor TlcOutputEvent.Destroy;
begin
  inherited Destroy;
end;

procedure TlcOutputEvent.SetOutputControl(Value: TComponent);
begin
  if FOutputControl <> Value then
  begin
    FOutputControl := Value;
  end;
end;

procedure TlcOutputEvent.SetOutputEvent(Value: TLogicChangeEvent);
begin
  if @FOutputEvent <> @Value then
      FOutputEvent := Value;
end;

procedure TlcOutputEvent.SetOutputMove(Value: TLogicMoveOutputPos);
begin
  if @FOutputMove <> @Value then
    FOutputMove := Value;
end;

procedure TlcOutputEvent.Assign(Source: TPersistent);
var
  AOutputEvent: TlcOutputEvent;
begin
  if Source is TlcOutputEvent then
  begin
    AOutputEvent := TlcOutputEvent(Source);
    OutputControl := AOutputEvent.OutputControl;
    OutputEvent   := AOutputEvent.OutputEvent;
    OutputMove    := AOutputEvent.OutputMove;
  end
  else inherited Assign(Source);
end;

{ TOutputEvents based on TListColumns }

constructor TlcOutputEvents.Create(AOwner: TLogicConnect);
begin
  inherited Create(TlcOutputEvent);
  FOwner := AOwner;
end;

function TlcOutputEvents.GetItem(Index: Integer): TlcOutputEvent;
begin
  Result := TlcOutputEvent(inherited GetItem(Index));
end;

procedure TlcOutputEvents.SetItem(Index: Integer; Value: TlcOutputEvent);
begin
  inherited SetItem(Index, Value);
end;

function TlcOutputEvents.Add: TlcOutputEvent;
begin
  Result := TlcOutputEvent(inherited Add);
end;

function TlcOutputEvents.Owner: TLogicConnect;
begin
  Result := FOwner;
end;

function TlcOutputEvents.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

{ TLogicConnect }
procedure TLogicConnect.SetOutputEvents(Value: TlcOutputEvents);
begin
  FOutputEvents.Assign(Value);
end;

constructor TLogicConnect.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FColorInput := clBlack;
  FColorOutput := clBlack;
  FColorState  := clBlack;
  FGridEnabled := false;
  FGridSize    := 0;
  GridSize     := 5;
  GridEnabled  := true;
  FLogicInputSource := nil;
  FOutputEvents := TlcOutputEvents.Create(Self);
  FInput := true;
  FInputRecursionCounter := 0; //V2.27
  FInvert := false;
  Input := false; //force input update
  FInvertOut := false;
  FMinHeight := 20;
  FMinWidth  := 60;
  FOutputState := true;
  FLogicMode := lmDesign;
  FLogicState := false;
  FInitLogic := true;
  LogicState := false; //force logic/output update
  OnMouseDown := uoMouseDown;
  OnMouseMove := uoMouseMove;
  OnMouseUp   := uoMouseUp;
  with Canvas do
  begin
    Pen.Style := psSolid;
    Pen.Color := clBlack;
    Brush.Style := bsClear;
  end;
  Top := 50;
  Left := 50;
  Width := FMinWidth;
  Height := FMinHeight;
  Color := clWhite;
  Visible := true;

end;

destructor TLogicConnect.Destroy;
begin
  try
  if Assigned(FLogicInputSource) then
  begin
    //Before leaving, delete input links
    LogicInputSource := nil;
  end;
  except
  end;
  inherited Destroy;
end;

//This is an important tool for removing dependencies to linked components
procedure TLogicConnect.Notification(AComponent:TComponent ; Operation:TOperation);
begin
  if (Operation=opRemove) and (AComponent=LogicInputSource) then
    begin
    //Unlink myself from the component being destroyed
    LogicInputSource := nil;
    end;
  inherited Notification(Acomponent,Operation);
end;

procedure TLogicConnect.SetInput(Value: Boolean);
begin
  if (Value xor FInvert) <> FInput then
  begin
    FInput := Value xor FInvert;
    if FInput then FColorInput := clLime else FColorInput := clRed;
    invalidate;
  end;
end;

procedure TLogicConnect.SetInvert(Value:Boolean);
var
  RealInput : boolean;
begin
  if Value <> FInvert then
  begin
    if FInvert then RealInput := not Input else RealInput := Input;
    FInvert := Value;
    ReceiveLogicChange(RealInput,True);
  end;
end;

procedure TLogicConnect.SetInvertOut(Value:Boolean);
begin
  if Value <> FInvertOut then
  begin
    FInvertOut := Value;
    InitLogic := True;
    LogicState := LogicState;
  end;
end;

//Store my input source object, and add myself to its output list
procedure TLogicConnect.SetLogicInputSource(ALogicElement: TComponent);
begin
  //Before assigning new InputSource, delete old InputSource events
  if Assigned(FLogicInputSource) then
  begin
    if FLogicInputSource is TLogicLine then
    begin
        TLogicLine(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,oaDelete);
        TLogicLine(FLogicInputSource).MoveEndPoint(-1,-10);
    end;
    if FLogicInputSource is TLogicConnect then
        TLogicConnect(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,nil,
          oaDelete);
    //Force input to false on disconnecting input
    ReceiveLogicChange(false,false);
  end;

  try
    FLogicInputSource := ALogicElement;
  except
    FLogicInputSource := nil;
  end;

  if Assigned(FLogicInputSource) then
  begin
    if FLogicInputSource is TLogicLine then
    begin
      TLogicLine(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,oaAdd);
      MoveConnectedInputs;
    end;
    if FLogicInputSource is TLogicConnect then
      TLogicConnect(FLogicInputSource).AssignOutput(Self,ReceiveLogicChange,nil,
        oaAdd);
  end;
  Invalidate;
end;

procedure TLogicConnect.AssignOutput(ALogicElement:TComponent;
          ALogicChangeMethod:TLogicChangeEvent;
          ALogicMoveOutputMethod:TLogicMoveOutputPos;
          AnOutputAction:TOutputActionType);
begin 
  case AnOutputAction of
  oaAdd:
    //add ALogicChange (pointer to an OnLogicChange event)
    // to a list of TLogicChangeEvents
    AddOutput(ALogicElement,ALogicChangeMethod,ALogicMoveOutputMethod);
  oaDelete:
    //Lookup item = ALogicChangeMethod and delete it
    DeleteOutput(ALogicElement);
  end; //case
end;

//Override this function to perform custom logic
function TLogicConnect.DetermineLogicalState(ALogicInput:Boolean):Boolean;
begin
  Result := ALogicInput;
end;

procedure TLogicConnect.ReceiveLogicChange(ALogicState,Init:Boolean);
begin
  inc(FInputRecursionCounter);  //V2.27
  if FInputRecursionCounter < 100 then  //V2.27
    begin
    InitLogic := Init;
    LogicState := DetermineLogicalState(ALogicState);
    if FInputRecursionCounter > 0 then dec(FInputRecursionCounter); //V2.27
    end
  else
    begin
    if FInputRecursionCounter > 0 then dec(FInputRecursionCounter); //V2.27
    FColorInput := clYellow;  //V2.27
    Invalidate;  //V2.27
    end;

end;

function TLogicConnect.GetPositionOnGrid(APos:Integer):Integer;
var
  Delta : integer;
begin
  if GridEnabled and (GridSize > 0) then
    begin
    Delta := APos mod GridSize;
    if (Delta / GridSize) >= 0.5 then
      Result := APos + GridSize - Delta
    else
      Result := APos - Delta;
    end
  else
    Result := APos;
end;

function TLogicConnect.GetTextBounds:TRect;
begin
  Result := Rect(0,0,0,0);
  Windows.DrawText(Self.Canvas.Handle, PChar(Self.Caption), -1,
    Result, DT_CALCRECT);
end;

procedure TLogicConnect.SetGridEnabled(Value:Boolean);
begin
  if Value <> FGridEnabled then
    if not FGridEnabled then  //object may not have been on grid
      begin
      FGridEnabled := Value;
      Left := GetPositionOnGrid(Left);
      Top  := GetPositionOnGrid(Top);
      end
  else
      FGridEnabled := Value;
end;

procedure TLogicConnect.SetGridSize(Value:TGridSize);
begin
  if Value <> FGridSize then
  begin
    FGridSize := Value;
  end;
end;

procedure TLogicConnect.SetInitLogic(Init:Boolean);
begin
  FInitLogic := Init;
end;

procedure TLogicConnect.SetOutputState(Value:Boolean);
begin
  if Value <> FOutputState then
  begin
    FOutputState := Value;
    if FOutputState then FColorOutput := clLime else FColorOutput := clRed;
    invalidate;
  end;
end;

procedure TLogicConnect.SetLogicState(State:Boolean);
var
  i : integer;
  AnOutputEvent : TlcOutputEvent;
begin
  if (State <> FLogicState) or InitLogic then
  begin
    FLogicState := State;
    OutputState := FLogicState xor InvertOut;
    for i := 0 to OutputEvents.Count-1 do
    begin
      try
      AnOutputEvent := OutputEvents.Items[i];
      AnOutputEvent.OutputEvent(OutputState,false);
      except
      end;
    end;
    If InitLogic then InitLogic := false;
    If FLogicState then FColorState := clLime else FColorState := clRed;
    Invalidate;
    if Assigned(OnLogicChange) then OnLogicChange(OutputState,FInitLogic);
  end;
end;

procedure TLogicConnect.SetNewSize;
var
  TextRect : TRect;
begin
    if assigned(Parent) then
      TextRect := GetTextBounds
    else
      TextRect := Rect(0,0,0,0);
    Width  := GetGridPosition(Max(FMinWidth,TextRect.Right+6+rNot*2));
    Height := GetGridPosition(Max(FMinHeight,TextRect.Bottom+6));
    Invalidate;
    MoveConnectedOutputs;
end;

procedure TLogicConnect.AddOutput(ALogicElement:TComponent;
          ALogicChangeMethod:TLogicChangeEvent;
          ALogicMoveOutputMethod:TLogicMoveOutputPos);
var
  AnOutputEvent : TlcOutputEvent;
begin
  //add ALogicChangeMethod (pointer to an LogicChange event)
  // to a list of TLogicChangeEvents
  AnOutputEvent := OutputEvents.Add;
  AnOutputEvent.OutputControl := ALogicElement;
  AnOutputEvent.OutputEvent := ALogicChangeMethod;
  AnOutputEvent.OutputMove  := ALogicMoveOutputMethod;
  //Connect the output
  MoveConnectedOutputs;
  //Initialize the output
  InitLogic := True;
  LogicState := LogicState;
end;

procedure TLogicConnect.DeleteOutput(ALogicElement:TComponent);
var
  i : integer;
begin
  //Lookup item = ALogicChangeMethod and delete it
  for i:=OutputEvents.Count-1 downto 0 do
  begin
    if OutputEvents[i].OutputControl = ALogicElement then
      OutputEvents.Delete(i);
  end;
end;

procedure TLogicConnect.Paint;
var
  TextRect : TRect;
begin
  inherited Paint;
  if Selected then Canvas.Rectangle(0,0,Width,Height);
end;

function TLogicConnect.GetBoundsRect:TRect;
begin
  Result := BoundsRect;
end;

procedure TLogicConnect.uoMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if (Button = mbLeft) and (LogicMode = lmDesign) then
    begin
    Selected := true;
    StartXY := ClientToParent(Point(X,Y),TPanel(Parent));
    ParentExtent := GetDeltaRect(TScrollBox(Parent.Parent).ClientRect,
      Point(-TPanel(Parent).Left,-TPanel(Parent).Top));
    LastRect:= GetBoundsRect;
    end;
end;

procedure TLogicConnect.uoMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if (Button = mbLeft) and (LogicMode = lmDesign) then
  begin
    Selected := false;
  end;
end;

procedure TLogicConnect.uoMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
var
  AbsXY, DeltaXY : TPoint;
  NewExtent : TRect;
begin
  if (ssLeft in Shift) and (LogicMode = lmDesign) then { make sure button is down }
  begin
    AbsXY := ClientToParent(Point(X,Y),TPanel(Parent));
    DeltaXY.X := GetPositionOnGrid(AbsXY.X - StartXY.X);
    DeltaXY.Y := GetPositionOnGrid(AbsXY.Y - StartXY.Y);
    NewExtent := GetDeltaRect(LastRect,DeltaXY);
    if ( PtInRect(ParentExtent, NewExtent.TopLeft) and
         PtInRect(ParentExtent, NewExtent.BottomRight) ) then
      begin
      Inc(StartXY.X,DeltaXY.X);
      Inc(StartXY.Y,DeltaXY.Y);
      end
    else
      begin
      if NewExtent.Left < ParentExtent.Left then
        DeltaXY.X := GetPositionOnGrid(ParentExtent.Left - LastRect.Left);
      if NewExtent.Right > ParentExtent.Right then
        DeltaXY.X := GetPositionOnGrid(ParentExtent.Right - LastRect.Right);
      if NewExtent.Top < ParentExtent.Top then
        DeltaXY.Y := GetPositionOnGrid(ParentExtent.Top - LastRect.Top);
      if NewExtent.Bottom > ParentExtent.Bottom then
        DeltaXY.Y := GetPositionOnGrid(ParentExtent.Bottom - LastRect.Bottom);
      NewExtent := GetDeltaRect(LastRect,DeltaXY);
      Inc(StartXY.X,DeltaXY.X);
      Inc(StartXY.Y,DeltaXY.Y);
      end;
    Self.Left := NewExtent.Left;
    Self.Top  := NewExtent.Top;
    LastRect := NewExtent;
    MoveConnectedIO;
  end;
end;

procedure TLogicConnect.MoveConnectedIO;
begin
  MoveConnectedInputs;
  MoveConnectedOutputs;
end;

procedure TLogicConnect.MoveConnectedInputs;
var
  X,Y : integer;
begin
  X := Left;
  Y := Top + (Height div 2);
  if Assigned(LogicInputSource) then
    if LogicInputSource is TLogicLine then
      TLogicLine(LogicInputSource).PlacePolyPoint(-1,X,Y);
end;

procedure TLogicConnect.MoveConnectedOutputs;
var
  i : integer;
  X,Y : integer;
  AnOutputMove : TlcOutputEvent;
begin
  X := Left + Width;
  Y := Top + (Height div 2);
  for i := 0 to OutputEvents.Count-1 do
  begin
    try
    AnOutputMove := OutputEvents.Items[i];
    if @AnOutputMove.OutputMove <> nil then
    //if Assigned(AnOutputMove) then
      AnOutputMove.OutputMove(0,X,Y);
    except
    end;
  end;
end;

procedure TLogicConnect.SetSelected(Value:Boolean);
begin
  if Value <> FSelected then
  begin
    FSelected := Value;
    Invalidate;
  end;
end;

procedure TLogicConnect.puiConnChangeHint(Sender: TObject);
begin
  if Self.Hint <> '' then
    Self.Hint := EditControlHint(Self,Self.Hint)
  else
    Self.Hint := EditControlHint(Self,Self.Caption);
  if Self.Hint <> '' then
    Self.ShowHint := true
  else
    Self.ShowHint := false;
  Invalidate;
end;

procedure TLogicConnect.puiConnDisconnectInput(Sender: TObject);
begin
  with Self as TLogicConnect do begin
    LogicInputSource := nil;
    Invalidate;
  end;
end;

procedure TLogicConnect.puiLogConnInvertInput(Sender: TObject);
begin
  with Self as TLogicConnect do begin
    Invert := not Invert;
  end;
end;

procedure TLogicConnect.puiLogConnInvertOutput(Sender: TObject);
begin
  with Self as TLogicConnect do begin
    InvertOut := not InvertOut;
  end;
end;

procedure TLogicConnect.puiConnFreeLogicElement(Sender: TObject);
var
  i : integer;
begin
  //Set priority of output logiclines to -1  //V2.26
  for i := FOutputEvents.Count-1 downto 0 do
    FOutputEvents[i].Priority := -1;
  FOutputEvents.UpdatePriorities;

  TLogicConnect(Self).Free;
end;

procedure TLogicConnect.SetLogicMode(Value: TLogicMode);
begin
  If FLogicMode <> Value then
  begin
    FLogicMode := Value;
    OnLogicModeChange;
  end;
end;

procedure TLogicConnect.OnLogicModeChange;
begin
  //Override in descendants to update popup menus
end;

procedure TLogicConnect.Loaded; //V2.26
begin
  inherited;
  FOutputEvents.PrioritizeOutputs;
end;

{TLogicConnSink}

constructor TLogicConnSink.Create(AOwner: TComponent);
var
  i : integer;
begin
  inherited Create(AOwner);
  Self.Caption := 'Receive';
  FPriority := -1;  //V2.26
  OnDragDrop := LogicLineDragDrop;
  OnDragOver := LogicLineDragOver;

  //Popup Menu
  FpumConnSink := TPopUpMenu.Create(self);
  FpumConnSink.AutoPopup := true;
  SetLength(FpuiConnSink,9);  //V2.26
  for i := 0 to 8 do begin    //V2.26
    FpuiConnSink[i] := TMenuItem.Create(Self);
    FpumConnSink.Items.Add(FpuiConnSink[i]);
  end;
  FpuiConnSink[0].Caption := 'Change description';
  FpuiConnSink[0].OnClick := puiConnSinkChangeDescription;
  FpuiConnSink[1].Caption := 'Change hint';
  FpuiConnSink[1].OnClick := puiConnChangeHint;
  FpuiConnSink[2].Caption := 'Show connected source';
  FpuiConnSink[2].OnClick := puiConnSinkShowConnection;
  FpuiConnSink[3].Caption := 'Connect source';
  FpuiConnSink[3].OnClick := puiConnSinkLogicInputSource;
  FpuiConnSink[4].Caption := 'Invert Output';
  FpuiConnSink[4].OnClick := puiLogConnInvertOutput;
  FpuiConnSink[5].Caption := 'Add Output LogicLine';  //v2.28
  FpuiConnSink[5].OnClick := puiConnSinkAddOutputLogicLine;
  FpuiConnSink[6].Caption := 'Set Output Priority (-1)';   //V2.26
  FpuiConnSink[6].OnClick := puiConnSinkSetPriority;  //V2.26
  FpuiConnSink[7].Caption := 'Disconnect source';
  FpuiConnSink[7].OnClick := puiConnSinkDisconnectSource;
  FpuiConnSink[8].Caption := 'Delete LogicElement';
  FpuiConnSink[8].OnClick := puiConnSinkFreeLogicElement; //V2.26
  PopupMenu := FpumConnSink;

  with Canvas do
  begin
    Pen.Style := psSolid;
    Pen.Color := clBlack;
    Brush.Style := bsClear;
  end;
end;

destructor TLogicConnSink.Destroy;
begin
  try
    FpumConnSink.Free;
    //also frees each instance of FpuiLogicDelay
  except
  end;
  try
    SetLength(FpuiConnSink,0);
  except
  end;
  inherited Destroy;
end;

procedure TLogicConnSink.LogicLineDragDrop(Sender, Source: TObject; X,
  Y: Integer);
begin
  if IsDragObject(Source) and (Source is TLogicLineDragObject) then
    with (Source as TLogicLineDragObject) do
    begin
      if LinkType = ltOutput then
      begin
        LogicLineID.LogicInputSource := Sender as TLogicConnSink;
      end;
    end;
end;

procedure TLogicConnSink.LogicLineDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
  if IsDragObject(Source) and (Source is TLogicLineDragObject) then
    if TLogicLineDragObject(Source).LinkType = ltOutput then
      Accept := True else Accept := False;
end;

procedure TLogicConnSink.Paint;
var
  x0,x1,y : integer;
begin
  inherited Paint;
  Canvas.Pen.Color := FColorState;
  if Assigned(LogicInputSource) then
    Canvas.Pen.Style := psSolid
  else
    Canvas.Pen.Style := psDash;
  //Draw body
  y  := Height div 2;
  x0 := 0;
  x1 := Width - rNot*2;
  Canvas.Rectangle(x0,0,x1,Height);
  Canvas.TextOut(2,3,Self.Caption); //v2.28
  // Draw output
  Canvas.Pen.Color := FColorOutput;
  // Draw output triangle
  x0 := x1;
  x1 := Width;
  Canvas.PolyLine([Point(x0,0),Point(x1,y),Point(x0,Height)]);
  // Draw output not
  if FInvertOut then
      Canvas.Ellipse(x0,y-rNot,x1,y+rNot);
end;

function TLogicConnSink.DetermineLogicalState(ALogicInput:Boolean):Boolean;
begin
  Input := ALogicInput;
  Result := FInput;
end;

procedure TLogicConnSink.puiConnSinkLogicInputSource(Sender: TObject);
var
  i, j : integer;
  ConnSources : TStringList;
  MySource : string;
  MyComponent : TComponent;
  OrigSource : string;
begin
  ConnSources := TStringList.Create;
  OrigSource := '';
  try
  for i:=0 to Owner.ComponentCount-1 do
    if Owner.Components[i] is TLogicConnSource then
    begin
      j := ConnSources.AddObject(TLogicConnSource(Owner.Components[i]).Caption +
        ' - ' + Owner.Components[i].Name, Owner.Components[i]);
      if TLogicConnSource(ConnSources.Objects[j]) = FLogicInputSource then
        OrigSource := ConnSources.Strings[j];   //v2.28
    end;
  ConnSources.Sort;

  MySource := SelectConnSource('Source Selector',
    'Select a logic sending connector', ConnSources, OrigSource);  //v2.28
  if MySource <> '' then   //V2.28
  begin
    i := ConnSources.IndexOf(MySource);
    if i > -1 then
      MyComponent := TLogicConnSource(ConnSources.Objects[i])
    else
      MyComponent := nil;
    LogicInputSource := TLogicConnSource(MyComponent);
    //Set the caption
    if Assigned(LogicInputSource) then
      Caption := TLogicConnSource(LogicInputSource).Caption
    else
      Caption := 'Receive';
    SetNewSize;
  end; //do nothing if user cancels selection
  finally
  ConnSources.Free;
  end;
  Invalidate;
end;

procedure TLogicConnSink.puiConnSinkAddOutputLogicLine(Sender: TObject); //v2.28
var
  Xo,Yo : integer;
  ALogicLine : TLogicLine;
begin
  Xo := Left + Width;
  Yo := Top  + (Height div 2);
  ALogicLine := MultiPolyline.ArrayToNewPolyline([point(Xo,Yo),
    point(Xo+30,Yo+30)]);
  ALogicLine.Selected := true;
  LogicCount.Count:=LogicCount.Count+1;
  ALogicLine.Name := 'LogicLine' + IntToStr(LogicCount.Count);
  ALogicLine.LogicInputSource := Self as TLogicConnSink;
end;

procedure TLogicConnSink.puiConnSinkChangeDescription(Sender: TObject);
begin
  Self.Caption := EditControlDescription(Self,Self.Caption);
  SetNewSize;
end;

procedure TLogicConnSink.puiConnSinkShowConnection(Sender: TObject);
begin
  if LogicInputSource = nil then
    ShowMessage('Source disconnected')
  else
    ShowMessage('Source connection' + Chr(vk_Return) +
      '  Tag:  ' +
      TLogicConnSource(LogicInputSource).Caption + Chr(vk_Return) +
      '  Name: ' +
      TLogicConnSource(LogicInputSource).Name);
end;

procedure TLogicConnSink.puiConnSinkDisconnectSource(Sender: TObject);
begin
  if Priority <> -1 then RequestNewPriority(-1); //V2.26
  LogicInputSource := nil;
  Caption := 'Receive';
  Invalidate;
end;

procedure TLogicConnSink.OnLogicModeChange;
begin
{
  FpuiConnSink[0].Caption := 'Change description';
  FpuiConnSink[1].Caption := 'Change hint';
  FpuiConnSink[2].Caption := 'Show connected source';
  FpuiConnSink[3].Caption := 'Connect source';
  FpuiConnSink[4].Caption := 'Invert Output';
  FpuiConnSink[5].Caption := 'Add Output LogicLine';  //v2.28
  FpuiConnSink[6].Caption := 'Set Output Priority';   //V2.26
  FpuiConnSink[7].Caption := 'Disconnect source';
  FpuiConnSink[8].Caption := 'Delete LogicElement';
}
  FpuiConnSink[3].Enabled := LogicMode = lmDesign;
  FpuiConnSink[5].Enabled := LogicMode = lmDesign;  //v2.28
  FpuiConnSink[7].Enabled := LogicMode = lmDesign;
  FpuiConnSink[8].Enabled := LogicMode = lmDesign;

end;

procedure TLogicConnSink.puiConnSinkSetPriority(Sender: TObject);  //V2.26
var
  NewPriority : integer;
begin
  //Call a dialog with existing priority default, show range of priority
  NewPriority := EditOutputPriority(Self,Priority);
  if Priority <> NewPriority then
    RequestNewPriority(NewPriority);
end;

procedure TLogicConnSink.RequestNewPriority(APriority: integer); //V2.26
begin
  //Change the published priority of this connection sink (Receiver)
  Priority := APriority;
  //Update priority in the LogicInputSource output event.
  if LogicInputSource is TLogicConnSource then
    TLogicConnSource(LogicInputSource).ChangePriority(Self,Priority);
end;

procedure TLogicConnSink.SetPriority(const Value: Integer);  //V2.26
begin
  if FPriority <> Value then
  begin
    FPriority := Value;
    FpuiConnSink[5].Caption :=
      'Set Output Priority (' + IntToStr(Priority) + ')';
  end;
end;

procedure TLogicConnSink.puiConnSinkFreeLogicElement(Sender: TObject); //V2.26
var
  i : integer;
  ALogicLine : TLogicLine;
begin
  if Priority <> -1 then RequestNewPriority(-1);
  for i := FOutputEvents.Count-1 downto 0 do  //v2.28
  begin
    ALogicLine := TLogicLine(FOutputEvents[i].FOutputControl); //v2.28
    if ALogicLine.OutputEvents.Count = 0 then  //v2.28
      FOutputEvents[i].FOutputControl.Free;
  end;
  puiConnFreeLogicElement(Sender);
end;

procedure TLogicConnSink.ChangePriority(ALogicReceiver:TLogicLine;
  ALogicPriority: Integer);  //V2.26
var
  i : integer;
begin
  //Lookup ALogicLine and change the priority of its output event
  for i:=0 to OutputEvents.Count-1 do
  begin
    if OutputEvents[i].OutputControl = ALogicReceiver then
    begin
      OutputEvents[i].Priority := ALogicPriority;
      OutputEvents.PrioritizeOutputs;
    end;
  end;
end;

{TLogicConnSource}
constructor TLogicConnSource.Create(AOwner: TComponent);
var
  i : integer;
begin
  inherited Create(AOwner);
  Self.Caption := 'Send';
  OnDragDrop := LogicLineDragDrop;
  OnDragOver := LogicLineDragOver;

  //Popup Menu
  FpumConnSource := TPopUpMenu.Create(self);
  FpumConnSource.AutoPopup := true;
  SetLength(FpuiConnSource,5);
  for i := 0 to 4 do begin
    FpuiConnSource[i] := TMenuItem.Create(Self);
    FpumConnSource.Items.Add(FpuiConnSource[i]);
  end;
  FpuiConnSource[0].Caption := 'Change description';
  FpuiConnSource[0].OnClick := puiConnSourceChangeDescription;
  FpuiConnSource[1].Caption := 'Change hint';
  FpuiConnSource[1].OnClick := puiConnChangeHint;
  FpuiConnSource[2].Caption := 'Invert Input';
  FpuiConnSource[2].OnClick := puiLogConnInvertInput;
  FpuiConnSource[3].Caption := 'Disconnect Input';
  FpuiConnSource[3].OnClick := puiConnDisconnectInput;
  FpuiConnSource[4].Caption := 'Delete LogicElement';
  FpuiConnSource[4].OnClick := puiConnFreeLogicElement;
  PopupMenu := FpumConnSource;

  with Canvas do
  begin
    Pen.Style := psSolid;
    Pen.Color := clBlack;
    Brush.Style := bsClear;
  end;
end;

destructor TLogicConnSource.Destroy;
begin
  try
    FpumConnSource.Free;
    //also frees each instance menu items
  except
  end;
  try
    SetLength(FpuiConnSource,0);
  except
  end;
  inherited Destroy;
end;

procedure TLogicConnSource.puiConnSourceChangeDescription(Sender: TObject);
begin
  Self.Caption := EditControlDescription(Self,Self.Caption);
  SetNewSize;
end;

function TLogicConnSource.DetermineLogicalState(
  ALogicInput: Boolean): Boolean;
begin
  Input := ALogicInput;
  Result := FInput;
end;

procedure TLogicConnSource.LogicLineDragDrop(Sender, Source: TObject; X,
  Y: Integer);
begin
  if IsDragObject(Source) and (Source is TLogicLineDragObject) then
    with (Source as TLogicLineDragObject) do
    begin
      if LinkType = ltInput then
      begin
        LogicInputSource := LogicLineID as TLogicLine;
      end;
    end;
end;

procedure TLogicConnSource.LogicLineDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
  if IsDragObject(Source) and (Source is TLogicLineDragObject) then
    if TLogicLineDragObject(Source).LinkType = ltInput then
      Accept := True else Accept := False;
end;

procedure TLogicConnSource.Paint;
var
  x0,x1,y : integer;
begin
  inherited Paint;
  if Assigned(LogicInputSource) then
    Canvas.Pen.Style := psSolid
  else
    Canvas.Pen.Style := psDash;
  // Draw output
  Canvas.Pen.Color := FColorInput;
  // Draw input triangle
  y  := Height div 2;
  x0 := 0;
  x1 := rNot*2;
  Canvas.PolyLine([Point(x1,0),Point(x0,y),Point(x1,Height)]);
  // Draw input not
  if FInvert then
      Canvas.Ellipse(x0,y-rNot,x1,y+rNot);
  //Draw body
  Canvas.Pen.Color := FColorState;
  x0 := x1;
  x1 := Width;
  Canvas.Rectangle(x0,0,x1,Height);
  Canvas.TextOut(x0+2,3,Self.Caption); //v2.28
end;

procedure TLogicConnSource.OnLogicModeChange;
begin
{
  FpuiConnSource[0].Caption := 'Change description';
  FpuiConnSource[1].Caption := 'Change hint';
  FpuiConnSource[2].Caption := 'Invert Input';
  FpuiConnSource[3].Caption := 'Disconnect Input';
  FpuiConnSource[4].Caption := 'Delete LogicElement';
}
  FpuiConnSource[3].Enabled := LogicMode = lmDesign;
  FpuiConnSource[4].Enabled := LogicMode = lmDesign;
end;

procedure TLogicConnSource.ChangePriority(ALogicReceiver:TLogicConnSink;
  ALogicPriority: Integer);  //V2.26
var
  i : integer;
begin
  //Lookup ALogicLine and change the priority of its output event
  for i:=0 to OutputEvents.Count-1 do
  begin
    if OutputEvents[i].OutputControl = ALogicReceiver then
    begin
      OutputEvents[i].Priority := ALogicPriority;
      OutputEvents.PrioritizeOutputs;
    end;
  end;
end;

procedure TlcOutputEvent.SetPriority(Value: Integer); //V2.26
begin
  if Value <> FPriority then
  begin
    FPriority := Value;
  end;
end;

procedure TlcOutputEvents.PrioritizeOutputs;  //V2.26
var
  i, j : integer;
begin
  //set Priority from LogicLines or LogicReceivers
  for i:= 0 to Self.Count-1 do
  begin
    if Items[i].OutputControl is TLogicLine then
      Items[i].Priority := TLogicLine(Items[i].OutputControl).Priority;
    if Items[i].OutputControl is TLogicConnSink then
      Items[i].Priority := TLogicConnSink(Items[i].OutputControl).Priority;
  end;
  //Change Output order to match Self.Priority
  QuickSort(0,Count-1, ComparePriorities);
  //Synch Output receivers to output events
  UpdatePriorities;
end;

procedure TlcOutputEvents.QuickSort(L, R: Integer;
  SCompare: TCollectionSortCompare);  //V2.26
var
  I, J: Integer;
  P: Integer; //Priority Value, not position
  T: TlcOutputEvent;
begin
  if R < L then exit; //true if no OutputEvents defined
  repeat
    I := L;
    J := R;
    P := Items[(L + R) shr 1].FPriority;
    repeat
      while SCompare(Items[I].FPriority, P) < 0 do
        Inc(I);
      while SCompare(Items[J].FPriority, P) > 0 do
        Dec(J);
      if I <= J then
      begin
        Items[J].Index := I;
        if I+1 < R then Items[I+1].Index := J;
        {T := Items[I];
        Items[I] := Items[J];
        Items[J] := T; }
        Inc(I);
        Dec(J);
      end;
    until I > J;
    if L < J then
      QuickSort(L, J, SCompare);
    L := I;
  until I >= R;
end;

procedure TlcOutputEvents.UpdatePriorities;  //V2.26
var
  i : integer;
begin
  for i := 0 to Self.Count-1 do
  begin
    //set FPriority for each output if not default
    if Items[i].FPriority <> -1 then
      Items[i].FPriority := i;
    //feed back priority to LogicLine or LogicReceiver
    if Items[i].OutputControl is TLogicLine then
      TLogicLine(Items[i].OutputControl).Priority := Items[i].FPriority;
    if Items[i].OutputControl is TLogicConnSink then
      TLogicConnSink(Items[i].OutputControl).Priority := Items[i].FPriority;
  end;
end;

initialization
RegisterClasses([TLogicConnect, TLogicNot, TLogicConnSink, TLogicConnSource]);
finalization

end.
