unit LogicBinaryOp;

{
State Inversion Table
A is Input, B is Invert Directive, C is Result
A xor B = C
0 0 0
1 0 1
0 1 1
1 1 0
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, Math, stdctrls, TypInfo, Types, Menus, Globals, gcMultiPolyline
  ;

const
  //LogicBinaryOp (lbo) dimensions: 50x50 Outer Dimension
  //Horizontal positions
  lboXi : Integer = 15; //End of Input, Start of Main
  lboXo : Integer = 35; //End of Main, Start of Output
  //Vertical positions
  lboYo : Integer = 25; //Output
  lboY1 : Integer = 10; //Input1
  lboY2 : Integer = 40; //Input2
  //Circle radius
  lboR  : Integer =  5; //Radius of "not"

type
  TboOutputEvents = class;
  TLogicBinaryOp = class;

  TboOutputEvent = class(TCollectionItem) 
  private
    FOutputControl: TLogicLine;
    FOutputMove: TLogicMoveOutputPos;
    FOutputEvent: TLogicChangeEvent;
    FPriority: Integer;  //V2.26
    procedure SetOutputControl(Value: TLogicLine);
    procedure SetOutputEvent(Value: TLogicChangeEvent);
    procedure SetOutputMove(Value: TLogicMoveOutputPos);
    procedure SetPriority(Value: Integer);  //V2.26
  protected
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
  //published
    property OutputControl:TLogicLine read FOutputControl
      write SetOutputControl;
    property OutputEvent:TLogicChangeEvent read FOutputEvent
      write SetOutputEvent stored true default nil;
    property OutputMove:TLogicMoveOutputPos read FOutputMove
      write SetOutputMove default nil;
    property Priority:Integer read FPriority write SetPriority; //V2.26
  end;

  TboOutputEvents = class(TCollection)
  private
    FOwner: TLogicBinaryOp;
    function GetItem(Index: Integer): TboOutputEvent;
    procedure QuickSort(L, R: Integer; SCompare: TCollectionSortCompare);  //V2.26
    procedure SetItem(Index: Integer; Value: TboOutputEvent);
    procedure UpdatePriorities;
  protected
    function GetOwner: TPersistent; override;
    procedure PrioritizeOutputs; //V2.26
  public
    constructor Create(AOwner: TLogicBinaryOp);
    function Add: TboOutputEvent;
    function Owner: TLogicBinaryOp;
  //published
    property Items[Index: Integer]: TboOutputEvent read GetItem
      write SetItem; default;
  end;

TLogicBinaryOp = class(TCustomControl)
private
  FColorInput1 : TColor;
  FColorInput2 : TColor;
  FColorOutput : TColor;
  FColorState  : TColor;
  FGridSize: TGridSize;
  FGridEnabled: Boolean;
  FInitLogic : Boolean;
  FInput1 : Boolean;
  FInput2 : Boolean;
  FInputRecursionCounter1 : integer;  //V2.27
  FInputRecursionCounter2 : integer;  //V2.27
  FInvert1 : Boolean;
  FInvert2 : Boolean;
  FInvertOut : Boolean;
  FLogicInputSource1 : TLogicLine;
  FLogicInputSource2 : TLogicLine;
  FLogicMode : TLogicMode;
  FLogicState : Boolean;
  FOnLogicChange : TLogicChangeEvent;
  FOutputEvents : TboOutputEvents;
  FOutputState : Boolean;
  FPopupMenu : TPopupMenu;
  FPopupItem : array of TMenuItem;
  FSelected   : Boolean;
  StartXY : TPoint;
  LastRect, ParentExtent: TRect;
  procedure OnLogicModeChange; dynamic;
  procedure AddOutput(ALogicLine:TLogicLine;
            ALogicChangeMethod:TLogicChangeEvent;
            ALogicMoveOutputMethod:TLogicMoveOutputPos);
  procedure DeleteOutput(ALogicLine:TLogicLine);
  function  GetPositionOnGrid(APos: Integer): Integer;
  procedure MoveConnectedIO;
  procedure MoveConnectedInputs;
  procedure MoveConnectedOutputs;
  procedure SetInitLogic(Init:Boolean);
  procedure SetInvert1(Value:Boolean);
  procedure SetInvert2(Value:Boolean);
  procedure SetInvertOut(Value:Boolean);
  procedure SetLogicInputSource1(ALogicElement: TLogicLine);
  procedure SetLogicInputSource2(ALogicElement: TLogicLine);
  procedure SetLogicState(State:Boolean);
  procedure SetOutputState(Value:Boolean);
  procedure SetOutputEvents(Value: TboOutputEvents);
  procedure puiBinOpInvertInput1(Sender: TObject);
  procedure puiBinOpInvertInput2(Sender: TObject);
  procedure puiBinOpInvertOutput(Sender: TObject);
  procedure puiBinOpDisconnectInput1(Sender: TObject);
  procedure puiBinOpDisconnectInput2(Sender: TObject);
  procedure puiBinOpDeleteLogicElement(Sender: TObject);
  procedure SetInput1(Value: Boolean);
  procedure SetInput2(Value: Boolean);
  procedure SetGridEnabled(Value: Boolean);
  procedure SetGridSize(Value: TGridSize);
  procedure SetLogicMode(Value: TLogicMode);
    procedure puiBinOpAddOutputLogicLine(Sender: TObject);

protected
  function  DetermineLogicalState(ALogicInput:Boolean;
            AIndex:Integer):Boolean; dynamic;
  function  GetBoundsRect:TRect; dynamic;
  procedure Loaded; override; //v2.26
  procedure LogicLineDragOver(Sender, Source: TObject; X,
            Y: Integer; State: TDragState; var Accept: Boolean);
  procedure Paint; override; //Implements TCustomControl virtual method
  procedure SetSelected(Value:Boolean);
  procedure uoMouseDown(Sender: TObject; Button: TMouseButton;
    Shift: TShiftState; X, Y: Integer);
  procedure uoMouseUp(Sender: TObject; Button: TMouseButton;
    Shift: TShiftState; X, Y: Integer);
  procedure uoMouseMove(Sender: TObject; Shift: TShiftState;
    X, Y: Integer);
  procedure ReceiveLogicChange1(ALogicState,Init:Boolean);
  procedure ReceiveLogicChange2(ALogicState,Init:Boolean);
  property  InitLogic:Boolean read FInitLogic write SetInitLogic;
  property  OutputState:Boolean read FOutputState write SetOutputState;
  property  OnLogicChange: TLogicChangeEvent
            read FOnLogicChange write FOnLogicChange;
  property  Canvas;
public
  constructor Create(AOwner: TComponent); override;
  destructor  Destroy;override;
  procedure AssignOutput(ALogicLine:TLogicLine;
          ALogicChangeMethod:TLogicChangeEvent;
          ALogicMoveOutputMethod:TLogicMoveOutputPos;
          AnOutputAction:TOutputActionType);
  procedure Notification(AComponent:TComponent; Operation:TOperation); override;
  procedure ChangePriority(ALogicLine:TLogicLine;ALogicPriority:Integer);  //V2.26
  property  GridEnabled : boolean read FGridEnabled write SetGridEnabled;
  property  GridSize:TGridSize read FGridSize write SetGridSize;
  property  Input1:Boolean read FInput1 write SetInput1;
  property  Input2:Boolean read FInput2 write SetInput2;
  property  LogicMode:TLogicMode read FLogicMode write SetLogicMode;
  property  LogicState:Boolean read FLogicState write SetLogicState;
  property  OutputEvents: TboOutputEvents read FOutputEvents write SetOutputEvents;
  property  Selected:Boolean read FSelected write SetSelected;
  property  OnDragDrop;
  property  OnDragOver;
  property  PopupMenu;
published
  property Invert1:Boolean read FInvert1 write SetInvert1 stored true default false;
  property Invert2:Boolean read FInvert2 write SetInvert2 stored true default false;
  property InvertOut:Boolean read FInvertOut write SetInvertOut stored true default false;
  property LogicInputSource1:TLogicLine
           read FLogicInputSource1 write SetLogicInputSource1
           stored true default nil;
  property LogicInputSource2:TLogicLine
           read FLogicInputSource2 write SetLogicInputSource2
           stored true default nil;
end;

TLogicAnd = class(TLogicBinaryOp)
private
  { Private declarations }
protected
    procedure OnLogicModeChange; override;
  { Protected declarations }
  function  DetermineLogicalState(ALogicInput:Boolean;
            AIndex:Integer):Boolean; override;
  procedure LogicLineDragDrop(Sender, Source: TObject; X,
            Y: Integer);
  procedure Paint; override; //Implements TCustomControl virtual method
public
  { Public declarations }
  constructor Create(AOwner: TComponent); override;
  destructor  Destroy;override;
published
  { Published declarations }
end;

TLogicOr = class(TLogicBinaryOp)
private
  { Private declarations }
protected
  procedure OnLogicModeChange; override;
  { Protected declarations }
  function  DetermineLogicalState(ALogicInput:Boolean;
            AIndex:Integer):Boolean; override;
  procedure LogicLineDragDrop(Sender, Source: TObject; X,
            Y: Integer);
  procedure Paint; override; //Implements TCustomControl virtual method
public
  { Public declarations }
  constructor Create(AOwner: TComponent); override;
  destructor  Destroy;override;
published
  { Published declarations }
end;

TLogicSetReset = class(TLogicBinaryOp)
private
  { Private declarations }
protected
    procedure OnLogicModeChange; override;
  { Protected declarations }
  function  DetermineLogicalState(ALogicInput:Boolean;
            AIndex:Integer):Boolean; override;
  procedure LogicLineDragDrop(Sender, Source: TObject; X,
            Y: Integer);
  procedure Paint; override; //Implements TCustomControl virtual method
public
  { Public declarations }
  constructor Create(AOwner: TComponent); override;
  destructor  Destroy;override;
published
  { Published declarations }
end;

implementation

uses LogicUnaryOp, LogicConnect, LogicWindow;

{ TOutputEvent based on TListColumn }

constructor TboOutputEvent.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FOutputControl := nil; //V2.26
  FOutputEvent := nil;
  FOutputMove := nil;  //V2.26
  FPriority := -1; //V2.26
end;

destructor TboOutputEvent.Destroy;
begin
  inherited Destroy;
end;

procedure TboOutputEvent.SetOutputControl(Value: TLogicLine);
begin
  if FOutputControl <> Value then
  begin
    FOutputControl := Value;
  end;
end;

procedure TboOutputEvent.SetOutputEvent(Value: TLogicChangeEvent);
begin
  if @FOutputEvent <> @Value then
  begin
    FOutputEvent := Value;
  end;
end;

procedure TboOutputEvent.SetOutputMove(Value: TLogicMoveOutputPos);
begin
  if @FOutputMove <> @Value then
  begin
    FOutputMove := Value;
  end;
end;

procedure TboOutputEvent.Assign(Source: TPersistent);
var
  AOutputEvent: TboOutputEvent;
begin
  if Source is TboOutputEvent then
  begin
    AOutputEvent := TboOutputEvent(Source);
    OutputControl := AOutputEvent.OutputControl;
    OutputEvent   := AOutputEvent.OutputEvent;
    OutputMove    := AOutputEvent.OutputMove;
  end
  else inherited Assign(Source);
end;

{ TOutputEvents based on TListColumns }

constructor TboOutputEvents.Create(AOwner: TLogicBinaryOp);
begin
  inherited Create(TboOutputEvent);
  FOwner := AOwner;
end;

function TboOutputEvents.GetItem(Index: Integer): TboOutputEvent;
begin
  Result := TboOutputEvent(inherited GetItem(Index));
end;

procedure TboOutputEvents.SetItem(Index: Integer; Value: TboOutputEvent);
begin
  inherited SetItem(Index, Value);
end;

function TboOutputEvents.Add: TboOutputEvent;
begin
  Result := TboOutputEvent(inherited Add);
end;

function TboOutputEvents.Owner: TLogicBinaryOp;
begin
  Result := FOwner;
end;

function TboOutputEvents.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

{ TLogicBinaryOp }
procedure TLogicBinaryOp.SetOutputEvents(Value: TboOutputEvents);
begin
  FOutputEvents.Assign(Value);
end;

constructor TLogicBinaryOp.Create(AOwner: TComponent);
var
  i : integer;
begin
  inherited Create(AOwner);
  FLogicInputSource1 := nil;
  FLogicInputSource2 := nil;
  FOutputEvents := TboOutputEvents.Create(Self);
  FColorInput1 := clBlack;
  FColorInput2 := clBlack;
  FColorOutput := clBlack;
  FColorState  := clBlack;
  FGridEnabled := false;
  FGridSize    := 0;
  GridSize     := 5;
  GridEnabled  := true;
  Height := 50;
  Width  := 50;
  Color := clWhite;
  FInput1 := true;
  FInput2 := true;
  FInputRecursionCounter1 := 0; //V2.27
  FInputRecursionCounter2 := 0; //V2.27
  FInvert1 := false;
  FInvert2 := false;
  Input1 := false; //force input update
  Input2 := false;
  FInvertOut := false;
  FOutputState := true;
  FLogicState := false;
  FInitLogic := true;
  LogicState := false; //force logic/output update
  FLogicMode := lmDesign;
  with Canvas do
  begin
    Pen.Style := psSolid;
    Pen.Color := clBlack;
    Brush.Style := bsClear;
  end;

  //Popup Menu
  FPopupMenu := TPopUpMenu.Create(self);
  FPopupMenu.AutoPopup := true;
  SetLength(FPopupItem,7);
  for i := 0 to 6 do begin
    FPopupItem[i] := TMenuItem.Create(Self);
    FPopupMenu.Items.Add(FPopupItem[i]);
  end;
  FPopupItem[0].Caption := 'Invert Input 1';
  FPopupItem[0].OnClick := puiBinOpInvertInput1;
  FPopupItem[1].Caption := 'Invert Input 2';
  FPopupItem[1].OnClick := puiBinOpInvertInput2;
  FPopupItem[2].Caption := 'Invert Output';
  FPopupItem[2].OnClick := puiBinOpInvertOutput;
  FPopupItem[3].Caption := 'Disconnect Input 1';
  FPopupItem[3].OnClick := puiBinOpDisconnectInput1;
  FPopupItem[4].Caption := 'Disconnect Input 2';
  FPopupItem[4].OnClick := puiBinOpDisconnectInput2;
  FPopupItem[5].Caption := 'Add Output LogicLine'; //v2.28
  FPopupItem[5].OnClick := puiBinOpAddOutputLogicLine; //v2.28
  FPopupItem[6].Caption := 'Delete LogicElement';
  FPopupItem[6].OnClick := puiBinOpDeleteLogicElement;
  PopupMenu := FPopupMenu;

  OnDragOver := LogicLineDragOver;
  OnMouseDown := uoMouseDown;
  OnMouseMove := uoMouseMove;
  OnMouseUp   := uoMouseUp;
end;

destructor TLogicBinaryOp.Destroy;
begin
  if Assigned(FLogicInputSource1) then
  try
    //Before leaving, delete input links
    LogicInputSource1 := nil;
  except
  end;
  if Assigned(FLogicInputSource2) then
  try
    //Before leaving, delete input links
    LogicInputSource2 := nil;
  except
  end;
  try
    FPopupMenu.Free;
    //also frees each instance of FpuiLogicDelay
  except
  end;
  try
    SetLength(FPopupItem,0);
  except
  end;
  inherited Destroy;
end;

//This is an important tool for removing dependencies to linked components
procedure TLogicBinaryOp.Notification(AComponent:TComponent ; Operation:TOperation);
begin
  if (Operation=opRemove) and (AComponent=LogicInputSource1) then
    begin
    //Unlink myself from the component being destroyed
    LogicInputSource1 := nil;
    end;
  if (Operation=opRemove) and (AComponent=LogicInputSource2) then
    begin
    //Unlink myself from the component being destroyed
    LogicInputSource2 := nil;
    end;
  inherited Notification(Acomponent,Operation);
end;

procedure TLogicBinaryOp.LogicLineDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
end;

procedure TLogicBinaryOp.SetInvert1(Value:Boolean);
var
  RealInput : boolean;
begin
  if Value <> FInvert1 then
  begin
    if FInvert1 then RealInput := not Input1 else RealInput := Input1;
    FInvert1 := Value;
    ReceiveLogicChange1(RealInput,True);
  end;
end;

procedure TLogicBinaryOp.SetInvert2(Value:Boolean);
var
  RealInput : boolean;
begin
  if Value <> FInvert2 then
  begin
    if FInvert2 then RealInput := not Input2 else RealInput := Input2;
    FInvert2 := Value;
    ReceiveLogicChange2(RealInput,True);
  end;
end;

procedure TLogicBinaryOp.SetInvertOut(Value:Boolean);
begin
  if Value <> FInvertOut then
  begin
    FInvertOut := Value;
    InitLogic := True;
    LogicState := LogicState;
  end;
end;

  //Store my input source object, and add myself to its output list
procedure TLogicBinaryOp.SetLogicInputSource1(ALogicElement: TLogicLine); 
begin
  //Before assigning new InputSource, delete old InputSource events
  if Assigned(FLogicInputSource1) then
  begin
    if FLogicInputSource1 is TLogicLine then
    begin
        TLogicLine(FLogicInputSource1).AssignOutput(Self,ReceiveLogicChange1,oaDelete);
        TLogicLine(FLogicInputSource1).MoveEndPoint(-1,-10);
    end;
    //Force input to false on disconnecting input
    ReceiveLogicChange1(false,false);
  end;

  try
    FLogicInputSource1 := TLogicLine(ALogicElement);
  except
    FLogicInputSource1 := nil;
  end;

  if Assigned(FLogicInputSource1) then
  begin
    if FLogicInputSource1 is TLogicLine then
    begin
      TLogicLine(FLogicInputSource1).AssignOutput(Self,ReceiveLogicChange1,oaAdd);
      MoveConnectedInputs;
    end;
  end;
end;

//Store my input source object, and add myself to its output list
procedure TLogicBinaryOp.SetLogicInputSource2(ALogicElement: TLogicLine); 
begin
  //Before assigning new InputSource, delete old InputSource events
  if Assigned(FLogicInputSource2) then
  begin
    if FLogicInputSource2 is TLogicLine then
    begin
        TLogicLine(FLogicInputSource2).AssignOutput(Self,ReceiveLogicChange2,oaDelete);
        TLogicLine(FLogicInputSource2).MoveEndPoint(-1,-10);
    end;
    //Force input to false on disconnecting input
    ReceiveLogicChange2(false,false);
  end;

  try
    FLogicInputSource2 := TLogicLine(ALogicElement);
  except
    FLogicInputSource2 := nil;
  end;
  
  if Assigned(FLogicInputSource2) then
  begin
    if FLogicInputSource2 is TLogicLine then
    begin
      TLogicLine(FLogicInputSource2).AssignOutput(Self,ReceiveLogicChange2,oaAdd);
      MoveConnectedInputs;
    end;
  end;
end;

//Add/delete methods in my output list
procedure TLogicBinaryOp.AssignOutput(ALogicLine:TLogicLine;
          ALogicChangeMethod:TLogicChangeEvent;
          ALogicMoveOutputMethod:TLogicMoveOutputPos;
          AnOutputAction:TOutputActionType);
begin 
  case AnOutputAction of
  oaAdd:
    //add ALogicChange (pointer to an OnLogicChange event)
    // to a list of TLogicChangeEvents
    AddOutput(ALogicLine,ALogicChangeMethod,ALogicMoveOutputMethod);
  oaDelete:
    //Lookup item = ALogicChangeMethod and delete it
    DeleteOutput(ALogicLine);
  end; //case
end;

//Override this function to perform custom logic
function TLogicBinaryOp.DetermineLogicalState(ALogicInput:Boolean;
         AIndex:Integer):Boolean;
begin
  Case AIndex of
  1: begin
    Input1 := ALogicInput;
    end;
  2: begin
    Input2 := ALogicInput;
    end;
  end;
  Result := ALogicInput;
end;

procedure TLogicBinaryOp.ReceiveLogicChange1(ALogicState,Init:Boolean);
begin
  inc(FInputRecursionCounter1);  //V2.27
  if FInputRecursionCounter1 < 100 then  //V2.27
    begin
    InitLogic := Init;
    LogicState := DetermineLogicalState(ALogicState, 1);
    if FInputRecursionCounter1 > 0 then dec(FInputRecursionCounter1); //V2.27
    end
  else
    begin
    if FInputRecursionCounter1 > 0 then dec(FInputRecursionCounter1); //V2.27
    FColorInput1 := clYellow;  //V2.27
    Invalidate;  //V2.27
    end;
end;

procedure TLogicBinaryOp.ReceiveLogicChange2(ALogicState,Init:Boolean);
begin
  inc(FInputRecursionCounter2);  //V2.27
  if FInputRecursionCounter2 < 100 then  //V2.27
    begin
    InitLogic := Init;
    LogicState := DetermineLogicalState(ALogicState, 2);
    if FInputRecursionCounter2 > 0 then dec(FInputRecursionCounter2); //V2.27
    end
  else
    begin
    if FInputRecursionCounter2 > 0 then dec(FInputRecursionCounter2); //V2.27
    FColorInput2 := clYellow;  //V2.27
    Invalidate;  //V2.27
    end;
end;

function TLogicBinaryOp.GetPositionOnGrid(APos:Integer):Integer;
var
  Delta : integer;
begin
  if GridEnabled and (GridSize > 0) then
    begin
    Delta := APos mod GridSize;
    if (Delta / GridSize) >= 0.5 then
      Result := APos + GridSize - Delta
    else
      Result := APos - Delta;
    end
  else
    Result := APos;
end;

procedure TLogicBinaryOp.SetGridEnabled(Value:Boolean);
begin
  if Value <> FGridEnabled then
    if not FGridEnabled then  //object may not have been on grid
      begin
      FGridEnabled := Value;
      Left := GetPositionOnGrid(Left);
      Top  := GetPositionOnGrid(Top);
      end
  else
      FGridEnabled := Value;
end;

procedure TLogicBinaryOp.SetGridSize(Value:TGridSize);
begin
  if Value <> FGridSize then
  begin
    FGridSize := Value;
  end;
end;

procedure TLogicBinaryOp.SetInitLogic(Init:Boolean);
begin
  FInitLogic := Init;
end;

procedure TLogicBinaryOp.SetLogicState(State:Boolean);
var
  i : integer;
  AnOutputEvent : TboOutputEvent;
begin
  if (State <> FLogicState) or InitLogic then
  begin
    FLogicState := State;
    OutputState := FLogicState xor InvertOut;
    for i := 0 to OutputEvents.Count-1 do
    begin
      try
      AnOutputEvent := OutputEvents.Items[i];
      AnOutputEvent.OutputEvent(OutputState,false);
      except
      end;
    end;
    If InitLogic then InitLogic := false;
    If FLogicState then FColorState := clLime else FColorState := clRed;
    Invalidate;
    if Assigned(OnLogicChange) then OnLogicChange(OutputState,FInitLogic);
  end;
end;

procedure TLogicBinaryOp.SetOutputState(Value:Boolean);
begin
  if Value <> FOutputState then
  begin
    FOutputState := Value;
    if FOutputState then FColorOutput := clLime else FColorOutput := clRed;
    invalidate;
  end;
end;

procedure TLogicBinaryOp.AddOutput(ALogicLine:TLogicLine;
          ALogicChangeMethod:TLogicChangeEvent;
          ALogicMoveOutputMethod:TLogicMoveOutputPos);
var
  AnOutputEvent : TboOutputEvent;
begin
  //add ALogicChangeMethod (pointer to an LogicChange event)
  // to a list of TLogicChangeEvents
  AnOutputEvent := OutputEvents.Add;
  AnOutputEvent.OutputControl := ALogicLine;
  AnOutputEvent.OutputEvent := ALogicChangeMethod;
  AnOutputEvent.OutputMove  := ALogicMoveOutputMethod;
  //Connect the output
  MoveConnectedOutputs;
  //Initialize the output
  InitLogic := True;
  LogicState := LogicState;
end;

procedure TLogicBinaryOp.DeleteOutput(ALogicLine:TLogicLine);
var
  i : integer;
begin
  //Lookup item = ALogicChangeMethod and delete it
  for i:=OutputEvents.Count-1 downto 0 do
  begin
    if OutputEvents[i].OutputControl = ALogicLine then
      OutputEvents.Delete(i);
  end;
end;

procedure TLogicBinaryOp.Paint;
begin
  inherited Paint;
  if Selected then Canvas.Rectangle(0,0,Width,Height);
  //Draw Input1
  Canvas.Pen.Color := FColorInput1;
  if FInvert1 then
  begin
    Canvas.Ellipse(0,lboY1-lboR,lboXi-lboR,lboY1+lboR);
    Canvas.Polyline([Point(lboXi-lboR,lboY1),Point(lboXi,lboY1)]);
  end else begin
    Canvas.Polyline([Point(0,lboY1),Point(lboXi,lboY1)]);
  end;
  //Draw Input 2
  Canvas.Pen.Color := FColorInput2;
  if FInvert2 then
  begin
    Canvas.Ellipse(0,lboY2-lboR,lboXi-lboR,lboY2+lboR);
    Canvas.Polyline([Point(lboXi-lboR,lboY2),Point(lboXi,lboY2)]);
  end else begin
    Canvas.Polyline([Point(0,lboY2),Point(lboXi,lboY2)]);
  end;
  //Draw Output
  Canvas.Pen.Color := FColorOutput;
  if FInvertOut then
  begin
    Canvas.Polyline([Point(lboXo,lboYo),Point(lboXo+lboR,lboYo)]);
    Canvas.Ellipse(lboXo+lboR,lboYo-lboR,Width,lboYo+lboR);
  end else begin
    Canvas.Polyline([Point(lboXo,lboYo),Point(Width,lboYo)]);
  end;
end;

function TLogicBinaryOp.GetBoundsRect:TRect;
begin
  Result := BoundsRect;
end;

procedure TLogicBinaryOp.uoMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if LogicMode = lmDesign then
  begin
    if Button = mbLeft then
      begin
      Selected := true;
      StartXY := ClientToParent(Point(X,Y),TPanel(Parent));
      ParentExtent := GetDeltaRect(TScrollBox(Parent.Parent).ClientRect,
        Point(-TPanel(Parent).Left,-TPanel(Parent).Top));
      LastRect:= GetBoundsRect;
      end;
  end;
end;

procedure TLogicBinaryOp.uoMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if (Button = mbLeft) and (LogicMode = lmDesign) then
  begin
    Selected := false;
  end;
end;

procedure TLogicBinaryOp.uoMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
var
  AbsXY, DeltaXY : TPoint;
  NewExtent : TRect;
begin
  if (ssLeft in Shift) and (LogicMode = lmDesign) then { make sure button is down }
  begin
    AbsXY := ClientToParent(Point(X,Y),TPanel(Parent));
    DeltaXY.X := GetPositionOnGrid(AbsXY.X - StartXY.X);
    DeltaXY.Y := GetPositionOnGrid(AbsXY.Y - StartXY.Y);
    NewExtent := GetDeltaRect(LastRect,DeltaXY);
    if ( PtInRect(ParentExtent, NewExtent.TopLeft) and
         PtInRect(ParentExtent, NewExtent.BottomRight) ) then
      begin
      Inc(StartXY.X,DeltaXY.X);
      Inc(StartXY.Y,DeltaXY.Y);
      end
    else
      begin
      if NewExtent.Left < ParentExtent.Left then
        DeltaXY.X := GetPositionOnGrid(ParentExtent.Left - LastRect.Left);
      if NewExtent.Right > ParentExtent.Right then
        DeltaXY.X := GetPositionOnGrid(ParentExtent.Right - LastRect.Right);
      if NewExtent.Top < ParentExtent.Top then
        DeltaXY.Y := GetPositionOnGrid(ParentExtent.Top - LastRect.Top);
      if NewExtent.Bottom > ParentExtent.Bottom then
        DeltaXY.Y := GetPositionOnGrid(ParentExtent.Bottom - LastRect.Bottom);
      NewExtent := GetDeltaRect(LastRect,DeltaXY);
      Inc(StartXY.X,DeltaXY.X);
      Inc(StartXY.Y,DeltaXY.Y);
      end;
    Self.Left := NewExtent.Left;
    Self.Top  := NewExtent.Top;
    LastRect := NewExtent;
    MoveConnectedIO;
  end;
end;

procedure TLogicBinaryOp.MoveConnectedIO;
begin
  MoveConnectedInputs;
  MoveConnectedOutputs;
end;

procedure TLogicBinaryOp.MoveConnectedInputs;
var
  X,Y : integer;
begin
  X := Left;
  Y := Top;
  if Assigned(LogicInputSource1) then
    if LogicInputSource1 is TLogicLine then
      TLogicLine(LogicInputSource1).PlacePolyPoint(-1,X,Y+lboY1);
  if Assigned(LogicInputSource2) then
    if LogicInputSource2 is TLogicLine then
      TLogicLine(LogicInputSource2).PlacePolyPoint(-1,X,Y+lboY2);
end;

procedure TLogicBinaryOp.MoveConnectedOutputs;
var
  i : integer;
  Xo,Yo : integer;
  AnOutputMove : TboOutputEvent;
begin
  Xo := Left + Width;
  Yo := Top  + (Height div 2);
  LogicForm.StatusBar.Panels[0].Text := 'Output position X = ' +
    IntToStr(Xo) + ', Y = ' + IntToStr(Yo);  
  for i := 0 to OutputEvents.Count-1 do
  begin
    try
    AnOutputMove := OutputEvents.Items[i];
    AnOutputMove.OutputMove(0,Xo,Yo);
    except
    end;
  end;
end;

procedure TLogicBinaryOp.SetSelected(Value:Boolean);
begin
  if Value <> FSelected then
  begin
    FSelected := Value;
    Invalidate;
  end;
end;

procedure TLogicBinaryOp.puiBinOpInvertInput1(Sender: TObject);
begin
  with Self as TLogicBinaryOp do begin
    Invert1 := not Invert1;
  end;
end;

procedure TLogicBinaryOp.puiBinOpInvertInput2(Sender: TObject);
begin
  with Self as TLogicBinaryOp do begin
    Invert2 := not Invert2;
  end;
end;

procedure TLogicBinaryOp.puiBinOpInvertOutput(Sender: TObject);
begin
  with Self as TLogicBinaryOp do begin
    InvertOut := not InvertOut;
  end;
end;

procedure TLogicBinaryOp.puiBinOpDisconnectInput1(Sender: TObject);
begin
  with Self as TLogicBinaryOp do begin
    LogicInputSource1 := nil;
    Invalidate;
  end;
end;

procedure TLogicBinaryOp.puiBinOpDisconnectInput2(Sender: TObject);
begin
  with Self as TLogicBinaryOp do begin
    LogicInputSource2 := nil;
    Invalidate;
  end;
end;

procedure TLogicBinaryOp.puiBinOpDeleteLogicElement(Sender: TObject);
var
  i : integer;
begin
  //Set priority of output logiclines to -1  //V2.26
  for i := FOutputEvents.Count-1 downto 0 do
  begin
    FOutputEvents[i].Priority := -1;
    if FOutputEvents[i].FOutputControl.OutputEvents.Count = 0 then  //v2.28
      FOutputEvents[i].FOutputControl.Free;
  end;
  FOutputEvents.UpdatePriorities;

  TLogicBinaryOp(Self).Free;
end;

procedure TLogicBinaryOp.puiBinOpAddOutputLogicLine(Sender: TObject); //v2.28
var
  Xo,Yo : integer;
  ALogicLine : TLogicLine;
begin
  Xo := Left + Width;
  Yo := Top  + (Height div 2);
  ALogicLine := MultiPolyline.ArrayToNewPolyline([point(Xo,Yo),
    point(Xo+30,Yo+30)]);
  ALogicLine.Selected := true;
  LogicCount.Count:=LogicCount.Count+1;
  ALogicLine.Name := 'LogicLine' + IntToStr(LogicCount.Count);
  ALogicLine.LogicInputSource := Self as TLogicBinaryOp;
end;

procedure TLogicBinaryOp.SetInput1(Value: Boolean);
begin
  if (Value xor FInvert1) <> FInput1 then
  begin
    FInput1 := Value xor FInvert1;
    if FInput1 then FColorInput1 := clLime else FColorInput1 := clRed;
    invalidate;
  end;
end;

procedure TLogicBinaryOp.SetInput2(Value: Boolean);
begin
  if (Value xor FInvert2) <> FInput2 then
  begin
    FInput2 := Value xor FInvert2;
    if FInput2 then FColorInput2 := clLime else FColorInput2 := clRed;
    invalidate;
  end;
end;

procedure TLogicBinaryOp.SetLogicMode(Value: TLogicMode);
begin
  if FLogicMode <> Value then
  begin
    FLogicMode := Value;
    OnLogicModeChange;
  end;
end;

procedure TLogicBinaryOp.OnLogicModeChange;
begin
  //Override in descendants to update popup menus
end;

procedure TLogicBinaryOp.Loaded; //V2.26
begin
  inherited;
  FOutputEvents.PrioritizeOutputs;
end;

procedure TLogicBinaryOp.ChangePriority(ALogicLine: TLogicLine;
  ALogicPriority: Integer);  //V2.26
var
  i : integer;
begin
  //Lookup ALogicLine and change the priority of its output event
  for i:=0 to OutputEvents.Count-1 do
  begin
    if OutputEvents[i].OutputControl = ALogicLine then
    begin
      OutputEvents[i].Priority := ALogicPriority;
      OutputEvents.PrioritizeOutputs;
    end;
  end;
end;

{TLogicAnd}

constructor TLogicAnd.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Top := 50;
  Left := 50;
  Visible := true;
  OnDragDrop := LogicLineDragDrop;
end;

destructor TLogicAnd.Destroy;
begin
  //Inherited TLogicBinaryOp unlinks all inputs
  inherited Destroy;
end;

procedure TLogicAnd.LogicLineDragDrop(Sender, Source: TObject; X,
  Y: Integer);
begin
  if IsDragObject(Source) and (Source is TLogicLineDragObject) then
    with (Source as TLogicLineDragObject) do
    begin
      if LinkType = ltInput then
      begin
        if ( Y < (Height div 2) ) then
        begin
          LogicInputSource1 := LogicLineID as TLogicLine;
        end else begin
          LogicInputSource2 := LogicLineID as TLogicLine;
        end;
      end else begin
        LogicLineID.LogicInputSource := Sender as TLogicAnd;
      end;
    end;
end;

procedure TLogicAnd.Paint;
var
  x,y : integer;
begin
  inherited Paint;
  //Draw main body in LogicState color
  Canvas.Pen.Color := FColorState;
  x := Width div 3;
  y := Height div 3;
  Canvas.MoveTo(x,0);
  Canvas.LineTo(x,Height);
  Canvas.MoveTo(x,y);
  Canvas.LineTo(x*2,y);
  Canvas.LineTo(x*2,y*2);
  Canvas.LineTo(x,y*2);
  Canvas.TextOut((Width div 2)-3, (Height div 2)-6,'A');
end;

function TLogicAnd.DetermineLogicalState(ALogicInput:Boolean;
         AIndex:Integer):Boolean;
begin
  Case AIndex of
  1: begin
    Input1 := ALogicInput;
    end;
  2: begin
    Input2 := ALogicInput;
    end;
  end;
  Result := Input1 and Input2;
end;

procedure TLogicAnd.OnLogicModeChange;
begin
{
  FPopupItem[0].Caption := 'Invert Input 1';
  FPopupItem[1].Caption := 'Invert Input 2';
  FPopupItem[2].Caption := 'Invert Output';
  FPopupItem[3].Caption := 'Disconnect Input 1';
  FPopupItem[4].Caption := 'Disconnect Input 2';
  FPopupItem[5].Caption := 'Add Output LogicLine'; //v2.28
  FPopupItem[6].Caption := 'Delete LogicElement';
}
  FPopupItem[3].Enabled := LogicMode = lmDesign;
  FPopupItem[4].Enabled := LogicMode = lmDesign;
  FPopupItem[5].Enabled := LogicMode = lmDesign; //v2.28
  FPopupItem[6].Enabled := LogicMode = lmDesign;
end;

{TLogicOr}

constructor TLogicOr.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Top := 50;
  Left := 50;
  Visible := true;
  OnDragDrop := LogicLineDragDrop;
end;

destructor TLogicOr.Destroy;
begin
  //Inherited TLogicBinaryOp unlinks all inputs
  inherited Destroy;
end;

procedure TLogicOr.LogicLineDragDrop(Sender, Source: TObject; X,
  Y: Integer);
begin
  if IsDragObject(Source) and (Source is TLogicLineDragObject) then
    with (Source as TLogicLineDragObject) do
    begin
      if LinkType = ltInput then
      begin
        if ( Y < (Height div 2) ) then
        begin
          LogicInputSource1 := LogicLineID as TLogicLine;
        end else begin
          LogicInputSource2 := LogicLineID as TLogicLine;
        end;
      end else begin
        LogicLineID.LogicInputSource := Sender as TLogicOr;
      end;
    end;
end;

procedure TLogicOr.Paint;
var
  x,y : integer;
begin
  inherited Paint;
  //Draw main body in LogicState color
  Canvas.Pen.Color := FColorState;
  x := Width div 3;
  y := Height div 3;
  Canvas.MoveTo(x,0);
  Canvas.LineTo(x,Height);
  Canvas.MoveTo(x,y);
  Canvas.LineTo(x*2,y);
  Canvas.LineTo(x*2,y*2);
  Canvas.LineTo(x,y*2);
  Canvas.TextOut((Width div 2)-3, (Height div 2)-6,'O');
end;

function TLogicOr.DetermineLogicalState(ALogicInput:Boolean;
         AIndex:Integer):Boolean;
begin
  Case AIndex of
  1: begin
    Input1 := ALogicInput;
    end;
  2: begin
    Input2 := ALogicInput;
    end;
  end;
  Result := Input1 or Input2;
end;

procedure TLogicOr.OnLogicModeChange;
begin
{
  FPopupItem[0].Caption := 'Invert Input 1';
  FPopupItem[1].Caption := 'Invert Input 2';
  FPopupItem[2].Caption := 'Invert Output';
  FPopupItem[3].Caption := 'Disconnect Input 1';
  FPopupItem[4].Caption := 'Disconnect Input 2';
  FPopupItem[5].Caption := 'Add Output LogicLine'; //v2.28
  FPopupItem[6].Caption := 'Delete LogicElement';
}
  FPopupItem[3].Enabled := LogicMode = lmDesign;
  FPopupItem[4].Enabled := LogicMode = lmDesign;
  FPopupItem[5].Enabled := LogicMode = lmDesign;  //v2.28
  FPopupItem[6].Enabled := LogicMode = lmDesign;
end;

{TLogicSetReset}

constructor TLogicSetReset.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Top := 50;
  Left := 50;
  Visible := true;
  OnDragDrop := LogicLineDragDrop;
end;

destructor TLogicSetReset.Destroy;
begin
  //Inherited TLogicBinaryOp unlinks all inputs
  inherited Destroy;
end;

procedure TLogicSetReset.LogicLineDragDrop(Sender, Source: TObject; X,
  Y: Integer);
begin
  if IsDragObject(Source) and (Source is TLogicLineDragObject) then
    with (Source as TLogicLineDragObject) do
    begin
      if LinkType = ltInput then
      begin
        if ( Y < (Height div 2) ) then
        begin
          LogicInputSource1 := LogicLineID as TLogicLine;
        end else begin
          LogicInputSource2 := LogicLineID as TLogicLine;
        end;
      end else begin
        LogicLineID.LogicInputSource := Sender as TLogicSetReset;
      end;
    end;
end;

procedure TLogicSetReset.Paint;
var
  x : integer;
begin
  inherited Paint;
  //Draw main body in LogicState color
  Canvas.Pen.Color := FColorState;
  x := Width div 3;
  Canvas.MoveTo(x,0);
  Canvas.LineTo(x,Height-1);
  Canvas.LineTo(x*2,Height-1);
  Canvas.LineTo(x*2,0);
  Canvas.LineTo(x,0);
  Canvas.TextOut((Width div 2)-3, (Height div 3)-6,'S');
  Canvas.TextOut((Width div 2)-3, ((Height div 3)*2)-6,'R');
end;

function TLogicSetReset.DetermineLogicalState(ALogicInput:Boolean;
         AIndex:Integer):Boolean;
begin
  Case AIndex of
  1: begin
    Input1 := ALogicInput;
    end;
  2: begin
    Input2 := ALogicInput;
    end;
  end;
  Result := (Input1 or (LogicState and not Input2)) and not Input2;
end;

procedure TLogicSetReset.OnLogicModeChange;
begin
{
  FPopupItem[0].Caption := 'Invert Input 1';
  FPopupItem[1].Caption := 'Invert Input 2';
  FPopupItem[2].Caption := 'Invert Output';
  FPopupItem[3].Caption := 'Disconnect Input 1';
  FPopupItem[4].Caption := 'Disconnect Input 2';
  FPopupItem[5].Caption := 'Add Output LogicLine'; //v2.28
  FPopupItem[6].Caption := 'Delete LogicElement';
}
  FPopupItem[3].Enabled := LogicMode = lmDesign;
  FPopupItem[4].Enabled := LogicMode = lmDesign;
  FPopupItem[5].Enabled := LogicMode = lmDesign; //v2.28
  FPopupItem[6].Enabled := LogicMode = lmDesign;
end;

procedure TboOutputEvent.SetPriority(Value: Integer);  //V2.26
begin
  if Value <> FPriority then
  begin
    FPriority := Value;
  end;
end;

procedure TboOutputEvents.PrioritizeOutputs; //V2.26
var
  i, j : integer;
begin
  //set Priority from LogicLines
  for i:= 0 to Self.Count-1 do
  begin
    Items[i].Priority := Items[i].OutputControl.Priority;
  end;
  //Change Output order to match Self.Priority
  QuickSort(0,Count-1, ComparePriorities);
  //Synch Output receivers to output events
  UpdatePriorities;
end;

procedure TboOutputEvents.QuickSort(L, R: Integer;
  SCompare: TCollectionSortCompare);  //V2.26
var
  I, J: Integer;
  P: Integer; //Priority Value, not position
  T: TboOutputEvent;
begin
  if R < L then exit; //true if no OutputEvents defined
  repeat
    I := L;
    J := R;
    P := Items[(L + R) shr 1].FPriority;
    repeat
      while SCompare(Items[I].FPriority, P) < 0 do
        Inc(I);
      while SCompare(Items[J].FPriority, P) > 0 do
        Dec(J);
      if I <= J then
      begin
        Items[J].Index := I;
        if I+1 < R then Items[I+1].Index := J;
        {T := Items[I];
        Items[I] := Items[J];
        Items[J] := T; }
        Inc(I);
        Dec(J);
      end;
    until I > J;
    if L < J then
      QuickSort(L, J, SCompare);
    L := I;
  until I >= R;
end;

procedure TboOutputEvents.UpdatePriorities; //V2.26
var
  i : integer;
begin
  for i := 0 to Self.Count-1 do
  begin
    //set FPriority for each output if not default
    if Items[i].FPriority <> -1 then
      Items[i].FPriority := i;
    //feed back priority to LogicLine
    TLogicLine(Items[i].OutputControl).Priority := Items[i].FPriority;
  end;
end;

initialization
RegisterClasses([TLogicBinaryOp, TLogicAnd, TLogicOr, TLogicSetReset]);
finalization

end.
