unit Globals;

interface

uses Classes, ExtCtrls, StdCtrls, Dialogs, Controls, Menus, SysUtils, Forms,
      Types, Windows, Graphics, Consts, Math;

const
  TNotRadius = 4;  //radius of not circles
  // A list of handy constants
  Numeric = ['0'..'9'];
  Alpha = ['A'..'Z', 'a'..'z'];
  AlphaNum = Alpha + Numeric;
  CRLF = Chr(13); //VK_RETURN = 13

type
  TGridSize = 0..10;
  TLogicMode = (lmRun,lmDesign);
  TLogicAutoFit = (afAuto,afFixed);
  //QuickSort types from TList in Classes.pas
  TCollectionSortCompare = function (Item1, Item2: Integer): Integer;


  {TxxOutputEvent Support}
  //Points to TLogicXxx.ReceiveLogicChange(ALogicState,Init:Boolean);
  TLogicChangeEvent = procedure(InputLogicState,Init:Boolean) of object;
  //Points to TgcPolyLine.PlacePolyPoint(PointId,x,y: Integer);
  TLogicMoveOutputPos = procedure(PointId,x,y: Integer) of object;
  TOutputActionType = (oaAdd,oaDelete);

  TLogicTitle = class(TLabel)
  private
    FCount: Integer;
    FPopupMenu : TPopupMenu;
    FPopupItem : array of TMenuItem;
    FVersion : string;
    StartX, StartY : integer;
    procedure ltMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ltMouseMove(Sender: TObject; Shift: TShiftState;
      X, Y: Integer);
    procedure ltMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure SetCount(Value:Integer);
    procedure SetVersion(Value:string);
  protected
    procedure puiChangeDescription(Sender: TObject);
    procedure puiShowLogicCount(Sender: TObject);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Count:Integer read FCount write SetCount stored true default 0;
    property Version: string read FVersion write SetVersion stored true;
  end;

  TLogicDescription = class(TLabel)
  private
    FPopupMenu : TPopupMenu;
    FPopupItem : array of TMenuItem;
    StartX, StartY : integer;
    procedure ltMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ltMouseMove(Sender: TObject; Shift: TShiftState;
      X, Y: Integer);
    procedure ltMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
  protected
    procedure puiEditText(Sender: TObject);
    procedure puiDelete(Sender: TObject);
    procedure puiSetWidth(Sender: TObject);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Caption;
    property Height;
    property Left;
    property Top;
    property Width;
  end;

function EditControlDescription(AControl:TControl; ADefault:string):string;
function EditControlHint(AControl:TControl; ADefault:string):string;
function EditControlWidth(AControl:TControl; ADefault:integer):integer;
function EditOutputPriority(AControl:TComponent; ADefault:integer):integer; //V2.26
function EditTextDescription(AControl:TControl; const ACaption, APrompt,
    ADefault: string):string;
function GetAveCharSize(Canvas: TCanvas): TPoint;
function GetDeltaRect(OrigRect:TRect; DeltaXY:TPoint):TRect;
function GetGridPosition(APos:Integer):Integer;
function InputQuery1(const ACaption, APrompt: string; AList: TStringList;
          var Value: string): Boolean;
function IsXYOnLine(X,Y,XA,YA,XB,YB,Tolerance: Double): Boolean;
  //(X,Y) is test point, (XA,YA) and (XB,YB) are first & second points on line
function ComparePriorities(Item1, Item2: Integer): Integer; //V2.26
function SelectConnSource(const ACaption, APrompt: string; AList: TStringList;
          ADefault: string): string;

implementation

uses LogicWindow;

{ Global Utilities }

function GetAveCharSize(Canvas: TCanvas): TPoint;
var
  I: Integer;
  Buffer: array[0..51] of Char;
begin
  for I := 0 to 25 do Buffer[I] := Chr(I + Ord('A'));
  for I := 0 to 25 do Buffer[I + 26] := Chr(I + Ord('a'));
  GetTextExtentPoint(Canvas.Handle, Buffer, 52, TSize(Result));
  Result.X := Result.X div 52;
end;

function EditControlDescription(AControl:TControl; ADefault:string):string;
var
  ACaption: string;
  APrompt : string;
begin
  if Assigned(AControl) then
  begin
    APrompt  := 'Enter new description for ' + AControl.Name;
    ACaption := 'Description Editor';
    Result := InputBox(ACaption,APrompt,ADefault);
  end else begin
    Result := ADefault;
  end;
end;

function EditControlHint(AControl:TControl; ADefault:string):string;
var
  ACaption: string;
  APrompt : string;
begin
  if Assigned(AControl) then
  begin
    APrompt  := 'Enter new hint for ' + AControl.Name;
    ACaption := 'Hint Editor';
    Result := InputBox(ACaption,APrompt,ADefault);
  end else begin
    Result := ADefault;
  end;
end;

function EditControlWidth(AControl:TControl; ADefault:integer):integer;
var
  ACaption: string;
  APrompt : string;
  sWidth : string;
begin
  if Assigned(AControl) then
  begin
    APrompt  := 'Enter new width for ' + AControl.Name;
    ACaption := 'Width Editor';
    sWidth := InputBox(ACaption,APrompt,IntToStr(ADefault));
    try
      Result := StrToInt(sWidth);
    except
      Result := ADefault;
    end;
  end else begin
    Result := ADefault;
  end;
end;

function EditOutputPriority(AControl:TComponent; ADefault:integer):integer;
var
  ACaption: string;
  APrompt : string;
  sPriority : string;
begin
  if Assigned(AControl) then
  begin
    APrompt  := 'Enter new priority for ' + AControl.Name;
    ACaption := 'Priority Editor';
    sPriority := InputBox(ACaption,APrompt,IntToStr(ADefault));
    try
      Result := StrToInt(sPriority);
    except
      Result := ADefault;
    end;
  end else begin
    Result := ADefault;
  end;
end;

function GetGridPosition(APos:Integer):Integer;
var
  Delta : integer;
  GridEnabled : boolean;
  GridSize : integer;
begin
  GridEnabled := True;
  GridSize := 5;
  if GridEnabled and (GridSize > 0) then
    begin
    Delta := APos mod GridSize;
    if (Delta / GridSize) >= 0.5 then
      Result := APos + GridSize - Delta
    else
      Result := APos - Delta;
    end
  else
    Result := APos;
end;

function IsXYOnLine(X,Y,XA,YA,XB,YB,Tolerance: Double): Boolean;
//(X,Y) is test point, (XA,YA) and (XB,YB) are first & second points on line
var

  L,R,S: Double;

begin

  Result:=False;

  L:=SQRT(((XB-XA)*(XB-XA)+(YB-YA)*(YB-YA)));

  if l<>0 then
  begin
    R:= ((YA-Y)*(YA-YB)-(XA-X)*(XB-XA))/(L*L);
    S:= ((YA-Y)*(XB-XA)-(XA-X)*(YB-YA))/(L*L);
    //if (r>0) and (r<1) then if Abs(S*L)<=Tolerance then Result:=True;  //s*l=distance
    //Note that R is the % distance along the axis of line and S*L is the
    //perpendicular distance from the axis of the line
    //Modified original formula to hit before start and beyond end of line
    if (R>=0-Tolerance/100.0) and (R<=1+Tolerance/100.0) and
      (ABS(S*L)<=Tolerance) then Result:=True;
  end;
end;

function GetDeltaRect(OrigRect:TRect; DeltaXY:TPoint):TRect;
begin
  with OrigRect do
    Result := Bounds(Left+DeltaXY.X, Top+DeltaXY.Y,
              Right-left, Bottom-Top);
end;

function ComparePriorities(Item1, Item2: Integer): Integer;  //V2.26
begin
{  Note: Delphi help on TListSortCompare type is WRONG (used AnsiComparStr)
   -1 Item1 is less than Item2
    0	Item1 is equal to Item2
    1	Item1 is greater than Item2
}
  Result := 0;
  if Item1 = Item2 then Result := 0
  else if (Item1 = -1) then Result :=  1
  else if (Item2 = -1) then Result := -1
  else if (Item1 < Item2) then Result := -1 else Result := 1;
end;

{ TLogicTitle }

procedure TLogicTitle.SetCount(Value:Integer);
begin
  if FCount <> Value then
    FCount := Value;
end;

procedure TLogicTitle.SetVersion(Value:string);
begin
  if FVersion <> Value then
    FVersion := Value;
end;

constructor TLogicTitle.Create(AOwner: TComponent);
var
  i : integer;
begin
  inherited;
  FCount := 0;
  Left := 344;
  Top  := 8;
  Color := clWhite;
  Caption := 'BLD Title';

  OnMouseDown := ltMouseDown;
  OnMouseMove := ltMouseMove;
  OnMouseUp   := ltMouseUp;

  //Popup Menu
  FPopupMenu := TPopUpMenu.Create(self);
  FPopupMenu.AutoPopup := true;
  SetLength(FPopupItem,1);
  for i := 0 to 0 do begin
    FPopupItem[i] := TMenuItem.Create(Self);
    FPopupMenu.Items.Add(FPopupItem[i]);
  end;
  FPopupItem[0].Caption := 'Change description';
  FPopupItem[0].OnClick := puiChangeDescription;
  PopupMenu := FPopupMenu;

end;

destructor TLogicTitle.Destroy;
begin
  inherited Destroy;
end;

procedure TLogicTitle.puiChangeDescription(Sender: TObject);
begin
  Self.Caption := EditControlDescription(Self,Self.Caption);
end;

procedure TLogicTitle.puiShowLogicCount(Sender: TObject);
begin
  ShowMessage(Self.Name + ' counter value: ' + IntToStr(Self.Count));
end;

procedure TLogicTitle.ltMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if Button = mbLeft then
    begin
    StartX := X;
    StartY := Y;
    end;
end;

procedure TLogicTitle.ltMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if Button = mbLeft then
  begin
    Left := Left + X - StartX;
    Top  := Top  + Y - StartY;
  end;
end;

procedure TLogicTitle.ltMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
begin
  if ssLeft in Shift then { make sure button is down }
  begin
    Left := Left + X - StartX;
    Top  := Top  + Y - StartY;
    Invalidate;
  end;
end;

{ TLogicDescription }

constructor TLogicDescription.Create(AOwner: TComponent);
var
  i : integer;
begin
  inherited;
  //Visible := false;
  WordWrap := true;
  Caption := 'Logic description';
  //AutoSize := true;
  Transparent := true;
  Left := 50;
  Top  := 50;
  Height := 50;
  Width  := 150;
  //Color := clWhite;

  OnMouseDown := ltMouseDown;
  OnMouseMove := ltMouseMove;
  OnMouseUp   := ltMouseUp;
  //OnDblClick  := puiEditText;

  //Popup Menu
  FPopupMenu := TPopUpMenu.Create(self);
  FPopupMenu.AutoPopup := false;
  SetLength(FPopupItem,3);
  for i := 0 to 2 do begin
    FPopupItem[i] := TMenuItem.Create(Self);
    FPopupMenu.Items.Add(FPopupItem[i]);
  end;
  FPopupItem[0].Caption := 'Edit description';
  FPopupItem[0].OnClick := puiEditText;
  FPopupItem[1].Caption := 'Set width';
  FPopupItem[1].OnClick := puiSetWidth;
  FPopupItem[2].Caption := 'Delete description';
  FPopupItem[2].OnClick := puiDelete;
  PopupMenu := FPopupMenu;

end;

destructor TLogicDescription.Destroy;
begin
  inherited Destroy;
end;

procedure TLogicDescription.puiDelete(Sender: TObject);
begin
  Self.Free;
end;

procedure TLogicDescription.puiEditText(Sender: TObject);
begin
  //Edit label caption with a popup dialog
  //Caption := EditControlDescription(self,self.Caption);
  Caption := EditTextDescription(self,self.Name,'Enter description:',self.Caption);
end;

procedure TLogicDescription.puiSetWidth(Sender: TObject);
begin
  //Set width of description (height adjusts to fit text
  Self.Width := EditControlWidth(self,self.width);
end;

procedure TLogicDescription.ltMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  MyPos : TPoint;
begin
  if (LogicForm.LogicMode = lmDesign) and (Button = mbLeft) then
    begin
    StartX := X;
    StartY := Y;
    end;
  //Call popup menu if simple right click
  if (Button = mbRight) and (not (ssCtrl in Shift)) and
    (not (ssDouble in Shift)) then
  begin
    MyPos := ClientToScreen(Point(X,Y));
    //disable FPopupItem[2].Caption := 'Delete description'; if lmDesign mode
    FPopupItem[2].Enabled := LogicForm.LogicMode = lmDesign;
    FPopupMenu.Popup(MyPos.X,MyPos.Y);
  end;
end;

procedure TLogicDescription.ltMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if (LogicForm.LogicMode = lmDesign) and (Button = mbLeft) then
  begin
    Left := Left + X - StartX;
    Top  := Top  + Y - StartY;
  end;
end;

procedure TLogicDescription.ltMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
begin
  if (LogicForm.LogicMode = lmDesign) and (ssLeft in Shift) then { make sure button is down }
  begin
    Left := Left + X - StartX;
    Top  := Top  + Y - StartY;
    Invalidate;
  end;
end;

{ InputQuery1 }

function InputQuery1(const ACaption, APrompt: string; AList: TStringList;
  var Value: string): Boolean;
var
  Form: TForm;
  Prompt: TLabel;
  Sources: TComboBox;
  DialogUnits: TPoint;
  ButtonTop, ButtonWidth, ButtonHeight: Integer;
  i : integer;
begin
  Result := False;
  Form := TForm.Create(Application);
  with Form do
    try
      Canvas.Font := Font;
      DialogUnits := GetAveCharSize(Canvas);
      BorderStyle := bsDialog;
      Caption := ACaption;
      ClientWidth := MulDiv(180, DialogUnits.X, 4);
      Position := poScreenCenter;
      Prompt := TLabel.Create(Form);
      with Prompt do
      begin
        Parent := Form;
        Caption := APrompt;
        Left := MulDiv(8, DialogUnits.X, 4);
        Top := MulDiv(8, DialogUnits.Y, 8);
        Constraints.MaxWidth := MulDiv(164, DialogUnits.X, 4);
        WordWrap := True;
      end;

      Sources := TComboBox.Create(Form);
      with Sources do
      begin
        Parent := Form;
        Left   := Prompt.Left;
        Top    := Prompt.Top + Prompt.Height + 5;
        Width  := MulDiv(164, DialogUnits.X, 4);
        Items.Assign(AList);
        ItemIndex := Items.IndexOf(Value); //V2.28
      end;

      ButtonTop := Sources.Top + Sources.Height + 15;
      ButtonWidth := MulDiv(50, DialogUnits.X, 4);
      ButtonHeight := MulDiv(14, DialogUnits.Y, 8);
      with TButton.Create(Form) do
      begin
        Parent := Form;
        Caption := SMsgDlgOK;
        ModalResult := mrOk;
        Default := True;
        SetBounds(MulDiv(38, DialogUnits.X, 4), ButtonTop, ButtonWidth,
          ButtonHeight);
      end;
      with TButton.Create(Form) do
      begin
        Parent := Form;
        Caption := SMsgDlgCancel;
        ModalResult := mrCancel;
        Cancel := True;
        SetBounds(MulDiv(92, DialogUnits.X, 4), ButtonTop, ButtonWidth,
          ButtonHeight);
        Form.ClientHeight := Top + Height + 13;
      end;
      if ShowModal = mrOk then
      begin
        Value := Sources.Text;
        Result := True;
      end;
    finally
      Form.Free;
    end;
end;

function SelectConnSource(const ACaption, APrompt: string; AList: TStringList;
   ADefault: string): string;
begin
  Result := ADefault;
  if not InputQuery1(ACaption, APrompt, AList, Result) then
    Result := ''; //v2.28 InputQuery1 returns true and sets Result if user hits Ok button.
end;

{ Input Description Editor }

function EditTextDescription(AControl:TControl; const ACaption, APrompt,
    ADefault: string):string;
var
  Form: TForm;
  Prompt: TLabel;
  TextMemo: TMemo;
  DialogUnits: TPoint;
  ButtonTop, ButtonWidth, ButtonHeight: Integer;
begin
  EditTextDescription := ADefault;
  Form := TForm.Create(Application);
  with Form do
    try
      Canvas.Font := Font;
      DialogUnits := GetAveCharSize(Canvas);
      BorderStyle := bsDialog;
      Caption := ACaption;
      ClientWidth := MulDiv(180, DialogUnits.X, 4);
      Position := poScreenCenter;
      Prompt := TLabel.Create(Form);
      with Prompt do
      begin
        Parent := Form;
        Caption := APrompt;
        Left := MulDiv(8, DialogUnits.X, 4);
        Top := MulDiv(8, DialogUnits.Y, 8);
        Constraints.MaxWidth := MulDiv(164, DialogUnits.X, 4);
        WordWrap := True;
      end;

      TextMemo := TMemo.Create(Form);
      with TextMemo do
      begin
        Parent := Form;
        Left   := Prompt.Left;
        Top    := Prompt.Top + Prompt.Height + 5;
        Width  := AControl.Width+30;
        Height := min(AControl.Height+60, 100);
        WordWrap := true;
        WantReturns := true;
        ScrollBars := ssVertical;
        Text   := ADefault;
        if (Text = 'Logic description') or (Text = 'Edit description...')
          then SelectAll;
      end;
      ClientWidth := max(ClientWidth, TextMemo.Left * 2 + TextMemo.Width);

      ButtonTop := TextMemo.Top + TextMemo.Height + 15;
      ButtonWidth := MulDiv(50, DialogUnits.X, 4);
      ButtonHeight := MulDiv(14, DialogUnits.Y, 8);
      with TButton.Create(Form) do
      begin
        Parent := Form;
        Caption := SMsgDlgOK;
        ModalResult := mrOk;
        Default := True;
        SetBounds(MulDiv(38, DialogUnits.X, 4), ButtonTop, ButtonWidth,
          ButtonHeight);
      end;
      with TButton.Create(Form) do
      begin
        Parent := Form;
        Caption := SMsgDlgCancel;
        ModalResult := mrCancel;
        Cancel := True;
        SetBounds(MulDiv(92, DialogUnits.X, 4), ButtonTop, ButtonWidth,
          ButtonHeight);
        Form.ClientHeight := Top + Height + 13;
      end;
      if ShowModal = mrOk then
      begin
        if TextMemo.Text = '' then EditTextDescription := 'Edit description...'
        else EditTextDescription := TextMemo.Text;
      end;
    finally
      Form.Free;
    end;
end;

initialization
RegisterClasses([TLogicTitle,TLogicDescription]);
finalization

end.
