{**********************************************}
{  TeeTree --> QuickReports 2.0 & 3.0          }
{  Copyright (c) 1996-98 by David Berneda      }
{  All Rights Reserved                         }
{**********************************************}
{$I teedefs.inc}
unit TreeQR;

interface

uses
  WinTypes, WinProcs, Messages, SysUtils, Classes, Graphics, Controls,
  Forms, Dialogs, TeeProcs, TeeTree, QuickRpt, TeCanvas, StdCtrls, Menus,
  ExtCtrls;

type
  TTeeQuickMethod=(qtmBitmap,qtmMetafile);

  TQuickReportTree=class(TCustomTree)
  public
    constructor Create(AOwner : TComponent); override;
    Function GetRectangle:TRect; override;
    procedure Invalidate; override;
    procedure Paint; override;
  published
    property AllowPanning;
    property AllowZoom;
    property AnimatedZoom;
    property AnimatedZoomSteps;
    property BackImage;
    property BackImageMode;
    property BufferedDisplay;
    property CrossBox;
    property Designing;
    property Gradient;
    property GridColor;
    property GridStep;
    property PrintProportional;
    property Selected;
    property SnapToGrid;
    property View3DOptions;
    property HorzScrollBar;
    property VertScrollBar;

    property OnAfterDraw;
    property OnNewConnection;
    property OnNewShape;
    property OnSelectShape;
    property OnUnSelectShape;

    { TPanel properties }
    property Align;
    property BevelInner;
    property BevelOuter default bvNone;
    property BevelWidth;
    property BorderWidth;
    property BorderStyle default bsSingle;
    property Color default clWhite;
    property DragCursor;
    property DragMode;
    property Enabled;
    property ParentColor;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint default True;
    property TabOrder;
    property TabStop;
    property Visible;

    { TPanel events }
    property OnResize;
  end;

  TQRTree=class;

  TPrintQRTreeEvent = procedure( Sender: TQRTree;
	   			 Var PaperRect,TreeRect:TRect ) of object;

  TQRTree = class(TQRPrintable)
  private
    { Private declarations }
    FOnPrint        : TPrintQRTreeEvent;
    FTeePrintMethod : TTeeQuickMethod;

    Function GetTree:TQuickReportTree;
    Procedure SetPrintMethod(Value:TTeeQuickMethod);
  protected
    { Protected declarations }
    procedure ReadState(Reader: TReader); override;
  public
    { Public declarations }
    Constructor Create(AOwner : TComponent); override;
    procedure Print(OfsX, OfsY : integer); override;
    procedure Paint; override;
  published
    { Published declarations }
    property Tree : TQuickReportTree read GetTree;
    property TeePrintMethod:TTeeQuickMethod read FTeePrintMethod
                                            write SetPrintMethod default qtmMetafile;
    { Published QR events }
    property OnPrint:TPrintQRTreeEvent read FOnPrint write FOnPrint;
  end;

implementation

Uses DsgnIntf,TeeConst;

Const TeeMsg_DefaultQRTree='QuickReportTree'; { <-- dont translate }

{ TQuickReportTree }
Constructor TQuickReportTree.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  Color:=clWhite;
  BevelOuter:=bvNone;
  BorderStyle:=bsNone;
  BufferedDisplay:=False;
  SetBounds(-1,-1,1,1);
  Hide;
end;

Function TQuickReportTree.GetRectangle:TRect;

  Function InternalGetRectangle(AControl:TControl; Const AZoom:Double):TRect;
  begin
    result:=Rect(0,0, Round(AControl.ClientWidth*AZoom),
                      Round(AControl.ClientHeight*AZoom) );
  end;

Var tmpZoomFactor : Double;
begin
  if Assigned(Parent) then
  begin
    if Parent is TQRTree then
    With TQRTree(Parent) do
    begin
      if Assigned(ParentReport) then tmpZoomFactor:=100.0/ParentReport.Zoom
                                else tmpZoomFactor:=1;
      result:=InternalGetRectangle(Self.Parent,tmpZoomFactor);
    end
    else result:=InternalGetRectangle(Parent,1);
  end
  else result:=InternalGetRectangle(Self,1);
end;

procedure TQuickReportTree.Paint;
var Meta   : TMetaFile;
    Bitmap : TBitmap;
begin
  if Assigned(Parent) and (Parent is TQRTree) then
  With TQRTree(Parent) do
  Case FTeePrintMethod of
  qtmMetafile: begin
                 Meta:=TeeCreateMetafile(True,GetRectangle);
                 try
                   Canvas.StretchDraw(GetClientRect,Meta);
                 finally
                   Meta.Free;
                 end;
               end;
  qtmBitmap:   begin
                 Bitmap:=TeeCreateBitmap(clWhite,GetRectangle);
                 try
                   Canvas.StretchDraw(GetClientRect,Bitmap);
                 finally
                   Bitmap.Free;
                 end;
               end;
  end
  else inherited Paint;
end;

procedure TQuickReportTree.Invalidate;
begin
  if Assigned(Parent) then
     if Parent is TQRTree then Parent.Invalidate
                          else inherited Invalidate;
end;

{ TQRTree }
Constructor TQRTree.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  FTeePrintMethod:=qtmMetafile;
  Width :=350;
  Height:=200;
  if (csDesigning in ComponentState) and
     (not (csLoading in Owner.ComponentState)) then
  With TQuickReportTree.Create(AOwner) do
  begin
    Parent:=TWinControl(Self);
    Name:=TeeGetUniqueName(AOwner,TeeMsg_DefaultQRTree);
  end;
end;

procedure TQRTree.ReadState(Reader: TReader);
Var tmpTree:TQuickReportTree;
begin
  tmpTree:=Tree;
  if Assigned(tmpTree) and
     (not (csLoading in tmpTree.ComponentState))
     and (not (csAncestor in tmpTree.ComponentState))
        then
          tmpTree.Free;
  inherited ReadState(Reader);
end;

Procedure TQRTree.SetPrintMethod(Value:TTeeQuickMethod);
begin
  if Value<>FTeePrintMethod then
  begin
    FTeePrintMethod:=Value;
    Repaint;
  end;
end;

procedure TQRTree.Print(OfsX, OfsY : Integer);
Var tmpMeta   : TMetafile;
    QuickRect : TRect;
    tmpRect   : TRect;
    tmpTree   : TQuickReportTree;
    tmpBitmap : TBitmap;
begin
  tmpTree:=Tree;
  if Assigned(tmpTree) then
  begin
    With ParentReport.QRPrinter do
    begin
      QuickRect:=Rect( Xpos(OfsX+Size.Left),
                       Ypos(OfsY+Size.Top),
                       Xpos(OfsX+Size.Left+Size.Width),
                       Ypos(OfsY+Size.Top+Size.Height));

      tmpRect:=tmpTree.GetRectangle;

      if Assigned( FOnPrint ) then FOnPrint( Self, QuickRect, tmpRect );

      Case FTeePrintMethod of
      qtmMetafile: begin
                     tmpMeta:=tmpTree.TeeCreateMetafile(True, tmpRect);
                     try
                       Canvas.StretchDraw(QuickRect,tmpMeta);
                     finally
                       tmpMeta.Free;
                     end;
                   end;
        qtmBitmap: begin
                     tmpBitmap:=tmpTree.TeeCreateBitmap(clWhite, tmpRect);
                     try
                       Canvas.StretchDraw(QuickRect,tmpBitmap);
                     finally
                       tmpBitmap.Free;
                     end;
                   end;
      end;
    end;
  end;
  inherited Print(OfsX,OfsY);
end;

procedure TQRTree.Paint;
begin
  if Assigned(Tree) then Tree.Paint;
  inherited Paint;
end;

Function TQRTree.GetTree:TQuickReportTree;
begin
  if ControlCount>0 then result:=TQuickReportTree(Controls[0])
                    else result:=nil;
end;

initialization
  RegisterClass( TQuickReportTree );
end.
