{**********************************************}
{   TDBTree Component Example                  }
{   Copyright (c) 1998 by David Berneda        }
{**********************************************}
unit UDBTree;

interface

{ This example shows the different features of DBTree component.

  A DBTree is automatically connected to a Table.

  See the TeeDBTre.pas unit for a complete explanation of DBTree
  modes of operation.

  3 situations are supported and showed in this example:

    Situation 1:  Just one field of one Table.

    Situation 2:  With one table, one field groups another field.

    Situation 3:  With a Master-Detail relation, master groups details.

  The DBTree also supports parent-child relations in within the
  same Table or Query, using the "CodeField" property.
}
uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  TeeTree, ExtCtrls, TeeProcs, TeeDBTre, StdCtrls, Db, DBTables, Grids,
  DBGrids, DBCtrls;

type
  TFormDBTreeDemo = class(TForm)
    Table1: TTable;
    Table2: TTable;
    Table3: TTable;
    DataSource1: TDataSource;
    MultiLine: TCheckBox;
    DBGrid1: TDBGrid;
    DataSource2: TDataSource;
    DBGrid2: TDBGrid;
    DataSource3: TDataSource;
    DBTree1: TDBTree;
    Button4: TButton;
    Sorted: TCheckBox;
    Synchronize: TCheckBox;
    DBImage1: TDBImage;
    Situation: TRadioGroup;
    Button1: TButton;
    CheckBox1: TCheckBox;
    procedure FormCreate(Sender: TObject);
    procedure MultiLineClick(Sender: TObject);
    procedure Button4Click(Sender: TObject);
    procedure DBTree1NewShape(Sender: TCustomTreePanel;
      NewShape: TTreeNodeShape);
    procedure SortedClick(Sender: TObject);
    procedure DBTree1SelectShape(Sender: TTreeNodeShape);
    procedure DataSource1DataChange(Sender: TObject; Field: TField);
    procedure SynchronizeClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure SituationClick(Sender: TObject);
    procedure Button1Click(Sender: TObject);
    procedure CheckBox1Click(Sender: TObject);
  private
    { Private declarations }
    Selecting:Boolean;
    Procedure RefreshTree;
  public
    { Public declarations }
    ShowWizard:Boolean;

    procedure JustOneField;  { situation 1 }
    procedure GroupByField;  { situation 2 }
    procedure MasterDetail;  { situation 3 }
  end;

implementation

{$R *.DFM}
Uses TreeEd,TreeDBEd; { <-- the Tree and DBTree Editor dialog }

{ Initialize some Tree properties... }
procedure TFormDBTreeDemo.FormCreate(Sender: TObject);
begin
  ShowWizard:=False; { internal }

  With DBTree1 do
  begin
    AllowPanning:=pmVertical;      { allow scrolling vertical only }
    VertScrollBar.Automatic:=True; { automatic (appears when needed)}
  end;
  Selecting:=False;
end;

{ Situation 1:  Just one field of one Table }
procedure TFormDBTreeDemo.JustOneField;
begin
  With DBTree1 do
  begin
    DataSet     := Table1 ;   { Animals.dbf }
    CodeField   := '' ;
    ParentField := '' ;
    TextFields  := 'Name' ;   { show Animal names }
    Detail      := nil ;
    DetailFields:= '' ;
  end;
  RefreshTree;

  { Show Animals in grid... }
  DBGrid1.DataSource:=DataSource1;
  DBGrid2.Visible:=False;
  DBImage1.Visible:=True;
end;

{ Situation 2: Group one Table by one Field and show another Field. }
procedure TFormDBTreeDemo.GroupByField;
begin
  With DBTree1 do
  begin
    DataSet     := Table1 ;  { Animals.dbf }
    CodeField   := '' ;
    ParentField := 'Area' ;  { Grouped by "Area" field }
    TextFields  := 'Name' ;  { Show Animals names }
    Detail      := nil ;
    DetailFields:= '' ;
  end;
  RefreshTree;

  { Show Animals in grid... }
  DBGrid1.DataSource:=DataSource1;
  DBGrid2.Visible:=False;
  DBImage1.Visible:=True;
end;

{ Situation 3: Master-Detail relationship.
  "The Master table groups Detail fields".
}
procedure TFormDBTreeDemo.MasterDetail;
begin
  With DBTree1 do
  begin
    DataSet     := Table2 ;   { Customer.db }
    CodeField   := '';
    ParentField := '' ;
    TextFields  := 'Company'; { Show Customers Company }
    Detail      := Table3 ;   { Link customers.db with orders.db }

    { Show all orders of each customer with this data: }
    DetailFields:='SaleDate;EmpNo;AmountPaid' ;
  end;
  RefreshTree;

  { Show Customers and Orders in grids... }
  DBGrid1.DataSource:=DataSource2;
  DBGrid2.Visible:=True;
  DBImage1.Visible:=False;
end;

{ Show node's text with multiple-lines or single line of text }
procedure TFormDBTreeDemo.MultiLineClick(Sender: TObject);
begin
  DBTree1.MultiLineText:=MultiLine.Checked;
  if DBTree1.Shapes.Count>0 then RefreshTree;
end;

{ Call "DBTree1.Refresh" to retrieve the appropiate database records
  to fill the Tree.

  Re-order nodes by text if desired.
}
Procedure TFormDBTreeDemo.RefreshTree;
Var OldSource:TDataSource;
begin
  { "Refresh" clears and re-fills again the DBTree with
    database records. }
  Screen.Cursor:=crHourGlass;
  OldSource:=DBGrid1.DataSource;
  DBGrid1.DataSource:=nil;
  DBTree1.Refresh;
  DBGrid1.DataSource:=OldSource;
  Screen.Cursor:=crDefault;

  { Move Tree to top }
  DBTree1.View3DOptions.VertOffset:=0;

  { Select the first child or the root if there are no childs }
  With DBTree1.Roots[0] do
  begin
    Expanded:=True;
    if Childs.Count>0 then Childs[0].Selected:=True
                      else Selected:=True;
  end;

  { Re-order nodes... }
  if Sorted.Checked then DBTree1.Sort(True,True);
end;

procedure TFormDBTreeDemo.Button4Click(Sender: TObject);
begin
  Close;
end;

{ Hide all node shapes border and, if the new shape is a
  child of a child, remove the image picture.
}
procedure TFormDBTreeDemo.DBTree1NewShape(Sender: TCustomTreePanel;
  NewShape: TTreeNodeShape);
begin
  { No border and no background for all shapes... }
  NewShape.Border.Visible:=False;
  NewShape.Transparent:=True;

  { How to remove the picture for sub-sub-nodes...? }
  if Assigned(NewShape.Parent) and Assigned(NewShape.Parent.Parent) then
     NewShape.ImageIndex:=tiNone
  else
  Case Random(3) of
    0: NewShape.Font.Color:=clRed;
    1: NewShape.Font.Color:=clTeal;
    2: NewShape.Font.Color:=clBlue;
  end;
end;

{ Re-order nodes by text Alphabetical, Ascending order, IgnoreCase }
procedure TFormDBTreeDemo.SortedClick(Sender: TObject);
begin
  if Sorted.Checked then DBTree1.Sort(True,True);
end;

{ When clicking a Tree node shape, search the record in the DataSet
  that corresponds to that shape, and move the Table or Query current
  record to it.
}
procedure TFormDBTreeDemo.DBTree1SelectShape(Sender: TTreeNodeShape);
var tmpField:String;
begin
  if Synchronize.Checked then
  if not Selecting then
  if Assigned(Sender.Parent) and
     (not Assigned(Sender.Parent.Parent)) then  {  not for sub-sub-nodes... }
  begin
    With DBTree1 do
    if ParentField<>'' then tmpField:=ParentField
                       else tmpField:=TeeExtractField(TextFields,1);

    { Search which record is... }
    With DBTree1.DataSet do
    begin
      DisableControls;
      First;
      While not eof do
      begin
        if FieldByName(tmpField).AsString=Sender.Text[0] then
        begin
          EnableControls;
          Exit;
        end;
        Next;
      end;
      EnableControls;
    end;
  end;
end;

{ When the DataSource current record changes position (because
  clicking the grid for example), we should re-position the current
  Tree node as well.
}
procedure TFormDBTreeDemo.DataSource1DataChange(Sender: TObject;
  Field: TField);
var t   : Integer;
    tmp : String;
    tmpSt:String;
begin
  if Synchronize.Checked then
  With DBTree1 do
  if Roots.Count>0 then { if tree is not empty... }
  begin
    if ParentField<>'' then tmpSt:=ParentField
                       else tmpSt:=TextFields;
    tmp:=DataSet.FieldByName(TeeExtractField(tmpSt,1)).AsString;

    { Search which node shape in the tree correspond to the current
      DataSet record... }
    for t:=0 to Roots[0].Childs.Count-1 do
    if tmp=Roots[0].Childs[t].Text[0] then
    begin
      { If found, make this node to be the Selected one... }
      Selected.Clear;
      Selecting:=True;
      Roots[0].Childs[t].Selected:=True;
      Selecting:=False;
      { quit... }
      Exit;
    end;
  end;
end;

procedure TFormDBTreeDemo.SynchronizeClick(Sender: TObject);
begin
  DataSource1DataChange(Self,nil);
end;

procedure TFormDBTreeDemo.FormShow(Sender: TObject);
begin
  DBTree1.MultiLineText:=MultiLine.Checked;
  Situation.ItemIndex:=0;
  if ShowWizard then ShowDBTreeEditor(Self,DBTree1);
end;

procedure TFormDBTreeDemo.SituationClick(Sender: TObject);
begin
  Case Situation.ItemIndex+1 of
    1: JustOneField;  { situation 1 }
    2: GroupByField;  { situation 2 }
    3: MasterDetail;  { situation 3 }
  end;
end;

procedure TFormDBTreeDemo.Button1Click(Sender: TObject);
begin
  EditDBTree(Self,DBTree1); { Show the DBTree editor... }
end;

procedure TFormDBTreeDemo.CheckBox1Click(Sender: TObject);
begin
  { these settings can speed when adding many childs... }
  DBTree1.NoOwnerShapes:=CheckBox1.Checked;
  DBTree1.CreateConnections:=not CheckBox1.Checked;
end;

end.
