unit PBShareStringList;

{$INCLUDE PBDefines.inc}

interface

uses
	Windows, Messages, SysUtils, Classes, PBShare, PBShareManager;

type
	TPBShareStringList = class(TPBCustomShare)
	private
		{ Private declarations }
		FExternalList, FAutoUpdate : Boolean;
		FStrings, InternStrings : TStrings;
		function GetStrings : TStrings;
		function GetValues(const Name : string) : string;
		procedure DoWriteList(Sender : TObject; const Stream : TMemoryStream;
			const Writer : TPBWriter; var ExtraInfo : integer);
		procedure DoReadList(Sender : TObject; const Stream : TMemoryStream;
			const Reader : TPBReader; const FromHandle : HWnd;
			const ExtraInfo : integer);
		procedure SetExternalList(Value : Boolean);
		procedure SetStrings(Value : TStrings);
		procedure SetValues(const Name, Value : string);
	protected
		{ Protected declarations }
	public
		{ Public declarations }
		constructor Create(AOwner : TComponent); override;
		constructor CreateNoOwner(AShareName : string; AAllocBy : integer;
			AFlags : TShareFlags; AAutoUpdate, AOpenNow : Boolean;
			AShareFunction : TShareFunction; AOnClosing : TClosingShareEvent;
			AOnOpen : TOpenShareEvent; AOnUpdateAll : TUpdateAllEvent;
			AOnRemoteControl : TRemoteControlEvent); reintroduce;
		constructor CreateNoEvents(AShareName : string; AAllocBy : integer;
			AFlags : TShareFlags; AAutoUpdate, AOpenNow : Boolean;
			AShareFunction : TShareFunction; AOnClosing : TClosingShareProc;
			AOnOpen : TOpenShareProc; AOnUpdateAll : TUpdateAllProc;
			AOnRemoteControl : TRemoteControlProc); reintroduce;
		destructor Destroy; override;
		function OpenShare : Boolean; override;
		procedure Clear; override;
		procedure ReadAll; override;
		procedure StringListChange(Sender : TObject);
		procedure WriteAll; override;
		property ExtraInfo;
		property Values[const Name : string] : string read GetValues
			write SetValues; default;
	published
		{ Published declarations }
		property AutoUpdate : Boolean read FAutoUpdate
			write FAutoUpdate default True;
		property ExternalList : Boolean read FExternalList
			write SetExternalList default False;
		property Strings : TStrings read GetStrings write SetStrings;
	end;

implementation

//  ---------------  TPBShareStringList  ------------------------
constructor TPBShareStringList.Create(AOwner : TComponent);
begin
	inherited;
	FAutoUpdate := True;
	FExternalList := False;
	FStrings := TStringList.Create;
	InternStrings := FStrings;
	TStringList(FStrings).OnChange := StringListChange;
	OnDoWriteAll := DoWriteList;
	OnDoReadAll := DoReadList;
end;

constructor TPBShareStringList.CreateNoOwner(AShareName : string;
	AAllocBy : integer; AFlags : TShareFlags; AAutoUpdate, AOpenNow : Boolean;
	AShareFunction : TShareFunction; AOnClosing : TClosingShareEvent;
	AOnOpen : TOpenShareEvent; AOnUpdateAll : TUpdateAllEvent;
	AOnRemoteControl : TRemoteControlEvent);
begin
	Create(nil);
	FAutoUpdate := AAutoUpdate;
	Flags := AFlags;
	ShareFunction := AShareFunction;
	ShareName := AShareName;
	OnClosing := AOnClosing;
	OnOpen := AOnOpen;
	OnUpdateAll := AOnUpdateAll;
	OnRemoteControl := AOnRemoteControl;
	if AOpenNow then LazyOpenShare;
end;

constructor TPBShareStringList.CreateNoEvents(AShareName : string;
	AAllocBy : integer; AFlags : TShareFlags; AAutoUpdate, AOpenNow : Boolean;
	AShareFunction : TShareFunction; AOnClosing : TClosingShareProc;
	AOnOpen : TOpenShareProc;	AOnUpdateAll : TUpdateAllProc;
	AOnRemoteControl : TRemoteControlProc);
begin
	Create(nil);
	FAutoUpdate := AAutoUpdate;
	Flags := AFlags;
	ShareFunction := AShareFunction;
	ShareName := AShareName;
	OnClosing := TClosingShareEvent(MakeMethod(@AOnClosing));
	OnOpen := TOpenShareEvent(MakeMethod(@AOnOpen));
	OnUpdateAll := TUpdateAllEvent(MakeMethod(@AOnUpdateAll));
	OnRemoteControl := TRemoteControlEvent(MakeMethod(@AOnRemoteControl));
	if AOpenNow then LazyOpenShare;
end;

destructor TPBShareStringList.Destroy;
begin
	if IsOpen then CloseShare;
	if Assigned(InternStrings) and (InternStrings = FStrings) then
	begin
		FStrings.Free;
		FStrings := nil;
		InternStrings := nil;
	end;
	inherited;
end;

function TPBShareStringList.GetStrings : TStrings;
begin
	Result := FStrings;
end;

function TPBShareStringList.GetValues(const Name : string) : string;
begin
	Result := FStrings.Values[Name];
end;

function TPBShareStringList.OpenShare : Boolean;
begin
	if FStrings = nil then Raise Exception.Create
		('PBShareStringList: Strings property (external) not assigned !')
	else Result := inherited OpenShare;
end;

procedure TPBShareStringList.Clear;
var
	TempAuto : Boolean;
begin
	TempAuto := FAutoUpdate;
	FAutoUpdate := False;
	FStrings.Clear;
	WriteAll;
	FAutoUpdate := TempAuto;
end;

procedure TPBShareStringList.SetExternalList(Value : Boolean);
begin
	if FExternalList <> Value then
	begin
		FExternalList := Value;
		if FExternalList and (FStrings = InternStrings) then
		begin
			FStrings.Free;
			FStrings := nil;
			InternStrings := nil;
		end
		else if InternStrings = nil then
		begin
			FStrings := TStringList.Create;
			InternStrings := FStrings;
			TStringList(FStrings).OnChange := StringListChange;
		end;
	end;
end;

procedure TPBShareStringList.SetStrings(Value : TStrings);
begin
	if FExternalList then	FStrings := Value
	else if Assigned(FStrings) and (Value.Text <> FStrings.Text)
		then FStrings.Assign(Value);
end;

procedure TPBShareStringList.SetValues(const Name, Value : string);
begin
	FStrings.Values[Name] := Value;
end;

procedure TPBShareStringList.DoWriteList(Sender : TObject;
	const Stream : TMemoryStream; const Writer : TPBWriter;
	var ExtraInfo : integer);
begin
	FStrings.SaveToStream(Stream);
end;

procedure TPBShareStringList.DoReadList(Sender : TObject;
	const Stream : TMemoryStream; const Reader : TPBReader;
	const FromHandle : HWnd; const ExtraInfo : integer);
begin
	FStrings.LoadFromStream(Stream);
end;

procedure TPBShareStringList.ReadAll;
begin
	inherited;
end;

procedure TPBShareStringList.WriteAll;
begin
	inherited;
end;

procedure TPBShareStringList.StringListChange(Sender : TObject);
begin
	if (FAutoUpdate or FExternalList) and (not (msReadingData in FPStatus^))
		then WriteAll;
end;

end.
