unit PBShareStream;

interface

uses
	Windows, SysUtils, Messages, Classes;

type
	PHandle = ^THandle;
	PPInteger = ^PInteger;
	PBoolean = ^Boolean;

	TShareLock = (slNone, slAnotherRead, slRead, slAnotherWrite, slWrite);

	EShareException = class(Exception);

//  --------------------  TPBReader  -------------------------
	TPBReader = class(TPersistent)
	private
		FStream : TStream;
	public
		constructor Create(AStream : TStream);
		function ReadBoolean : Boolean;
		function ReadByte : Byte;
		function ReadCardinal : Cardinal;
		function ReadChar : Char;
		function ReadCurrency : Currency;
		function ReadDateTime : TDateTime;
		function ReadDouble : Double;
		function ReadExtended : Extended;
		function ReadInt64 : Int64;
		function ReadInteger : integer;
		function ReadLongInt : Longint;
		function ReadLongWord : LongWord;
		function ReadShortInt : Shortint;
		function ReadSingle : Single;
		function ReadSmallInt : Smallint;
		function ReadString : string;
		function ReadWideChar : WideChar;
		function ReadWideString : WideString;
		function ReadWord : Word;
		procedure Read(var Buffer; const Count : integer);
	end;

//  --------------------  TPBWriter  -------------------------
	TPBWriter = class(TPersistent)
	private
		FStream : TStream;
	public
		constructor Create(AStream : TStream);
		procedure Write(const Buffer; const Count : integer);
		procedure WriteBoolean(const Value : Boolean);
		procedure WriteByte(const Value : Byte);
		procedure WriteCardinal(const Value : Cardinal);
		procedure WriteChar(const Value : Char);
		procedure WriteCurrency(const Value : Currency);
		procedure WriteDateTime(const Value : TDateTime);
		procedure WriteDouble(const Value : Double);
		procedure WriteExtended(const Value : Extended);
		procedure WriteInt64(const Value : Int64);
		procedure WriteInteger(const Value : integer);
		procedure WriteLongInt(const Value : LongInt);
		procedure WriteLongWord(const Value : LongWord);
		procedure WriteShortInt(const Value : ShortInt);
		procedure WriteSingle(const Value : Single);
		procedure WriteSmallInt(const Value : SmallInt);
		procedure WriteString(const Value : string);
		procedure WriteWideChar(const Value : WideChar);
		procedure WriteWideString(const Value : WideString);
		procedure WriteWord(const Value : Word);
	end;

//  -----------------  TPBShareStream  -----------------------
	TPBShareStream = class(TStream)
	private
		FBasePointer, FDataPointer, FSizePointer : PInteger;
		FPosition : integer;
		PWriteMutex, PReadEvent : PHandle;
		PWriteLockCount, PReadLockCount : PInteger;
		PFileMapPointer : PPInteger;
		PSynchronizeRead : PBoolean;
		function GetCapacity : integer;
		function GetSize : integer;
		procedure SetCapacity(const Value : integer);
		procedure SetPosition(const Value : integer);
		procedure SetSize(const Value : integer); reintroduce;
	protected
		{ Protected declarations }
		procedure LockShare(const Lock : TShareLock);
		procedure UnLockShare(const Lock : TShareLock);
	public
		{ Public declarations }
		constructor Create(MemPtr : PInteger; ACapacity : integer;
			APWriteMutex, APReadEvent : PHandle; APFileMapPointer : PPInteger;
			APWriteLockCount, APReadLockCount : PInteger;
			APSynchronizeRead : PBoolean); virtual;
		function GetItemPointer(const StreamPosition : integer) : PInteger;
		function GetPosition(const ItemPointer : PInteger) : integer;
		function Move(const FromPosition, ToPosition : integer;
			Count : integer) : integer;
		function Read(var Buffer; Count: integer) : integer; override;
		function Seek(Offset: integer; Origin: Word) : integer; override;
		function Write(const Buffer; Count: integer) : integer; override;
		procedure Clear;
		procedure SetVariables(MemPtr : PInteger; ACapacity : integer);
		property Capacity : integer read GetCapacity write SetCapacity;
		property Position : integer read FPosition write SetPosition;
		property Size : integer read GetSize write SetSize;
	end;

function LockStatus(const PWriteMutex, PReadEvent : PHandle;
	const PWriteLockCount, PReadLockCount : PInteger;
	const PMultiReadCounter : PPInteger) : TShareLock;
	stdcall; external 'Unregistrered_PBSharePack.dll';
procedure LockShare(const Lock : TShareLock; const PSynchronizeRead : PBoolean;
	const PWriteMutex, PReadEvent : PHandle;
	const PWriteLockCount, PReadLockCount : PInteger;
	const PMultiReadCounter : PPInteger);
	stdcall; external 'Unregistrered_PBSharePack.dll';
procedure UnLockShare(const Lock : TShareLock; const PSynchronizeRead : PBoolean;
	const PWriteMutex, PReadEvent : PHandle;
	const PWriteLockCount, PReadLockCount : PInteger;
	const PMultiReadCounter : PPInteger);
	stdcall; external 'Unregistrered_PBSharePack.dll';

//  -----------------  implementation  -----------------------------
implementation

//  --------------------  TPBReader  --------------------------
constructor TPBReader.Create(AStream : TStream);
begin
	inherited Create;
	FStream := AStream;
end;

function TPBReader.ReadBoolean : Boolean;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadByte : Byte;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadCardinal : Cardinal;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadChar : Char;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadCurrency : Currency;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadDateTime : TDateTime;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadDouble : Double;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadExtended : Extended;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadInt64 : Int64;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadInteger : integer;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadLongInt : LongInt;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadLongWord : LongWord;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadShortInt : ShortInt;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadSingle : Single;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadSmallInt : SmallInt;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadString : string;
var
	Size : integer;
begin
	Read(Size, SizeOf(Integer));
	SetLength(Result, Size);
	Read(PChar(Result)^, Size);
end;

function TPBReader.ReadWideChar : WideChar;
begin
	Read(Result, SizeOf(Result));
end;

function TPBReader.ReadWideString : WideString;
var
	Size : integer;
begin
	Read(Size, SizeOf(Integer));
	SetLength(Result, Size div SizeOf(WideChar));
	Read(PWideChar(Result)^, Size);
end;

function TPBReader.ReadWord : Word;
begin
	Read(Result, SizeOf(Result));
end;

procedure TPBReader.Read(var Buffer; const Count : integer);
begin
	FStream.ReadBuffer(Buffer, Count);
end;

//  --------------------  TPBWriter  --------------------------
constructor TPBWriter.Create(AStream : TStream);
begin
	inherited Create;
	FStream := AStream;
end;

procedure TPBWriter.Write(const Buffer; const Count : integer);
begin
	FStream.WriteBuffer(Buffer, Count);
end;

procedure TPBWriter.WriteBoolean(const Value : Boolean);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteByte(const Value : Byte);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteCardinal(const Value : Cardinal);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteChar(const Value : Char);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteCurrency(const Value : Currency);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteDateTime(const Value : TDateTime);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteDouble(const Value : Double);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteExtended(const Value : Extended);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteInt64(const Value : Int64);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteInteger(const Value : integer);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteLongInt(const Value : LongInt);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteLongWord(const Value : LongWord);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteShortInt(const Value : ShortInt);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteSingle(const Value : Single);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteSmallInt(const Value : SmallInt);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteString(const Value : string);
var
	Size : integer;
begin
	Size := Length(Value);
	Write(Size, SizeOf(Integer));
	Write(PChar(Value)^, Size);
end;

procedure TPBWriter.WriteWideChar(const Value : WideChar);
begin
	Write(Value, SizeOf(Value));
end;

procedure TPBWriter.WriteWideString(const Value : WideString);
var
	Size : integer;
begin
	Size := Length(Value) * SizeOf(WideChar);
	Write(Size, SizeOf(Integer));
	Write(PWideChar(Value)^, Size);
end;

procedure TPBWriter.WriteWord(const Value : Word);
begin
	Write(Value, SizeOf(Value));
end;

//  --------------------  TPBShareStream  -------------------------
constructor TPBShareStream.Create(MemPtr : PInteger; ACapacity : integer;
	APWriteMutex, APReadEvent : PHandle; APFileMapPointer : PPInteger;
	APWriteLockCount, APReadLockCount : PInteger; APSynchronizeRead : PBoolean);
begin
	inherited Create;
	PWriteMutex := APWriteMutex;
	PReadEvent := APReadEvent;
	PFileMapPointer := APFileMapPointer;
	PWriteLockCount := APWriteLockCount;
	PReadLockCount := APReadLockCount;
	PSynchronizeRead := APSynchronizeRead;
	SetVariables(MemPtr, ACapacity);
end;

function TPBShareStream.GetCapacity : integer;
begin
	Result := FBasePointer^;
end;

function TPBShareStream.GetSize : integer;
begin
	Result := FSizePointer^;
end;

function TPBShareStream.GetItemPointer(const StreamPosition : integer) : PInteger;
begin
	Result := Ptr(Integer(FDataPointer) + StreamPosition);
end;

function TPBShareStream.GetPosition(const ItemPointer : PInteger) : integer;
begin
	Result := Integer(ItemPointer) - Integer(FDataPointer);
end;

function TPBShareStream.Move(const FromPosition, ToPosition : integer;
	Count : integer) : integer;
var
	FromPointer, ToPointer : PInteger;
begin
	LockShare(slWrite);
	if (FromPosition + Count > Size) then Count := Size - FromPosition
	else if (ToPosition + Count > Capacity) then Count := 0;
	FromPointer := GetItemPointer(FromPosition);
	ToPointer := GetItemPointer(ToPosition);
	MoveMemory(ToPointer, FromPointer, Count);
	UnLockShare(slWrite);
	Result := Count;
end;

function TPBShareStream.Read(var Buffer; Count: integer): integer;
var
	TempPointer : PInteger;
begin
	LockShare(slRead);
	if FPosition + Count > Size
		then Count := Size - FPosition;
	TempPointer := GetItemPointer(FPosition);
	CopyMemory(@Buffer, TempPointer, Count);
	UnLockShare(slRead);
	Seek(Count, soFromCurrent);
	Result := Count;
end;

function TPBShareStream.Seek(Offset: integer; Origin: Word) : integer;
begin
	case Origin of
		soFromBeginning: FPosition := Offset;
		soFromCurrent: FPosition := FPosition + Offset;
		soFromEnd: FPosition := Size + Offset;
	end;
	Result := FPosition;
end;

function TPBShareStream.Write(const Buffer; Count: integer): integer;
var
	TempPointer : PInteger;
begin
	LockShare(slWrite);
	if FPosition + Count > Capacity
		then Count := Capacity - FPosition;
	TempPointer := GetItemPointer(FPosition);
	CopyMemory(TempPointer, @Buffer, Count);
	UnLockShare(slWrite);
	Seek(Count, soFromCurrent);
	Result := Count;
end;

procedure TPBShareStream.Clear;
begin
	FPosition := 0;
	LockShare(slWrite);
	ZeroMemory(FSizePointer, SizeOf(Integer) + Capacity);
	UnLockShare(slWrite);
end;

procedure TPBShareStream.LockShare(const Lock : TShareLock);
begin
	PBShareStream.LockShare(Lock, PSynchronizeRead, PWriteMutex, PReadEvent,
		PWriteLockCount, PReadLockCount, PFileMapPointer);
end;

procedure TPBShareStream.UnLockShare(const Lock : TShareLock);
begin
	PBShareStream.UnLockShare(Lock, PSynchronizeRead, PWriteMutex, PReadEvent,
		PWriteLockCount, PReadLockCount, PFileMapPointer);
end;

procedure TPBShareStream.SetPosition(const Value : integer);
begin
	if (FPosition <> Value) then
	begin
		if (Value < 0) then FPosition := 0
		else if (Value > Size) then FPosition := Size
		else FPosition := Value;
	end;
end;

procedure TPBShareStream.SetCapacity(const Value : integer);
begin
	LockShare(slWrite);
	FBasePointer^ := Value;
	UnLockShare(slWrite);
end;

procedure TPBShareStream.SetSize(const Value : integer);
begin
	LockShare(slWrite);
	FSizePointer^ := Value;
	UnLockShare(slWrite);
end;

procedure TPBShareStream.SetVariables(MemPtr : PInteger; ACapacity : integer);
begin
	FBasePointer := PInteger(MemPtr);
	FSizePointer := PInteger(Integer(FBasePointer) + SizeOf(integer));
	FDataPointer := PInteger(Integer(FSizePointer) + SizeOf(integer));
	if ACapacity > 0 then Capacity := ACapacity;
end;

end.
