unit PBShareMulti;

{$INCLUDE PBDefines.inc}

interface

uses
	Windows, Messages, SysUtils, Classes, PBShare, PBShareManager,
	PBShareStream;

type
	TPBCustomShareMulti = class(TPBCustomShare)
	private
		FOnUpdateItem : TUpdateItemEvent;
		FOnDoReadItem : TDoReadItemEvent;
		FOnDoWriteItem : TDoWriteItemEvent;
		function GetCount : integer;
		function GetShareManager : TPBMultiShareManager;
	protected
		procedure DataChange(Sender : TObject; const Index : integer;
			const Resized : Boolean; const ItemOperation : TMultiOperation); override;
		procedure ReadAll; override;
		procedure WriteAll; override;
		procedure WndProc(var FMessage : TMessage); override;
		procedure WriteItemSize(const NumberOfStrings,
			TotalItemSize : integer); virtual;
		property OnDoReadItem : TDoReadItemEvent read FOnDoReadItem
			write FOnDoReadItem;
		property OnDoWriteItem : TDoWriteItemEvent read FOnDoWriteItem
			write FOnDoWriteItem;
		property ShareManager : TPBMultiShareManager read GetShareManager;
	public
		constructor Create(AOwner : TComponent); override;
		constructor CreateNoOwner(AShareName : string; AAllocBy : integer;
			AFlags : TShareFlags; AShareFunction : TShareFunction;
			AOnClosing : TClosingShareEvent; AOnOpen : TOpenShareEvent;
			AOnUpdateAll : TUpdateAllEvent; AOnUpdateItem : TUpdateItemEvent;
			AOnRemoteControl : TRemoteControlEvent); reintroduce;
		constructor CreateNoEvents(AShareName : string; AAllocBy : integer;
			AFlags : TShareFlags; AShareFunction : TShareFunction;
			AOnClosing : TClosingShareProc; AOnOpen : TOpenShareProc;
			AOnUpdateAll : TUpdateAllProc; AOnUpdateItem : TUpdateItemProc;
			AOnRemoteControl : TRemoteControlProc); reintroduce;
		function Add : integer;
		procedure Delete(const Index : integer);
		procedure Insert(const Index : integer);
		procedure ReadItem(const Index : integer);
		procedure WriteItem(const Index : integer);
		property Count : integer read GetCount;
	published
		property OnUpdateItem : TUpdateItemEvent read FOnUpdateItem
			write FOnUpdateItem;
	end;

	TPBShareMulti = class(TPBCustomShareMulti)
	private
	public
		constructor CreateNoOwner(AShareName : string; AAllocBy : integer;
			AFlags : TShareFlags; AOpenNow : Boolean; AShareFunction : TShareFunction;
			AOnClosing : TClosingShareEvent; AOnOpen : TOpenShareEvent;
			AOnUpdateAll : TUpdateAllEvent; AOnDoReadItem : TDoReadItemEvent;
			AOnDoWriteItem : TDoWriteItemEvent; AOnUpdateItem : TUpdateItemEvent;
			AOnRemoteControl : TRemoteControlEvent); reintroduce;
		constructor CreateNoEvents(AShareName : string; AAllocBy : integer;
			AFlags : TShareFlags; AOpenNow : Boolean; AShareFunction : TShareFunction;
			AOnClosing : TClosingShareProc; AOnOpen : TOpenShareProc;
			AOnUpdateAll : TUpdateAllProc; AOnDoReadItem : TDoReadItemProc;
			AOnDoWriteItem : TDoWriteItemProc; AOnUpdateItem : TUpdateItemProc;
			AOnRemoteControl : TRemoteControlProc); reintroduce;
		procedure WriteItemSize(const NumberOfStrings,
			TotalItemSize : integer); override;
	published
		property OnDoReadItem;
		property OnDoWriteItem;
	end;

implementation

var
	FUpdateItemMessageID : Cardinal;

//  -------------------  TPBCustomShareMulti  ----------------------
constructor TPBCustomShareMulti.Create(AOwner : TComponent);
begin
	inherited;
	FShareManagerClass := TPBMultiShareManager;
end;

constructor TPBCustomShareMulti.CreateNoOwner(AShareName : string; AAllocBy : integer;
	AFlags : TShareFlags; AShareFunction : TShareFunction;
	AOnClosing : TClosingShareEvent; AOnOpen : TOpenShareEvent;
	AOnUpdateAll : TUpdateAllEvent;	AOnUpdateItem : TUpdateItemEvent;
	AOnRemoteControl : TRemoteControlEvent);
begin
	inherited CreateNoOwner(AShareName, AAllocBy, AFlags,
		AShareFunction, AOnClosing, AOnOpen, AOnUpdateAll, AOnRemoteControl);
	FShareManagerClass := TPBMultiShareManager;
	FOnUpdateItem := AOnUpdateItem;
end;

constructor TPBCustomShareMulti.CreateNoEvents(AShareName : string; AAllocBy : integer;
	AFlags : TShareFlags; AShareFunction : TShareFunction;
	AOnClosing : TClosingShareProc; AOnOpen : TOpenShareProc;
	AOnUpdateAll : TUpdateAllProc; AOnUpdateItem : TUpdateItemProc;
	AOnRemoteControl : TRemoteControlProc);
begin
	inherited CreateNoEvents(AShareName, AAllocBy, AFlags,
		AShareFunction, AOnClosing, AOnOpen, AOnUpdateAll, AOnRemoteControl);
	FShareManagerClass := TPBMultiShareManager;
	FOnUpdateItem := TUpdateItemEvent(MakeMethod(@AOnUpdateItem));
end;

function TPBCustomShareMulti.Add : integer;
begin
	Result := -1;
	if (ShareFunction = sfReader) then Exit;
	if ([msOpen, msOpening] * FPStatus^ <> [])
		then Result := ShareManager.Add(Self, FOnDoWriteItem)
	else Raise EShareException.Create('PBShare must be open for this operation !');
end;

function TPBCustomShareMulti.GetCount : integer;
begin
	if (ShareManager <> nil) then Result := ShareManager.Count
	else Result := 0;
end;

function TPBCustomShareMulti.GetShareManager : TPBMultiShareManager;
begin
	Result := TPBMultiShareManager(FShareManager);
end;

procedure TPBCustomShareMulti.DataChange(Sender : TObject; const Index : integer;
	const Resized : Boolean; const ItemOperation : TMultiOperation);
begin
	inherited;
	if (ShareCount = 1) or (UpdateCount > 0) then Exit;
	if ItemOperation <> moWriteAll then
	begin
		FExtraInfo := Index mod (MAXINT div 16) + Ord(ItemOperation) * (MAXINT div 16);
		NotifyAll(FUpdateItemMessageID, FExtraInfo, (sfAsynchronUpdate in Flags));
	end;
end;

procedure TPBCustomShareMulti.Delete(const Index : integer);
begin
	if (ShareFunction = sfReader) then Exit;
	if ([msOpen, msOpening] * FPStatus^ <> [])
		then ShareManager.Delete(Index)
	else Raise EShareException.Create('PBShare must be open for this operation !');
end;

procedure TPBCustomShareMulti.Insert(const Index : integer);
begin
	if (ShareFunction = sfReader) then Exit;
	if ([msOpen, msOpening] * FPStatus^ <> [])
		then ShareManager.Insert(Self, FOnDoWriteItem, Index)
	else Raise EShareException.Create('PBShare must be open for this operation !');
end;

procedure	TPBCustomShareMulti.WndProc(var FMessage : TMessage);
var
	Index : integer;
	ItemOperation : TMultiOperation;
begin
	with FMessage do if ([msClosing, msDestroying] * FPStatus^ = []) then
	begin
		if IsOpen and (Msg = FUpdateItemMessageID)
			and (ShareFunction <> sfWriter) then
		begin
			Include(FPStatus^, msUpdating);
			FFromHandle := HWnd(WParam);
			Index := LParam mod (MAXINT div 16);
			ItemOperation := TMultiOperation(LParam div (MAXINT div 16));
			if InSendMessage then PSynchronizeRead^ := True
			else PSynchronizeRead^ := False;
			LockShare(slRead);
			if ItemOperation = moWrite then ShareManager.UpdateList(Index)
			else FShareManager.MakeList;
			if Assigned(FOnUpdateItem) then FOnUpdateItem(Self, FFromHandle, Index,
				ItemOperation);
			PSynchronizeRead^ := False;
			UnLockShare(slRead);
			Exclude(FPStatus^, msUpdating);
		end;
	end;
	inherited;
end;

procedure TPBCustomShareMulti.ReadAll;
begin
	if ([msOpen, msOpening] * FPStatus^ <> []) and (ShareFunction <> sfWriter) then
	begin
		Include(FPStatus^, msReadingData);
		ShareManager.MakeList;
		Exclude(FPStatus^, msReadingData);
	end;
end;

procedure TPBCustomShareMulti.ReadItem(const Index : integer);
begin
	if ([msOpen, msOpening] * FPStatus^ <> []) and (ShareFunction <> sfWriter) then
	begin
		Include(FPStatus^, msReadingData);
		ShareManager.ReadItem(Self, FOnDoReadItem, Index);
		Exclude(FPStatus^, msReadingData);
	end;
end;

procedure TPBCustomShareMulti.WriteAll; begin {Do nothing} end;

procedure TPBCustomShareMulti.WriteItem(const Index : integer);
begin
	if (ShareFunction = sfReader) then Exit;
	if ([msOpen, msOpening] * FPStatus^ <> [])
		then ShareManager.WriteItem(Self, FOnDoWriteItem, Index)
	else Raise EShareException.Create('PBShare must be open for this operation !');
end;

procedure TPBCustomShareMulti.WriteItemSize(const NumberOfStrings, TotalItemSize : integer);
begin
	if (ShareFunction = sfReader) then Exit;
	if ([msOpen, msOpening] * FPStatus^ <> [])
		then ShareManager.WriteItemSize(ShareManager.CallbackIndex,
		ShareManager.CallBackOldSize, ShareManager.CallBackAddSize,
		NumberOfStrings * SizeOf(Integer) + TotalItemSize)
	else Raise EShareException.Create('PBShare must be open for this operation !');
end;

//  -------------------  TPBShareMulti  ----------------------
constructor TPBShareMulti.CreateNoOwner(AShareName : string; AAllocBy : integer;
	AFlags : TShareFlags; AOpenNow : Boolean; AShareFunction : TShareFunction;
	AOnClosing : TClosingShareEvent; AOnOpen : TOpenShareEvent;
	AOnUpdateAll : TUpdateAllEvent;	AOnDoReadItem : TDoReadItemEvent;
	AOnDoWriteItem : TDoWriteItemEvent;	AOnUpdateItem : TUpdateItemEvent;
	AOnRemoteControl : TRemoteControlEvent);
begin
	inherited CreateNoOwner(AShareName, AAllocBy, AFlags,
		AShareFunction, AOnClosing, AOnOpen, AOnUpdateAll, AOnUpdateItem,
		AOnRemoteControl);
	FOnDoReadItem := AOnDoReadItem;
	FOnDoWriteItem := AOnDoWriteItem;
	if AOpenNow then LazyOpenShare;
end;

constructor TPBShareMulti.CreateNoEvents(AShareName : string; AAllocBy : integer;
	AFlags : TShareFlags; AOpenNow : Boolean; AShareFunction : TShareFunction;
	AOnClosing : TClosingShareProc; AOnOpen : TOpenShareProc;
	AOnUpdateAll : TUpdateAllProc; AOnDoReadItem : TDoReadItemProc;
	AOnDoWriteItem : TDoWriteItemProc; AOnUpdateItem : TUpdateItemProc;
	AOnRemoteControl : TRemoteControlProc);
begin
	inherited CreateNoEvents(AShareName, AAllocBy, AFlags,
		AShareFunction, AOnClosing, AOnOpen, AOnUpdateAll, AOnUpdateItem,
		AOnRemoteControl);
	FOnDoReadItem := TDoReadItemEvent(MakeMethod(@AOnDoReadItem));
	FOnDoWriteItem := TDoWriteItemEvent(MakeMethod(@AOnDoWriteItem));
	if AOpenNow then OpenShare;
end;

procedure TPBShareMulti.WriteItemSize(const NumberOfStrings, TotalItemSize : integer);
begin
	inherited;
end;

//  ---------------------  initialization  -----------------------
initialization
	FUpdateItemMessageID := RegisterWindowMessage('PBShare-UpdateItem-Message');

end.
