unit PBShareMemIni;

{$INCLUDE PBDefines.inc}

interface

uses
	Windows, Messages, SysUtils, Classes, IniFiles, PBShare, PBShareManager;

type
	TPBShareMemIni = class;

	TPBShareMemIniFile = class(TMemIniFile)
	private
		FOnChange : TNotifyEvent;
		FOwner : TPBShareMemIni;
	public
		constructor Create(AOwner : TPBShareMemIni; const FileName : string);
		procedure Clear;
		procedure Deletekey(const Section, Ident: string); override;
		procedure EraseSection(const Section: string); override;
		procedure Rename(const FileName: string; Reload: Boolean);
		procedure UpdateFile; override;
		procedure WriteString(const Section, Ident, Value: string); override;
		property OnChange : TNotifyEvent read FOnChange write FOnChange;
	end;

	TPBShareMemIni = class(TPBCustomShare)
	private
		{ Private declarations }
		FMemIni : TPBShareMemIniFile;
		FStrings : TStringList;
		FAutoUpdate : Boolean;
		DesignFileName : string;
		function GetFileName : string;
		procedure DoWriteList(Sender : TObject; const Stream : TMemoryStream;
			const Writer : TPBWriter; var ExtraInfo : integer);
		procedure DoReadList(Sender : TObject; const Stream : TMemoryStream;
			const Reader : TPBReader; const FromHandle : HWnd;
			const ExtraInfo : integer);
		procedure MemIniChange(Sender : TObject);
		procedure SetFileName(Value : string);
	protected
		{ Protected declarations }
	public
		{ Public declarations }
		constructor Create(AOwner : TComponent); override;
		constructor CreateNoOwner(AFileName, AShareName : string; AAllocBy : integer;
			AFlags : TShareFlags; AAutoUpdate, AOpenNow : Boolean;
			AShareFunction : TShareFunction; AOnClosing : TClosingShareEvent;
			AOnOpen : TOpenShareEvent; AOnUpdateAll : TUpdateAllEvent;
			AOnRemoteControl : TRemoteControlEvent); reintroduce;
		constructor CreateNoEvents(AFileName, AShareName : string; AAllocBy : integer;
			AFlags : TShareFlags; AAutoUpdate, AOpenNow : Boolean;
			AShareFunction : TShareFunction; AOnClosing : TClosingShareProc;
			AOnOpen : TOpenShareProc; AOnUpdateAll : TUpdateAllProc;
			AOnRemoteControl : TRemoteControlProc); reintroduce;
		destructor Destroy; override;
		procedure Clear; override;
		procedure LoadFromFile(FileName : string); override;
		procedure ReadAll; override;
		procedure SaveToFile(FileName : string); override;
		procedure WriteAll; override;
		property ExtraInfo;
		property MemIni : TPBShareMemIniFile read FMemIni;
	published
		{ Published declarations }
		property AutoUpdate : Boolean read FAutoUpdate
			write FAutoUpdate default True;
		property FileName : string read GetFileName write SetFileName;
	end;

implementation

//  ------------------  TPBShareMemIniFile  --------------------------
constructor TPBShareMemIniFile.Create(AOwner : TPBShareMemIni;
	const FileName : string);
begin
	inherited Create(Filename);
	FOwner := AOwner;
end;

procedure TPBShareMemIniFile.Clear;
begin
	inherited;
	if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TPBShareMemIniFile.Deletekey(const Section, Ident: string);
begin
	inherited;
	if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TPBShareMemIniFile.EraseSection(const Section: string);
begin
	inherited;
	if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TPBShareMemIniFile.Rename(const FileName: string; Reload: Boolean);
begin
	inherited;
	FOwner.FileName := FileName;
	if Reload and Assigned(FOnChange) then FOnChange(Self);
end;

procedure TPBShareMemIniFile.UpdateFile;
begin
	inherited;
end;

procedure TPBShareMemIniFile.WriteString(const Section, Ident, Value: string);
begin
	inherited;
	if Assigned(FOnChange) then FOnChange(Self);
end;

//  -------------------  TPBShareMemIni  --------------------------
constructor TPBShareMemIni.Create(AOwner : TComponent);
begin
	inherited;
	FAutoUpdate := True;
	DesignFileName := '';
	if not (csDesigning in ComponentState) then
	begin
		FMemIni := TPBShareMemIniFile.Create(Self, '');
		FMemIni.OnChange := MemIniChange;
		FStrings := TStringList.Create;
		OnDoWriteAll := DoWriteList;
		OnDoReadAll := DoReadList;
	end;
end;

constructor TPBShareMemIni.CreateNoOwner(AFileName, AShareName : string;
	AAllocBy : integer;	AFlags : TShareFlags; AAutoUpdate, AOpenNow : Boolean;
	AShareFunction : TShareFunction; AOnClosing : TClosingShareEvent;
	AOnOpen : TOpenShareEvent; AOnUpdateAll : TUpdateAllEvent;
	AOnRemoteControl : TRemoteControlEvent);
begin
	Create(nil);
	FileName := AFileName;
	FAutoUpdate := AAutoUpdate;
	Flags := AFlags;
	ShareFunction := AShareFunction;
	ShareName := AShareName;
	OnClosing := AOnClosing;
	OnOpen := AOnOpen;
	OnUpdateAll := AOnUpdateAll;
	OnRemoteControl := AOnRemoteControl;
	if AOpenNow then LazyOpenShare;
end;

constructor TPBShareMemIni.CreateNoEvents(AFileName, AShareName : string;
	AAllocBy : integer; AFlags : TShareFlags; AAutoUpdate, AOpenNow : Boolean;
	AShareFunction : TShareFunction; AOnClosing : TClosingShareProc;
	AOnOpen : TOpenShareProc;	AOnUpdateAll : TUpdateAllProc;
	AOnRemoteControl : TRemoteControlProc);
begin
	Create(nil);
	FileName := AFileName;
	FAutoUpdate := AAutoUpdate;
	Flags := AFlags;
	ShareFunction := AShareFunction;
	ShareName := AShareName;
	OnClosing := TClosingShareEvent(MakeMethod(@AOnClosing));
	OnOpen := TOpenShareEvent(MakeMethod(@AOnOpen));
	OnUpdateAll := TUpdateAllEvent(MakeMethod(@AOnUpdateAll));
	OnRemoteControl := TRemoteControlEvent(MakeMethod(@AOnRemoteControl));
	if AOpenNow then LazyOpenShare;
end;

destructor TPBShareMemIni.Destroy;
begin
	if not (csDesigning in ComponentState) then
	begin
		if IsOpen then CloseShare;
		FStrings.Free;
		FStrings := nil;
		FMemIni.Free;
		FMemIni := nil;
	end;
	inherited;
end;

function TPBShareMemIni.GetFileName : string;
begin
	if FMemIni = nil then Result := DesignFileName
	else Result := FMemIni.FileName;
end;

procedure TPBShareMemIni.Clear;
var
	TempAuto : Boolean;
begin
	TempAuto := FAutoUpdate;
	FAutoUpdate := False;
	FMemIni.Clear;
	WriteAll;
	FAutoUpdate := TempAuto;
end;

procedure TPBShareMemIni.DoWriteList(Sender : TObject;
	const Stream : TMemoryStream; const Writer : TPBWriter;
	var ExtraInfo : integer);
begin
	FStrings.Clear;
	FStrings.Add(FileName);
	FMemIni.GetStrings(FStrings);
	FStrings.SaveToStream(Stream);
end;

procedure TPBShareMemIni.DoReadList(Sender : TObject;
	const Stream : TMemoryStream; const Reader : TPBReader;
	const FromHandle : HWnd; const ExtraInfo : integer);
begin
	FStrings.LoadFromStream(Stream);
	if (FStrings.Count > 0) then
	begin
		if (FileName <> FStrings[0]) then FMemIni.Rename(FStrings[0], False);
		FStrings.Delete(0);
	end;
	FMemIni.SetStrings(FStrings);
end;

procedure TPBShareMemIni.LoadFromFile(FileName : string);
begin
	FMemIni.Rename(FileName, True);
end;

procedure TPBShareMemIni.MemIniChange(Sender : TObject);
begin
	if FAutoUpdate and IsOpen then WriteAll;
end;

procedure TPBShareMemIni.ReadAll;
begin
	inherited;
end;

procedure TPBShareMemIni.SaveToFile(FileName : string);
begin
	FMemIni.Rename(FileName, False);
	FMemIni.UpdateFile;
end;

procedure TPBShareMemIni.SetFileName(Value : string);
begin
	if FMemIni = nil then DesignFileName := Value
	else if FMemIni.FileName <> Value then FMemIni.Rename(Value, True);
end;

procedure TPBShareMemIni.WriteAll;
begin
	inherited;
end;

end.
