unit PBShareManager;

{$INCLUDE PBDefines.inc}

interface

uses
	Windows, SysUtils, Classes, PBShareStream;

type
	TShareStatus = (msClosing, msDestroying, msFirstShare, msOpen, msOpening,
		msReadingShareList,	msReadingData, msActiveResizing, msPassiveResizing,
		msUpdating, msWritingShareList,	msWritingData);
	TShareStatusSet = set of TShareStatus;
	PShareStatusSet = ^TShareStatusSet;

	TMultiOperation = (moNone, moWriteAll, moAdd, moDelete, moInsert, moWrite,
		moRepair);

	TClosingShareEvent = procedure(Sender : TObject;
		LastShare : Boolean) of object;
	TClosingShareProc = procedure(Self, Sender : TObject; LastShare : Boolean);
	TOpenShareEvent = procedure(Sender : TObject;
		FirstShare : Boolean) of object;
	TOpenShareProc = procedure(Self, Sender : TObject; FirstShare : Boolean);
	TDoReadAllEvent = procedure(Sender : TObject; const Stream : TMemoryStream;
		const Reader : TPBReader; const FromHandle : HWnd;
		const ExtraInfo : integer) of object;
	TDoReadAllProc = procedure(Self, Sender : TObject; const Stream : TMemoryStream;
		const Reader : TPBReader; const FromHandle : HWnd;
		const ExtraInfo : integer);
	TDoWriteAllEvent = procedure(Sender : TObject; const Stream : TMemoryStream;
		const Writer : TPBWriter; var ExtraInfo : integer) of object;
	TDoWriteAllProc = procedure(Self, Sender : TObject; const Stream : TMemoryStream;
		const Writer : TPBWriter; var ExtraInfo : integer);
	TDoReadItemEvent = procedure(Sender : TObject; const Stream : TMemoryStream;
		const Reader : TPBReader; const Index, ItemSize : integer) of object;
	TDoReadItemProc = procedure(Self, Sender : TObject; const Stream : TMemoryStream;
		const Reader : TPBReader; const Index, ItemSize : integer);
	TDoWriteItemEvent = procedure(Sender : TObject; const Stream : TMemoryStream;
		const Writer : TPBWriter; const Index : integer) of object;
	TDoWriteItemProc = procedure(Self, Sender : TObject; const Stream : TMemoryStream;
		const Writer : TPBWriter; const Index : integer);
	TDataChangeEvent = procedure(Sender : TObject; const Index : integer;
		const Resized : Boolean; const ItemOperation : TMultiOperation) of object;
	TRequestResizeShareEvent = procedure(Sender : TObject;
		NewCapacity : integer) of object;
	TUpdateAllEvent = procedure(Sender : TObject; const FromHandle : HWnd;
		const ExtraInfo : integer) of object;
	TUpdateAllProc = procedure(Self, Sender : TObject; const FromHandle : HWnd;
		const ExtraInfo : integer);
	TUpdateItemEvent = procedure(Sender : TObject; const FromHandle : HWnd;
		const Index : integer; const ItemOperation : TMultiOperation) of object;
	TUpdateItemProc = procedure(Self, Sender : TObject; const FromHandle : HWnd;
		const Index : integer; const ItemOperation : TMultiOperation);
	TRemoteControlEvent = procedure(Sender : TObject; const FromHandle : HWnd;
		const Param : integer) of object;
	TRemoteControlProc = procedure(Self, Sender : TObject; const FromHandle : HWnd;
		const Param : integer);

	TPBSingleShareManager = class(TObject)
	private
		FStream : TPBShareStream;
		FTempStream : TMemoryStream;
		FReader : TPBReader;
		FWriter : TPBWriter;
		FOnDataChange : TDataChangeEvent;
		FOnRequestResizeShare : TRequestResizeShareEvent;
		FResized, MustCallWriteSize : Boolean;
		PWriteMutex, PReadEvent : PHandle;
		PFileMapPointer : PPInteger;
		FPAllocBy, PWriteLockCount, PReadLockCount : PInteger;
		PSynchronizeRead, PNeverResizeDown : PBoolean;
		FPStatus : PShareStatusSet;
		TempPosition : integer;
	protected
		FOwner : TObject;
		TempWriteSize : integer;
		function CheckShareSize(const NewStreamSize : integer) : Boolean;
		procedure LockShare(const Lock : TShareLock);
		procedure UnLockShare(const Lock : TShareLock);
	public
		constructor Create(MemPtr : PInteger; APAllocBy : PInteger;
			APWriteMutex, APReadEvent : PHandle; APFileMapPointer : PPInteger;
			APWriteLockCount, APReadLockCount : PInteger;
			APSynchronizeRead, APNeverResizeDown : PBoolean; APStatus : PShareStatusSet;
			ATempStream : TMemoryStream; AOnRequestResizeShare : TRequestResizeShareEvent;
			AOnDataChange : TDataChangeEvent); virtual;
		destructor Destroy; override;
		procedure Clear; virtual;
		procedure LoadFromStream(Stream : TStream); virtual;
		procedure MakeList; virtual;
		procedure ReadAll(Sender : TObject; Event : TDoReadAllEvent;
			const FromHandle : HWnd; const Extrainfo : integer); virtual;
		procedure SaveToStream(Stream : TStream);
		procedure WriteAll(Sender : TObject; Event : TDoWriteAllEvent;
			var Extrainfo : integer); virtual;
		procedure WriteSize(Size : integer); virtual;
		property Stream : TPBShareStream read FStream;
	end;

	TPBMultiShareManager = class(TPBSingleShareManager)
	private
		FList : TList;
		FCount, FItemSize : integer;
		MustCallWriteItemSize : Boolean;
		function ReadItemSize(const Index : integer) : integer;
		function CheckItemSize(const Index : integer;
			const OldSize, NewSize : integer) : Boolean;
		procedure GetAndWriteItem(Sender : TObject; Event : TDoWriteItemEvent;
			const Index : integer);
	public
		CallBackIndex, CallBackOldsize, CallBackAddSize : integer;
		constructor Create(MemPtr : PInteger; APAllocBy : PInteger;
			APWriteMutex, APReadEvent : PHandle; APFileMapPointer : PPInteger;
			APWriteLockCount, APReadLockCount : PInteger;
			APSynchronizeRead, APNeverResizeDown : PBoolean; APStatus : PShareStatusSet;
			ATempStream : TMemoryStream; AOnRequestResizeShare : TRequestResizeShareEvent;
			AOnDataChange : TDataChangeEvent); override;
		destructor Destroy; override;
		function Add(Sender : TObject; Event : TDoWriteItemEvent) : integer; virtual;
		procedure Clear; override;
		procedure LoadFromStream(Stream : TStream); override;
		procedure MakeList; override;
		procedure Delete(const Index : integer); virtual;
		procedure Insert(Sender : TObject; Event : TDoWriteItemEvent;
			const Index : integer); virtual;
		procedure UpdateList(const FromIndex : integer);
		procedure ReadItem(Sender : TObject; Event : TDoReadItemEvent;
			const Index : integer); virtual;
		procedure WriteAll(Sender : TObject; Event : TDoWriteAllEvent;
			var Extrainfo : integer); override;
		procedure WriteItem(Sender : TObject; Event : TDoWriteItemEvent;
			const Index : integer); virtual;
		procedure WriteItemSize(const Index, Oldsize, AddSize, ItemSize : integer);
		property Count : integer read FCount;
	end;

	TShareInfo = record
		Handle : HWnd;
		ProcessID, ThreadID : Cardinal;
		ExeName, ModuleName, Version : string;
		IsDLL, IsReader : Boolean;
	end;

	TPBShareInfoManager = class(TPBMultiShareManager)
	private
		function GetShareInfo(Index : integer) : TShareInfo;
	public
		FShareInfo : TShareInfo;
		function Add(Sender : TObject; Event : TDoWriteItemEvent) : integer; override;
		function IndexOfHandle(const Handle : HWnd) : integer;
		procedure Delete(const Index : integer); override;
		procedure DoWrite(Sender : TObject; const Stream : TMemoryStream;
			const Writer : TPBWriter; const Index : integer);
		procedure DoRead(Sender : TObject; const Stream : TMemoryStream;
			const Reader : TPBReader; const Index, ItemSize : integer);
		property ShareInfos[Index : integer] : TShareInfo read GetShareInfo; default;
	end;

	TPBShareManagerClass = class of TPBSingleShareManager;

const
	msReading = [msReadingShareList, msReadingData];
	msWriting = [msWritingShareList, msWritingData];
	msResizing = [msActiveResizing, msPassiveResizing];

implementation

//  -------------------  TPBSingleShareManager  --------------------
constructor TPBSingleShareManager.Create(MemPtr : PInteger; APAllocBy : PInteger;
	APWriteMutex, APReadEvent : PHandle; APFileMapPointer : PPInteger;
	APWriteLockCount, APReadLockCount : PInteger;
	APSynchronizeRead, APNeverResizeDown : PBoolean; APStatus : PShareStatusSet;
	ATempStream : TMemoryStream; AOnRequestResizeShare : TRequestResizeShareEvent;
	AOnDataChange : TDataChangeEvent);
begin
	inherited Create;
	PWriteMutex := APWriteMutex;
	PReadEvent := APReadEvent;
	PFileMapPointer := APFileMapPointer;
	PWriteLockCount := APWriteLockCount;
	PReadLockCount := APReadLockCount;
	PSynchronizeRead := APSynchronizeRead;
	PNeverResizeDown := APNeverResizeDown;
	FPAllocBy := APAllocBy;
	FPStatus := APStatus;
	FOnRequestResizeShare := AOnRequestResizeShare;
	FOnDataChange := AOnDataChange;
	FTempStream := ATempStream;
	if msFirstShare in FPStatus^
		then FStream := TPBShareStream.Create(MemPtr, APAllocBy^, APWriteMutex,
		APReadEvent, APFilemapPointer, APWriteLockCount, APReadLockCount,
		APSynchronizeRead)
	else FStream := TPBShareStream.Create(MemPtr, -1, APWriteMutex,
		APReadEvent, APFilemapPointer, APWriteLockCount, APReadLockCount,
		APSynchronizeRead);
	FReader := TPBReader.Create(FStream);
	FWriter := TPBWriter.Create(FStream);
end;

destructor TPBSingleShareManager.Destroy;
begin
	FWriter.Free;
	FWriter := nil;
	FReader.Free;
	FReader := nil;
	FStream.Free;
	FStream := nil;
	inherited;
end;

function TPBSingleShareManager.CheckShareSize(const NewStreamSize : integer) : Boolean;
var
	NewCapacity : integer;
begin
	Result := False;
	if (not (msDestroying in FPStatus^))
		and ((NewStreamSize > FStream.Capacity) or ((not PNeverResizeDown^)
		and (NewStreamSize < FStream.Capacity - FPAllocBy^))) then
	begin
		Result := True;
		NewCapacity := (NewStreamSize div FPAllocBy^ + 1) * FPAllocBy^;
		if Assigned(FOnRequestResizeShare)
			then FOnRequestResizeShare(Self, NewCapacity)
		else Raise EShareException.Create
			('PBShareManager: OnRequestResizeShare event not assigned !');
	end;
end;

procedure TPBSingleShareManager.Clear;
begin
	LockShare(slWrite);
	FStream.Clear;
	FResized := CheckShareSize(FPAllocBy^);
	FOnDataChange(Self, -1, FResized, moWriteAll);
	UnLockShare(slWrite);
end;

procedure TPBSingleShareManager.LoadFromStream(Stream : TStream);
begin
	LockShare(slWrite);
	FStream.Clear;
	Stream.Position := 0;
	MustCallWriteSize := True;
	WriteSize(Stream.Size);
	FStream.CopyFrom(Stream, Stream.Size);
	FOnDataChange(Self, -1, FResized, moWriteAll);
	UnLockShare(slWrite);
end;

procedure TPBSingleShareManager.LockShare(const Lock : TShareLock);
begin
	PBShareStream.LockShare(Lock, PSynchronizeRead, PWriteMutex, PReadEvent,
		PWriteLockCount, PReadLockCount, PFileMapPointer);
end;

procedure TPBSingleShareManager.UnLockShare(const Lock : TShareLock);
begin
	PBShareStream.UnLockShare(Lock, PSynchronizeRead, PWriteMutex, PReadEvent,
		PWriteLockCount, PReadLockCount, PFileMapPointer);
end;

procedure TPBSingleShareManager.MakeList; begin {DoNothing} end;

procedure TPBSingleShareManager.ReadAll(Sender : TObject; Event : TDoReadAllEvent;
	const FromHandle : HWnd; const Extrainfo : integer);
begin
	if not Assigned(Event) then Raise EShareException.Create
		('PBShareManager: OnDoReadAll Event not assigned !');
	LockShare(slRead);
	FTempStream.LoadFromStream(FStream);
	FTempStream.Position := 0;
	FStream.Position := 0;
	Event(Sender, FTempStream, FReader, FromHandle, Extrainfo);
	FTempStream.Clear;
	UnLockShare(slRead);
end;

procedure TPBSingleShareManager.SaveToStream(Stream : TStream);
begin
	FStream.Position := 0;
	Stream.Position := 0;
	Stream.CopyFrom(FStream, FStream.Size);
end;

procedure TPBSingleShareManager.WriteAll(Sender : TObject; Event : TDoWriteAllEvent;
	var Extrainfo : integer);
begin
	if not Assigned(Event) then Raise EShareException.Create
		('PBShareManager: OnDoWriteAll Event not assigned !')
	else if not Assigned(FOnDataChange) then Raise EShareException.Create
		('PBShareManager: OnDataChange Event not assigned !');
	LockShare(slWrite);
	FTempStream.Clear;
	MustCallWriteSize := True;
	FStream.Position := 0;
	TempWriteSize := 0;
	Event(Sender, FTempStream, FWriter, Extrainfo);
	if (FStream.Position > 0) and MustCallWriteSize
		then Raise EShareException.Create('PBShareManager - OnDoWriteAll: '
		+ 'When using Writer you must first call ''WriteSize'' !')
	else if TempWriteSize <> FStream.Position
		then Raise EShareException.Create('PBShareManager - OnDoWriteAll: '
		+ 'The specified and actual size differs !')
	else if MustCallWriteSize then
	begin
		WriteSize(FTempStream.Size);
		FTempStream.SaveToStream(FStream);
		FTempStream.Clear;
	end;
	FOnDataChange(Self, Extrainfo, FResized, moWriteAll);
	UnLockShare(slWrite);
end;

procedure TPBSingleShareManager.WriteSize(Size : integer);
begin
	if not MustCallWriteSize then Raise EShareException.Create
		('PBShareManager: ''WriteSize'' may only be called (once)'
		+ #13#10'from inside TDoWriteAll event-handlers !');
	MustCallWriteSize := False;
	TempWriteSize := Size;
	FResized := CheckShareSize(Size);
	FStream.Size := Size;
end;

//  ------------------  TPBMultiShareManager  ----------------------
constructor TPBMultiShareManager.Create(MemPtr : PInteger; APAllocBy : PInteger;
	APWriteMutex, APReadEvent : PHandle; APFileMapPointer : PPInteger;
	APWriteLockCount, APReadLockCount : PInteger;
	APSynchronizeRead, APNeverResizeDown : PBoolean; APStatus : PShareStatusSet;
	ATempStream : TMemoryStream; AOnRequestResizeShare : TRequestResizeShareEvent;
	AOnDataChange : TDataChangeEvent);
begin
	inherited;
	FList := TList.Create;
end;

destructor TPBMultiShareManager.Destroy;
begin
	FList.Free;
	FList := nil;
	inherited;
end;

function TPBMultiShareManager.Add(Sender : TObject; Event : TDoWriteItemEvent) : integer;
begin
	LockShare(slWrite);
	FStream.Position := FStream.Size;
	CallBackIndex := FCount;
	CallBackOldsize := 0;
	CallBackAddSize := SizeOf(integer);
	FList := FList.Expand;
	GetAndWriteItem(Sender, Event, FCount);
	Result := FCount - 1;
	FOnDataChange(Self, Result, FResized, moAdd);
	UnLockShare(slWrite);
end;

function TPBMultiShareManager.CheckItemSize(const Index : integer;
	const OldSize, NewSize : integer) : Boolean;
var
	OldPosition, NewStreamSize : integer;
begin
	Result := False;
	if (NewSize = OldSize) then Exit
	else
	begin
		NewStreamSize := FStream.Size - OldSize + NewSize;
		if (NewSize > OldSize)
			then Result := CheckShareSize(NewStreamSize);
		if Index < FCount - 1 then
		begin
			OldPosition := FStream.GetPosition(FList[Index + 1]);
			FStream.Move(OldPosition, OldPosition - OldSize + NewSize,
				FStream.Size - OldPosition);
		end;
		FStream.Size := NewStreamSize;
		if (NewSize < OldSize)
			then Result := CheckShareSize(NewStreamSize);
	end;
end;

function TPBMultiShareManager.ReadItemSize(const Index : integer) : integer;
begin
	FStream.Position := FStream.GetPosition(FList[Index]);
	FStream.Read(Result, SizeOf(Integer));
end;

procedure TPBMultiShareManager.Clear;
begin
	FList.Clear;
	FCount := 0;
	inherited;
end;

procedure TPBMultiShareManager.ReadItem(Sender : TObject;
	Event : TDoReadItemEvent; const Index : integer);
begin
	if (Index < 0) or (Index >= FCount) then Raise EShareException.Create
		('PBShareManager: Read Index out of range (' + IntToStr(Index) + ') !')
	else if not Assigned(Event) then Raise EShareException.Create
		('PBShareManager: OnDoReadItem Event not assigned !');
	LockShare(slRead);
	FItemSize := ReadItemSize(Index);
	FTempStream.Clear;
	if FItemSize <> 0 then FTempStream.CopyFrom(FStream, FItemSize);
	FStream.Seek(-FItemSize, soFromCurrent);
	FTempStream.Position := 0;
	Event(Sender, FTempStream, FReader, Index, FItemSize);
	FTempStream.Clear;
	UnLockShare(slRead);
end;

procedure TPBMultiShareManager.WriteItem(Sender : TObject;
	Event : TDoWriteItemEvent; const Index : integer);
begin
	if (Index < 0) or (Index >= FCount) then Raise EShareException.Create
		('PBShareManager: Write Index out of range (' + IntToStr(Index) + ') !');
	LockShare(slWrite);
	CallBackOldsize := ReadItemSize(Index);
	FStream.Seek(-SizeOf(Integer), soFromCurrent);
	CallBackIndex := Index;
	CallBackAddSize := 0;
	GetAndWriteItem(Sender, Event, Index);
	if not FResized then UpdateList(Index);
	FOnDataChange(Self, Index, FResized, moWrite);
	UnLockShare(slWrite);
end;

procedure TPBMultiShareManager.WriteItemSize(const Index, Oldsize,
	AddSize, ItemSize : integer);
var
	TempItemSize : integer;
begin
	if not MustCallWriteItemSize then Raise EShareException.Create
		('PBShareManager: ''WriteItemSize'' may only be called (once)'
		+ #13#10'from inside TDoWriteItem event-handlers !');
	FResized := CheckItemSize(Index, OldSize, ItemSize + AddSize);
	if Index >= FCount then
	begin
		if FCount > 0 then
		begin
			TempItemSize := ReadItemSize(FCount - 1);
			FStream.Seek(TempItemSize, soFromCurrent);
		end
		else FStream.Position := 0;
		FCount := FList.Add(FStream.GetItemPointer(FStream.Position)) + 1;
	end
	else FStream.Position := FStream.GetPosition(FList[Index]);
	FStream.Write(ItemSize, SizeOf(Integer));
	MustCallWriteItemSize := False;
	TempWriteSize := ItemSize;
end;

procedure TPBMultiShareManager.Delete(const Index : integer);
begin
	if (Index < 0) or (Index >= FCount)
		then Raise ERangeError.Create('PBShareManager: Index out of range !')
	else if not Assigned(FOnDataChange) then Raise EShareException.Create
		('PBShareManager: OnDataChange Event not assigned !');
	LockShare(slWrite);
	FItemSize := ReadItemSize(Index);
	FResized := CheckItemSize(Index, FItemSize + SizeOf(integer), 0);
	if not FResized then MakeList;
	FOnDataChange(Self, Index, FResized, moDelete);
	UnLockShare(slWrite);
end;

procedure TPBMultiShareManager.Insert(Sender : TObject;
	Event : TDoWriteItemEvent; const Index : integer);
begin
	if (Index < 0) or (Index >= FCount) then Raise EShareException.Create
		('PBShareManager: Insert Index out of range !');
	LockShare(slWrite);
	FList := FList.Expand;
	FList.Insert(Index, FList[Index]);
	Inc(FCount);
	FStream.Position := FStream.GetPosition(FList[Index]);
	CallBackIndex := Index;
	CallBackOldsize := 0;
	CallBackAddSize := SizeOf(integer);
	GetAndWriteItem(Sender, Event, Index);
	if FResized then MakeList
	else UpdateList(Index);
	FOnDataChange(Self, Index, FResized, moInsert);
	UnLockShare(slWrite);
end;

procedure TPBMultiShareManager.LoadFromStream(Stream : TStream);
begin
	inherited;
	Makelist;
end;

procedure TPBMultiShareManager.MakeList;
var
	TempItemSize, TempStreamSize : integer;
begin
	LockShare(slRead);
	FList.Clear;
	FStream.Position := 0;
	TempStreamSize := FStream.Size;
	while FStream.Position < TempStreamSize do
	begin
		FList := FList.Expand;
		FList.Add(FStream.GetItemPointer(FStream.Position));
		FStream.Read(TempItemSize, SizeOf(Integer));
		FStream.Seek(TempItemSize, soFromCurrent);
	end;
	FCount := FList.Count;
	UnLockShare(slRead);
end;

procedure TPBMultiShareManager.UpdateList(const FromIndex : integer);
var
	t, TempItemSize : integer;
begin
	LockShare(slRead);
	for t := FromIndex to FCount - 1 do
	begin
		TempItemSize := ReadItemSize(t);
		FStream.Seek(TempItemSize, soFromCurrent);
		if t < FCount - 1
			then FList[t + 1] := FStream.GetItemPointer(FStream.Position);
	end;
	UnLockShare(slRead);
end;

procedure TPBMultiShareManager.WriteAll(Sender : TObject; Event : TDoWriteAllEvent;
	var Extrainfo : integer);
begin
	LockShare(slWrite);
	inherited;
	MakeList;
	UnLockShare(slWrite);
end;

procedure TPBMultiShareManager.GetAndWriteItem(Sender : TObject;
	Event : TDoWriteItemEvent; const Index : integer);
begin
	if not Assigned(Event) then Raise EShareException.Create
		('PBShareManager: OnDoWriteItem Event not assigned !')
	else if not Assigned(FOnDataChange) then Raise EShareException.Create
		('PBShareManager: OnDataChange Event not assigned !');
	LockShare(slWrite);
	FTempStream.Clear;
	TempPosition := FStream.Position;
	MustCallWriteItemSize := True;
	TempWriteSize := 0;
	Event(Sender, FTempStream, FWriter, Index);
	if (TempPosition <> FStream.Position) and MustCallWriteItemSize
		then Raise EShareException.Create('PBShareManager: When using Writer '
		+ 'you must first call ''WriteItemSize'' !')
	else if (TempPosition + TempWriteSize + SizeOf(Integer) <> FStream.Position)
		and (not MustCallWriteItemSize)
		then Raise EShareException.Create('PBShareManager: The specified and '
		+ 'actual size differs !')
	else if MustCallWriteItemSize and (FTempStream.Size = 0)
		then Raise EShareException.Create('PBSharemanager: ItemSize is zero !')
	else if (FTempStream.Size <> 0) then
	begin
		FItemSize := FTempStream.Size;
		WriteItemSize(Index, CallBackOldsize, CallBackAddSize, FItemSize);
		FTempStream.Position := 0;
		FStream.CopyFrom(FTempStream, FItemSize);
		FTempStream.Clear;
	end;
	UnLockShare(slWrite);
end;

//  ------------------  TPBShareInfoManager  ----------------------
function TPBShareInfoManager.Add(Sender : TObject;
	Event : TDoWriteItemEvent) : integer;
begin
	Include(FPStatus^, msWritingShareList);
	Result := inherited Add(Sender, Event);
	Exclude(FPStatus^, msWritingShareList);
end;

function TPBShareInfoManager.GetShareInfo(Index : integer) : TShareInfo;
begin
	Include(FPStatus^, msReadingShareList);
	ReadItem(Self, DoRead, Index);
	Result := FShareInfo;
	Exclude(FPStatus^, msReadingShareList);
end;

function TPBShareInfoManager.IndexOfHandle(const Handle : HWnd) : integer;
var
	t : integer;
begin
	Result := -1;
	for t := 0 to FCount - 1 do
	begin
		if GetShareInfo(t).Handle = Handle then
		begin
			Result := t;
			Break;
		end;
	end;
end;

procedure TPBShareInfoManager.Delete(const Index : integer);
begin
	Include(FPStatus^, msWritingShareList);
	inherited;
	Exclude(FPStatus^, msWritingShareList);
end;

procedure TPBShareInfoManager.DoWrite(Sender : TObject;
	const Stream : TMemoryStream; const Writer : TPBWriter; const Index : integer);
begin
	WriteItemSize(CallBackIndex, CallBackOldSize, CallBackAddSize,
		3 * SizeOf(Integer) + 3 * SizeOf(Cardinal) + Length(FShareInfo.ExeName)
		+ Length(FShareInfo.ModuleName) + Length(FShareInfo.Version)
		+ 2 * SizeOf(Boolean));
	Writer.WriteString(FShareInfo.Version);
	Writer.WriteCardinal(FShareInfo.Handle);
	Writer.WriteCardinal(FShareInfo.ProcessID);
	Writer.WriteCardinal(FShareInfo.ThreadID);
	Writer.WriteString(FShareInfo.ExeName);
	Writer.WriteString(FShareInfo.ModuleName);
	Writer.WriteBoolean(FShareInfo.IsDLL);
	Writer.WriteBoolean(FShareInfo.IsReader);
end;

procedure TPBShareInfoManager.DoRead(Sender : TObject;
	const Stream : TMemoryStream; const Reader : TPBReader;
	const Index, ItemSize : integer);
begin
	FShareInfo.Version := Reader.ReadString;
	FShareInfo.Handle := Reader.ReadCardinal;
	FShareInfo.ProcessID := Reader.ReadCardinal;
	FShareInfo.ThreadID := Reader.ReadCardinal;
	FShareInfo.ExeName := Reader.ReadString;
	FShareInfo.ModuleName := Reader.ReadString;
	FShareInfo.IsDLL := Reader.ReadBoolean;
	FShareInfo.IsReader := Reader.ReadBoolean;
end;

end.
