unit PBShare;

{$INCLUDE PBDefines.inc}

interface

uses
	Windows, Messages, SysUtils, Classes, TypInfo,
	{$IFDEF COMPILER_MAX_5} Forms, Controls, {$ENDIF}
	PBShareManager, PBShareStream;

type
	TAutoOpenShare = (omManual, omOnLoaded, omOnFormCreate);
	TAutoCloseShare = (cmManual, omOnDestroy, cmOnFormDestroy);

	TShareFunction = (sfReader, sfAutoSwitch, sfWriter, sfUpdateChain);

	TShareFlags = set of (sfAsynchronUpdate, sfAutoRepairList, sfInterProcess,
		{sfInterUser,} sfNeverResizeDown, sfUpdateOnOpen);

	TMultiOperation = PBShareManager.TMultiOperation;

	TPBWriter = PBShareStream.TPBWriter;
	TPBReader = PBShareStream.TPBReader;
	TShareLock = PBShareStream.TShareLock;

	TPBCustomShare = class(TComponent)
	private
		{ Private declarations }
		FShareList : TPBShareInfoManager;
		FSharePointer, FShareListPointer : PInteger;
		FTempStream : TMemoryStream;
		FShareHandle : THandle;
		FPAllocBy, FPShareListAllocBy, PWriteLockCount, PReadLockCount : PInteger;
		PWriteMutex, PReadEvent : PHandle;
		PFileMapPointer : PPInteger;
		PNeverResizeDown : PBoolean;
		FNameSuffix, FShareListExtra : integer;
		FWindowList : array of Hwnd;
		FAutoOpenShare : TAutoOpenShare;
		FAutoCloseShare : TAutoCloseShare;
		HasResized, HasClosedHandles : Boolean;
		FGlobalString : string;
		FShareFunction : TShareFunction;
		FFlags : TShareFlags;
		FOnClosing : TClosingShareEvent;
		FOnOpen : TOpenShareEvent;
		FOnDoReadAll : TDoReadAllEvent;
		FOnDoWriteAll : TDoWriteAllEvent;
		FOnUpdateShareList : TUpdateItemEvent;
		FOnUpdateAll : TUpdateAllEvent;
		FOnRemoteControl : TRemoteControlEvent;
		OldOnCreate, OldOnDestroy : TNotifyEvent;
		function AllocateShareHWnd(Method : TWndMethod) : HWnd;
		function GetAllocBy : integer;
		function GetCapacity : integer;
		function GetEventProperty(const FObject : TPersistent;
			const PropertyName : string; var Method : TMethod) : Boolean;
		function GetShareInfos(Index : integer) : TShareInfo;
		function GetShareCount : integer;
		function GetShareIndex : integer;
		function GetSize : integer;
		function GetStatus : TShareStatusSet;
		function GetNameSuffix : Char;
		function OwnerIsForm : Boolean;
		function SetEventProperty(const FObject : TPersistent;
			const PropertyName : string; const NewValue : TMethod;
			const OnlyEmpty : Boolean) : Boolean;
		function SwitchNameSuffix : Char;
		function VersionAsInteger : integer;
		procedure DeallocateShareHWnd(Wnd: HWND);
		procedure Dummy(Value : string);
		procedure FinishResizeShare;
		procedure NewCreate(Sender : TObject);
		procedure NewDestroy(Sender : TObject);
		procedure PassiveResizeShare(const UpdateCounter, NameSuffix : integer);
		procedure ReadShareList(LParam : integer);
		procedure ReSizeShare(Sender : TObject; NewCapacity : integer);
		procedure SetAutoCloseShare(Value : TAutoCloseShare);
		procedure SetAutoOpenShare(Value : TAutoOpenShare);
		procedure SetAllocBy(Value : integer);
		procedure SetFlags(Value : TShareFlags);
		procedure SetShareFunction(Value : TShareFunction);
		procedure ShareListChange(Sender : TObject; const Index : integer;
			const Resized : Boolean; const ItemOperation : TMultiOperation);
		procedure TimerTimer(Sender : TObject);
	protected
		{ Protected declarations }
		FExtraInfo, UpdateCount, MessageStack : integer;
		FShareWindowhandle, FFromHandle : HWnd;
		FPStatus : PShareStatusSet;
		PSynchronizeRead : PBoolean;
		FShareManager : TPBSingleShareManager;
		FShareManagerClass : TPBShareManagerClass;
		FShareName, FVersion, FWindowName, ProcessStr : string;
		function GetTotalShareSize : integer; virtual;
		function MakeMethod(Proc : Pointer) : TMethod;
		procedure DataChange(Sender : TObject; const Index : integer;
			const Resized : Boolean; const ItemOperation : TMultiOperation); virtual;
		procedure LazyOpenShare;
		procedure LockShare(const Lock : TShareLock);
		procedure Loaded; override;
		procedure NotifyAll(const MessageId : Cardinal; const Extra : integer;
			const Asynchron : Boolean);
		procedure ReadAll; virtual;
		procedure SetShareName(Value : string); virtual;
		procedure UnLockShare(const Lock : TShareLock);
		procedure WndProc(var FMessage : TMessage); virtual;
		procedure WriteAll; virtual;
		procedure WriteSize(const NumberOfStrings, TotalSize : integer); virtual;
		property ExtraInfo : integer read FExtraInfo write FExtraInfo;
		property OnDoReadAll : TDoReadAllEvent read FOnDoReadAll
			write FOnDoReadAll;
		property OnDoWriteAll : TDoWriteAllEvent read FOnDoWriteAll
			write FOnDoWriteAll;
	public
		{ Public declarations }
		constructor Create(AOwner : TComponent); override;
		constructor CreateNoOwner(AShareName : string; AAllocBy : integer;
			AFlags : TShareFlags; AShareFunction : TShareFunction;
			AOnClosing : TClosingShareEvent; AOnOpen : TOpenShareEvent;
			AOnUpdateAll : TUpdateAllEvent;
			AOnRemoteControl : TRemoteControlEvent); virtual;
		constructor CreateNoEvents(AShareName : string; AAllocBy : integer;
			AFlags : TShareFlags; AShareFunction : TShareFunction;
			AOnClosing : TClosingShareProc; AOnOpen : TOpenShareProc;
			AOnUpdateAll : TUpdateAllProc;
			AOnRemoteControl : TRemoteControlProc); virtual;
		destructor Destroy; override;
		function CloseShare : Boolean; virtual;
		function OpenShare : Boolean; virtual;
		function IndexOfHandle(const HandleOfWindow : HWnd) : integer;
		function LockStatus : TShareLock;
		function IsOpen : Boolean;
		procedure BeginUpdate;
		procedure Clear; virtual;
		procedure EndUpdate;
		procedure LoadFromFile(FileName : string); virtual;
		procedure LoadFromStream(Stream : TStream); virtual;
		procedure SaveToFile(FileName : string); virtual;
		procedure SaveToStream(Stream : TStream); virtual;
		procedure RemoteControl(const Param : integer; const ToHandle : HWnd = 0;
			const Asynchron : Boolean = False; const AlsoSendToClosed : Boolean = False);
		property Capacity : integer read GetCapacity;
		property Handle : HWnd read FShareWindowHandle;
		property ShareIndex : integer read GetShareIndex;
		property ShareInfos[Index : integer] : TShareInfo read GetShareInfos;
		property ShareCount : integer read GetShareCount;
		property Size : integer read GetSize;
		property Status : TShareStatusSet read GetStatus;
	published
		{ Published declarations }
		property AllocBy : integer read GetAllocBy write SetAllocBy default $1000;
		property AutoCloseShare : TAutoCloseShare read FAutoCloseShare
			write SetAutoCloseShare;
		property AutoOpenShare : TAutoOpenShare read FAutoOpenShare
			write SetAutoOpenShare;
		property Flags : TShareFlags read FFlags write SetFlags
			default [sfInterProcess, sfUpdateOnOpen, sfAutoRepairList];
		property OnClosing : TClosingShareEvent read FOnClosing write FOnClosing;
		property OnUpdateShareList : TUpdateItemEvent read FOnUpdateShareList
			write FOnUpdateShareList;
		property OnOpen : TOpenShareEvent read FOnOpen write FOnOpen;
		property OnUpdateAll : TUpdateAllEvent read FOnUpdateAll write FOnUpdateAll;
		property OnRemoteControl : TRemoteControlEvent read FOnRemoteControl
			write FOnRemoteControl;
		property ShareFunction : TShareFunction read FShareFunction
			write SetShareFunction default sfAutoSwitch;
		property ShareName : string read FShareName write SetShareName;
		property Version : string read FVersion write Dummy stored False;
	end;

	TPBShareSingle = class(TPBCustomShare)
	public
		constructor CreateNoOwner(AShareName : string; AAllocBy : integer;
			AFlags : TShareFlags; AOpenNow : Boolean; AShareFunction : TShareFunction;
			AOnClosing : TClosingShareEvent; AOnOpen : TOpenShareEvent;
			AOnUpdateAll : TUpdateAllEvent;	AOnDoReadAll : TDoReadAllEvent;
			AOnDoWriteAll : TDoWriteAllEvent;
			AOnRemoteControl : TRemoteControlEvent); reintroduce;
		constructor CreateNoEvents(AShareName : string; AAllocBy : integer;
			AFlags : TShareFlags; AOpenNow : Boolean; AShareFunction : TShareFunction;
			AOnClosing : TClosingShareProc; AOnOpen : TOpenShareProc;
			AOnUpdateAll : TUpdateAllProc; AOnDoReadAll : TDoReadAllProc;
			AOnDoWriteAll : TDoWriteAllProc;
			AOnRemoteControl : TRemoteControlProc); reintroduce;
		procedure ReadAll; override;
		procedure WriteAll; override;
		procedure WriteSize(const NumberOfStrings, TotalSize : integer); override;
		property ExtraInfo;
	published
		property OnDoReadAll;
		property OnDoWriteAll;
	end;

const
	moWriteAll = PBShareManager.moWriteAll;
	moAdd = PBShareManager.moAdd;
	moDelete = PBShareManager.moDelete;
	moInsert = PBShareManager.moInsert;
	moWrite = PBShareManager.moWrite;
	moRepair = PBShareManager.moRepair;

	msClosing = PBShareManager.msClosing;
	msDestroying = PBShareManager.msDestroying;
	msFirstShare = PBShareManager.msFirstShare;
	msOpen = PBShareManager.msOpen;
	msOpening = PBShareManager.msOpening;
	msReadingShareList = PBShareManager.msReadingShareList;
	msReadingData = PBShareManager.msReadingData;
	msActiveResizing = PBShareManager.msActiveResizing;
	msPassiveResizing = PBShareManager.msPassiveResizing;
	msWritingShareList = PBShareManager.msWritingShareList;
	msWritingData = PBShareManager.msWritingData;
	msUpdating = PBShareManager.msUpdating;
	msReading = PBShareManager.msReading;
	msWriting = PBShareManager.msWriting;
	msResizing = PBShareManager.msResizing;

implementation

const
	ComponentVersion = '5.00.00.00';
	DefaultAllocBy = $1000;
	MinAllocBy = $100;
	MaxAllocBy = MAXINT div 16;
	ShareListAllocBy = $800;
	TimerIdent = 1;
	TimerInterVal = 10000;
	StreamHeaderSize = 2 * Sizeof(Integer);
	WindowClassName = 'TPBShareWindow';
	MaxNameSuffix = 2;
	MessageTimeOut = 5000;
	OpenNameSuffix = '-Open';

var
	FUpdateMessageID, FPassiveResizeMessageID, FShareListMessageID : Cardinal;
	FRequestExtraInfoMessageID, FRequestNameSuffixMessageID : Cardinal;
	FRemoteControlMessageID, FLazyOpenShareMessageID, FVersionMessageID : Cardinal;
	SecurityDesc : TSecurityDescriptor;
	SecurityAttr : TSecurityAttributes;

{$IFDEF COMPILER_MAX_5}
procedure RaiseLastOSError;
begin
	RaiseLastWin32Error;
end;
{$ENDIF}


//  ----------------  TPBCustomShare  ----------------------
constructor TPBCustomShare.Create(AOwner : TComponent);
var
	t : integer;
	TempShareName : string;
begin
	inherited Create(AOwner);
	FVersion := ComponentVersion;
	FPAllocBy := AllocMem(SizeOf(PInteger));
	FPAllocBy^ :=	DefaultAllocBy;
	FPShareListAllocBy := AllocMem(SizeOf(PInteger));
	FPShareListAllocBy^ := ShareListAllocBy;
	FShareManagerClass := TPBSingleShareManager;
	FShareFunction := sfAutoSwitch;
	FFlags := [sfInterProcess, sfUpdateOnOpen, sfAutoRepairList];
	FPStatus := AllocMem(SizeOf(FPStatus));
	PFileMapPointer := AllocMem(SizeOf(PPInteger));
	PWriteLockCount := AllocMem(SizeOf(PInteger));
	PReadLockCount := AllocMem(SizeOf(PInteger));
	PSynchronizeRead := AllocMem(SizeOf(PBoolean));
	PNeverResizeDown := AllocMem(SizeOf(PBoolean));
	PWriteMutex := AllocMem(SizeOf(PHandle));
	PReadEvent := AllocMem(SizeOf(PHandle));
	if (csDesigning in ComponentState) then
	begin
		FAutoOpenShare := omOnFormCreate;
		FAutoCloseShare := cmOnFormDestroy;
	end
	else
	begin
		FAutoOpenShare := omManual;
		FAutoCloseShare := cmManual;
		FShareWindowHandle := AllocateShareHWnd(WndProc);
	end;
	TempShareName := 'PBShare-';
	for t := 0 to 6 do TempShareName := TempShareName + Chr(Random(26) + Ord('A'));
	SetShareName(TempShareName);
end;

constructor TPBCustomShare.CreateNoOwner(AShareName : string;
	AAllocBy : integer; AFlags : TShareFlags;
	AShareFunction : TShareFunction; AOnClosing : TClosingShareEvent;
	AOnOpen : TOpenShareEvent; AOnUpdateAll : TUpdateAllEvent;
	AOnRemoteControl : TRemoteControlEvent);
begin
	Create(nil);
	Flags := AFlags;
	ShareName := AShareName;
	FPAllocBy^ := AAllocBy;
	FShareFunction := AShareFunction;
	FOnClosing := AOnClosing;
	FOnOpen := AOnOpen;
	FOnUpdateAll := AOnUpdateAll;
	FOnRemoteControl := AOnRemoteControl;
end;

constructor TPBCustomShare.CreateNoEvents(AShareName : string; AAllocBy : integer;
	AFlags : TShareFlags; AShareFunction : TShareFunction;
	AOnClosing : TClosingShareProc; AOnOpen : TOpenShareProc;
	AOnUpdateAll : TUpdateAllProc; AOnRemoteControl : TRemoteControlProc);
begin
	Create(nil);
	Flags := AFlags;
	ShareName := AShareName;
	FPAllocBy^ := AAllocBy;
	FShareFunction := AShareFunction;
	FOnClosing := TClosingShareEvent(MakeMethod(@AOnClosing));
	FOnOpen := TOpenShareEvent(MakeMethod(@AOnOpen));
	FOnUpdateAll := TUpdateAllEvent(MakeMethod(@AOnUpdateAll));
	FOnRemoteControl := TRemoteControlEvent(MakeMethod(@AOnRemoteControl));
end;

procedure TPBCustomShare.Loaded;
begin
	inherited;
	if (not (csDesigning in ComponentState))
		and (FAutoOpenShare = omOnLoaded) then OpenShare;
end;

destructor TPBCustomShare.Destroy;
begin
	if msOpen in FPStatus^ then CloseShare;
	if not (csDesigning in ComponentState)
		then DeallocateShareHWnd(FShareWindowhandle);
	FreeMem(PReadEvent);
	PReadEvent := nil;
	FreeMem(PWriteMutex);
	PWriteMutex := nil;
	FreeMem(PNeverResizeDown);
	PNeverResizeDown := nil;
	FreeMem(PSynchronizeRead);
	PSynchronizeRead := nil;
	FreeMem(PReadLockCount);
	PReadLockCount := nil;
	FreeMem(PWriteLockCount);
	PWriteLockCount := nil;
	FreeMem(PFileMapPointer);
	PFileMapPointer := nil;
	FreeMem(FPShareListAllocBy);
	FPShareListAllocBy := nil;
	FreeMem(FPAllocBy);
	FPAllocBy := nil;
	FreeMem(FPStatus);
	FPStatus := nil;
	inherited Destroy;
end;

function TPBCustomShare.AllocateShareHWnd(Method : TWndMethod) : HWnd;
var
	ShareWndClass : TWndClass;
begin
	ZeroMemory(@ShareWndClass, SizeOf(TWndClass));
	ShareWndClass.hInstance := HInstance;
	ShareWndClass.lpfnWndProc := @DefWindowProc;
	ShareWndClass.lpszClassName := WindowClassName;
	Windows.RegisterClass(ShareWndClass);
	Result := CreateWindowEx(WS_EX_TOOLWINDOW, ShareWndClass.lpszClassName,
		'', WS_POPUP, 0, 0, 0, 0, 0, 0, HInstance, nil);
	if Assigned(Method) then
		SetWindowLong(Result, GWL_WNDPROC, Longint(MakeObjectInstance(Method)));
end;

function TPBCustomShare.GetAllocBy : integer;
begin
	Result := FPAllocBy^;
end;

function TPBCustomShare.GetCapacity : integer;
begin
	if (FShareManager <> nil) and (FShareManager.Stream <> nil)
		then Result := FShareManager.Stream.Capacity
	else Result := 0;
end;

function TPBCustomShare.GetEventProperty(const FObject : TPersistent;
	const PropertyName : string; var Method : TMethod) : Boolean;
var
	PInfo: PPropInfo;
begin
	Result := False;
	PInfo := GetPropInfo(FObject.ClassInfo, PropertyName);
	if (PInfo <> nil) and (PInfo^.PropType^.Kind = tkmethod) then
	begin
		Method := GetMethodProp(FObject, PInfo);
		Result := True;
	end;
end;

function TPBCustomShare.GetNameSuffix : Char;
begin
	Result := Chr(FNameSuffix + Ord('0'));
end;

function TPBCustomShare.IndexOfHandle(const HandleOfWindow : HWnd) : integer;
begin
	if IsOpen then Result := FShareList.IndexOfHandle(HandleOfWindow)
	else Result := -1;
end;

function TPBCustomShare.OwnerIsForm : Boolean;
var
	ClassRef : TClass;
begin
	Result := False;
	if Owner = nil then Exit
	else
	begin
		ClassRef := Owner.ClassType;
		while (not Result) and (ClassRef <> nil) do
		begin
			if ClassRef.ClassName = 'TCustomForm' then Result := True;
			ClassRef := ClassRef.ClassParent;
		end;
	end;
end;

function TPBCustomShare.SetEventProperty(const FObject : TPersistent;
	const PropertyName : string; const NewValue : TMethod;
	const OnlyEmpty : Boolean) : Boolean;
var
	PInfo: PPropInfo;
begin
	Result := False;
	PInfo := GetPropInfo(FObject.ClassInfo, PropertyName);
	if (PInfo <> nil) and (PInfo^.PropType^.Kind = tkmethod)
		and ((GetMethodProp(FObject, PInfo).Code = nil) or (not OnlyEmpty)) then
	begin
		SetMethodProp(FObject, PInfo, NewValue);
		Result := True;
	end;
end;

function TPBCustomShare.SwitchNameSuffix : Char;
begin
	Inc(FNameSuffix);
	if FNameSuffix > MaxNameSuffix then FNameSuffix := 0;
	Result := GetNameSuffix;
end;

function TPBCustomShare.GetShareInfos(Index : integer) : TShareInfo;
begin
	Result := FShareList[Index];
end;

function TPBCustomShare.GetShareCount : integer;
begin
	if IsOpen and (FShareList <> nil) then Result := FShareList.Count
	else Result := 0;
end;

function TPBCustomShare.GetShareIndex : integer;
begin
	if IsOpen then Result := FShareList.IndexOfHandle(Handle)
	else Result := -1;
end;

function TPBCustomShare.GetSize : integer;
begin
	if (FShareManager <> nil) and (FShareManager.Stream <> nil)
		then Result := FShareManager.Stream.Size
	else Result := 0;
end;

function TPBCustomShare.GetStatus : TShareStatusSet;
begin
	Result := FPStatus^;
end;

function TPBCustomShare.IsOpen : Boolean;
begin
	Result := (msOpen in FPStatus^);
end;

function TPBCustomShare.LockStatus : TShareLock;
begin
	if IsOpen then Result := PBShareStream.LockStatus(PWriteMutex, PReadEvent,
		PWriteLockCount, PReadLockCount, PFileMapPointer)
	else Result := slNone;
end;

function TPBCustomShare.MakeMethod(Proc : Pointer) : TMethod;
begin
	Result.Data := nil;
	Result.Code := Proc;
end;

function TPBCustomShare.VersionAsInteger : integer;
var
	TempString : string;
begin
	TempString := StringReplace(Copy(FVersion, 1, 2), '.', '',
		[rfReplaceAll, rfIgnoreCase]);
	Result := StrToIntDef(TempString, Random(100) + 100);
end;

procedure TPBCustomShare.BeginUpdate;
begin
	if IsOpen then
	begin
		LockShare(slWrite);
		Inc(UpdateCount);
	end;
end;

procedure TPBCustomShare.Clear;
begin
	FShareManager.Clear;
end;

procedure TPBCustomShare.DeallocateShareHWnd(Wnd: HWND);
var
	Instance: Pointer;
begin
  Instance := Pointer(GetWindowLong(Wnd, GWL_WNDPROC));
  DestroyWindow(Wnd);
  if Instance <> @DefWindowProc then FreeObjectInstance(Instance);
end;

procedure TPBCustomShare.EndUpdate;
begin
	if IsOpen then
	begin
		Dec(UpdateCount);
		UnLockShare(slWrite);
		if (UpdateCount = 0) then DataChange(Self, FExtraInfo, HasResized, moWriteAll);
	end;
end;

procedure TPBCustomShare.LockShare(const Lock : TShareLock);
begin
	PBShareStream.LockShare(Lock, PSynchronizeRead, PWriteMutex, PReadEvent,
		PWriteLockCount, PReadLockCount, PFileMapPointer);
end;

procedure TPBCustomShare.UnLockShare(const Lock : TShareLock);
begin
	PBShareStream.UnLockShare(Lock, PSynchronizeRead, PWriteMutex, PReadEvent,
		PWriteLockCount, PReadLockCount, PFileMapPointer);
end;

procedure TPBCustomShare.ResizeShare(Sender : TObject; NewCapacity : integer);
var
	NewShareSize, KeepCapacity : integer;
	FileMapPointer2 : PInteger;
	FShareListPointer2, FSharePointer2 : PInteger;
	FShareHandle2 : THandle;
	Exists : Boolean;
begin
	Include(FPStatus^, msActiveResizing);
	LockShare(slWrite);
	if Sender = FShareList then KeepCapacity := FShareManager.Stream.Capacity
	else KeepCapacity := FShareList.Stream.Capacity;
	NewShareSize := 2 * StreamHeaderSize + KeepCapacity + NewCapacity;
	repeat
		FShareHandle2 := CreateFileMapping($FFFFFFFF, @SecurityAttr, PAGE_READWRITE
			or SEC_COMMIT or SEC_NOCACHE,	0, NewShareSize,
			PChar(FShareName + ProcessStr + SwitchNameSuffix));
		if (FShareHandle2 = INVALID_HANDLE_VALUE) or (FShareHandle2 = 0)
			then RaiseLastOSError;
		Exists := (GetLastError = ERROR_ALREADY_EXISTS);
		if Exists then CloseHandle(FShareHandle2);
	until not Exists;
	FileMapPointer2 := MapViewOfFile(FShareHandle2, FILE_MAP_ALL_ACCESS, 0, 0, 0);
	if FileMapPointer2 = nil then RaiseLastOSError;
	FSharePointer2 := PInteger(Integer(FileMapPointer2) + SizeOf(Integer));
	if Sender = FShareList then	FShareListPointer2 := PInteger(Integer(FSharePointer2)
		+ StreamHeaderSize + KeepCapacity)
	else FShareListPointer2 := PInteger(Integer(FSharePointer2) + StreamHeaderSize
		+ NewCapacity);
	CopyMemory(FileMapPointer2, PFileMapPointer^, SizeOf(Integer));
	if (Sender = FShareManager)
		and (NewCapacity < FShareManager.Stream.Capacity)
		then CopyMemory(FSharePointer2, FSharePointer, StreamHeaderSize + NewCapacity)
	else CopyMemory(FSharePointer2, FSharePointer, StreamHeaderSize
		+ FShareManager.Stream.Capacity);
	if (Sender = FShareList) and (NewCapacity < FShareList.Stream.Capacity)
		then CopyMemory(FShareListPointer2, FShareListPointer,
		StreamHeaderSize + NewCapacity)
	else CopyMemory(FShareListPointer2, FShareListPointer,
		StreamHeaderSize + FShareList.Stream.Capacity);
	UnMapViewOfFile(PFileMapPointer^);
	PFileMapPointer^ := FileMapPointer2;
	FSharePointer := FSharePointer2;
	FShareListPointer := FShareListPointer2;
	CloseHandle(FShareHandle);
	FShareHandle := FShareHandle2;
	if Sender = FShareList then
	begin
		FShareManager.Stream.SetVariables(FSharePointer, KeepCapacity);
		FShareList.Stream.SetVariables(FShareListPointer, NewCapacity);
	end
	else
	begin
		FShareManager.Stream.SetVariables(FSharePointer, NewCapacity);
		FShareList.Stream.SetVariables(FShareListPointer, KeepCapacity);
	end;
	FShareList.MakeList;
	FShareManager.MakeList;
	UnLockShare(slWrite);
	Exclude(FPStatus^, msActiveResizing);
	HasResized := True;
end;

procedure TPBCustomShare.PassiveResizeShare
	(const UpdateCounter, NameSuffix : integer);
var
	Msg : TMsg;
begin
	if IsOpen then
	begin
		Include(FPStatus^, msPassiveResizing);
		Exclude(FPStatus^, msOpen);
		UnMapViewOfFile(PFileMapPointer^);
		PFileMapPointer^ := nil;
		CloseHandle(FShareHandle);
		FShareHandle := 0;
		if UpdateCounter <> 0 then
		begin
			ReplyMessage(1);
			repeat
				if PeekMessage(Msg, FShareWindowhandle, FPassiveResizeMessageID,
					FPassiveResizeMessageID, PM_REMOVE) then DispatchMessage(Msg);
			until IsOpen;
			Exit;
		end;
	end
	else if UpdateCounter <> 0 then Exit;
	FNameSuffix := NameSuffix;
	FShareHandle := OpenFileMapping(FILE_MAP_ALL_ACCESS,
		False, PChar(FGlobalString + FWindowName + GetNameSuffix));
	if (FShareHandle = INVALID_HANDLE_VALUE) or (FShareHandle = 0)
		then RaiseLastOSError;
	PFileMapPointer^ := MapViewOfFile(FShareHandle, FILE_MAP_ALL_ACCESS, 0, 0, 0);
	if PFileMapPointer^ = nil then RaiseLastOSError;
	FSharePointer := PInteger(Integer(PFileMapPointer^) + SizeOf(Integer));
	FShareManager.Stream.SetVariables(FSharePointer, -1);
	PSynchronizeRead^ := True;
	FShareListPointer := PInteger(Integer(FSharePointer) + StreamHeaderSize
		+ FShareManager.Stream.Capacity);
	FShareList.Stream.SetVariables(FShareListPointer, -1);
	FShareList.MakeList;
	FShareManager.MakeList;
	PSynchronizeRead^ := False;
	Include(FPStatus^, msOpen);
	Exclude(FPStatus^, msPassiveResizing);
end;

procedure TPBCustomShare.LazyOpenShare;
begin
	PostMessage(FShareWindowHandle, FLazyOpenShareMessageID, 0, 0);
end;

procedure TPBCustomShare.Dummy(Value : string); begin end; // read-only

procedure TPBCustomShare.NewCreate(Sender : TObject);
begin
	if Assigned(OldOnCreate) then OldOnCreate(Owner);
	if FAutoOpenShare = omOnFormCreate then OpenShare;
end;

procedure TPBCustomShare.NewDestroy(Sender : TObject);
begin
	if FAutoCloseShare = cmOnFormDestroy then CloseShare;
	if Assigned(OldOnDestroy) then OldOnDestroy(Owner);
end;

function EnumShareWindows(HWndNext : HWnd; lParam : integer): Boolean; stdcall;
var
	Instance : TPBCustomShare;
	PTitel : array[0..256] of Char;
begin
	Instance := TPBCustomShare(lParam);
	Result := True;
	if HWndNext = 0 then Result := False
	else if HWndNext <> Instance.FShareWindowhandle then
	begin
		PTitel[0] := #0;
		GetClassName(HWndNext, PTitel, 255);
		if PTitel = WindowClassName then
		begin
			PTitel[0] := #0;
			GetWindowText(HWndNext, PTitel, 255);
			if (PTitel = Instance.FWindowName) or (PTitel = Instance.FWindowName
				+ OpenNameSuffix) then
			begin
				if Instance.FWindowList = nil then SetLength(Instance.FWindowList, 1)
				else SetLength(Instance.FWindowList, Length(Instance.FWindowList) + 1);
				Instance.FWindowList[High(Instance.FWindowList)] := HwndNext;
			end;
		end;
	end;
end;

procedure TPBCustomShare.RemoteControl(const Param : integer;
	const ToHandle : HWnd = 0; const Asynchron : Boolean = False;
	const AlsoSendToClosed : Boolean = False);
var
	t, Dummy : Cardinal;
	PTitel : array[0..256] of Char;
	TempAsynchron : Boolean;
begin
	if (ToHandle = FShareWindowhandle) then Exit;
	if InSendMessage then TempAsynchron := True
	else TempAsynchron := Asynchron;
	if (ToHandle <> 0) and IsWindow(ToHandle) then
	begin
		if TempAsynchron then PostMessage(ToHandle, FRemoteControlMessageID,
			Integer(FShareWindowhandle), Param)
		else SendmessageTimeOut(ToHandle, FRemoteControlMessageID,
			Integer(FShareWindowhandle), Param, SMTO_ABORTIFHUNG or SMTO_NORMAL,
			MessageTimeOut, Dummy);
	end
	else if AlsoSendToClosed or not IsOpen then
	begin
		FWindowList := nil;
		EnumWindows(@EnumShareWindows, Integer(Self));
		if FWindowList = nil then Exit
		else for t := 0 to High(FWindowList) do
		begin
			PTitel[0] := #0;
			GetWindowText(FWindowList[t], PTitel, 255);
			if (PTitel = FWindowName + OpenNameSuffix) or AlsoSendToClosed then
			begin
				if TempAsynchron then PostMessage(FWindowList[t], FRemoteControlMessageID,
					Integer(FShareWindowhandle), Param)
				else SendMessageTimeOut(FWindowList[t], FRemoteControlMessageID,
					Integer(FShareWindowhandle), Param, SMTO_ABORTIFHUNG or SMTO_NORMAL,
					MessageTimeOut, Dummy);
			end;
		end;
	end
	else NotifyAll(FRemoteControlMessageID, Param, Asynchron);
end;

procedure	TPBCustomShare.WndProc(var FMessage : TMessage);
begin
	with FMessage do
	begin
		if (Msg = FRemoteControlMessageID)
			and ([msClosing, msDestroying] * FPStatus^ = [])
			and Assigned(FOnRemoteControl)
			then FOnRemoteControl(Self, HWnd(WParam), LParam)
		else if IsOpen then
		begin
			if (Msg = FUpdateMessageID) and (ShareFunction <> sfWriter)
				and ([msClosing, msDestroying] * FPStatus^ = []) then
			begin
				Include(FPStatus^, msUpdating);
				FFromHandle := HWnd(WParam);
				FExtraInfo := LParam;
				if InSendMessage then PSynchronizeRead^ := True
				else PSynchronizeRead^ := False;
				LockShare(slRead);
				ReadAll;
				if Assigned(FOnUpdateAll)
					then FOnUpdateAll(Self, FFromHandle, FExtraInfo);
				PSynchronizeRead^ := False;
				UnLockShare(slRead);
				Exclude(FPStatus^, msUpdating);
			end
			else if (Msg = FShareListMessageID)
				and ([msClosing, msDestroying] * FPStatus^ = []) then
			begin
				PSynchronizeRead^ := True;
				LockShare(slRead);
				FFromHandle := HWnd(WParam);
				ReadShareList(LParam);
				PSynchronizeRead^ := False;
				UnLockShare(slRead);
			end
			else if (Msg = FRequestExtraInfoMessageID)
				and ([msClosing, msDestroying] * FPStatus^ = []) then
			begin
				Result := FExtraInfo;
				Exit;
			end
			else if (Msg = FPassiveResizeMessageID)
				then PassiveResizeShare(WParam, LParam)
			else if (Msg = WM_TIMER) and (WParam = TimerIdent)
				and ([msClosing, msDestroying] * FPStatus^ = [])
				then TimerTimer(Self)
			else if Msg = FRequestNameSuffixMessageID then
			begin
				Result := FNameSuffix;
				Exit;
			end
			else if Msg = FVersionMessageID then
			begin
				if WParam = VersionAsInteger then Result := 1
				else Result := 0;
				Exit;
			end;
		end
		else if (Msg = FPassiveResizeMessageID) and (WParam = 0)
			then PassiveResizeShare(0, LParam)
		else if (Msg = FLazyOpenShareMessageID) then OpenShare;
		Result := DefWindowProc(FShareWindowHandle, Msg, wParam, lParam);
	end;
end;

procedure TPBCustomShare.NotifyAll(const MessageID : Cardinal;
	const Extra : integer; const Asynchron : Boolean);
var
	t, HandleIndex, TempCount : integer;
	SendToShareInfos : array of TShareInfo;
	Dummy : Cardinal;
begin
	TempCount := FShareList.Count;
	SetLength(SendToShareInfos, TempCount);
	for t := 0 to TempCount - 1 do SendToShareInfos[t] := FShareList.ShareInfos[t];
	if (TempCount = 0) or ((TempCount = 1)
		and (SendToShareInfos[0].Handle = FShareWindowhandle)) then Exit;
	HandleIndex := ShareIndex;
	if HandleIndex = High(SendToShareInfos) then HandleIndex := -1;
	t := 0;
	while t <= TempCount - 1 do
	begin
		if (SendToShareInfos[t].Handle <> FShareWindowhandle)
			and IsWindow(SendToShareInfos[t].Handle) then
		begin
			if (MessageID = FPassiveResizeMessageID) then
			begin
				if (UpdateCount = 0) or (not HasClosedHandles)
					then SendMessageTimeout(SendToShareInfos[t].Handle,
					FPassiveResizeMessageID, UpdateCount, FNameSuffix,
					SMTO_ABORTIFHUNG or SMTO_NORMAL, MessageTimeOut, Dummy);
			end
			else if (MessageID = FShareListMessageId)
				or (MessageID = FRemoteControlMessageID)
				or (((FShareFunction <> sfUpdateChain) or (t = HandleIndex + 1))
				and SendToShareInfos[t].IsReader) then
			begin
				if InSendMessage or Asynchron then PostMessage(SendToShareInfos[t].Handle,
					MessageID, Integer(FShareWindowhandle), Extra)
				else SendMessageTimeout(SendToShareInfos[t].Handle, MessageID,
					Integer(FShareWindowhandle), Extra, SMTO_ABORTIFHUNG or SMTO_NORMAL,
					MessageTimeOut, Dummy);
			end;
		end;
		Inc(t);
	end;
	if (MessageID = FPassiveResizeMessageID) then FinishResizeShare;
end;

procedure TPBCustomShare.FinishResizeShare;
begin
	if UpdateCount = 0 then
	begin
		HasResized := False;
		HasClosedHandles := False;
	end
	else HasClosedHandles := True;
end;

procedure TPBCustomShare.SetAllocBy(Value : integer);
begin
	if (FPAllocBy^ <> Value) then
	begin
		if (Value < MinAllocBy) then FPAllocBy^ := MinAllocBy
		else if (Value > MaxAllocBy) then FPAllocBy^ := MaxAllocBy
		else FPAllocBy^ := Value;
	end;
end;

procedure TPBCustomShare.SetAutoCloseShare(Value : TAutoCloseShare);
var
	TempMethod : TMethod;
begin
	if (FAutoCloseShare <> Value)
		and ([csDesigning, csLoading] * ComponentState <> []) then
	begin
		FAutoCloseShare := Value;
		if OwnerIsForm and (not (csDesigning in ComponentState))
			and (Value = cmOnFormDestroy) then
		begin
			TempMethod.Data := Self;
			TempMethod.Code := @TPBCustomShare.NewDestroy;
			if GetEventProperty(Owner, 'OnDestroy', TMethod(OldOnDestroy))
				then SetEventProperty(Owner, 'OnDestroy', TempMethod, False);
		end;
	end
	else if (FAutoCloseShare <> Value)
		then Raise Exception.Create('Property AutoCloseShare must be set at designtime');
end;

procedure TPBCustomShare.SetAutoOpenShare(Value : TAutoOpenShare);
var
	TempMethod : TMethod;
begin
	if (FAutoOpenShare <> Value)
		and ([csDesigning, csLoading] * ComponentState <> []) then
	begin
		FAutoOpenShare := Value;
		if OwnerIsForm and (not (csDesigning in ComponentState))
			and (Value = omOnFormCreate) then
		begin
			TempMethod.Data := Self;
			TempMethod.Code := @TPBCustomShare.NewCreate;
			if GetEventProperty(Owner, 'OnCreate', TMethod(OldOnCreate))
				then SetEventProperty(Owner, 'OnCreate', TempMethod, False);
		end;
	end
	else if (FAutoOpenShare <> Value)
		then Raise Exception.Create('Property AutoOpenShare must be set at designtime');
end;

procedure TPBCustomShare.SetFlags(Value : TShareFlags);
var
	OldFlags : TShareFlags;
begin
	if FFlags <> Value then
	begin
		OldFlags := FFlags;
		FFlags := Value;
		if ((sfInterProcess in FFlags) <> (sfInterProcess in OldFlags)) then
		begin
			if (sfInterProcess in FFlags) then ProcessStr := ''
			else ProcessStr := IntToStr(GetCurrentProcessId);
			SetShareName(FShareName);
		end;
{		if (sfInterUser in FFlags) <> (sfInterUser in OldFlags) then
		begin
			if (sfInterUser in FFlags) and (Win32MajorVersion >= 5)
				and (Win32Platform = VER_PLATFORM_WIN32_NT)
				then FGlobalString := 'Global\'
			else FGlobalString := '';
		end;}
		PNeverResizeDown^ := (sfNeverResizeDown in FFlags);
	end;
end;

procedure TPBCustomShare.SetShareFunction(Value : TShareFunction);
var
	Index : integer;
begin
	if FShareFunction <> Value then
	begin
		FShareFunction := Value;
		if IsOpen and (FShareList <> nil) then
		begin
			LockShare(slWrite);
			Index := FShareList.IndexOfHandle(FShareWindowHandle);
			FShareList.ReadItem(FShareList, FShareList.DoRead, Index);
			FShareList.FShareInfo.IsReader := (FShareFunction <> sfWriter);
			FShareList.WriteItem(FShareList, FShareList.DoWrite, Index);
			UnLockShare(slWrite);
		end;
	end;
end;

procedure TPBCustomShare.SetShareName(Value : string);
begin
	if (FPStatus^ = []) then
	begin
		if (Length(Value) > MAX_PATH - 50)
			then Value := Copy(Value, 1, MAX_PATH - 50);
		FShareName := Value;
		FWindowName := FShareName + ProcessStr;
		if FShareWindowHandle <> 0
			then SetWindowText(FShareWindowHandle, PChar(FWindowName));
	end;
end;

procedure TPBCustomShare.TimerTimer(Sender : TObject);
var
	t : integer;
	ShareWinHandle : HWnd;
	WinText : array[0..1000] of Char;
	ShareInfos : array of TShareInfo;
begin
	if (not IsOpen)	or (FPStatus^ * ([msClosing, msUpdating] + msReading
		+ msWriting	+ msResizing) <> []) or (LockStatus > slRead) then Exit;
	SetLength(ShareInfos, FShareList.Count);
	for t := 0 to High(ShareInfos) do ShareInfos[t] := FShareList.ShareInfos[t];
	KillTimer(FShareWindowhandle, TimerIdent);
	for t := 0 to High(ShareInfos) do
	begin
		ShareWinHandle := ShareInfos[t].Handle;
		GetWindowText(ShareWinHandle, WinText, 1000);
		if (ShareWinHandle <> 0) and (ShareWinHandle <> FShareWindowHandle)
			and ((not IsWindow(ShareWinHandle)) or (Pos(OpenNameSuffix, WinText) = 0)) then
		begin
			while FShareList.Count - 2 < PFileMapPointer^^
				do InterLockedDecrement(PFileMapPointer^^);
			PSynchronizeRead^ := False;
			FShareList.Delete(t);
			Break;
		end;
	end;
	if (msOpen in FPStatus^) and (sfAutoRepairList in FFlags)
		then SetTimer(FShareWindowhandle, TimerIdent, TimerInterval, nil);
end;

function TPBCustomShare.GetTotalShareSize : integer;
begin
	Result := SizeOf(Integer) + 2 * StreamHeaderSize + FPAllocBy^
		+ FPShareListAllocBy^;
end;

function TPBCustomShare.OpenShare : Boolean;
var
	PExePath, PModuleName : array[0..MAX_PATH] of Char;
	OtherHandle : HWnd;
	VersionResult : Cardinal;
begin
	Result := False;
	if (FShareHandle = 0) and (PFileMapPointer^ = nil)
		and ([msOpen, msOpening, msPassiveResizing] * FPStatus^ = []) then
	begin
		Include(FPStatus^, msOpening);
		PReadEvent^ := CreateEvent(@SecurityAttr, True, False, PChar(FGlobalString
			+ FWindowName + 'ReadEvent'));
		if PReadEvent^ = 0 then RaiseLastOSError;
		PWriteMutex^ := CreateMutex(@SecurityAttr, False, PChar(FGlobalString
			+ FWindowName + 'WriteMtx'));
		if PWriteMutex^ = 0 then RaiseLastOSError;
		if (GetLastError = ERROR_ALREADY_EXISTS)
			then Exclude(FPStatus^, msFirstShare)
		else Include(FPStatus^, msFirstShare);
		WaitForSingleObject(PWriteMutex^, INFINITE);
		if not (msFirstShare in FPStatus^) then
		begin
			OtherHandle := FindWindow(PChar(WindowClassName), PChar(FWindowName
				+ OpenNameSuffix));
			SendMessageTimeout(OtherHandle, FVersionMessageID, VersionAsInteger, 0,
				SMTO_ABORTIFHUNG or SMTO_NORMAL, MessageTimeOut, VersionResult);
			if VersionResult <> 1
				then Raise EShareException.Create('PBShare version conflict');
			SendMessageTimeOut(OtherHandle, FRequestNameSuffixMessageID, 0, 0,
				SMTO_ABORTIFHUNG or SMTO_NORMAL, MessageTimeOut, Cardinal(FNameSuffix));
		end
		else
		begin
			SetEvent(PReadEvent^);
			OtherHandle := 0;
		end;
		WaitForSingleObject(PReadEvent^, INFINITE);
		FShareHandle := CreateFileMapping($FFFFFFFF, @SecurityAttr, PAGE_READWRITE
			or SEC_COMMIT or SEC_NOCACHE, 0, GetTotalShareSize,
			PChar(FGlobalString + FWindowName + GetNameSuffix));
		if (FShareHandle = INVALID_HANDLE_VALUE) or (FShareHandle = 0)
			then RaiseLastOSError;
		PFileMapPointer^ := MapViewOfFile(FShareHandle, FILE_MAP_ALL_ACCESS, 0, 0, 0);
		if PFileMapPointer^ = nil then RaiseLastOSError;
		if msFirstShare in FPStatus^
			then ZeroMemory(PFileMapPointer^, GetTotalShareSize);
		LockShare(slWrite);
		ReleaseMutex(PWriteMutex^);
		FSharePointer := PInteger(Integer(PFileMapPointer^) + SizeOf(Integer));
		FTempStream := TMemoryStream.Create;
		FShareManager := FShareManagerClass.Create(FSharePointer, FPAllocBy,
			PWriteMutex, PReadEvent, PFileMapPointer, PWriteLockCount, PReadLockCount,
			PSynchronizeRead, PNeverResizeDown, FPStatus, FTempStream, ResizeShare,
			DataChange);
		FShareListPointer := PInteger(Integer(FSharePointer) + StreamHeaderSize
			+ FShareManager.Stream.Capacity);
		FShareList := TPBShareInfoManager.Create(FShareListPointer, FPShareListAllocBy,
			PWriteMutex, PReadEvent, PFileMapPointer, PWriteLockCount, PReadLockCount,
			PSynchronizeRead, PNeverResizeDown, FPStatus, FTempStream, ResizeShare,
			ShareListChange);
		if not (msFirstShare in FPStatus^) then ReadShareList(-1);
		FFromHandle := FShareWindowhandle;
		with FShareList.FShareInfo do
		begin
			Handle := FShareWindowhandle;
			ProcessID := GetCurrentProcessId;
			ThreadID := GetCurrentThreadId;
			GetModuleFileName(0, PExePath, SizeOf(PExePath));
			ExeName := PExePath;
			GetModuleFileName(HInstance, PModuleName, SizeOf(PModuleName));
			ModuleName := PModuleName;
			Version := Self.Version;
			IsDLL := IsLibrary;
			IsReader := (FShareFunction <> sfWriter);
		end;
		Include(FPStatus^, msOpen);
		FShareList.Add(FShareList, FShareList.DoWrite);
		if (msFirstShare in FPStatus^) then WriteAll
		else LockShare(slRead);
		UnLockShare(slWrite);
		if not (msFirstShare in FPStatus^) then
		begin
			SendMessageTimeout(OtherHandle, FRequestExtraInfoMessageID, 0, 0,
				SMTO_ABORTIFHUNG or SMTO_NORMAL, MessageTimeOut, Cardinal(FExtraInfo));
			ReadAll;
			UnLockShare(slRead);
		end;
		if Assigned(FOnOpen) then FOnOpen(Self, (msFirstShare in FPStatus^));
		if (not (msFirstShare in FPStatus^)) and (sfUpdateOnOpen in FFlags)
			and Assigned(FOnUpdateAll) then
		begin
			Include(FPStatus^, msUpdating);
			FOnUpdateAll(Self, FFromHandle, FExtraInfo);
			Exclude(FPStatus^, msUpdating);
		end;
		SetWindowText(FShareWindowHandle, PChar(FWindowName + OpenNameSuffix));
		if (sfAutoRepairList in FFlags) then SetTimer(FShareWindowhandle, TimerIdent,
			TimerInterval, nil);
		Exclude(FPStatus^, msOpening);
		Result := True;
	end;
end;

function TPBCustomShare.CloseShare : Boolean;
begin
	Result := False;
	if csDestroying in ComponentState then Include(FPStatus^, msDestroying);
	KillTimer(FShareWindowhandle, TimerIdent);
	if msOpen in FPStatus^ then
	begin
		Include(FPStatus^, msClosing);
		SetWindowText(FShareWindowHandle, PChar(FWindowName));
		if Assigned(FOnClosing) then FOnClosing(Self, (FShareList.Count = 1));
		FFromHandle := FShareWindowHandle;
		Exclude(FPStatus^, msOpen);
		LockShare(slWrite);
		FShareList.Delete(FShareList.IndexOfHandle(FShareWindowHandle));
		UnLockShare(slWrite);
		FShareManager.Free;
		FShareManager := nil;
		FShareList.Free;
		FShareList := nil;
		FTempStream.Free;
		FTempStream := nil;
		UnMapViewOfFile(PFileMapPointer^);
		PFileMapPointer^ := nil;
		FSharePointer := nil;
		FShareListPointer := nil;
		CloseHandle(FShareHandle);
		FShareHandle := 0;
		CloseHandle(PReadEvent^);
		PReadEvent^ := 0;
		ReleaseMutex(PWriteMutex^);
		CloseHandle(PWriteMutex^);
		PWriteMutex^ := 0;
		FPStatus^ := FPStatus^ * [msDestroying];
		Result := True;
	end;
end;

procedure TPBCustomShare.DataChange(Sender : TObject; const Index : integer;
	const Resized : Boolean; const ItemOperation : TMultiOperation);
begin
	FFromHandle := FShareWindowHandle;
	FExtraInfo := Index;
	if Resized then NotifyAll(FPassiveResizeMessageID, 0, False);
	if (FShareList.Count = 1) or (UpdateCount > 0) then Exit;
	if ItemOperation = moWriteAll then NotifyAll(FUpdateMessageId, FExtraInfo,
		(sfAsynchronUpdate in FFlags));
end;

procedure TPBCustomShare.ShareListChange(Sender : TObject; const Index : integer;
	const Resized : Boolean; const ItemOperation : TMultiOperation);
var
	TempOperation : TMultiOperation;
begin
	if (ItemOperation = moDelete) and not (msClosing in FPStatus^)
		then TempOperation := moRepair
	else TempOperation := ItemOperation;
	FFromHandle := FShareWindowHandle;
	FShareListExtra := Index + Ord(TempOperation) * (MAXINT div 16);
	if Resized then NotifyAll(FPassiveResizeMessageID, 0, False);
	if not (csDestroying in ComponentState) and Assigned(FOnUpdateShareList) then
	begin
		Include(FPStatus^, msUpdating);
		FOnUpdateShareList(Self, FFromHandle, Index, TempOperation);
		Exclude(FPStatus^, msUpdating);
	end;
 	NotifyAll(FShareListMessageId, FShareListExtra,	False);
end;

procedure TPBCustomShare.ReadShareList(LParam : integer);
var
	Index : integer;
	ItemOperation : TMultiOperation;
begin
	if ([msOpen, msOpening] * FPStatus^ <> []) then
	begin
		Include(FPStatus^, msUpdating);
		Include(FPStatus^, msReadingShareList);
		LockShare(slRead);
		FShareList.MakeList;
		UnLockShare(slRead);
		if (msOpen in FPStatus^) and (LParam > -1) then
		begin
			Index := LParam mod (MAXINT div 16);
			ItemOperation := TMultiOperation(LParam div (MAXINT div 16));
			if Assigned(FOnUpdateShareList)
				then FOnUpdateShareList(Self, FFromHandle, Index, ItemOperation);
		end;
		Exclude(FPStatus^, msReadingShareList);
		Exclude(FPStatus^, msUpdating);
	end;
end;

procedure TPBCustomShare.ReadAll;
begin
	if (ShareFunction = sfWriter) then Exit;
	if ([msOpen, msOpening] * FPStatus^ <> []) then
	begin
		Include(FPStatus^, msReadingData);
		FShareManager.ReadAll(Self, FOnDoReadAll, FFromHandle, FExtraInfo);
		Exclude(FPStatus^, msReadingData);
	end
	else Raise EShareException.Create('PBShare must be open when calling ''ReadAll''');
end;

procedure TPBCustomShare.WriteAll;
begin
	if (ShareFunction = sfReader) then Exit;
	if ([msOpen, msOpening] * FPStatus^ <> []) then
	begin
		Include(FPStatus^, msWritingData);
		FShareManager.WriteAll(Self, FOnDoWriteAll, FExtraInfo);
		Exclude(FPStatus^, msWritingData);
	end
	else Raise EShareException.Create('PBShare must be open when calling ''WriteAll''');
end;

procedure TPBCustomShare.WriteSize(const NumberOfStrings, TotalSize : integer);
begin
	if (ShareFunction = sfReader) then Exit;
	if ([msOpen, msOpening] * FPStatus^ <> [])
		then FShareManager.WriteSize(NumberOfStrings * SizeOf(Integer) + TotalSize)
	else Raise EShareException.Create('PBShare must be open when calling ''WriteSize''');
end;

procedure TPBCustomShare.LoadFromFile(FileName : string);
var
	FileStream : TFileStream;
begin
	FileStream := TFileStream.Create(FileName, fmOpenRead or fmShareDenyWrite);
	LoadFromStream(FileStream);
	FileStream.Free;
end;

procedure TPBCustomShare.LoadFromStream(Stream : TStream);
begin
	FShareManager.LoadFromStream(Stream);
	ReadAll;
end;

procedure TPBCustomShare.SaveToFile(FileName : string);
var
	FileStream : TFileStream;
begin
	FileStream := TFileStream.Create(FileName, fmCreate or fmShareExclusive);
	SaveToStream(FileStream);
	FileStream.Free;
end;

procedure TPBCustomShare.SaveToStream(Stream : TStream);
begin
	FShareManager.SaveToStream(Stream);
end;

//  -------------------  TPBShareSingle  --------------------------
constructor TPBShareSingle.CreateNoOwner(AShareName : string; AAllocBy : integer;
	AFlags : TShareFlags; AOpenNow : Boolean; AShareFunction : TShareFunction;
	AOnClosing : TClosingShareEvent; AOnOpen : TOpenShareEvent;
	AOnUpdateAll : TUpdateAllEvent;	AOnDoReadAll : TDoReadAllEvent;
	AOnDoWriteAll : TDoWriteAllEvent; AOnRemoteControl : TRemoteControlEvent);
begin
	inherited CreateNoOwner(AShareName, AAllocBy, AFlags,
		AShareFunction, AOnClosing, AOnOpen, AOnUpdateAll, AOnRemoteControl);
	FOnDoReadAll := AOnDoReadAll;
	FOnDoWriteAll := AOnDoWriteAll;
	if AOpenNow then LazyOpenShare;
end;

constructor TPBShareSingle.CreateNoEvents(AShareName : string; AAllocBy : integer;
	AFlags : TShareFlags; AOpenNow : Boolean; AShareFunction : TShareFunction;
	AOnClosing : TClosingShareProc; AOnOpen : TOpenShareProc;
	AOnUpdateAll : TUpdateAllProc; AOnDoReadAll : TDoReadAllProc;
	AOnDoWriteAll : TDoWriteAllProc; AOnRemoteControl : TRemoteControlProc);
begin
	inherited CreateNoEvents(AShareName, AAllocBy, AFlags,
		AShareFunction, AOnClosing, AOnOpen, AOnUpdateAll, AOnRemoteControl);
	FOnDoReadAll := TDoReadAllEvent(MakeMethod(@AOnDoReadAll));
	FOnDoWriteAll := TDoWriteAllEvent(MakeMethod(@AOnDoWriteAll));
	if AOpenNow then LazyOpenShare;
end;

procedure TPBShareSingle.ReadAll;
begin
	inherited;
end;

procedure TPBShareSingle.WriteAll;
begin
	inherited;
end;

procedure TPBShareSingle.WriteSize(const NumberOfStrings, TotalSize : integer);
begin
	inherited;
end;

//  ----------------------  initialization  -----------------------
initialization
	FRemoteControlMessageID := RegisterWindowMessage('PBShare-RemoteControl-Message');
	FRequestNameSuffixMessageID := RegisterWindowMessage('PBShare-NameSuffix-Message');
	FUpdateMessageID := RegisterWindowMessage('PBShare-UpdateAll-Message');
	FRequestExtraInfoMessageID := RegisterWindowMessage('PBShare-ExtraInfo-Message');
	FPassiveResizeMessageID := RegisterWindowMessage('PBShare-PassiveResize-Message');
	FShareListMessageID := RegisterWindowMessage('PBShare-ShareList-Message');
	FLazyOpenShareMessageID := RegisterWindowMessage('PBShare-LazyOpen-Message');
	FVersionMessageID := RegisterWindowMessage('PBShare-Version-Message');
	Win32Check(InitializeSecurityDescriptor(@SecurityDesc,
		SECURITY_DESCRIPTOR_REVISION));
	Win32Check(SetSecurityDescriptorDacl(@SecurityDesc, True, nil, False));
	SecurityAttr.nLength := SizeOf(TSecurityAttributes);
	SecurityAttr.lpSecurityDescriptor := @SecurityDesc;
	SecurityAttr.bInheritHandle := False;

end.


