{*****************************************************************}
{                                                                 }
{ ModLink                                                         }
{ Copyright (C) 2002 - 2003 Ing. Ivo Bauer. All Rights Reserved.  }
{                                                                 }
{ http://www.ozm.cz/ivobauer/modlink/                             }
{ ivo.bauer@tiscali.cz (bauer@ozm.cz)                             }
{                                                                 }
{ For a detailed information regarding the distribution and use   }
{ of this software product, please refer to the License Agreement }
{ embedded in the accompanying online documentation (ModLink.chm) }
{                                                                 }
{*****************************************************************}

unit ModLinkDemoMain;

{$I ModLink.inc}

interface

//--------------------------------------------------------------------------------------------------

uses
  { Windows } Windows, Messages,
  { Delphi  } SysUtils, {$IFDEF COMPILER_6_UP} Variants , {$ENDIF COMPILER_6_UP} Classes, Graphics,
              Controls, Forms, Dialogs, Menus, ComCtrls, StdCtrls, ExtCtrls,
  { ModLink } ModLink;

//--------------------------------------------------------------------------------------------------

type
  TModLinkDemoMainForm = class(TForm)
    MainMenu1: TMainMenu;
    FileMenu: TMenuItem;
    ViewMenu: TMenuItem;
    ToolsMenu: TMenuItem;
    HelpMenu: TMenuItem;
    FileExitItem: TMenuItem;
    NativeItem: TMenuItem;
    SmallIntItem: TMenuItem;
    FloatingPointItem: TMenuItem;
    N1: TMenuItem;
    DecPlaces0Item: TMenuItem;
    DecPlaces1Item: TMenuItem;
    DecPlaces2Item: TMenuItem;
    DecPlaces3Item: TMenuItem;
    DecPlaces4Item: TMenuItem;
    ConnectionOptionsItem: TMenuItem;
    ClientOptionsItem: TMenuItem;
    HelpAboutItem: TMenuItem;
    ModbusConnection1: TModbusConnection;
    ModbusClient1: TModbusClient;
    StatusBar1: TStatusBar;
    PageControl1: TPageControl;
    LogMemo: TMemo;
    IntroductionTabSheet: TTabSheet;
    DiscreteAccessTabSheet: TTabSheet;
    RegisterAccessTabSheet: TTabSheet;
    DiagnosticsTabSheet: TTabSheet;
    Panel1: TPanel;
    DiscreteAccessPageControl: TPageControl;
    ReadDiscretesTabSheet: TTabSheet;
    WriteDiscretesTabSheet: TTabSheet;
    Label3: TLabel;
    StartBitEdit: TEdit;
    Label2: TLabel;
    BitCountEdit: TEdit;
    ReadCoilsButton: TButton;
    ReadDiscreteInputsButton: TButton;
    DiscreteListView: TListView;
    WriteSingleCoilButton: TButton;
    WriteMultipleCoilsButton: TButton;
    FileRecordAccessTabSheet: TTabSheet;
    RegisterAccessPageControl: TPageControl;
    ReadRegistersTabSheet: TTabSheet;
    Label4: TLabel;
    Label5: TLabel;
    StartRegEdit: TEdit;
    RegCountEdit: TEdit;
    ReadHoldingRegistersButton: TButton;
    ReadInputRegistersButton: TButton;
    WriteRegistersTabSheet: TTabSheet;
    RegisterListView: TListView;
    WriteSingleRegisterButton: TButton;
    WriteMultipleRegistersButton: TButton;
    MaskWriteRegisterTabSheet: TTabSheet;
    Label6: TLabel;
    AndMaskEdit: TEdit;
    OrMaskEdit: TEdit;
    Label8: TLabel;
    MaskWriteSingleRegisterButton: TButton;
    DiagnosticActionRadioGroup: TRadioGroup;
    DiagnosticsButton: TButton;
    N2: TMenuItem;
    ClearLogItem: TMenuItem;
    IntroductionMemo: TMemo;
    FileRecordAccessMemo: TMemo;
    DiscreteBroadcastCheckBox: TCheckBox;
    RegisterBroadcastCheckBox: TCheckBox;
    procedure FileExitItemClick(Sender: TObject);
    procedure StorageFormatChange(Sender: TObject);
    procedure DecimalPlacesChange(Sender: TObject);
    procedure ConnectionOptionsItemClick(Sender: TObject);
    procedure ClientOptionsItemClick(Sender: TObject);
    procedure HelpAboutItemClick(Sender: TObject);
    procedure DiscreteAccessPageControlChanging(Sender: TObject;
      var AllowChange: Boolean);
    procedure DiscreteAccessPageControlChange(Sender: TObject);
    procedure ReadCoilsButtonClick(Sender: TObject);
    procedure ReadDiscreteInputsButtonClick(Sender: TObject);
    procedure WriteSingleCoilButtonClick(Sender: TObject);
    procedure WriteMultipleCoilsButtonClick(Sender: TObject);
    procedure ModbusClient1CoilsRead(Sender: TModbusClient;
      const Info: TTransactionInfo; BitStart, BitCount: Word;
      const BitValues: TBitValues);
    procedure ModbusClient1DiscreteInputsRead(Sender: TModbusClient;
      const Info: TTransactionInfo; BitStart, BitCount: Word;
      const BitValues: TBitValues);
    procedure ModbusClient1SingleCoilWrite(Sender: TModbusClient;
      const Info: TTransactionInfo; BitAddr: Word; BitValue: Boolean);
    procedure ModbusClient1MultipleCoilsWrite(Sender: TModbusClient;
      const Info: TTransactionInfo; BitStart, BitCount: Word;
      const BitValues: TBitValues);
    procedure ModbusConnection1FrameSend(Sender: TModbusConnection;
      const Data: TFrameData);
    procedure ModbusConnection1FrameReceive(Sender: TModbusConnection;
      const Data: TFrameData);
    procedure RegisterAccessPageControlChanging(Sender: TObject;
      var AllowChange: Boolean);
    procedure RegisterAccessPageControlChange(Sender: TObject);
    procedure ReadHoldingRegistersButtonClick(Sender: TObject);
    procedure ReadInputRegistersButtonClick(Sender: TObject);
    procedure WriteSingleRegisterButtonClick(Sender: TObject);
    procedure WriteMultipleRegistersButtonClick(Sender: TObject);
    procedure RegisterListViewKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure RegisterListViewDblClick(Sender: TObject);
    procedure ModbusClient1HoldingRegistersRead(Sender: TModbusClient;
      const Info: TTransactionInfo; StartReg, RegCount: Word;
      const RegValues: TRegValues);
    procedure ModbusClient1InputRegistersRead(Sender: TModbusClient;
      const Info: TTransactionInfo; StartReg, RegCount: Word;
      const RegValues: TRegValues);
    procedure ModbusClient1SingleRegisterWrite(Sender: TModbusClient;
      const Info: TTransactionInfo; RegAddr, RegValue: Word);
    procedure ModbusClient1MultipleRegistersWrite(Sender: TModbusClient;
      const Info: TTransactionInfo; StartReg, RegCount: Word;
      const RegValues: TRegValues);
    procedure MaskWriteSingleRegisterButtonClick(Sender: TObject);
    procedure ModbusClient1SingleRegisterMaskWrite(Sender: TModbusClient;
      const Info: TTransactionInfo; RegAddr, AndMask, OrMask: Word);
    procedure DiagnosticsButtonClick(Sender: TObject);
    procedure ModbusClient1Diagnostics(Sender: TModbusClient;
      const Info: TTransactionInfo; Action: TDiagnosticAction;
      Result: Word);
    procedure ClearLogItemClick(Sender: TObject);
  private
    procedure LogBroadcast;
    procedure LogDone(ID: Cardinal; const CmdDesc: string);
    procedure LogFrame(const Data: TFrameData; Send: Boolean);
    procedure LogInit(ID: Cardinal; const CmdDesc: string);
    procedure LogProcessedBits(BitCount: Word; Coils: Boolean);
    procedure LogProcessedRegs(RegCount: Word; Holding: Boolean);
    procedure LogSingleBit(BitIndex: Word; BitValue: Boolean; Coil: Boolean);
    procedure LogSingleRegister(RegIndex: Word; RegValue: Word; Holding: Boolean);
    procedure LogStatus(const Info: TTransactionInfo);
    procedure StorageFormatChanged;
    procedure UpdateConnectionStatus;
    procedure ValidateMaskWriteRegisterTabSheet;
    procedure ValidateReadDiscretesTabSheet;
    procedure ValidateReadRegistersTabSheet;
    function ValidateRegisterListViewItem(Item: TListItem): Word;
  public
    constructor Create(AOwner: TComponent); override;
  end;

//--------------------------------------------------------------------------------------------------

var
  ModLinkDemoMainForm: TModLinkDemoMainForm;

//--------------------------------------------------------------------------------------------------

implementation

//--------------------------------------------------------------------------------------------------

uses SysConst, Consts, ModbusConnectionEditor, ModLinkAboutBox;

//--------------------------------------------------------------------------------------------------

{$R *.dfm}

//--------------------------------------------------------------------------------------------------

type
  TStorageFormat = (sfNative, sfSmallInt, sfFloat);

//--------------------------------------------------------------------------------------------------

var
  StorageFormat: TStorageFormat = sfNative;
  DecimalPlaces: Integer = 0;

//--------------------------------------------------------------------------------------------------

const

  ServerReplies: array [TServerReply] of string = (
    'Transaction failed (no reply was received)',
    'Transaction failed (unmatched reply was received)',
    'Transaction failed (exception reply was received)',
    'Transaction succeeded'
  );

  ServerExceptions: array [TServerException] of string = (
    'Unknown exception code',
    'Illegal command',
    'Illegal data address',
    'Illegal data value',
    'Server failure',
    'Acknowledge',
    'Server busy',
    'Negative acknowledge',
    'Memory parity error'
  );

//--------------------------------------------------------------------------------------------------

procedure ValidateNumberInEditBox(AEdit: TEdit; MinValue, MaxValue: Int64);

  // begin of local block --------------------------------------------------------------------------

  procedure Error(ResStringRec: PResStringRec; const Args: array of const);
  begin
    {$IFDEF COMPILER_5_UP}
    raise EModLinkError.CreateResFmt(ResStringRec, Args);
    {$ELSE}
    raise EModLinkError.CreateFmt(LoadResString(ResStringRec), Args);
    {$ENDIF COMPILER_5_UP}
  end;

  procedure InvalidInteger;
  begin
    if AEdit.CanFocus then AEdit.SetFocus;
    Error(@SInvalidInteger, [AEdit.Text]);
  end;

  procedure InvalidRange;
  begin
    if AEdit.CanFocus then AEdit.SetFocus;
    Error(@SOutOfRange, [MinValue, MaxValue]);
  end;

  // end of local block ----------------------------------------------------------------------------

var
  I: Int64;
begin
  try
    I := StrToInt64(AEdit.Text);
    if (MinValue <> MaxValue) and ((I < MinValue) or (I > MaxValue)) then
      InvalidRange;
  except
    on E: Exception do
      if E is EConvertError then
      begin
        InvalidInteger;
      end
      else
        raise;
  end;
end;

//--------------------------------------------------------------------------------------------------
// TModLinkDemoMainForm class
//--------------------------------------------------------------------------------------------------

constructor TModLinkDemoMainForm.Create(AOwner: TComponent);
begin
  inherited;

  try
    ModbusConnection1.Open;
  except
    {$IFDEF COMPILER6_UP}
    if Assigned(ApplicationHandleException) then
      ApplicationHandleException(Self);
    {$ELSE}
    Application.HandleException(Self);
    {$ENDIF}
  end;

  Application.HintHidePause := 20000;

  PageControl1.ActivePage := IntroductionTabSheet;
  DiscreteAccessPageControl.ActivePage := ReadDiscretesTabSheet;
  RegisterAccessPageControl.ActivePage := ReadRegistersTabSheet;
  RegisterListView.Column[0].Index := 1;

  LogMemo.Lines.Add(Format('Running on ModLink version %s'#13#10, [ModLinkVersion]));

  StorageFormatChanged;
  UpdateConnectionStatus;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.LogBroadcast;
begin
  LogMemo.Lines.Add('Broadcast query: No reply is expected to be returned from the server(s).');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.LogDone(ID: Cardinal; const CmdDesc: string);
begin
  LogMemo.Lines.Add(Format('DONE: %s [ID: %d]', [CmdDesc, ID]));
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.LogFrame(const Data: TFrameData; Send: Boolean);
var
  S: string;
  I: Integer;
begin
  if Send then
    S := 'TX: '
  else
    S := 'RX: ';

  if Length(Data) = 0 then
    S := S + '<empty>'
  else
    for I := 0 to High(Data) do
      S := S + IntToHex(Data[I], 2) + ' ';

  LogMemo.Lines.Add(S);
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.LogInit(ID: Cardinal; const CmdDesc: string);
begin
  LogMemo.Lines.Add(Format('INIT: %s [ID: %d]', [CmdDesc, ID]));
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.LogProcessedBits(BitCount: Word; Coils: Boolean);
var
  Temp: string;
begin
  if BitCount > 1 then
    if Coils then
      Temp := 'coils were'
    else
      Temp := 'discrete inputs were'
  else
    if Coils then
      Temp := 'coil was'
    else
      Temp := 'discrete input was';

  LogMemo.Lines.Add(Format('%d %s processed.', [BitCount, Temp]));
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.LogProcessedRegs(RegCount: Word; Holding: Boolean);
var
  Temp: string;
begin
  if RegCount > 1 then
    if Holding then
      Temp := 'holding registers were'
    else
      Temp := 'input registers were'
  else
    if Holding then
      Temp := 'holding register was'
    else
      Temp := 'input register was';

  LogMemo.Lines.Add(Format('%d %s processed.', [RegCount, Temp]));
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.LogSingleBit(BitIndex: Word; BitValue: Boolean; Coil: Boolean);
const
  BitStates: array [Boolean] of string = ('OFF', 'ON');
var
  Temp: string;
begin
  if Coil then
    Temp := 'Coil'
  else
    Temp := 'Discrete input';

  LogMemo.Lines.Add(Format('%s %d is %s.', [Temp, BitIndex, BitStates[BitValue]]));
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.LogSingleRegister(RegIndex: Word; RegValue: Word; Holding: Boolean);
const
  RegNames: array [Boolean] of string = ('input', 'holding');
begin
  case StorageFormat of
    sfNative:
      LogMemo.Lines.Add(Format('Value of %s register %d is %d',
        [RegNames[Holding], RegIndex, RegValue]));
    sfSmallInt:
      LogMemo.Lines.Add(Format('Value of %s register %d is %d',
        [RegNames[Holding], RegIndex, Mod2Int(RegValue)]));
    sfFloat:
      LogMemo.Lines.Add(Format('Value of %s register %d is %.*f',
        [RegNames[Holding], RegIndex, DecimalPlaces, Mod2Float(RegValue, DecimalPlaces)]));
  end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.LogStatus(const Info: TTransactionInfo);
begin
  LogMemo.Lines.Add(ServerReplies[Info.Reply]);
  if Info.Reply = srExceptionReply then
    LogMemo.Lines.Add('Server exception: ' + ServerExceptions[Info.Exception]);
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.StorageFormatChanged;
begin
  DecPlaces0Item.Enabled := StorageFormat = sfFloat;
  DecPlaces1Item.Enabled := StorageFormat = sfFloat;
  DecPlaces2Item.Enabled := StorageFormat = sfFloat;
  DecPlaces3Item.Enabled := StorageFormat = sfFloat;
  DecPlaces4Item.Enabled := StorageFormat = sfFloat;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.UpdateConnectionStatus;
begin
  if ModbusConnection1.Active then
    StatusBar1.Panels[0].Text := Format('Connected to %s', [ModbusConnection1.Port])
  else
    StatusBar1.Panels[0].Text := 'No active connection';
  StatusBar1.Panels[1].Text := Format('Server address: %d', [ModbusClient1.ServerAddress]);
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ValidateMaskWriteRegisterTabSheet;
begin
  ValidateNumberInEditBox(AndMaskEdit, 0, High(Word));
  ValidateNumberInEditBox(OrMaskEdit, 0, High(Word));
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ValidateReadDiscretesTabSheet;
begin
  ValidateNumberInEditBox(StartBitEdit, 0, High(Word));
  ValidateNumberInEditBox(BitCountEdit, 1, High(Word));
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ValidateReadRegistersTabSheet;
begin
  ValidateNumberInEditBox(StartRegEdit, 0, High(Word));
  ValidateNumberInEditBox(RegCountEdit, 1, High(Word));
end;

function TModLinkDemoMainForm.ValidateRegisterListViewItem(Item: TListItem): Word;
begin
  try
    case StorageFormat of
      sfNative:
        Result := Word(StrToInt(Item.Caption));
      sfSmallInt:
        Result := Int2Mod(SmallInt(StrToInt(Item.Caption)));
      sfFloat:
        Result := Float2Mod(StrToFloat(Item.Caption), DecimalPlaces);
    else
      raise Exception.Create('Internal error.');
    end;
  except
    on E: EConvertError do
    begin
      E.Message := Format('You''ve entered an invalid value for ''%s''', [Item.SubItems[0]]);
      raise;
    end;
  else
    raise;
  end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.FileExitItemClick(Sender: TObject);
begin
  Close;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.StorageFormatChange(Sender: TObject);
begin
  with Sender as TMenuItem do
  begin
    StorageFormat := TStorageFormat(Tag);
    Checked := True;
  end;
  StorageFormatChanged;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.DecimalPlacesChange(Sender: TObject);
begin
  with Sender as TMenuItem do
  begin
    Checked := True;
    DecimalPlaces := Tag;
  end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ConnectionOptionsItemClick(Sender: TObject);
begin
  try
    if EditModbusConnection(ModbusConnection1, 'Modbus Connection Options') then
      ModbusConnection1.Open;
  finally
    UpdateConnectionStatus;
  end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ClientOptionsItemClick(Sender: TObject);
const
  SCaptionFmt = 'Modbus Client Options';
  SPrompt = 'Enter the address of a remote server ' +
    '(acceptable values are 1 through 247):';
var
  S: string;
begin
  S := IntToStr(ModbusClient1.ServerAddress);
  if InputQuery(SCaptionFmt, SPrompt, S) then
  begin
    try
      ModbusClient1.ServerAddress := StrToInt(S);
      UpdateConnectionStatus;
    except
      on E: EConvertError do
      begin
        E.Message := Format('''%s'' is not a valid server address.', [S]);
        raise;
      end
      else raise;
    end;
  end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.HelpAboutItemClick(Sender: TObject);
begin
  ShowModLinkAboutBox;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.DiscreteAccessPageControlChanging(Sender: TObject;
  var AllowChange: Boolean);
begin
  try
    if DiscreteAccessPageControl.ActivePage = ReadDiscretesTabSheet then
      ValidateReadDiscretesTabSheet;
  except
    AllowChange := False;
  end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.DiscreteAccessPageControlChange(Sender: TObject);
var
  Start, I: Integer;
begin
  if DiscreteAccessPageControl.ActivePage = ReadDiscretesTabSheet then Exit;

  Start := StrToInt(StartBitEdit.Text);
  I := StrToInt(BitCountEdit.Text);
  WriteSingleCoilButton.Enabled := I = 1;

  with DiscreteListView do
  begin
    Items.BeginUpdate;
    try
      Items.Clear;
      Items.Count := I;
      for I := 0 to I - 1 do
        Items.Add.Caption := Format('Coil %d', [I + Start]);
    finally
      Items.EndUpdate;
    end;
  end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ReadCoilsButtonClick(Sender: TObject);
var
  StartBit, BitCount: Word;
  ID: Cardinal;
begin
  ValidateReadDiscretesTabSheet;
  StartBit := Word(StrToInt(StartBitEdit.Text));
  BitCount := Word(StrToInt(BitCountEdit.Text));
  ID := ModbusClient1.ReadCoils(StartBit, BitCount);
  LogInit(ID, 'Read Coils (code $01)');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ReadDiscreteInputsButtonClick(Sender: TObject);
var
  StartBit, BitCount: Word;
  ID: Cardinal;
begin
  ValidateReadDiscretesTabSheet;
  StartBit := Word(StrToInt(StartBitEdit.Text));
  BitCount := Word(StrToInt(BitCountEdit.Text));
  ID := ModbusClient1.ReadDiscreteInputs(StartBit, BitCount);
  LogInit(ID, 'Read Discrete Inputs (code $02)');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.WriteSingleCoilButtonClick(Sender: TObject);
var
  BitAddr: Word;
  BitValue: Boolean;
  ID: Cardinal;
begin
  ValidateReadDiscretesTabSheet;
  BitAddr := Word(StrToInt(StartBitEdit.Text));
  BitValue := DiscreteListView.Items[0].Checked;
  if DiscreteBroadcastCheckBox.Checked then
  begin
    ID := ModbusConnection1.WriteSingleCoil(BitAddr, BitValue);
    LogInit(ID, 'Write Single Coil (code $05)');
    LogBroadcast;
  end
  else
  begin
    ID := ModbusClient1.WriteSingleCoil(BitAddr, BitValue);
    LogInit(ID, 'Write Single Coil (code $05)');
  end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.WriteMultipleCoilsButtonClick(Sender: TObject);
var
  StartBit, BitCount: Word;
  BitValues: TBitValues;
  I: Integer;
  ID: Cardinal;
begin
  ValidateReadDiscretesTabSheet;
  StartBit := Word(StrToInt(StartBitEdit.Text));
  BitCount := Word(StrToInt(BitCountEdit.Text));
  SetLength(BitValues, BitCount);
  try
    for I := 0 to BitCount - 1 do
      BitValues[I] := DiscreteListView.Items[I].Checked;

    if DiscreteBroadcastCheckBox.Checked then
    begin
      ID := ModbusConnection1.WriteMultipleCoils(StartBit, BitValues);
      LogInit(ID, 'Write Multiple Coils (code $0F)');
      LogBroadcast;
    end
    else
    begin
      ID := ModbusClient1.WriteMultipleCoils(StartBit, BitValues);
      LogInit(ID, 'Write Multiple Coils (code $0F)');
    end;
  finally
    Finalize(BitValues);
  end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ModbusClient1CoilsRead(Sender: TModbusClient;
  const Info: TTransactionInfo; BitStart, BitCount: Word;
  const BitValues: TBitValues);
var
  I: Integer;
begin
  LogDone(Info.ID, 'Read Coils (code $01)');
  LogStatus(Info);
  if Info.Reply = srNormalReply then
  begin
    LogProcessedBits(BitCount, True);
    for I := 0 to BitCount - 1 do
      LogSingleBit(BitStart + I, BitValues[I], True);
  end;
  LogMemo.Lines.Add('');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ModbusClient1DiscreteInputsRead(
  Sender: TModbusClient; const Info: TTransactionInfo; BitStart,
  BitCount: Word; const BitValues: TBitValues);
var
  I: Integer;
begin
  LogDone(Info.ID, 'Read Discrete Inputs (code $02)');
  LogStatus(Info);
  if Info.Reply = srNormalReply then
  begin
    LogProcessedBits(BitCount, False);
    for I := 0 to BitCount - 1 do
      LogSingleBit(BitStart + I, BitValues[I], False);
  end;
  LogMemo.Lines.Add('');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ModbusClient1SingleCoilWrite(Sender: TModbusClient;
  const Info: TTransactionInfo; BitAddr: Word; BitValue: Boolean);
begin
  LogDone(Info.ID, 'Write Single Coil (code $05)');
  LogStatus(Info);
  if Info.Reply = srNormalReply then
  begin
    LogProcessedBits(1, True);
    LogSingleBit(BitAddr, BitValue, True);
  end;
  LogMemo.Lines.Add('');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ModbusClient1MultipleCoilsWrite(
  Sender: TModbusClient; const Info: TTransactionInfo; BitStart,
  BitCount: Word; const BitValues: TBitValues);
var
  I: Integer;
begin
  LogDone(Info.ID, 'Write Multiple Coils (code $0F)');
  LogStatus(Info);
  if Info.Reply = srNormalReply then
  begin
    LogProcessedBits(BitCount, True);
    for I := 0 to BitCount - 1 do
      LogSingleBit(BitStart + I, BitValues[I], True);
  end;
  LogMemo.Lines.Add('');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ModbusConnection1FrameSend(
  Sender: TModbusConnection; const Data: TFrameData);
begin
  LogFrame(Data, True);
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ModbusConnection1FrameReceive(
  Sender: TModbusConnection; const Data: TFrameData);
begin
  LogFrame(Data, False);
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.RegisterAccessPageControlChanging(Sender: TObject;
  var AllowChange: Boolean);
begin
  try
    if RegisterAccessPageControl.ActivePage = ReadRegistersTabSheet then
      ValidateReadRegistersTabSheet;
  except
    AllowChange := False;
  end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.RegisterAccessPageControlChange(Sender: TObject);
var
  Start, I: Integer;
begin
  if RegisterAccessPageControl.ActivePage = ReadRegistersTabSheet then Exit;

  Start := StrToInt(StartRegEdit.Text);
  I := StrToInt(RegCountEdit.Text);

  WriteSingleRegisterButton.Enabled := I = 1;
  MaskWriteSingleRegisterButton.Enabled := I = 1;

  with RegisterListView do
  begin
    Items.BeginUpdate;
    try
      Items.Clear;
      Items.Count := I;
      for I := 0 to I - 1 do
        with Items.Add do
        begin
          Caption := '0';
          SubItems.Add(Format('Register %d', [I + Start]));
        end;
    finally
      Items.EndUpdate;
    end;
  end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ReadHoldingRegistersButtonClick(Sender: TObject);
var
  StartReg, RegCount: Word;
  ID: Cardinal;
begin
  ValidateReadRegistersTabSheet;
  StartReg := Word(StrToInt(StartRegEdit.Text));
  RegCount := Word(StrToInt(RegCountEdit.Text));
  ID := ModbusClient1.ReadHoldingRegisters(StartReg, RegCount);
  LogInit(ID, 'Read Holding Registers (code $03)');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ReadInputRegistersButtonClick(Sender: TObject);
var
  StartReg, RegCount: Word;
  ID: Cardinal;
begin
  ValidateReadRegistersTabSheet;
  StartReg := Word(StrToInt(StartRegEdit.Text));
  RegCount := Word(StrToInt(RegCountEdit.Text));
  ID := ModbusClient1.ReadInputRegisters(StartReg, RegCount);
  LogInit(ID, 'Read Input Registers (code $04)');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.WriteSingleRegisterButtonClick(Sender: TObject);
var
  RegAddr, RegValue: Word;
  ID: Cardinal;
begin
  RegAddr := Word(StrToInt(StartRegEdit.Text));
  RegValue := ValidateRegisterListViewItem(RegisterListView.Items[0]);
  if RegisterBroadcastCheckBox.Checked then
  begin
    ID := ModbusConnection1.WriteSingleRegister(RegAddr, RegValue);
    LogInit(ID, 'Write Single Register (code $06)');
    LogBroadcast;
  end
  else
  begin
    ID := ModbusClient1.WriteSingleRegister(RegAddr, RegValue);
    LogInit(ID, 'Write Single Register (code $06)');
  end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.WriteMultipleRegistersButtonClick(Sender: TObject);
var
  StartReg, RegCount: Word;
  RegValues: TRegValues;
  I: Integer;
  ID: Cardinal;
begin
  StartReg := Word(StrToInt(StartRegEdit.Text));
  RegCount := Word(StrToInt(RegCountEdit.Text));
  SetLength(RegValues, RegCount);
  try
    for I := 0 to RegCount - 1 do
      RegValues[I] := ValidateRegisterListViewItem(RegisterListView.Items[I]);
    if RegisterBroadcastCheckBox.Checked then
    begin
      ID := ModbusConnection1.WriteMultipleRegisters(StartReg, RegValues);
      LogInit(ID, 'Write Multiple Registers (code $10)');
      LogBroadcast;
    end
    else
    begin
      ID := ModbusClient1.WriteMultipleRegisters(StartReg, RegValues);
      LogInit(ID, 'Write Multiple Registers (code $10)');
    end;
  finally
    Finalize(RegValues);
  end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.RegisterListViewKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  with Sender as TListView do
    if (Key = VK_RETURN) and (not IsEditing) and (Selected <> nil) then
    begin
      Selected.EditCaption;
      Key := 0;
    end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.RegisterListViewDblClick(Sender: TObject);
var
  P: TPoint;
  Item: TListItem;
begin
  P := Mouse.CursorPos;
  with Sender as TListView do
  begin
    P := ScreenToClient(P);
    Item := GetItemAt(P.X, P.Y);
    if Assigned(Item) then Item.EditCaption;
  end;
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ModbusClient1HoldingRegistersRead(
  Sender: TModbusClient; const Info: TTransactionInfo; StartReg,
  RegCount: Word; const RegValues: TRegValues);
var
  I: Integer;
begin
  LogDone(Info.ID, 'Read Holding Registers (code $03)');
  LogStatus(Info);
  if Info.Reply = srNormalReply then
  begin
    LogProcessedRegs(RegCount, True);
    for I := 0 to RegCount - 1 do
      LogSingleRegister(StartReg + I, RegValues[I], True);
  end;
  LogMemo.Lines.Add('');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ModbusClient1InputRegistersRead(
  Sender: TModbusClient; const Info: TTransactionInfo; StartReg,
  RegCount: Word; const RegValues: TRegValues);
var
  I: Integer;
begin
  LogDone(Info.ID, 'Read Input Registers (code $04)');
  LogStatus(Info);
  if Info.Reply = srNormalReply then
  begin
    LogProcessedRegs(RegCount, False);
    for I := 0 to RegCount - 1 do
      LogSingleRegister(StartReg + I, RegValues[I], False);
  end;
  LogMemo.Lines.Add('');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ModbusClient1SingleRegisterWrite(
  Sender: TModbusClient; const Info: TTransactionInfo; RegAddr,
  RegValue: Word);
begin
  LogDone(Info.ID, 'Write Single Register (code $06)');
  LogStatus(Info);
  if Info.Reply = srNormalReply then
  begin
    LogProcessedRegs(1, True);
    LogSingleRegister(RegAddr, RegValue, True);
  end;
  LogMemo.Lines.Add('');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ModbusClient1MultipleRegistersWrite(
  Sender: TModbusClient; const Info: TTransactionInfo; StartReg,
  RegCount: Word; const RegValues: TRegValues);
var
  I: Integer;
begin
  LogDone(Info.ID, 'Write Multiple Registers (code $10)');
  LogStatus(Info);
  if Info.Reply = srNormalReply then
  begin
    LogProcessedRegs(RegCount, True);
    for I := 0 to RegCount - 1 do
      LogSingleRegister(StartReg + I, RegValues[I], True);
  end;
  LogMemo.Lines.Add('');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.MaskWriteSingleRegisterButtonClick(Sender: TObject);
var
  RegAddr, AndMask, OrMask: Word;
  ID: Cardinal;
begin
  ValidateMaskWriteRegisterTabSheet;
  RegAddr := Word(StrToInt(StartRegEdit.Text));
  AndMask := Word(StrToInt(AndMaskEdit.Text));
  OrMask := Word(StrToInt(OrMaskEdit.Text));
  ID := ModbusClient1.MaskWriteSingleRegister(RegAddr, AndMask, OrMask);
  LogInit(ID, 'Mask Write Register (code $16)');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ModbusClient1SingleRegisterMaskWrite(
  Sender: TModbusClient; const Info: TTransactionInfo; RegAddr, AndMask,
  OrMask: Word);
begin
  LogDone(Info.ID, 'Mask Write Register (code $16)');
  LogStatus(Info);
  if Info.Reply = srNormalReply then
  begin
    LogProcessedRegs(1, True);
    LogMemo.Lines.Add(Format('Register: %d | AND mask: %d | OR mask: %d',
      [RegAddr, AndMask, OrMask]));
  end;
  LogMemo.Lines.Add('');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.DiagnosticsButtonClick(Sender: TObject);
var
  Action: TDiagnosticAction;
  ID: Cardinal;
begin
  Action := TDiagnosticAction(DiagnosticActionRadioGroup.ItemIndex);
  ID := ModbusClient1.Diagnostics(Action);
  LogInit(ID, 'Diagnostics (code $08)');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ModbusClient1Diagnostics(Sender: TModbusClient;
  const Info: TTransactionInfo; Action: TDiagnosticAction; Result: Word);
begin
  LogDone(Info.ID, 'Diagnostics (code $08)');
  LogStatus(Info);
  if Info.Reply = srNormalReply then
  begin
    case Action of
      daReturnQueryData:
        LogMemo.Lines.Add('"Return Query Data" action was performed by the server. It appears to be responding.');
      daRestartCommsOption:
        LogMemo.Lines.Add('"Restart Comms Option" action was performed by the server.');
      daRestartCommsOptionAndClearEventLog:
        LogMemo.Lines.Add('"Restart Comms Option And Clear Event Log" action was performed by the server.');
      daReturnDiagnosticRegister:
        LogMemo.Lines.Add(Format('"Return Diagnostic Register" action returned the value %d.', [Result]));
      daForceListenOnlyMode:
        LogMemo.Lines.Add('"Force Listen Only Mode" action was performed by the server. It should never go here.');
      daClearCountersAndDiagnosticRegister:
        LogMemo.Lines.Add('"Clear Counters And Diagnostic Register" action was performed by the server.');
      daReturnBusMessageCount:
        LogMemo.Lines.Add(Format('"Return Bus Message Count" action returned the value %d.', [Result]));
      daReturnBusCommErrorCount:
        LogMemo.Lines.Add(Format('"Return Bus Comm Error Count" action returned the value %d.', [Result]));
      daReturnBusExceptionErrorCount:
        LogMemo.Lines.Add(Format('"Return Bus Exception Error Count" action returned the value %d.', [Result]));
      daReturnServerMessageCount:
        LogMemo.Lines.Add(Format('"Return Server Message Count" action returned the value %d.', [Result]));
      daReturnServerNoReplyCount:
        LogMemo.Lines.Add(Format('"Return Server No Reply Count" action returned the value %d.', [Result]));
      daReturnServerNegativeAcknowledgeCount:
        LogMemo.Lines.Add(Format('"Return Server Negative Acknowledge Count" action returned the value %d.', [Result]));
      daReturnServerBusyCount:
        LogMemo.Lines.Add(Format('"Return Server Busy Count" action returned the value %d.', [Result]));
      daReturnBusCharacterOverrunCount:
        LogMemo.Lines.Add(Format('"Return Bus Character Overrun Count" action returned the value %d.', [Result]));
      daClearOverrunCounterAndFlag:
        LogMemo.Lines.Add('"Clear Overrun Counter And Flag" action was performed by the server.');
    end;
  end;
  LogMemo.Lines.Add('');
end;

//--------------------------------------------------------------------------------------------------

procedure TModLinkDemoMainForm.ClearLogItemClick(Sender: TObject);
begin
  LogMemo.Clear;
end;

//--------------------------------------------------------------------------------------------------

end.
 