{ ****************************************************************
  Info               :  TAnalogMeter2000X
                        Freeware

  Source File Name   :  X2000AnalogMeter.PAS
  Author             :  Baldemaier Florian (Baldemaier.Florian@gmx.net)
  Original           :  Hannes Breuer (hannes.breuer@talknet.de) 
  Compiler           :  Delphi 5.0 Professional
  Decription         :  AnalogMeter Gauge

**************************************************************** }
unit X2000AnalogMeter;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, X2000AboutInfo;

type
  TAnalogMeter2000X = class(TPaintBox)
  private
    fAbout  : TAboutInfo2000X;
    fBitmap : TBitMap;
    fZeroAngle     : Single;
    fCentrePoint   : TPoint;
    fOuterHalfAxes,
    fInnerHalfAxes : TPoint;
    fFrameWidth : LongInt;
    fTxtHeight     : LongInt;
    fOldValue : LongInt;
    //events:
    FonResize : TNotifyEvent;
    FonChange : TNotifyEvent;
    FonRiseAboveHigh : TNotifyEvent;
    FonFallBelowHigh : TNotifyEvent;
    FonRiseAboveLow : TNotifyEvent;
    FonFallBelowLow : TNotifyEvent;
    fEnableZoneEvents : Boolean;
    //variables for properties:
    fCaption : String;
    fMin, fMax, fValue : LongInt; 
    fAngularRange : LongInt;
    fTickCount : LongInt;         
    fLowZone,
    fHighZone : Single;           
    fShowFrame,
    fShowTicks,
    fShowValue : Boolean;
    fTickColor,
    fHighZoneColor,
    fLowZoneColor,
    fOKZoneColor : TColor;
    Procedure SetCaption(s : String);
    Procedure SetMin(m : LongInt);
    Procedure SetMax(m : LongInt);
    Procedure SetValue(v : LongInt);
    Procedure SetAngularRange(r : LongInt);
    Procedure SetTickCount(t : LongInt);
    Procedure SetLowZone(percent : Byte);
    Function GetLowZone : Byte;
    Procedure SetHighZone(percent : Byte);
    Function GetHighZone : Byte;
    Procedure SetShowValue(b : Boolean);
    Procedure SetHighZoneColor(c : TColor);
    Procedure SetLowZoneColor(c : TColor);
    Procedure SetOKZoneColor(c : TColor);
    Procedure SetTickColor(c : TColor);
    Procedure SetShowTicks(b : Boolean);
    Procedure SetShowFrame(b : Boolean);
    Procedure SetFont(F : TFont);
    Function GetFont : TFont;
    Procedure WMSize(var Message : TWMSize); message 15;
    Function AngleOf(v : Single) : Single; 
    Procedure OptimizeSizes; 
    Procedure ClearCanvas;
    Procedure DrawFrame;
    Procedure DrawRegions;
    Procedure DrawTicks;
    Procedure DrawCaption;
    Procedure DrawPointer;
    Procedure DrawValue;
    Procedure DrawFace;
  protected
    Procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property About: TAboutInfo2000X read FAbout write FAbout Stored False;
    property Caption : String read fCaption write SetCaption;
    property Min : LongInt read fMin write SetMin default 0;
    property Max : LongInt read fMax write SetMax default 100;
    property Value : LongInt read fValue write SetValue default 0;
    property AngularRange : LongInt read fAngularRange write SetAngularRange default 270;
    property TickCount : LongInt read fTickCount write SetTickCount default 7;
    property LowZone : Byte read GetLowZone write SetLowZone default 10;
    property HighZone : Byte read GetHighZone write SetHighZone default 90;
    property ShowValue : Boolean read fShowValue write SetShowValue default True;
    property HighZoneColor : TColor read fHighZoneColor write SetHighZoneColor default clRed;
    property LowZoneColor  : TColor read fLowZoneColor write SetLowZoneColor default clYellow;
    property OKZoneColor   : TColor read fOKZoneColor write SetOKZoneColor default cl3DLight;
    property TickColor : TColor read fTickColor write SetTickColor default clBlack;
    property ShowTicks : Boolean read fShowTicks write SetShowTicks default True;
    property ShowFrame : Boolean read fShowFrame write SetShowFrame default True;
    property onResize : TNotifyEvent read fOnResize write FonResize;
    property onChange : TNotifyEvent read FonChange write FonChange;
    property EnableZoneEvents : Boolean read fEnableZoneEvents write fEnableZoneEvents default True;
    property onRiseAboveHigh : TNotifyEvent
        read FonRiseAboveHigh
       write FonRiseAboveHigh;
    property onFallBelowHigh : TNotifyEvent
        read FonFallBelowHigh
       write FonFallBelowHigh;
    property onRiseAboveLow : TNotifyEvent
        read FonRiseAboveLow
       write FonRiseAboveLow;
    property onFallBelowLow : TNotifyEvent
        read FonFallBelowLow
       write FonFallBelowLow;
    property Align;
    property Color;
    property Font : TFont read GetFont write SetFont;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  end;

implementation

Function Radius(Theta : Single; HalfAxes : TPoint) : Single;
Var C, S : Single;  
    w, h : LongInt; 
Begin
  C := Cos(Theta*Pi/180);
  S := Sin(Theta*Pi/180);
  w := HalfAxes.X;
  h := HalfAxes.Y;
  If (w <= 0) or (h <= 0) then begin
    Result := 0;
    Exit;
  end;
  Result := w*h/Sqrt(h*h*S*S + w*w*C*C);
End;

Function XinEllipse(Theta : Single; HalfAxes : TPoint) : LongInt;
Var R : Single;
Begin
  R := Radius(Theta,Halfaxes);
  Result := Round(-R*Sin(Theta*Pi/180)); 
End;

Function YinEllipse(Theta : Single; HalfAxes : TPoint) : LongInt;
Var R : Single;
Begin
  R := Radius(Theta,Halfaxes);
  Result := Round(R*Cos(Theta*Pi/180));
End;

Function OptimalHalfHeight(AvailableHeight : LongInt; Theta : Single; Width : LongInt) : LongInt;
Var UsedHeight : Single;
    H : LongInt;
    C, S, T : Single; 
Begin
  If Theta >= 90 then begin 
    Result := AvailableHeight;
    Exit;
  end;
  S := Sin(Theta*Pi/180);
  S := Sqr(S);  
  C := Cos(Theta*Pi/180);
  C := Width*C/2;
  T := Sqr(C);
  H := AvailableHeight div 2;
  UsedHeight := H + H*C / Sqrt(Sqr(H)*S + T); 
  While UsedHeight < AvailableHeight do begin
    Inc(H);
    UsedHeight := H + H*C / Sqrt(Sqr(H)*S + T);
  end;
  Dec(H); 
  Result := H;
End;

Constructor TAnalogMeter2000X.Create(AOwner: TComponent);
Begin
  inherited Create(AOwner);
  fBitmap := TBitmap.Create;
  Width  := 121;
  Height := 117;
  fCaption := 'mV';
  fMin   := 0;
  fMax   := 100;
  fValue := 0;
  fOldValue := fValue;
  fAngularRange  := 270;
  fTickCount     := 11;
  fLowZone   := 10;
  fHighZone  := 90;
  fShowValue := True;
  //internal stuff:
  fZeroAngle := 45;
  fTxtHeight := 8;
  fHighZoneColor := clRed;
  fLowZoneColor  := clYellow;
  fOKZoneColor   := cl3DLight;
  fTickColor     := clBlack;
  fShowTicks     := True;
  fShowFrame     := True;
  fEnableZoneEvents := True;
End;

Destructor TAnalogMeter2000X.Destroy;
Begin
  fBitmap.Free;
  inherited Destroy;
End;

Procedure TAnalogMeter2000X.WMSize(var Message : TWMSize);
Begin
  If Height < 37 then Height := 37; 
  If Width  < 52 then Width  := 52;
  If Assigned(FonResize) then FonResize(Self);
  OptimizeSizes;
  fBitmap.Width  := Width;
  fBitmap.Height := Height;
  DrawFace;
  inherited;
End;

Function TAnalogMeter2000X.GetFont : TFont;
Begin
  Result := fBitmap.Canvas.Font;
End;

Procedure TAnalogMeter2000X.SetFont(f : TFont);
Begin
  If fBitmap.Canvas.Font = f then Exit;
  fBitmap.Canvas.Font := f;
  fTxtHeight := fBitmap.Canvas.TextHeight('S');
  If fShowValue or (Trim(fCaption) <> '') then begin
    DrawFace;
    Paint;
  end;
End;

Procedure TAnalogMeter2000X.SetCaption(s : String);
Begin
  s := Trim(s);
  If fCaption = s then Exit;
  fCaption := s;
  DrawFace;
  Paint;
End;

Procedure TAnalogMeter2000X.SetMin(m : LongInt);
Var OldLow, OldHigh : Longint;
Begin
  If m > fMax then m := fMax; 
  If fMin = m then Exit;      
  OldLow := GetLowZone;       
  OldHigh := GetHighZone;
  fMin := m;
  If fValue < m then Value := m;
  SetLowZone(OldLow);           
  SetHighZone(OldHigh);
End;

Procedure TAnalogMeter2000X.SetMax(m : LongInt);
Var OldLow, OldHigh : Longint;
Begin
  If m < fMin then m := fMin;
  If fMax = m then Exit;     
  OldLow := GetLowZone;      
  OldHigh := GetHighZone;
  fMax := m;
  If fValue > m then Value := m;
  SetLowZone(OldLow);           
  SetHighZone(OldHigh);
End;

Procedure TAnalogMeter2000X.SetValue(v : LongInt);
Var TriggerHi, TriggerLo : Boolean;
Begin
  If v < fMin then v := fMin;
  If v > fMax then v := fMax;
  If v = fValue then Exit;
  fOldValue := fValue;
  fValue := v;
  If Assigned(FonChange) then FonChange(Self);
  //redraw:
  DrawFace;
  Paint; 
  //call events if necessary:
  If not fEnableZoneEvents then Exit;
  TriggerHi := Assigned(FonRiseAboveHigh) and (fOldValue < fHighZone) and (fValue >= fHighZone);
  TriggerLo := Assigned(FonFallBelowLow) and (fOldValue >= fLowZone) and (fValue < fLowZone);
  If TriggerLo then FonFallBelowLow(Self);
  If TriggerHi then FonRiseAboveHigh(Self);
  If not (TriggerHi or TriggerLo) then begin
    If Assigned(FonFallBelowHigh) and (fOldValue >= fHighZone) and (fValue < fHighZone)
      then FonFallBelowHigh(Self);
    If Assigned(FonRiseAboveLow) and (fOldValue < fLowZone) and (fValue >= fLowZone)
      then FonRiseAboveLow(Self);
  end;
End;

Procedure TAnalogMeter2000X.SetAngularRange(r : LongInt);
Begin
  If r < 10 then r := 10;  
  If r > 360 then r := 360;
  If r = fAngularRange then Exit;
  fAngularRange := r;
  fZeroAngle := (360-AngularRange)/2;
  OptimizeSizes;
  DrawFace;
  Paint;
End;

Procedure TAnalogMeter2000X.SetTickCount(t : LongInt);
Begin
  If t < 0 then t := 0;          
  If t > (fMax-fMin) then t := Fmax - fMin;
  If t = fTickCount then Exit;   
  fTickCount := t;
  DrawFace;
  Paint;
End;

Procedure TAnalogMeter2000X.SetLowZone(percent : Byte);
Var fOldLow : Single;
Begin
  If percent < 0   then percent := 0; 
  If percent > 100 then percent := 100;
  If percent = LowZone then Exit; 
  If percent > HighZone then HighZone := percent; 
  fOldLow := fLowZone;
  fLowZone := Min + (Max-Min)*percent/100;
  DrawFace;
  Paint;
  If not fEnableZoneEvents then Exit;
  If Assigned(FonFallBelowLow) and (fValue >= fOldLow) and (fValue < fLowZone)
    then FonFallBelowLow(Self);
  If Assigned(FonRiseAboveLow) and (fValue < fOldLow) and (fValue >= fLowZone)
    then FonRiseAboveLow(Self);
End;

Function TAnalogMeter2000X.GetLowZone : Byte;
Begin
  If Max = Min
    then Result := 0
    else Result := Round(100*(fLowZone - Min)/(Max - Min));
End;

Procedure TAnalogMeter2000X.SetHighZone(percent : Byte);
Var fOldHigh : Single;
Begin
  If percent < 0   then percent := 0;  
  If percent > 100 then percent := 100;
  If percent = HighZone then Exit; 
  If percent < LowZone then LowZone := percent;
  fOldHigh := fHighZone;
  fHighZone := Min + (Max-Min)*percent/100;
  DrawFace;
  Paint;
  //call events if necessary:
  If not fEnableZoneEvents then Exit;
  If Assigned(FonFallBelowHigh) and (fValue >= fOldHigh) and (fValue < fHighZone)
    then FonFallBelowHigh(Self);
  If Assigned(FonRiseAboveHigh) and (fValue < fOldHigh) and (fValue >= fHighZone)
    then FonRiseAboveHigh(Self);
End;

Function TAnalogMeter2000X.GetHighZone : Byte;
Begin
  If Max = Min
    then Result := 0
    else Result := Round(100*(fHighZone - Min)/(Max - Min));
End;

Procedure TAnalogMeter2000X.SetShowValue(b : Boolean);
Begin
  If b = fShowValue then Exit;
  fShowValue := b;
  DrawFace;
  Paint;
End;

Procedure TAnalogMeter2000X.SetHighZoneColor(c : TColor);
Begin
  If fHighZoneColor = c then Exit;
  fHighZoneColor := c;
  DrawFace;
  Paint;
End;

Procedure TAnalogMeter2000X.SetLowZoneColor(c : TColor);
Begin
  If fLowZoneColor = c then Exit;
  fLowZoneColor := c;
  DrawFace;
  Paint;
End;

Procedure TAnalogMeter2000X.SetOKZoneColor(c : TColor);
Begin
  If fOKZoneColor = c then Exit;
  fOKZoneColor := c;
  DrawFace;
  Paint;
End;

Procedure TAnalogMeter2000X.SetTickColor(c : TColor);
Begin
  If fTickColor = c then Exit;
  fTickColor := c;
  DrawFace;
  Paint;
End;

Procedure TAnalogMeter2000X.SetShowTicks(b : Boolean);
Begin
  If fShowTicks = b then Exit;
  fShowTicks := b;
  DrawFace;
  Paint;
End;

Procedure TAnalogMeter2000X.SetShowFrame(b : Boolean);
Begin
  If fShowFrame = b then Exit;
  fShowFrame := b;
  DrawFace;
  Paint;
End;

Function TAnalogMeter2000X.AngleOf(v : Single) : Single;
Begin
  Result := fZeroAngle + fAngularRange*(v - fMin)/(fMax-fMin);
End;

Procedure TAnalogMeter2000X.OptimizeSizes;
var TH : LongInt; 
Begin
  TH := fTxtHeight + 2;
  fFrameWidth := (Width + Height) div 40;
  fOuterHalfAxes.Y := OptimalHalfHeight(Height - 2*fFrameWidth - TH, fZeroAngle, Width - 2*fFrameWidth);
  fCentrePoint.Y := fFrameWidth + fOuterHalfAxes.Y;
  fCentrePoint.X := Width Div 2;
  fOuterHalfAxes.X := fCentrePoint.X-fFrameWidth;
  fInnerHalfAxes := Point(fOuterHalfAxes.X - ((Width+Height)div 25),
                          fOuterHalfAxes.Y - ((Width+Height)div 25));
End;

Procedure TAnalogMeter2000X.ClearCanvas;
Begin
  with fBitmap.Canvas do begin
    Brush.Style := bsClear;
    Brush.Color := Self.Color;
    FillRect(ClientRect);
  end;
End;

Procedure TAnalogMeter2000X.DrawFrame;
Begin
  with fBitmap.Canvas do begin
    Pen.Color := clBtnShadow;
    Pen.Width := 1;
    MoveTo(0,Height);
    LineTo(0,0);
    LineTo(Width-1,0);
    Pen.Color := clBtnHighLight;
    LineTo(Width-1,Height-1);
    LineTo(0,Height-1);
  end;
End;

Procedure TAnalogMeter2000X.DrawRegions;
Var Xzero, Yzero,
    Xlow, Ylow,
    XHi, YHi : LongInt;
    LowAngle, HiAngle : Single;
Begin
  LowAngle := AngleOf(fLowZone);
  HiAngle  := AngleOf(fHighZone);
  Xzero := XInEllipse(fZeroAngle,fOuterHalfAxes);
  Yzero := YInEllipse(fZeroAngle,fOuterHalfAxes);
  XLow  := XInEllipse(LowAngle,fOuterHalfAxes);
  YLow  := YInEllipse(LowAngle,fOuterHalfAxes);
  XHi   := XInEllipse(HiAngle,fOuterHalfAxes);
  YHi   := YInEllipse(HiAngle,fOuterHalfAxes);
  with fBitmap.Canvas do begin
    Pen.Width := 2;
    If Color = clBtnShadow then Pen.Color := cl3DLight
                           else Pen.Color := clBtnShadow;
    If fLowZone > fMin then begin
      Brush.Color := fLowZoneColor;
      Pie(fCentrePoint.X-fOuterHalfAxes.X, fCentrePoint.Y-fOuterHalfAxes.Y,
          fCentrePoint.X+fOuterHalfAxes.X, fCentrePoint.Y+fOuterHalfAxes.Y,
          fCentrePoint.X + XLow, fCentrePoint.Y + YLow,
          fCentrePoint.X + Xzero, fCentrePoint.Y + Yzero);
    end;
    If (fHighZone > fLowZone) then begin
      Brush.Color := fOKZoneColor;
      Pie(fCentrePoint.X-fOuterHalfAxes.X, fCentrePoint.Y-fOuterHalfAxes.Y,
          fCentrePoint.X+fOuterHalfAxes.X, fCentrePoint.Y+fOuterHalfAxes.Y,
          fCentrePoint.X + XHi, fCentrePoint.Y + YHi,
          fCentrePoint.X + XLow, fCentrePoint.Y + YLow);
    end;
    If (fHighZone < fMax) then begin
      Brush.Color := fHighZoneColor;
      Pie(fCentrePoint.X-fOuterHalfAxes.X, fCentrePoint.Y-fOuterHalfAxes.Y,
          fCentrePoint.X+fOuterHalfAxes.X, fCentrePoint.Y+fOuterHalfAxes.Y,
          fCentrePoint.X - Xzero, fCentrePoint.Y + Yzero,
          fCentrePoint.X + XHi, fCentrePoint.Y + YHi);
    end;
    Pen.Width := 2;
    If Color = clBtnShadow then Pen.Color := cl3DLight
                           else Pen.Color := clBtnShadow;
    Arc(fCentrePoint.X-fInnerHalfAxes.X, fCentrePoint.Y-fInnerHalfAxes.Y,
        fCentrePoint.X+fInnerHalfAxes.X, fCentrePoint.Y+fInnerHalfAxes.Y,
        fCentrePoint.X - Xzero, fCentrePoint.Y + Yzero,
        fCentrePoint.X + Xzero, fCentrePoint.Y + Yzero);
    Pen.Color := Color; 
    Brush.Color := Color;
    Ellipse(fCentrePoint.X-fInnerHalfAxes.X+2, fCentrePoint.Y-fInnerHalfAxes.Y+2,
        fCentrePoint.X+fInnerHalfAxes.X-2, fCentrePoint.Y+fInnerHalfAxes.Y-2);
  end;
End;

Procedure TAnalogMeter2000X.DrawTicks;
var i : LongInt;
    SmallEllipse : TPoint;
    Angle : Single;
Begin
  SmallEllipse.X := fInnerHalfAxes.X-((Width+Height) div 35);
  SmallEllipse.Y := fInnerHalfAxes.Y-((Width+Height) div 35);
  with fBitmap.Canvas do begin
    Pen.Color := fTickColor;
    Pen.Width := 1;
    For i := 1 to fTickCount do begin
      Angle := AngleOf(Min + (Max-Min)*(i-1)/(fTickCount-1));
      MoveTo(fCentrePoint.X + XInEllipse(Angle,fInnerHalfAxes),
             fCentrePoint.Y + YInEllipse(Angle,fInnerHalfAxes));
      LineTo(fCentrePoint.X + XInEllipse(Angle,SmallEllipse),
             fCentrePoint.Y + YInEllipse(Angle,SmallEllipse));
    end;
  end;
End;

Procedure TAnalogMeter2000X.DrawCaption;
var TW: LongInt; 
    H, H1, H2 : LongInt;
Begin
  H1 := fCentrePoint.Y - ((Width + Height) div 40);
  H2 := fCentrePoint.Y - fInnerHalfAxes.Y - 1;     
  H  := (H1 + H2) div 2;
  with fBitmap.Canvas do begin
    Brush.Color := Color; 
    TW := TextWidth(fCaption) div 2;
    TextOut(fCentrePoint.X - TW, H, fCaption);
  end;
End;

Procedure TAnalogMeter2000X.DrawPointer;
var angle : Single;
    X,Y : LongInt;
    SmallEllipse : TPoint;
    Radius : LongInt;
Begin
  with fBitmap.Canvas do begin
    SmallEllipse.X := fInnerHalfAxes.X-3;
    SmallEllipse.Y := fInnerHalfAxes.Y-3;
    angle := AngleOf(fValue);
    X := XInEllipse(Angle,SmallEllipse);
    Y := YInEllipse(Angle,SmallEllipse);
    Pen.Width := 2;
    Pen.Color := fTickColor;
    MoveTo(fCentrePoint.X, fCentrePoint.Y);
    LineTo(fCentrePoint.X + X, fCentrePoint.Y + Y);
    Radius := (Width + Height) div 40;
    Pen.Width := 3;
    Pen.Color := clBtnShadow;
    Ellipse(fCentrePoint.X - Radius+1, fCentrePoint.Y - Radius,
            fCentrePoint.X + Radius+1, fCentrePoint.Y + Radius);
    Pen.Width := 1;
    If (fLowZone > fMin) and (fValue <= fLowZone)
      then Brush.Color := fLowZoneColor
      else If (fHighZone < fMax) and (fValue >= fHighZone)
        then Brush.Color := fHighzoneColor
        else Brush.Color := fOKzoneColor;
    Pen.Color := Brush.Color;
    Ellipse(fCentrePoint.X - Radius, fCentrePoint.Y - Radius,
            fCentrePoint.X + Radius, fCentrePoint.Y + Radius);
  end;
End;

Procedure TAnalogMeter2000X.DrawValue;
var s : String;
Begin
  s := IntToStr(Value);
  with fBitmap.Canvas do begin
    Brush.Style := bsClear;
    TextOut(fCentrePoint.X - (TextWidth(s) div 2),
            Height- fFrameWidth -fTxtHeight -1, s);
  end;
End;

Procedure TAnalogMeter2000X.DrawFace;
Begin
  ClearCanvas;
  DrawRegions;
  If fShowTicks and (TickCount > 1) then DrawTicks;
  If fCaption <> '' then DrawCaption;
  DrawPointer;
  If fShowValue then DrawValue;
  If fShowFrame then DrawFrame;
End;

Procedure TAnalogMeter2000X.Paint;
Begin
  BitBlt(Canvas.Handle, 0, 0, Width, Height,
         fBitmap.Canvas.Handle, 0, 0, SRCCOPY);
End;

end.
