{*******************************************************}
{            Database structure checker                 }
{               for local databases                     }
{                 DBASE & PARADOX                       }
{                                                       }
{      Copyright (c) 1999-2000 Field of Miracles        }
{            e-mail: deleon@mailru.com                  }
{                                                       }
{*******************************************************}
unit DbChk;
{$R DBCHK.DCR}
interface

uses
  BDE,       Db,        DbTables,   DsgnIntf,
  Windows,   Classes,   Forms,      Dialogs,
  FileCtrl,  SysUtils,  Controls,   BDE32;

type
  TCharSet = set of Char;
  TChkMessage = (CHK_NONE, CHK_TABLECREATE, CHK_TABLEMODIFY, CHK_AUTOINCREPAIR);

  TCheckEvent    = procedure (Sender: TObject; ChkMessage: TChkMessage; TblName: string) of object;
  TFindDiffEvent = procedure (Sender: TObject; ChkMessage: TChkMessage; TblName: string; var Correct: boolean) of object;
  TErrorEvent    = procedure (Sender: TObject; Message: EDBEngineError) of object;
  TAddFieldEvent = procedure (Sender: TObject; TblName, FldName: string) of object;
  TTableEvent    = procedure (Sender: TObject; TblName: string) of object;

type
  TDbCheck = class(TComponent)
  private
   { Private declarations }
   FPos              : Longint;
   FTable            : TTable;
   FCorrect          : Boolean;
   FStrings          : TStrings;
   FReadFiles        : TStrings;
   FTableName        : string;
   FReadPath         : string;
   FCorrectPath      : string;
   FAfterCheck       : TCheckEvent;
   FBeforeCheck      : TCheckEvent;
   FCompleteEvent    : TNotifyEvent;
   FFindDiffEvent    : TFindDiffEvent;
   FErrorEvent       : TErrorEvent;
   FAddFieldEvent    : TAddFieldEvent;
   FModifyFieldEvent : TAddFieldEvent;
   FCreateTableEvent : TTableEvent;
   FModifyTableEvent : TTableEvent;
   procedure SetStrings(Value: TStrings);
   procedure SetReadFiles(Value: TStrings);
   function  LoadObjectDesc(var FldCount, IDXCount: Word;
                            var pFlds: pFLDDesc;
                            var pIdxs: pIDXDesc): DBIRESULT;
   function  IName: string;
   function  CheckFieldDescs(var NewFlds, OldFlds: Word; var pNewFlds, pOldFlds: pFLDDesc): boolean;
   function  CheckIndexDescs(var NewIdxs, OldIdxs: Word; var pNewIdxs, pOldIdxs: pIDXDesc): boolean;
   function  GetTblDesc(TblName: string): string;
  protected
    { Protected declarations }
  public
    { Public declarations }
   constructor Create(AOwner: TComponent);override;
   destructor  Destroy; override;
   function    CheckDatabase: boolean;
   procedure   ReadDatabase;
   property    CheckTableName: string read FTableName;
  published
    { Published declarations }
   property ReadPath      : string read FReadPath write FReadPath;
   property CorrectPath   : string read FCorrectPath write FCorrectPath;
   property Descriptor    : TStrings read FStrings write SetStrings;
   property ReadFiles     : TStrings read FReadFiles write SetReadFiles;
   property OnAfterCheck  : TCheckEvent read FAfterCheck write FAfterCheck;
   property OnBeforeCheck : TCheckEvent read FBeforeCheck write FBeforeCheck;
   property OnComplete    : TNotifyEvent read FCompleteEvent write FCompleteEvent;
   property OnError       : TErrorEvent read FErrorEvent write FErrorEvent;
   property OnFoundDiff   : TFindDiffEvent read FFindDiffEvent write FFindDiffEvent;
   property OnAddField    : TAddFieldEvent read FAddFieldEvent write FAddFieldEvent;
   property OnModifyField : TAddFieldEvent read FModifyFieldEvent write FModifyFieldEvent;
   property OnCreateTable : TTableEvent read FCreateTableEvent write FCreateTableEvent;
   property OnModifyTable : TTableEvent read FModifyTableEvent write FModifyTableEvent;
  end;

procedure Register;

implementation

const
  TBLEXT_DBASE       = '.DBF';
  TBLEXT_PARADOX     = '.DB';
  OFFSET_PDX_AUTOINC = 73;

{-----------------------------------------------------------------------}
{  WordPosition and ExtractWord functions                               }
{  are copied from RX's STRUTILS.PAS unit                               }
{-----------------------------------------------------------------------}
function WordPosition(const N: Integer; const S: string;
         const WordDelims: TCharSet): Integer;
var
  Count, I: Integer;
begin
  Count := 0;
  I := 1;
  Result := 0;
  while (I <= Length(S)) and (Count <> N) do begin
    { skip over delimiters }
    while (I <= Length(S)) and (S[I] in WordDelims) do Inc(I);
    { if we're not beyond end of S, we're at the start of a word }
    if I <= Length(S) then Inc(Count);
    { if not finished, find the end of the current word }
    if Count <> N then
      while (I <= Length(S)) and not (S[I] in WordDelims) do Inc(I)
    else Result := I;
  end;
end;

function ExtractWord(N: Integer; const S: string;
         const WordDelims: TCharSet): string;
var
  I: Integer;
  Len: Integer;
begin
  Len := 0;
  I := WordPosition(N, S, WordDelims);
  if I <> 0 then
    { find the end of the current word }
    while (I <= Length(S)) and not(S[I] in WordDelims) do begin
      { add the I'th character to result }
      Inc(Len);
      SetLength(Result, Len);
      Result[Len] := S[I];
      Inc(I);
    end;
  SetLength(Result, Len);
end;

function FLDComp(pFld1, pFld2: pFLDDesc): boolean;
begin
 Result := (StrIComp(pFld1^.szName, pFld2^.szName) = 0)and
           (pFld1^.iFldType = pFld2^.iFldType)and
           (pFld1^.iSubType = pFld2^.iSubType)and
           (pFld1^.iUnits1  = pFld2^.iUnits1)and
           (pFld1^.iUnits2  = pFld2^.iUnits2);
end;

function IDXComp(pIdx1, pIdx2: pIDXDesc): boolean;
begin
 Result := ( ANSIStrIComp(pIdx1^.szName, pIdx2^.szName) = 0)and
           ( ANSIStrIComp(pIdx1^.szTagName, pIdx2^.szTagName) = 0)and
           ( StrLIComp(@pIdx1^.aiKeyFld, @pIdx2^.aiKeyFld, SizeOf(DBIKEY)) = 0)and
           ( pIdx1^.bPrimary          = pIdx2^.bPrimary)and
           ( pIdx1^.bExpIdx           = pIdx2^.bExpIdx)and
           ( pIdx1^.bUnique           = pIdx2^.bUnique)and
           ( pIdx1^.bDescending       = pIdx2^.bDescending)and
           ( pIdx1^.bCaseInsensitive  = pIdx2^.bCaseInsensitive)and
           ( pIdx1^.bSubset           = pIdx2^.bSubset)and
           ( pIdx1^.bOutOfDate        = pIdx2^.bOutOfDate)and
           ( pIdx1^.iFldsInKey        = pIdx2^.iFldsInKey)and
           ( ANSIStrIComp(pIdx1^.szFormat, pIdx2^.szFormat) = 0)and
           ( ANSIStrIComp(pIdx1^.szKeyExp, pIdx2^.szKeyExp) = 0)and
           ( ANSIStrIComp(pIdx1^.szKeyCond,pIdx2^.szKeyCond)= 0);
end;

function  StrToAiKeyFld(const Value: string): DBIKEY;
var
 I, F: integer;
 xStr: string;
begin
 FillChar(Result, SizeOf(DBIKEY), 0);
 I := 1; F := 0;
 while (I <= Length(Value))do
 begin
   xStr := '';
   while (I <= Length(Value))and(Value[I]<>',')do
   begin
    xStr := xStr + Value[I];
    inc(I);
   end;
  Result[F] := StrToIntDef(xStr, 0);
  inc(F);
  inc(I);
 end;

end;

procedure GetCompTableInfo(Tbl: TTable; xInfoList: TStrings);
var
  I: integer;
  CURPROP: CURProps;
  FldParams: pFLDDesc;
  IdxParams: pIDXDesc;
  IsActive: boolean;

  { CHILD }
  function GetFieldNamesInKey(IDX: IDXDesc): string;
  var
    J: integer;
  begin
    J := 0;
    Result := '';
    while( True )do
    begin
     Result := Result + IntToStr(IDX.aiKeyFld[J]);
     inc(J);
     if IDX.aiKeyFld[J] = 0 then Break;
     Result := Result + ',';
    end;
  end;{ CHILD }

begin
  IsActive := Tbl.Active;
  if not( IsActive )then Tbl.Active := True;
  { GET TABLE PROPS }
  Check(DbiGetCursorProps(Tbl.Handle, CURPROP));
  { ADD HEADER }
  xInfoList.Add('object Table: ' + Tbl.TableName);

  {  F I E L D   D E S C R I P T O R S  }

  { ALLOC MEM }
  FldParams := AllocMem(SizeOf(FLDDesc)* CURPROP.iFields);
  { READ FIELDS DESCRIPTOR }
  if( DbiGetFieldDescs(Tbl.Handle, FldParams) = DBIERR_NONE )then
   for I := 0 to CURPROP.iFields - 1 do
   with( pFieldDescList(FldParams)^[I] )do
   begin
    xInfoList.Add('  field' +
                  Format('(%s;%s;%d;%d;%d;%d)',
    {1 - szName     s}   [szNAME,
    {2 - szOldName  s}    szNAME,
    {3 - iFldType   d}    iFldType,
    {4 - iSubType   d}    iSubType,
    {5 - iUnits1    d}    iUnits1,
    {6 - iUnits2    d}    iUnits2]));
   end;{ FOR - WITH }
  { FREE MEM }
  FreeMem(FldParams, SizeOf(FLDDesc) * CURPROP.iFields);


  {  I N D E X   D E S C R I P T O R S  }

  { ALLOC MEM }
  IDXParams := AllocMem(SizeOf(IDXDesc)* CURPROP.iIndexes);
  { READ INDEXES DESCRIPTOR }
  if( DbiGetIndexDescs(Tbl.Handle, IDXParams) = DBIERR_NONE )then
   for I := 0 to CURPROP.iIndexes - 1 do
   with( pIndexDescList(IDXParams)^[I] )do
   begin
    xInfoList.Add('  index' +
                  Format('(%s;%s;%d;%s;%d;%d;%d;%d;%d;%d;%d;%s;%s;%d;%s;%s)',
    {1 - szName              s}  [szName,
    {2 - OldszName           s}   szTagName,
    {3 - iFldsInKey          n}   iFldsInKey,
    {4 - aiKeyFld            s}   GetFieldNamesInKey(pIndexDescList(IDXParams)^[I]),
    {5 - bPrimary            n}   Longint(bPrimary),
    {6 - bUnique             n}   Longint(bUnique),
    {7 - bDescending         n}   Longint(bDescending),
    {8 - bMaintained         n}   Longint(bMaintained),
    {9 - bCaseInsensitive    n}   Longint(bCaseInsensitive),
    {10- bExpIdx             n}   Longint(bExpIdx),
    {11- bSubset             n}   Longint(bSubset),
    {12- szTagName           s}   szTagName,
    {13- szFormat            s}   szFormat,
    {14- iKeyExpType         n}   iKeyExpType,
    {15- szKeyExp            s}   szKeyExp,
    {16- szKeyCond           s}   szKeyCond]));
   end;{ FOR - WITH }
  { FREE MEM }
  FreeMem(IDXParams, SizeOf(IDXDesc) * CURPROP.iIndexes);
  { ADDING END OF OBJECT }
  xInfoList.Add('end');
  Tbl.Active := IsActive;
end;

function  _RESTRUCTURE(Tbl: TTable; pFlds: pFldDesc; pIdxs: pIDXDesc;
                       FldCount, IdxCount: Word): DBIResult;
var
  I: integer;
  CurProp: CurProps;
  TblDesc: CRTblDesc;
  hDb: hDBIDb;
  TablePath: DBIPATH;
begin
  hDb := nil;
  { INITIALIZE THE TABLE DESCRIPTOR }
  FillChar(TblDesc, SizeOf(CRTblDesc), #0);
  { GET MEMS }
  TblDesc.pecrFldOp := AllocMem(FldCount * SizeOf(CROpType));
  TblDesc.pecrIdxOp := AllocMem(IdxCount * SizeOf(CROpType));

  Tbl.Active := True;
  Check(DbiGetCursorProps(Tbl.Handle, CurProp));

  StrCopy(TblDesc.szTblType, CurProp.szTableType);
  StrCopy(TblDesc.szTblName, CurProp.szName);
  TblDesc.pfldDesc  := pFlds;
  TblDesc.pidxDesc  := pIdxs;
  TblDesc.iFldCount := FldCount;
  TblDesc.iIdxCount := IdxCount;
  TblDesc.bPack     := True;

  for I := 0 to FldCount - 1 do
  pCROpTypeList(TblDesc.pecrFldOp)^[I] := CROpType(pFieldDescList(pFlds)^[I].iUnUsed[0]);
  for I := 0 to IdxCount - 1 do
  pCROpTypeList(TblDesc.pecrIdxOp)^[I] := CROpType(pIndexDescList(pIdxs)^[I].iUnUsed[0]);

{ GET TABLE DIRECTORY }
  Check(DbiGetDirectory(Tbl.DBHandle, False, TablePath));
  Tbl.Active := False;

  try
   Check(DbiOpenDatabase(nil, szCFGDBSTANDARD, dbiReadWrite, dbiOpenExcl, nil,
                         0, nil, nil, hDb));
   Check(DbiSetDirectory(hDb, TablePath));
   Result := DbiDoRestructure(hDb, 1, @TblDesc, nil, nil, nil, False);
  finally
   Check(DbiCloseDatabase(hDb));
   FreeMem(TblDesc.pecrFldOp, FldCount * SizeOf(CROpType));
   FreeMem(TblDesc.pecrIdxOp, IdxCount * SizeOf(CROpType));
  end;
end;

function  _CREATE(Tbl: TTable; pFlds: pFldDesc; pIdxs: pIDXDesc;
                  FldCount, IdxCount: Word): DBIResult;
var
  I: integer;
  TblDesc: CRTblDesc;
  hDb: hDbiDB;
begin
  { Initialize the table descriptor }
  hDb := nil;
  FillChar(TblDesc, SizeOf(CRTblDesc), #0);
  TblDesc.pecrFldOp := AllocMem(SizeOf(CROpType) * FldCount);
  TblDesc.pecrIdxOp := AllocMem(SizeOf(CROpType) * IdxCount);
  StrCopy(TblDesc.szTblType, szPARADOX);
  StrPCopy(TblDesc.szTblName, Tbl.TableName);
  TblDesc.iFldCount := FldCount;
  TblDesc.iIdxCount := IdxCount;
  TblDesc.pfldDesc  := pFlds;
  TblDesc.pidxDesc  := pIdxs;

  for I := 0 to FldCount - 1 do
  with( pFieldDescList( pFlds )^[I] )do
  begin
   iFldNum := 0;
   pCROpTypeList(TblDesc.pecrFldOp)^[I] := crADD;
  end;

  for I := 0 to IdxCount - 1 do
  with( pIndexDescList( pIdxs )^[I] )do
  begin
   iRestrNum := 0;
   pCROpTypeList(TblDesc.pecrIdxOp)^[I] := crADD;
  end;

  try
   Check(DbiOpenDatabase(nil, szCFGDBSTANDARD, dbiReadWrite, dbiOpenExcl, nil,
         0, nil, nil, hDb));
   Check(DbiSetDirectory(hDb, PChar(Tbl.DatabaseName)));
   Result := DbiCreateTable(hDb, True, TblDesc);
  finally
   Check(DbiCloseDatabase(hDb));
   FreeMem(TblDesc.pecrFldOp, SizeOf(CROpType) * FldCount);
   FreeMem(TblDesc.pecrIdxOp, SizeOf(CROpType) * IdxCount);
  end;
end;

procedure _REINDEX(Table: TTable);
var
  WasActive: Boolean;
  WasExclusive: Boolean;
begin
  with Table do begin
    WasActive := Active;
    WasExclusive := Exclusive;
    DisableControls;
    try
      if not (WasActive and WasExclusive) then Close;
      try
        Exclusive := True;
        Open;
        Check(dbiRegenIndexes(Handle));
      finally
        if not (WasActive and WasExclusive) then begin
          Close;
          Exclusive := WasExclusive;
          Active := WasActive;
        end;
      end;
    finally
      EnableControls;
    end;
  end;
end;


{-----------------------------------------------------------------------}
{     TDbCheck                                                    }
{-----------------------------------------------------------------------}
constructor TDbCheck.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FStrings := TStringList.Create;
  FReadFiles := TStringList.Create;
  FTable := TTable.Create(Self);
end;

destructor  TDbCheck.Destroy;
begin
  FStrings.Free;
  FReadFiles.Free;
  FTable.Free;
  inherited Destroy;
end;

function TDbCheck.GetTblDesc(TblName: string): string;
var
 I: integer;
 S, Value : string;
begin
 Result  := TblName;
 TblName := UpperCase(ChangeFileExt(TblName, ''));
 for I := 0 to FReadFiles.Count - 1 do
 begin
  S := FReadFiles[I];
  Value := ExtractWord(1, S, ['=','|']);
  Value := UpperCase(ChangeFileExt(Value, ''));
  if( Value = TblName )then
  begin
   Value := ExtractWord(2, S, ['=','|']);
   if( Value <> '' )then
   begin
    Result := Value;
    Break;
   end;{ IF }
  end;{ IF }
 end;{ FOR }
end;

function TDbCheck.IName: string;
begin
 Result := UpperCase( ExtractWord(1, FStrings[FPos], [' ','(']));
end;

function  TDbCheck.LoadObjectDesc(var FldCount, IDXCount: Word;
                                  var pFlds: pFLDDesc;
                                  var pIdxs: pIDXDesc): DBIRESULT;
var
 J, iFld, iIdx, Wrd: integer;
 xStr, wStr: string;
begin
 Result := DBIERR_NONE;
 FldCount := 0;
 IdxCount := 0;
 pFlds := nil;
 pIdxs := nil;

 xStr := FStrings[FPos];
 if iNAME = 'OBJECT' then
 begin
   inc(FPos);
   xStr := FStrings[FPos];
   iFld := 0;
   iIdx := 0;
   { CICLE #3 }
   while (IName <> 'END') do
   begin
    { LOAD FIELD DESCS}
    if iNAME = 'FIELD' then
    begin
     inc(FldCount);
     ReallocMem(pFlds, SizeOf(FLDDesc)* FldCount);
     FillChar(pFieldDescList(pFlds)^[FldCount - 1], SizeOf(FLDDesc), 0);
     with( pFieldDescList(pFlds)^[FldCount - 1] )do
     begin
      iFldNum := iFld + 1;
      iUnUsed[0] := Ord(crNOOP);

      J := pos('(',xStr);
      inc(J);
      Wrd := 0;
      while (J <= Length(xStr))do
      begin
       wStr := '';
       while (J <= Length(xStr))and not(xStr[J]in[';',')'])do
       begin
        wStr := wStr + xStr[J];
        inc(J);
       end;
       inc(Wrd);
       case Wrd of
       {1 - szName}
       1: StrCopy(szName, PChar(wStr));
       {2 - szOldName}
       2: ;
       {3 - iFldType}
       3: iFldType := StrToIntDef(wStr, 0);
       {4 - iSubType}
       4: iSubType := StrToIntDef(wStr, 0);
       {5 - iUnits1}
       5: iUnits1  := StrToIntDef(wStr, 1);
       {6 - iUnits2}
       6: iUnits2  := StrToIntDef(wStr, 0);
       end;
       inc(J);
      end;
      inc(FPos);
      xStr := FStrings[FPos];
      inc(iFld);
     end;{ FIELDS }
    end;{ IF }

    { LOAD INDEX DESCS}
    if( iNAME = 'INDEX' )then
    begin
     inc(IdxCount);
     ReallocMem(pIdxs, SizeOf(IDXDesc)* IdxCount);
     FillChar(pIndexDescList(pIdxs)^[IdxCount - 1], SizeOf(IDXDesc), 0);
     with( pIndexDescList(pIdxs)^[IdxCount - 1] )do
     begin
      iIndexId := iIdx + 1;
      iUnUsed[0] := Ord(crNOOP);

      J := pos('(',xStr);
      inc(J);
      Wrd := 0;
      while (J <= Length(xStr))do
      begin
       wStr := '';
       while (J <= Length(xStr))and not(xStr[J]in[';',')'])do
       begin
        wStr := wStr + xStr[J];
        inc(J);
       end;
       inc(Wrd);
       case Wrd of
       {1 - szName              s}  1:  StrCopy(szName, PChar(wStr));
       {2 - szOldName           s}  2:  StrCopy(szTagName, PChar(wStr));
       {3 - iFldsInKey          n}  3:  iFldsInKey  := StrToIntDef(wStr, 0);
       {4 - aiKeyFld            s}  4:  aiKeyFld    := StrToAiKeyFld(wStr);
       {5 - bPrimary            n}  5:  bPrimary    := Boolean(StrToIntDef(wStr, 0));
       {6 - bUnique             n}  6:  bUnique     := Boolean(StrToIntDef(wStr, 0));
       {7 - bDescending         n}  7:  bDescending := Boolean(StrToIntDef(wStr, 0));
       {8 - bMaintained         n}  8:  bMaintained := Boolean(StrToIntDef(wStr, 0));
       {9 - bCaseInsensitive    n}  9:  bCaseInsensitive := Boolean(StrToIntDef(wStr, 0));
       {10- bExpIdx             n}  10: bExpIdx     := Boolean(StrToIntDef(wStr, 0));
       {11- bSubset             n}  11: bSubset     := Boolean(StrToIntDef(wStr, 0));
       {12- szTagName           s}  12: StrCopy(szTagName, PChar(wStr));
       {13- szFormat            s}  13: StrCopy(szFormat, PChar(wStr));
       {14- iKeyExpType         n}  14: iKeyExpType := StrToIntDef(wStr, 0);
       {15- szKeyExp            s}  15: StrCopy(szKeyExp, PChar(wStr));
       {16- szKeyCond           s}  16: StrCopy(szKeyCond, PChar(wStr));
       end; //case
       inc(J);
      end;//while
      inc(FPos);
      xStr := FStrings[FPos];
      inc(iIdx);
     end;{ INDEX }
    end;{ IF }

   end;//while
 end;

 if FPOS >= FStrings.Count then Result := DBIERR_EOF;
end;

function TDbCheck.CheckFieldDescs(var NewFlds, OldFlds: Word; var pNewFlds, pOldFlds: pFLDDesc): boolean;
var
 I, Pos: integer;
 pFld1: pFLDDesc;
 { CHILD }
 function FldExists(pFld: pFldDesc): integer;
 var
  J: integer;
 begin
  Result := -1;
  for J := 0 to OldFlds - 1 do
  if ANSIStrIComp(pFld^.szName, pFieldDescList(pOldFlds)^[J].szName)= 0 then
  begin
   Result := J;
   Break;
  end;
 end;

begin
 Result := True;
 for I := 0 to NewFlds - 1 do
 begin
  pFld1 := Addr(pFieldDescList(pNewFlds)^[I]);
  { GET FIELD POSITION IN OLD DESCRIPTOR }
  Pos := FldExists(pFld1);
  { FIELD NOT FOUND }
  if( Pos = -1 )then
  begin
   pFld1^.iFldNum   := 0;
   pFld1^.iUnUsed[0]:= Ord(crADD);
   if Assigned( FAddFieldEvent )then
   FAddFieldEvent(Self, FTable.TableName, pFld1^.szName);
   Result := False;
  end else
  { FIELD FOUND }
  if not((Pos = I)and( FLDComp(pFld1, @pFieldDescList(pOldFlds)^[Pos])))then
  begin { FIELD WAS MODIFIED }
   pFld1^.iFldNum    := Pos + 1;
   pFld1^.iUnUsed[0] := Ord(crMODIFY);
   if( pFld1^.iSubType = fldstAUTOINC )then pFld1^.efldrRights := fldrREADONLY;
   if Assigned( FModifyFieldEvent )then
   FModifyFieldEvent(Self, FTable.TableName, pFld1^.szName);
   Result := False;
  end;
 end;
end;

function TDbCheck.CheckIndexDescs(var NewIdxs, OldIdxs: Word; var pNewIdxs, pOldIdxs: pIDXDesc): boolean;
var
 I, F     : integer;
 p, pFIND : pIDXDesc;
 _Count   : Word;
begin
 Result := True;
 _Count := NewIdxs;
 { FIND INDEXS IN OLD STRUCTURE }
 for I := 0 to NewIdxs - 1 do
 begin
  p  := Addr(pIndexDescList(pNewIdxs)^[I]);
  pFIND := nil;
  for F := 0 to OldIdxs - 1 do
  if(ANSIStrIComp(p^.szName, pIndexDescList(pOldIdxs)^[F].szName) = 0)then
  begin
   pFIND := Addr(pIndexDescList(pOldIdxs)^[F]);
   Break;
  end;{ IF - FOR }
  { INDEX NOT FOUND }
  if( pFIND = nil )then
  begin
   p^.iIndexId  := 0;
   p^.iRestrNum := 0;
   p^.iUnUsed[0]:= Ord(crADD);
   Result := False;
  end else
  if(IDXComp(pFIND, p))then
  begin
   p^.iRestrNum := I + 1;
   p^.iUnUsed[0] := Ord(crNOOP);
  end else
  begin
   p^.iRestrNum := I + 1;
   p^.iUnUsed[0] := Ord(crMODIFY);
   Result := False;
  end;
 end;{ FOR }


 { FIND INDEXS IN NEW STRUCTURE }
 for I := 0 to OldIdxs - 1 do
 begin
  p  := Addr(pIndexDescList(pOldIdxs)^[I]);
  pFIND := nil;
  for F := 0 to _Count - 1 do
  if(ANSIStrIComp(p^.szName, pIndexDescList(pNewIdxs)^[F].szName) = 0)then
  begin
   pFIND := Addr(pIndexDescList(pNewIdxs)^[F]);
   Break;
  end;

  if( pFIND = nil )then{    }
  begin
   inc(NewIdxs);
   ReallocMem(pNewIdxs, NewIdxs * SizeOf(IDXDesc));
   pFIND  := Addr(pIndexDescList(pNewIdxs)^[NewIdxs - 1]);
   pFIND^ := p^;
   pFIND^.iRestrNum  := I + 1;
   pFIND^.iUnUsed[0] := Ord(crDROP);
   Result := False;
  end;
 end;{ FOR }
end;

function TDbCheck.CheckDatabase: boolean;
var
 I               : Integer;
 Fld             : TField;
 FileHnd         : Integer;
 AutoInc1        : Integer;
 AutoInc2        : Integer;
 pLoadFlds       : pFLDDesc;
 pFldChk         : pFLDDesc;
 pLoadIdxs       : pIDXDesc;
 pIdxChk         : pIDXDesc;
 LoadFld         : word;
 LoadIdx         : word;
 CURPROP         : CurProps;
 Value           : string;
 CorrectPath     : string;
 TblExt          : string;
 TblType         : DBINAME;
 RES             : DBIResult;
 EDb             : EDBEngineError;
 EIdx            : boolean;
 EFld            : boolean;
 ChkMessage      : TChkMessage;
begin
 FPos := 0;
 pFldChk := nil;
 pIdxChk := nil;
 Result := True;
 if SESSION.IsAlias(FCorrectPath)then
  CorrectPath := BDE32.GetAliasPath(PChar(FCorrectPath))
 else
  CorrectPath := FCorrectPath;
 FTable.DatabaseName := CorrectPath;

 while( FPos < FStrings.Count )do
 begin
  Value := FStrings[FPos];
  if( Length( Value ) = 0 )then Continue;

  if( IName = 'OBJECT' )then
  begin
   FTable.TableName := ExtractWord(3, Value, [' ']);
   FTableName := UpperCase(FTable.TableName);
   TblExt     := ExtractFileExt(FTableName);
   if( TblExt = TBLEXT_DBASE )then
   TblType    := szDBASE else
   if( TblExt = TBLEXT_PARADOX )then
   TblType    := szPARADOX else
   TblType    := '';
   LoadObjectDesc(LoadFld, LoadIdx, pLoadFlds, pLoadIdxs);
  end;

{-----------------------------------------------------------------------}
{  Check table open                                                     }
{-----------------------------------------------------------------------}
  ChkMessage := CHK_NONE;
  RES := CheckOpenTable(PChar(FTable.DatabaseName),
                        PChar(FTable.TableName),
                        TblType, False);

{-----------------------------------------------------------------------}
{  ERROR: Table not found                                               }
{-----------------------------------------------------------------------}
  if( RES = DBIERR_NOSUCHTABLE )then
  begin
   Result   := False;
   FCorrect := True;
   if Assigned(FFindDiffEvent)then FFindDiffEvent(Self, CHK_TABLECREATE, GetTblDesc(FTable.TableName), FCorrect);
   if( FCorrect )then
   begin
   {#}RES := _CREATE(FTable, pLoadFlds, pLoadIdxs, LoadFld, LoadIdx);
      if( RES = DBIERR_NONE )then
      begin
       ChkMessage := CHK_TABLECREATE;
       if Assigned( FCreateTableEvent )then
       FCreateTableEvent(Self, FTable.TableName);
      end else
      if Assigned(FErrorEvent)then
      begin
       EDb := EDBEngineError.Create(RES);
       FErrorEvent(Self, EDb);
       EDb.Free;
      end;
   end;
  end else
{-----------------------------------------------------------------------}
{  ERROR: Index is out of date or None                                                          }
{-----------------------------------------------------------------------}
  if( RES = DBIERR_NONE )or( RES = DBIERR_INDEXOUTOFDATE )then
  begin
   if( RES = DBIERR_INDEXOUTOFDATE )then
   {}BDE32.DeleteIndexes(PChar(FTable.DatabaseName), PChar(FTable.TableName));
   if Assigned(FBeforeCheck)then FBeforeCheck(Self, CHK_NONE, GetTblDesc(FTable.TableName));
   FTable.Active := True;
    DbiGetCursorProps(FTable.Handle, CurProp);
    pFldChk := AllocMem(SizeOf(FLDDesc) * CurProp.iFields);
    pIdxChk := AllocMem(SizeOf(IDXDesc) * CurProp.iIndexes);
    DbiGetFieldDescs(FTable.Handle, pFldChk);
    DbiGetIndexDescs(FTable.Handle, pIdxChk);
   { CHECK PARADOX AUTOINC VALIDATE }
   Fld := nil;
   if StrIComp(CurProp.szTableType, szPARADOX) = 0 then
   begin
    for I := 0 to FTable.FieldCount - 1 do
    if( FTable.Fields[I].DataType = ftAutoInc )then
    begin
     Fld := FTable.Fields[I];
     FTable.Last;
     AutoInc1 := Fld.AsInteger;
     Break;
    end;{ if for }
   end;{ if }
   FTable.Active := False;

   if( Fld <> nil )then
   begin
    FileHnd := FileOpen(CorrectPath + FTable.TableName, fmOpenRead or fmShareDenyNone);
    if( FileHnd > -1 )then
    begin
     FileSeek( FileHnd, OFFSET_PDX_AUTOINC, soFromBeginning);
     FileRead( FileHnd, AutoInc2, SizeOf(Integer) );
     FileClose( FileHnd );
     if( AutoInc2 < AutoInc1 )then
     begin
      { REOPEN FILE WITH EXCLUSIVE MODE FOR REPAIRED AUTOINC }
      if Assigned(FBeforeCheck)then FBeforeCheck(Self, CHK_AUTOINCREPAIR, GetTblDesc(FTable.TableName));
      FileHnd := FileOpen(CorrectPath + FTable.TableName, fmOpenWrite or fmShareExclusive);
      if( FileHnd > -1 )then
      begin
       FileSeek( FileHnd, OFFSET_PDX_AUTOINC, soFromBeginning);
       FileWrite( FileHnd, AutoInc1, SizeOf(Integer) );
       FileClose( FileHnd );
      end else
      if Assigned(FErrorEvent)then
      begin
       EDb := EDBEngineError.Create( DBIERR_FILEBUSY );{ Table is busy ! }
       FErrorEvent(Self, EDb);
       EDb.Free;
      end;{ else if }
     end;{ if }
    end;{ if }
   end;{ if }
   { COMPARE FIELDS }
    EFld := CheckFieldDescs(LoadFld, CURPROP.iFields, pLoadFlds, pFldChk);
    EIdx := CheckIndexDescs(LoadIdx, CURPROP.iIndexes, pLoadIdxs, pIdxChk);
    if( not( EFld and EIdx ) )then
    begin
     FCorrect := True;
     if Assigned(FFindDiffEvent)then FFindDiffEvent(Self, CHK_TABLEMODIFY, GetTblDesc(FTable.TableName), FCorrect);
     if( FCorrect )then
     begin
     {#}RES := _RESTRUCTURE(FTable, pLoadFlds, pLoadIdxs, LoadFld, LoadIdx);
      if( RES = DBIERR_NONE )then
      begin
       ChkMessage := CHK_TABLEMODIFY;
       if Assigned( FModifyTableEvent )then
       FModifyTableEvent(Self, FTable.TableName);
      end else
       if Assigned(FErrorEvent)then
      begin
       EDb := EDBEngineError.Create(RES);
       FErrorEvent(Self, EDb);
       EDb.Free;
      end;
     end;
    end;
  end;


  FreeMem(pLoadFlds);
  FreeMem(pLoadIdxs);
  inc(FPos);
  if Assigned(FAfterCheck)then FAfterCheck(Self, ChkMessage, GetTblDesc(FTable.TableName));
 end;{ WHILE }
 if Assigned(FCompleteEvent)then FCompleteEvent(Self);
end;

procedure TDbCheck.SetStrings(Value: TStrings);
begin
  FStrings.Assign(Value);
end;

procedure TDbCheck.SetReadFiles(Value: TStrings);
begin
  FReadFiles.Assign(Value);
end;

procedure TDbCheck.ReadDatabase;
var
 SearchRec: TSearchRec;
 Found: integer;
 Value, ReadPath: string;
 xTblInfo: TStringList;
 { CHILD }
 function CanReadFile(FileName: string): boolean;
 var
  S: string;
  F: integer;
 begin
  Result   := FReadFiles.Count = 0;
  FileName := UpperCase(FileName);
  if( not Result )then
  begin
   for F := 0 to FReadFiles.Count - 1 do
   begin
    S := UpperCase(ExtractWord(1, FReadFiles[F], ['=','|']));
    if( FileName = S )then
    begin
     Result := True;
     Break;
    end;
   end;
  end;
 end;

begin
  Screen.Cursor := crHourGlass;
  xTblInfo := TStringList.Create;
  if(AliasExists(PChar(FReadPath)))then
   ReadPath := BDE32.GetAliasPath(PChar(FReadPath))
  else
   ReadPath := FReadPath;

  Found := FindFirst(ReadPath + '\*.DB*', faAnyFile, SearchRec);
  while( Found = 0 )do
  begin
   Value := UpperCase(ExtractFileExt(SearchRec.Name));
   if( Value = TBLEXT_DBASE )or( Value = TBLEXT_PARADOX )then
   begin
    FTable.DatabaseName := ReadPath;
    FTable.TableName    := SearchRec.Name;
    if CanReadFile(SearchRec.Name) then
     GetCompTableInfo(FTable, xTblInfo);
   end;
   Found := FindNext(SearchRec);
  end;
  FindClose(SearchRec);
  FStrings.Assign(xTblInfo);
  xTblInfo.Clear;
  xTblInfo.Free;
  Screen.Cursor := crDefault;
end;

{-----------------------------------------------------------------------}
{     TDirProperty                                                         }
{-----------------------------------------------------------------------}
type
  TDirProperty = class(TStringProperty)
  public
    procedure GetValues(Proc: TGetStrProc); override;
    function  GetAttributes: TPropertyAttributes; override;
  end;

procedure TDirProperty.GetValues(Proc: TGetStrProc);
var
 I: Integer;
 _List: TStringList;
begin
 _List := TStringList.Create;
 try
  SESSION.GetAliasNames(_List);
  for I := 0 to _List.Count - 1 do
  Proc(_List[I]);
 finally
 _List.Free;
 end;
end;

function TDirProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList];
end;

{-----------------------------------------------------------------------}
{     TDbCheckEditor                                                    }
{-----------------------------------------------------------------------}
type
  TDbCheckEditor = class(TComponentEditor)
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): string; override;
    function GetVerbCount: Integer; override;
  end;

procedure TDbCheckEditor.ExecuteVerb(Index: Integer);
begin
 case( Index )of
   0: TDbCheck(Component).ReadDatabase;
   1: TDbCheck(Component).Descriptor.Clear;
 end;{ case }
 Designer.Modified;
end;

function TDbCheckEditor.GetVerb(Index: Integer): string;
begin
  case( Index )of
    0: Result := '&Read database structure...';
    1: Result := '&Clear structure list...';
  end;
end;

function TDbCheckEditor.GetVerbCount: Integer;
begin
  Result := 2;
end;

procedure Register;
begin
  RegisterPropertyEditor(TypeInfo(string), TDbCheck, 'CorrectPath', TDirProperty);
  RegisterPropertyEditor(TypeInfo(string), TDbCheck, 'ReadPath', TDirProperty);
  RegisterComponentEditor(TDbCheck, TDbCheckEditor);
  RegisterComponents('VVMPage',[TDBCheck]);
end;

end.
