
{******************************************}
{                                          }
{           vtk Export library             }
{                                          }
{      Copyright (c) 2002 by vtkTools      }
{                                          }
{******************************************}

unit vteWriters;

interface

{$include vtk.inc}

uses
  Classes, SysUtils, graphics, windows, activex, axctrls,
  {$IFDEF VTK_D6} variants, {$ENDIF}

  BIFF8_Types, vteExcel, vteExcelTypes, vteConsts;

type

///////////////////////////
//
// TvteHtmlCell
//
///////////////////////////
TvteHtmlCell = record
  Range : TvteXLSRange;
  StyleId : word;
  Hide : word;
end;

THtmlCells = array [0..0] of TvteHtmlCell;

PHtmlCells = ^THtmlCells;


///////////////////////////
//
// TvteHTMLWriter
//
///////////////////////////
TvteHTMLWriter = class(TvteCustomWriter)
private
  FileStream : TFileStream;
  FFileName : string;
  FilesDir : string;
  FName, FileExt : string;
  DirName : string;
  FWorkBook : TvteXLSWorkbook;
  HtmlCells : PHtmlCells;
  MinPos : integer;
  RowCount, ColCount : integer;
  Styles : TStrings;
  SpansPresent : boolean;

  procedure AddRange(Range: TvteXLSRange);
  function GenStyle(Range: TvteXLSRange) : string;
  procedure SaveHeadFiles;
  procedure SaveMainFile;
  procedure SaveHeadFile;
  procedure WriteStyles;
  procedure WriteRowTag(Sheet : TvteXLSWorksheet; RowIndex,Level : integer);
  procedure WriteCellTag(Sheet : TvteXLSWorksheet; RowIndex,ColumnIndex,Level : integer);
  function GetSheetFileName(SheetNumber : integer) : string;
  function GetCellTagString(Range: TvteXLSRange) : string;
  procedure InitStrings;
public
  constructor Create;
  destructor Destroy; override;
  procedure SaveSheet(Sheet :TvteXLSWorksheet; FileName : string);
  procedure Save(WorkBook : TvteXLSWorkbook; const FileName : string);  override;
end;

////////////////////////////////
//
// rXLSRangeRec
//
////////////////////////////////
rXLSRangeRec = record
  iXF : integer;
  iSST : integer;
  iFont : integer;
  iFormat : integer;
end;
pXLSRangeRec = ^rXLSRangeRec;
rXLSRangesRec = array [0..0] of rXLSRangeRec;
pXLSRangesRec = ^rXLSRangesRec;

////////////////////////////////
//
// rXLSSheetRec
//
////////////////////////////////
rXLSSheetRec = record
  StreamBOFOffset : integer;
  StreamBOFOffsetPosition : integer;
end;
rXLSSheetsRecs = array [0..0] of rXLSSheetRec;
pXLSSheetsRecs = ^rXLSSheetsRecs;

////////////////////////////////
//
// TvteExcelWriter
//
////////////////////////////////
TvteExcelWriter = class(TvteCustomWriter)
private
  FBOFOffs : integer;
  FWorkBook : TvteXLSWorkbook;
  FUsedColors : TList;
  FRangesRecs : pXLSRangesRec;
  FColorPalette : array [0..XLSMaxColorsInPalette-1] of TColor;
  FPaletteModified : boolean;
  FSheetsRecs : pXLSSheetsRecs;
  function GetColorPaletteIndex(Color : TColor) : integer;
  procedure BuildFontList(l : TList);
  procedure BuildFormatList(sl : TStringList);
  procedure BuildXFRecord(Range : TvteXLSRange; var XF : rb8XF; prr : pXLSRangeRec);
  procedure BuildXFList(l : TList);
  procedure WriteSheetToStream(Stream : TStream; Sheet : TvteXLSWorksheet);
  procedure WriteRangeToStream(Stream : TStream; Range : TvteXLSRange; CurrentRow : integer; var IndexInCellsOffsArray : integer; var CellsOffs : Tb8DBCELLCellsOffsArray);
public
  constructor Create;
  destructor Destroy; override;
  procedure SaveAsBIFFToStream(WorkBook : TvteXLSWorkbook; Stream : TStream);
  procedure Save(WorkBook : TvteXLSWorkbook; const FileName : string); override;
end;

const
  aDefaultColorPalette : array [0..XLSMaxColorsInPalette-1] of TColor =
                         ($000000,
                          $FFFFFF,
                          $0000FF,
                          $00FF00,
                          $FF0000,
                          $00FFFF,
                          $FF00FF,
                          $FFFF00,
                          $000080,
                          $008000,
                          $800000,
                          $008080,
                          $800080,
                          $808000,
                          $C0C0C0,
                          $808080,
                          $FF9999,
                          $663399,
                          $CCFFFF,
                          $FFFFCC,
                          $660066,
                          $8080FF,
                          $CC6600,
                          $FFCCCC,
                          $800000,
                          $FF00FF,
                          $00FFFF,
                          $FFFF00,
                          $800080,
                          $000080,
                          $808000,
                          $FF0000,
                          $FFCC00,
                          $FFFFCC,
                          $CCFFCC,
                          $99FFFF,
                          $FFCC99,
                          $CC99FF,
                          $FF99CC,
                          $99CCFF,
                          $FF6633,
                          $CCCC33,
                          $00CC99,
                          $00CCFF,
                          $0099FF,
                          $0066FF,
                          $996666,
                          $969696,
                          $663300,
                          $669933,
                          $003300,
                          $003333,
                          $003399,
                          $663399,
                          $993333,
                          $333333);
  aDefaultColors : array [0..MaxDefaultColors-1] of integer =
                   (clWhite,clBlack,clSilver,clGray,
                    clRed,clMaroon,clYellow,clOlive,
                    clLime,clGreen,clAqua,clTeal,
                    clBlue,clNavy,clFuchsia,clPurple);

  aHtmlCellBorders : array[vtexlEdgeBottom..vtexlEdgeTop] of string = ('bottom','left','right','top');
  aBorderLineStyles : array[vtelsNone..vtelsSlantedDashDot] of string =
    ('none','.5pt solid','1.0pt solid',
     '.5pt dashed','.5pt dotted','1.5pt solid',
     '2.0pt double','.5pt hairline','1.0pt dashed',
     '.5pt dot-dash','1.0pt dot-dash','.5pt dot-dot-dash',
     '1.0pt dot-dot-dash','1.0pt dot-dash-slanted');

implementation

uses
  Math;

// MakeHTMLString
// Replaces special symbols according to the specification HTML
function MakeHTMLString(Value : string): string;
var
  i : integer;
begin
  Result := '';
  for i := 1 to Length(Value) do
    case Value[i] of
      '"' : Result := Result + vteHtml_quot;
      '<' : Result := Result + vteHtml_lt;
      '>' : Result := Result + vteHtml_gt;
      '&' : Result := Result + vteHtml_amp;
    else
      Result := Result + Value[i];
    end;
end;

// WriteBlockSeparator
// Writes down CRLF in the specified stream
procedure WriteBlockSeparator( AStream : TStream);
var
  P : PChar;
begin
  P := @vteBLOCKSEPARATOR[1];
  AStream.Write(P^,Length(vteBLOCKSEPARATOR));
end;

// WriteStringToStream
// Writes Value string in the specified stream
procedure WriteStringToStream(AStream: TStream; Value : string);
var
  P : PChar;
begin
  P := @Value[1];
  AStream.Write(P^,Length(Value));
end;

// WriteLevelMargin
// Writes down Level spaces in specified stream
procedure WriteLevelMargin( AStream : TStream; Level : integer);
begin
  AStream.Write(vteMAXMARGINSTRING,Min(Length(vteMAXMARGINSTRING),Level));
end;

// WriteStringWithFormatToStream
// Writes Value string in the specified stream with margin and line feed
procedure WriteStringWithFormatToStream( AStream : TStream; Value : string; Level : integer );
begin
  WriteLevelMargin(AStream, Level);
  WriteStringToStream(AStream, Value);
  WriteBlockSeparator(AStream);
end;

// WriteOpenTagFormat
procedure WriteOpenTagFormat( AStream : TStream; Tag : string; Level : integer );
begin
  WriteStringWithFormatToStream(AStream, Format('%s%s%s',[vteOPENTAGPREFIX,tag,vteTAGPOSTFIX]),Level);
end;

// WriteOpenTagClassFormat
// writes Tag to AStream with specified ClassId
procedure WriteOpenTagClassFormat( AStream : TStream; Tag : string; Level : integer ; ClassId : integer);
var
  ClName : string;
begin
  ClName := Format(vteSTYLEFORMAT,[ClassId]);
  WriteStringWithFormatToStream(AStream, Format('%s%s class=%s %s',[vteOPENTAGPREFIX,tag,ClName,vteTAGPOSTFIX]),Level);
end;

// WriteCloseTagFormat
procedure WriteCloseTagFormat( AStream : TStream; Tag : string; Level : integer );
begin
  WriteStringWithFormatToStream(AStream, Format('%s%s%s',[vteCLOSETAGPREFIX,tag,vteTAGPOSTFIX]),Level);
end;

///////////////////////////
//
// TvtkeHTMLWriter
//
///////////////////////////
constructor TvteHTMLWriter.Create;
begin
  inherited;
  Styles := TStringList.Create;
  Styles.Add(vteTABLESTYLE);
end;

destructor TvteHTMLWriter.Destroy;
begin
  Styles.Free;
  inherited;
end;

procedure TvteHTMLWriter.SaveHeadFiles;
var
  Code : integer;
begin
  Code := GetFileAttributes(PChar(FilesDir));
  if (Code=-1) or (FILE_ATTRIBUTE_DIRECTORY and Code = 0) then
    CreateDir(FilesDir);
  SaveMainFile;
  SaveHeadFile;
end;

procedure TvteHTMLWriter.SaveMainFile;
begin
    WriteStringWithFormatToStream(FileStream,vteHTML_VERSION,0);
    WriteOpenTagFormat(FileStream,vteHTMLTAG,0);
    WriteOpenTagFormat(FileStream,vteHEADTAG,0);
    WriteOpenTagFormat(FileStream,vteTITLETAG,1);
    WriteStringWithFormatToStream(FileStream,MakeHTMLString(FName),2);
    WriteCloseTagFormat(FileStream,vteTITLETAG,1);
    WriteCloseTagFormat(FileStream,vteHEADTAG,0);
    WriteStringWithFormatToStream(FileStream,'<FRAMESET rows="39,*" border=0 width=0 frameborder=no framespacing=0>',1);
    WriteStringWithFormatToStream(FileStream,Format('<FRAME name="header" src="%s/header.htm" marginwidth=0 marginheight=0>',[DirName]),2);
    WriteStringWithFormatToStream(FileStream,Format('<FRAME name="sheet" src="%s/Sheet0.htm">',[DirName]),2);
    WriteStringWithFormatToStream(FileStream,'</FRAMESET>',1);

    WriteCloseTagFormat(FileStream,vteHTMLTAG,0);
end;

procedure TvteHTMLWriter.SaveHeadFile;
var
  fs : TFileStream;
  i : integer;
begin
  fs := TFileStream.Create(FilesDir+'\header.htm',fmCreate or fmShareDenyWrite);
  try
    WriteStringWithFormatToStream(fs,vteHTML_VERSION,0);
    WriteOpenTagFormat(fs,vteHTMLTAG,0);
    WriteOpenTagFormat(fs,vteHEADTAG,0);
    WriteOpenTagFormat(fs,vteTITLETAG,1);
    WriteStringWithFormatToStream(fs,MakeHTMLString(FName),2);
    WriteCloseTagFormat(fs,vteTITLETAG,1);
    WriteOpenTagFormat(fs,vteSTYLETAG,0);
    WriteStringWithFormatToStream(fs,'<!--'#13#10'A { text-decoration:none; color:#000000; font-size:9pt; } A:Active { color : #0000E0}'#13#10'-->',1);
    WriteCloseTagFormat(fs,vteSTYLETAG,0);
    WriteCloseTagFormat(fs,vteHEADTAG,0);
    WriteStringWithFormatToStream(fs,'<body topmargin=0 leftmargin=0 bgcolor="#808080">',0);
    WriteStringWithFormatToStream(fs,'<table border=0 cellspacing=1 height=100%>',0);
    WriteStringWithFormatToStream(fs,'<tr height=10><TD>',1);
    WriteStringWithFormatToStream(fs,'<tr>',1);
    for i := 0 to FWorkBook.SheetsCount-1 do
    begin
      WriteStringToStream(fs,Format('<td bgcolor="#FFFFFF" nowrap><b><small><small>&nbsp;<A href="Sheet%d.htm" target=sheet><font face="Arial">%s</FONT></A>&nbsp;</small></small></b></td>',
        [i,TvteXLSWorksheet(FWorkBook.Sheets[i]).Title]));
    end;
    WriteCloseTagFormat(fs,vteROWTAG,0);
    WriteCloseTagFormat(fs,vteTABLETAG,0);
    WriteCloseTagFormat(fs,vteBODYTAG,0);
    WriteCloseTagFormat(fs,vteHTMLTAG,0);
  finally
    fs.Free;
  end;
end;

procedure TvteHTMLWriter.WriteStyles;
var
  i : integer;
begin
  WriteOpenTagFormat(FileStream,vteSTYLETAG,2);
  for i := 0 to Styles.Count-1 do
    WriteStringToStream(FileStream,Format('.'+vteSTYLEFORMAT+' { %s } '#13#10,[i,Styles[i]]));
  WriteCloseTagFormat(FileStream,vteSTYLETAG,2);
end;

procedure TvteHTMLWriter.WriteRowTag(Sheet : TvteXLSWorksheet; RowIndex,Level : integer);
var
  Row : TvteXLSRow;
begin
  Row := nil;
  if RowIndex >= 0 then
    Row := Sheet.FindRow(RowIndex);
  if Row = nil then
    WriteOpenTagFormat(FileStream,vteROWTAG,Level)
  else
    WriteStringWithFormatToStream(FileStream,Format('%s%s style="%s:%dpt" %s',[vteOPENTAGPREFIX,vteROWTAG,vteHEIGHTATTRIBUTE,Row.Height,vteTAGPOSTFIX]),Level);
end;

procedure TvteHTMLWriter.WriteCellTag(Sheet : TvteXLSWorksheet; RowIndex,ColumnIndex,Level : integer);
var
  S : string;
  Col : TvteXLSCol;
begin
  S := vteOPENTAGPREFIX+vteCELLTAG;
  if (RowIndex = MinPos) and (ColumnIndex >= 0) then
  begin
    Col := Sheet.FindCol(ColumnIndex);
    if Col <> nil then
      S := S + Format(' style="%s:%dem"',[vteWIDTHATTRIBUTE,Col.Width div 256]);
  end;
  if (RowIndex >= 0) and (ColumnIndex >= 0) and (HtmlCells^[RowIndex*ColCount+ColumnIndex].Range <> nil) then
  begin
    S := S + GetCellTagString(HtmlCells^[RowIndex*ColCount+ColumnIndex].Range);
    S := S + ' CLASS='+Format(vteSTYLEFORMAT,[HtmlCells^[RowIndex*ColCount+ColumnIndex].StyleId]);
  end;
  S := S + vteTAGPOSTFIX;
  WriteStringWithFormatToStream(FileStream,S,Level);
end;

procedure TvteHTMLWriter.AddRange(Range: TvteXLSRange);
var
  i,j : integer;
  StStr : string;
  StIndex : integer;
begin
  with Range do
  begin
    StStr := GenStyle(Range);
    StIndex := Styles.IndexOf(StStr);
    if StIndex < 0 then
    begin
      Styles.Add(StStr);
      StIndex := Styles.Count-1;
    end;
    for i := Place.Top to Place.Bottom do
      for j := Place.Left to Place.Right do
      begin
        if (i = Place.Top) and (j = Place.Left) then
        begin
          HtmlCells^[i*ColCount+j].Range := Range;
          HtmlCells^[i*ColCount+j].StyleId := StIndex;
        end
        else
        begin
          SpansPresent := True;
          HtmlCells^[i*ColCount+j].Hide := 1;
        end;
      end;
  end;
end;

function Getfont_family(Font: TFont) : string;
begin
  Result := Font.Name
end;

function Getfont_size(Font: TFont) : word;
begin
  Result := Font.Size
end;

function Getfont_weight(Font: TFont) : string;
begin
  if fsBold in Font.Style then
    Result := vteFONT_BOLD
  else
    Result := vteFONT_NORMAL;
end;

function Getfont_style(Font: TFont) : string;
begin
  if fsItalic in Font.Style then
    Result := vteFONT_ITALIC
  else
    Result := vteFONT_NORMAL;
end;

function GetText_decoration(Font: TFont) : string;
begin
  Result := '';
  if fsUnderline in Font.Style then
    Result := vteFONT_UNDERLINE;
  if fsStrikeout in Font.Style then
  begin
    if Result <> '' then
      Result := Result + ' ';
    Result := Result + vteFONT_STRIKE;
  end;
  if Result = '' then
    Result := vteFONT_NONE;
end;

function GetColor(Color : TColor): string;
var
  r,g,b : PByte;
begin
  r := @Color;
  g := @Color;
  Inc(g,1);
  b := @Color;
  Inc(b,2);
  Result := Format('#%.2x%.2x%.2x', [r^,g^,b^]);
end;

function GetVAlign(Align : TvteXLSVerticalAlignmentType) : string;
var
  Val : string;
begin
  if Align = vtexlVAlignJustify then
    Result := ''
  else
  begin
    Result := vteVALIGN+':';
    case Align of
      vtexlVAlignTop : Val := vteTEXTTOP;
      vtexlVAlignCenter : Val := vteMiddle;
      vtexlVAlignBottom : Val := vteTEXTBOTTOM;
    end;
    Result := Result+Val+';';
  end;
end;

function GetTextAlign(Align : TvteXLSHorizontalAlignmentType) : string;
var
  Val : string;
begin
  if not (Align in [vtexlHAlignLeft,vtexlHAlignCenter,vtexlHAlignRight,vtexlHAlignJustify]) then
    Result := ''
  else
  begin
    Result := vteTEXTALIGN+':';
    case Align of
      vtexlHAlignLeft: Val := vteLEFT;
      vtexlHAlignCenter: Val := vteCENTER;
      vtexlHAlignRight:  Val := vteRIGHT;
      vtexlHAlignJustify: Val := vteJustify;
    end;
    Result := Result+Val+';';
  end;
end;

// Returns the background color of style string by the Range
function GetBackgroundColor(Range : TvteXLSRange): string;
begin
  if Range.FillPattern = vtefpNone then
    Result := ''
  else
    Result := vteBackgroundColor+':'+GetColor(Range.ForegroundFillPatternColor)+';';
end;

function GetBorderId(Border : TvteXLSBorderType) : string;
begin
  if (Border >= vtexlEdgeBottom) and (Border <= vtexlEdgeTop) then
    Result := aHtmlCellBorders[Border]
  else
    Result := '';
end;

// Returns the border line style part of style string by the Range
function GetLineStyle(BorderLineStyle : TvteXLSLineStyleType): string;
begin
  Result := aBorderLineStyles[BorderLineStyle];
end;

// Returns the borders part of style string by the Range
function GetBorders(Range : TvteXLSRange): string;
var
  i : integer;
  Eq : boolean;
  lt : TvteXLSLineStyleType;
  lc : TColor;
begin
  Result := '';
  Eq := True;
  for i:=integer(vtexlEdgeBottom) to integer(High(TvteXLSBorderType)) do
  begin
    if (i > integer(vtexlEdgeBottom)) and(
       (Range.Borders[TvteXLSBorderType(i-1)].LineStyle <> Range.Borders[TvteXLSBorderType(i)].LineStyle) or
       (Range.Borders[TvteXLSBorderType(i-1)].Color <> Range.Borders[TvteXLSBorderType(i)].Color) or
       (Range.Borders[TvteXLSBorderType(i-1)].Weight <> Range.Borders[TvteXLSBorderType(i)].Weight)) then
      Eq := false;
    lt := Range.Borders[TvteXLSBorderType(i)].LineStyle;
    lc := Range.Borders[TvteXLSBorderType(i)].Color;
    if lt <> vtelsNone then
      Result := Result+'border-'+GetBorderId(TvteXLSBorderType(i))+': '+GetLineStyle(lt)+' '+GetColor(lc)+';';
  end;
  if Eq then
      Result := 'border:'+GetLineStyle(lt)+' '+GetColor(lc)+';';
end;

// TvteHTMLWriter.GenStyle
// Returns Style string for given Range
function TvteHTMLWriter.GenStyle(Range: TvteXLSRange) : string;
begin
  Result := Format('font-family : ''%s''; font-size : %dpt; font-weight : %s; font-style : %s;  text-decoration : %s ; color : %s; %s %s %s %s',
    [Getfont_family(Range.Font),
     Getfont_size(Range.Font),
     Getfont_weight(Range.Font),
     Getfont_style(Range.Font),
     Gettext_decoration(Range.Font),
     GetColor(Range.Font.Color),
     GetVAlign(Range.VerticalAlignment),
     GetTextAlign(Range.HorizontalAlignment),
     GetBackgroundColor(Range),
     GetBorders(Range)]);
end;

// TvteHTMLWriter.GetSheetFileName
// Returns FileName for Sheet by page number of Sheet
function TvteHTMLWriter.GetSheetFileName(SheetNumber : integer) : string;
begin
  Result := Format('%s/Sheet%d%s',[FilesDir,SheetNumber,'.htm']);
end;

procedure TvteHTMLWriter.InitStrings;
begin
  FileExt := ExtractFileExt(FFileName);
  FName := Copy(FFileName,1,Length(FFileName)-Length(FileExt));
  FilesDir := FName+'_files';
  DirName := ExtractFileName(FilesDir);
end;

// TvteHTMLWriter.Save
// Save Workbook with HTML format
procedure TvteHTMLWriter.Save(WorkBook : TvteXLSWorkbook; const FileName : string);
var
  i : integer;
  Writer : TvteHTMLWriter;
begin
  FFileName := FileName;
  InitStrings;
  FileStream := TFileStream.Create(FileName,fmCreate or fmShareDenyWrite);
  try

    FWorkBook := WorkBook;
    SaveHeadFiles;
    for i := 0 to WorkBook.SheetsCount - 1 do
    begin
      Writer := TvteHTMLWriter.Create;
      try
        Writer.SaveSheet(TvteXLSWorksheet(WorkBook.Sheets[i]),GetSheetFileName(i));
      finally
        Writer.Free;
      end;
    end;
  finally
    FileStream.Free;
  end;
end;

// TvteHTMLWriter.SaveSheet
// Saves Sheet with HTML format
procedure TvteHTMLWriter.SaveSheet(Sheet :TvteXLSWorksheet; FileName : string);
var
  i,j : integer;
begin
  FileStream := TFileStream.Create(FileName,fmCreate or fmShareDenyWrite);
  try
    with Sheet do
    begin
        SpansPresent := false;
        RowCount := Dimensions.Bottom+1;
        ColCount := Dimensions.Right+1;
        HtmlCells := AllocMem(RowCount*ColCount*SizeOf(TvteHtmlCell));
        try
          for i := 0 to RangesCount - 1 do
            Self.AddRange(RangeByIndex[i]);

          if SpansPresent then
            MinPos := -1
          else
            MinPos := 0;
          WriteStringWithFormatToStream(FileStream,vteHTML_VERSION,0);
          WriteOpenTagFormat(FileStream,vteHTMLTAG,0);
          WriteOpenTagFormat(FileStream,vteHEADTAG,0);
          WriteOpenTagFormat(FileStream,vteTITLETAG,1);
          WriteStringWithFormatToStream(FileStream,MakeHTMLString(Sheet.Title),2);
          WriteCloseTagFormat(FileStream,vteTITLETAG,1);
          WriteStyles;
          WriteCloseTagFormat(FileStream,vteHEADTAG,0);
          WriteOpenTagFormat(FileStream,vteBODYTAG,0);
          WriteOpenTagFormat(FileStream,vteFORMTAG,0);
          WriteOpenTagClassFormat(FileStream,vteTABLETAG,0,0);
          for i := MinPos to RowCount-1 do
          begin
            WriteRowTag(Sheet,i,1);
            for j := MinPos to ColCount-1 do
            begin
              if HtmlCells^[i*ColCount+j].Hide = 0 then
              begin
                WriteCellTag(Sheet,i,j,2);
                if (i >= 0) and (j >= 0) and (HtmlCells^[i*ColCount+j].Range <> nil) then
                  WriteStringWithFormatToStream(FileStream,'<PRE>'+MakeHTMLString(HtmlCells^[i*ColCount+j].Range.Value)+'</PRE>',2);
                WriteCloseTagFormat(FileStream,vteCELLTAG,1);
              end;
            end;
            WriteCloseTagFormat(FileStream,vteROWTAG,1);
          end;
          WriteCloseTagFormat(FileStream,vteTABLETAG,1);
          WriteCloseTagFormat(FileStream,vteFORMTAG,0);
          WriteCloseTagFormat(FileStream,vteBODYTAG,0);
          WriteCloseTagFormat(FileStream,vteHTMLTAG,0);
        finally
          FreeMem(HtmlCells);
        end;
      end;
  finally
    FileStream.Free;
  end;
end;

// TvteHTMLWriter.GetCellTagString
// Returns a line with rowspan, colspan attributes for formation tag
// TD according to Range.Place
function TvteHTMLWriter.GetCellTagString(Range: TvteXLSRange) : string;
var
  ColSpan, RowSpan : integer;
begin
  Result := '';
  with Range do
  begin
    RowSpan := Place.Bottom - Place.Top + 1;
    ColSpan := Place.Right - Place.Left + 1;
  end;
  if RowSpan > 1 then
    Result := Result + Format(' %s=%d',[vteROWSPANATTRIBUTE,rowspan]);
  if ColSpan > 1 then
    Result := Result + Format(' %s=%d',[vteCOLSPANATTRIBUTE,colspan]);
end;












procedure StringToWideChar(const Source: string; Dest: PWideChar;
  DestSize: Integer);
begin
MultiByteToWideChar(0,0,PChar(Source),Length(Source),Dest,DestSize);
end;

procedure wbiff(Stream : TStream; code : word; buf : pointer; size : integer);
var
  sz : word;
begin
repeat
  Stream.Write(code,2);
  sz := Min(size,MaxBiffRecordSize-4);
  Stream.Write(sz,2);
  if sz>0 then
    begin
      Stream.Write(buf^,sz);
      buf := PChar(buf)+sz;
      size := size-sz;
      code := b8_CONTINUE;
    end
until size=0;
end;

procedure wbiffFont(Stream : TStream; f : TFont; ColorPaletteIndex : word);
var
  font : pb8FONT;
  lf : TLogFont;
  lfont : integer;
begin
lfont := Length(f.Name)*sizeof(WideChar);
font := AllocMem(sizeof(rb8FONT)+lfont);
try
  GetObject(f.Handle, SizeOf(TLogFont), @lf);
  StringToWideChar(f.Name,PWideChar(PChar(font)+sizeof(rb8FONT)),lfont);
  font.dyHeight := f.Size*20;
  if fsItalic in f.Style then
    font.grbit := font.grbit or b8_FONT_grbit_fItalic;
  if fsStrikeout in f.Style then
    font.grbit := font.grbit or b8_FONT_grbit_fStrikeout;
  font.icv := ColorPaletteIndex; 
  if fsBold in f.Style then
    font.bls := $3E8  // from MSDN
  else
    font.bls := $64;  // from MSDN
  if fsUnderline in f.Style then
    font.uls := 1;  // from MSDN
  font.bFamily := lf.lfPitchAndFamily;
  font.bCharSet := lf.lfCharSet;
  font.cch := Length(f.Name);
  font.cchgrbit := $01;

  wbiff(Stream,b8_FONT,font,sizeof(rb8FONT)+lfont);
finally
  FreeMem(font);
end;
end;

procedure wbiffFormat(Stream : TStream; const FormatString : string; FormatCode : word);
var
  lformat : integer;
  format : pb8FORMAT;
begin
lformat := Length(FormatString)*sizeof(WideChar);
format := AllocMem(sizeof(rb8FORMAT)+lformat);
try
  StringToWideChar(FormatString,PWideChar(PChar(format)+sizeof(rb8FORMAT)),lformat);
  format.ifmt := FormatCode;
  format.cch := Length(FormatString);
  format.cchgrbit := $01;
  wbiff(Stream,b8_FORMAT,format,sizeof(rb8FORMAT)+lformat);
finally
  FreeMem(format);
end;
end;

function HexStringToString(const s : string) : string;
var
  b1 : string;
  i,ls : integer;
begin
Result := '';
ls := length(s);
i := 1;
while i<=ls do
  begin
    while (i<=ls) and not(s[i] in ['0'..'9','a'..'f','A'..'F']) do Inc(i);
    if i>ls then break;
    b1 := '';
    while (i<=ls) and (s[i] in ['0'..'9','a'..'f','A'..'F']) do
      begin
        b1 := b1+s[i];
        Inc(i);
      end;
    if b1<>'' then
      Result := Result+char(StrToInt('$'+b1));
    if (b1='') or (i>ls) then break;
  end;
end;

procedure wbiffHexString(Stream : TStream; const HexString : string);
var
  s : string;
begin
s := HexStringToString(HexString);
Stream.Write(s[1],Length(s));
end;

////////////////////////////////
//
// TvteExcelWriter
//
////////////////////////////////
constructor TvteExcelWriter.Create;
begin
  FUsedColors := TList.Create;
end;

destructor TvteExcelWriter.Destroy;
begin
FUsedColors.Free;
inherited;
end;

procedure TvteExcelWriter.BuildFontList(l : TList);
var
  f : TFont;
  sh : TvteXLSWorksheet;
  ran : TvteXLSRange;
  i,j,k,n : integer;
begin
n := 0;
for i:=0 to FWorkBook.SheetsCount-1 do
  begin
    sh := FWorkBook.Sheets[i];
    for j:=0 to sh.RangesCount-1 do
      begin
        ran := sh.RangeByIndex[j];
        ran.ExportData := @FRangesRecs[n];
        f := ran.Font;
        k := 0;
        while (k<L.Count) and
              ((TFont(L[k]).Charset<>f.Charset) or
               (TFont(L[k]).Color<>f.Color) or
               (TFont(L[k]).Height<>f.Height) or
               (TFont(L[k]).Name<>f.Name) or
               (TFont(L[k]).Pitch<>f.Pitch) or
               (TFont(L[k]).Size<>f.Size) or
               (TFont(L[k]).Style<>f.Style)) do Inc(k);
        if k>=L.Count then
          begin
            k := L.Add(TFont.Create);
            TFont(L[k]).Assign(f);
          end;
        FRangesRecs[n].iFont := k+1;
        Inc(n);
      end;
  end;
end;

procedure TvteExcelWriter.BuildFormatList(sl : TStringList);
var
  sh : TvteXLSWorksheet;
  ran : TvteXLSRange;
  i,j,k,n,m : integer;
begin
n := sl.Count;
m := 0;
for i:=0 to FWorkBook.SheetsCount-1 do
  begin
    sh := FWorkBook.Sheets[i];
    for j:=0 to sh.RangesCount-1 do
      begin
        ran := sh.RangeByIndex[j];
        if ran.Format='' then
          FRangesRecs[m].iFormat := 0
        else
          begin
            k := sl.IndexOf(ran.Format);
            if k=-1 then
              k := sl.AddObject(ran.Format,pointer(sl.Count-n+$32));
            FRangesRecs[m].iFormat := integer(sl.Objects[k]);
          end;
        Inc(m);
      end;
  end;
end;

procedure TvteExcelWriter.BuildXFRecord(Range : TvteXLSRange; var XF : rb8XF; prr : pXLSRangeRec);
const
  aFillPattern : array [TvteXLSFillPattern] of integer = (0,-4105,9,16,-4121,18,17,-4124,-4125,-4126,15,-4128,13,11,14,12,10,1,-4162,-4166);
  aHorizontalAlignment : array [TvteXLSHorizontalAlignmentType] of integer =
                         (b8_XF_Opt2_alcGeneral,
                          b8_XF_Opt2_alcLeft,
                          b8_XF_Opt2_alcCenter,
                          b8_XF_Opt2_alcRight,
                          b8_XF_Opt2_alcFill,
                          b8_XF_Opt2_alcJustify,
                          b8_XF_Opt2_alcCenterAcrossSelection);
  aVerticalAlignment : array [TvteXLSVerticalAlignmentType] of integer =
                         (b8_XF_Opt2_alcVTop,
                          b8_XF_Opt2_alcVCenter,
                          b8_XF_Opt2_alcVBottom,
                          b8_XF_Opt2_alcVJustify);
  aWrapText : array [boolean] of integer = (0,b8_XF_Opt2_fWrap);
  aBorderLineStyle : array [TvteXLSLineStyleType] of word =
                     (b8_XF_Border_None,
                      b8_XF_Border_Thin,
                      b8_XF_Border_Medium,
                      b8_XF_Border_Dashed,
                      b8_XF_Border_Dotted,
                      b8_XF_Border_Thick,
                      b8_XF_Border_Double,
                      b8_XF_Border_Hair,
                      b8_XF_Border_MediumDashed,
                      b8_XF_Border_DashDot,
                      b8_XF_Border_MediumDashDot,
                      b8_XF_Border_DashDotDot,
                      b8_XF_Border_MediumDashDotDot,
                      b8_XF_Border_SlantedDashDot);
  function GetBorderColorIndex(b : TvteXLSBorderType) : integer;
  begin
  if Range.Borders[b].LineStyle=vtelsNone then
    Result := 0
  else
    Result := GetColorPaletteIndex(Range.Borders[b].Color)+8; // ??? + 8 - dont know
  end;
var
  DiagBorderLineStyle : TvteXLSLineStyleType;
  DiagBorderColorIndex : integer;
begin
ZeroMemory(@XF,sizeof(XF));
XF.ifnt := prr.iFont;
XF.ifmt := pXLSRangeRec(Range.ExportData).iFormat;
XF.Opt1 := $0001;//b8_XF_Opt1_fLocked or b8_XF_Opt1_fHidden;
XF.Opt2 := aHorizontalAlignment[Range.HorizontalAlignment] or
           aWrapText[Range.WrapText] or
           aVerticalAlignment[Range.VerticalAlignment];
XF.trot := Range.Rotation;
XF.Opt3 := b8_XF_Opt3_fAtrNum or
           b8_XF_Opt3_fAtrFnt or
           b8_XF_Opt3_fAtrAlc or
           b8_XF_Opt3_fAtrBdr or
           b8_XF_Opt3_fAtrPat;
if (Range.Place.Left<>Range.Place.Right) or (Range.Place.Top<>Range.Place.Bottom) then
  XF.Opt3 := XF.Opt3 or b8_XF_Opt3_fMergeCell;

// borders
XF.Borders1 := (aBorderLineStyle[Range.Borders[vtexlEdgeLeft].LineStyle]) or
               (aBorderLineStyle[Range.Borders[vtexlEdgeRight].LineStyle] shl 4) or
               (aBorderLineStyle[Range.Borders[vtexlEdgeTop].LineStyle] shl 8) or
               (aBorderLineStyle[Range.Borders[vtexlEdgeBottom].LineStyle] shl 12);
DiagBorderLineStyle := vtelsNone;
DiagBorderColorIndex := 0;
XF.Borders2 := 0;
if Range.Borders[vtexlDiagonalDown].LineStyle<>vtelsNone then
  begin
    XF.Borders2 := XF.Borders2 or $4000;
    DiagBorderLineStyle := Range.Borders[vtexlDiagonalDown].LineStyle;
    DiagBorderColorIndex := GetColorPaletteIndex(Range.Borders[vtexlDiagonalDown].Color)+8;
  end;
if Range.Borders[vtexlDiagonalUp].LineStyle<>vtelsNone then
  begin
    XF.Borders2 := XF.Borders2 or $8000;
    DiagBorderLineStyle := Range.Borders[vtexlDiagonalUp].LineStyle;
    DiagBorderColorIndex := GetColorPaletteIndex(Range.Borders[vtexlDiagonalUp].Color)+8;
  end;
XF.Borders2 := XF.Borders2 or
               (GetBorderColorIndex(vtexlEdgeLeft)) or
               (GetBorderColorIndex(vtexlEdgeRight) shl 7);
XF.Borders3 := (GetBorderColorIndex(vtexlEdgeTop)) or
               (GetBorderColorIndex(vtexlEdgeBottom) shl 7) or
               (DiagBorderColorIndex shl 14) or
               (aBorderLineStyle[DiagBorderLineStyle] shl 21) or
               (aFillPattern[Range.FillPattern] shl 26);
XF.Colors := GetColorPaletteIndex(Range.ForegroundFillPatternColor) or
             (GetColorPaletteIndex(Range.BackgroundFillPatternColor) shl 7); // colors for fill pattern
end;

procedure TvteExcelWriter.BuildXFList(l : TList);
var
  p : pointer;
  XF : rb8XF;
  sh : TvteXLSWorksheet;
  ran : TvteXLSRange;
  i,j,k,n : integer;
begin
n := 0;
for i:=0 to FWorkBook.SheetsCount-1 do
  begin
    sh := FWorkBook.Sheets[i];
    for j:=0 to sh.RangesCount-1 do
      begin
        ran := sh.RangeByIndex[j];
        BuildXFRecord(ran,XF,@FRangesRecs[n]);
        k := 0;
        while (k<l.Count) and not CompareMem(l[k],@XF,sizeof(rb8XF)) do Inc(k);
        if k>=l.Count then
          begin
            GetMem(p,sizeof(rb8XF));
            CopyMemory(p,@XF,sizeof(rb8XF));
            k := l.Add(p);
          end;
        FRangesRecs[n].iXF := k+15; // 15 - count of STYLE XF records
        Inc(n);
      end;
  end;
end;

function TvteExcelWriter.GetColorPaletteIndex(Color : TColor) : integer;

  function DefaultColorIndex(c : TColor) : integer;
  begin
  Result := 0;
  while (Result<MaxDefaultColors) and (aDefaultColors[Result]<>c) do Inc(Result);
  if Result>=MaxDefaultColors then
    Result := -1;
  end;
  
begin
if (Color and $80000000)<>0 then
  Color := GetSysColor(Color and $00FFFFFF);
if FUsedColors.IndexOf(pointer(Color))=-1 then
  FUsedColors.Add(pointer(Color));
Result := 0;
while (Result<XLSMaxColorsInPalette) and (FColorPalette[Result]<>Color) do Inc(Result);
if Result<XLSMaxColorsInPalette then exit; // color exist in current palette
Result := 0;
while Result<XLSMaxColorsInPalette do
  begin
    if (DefaultColorIndex(FColorPalette[Result])=-1) and
       (FUsedColors.IndexOf(pointer(FColorPalette[Result]))=-1) then
      begin
        // replace color in palette with new color
        FColorPalette[Result] := Color;
        FPaletteModified := true;
        exit;
      end;
    Inc(Result);
  end;
Result := 1; // return index to BLACK color
end;

function sort(Item1, Item2: Pointer): Integer;
begin
Result := TvteXLSRange(Item1).Place.Left-TvteXLSRange(Item2).Place.Left;
end;

procedure TvteExcelWriter.WriteRangeToStream(Stream : TStream; Range : TvteXLSRange; CurrentRow : integer; var IndexInCellsOffsArray : integer; var CellsOffs : Tb8DBCELLCellsOffsArray);
var
  blank : rb8BLANK;
  i,Left : integer;
  number : rb8NUMBER;
  mulblank : pb8MULBLANK;
  labelsst : rb8LABELSST;

  procedure AddToCellsOffsArray;
  begin
  if IndexInCellsOffsArray=0 then
    CellsOffs[IndexInCellsOffsArray] := Stream.Position
  else
    CellsOffs[IndexInCellsOffsArray] := Stream.Position-CellsOffs[IndexInCellsOffsArray-1];
  Inc(IndexInCellsOffsArray);
  end;
  
begin
Left := Range.Place.Left;
if CurrentRow=Range.Place.Top then
  begin
    AddToCellsOffsArray;
    // write data cell, check cell type
    case Range.CellDataType of
      vtecdtNumber:
        begin
          number.rw := CurrentRow;
          number.col := Range.Place.Left;
          number.ixfe := pXLSRangeRec(Range.ExportData).iXF;
          number.num := Range.Value;
          wbiff(Stream,b8_NUMBER,@number,sizeof(rb8NUMBER));
        end;
      vtecdtString:
        begin
          labelsst.rw := CurrentRow;
          labelsst.col := Range.Place.Left;
          labelsst.ixfe := pXLSRangeRec(Range.ExportData).iXF;
          labelsst.isst := pXLSRangeRec(Range.ExportData).iSST;
          wbiff(Stream,b8_LABELSST,@labelsst,sizeof(labelsst));
        end;
    end;
    Inc(Left);
  end;

// write blank cells
if Left<Range.Place.Right then
  begin
    AddToCellsOffsArray;
    mulblank := AllocMem(sizeof(rb8MULBLANK)+(Range.Place.Right-Left+1)*2+2);
    try
      mulblank.rw := CurrentRow;
      mulblank.colFirst := Left;
      for i:=0 to Range.Place.Right-Left do
        PWordArray(PChar(mulblank)+sizeof(rb8MULBLANK))^[i] := pXLSRangeRec(Range.ExportData).iXF;
      PWord(PChar(mulblank)+sizeof(rb8MULBLANK)+(Range.Place.Right-Left+1)*2)^ := Range.Place.Right;
      wbiff(Stream,b8_MULBLANK,mulblank,sizeof(rb8MULBLANK)+(Range.Place.Right-Left+1)*2+2);
    finally
      FreeMem(mulblank);
    end;
  end
else
  if Left=Range.Place.Right then
    begin
      AddToCellsOffsArray;
      blank.rw := CurrentRow;
      blank.col := Left;
      blank.ixfe := pXLSRangeRec(Range.ExportData).iXF;
      wbiff(Stream,b8_BLANK,@blank,sizeof(blank));
    end;
end;

procedure TvteExcelWriter.WriteSheetToStream(Stream : TStream; Sheet : TvteXLSWorksheet);
var
  s : string;
  bof : rb8BOF;
  calcmode : rb8CALCMODE;
  calccount : rb8CALCCOUNT;
  refmode : rb8REFMODE;
  iteration : rb8ITERATION;
  saverecalc : rb8SAVERECALC;
  printheaders : rb8PRINTHEADERS;
  printgridlines : rb8PRINTGRIDLINES;
  gridset : rb8GRIDSET;
  guts : rb8GUTS;
  defaultrowheight : rb8DEFAULTROWHEIGHT;
  wsbool : rb8WSBOOL;
  hcenter : rb8HCENTER;
  vcenter : rb8VCENTER;
  defcolwidth : rb8DEFCOLWIDTH;
  dimensions : rb8DIMENSIONS;
  window2 : rb8WINDOW2;
  selection : pb8SELECTION;
  header : pb8HEADER;
  footer : pb8FOOTER;

  l : TList;
  bc,i,j : integer;
  ran : TvteXLSRange;
  rw : TvteXLSRow;
  row : rb8ROW;

  index : pb8INDEX;
  INDEXOffs : integer;
  BlocksInSheet : integer;
  IndexInDBCELLsOffs : integer;

  dbcell : rb8DBCELLfull;
  IndexInCellsOffsArray : integer;

  ms : TMemoryStream;
  FirstRowOffs,SecondRowOffs : integer;
  merge : pb8MERGE;
  colinfo : rb8COLINFO;

  leftmargin : rb8LEFTMARGIN;
  rightmargin : rb8RIGHTMARGIN;
  topmargin : rb8TOPMARGIN;
  bottommargin : rb8BOTTOMMARGIN;
  setup : rb8SETUP;
begin
ZeroMemory(@bof,sizeof(bof));
bof.vers := b8_BOF_vers;
bof.dt := b8_BOF_dt_Worksheet;
bof.rupBuild := b8_BOF_rupBuild_Excel97;
bof.rupYear := b8_BOF_rupYear_Excel07;
wbiff(Stream,b8_BOF,@bof,sizeof(bof));

if (Sheet.Dimensions.Bottom<>-1) and (Sheet.Dimensions.Top<>-1) then
  begin
    BlocksInSheet := (Sheet.Dimensions.Bottom-Sheet.Dimensions.Top+1) div XLSMaxRowsInBlock;
    if (Sheet.Dimensions.Bottom=Sheet.Dimensions.Top) or (((Sheet.Dimensions.Bottom-Sheet.Dimensions.Top+1) mod XLSMaxRowsInBlock)<>0) then
      Inc(BlocksInSheet);
  end
else
  BlocksInSheet := 0;

index := AllocMem(sizeof(rb8INDEX)+BlocksInSheet*4);
try
  if (Sheet.Dimensions.Bottom<>-1) and (Sheet.Dimensions.Top<>-1) then
    begin
      index.rwMic := Sheet.Dimensions.Top;
      index.rwMac := Sheet.Dimensions.Bottom+1;
    end;
  INDEXOffs := Stream.Position;
  IndexInDBCELLsOffs := 0;
  wbiff(Stream,b8_INDEX,index,sizeof(rb8INDEX)+BlocksInSheet*4); // corrected later

  calcmode.fAutoRecalc := 1; // automatic recalc
  wbiff(Stream,b8_CALCMODE,@calcmode,sizeof(calcmode));
  
  calccount.cIter := $0064; // see in biffview
  wbiff(Stream,b8_CALCCOUNT,@calccount,sizeof(calccount));
  
  refmode.fRefA1 := $0001; // 1 for A1 mode
  wbiff(Stream,b8_REFMODE,@refmode,sizeof(refmode));
  
  iteration.fIter := $0000; // 1 see in biffview
  wbiff(Stream,b8_ITERATION,@iteration,sizeof(iteration));
  
  // DELTA
  s := HexStringToString('10 00 08 00 fc a9 f1 d2 4d 62 50 3f');
  Stream.Write(s[1],length(s));

  saverecalc.fSaveRecalc := $0001; // see in biffview
  wbiff(Stream,b8_SAVERECALC,@saverecalc,sizeof(saverecalc));

  if Sheet.PageSetup.PrintHeaders then
    printheaders.fPrintRwCol := 1
  else
    printheaders.fPrintRwCol := 0;
  wbiff(Stream,b8_PRINTHEADERS,@printheaders,sizeof(printheaders));

  if Sheet.PageSetup.PrintGridLines then
    printgridlines.fPrintGrid := 1
  else
    printgridlines.fPrintGrid := 0;
  wbiff(Stream,b8_PRINTGRIDLINES,@printgridlines,sizeof(printgridlines));

  gridset.fGridSet := $0001; // see in biffview
  wbiff(Stream,b8_GRIDSET,@gridset,sizeof(gridset));
  
  ZeroMemory(@guts,sizeof(guts));  // all to zero see in biffview
  wbiff(Stream,b8_GUTS,@guts,sizeof(guts));
  
  defaultrowheight.grbit := $0000; // see in biffview
  defaultrowheight.miyRw := xlsDefaultRowHeight; // see in biffview
  wbiff(Stream,b8_DEFAULTROWHEIGHT,@defaultrowheight,sizeof(defaultrowheight));
  
  wsbool.grbit := $04C1; // see in biffview
  wbiff(Stream,b8_WSBOOL,@wsbool,sizeof(wsbool));

  s := '';
  if Sheet.PageSetup.LeftHeader<>'' then
    s := s+'&L'+Sheet.PageSetup.LeftHeader;
  if Sheet.PageSetup.CenterHeader<>'' then
    s := s+'&C'+Sheet.PageSetup.CenterHeader;
  if Sheet.PageSetup.RightHeader<>'' then
    s := s+'&R'+Sheet.PageSetup.RightHeader;
  if s<>'' then
    begin
      GetMem(header,sizeof(rb8HEADER)+Length(s)*2);
      try
        header.cch := Length(s);
        header.cchgrbit := 1;
        StringToWideChar(s,PWideChar(PChar(header)+sizeof(rb8HEADER)),Length(s)*2);
        wbiff(Stream,b8_HEADER,header,sizeof(rb8HEADER)+Length(s)*2);
      finally
        FreeMem(header);
      end;
    end;

  s := '';
  if Sheet.PageSetup.LeftFooter<>'' then
    s := s+'&L'+Sheet.PageSetup.LeftFooter;
  if Sheet.PageSetup.CenterFooter<>'' then
    s := s+'&C'+Sheet.PageSetup.CenterFooter;
  if Sheet.PageSetup.RightFooter<>'' then
    s := s+'&R'+Sheet.PageSetup.RightFooter;
  if s<>'' then
    begin
      GetMem(footer,sizeof(rb8FOOTER)+Length(s)*2);
      try
        footer.cch := Length(s);
        footer.cchgrbit := 1;
        StringToWideChar(s,PWideChar(PChar(footer)+sizeof(rb8HEADER)),Length(s)*2);
        wbiff(Stream,b8_FOOTER,footer,sizeof(rb8FOOTER)+Length(s)*2);
      finally
        FreeMem(footer);
      end;
    end;

  if Sheet.PageSetup.CenterHorizontally then
    hcenter.fHCenter := 1
  else
    hcenter.fHCenter := 0;
  wbiff(Stream,b8_HCENTER,@hcenter,sizeof(hcenter));

  if Sheet.PageSetup.CenterVertically then
    vcenter.fVCenter := 1
  else
    vcenter.fVCenter := 0;
  wbiff(Stream,b8_VCENTER,@vcenter,sizeof(vcenter));

  leftmargin.num := Sheet.PageSetup.LeftMargin;
  wbiff(Stream,b8_LEFTMARGIN,@leftmargin,sizeof(rb8LEFTMARGIN));
  rightmargin.num := Sheet.PageSetup.RightMargin;
  wbiff(Stream,b8_RIGHTMARGIN,@rightmargin,sizeof(rb8RIGHTMARGIN));
  topmargin.num := Sheet.PageSetup.TopMargin;
  wbiff(Stream,b8_TOPMARGIN,@topmargin,sizeof(rb8TOPMARGIN));
  bottommargin.num := Sheet.PageSetup.BottomMargin;
  wbiff(Stream,b8_BOTTOMMARGIN,@bottommargin,sizeof(rb8BOTTOMMARGIN));

  // SETUP - skipped
  ZeroMemory(@setup,sizeof(rb8SETUP));
  setup.iPaperSize := word(Sheet.PageSetup.PaperSize);
  setup.iPageStart := Sheet.PageSetup.FirstPageNumber;
  setup.iFitWidth := Sheet.PageSetup.FitToPagesWide;
  setup.iFitHeight := Sheet.PageSetup.FitToPagesTall;
  setup.numHdr := Sheet.PageSetup.HeaderMargin;
  setup.numFtr := Sheet.PageSetup.FooterMargin;
  setup.iCopies := Sheet.PageSetup.Copies;
  setup.iScale := Sheet.PageSetup.Zoom;
//  setup.grbit := b8_SETUP_fNoPls;
  if Sheet.PageSetup.Order=vtexlOverThenDown then
    setup.grbit := setup.grbit or b8_SETUP_fLeftToRight;
  if Sheet.PageSetup.Orientation=vtexlPortrait then
    setup.grbit := setup.grbit or b8_SETUP_fLandscape;
  if Sheet.PageSetup.BlackAndWhite then
    setup.grbit := setup.grbit or b8_SETUP_fNoColor;
  if Sheet.PageSetup.Draft then
    setup.grbit := setup.grbit or b8_SETUP_fDraft;
  if Sheet.PageSetup.PrintNotes then
    setup.grbit := setup.grbit or b8_SETUP_fNotes;
  if Sheet.PageSetup.FirstPageNumber<>1 then
    setup.grbit := setup.grbit or b8_SETUP_fUsePage;
  wbiff(Stream,b8_SETUP,@setup,sizeof(rb8SETUP));

  defcolwidth.cchdefColWidth := XLSDefaultColumnWidthInChars; // see in biffview
  wbiff(Stream,b8_DEFCOLWIDTH,@defcolwidth,sizeof(defcolwidth));
  
  // now write columns info
  for i:=0 to Sheet.ColsCount-1 do
    with Sheet.ColByIndex[i] do
      begin
        ZeroMemory(@colinfo,sizeof(colinfo));
        colinfo.colFirst := Ind;
        colinfo.colLast := Ind;
        colinfo.coldx := Width;
        wbiff(Stream,b8_COLINFO,@colinfo,sizeof(colinfo));
      end;

  ZeroMemory(@dimensions,sizeof(dimensions));
  if (Sheet.Dimensions.Left<>-1) and
     (Sheet.Dimensions.Right<>-1) and
     (Sheet.Dimensions.Top<>-1) and
     (Sheet.Dimensions.Bottom<>-1) then
    begin
      dimensions.rwMic := Sheet.Dimensions.Top;
      dimensions.rwMac := Sheet.Dimensions.Bottom+1;
      dimensions.colMic := Sheet.Dimensions.Left;
      dimensions.colMac := Sheet.Dimensions.Right+1;
    end;
  wbiff(Stream,b8_DIMENSIONS,@dimensions,sizeof(dimensions));

  // here must be writted cells
  if (Sheet.Dimensions.Top<>-1) and (Sheet.Dimensions.Bottom<>-1) then
    begin
      l := TList.Create;
      ms := TMemoryStream.Create;
      try
        bc := 0;
        FirstRowOffs := 0;
        SecondRowOffs := 0;
        for i:=Sheet.Dimensions.Top to Sheet.Dimensions.Bottom do
          begin
            // finding all regions what placed over row [i]
            l.Clear;
            for j:=0 to Sheet.RangesCount-1 do
              begin
                ran := Sheet.RangeByIndex[j];
                if (ran.Place.Top<=i) and (i<=ran.Place.Bottom) then
                  l.Add(ran);
              end;
            l.Sort(sort);
            // write row i to file
            if bc=0 then
              FirstRowOffs := Stream.Position;
            row.rw := i;
            if l.Count>0 then
              begin
                row.colMic := TvteXLSRange(l[0]).Place.Left;
                row.colMac := TvteXLSRange(l[l.Count-1]).Place.Right+1;
              end
            else
              begin
                row.colMic := 0;
                row.colMac := 0;
              end;
            // to determine row height find TvteXLSRow, if not found
            // simple set default height
            rw := Sheet.FindRow(i);
            if rw=nil then
              begin
                row.miyRw := XLSDefaultRowHeight;
                row.grbit := 0;
              end
            else
              begin
                row.miyRw := rw.Height*20;
                row.grbit := b8_ROW_grbit_fUnsynced;
              end;
            wbiff(Stream,b8_ROW,@row,sizeof(row));
            if bc=0 then
              SecondRowOffs := Stream.Position;

            // write row cells to temporary memorystream,
            // also save cell offset from SecondRowOffs to CellsOffs
            IndexInCellsOffsArray := 0;
            for j:=0 to l.Count-1 do
              WriteRangeToStream(ms,TvteXLSRange(l[j]),i,IndexInCellsOffsArray,dbcell.CellsOffs);

            Inc(bc);
            if (bc=XLSMaxRowsInBlock) or (i=Sheet.Dimensions.Bottom) then
              begin
                dbcell.CellsOffs[0] := Stream.Position-SecondRowOffs;
                // write from temporary memorystream to Stream
                ms.SaveToStream(Stream);
                // rows block ended - write DBCELL
                // save DBCell offset
                PCardinalArray(PChar(index)+sizeof(rb8INDEX))^[IndexInDBCELLsOffs] := Stream.Position-FBOFOffs;
                Inc(IndexInDBCELLsOffs);

                dbcell.dbRtrw := Stream.Position-FirstRowOffs;
                wbiff(Stream,b8_DBCELL,@dbcell,sizeof(rb8DBCELL)+IndexInCellsOffsArray*2);
                // reinit vars
                ms.Clear;
                bc := 0;
              end;
          end;
      finally
        l.Free;
        ms.Free;
      end;

      // correct index record
      Stream.Position := INDEXOffs;
      wbiff(Stream,b8_INDEX,index,sizeof(rb8INDEX)+BlocksInSheet*4);
      Stream.Seek(0,soFromEnd);
    end;
finally
  FreeMem(index);
end;

ZeroMemory(@window2,sizeof(window2));
window2.grbit := b8_WINDOW2_grbit_fPaged or // $06B6 - this value see in biffview
                 b8_WINDOW2_grbit_fDspGuts or
                 b8_WINDOW2_grbit_fDspZeros or
                 b8_WINDOW2_grbit_fDefaultHdr or
                 b8_WINDOW2_grbit_fDspGrid or
                 b8_WINDOW2_grbit_fDspRwCol;
if Sheet.IndexInWorkBook=0 then
  window2.grbit := window2.grbit+b8_WINDOW2_grbit_fSelected;
window2.rwTop := 0;
window2.colLeft := 0;
window2.icvHdr := $00000040;
window2.wScaleSLV := 0;
window2.wScaleNormal := 0;
wbiff(Stream,b8_WINDOW2,@window2,sizeof(window2));

selection := AllocMem(sizeof(rb8SELECTION)+6);
try
  selection.pnn := 3; // see in biffview
  selection.cref := 1;
  wbiff(Stream,b8_SELECTION,selection,sizeof(rb8SELECTION)+6);
finally
  FreeMem(selection);
end;

// write data about merge ranges
if Sheet.RangesCount>0 then
  begin
    j := 0;
    for i:=0 to Sheet.RangesCount-1 do
      begin
        ran := Sheet.RangeByIndex[i];
        if (ran.Place.Left<>ran.Place.Right) or
           (ran.Place.Top<>ran.Place.Bottom) then
          Inc(j);
      end;
    if j>0 then
      begin
        merge := AllocMem(sizeof(rb8MERGE)+j*8);
        try
          merge.cnt := j;
          j := 0;
          for i:=0 to Sheet.RangesCount-1 do
            begin
              ran := Sheet.RangeByIndex[i];
              if (ran.Place.Left<>ran.Place.Right) or (ran.Place.Top<>ran.Place.Bottom) then
                begin
                  with pb8MERGErec(PChar(merge)+sizeof(rb8MERGE)+j*8)^ do
                    begin
                      top := ran.Place.Top;
                      bottom := ran.Place.Bottom;
                      left := ran.Place.Left;
                      right := ran.Place.Right;
                    end;
                  Inc(j);
                end;
            end;
          wbiff(Stream,b8_MERGE,merge,sizeof(rb8MERGE)+j*8);
        finally
          FreeMem(merge);
        end;
      end;
  end;

wbiff(Stream,b8_EOF,nil,0);
end;

procedure TvteExcelWriter.SaveAsBIFFToStream(WorkBook : TvteXLSWorkbook; Stream : TStream);
var
  i,j,k,m,ltitle : integer;
  s : string;
  l : TList;
  sl : TStringList;
  sh : TvteXLSWorksheet;
  buf : pointer;
  bof : rb8BOF;
  mms : rb8MMS;
  codepage : rb8CODEPAGE;
  interfachdr : rb8INTERFACHDR;
  fngroupcount : rb8FNGROUPCOUNT;
  windowprotect : rb8WINDOWPROTECT;
  protect : rb8PROTECT;
  password : rb8PASSWORD;
  backup : rb8BACKUP;
  hideobj : rb8HIDEOBJ;
  s1904 : rb81904;
  precision : rb8PRECISION;
  bookbool : rb8BOOKBOOL;
  writeaccess : rb8WRITEACCESS;
  doublestreamfile : rb8DOUBLESTREAMFILE;
  prot4rev : rb8PROT4REV;
  prot4revpass : rb8PROT4REVPASS;
  window1 : rb8WINDOW1;
  refreshall : rb8REFRESHALL;
  useselfs : rb8USESELFS;
  boundsheet : pb8BOUNDSHEET;
  country : rb8COUNTRY;
  palette : rb8PALETTE;
  sst,sstbuf : PChar;
  extsst : pb8EXTSST;
  sstsizeoffset,ltitleoffset,sstblockoffset,lsstbuf,sstsize,extsstsize : integer;
  sz : word;
begin
FWorkBook := WorkBook;
j := 0;
for i:=0 to FWorkBook.SheetsCount-1 do
  j := j+FWorkBook.Sheets[i].RangesCount;
GetMem(FRangesRecs,j*sizeof(rXLSRangeRec));
GetMem(FSheetsRecs,FWorkBook.SheetsCount*sizeof(rXLSSheetRec));
try
  // set palette to default values
  CopyMemory(@FColorPalette[0],@aDefaultColorPalette[0],XLSMaxColorsInPalette*4);
  FPaletteModified := false;
  FUsedColors.Clear;
  
  FBOFOffs := Stream.Position;
  ZeroMemory(@bof,sizeof(bof));
  bof.vers := b8_BOF_vers;
  bof.dt := b8_BOF_dt_WorkbookGlobals;
  bof.rupBuild := b8_BOF_rupBuild_Excel97;
  bof.rupYear := b8_BOF_rupYear_Excel07;
  bof.sfo := b8_BOF_vers;
  wbiff(Stream,b8_BOF,@bof,sizeof(bof));
  ZeroMemory(@interfachdr,sizeof(interfachdr));
  interfachdr.cv := b8_INTERFACHDR_cv_ANSI;
  wbiff(Stream,b8_INTERFACHDR,@interfachdr,sizeof(interfachdr));
  ZeroMemory(@mms,sizeof(mms));
  wbiff(Stream,b8_MMS,@mms,sizeof(mms));
  wbiff(Stream,b8_INTERFACEND,nil,0);
  
  FillMemory(@writeaccess,sizeof(writeaccess),32);
  StringToWideChar(WorkBook.UserNameOfExcel,@writeaccess.stName,sizeof(writeaccess));
  wbiff(Stream,b8_WRITEACCESS,@writeaccess,sizeof(writeaccess));
  
  codepage.cv := b8_CODEPAGE_cv_ANSI;
  wbiff(Stream,b8_CODEPAGE,@codepage,sizeof(codepage));
  
  doublestreamfile.fDSF := 0;
  wbiff(Stream,b8_DOUBLESTREAMFILE,@doublestreamfile,sizeof(doublestreamfile));
  
  // see in biffview, not found in MSDN
  wbiff(Stream,$01C0,nil,0);
  
  GetMem(buf,WorkBook.SheetsCount*2);
  try
    for i:=0 to WorkBook.SheetsCount-1 do
      PWordArray(buf)^[i] := i;
    wbiff(Stream,b8_TABID,buf,WorkBook.SheetsCount*2);
  finally
    FreeMem(buf);
  end;
  fngroupcount.cFnGroup := $000E; // viewed in biffview
  wbiff(Stream,b8_FNGROUPCOUNT,@fngroupcount,sizeof(fngroupcount));
  
  windowprotect.fLockWn := 0; // viewed in biffview
  wbiff(Stream,b8_WINDOWPROTECT,@windowprotect,sizeof(windowprotect));
  
  protect.fLock := 0; // viewed in biffview
  wbiff(Stream,b8_PROTECT,@protect,sizeof(protect));
  
  password.wPassword := 0; // viewed in biffview
  wbiff(Stream,b8_PASSWORD,@password,sizeof(password));
  
  prot4rev.fRevLock := 0; // see in biffview
  wbiff(Stream,b8_PROT4REV,@prot4rev,sizeof(prot4rev));
  
  prot4revpass.wrevPass := 0; // see in biffview
  wbiff(Stream,b8_PROT4REVPASS,@prot4revpass,sizeof(prot4revpass));
  
  ZeroMemory(@window1,sizeof(window1));
  window1.xWn := $0168;
  window1.yWn := $001E;
  window1.dxWn := $1D1E;
  window1.dyWn := $1860;
  window1.grbit := $0038;
  window1.itabCur := $0000;
  window1.itabFirst := $0000;
  window1.ctabSel := $0001;
  window1.wTabRatio := $0258;
  wbiff(Stream,b8_WINDOW1,@window1,sizeof(window1));
  
  backup.fBackupFile := 0;  // set to 1 to enable backup
  wbiff(Stream,b8_BACKUP,@backup,sizeof(backup));
  
  hideobj.fHideObj := 0;  // viewed in biffview
  wbiff(Stream,b8_HIDEOBJ,@hideobj,sizeof(hideobj));
  
  s1904.f1904 := 0; // = 1 if the 1904 date system is used
  wbiff(Stream,b8_1904,@s1904,sizeof(s1904));
  
  precision.fFullPrec := 1; // viewed in biffview
  wbiff(Stream,b8_PRECISION,@precision,sizeof(precision));
  
  refreshall.fRefreshAll := 0;
  wbiff(Stream,b8_REFRESHALL,@refreshall,sizeof(refreshall));
  
  bookbool.fNoSaveSupp := 0; // viewed in biffview
  wbiff(Stream,b8_BOOKBOOL,@bookbool,sizeof(bookbool));
  
  // FONTS
  l := TList.Create;
  try
    // 1. Add default font records
    for i:=0 to 3 do
      with TFont(L[L.Add(TFont.Create)]) do
        begin
          Name := sDefaultFontName;
          Size := 10;
        end;
    // 2. Build list of unique FONT records and write them
    // and init ExportData
    BuildFontList(l);
    // 3. write fonts
    for i:=0 to l.Count-1 do
      wbiffFont(Stream,TFont(l[i]),GetColorPaletteIndex(TFont(l[i]).Color));
  finally
    for i:=0 to l.Count-1 do
      TFont(l[i]).Free;
    l.Free;
  end;
  
  // FORMATS
  sl := TStringList.Create;
  try
    // 1. Add default format records
    sl.AddObject('#,##0".";\-#,##0"."',pointer($0005));
    sl.AddObject('#,##0".";[Red]\-#,##0"."',pointer($0006));
    sl.AddObject('#,##0.00".";\-#,##0.00"."',pointer($0007));
    sl.AddObject('#,##0.00".";[Red]\-#,##0.00"."',pointer($0008));
    sl.AddObject('_-* #,##0"."_-;\-* #,##0"."_-;_-* "-""."_-;_-@_-',pointer($002A));
    sl.AddObject('_-* #,##0__._-;\-* #,##0__._-;_-* "-"__._-;_-@_-',pointer($0029));
    sl.AddObject('_-* #,##0.00"."_-;\-* #,##0.00"."_-;_-* "-"??"."_-;_-@_-',pointer($002C));
    sl.AddObject('_-* #,##0.00__._-;\-* #,##0.00__._-;_-* "-"??__._-;_-@_-',pointer($002B));
    // 2. build format records list
    BuildFormatList(sl);
    // 3. write formats
    for i:=0 to sl.Count-1 do
      wbiffFormat(Stream,sl[i],word(sl.Objects[i]));
  finally
    sl.Free;
  end;
  
  // Style XF
  wbiffHexString(Stream,'e0 00 14 00 00 00 00 00 f5 ff 20 00 00 00 00 00 00 00 00 00 00 00 c0 20');
  wbiffHexString(Stream,'e0 00 14 00 01 00 00 00 f5 ff 20 00 00 f4 00 00 00 00 00 00 00 00 c0 20');
  wbiffHexString(Stream,'e0 00 14 00 01 00 00 00 f5 ff 20 00 00 f4 00 00 00 00 00 00 00 00 c0 20');
  wbiffHexString(Stream,'e0 00 14 00 02 00 00 00 f5 ff 20 00 00 f4 00 00 00 00 00 00 00 00 c0 20');
  wbiffHexString(Stream,'e0 00 14 00 02 00 00 00 f5 ff 20 00 00 f4 00 00 00 00 00 00 00 00 c0 20');
  wbiffHexString(Stream,'e0 00 14 00 00 00 00 00 f5 ff 20 00 00 f4 00 00 00 00 00 00 00 00 c0 20');
  wbiffHexString(Stream,'e0 00 14 00 00 00 00 00 f5 ff 20 00 00 f4 00 00 00 00 00 00 00 00 c0 20');
  wbiffHexString(Stream,'e0 00 14 00 00 00 00 00 f5 ff 20 00 00 f4 00 00 00 00 00 00 00 00 c0 20');
  wbiffHexString(Stream,'e0 00 14 00 00 00 00 00 f5 ff 20 00 00 f4 00 00 00 00 00 00 00 00 c0 20');
  wbiffHexString(Stream,'e0 00 14 00 00 00 00 00 f5 ff 20 00 00 f4 00 00 00 00 00 00 00 00 c0 20');
  wbiffHexString(Stream,'e0 00 14 00 00 00 00 00 f5 ff 20 00 00 f4 00 00 00 00 00 00 00 00 c0 20');
  wbiffHexString(Stream,'e0 00 14 00 00 00 00 00 f5 ff 20 00 00 f4 00 00 00 00 00 00 00 00 c0 20');
  wbiffHexString(Stream,'e0 00 14 00 00 00 00 00 f5 ff 20 00 00 f4 00 00 00 00 00 00 00 00 c0 20');
  wbiffHexString(Stream,'e0 00 14 00 00 00 00 00 f5 ff 20 00 00 f4 00 00 00 00 00 00 00 00 c0 20');
  wbiffHexString(Stream,'e0 00 14 00 00 00 00 00 f5 ff 20 00 00 f4 00 00 00 00 00 00 00 00 c0 20');
  
  // XF
  l := TList.Create;
  try
    // 1. Add default XF record
    s := HexStringToString('00 00 00 00 01 00 20 00 00 00 00 00 00 00 00 00 00 00 c0 20');
    GetMem(buf,Length(s));
    CopyMemory(buf,@(s[1]),Length(s));
    l.Add(buf);
    // 2. Build list of unique XF records and write them
    BuildXFList(l);
    // 3. write XF
    for i:=0 to l.Count-1 do
      wbiff(Stream,b8_XF,l[i],sizeof(rb8XF));
  finally
    for i:=0 to l.Count-1 do
      FreeMem(l[i]);
    l.Free;
  end;
  
  // PALETTE
  if FPaletteModified then
    begin
      palette.ccv := XLSMaxColorsInPalette;
      for i:=0 to XLSMaxColorsInPalette-1 do
        palette.colors[i] := FColorPalette[i]{ shl 8};
      wbiff(Stream,b8_PALETTE,@palette,sizeof(palette));
    end;
  
  // STYLE see in biffview, i dont use this ability simple write bytes
  wbiffHexString(Stream,'93 02 04 00 10 80 04 FF');
  wbiffHexString(Stream,'93 02 04 00 11 80 07 FF');
  wbiffHexString(Stream,'93 02 04 00 00 80 00 FF');
  wbiffHexString(Stream,'93 02 04 00 12 80 05 FF');
  wbiffHexString(Stream,'93 02 04 00 13 80 03 FF');
  wbiffHexString(Stream,'93 02 04 00 14 80 06 FF');
  
  useselfs.fUsesElfs := 0;
  wbiff(Stream,b8_USESELFS,@useselfs,sizeof(useselfs));
  
  // Sheets information
  for i:=0 to FWorkBook.SheetsCount-1 do
    begin
      sh := FWorkBook.Sheets[i];
      FSheetsRecs[i].StreamBOFOffsetPosition := Stream.Position+4;
      ltitle := Length(sh.Title)*sizeof(WideChar);
      boundsheet := AllocMem(sizeof(rb8BOUNDSHEET)+ltitle);
      try
        boundsheet.grbit := 0;
        boundsheet.cch := Length(sh.Title);
        boundsheet.cchgrbit := 1;
        if boundsheet.cch>0 then
          StringToWideChar(sh.Title,PWideChar(PChar(boundsheet)+sizeof(rb8BOUNDSHEET)),ltitle);
        wbiff(Stream,b8_BOUNDSHEET,boundsheet,sizeof(rb8BOUNDSHEET)+ltitle);
      finally
        FreeMem(boundsheet);
      end;
    end;
  
  country.iCountryDef := $07;
  country.iCountryWinIni := $07;
  wbiff(Stream,b8_COUNTRY,@country,sizeof(country));
  
  // SST build sst table
  extsstsize := sizeof(rb8EXTSST);
  extsst := AllocMem(extsstsize);
  extsst.Dsst := 8;
  
  sstsize := sizeof(rb8SST)+4;
  sst := AllocMem(sstsize);
  PWord(sst)^ := b8_SST;
  sstsizeoffset := 2;
  PWord(sst+sstsizeoffset)^ := sizeof(rb8SST);
  sstblockoffset := sstsize;
  lsstbuf := 0;
  sstbuf := nil;
  
  k := 0;
  m := 0;
  try
    for i:=0 to FWorkBook.SheetsCount-1 do
      begin
        sh := FWorkBook.Sheets[i];
        for j:=0 to sh.RangesCount-1 do
          begin
            if sh.RangeByIndex[j].CellDataType=vtecdtString then
              begin
                s := VarToStr(sh.RangeByIndex[j].Value);
                if s<>'' then
                  begin
                    FRangesRecs[m].iSST := k;
                    Inc(k);

                    // convert string to UNICODE
                    ltitle := Length(s)*sizeof(WideChar);
                    if lsstbuf<ltitle then
                      begin
                        lsstbuf := ltitle;
                        ReallocMem(sstbuf,lsstbuf);
                      end;
                    StringToWideChar(s,PWideChar(sstbuf),ltitle);

                    if MaxBiffRecordSize-sstblockoffset<=4 then
                      begin
                        // start new CONTINUE record
                        ReallocMem(sst,sstsize+4);
                        PWord(sst+sstsize)^ := b8_CONTINUE;
                        sstsize := sstsize+2;
                        sstsizeoffset := sstsize;
                        PWord(sst+sstsize)^ := 0;
                        sstsize := sstsize+2;
                        sstblockoffset := 4;
                      end;

                    if (k mod 8)=1 then
                      begin
                        ReallocMem(extsst,extsstsize+sizeof(rb8ISSTINF));
                        pb8ISSTINF(PChar(extsst)+extsstsize).cb := sstblockoffset;
                        pb8ISSTINF(PChar(extsst)+extsstsize).ib := Stream.Position+sstsize;
                        pb8ISSTINF(PChar(extsst)+extsstsize).res1 := 0;
                        extsstsize := extsstsize+sizeof(rb8ISSTINF);
                      end;

                    ReallocMem(sst,sstsize+3);
                    PWord(sst+sstsize)^ := Length(s);
                    sstsize := sstsize+2;
                    PByte(sst+sstsize)^ := 1;
                    sstsize := sstsize+1;
                    PWord(sst+sstsizeoffset)^ := PWord(sst+sstsizeoffset)^+3;
                    sstblockoffset := sstblockoffset+3;

                    ltitleoffset := 0;
                    repeat
                      sz := (Min(ltitle-ltitleoffset,MaxBiffRecordSize-sstblockoffset)) and (not 1);
                      ReallocMem(sst,sstsize+sz);
                      CopyMemory(sst+sstsize,sstbuf+ltitleoffset,sz);
                      sstsize := sstsize+sz;
                      sstblockoffset := sstblockoffset+sz;
                      ltitleoffset := ltitleoffset+sz;
                      PWord(sst+sstsizeoffset)^ := PWord(sst+sstsizeoffset)^+sz;
                      if (ltitle>ltitleoffset) and ((MaxBiffRecordSize-sstblockoffset)<=4) then
                        begin
                          // begin CONTINUE record
                          ReallocMem(sst,sstsize+5);
                          PWord(sst+sstsize)^ := b8_CONTINUE;
                          sstsize := sstsize+2;
                          sstsizeoffset := sstsize;
                          PWord(sst+sstsize)^ := 1;
                          sstsize := sstsize+2;
                          PByte(sst+sstsize)^ := 1;
                          sstsize := sstsize+1;
                          sstblockoffset := 5;
                        end;
                    until ltitle<=ltitleoffset;
                  end;
              end;
            Inc(m);
          end;
      end;
    if k<>0 then
      begin
        pb8SST(sst+4).cstTotal := k;
        pb8SST(sst+4).cstUnique := k;
        Stream.Write(sst^,sstsize);
        wbiff(Stream,b8_EXTSST,extsst,extsstsize);
      end;
  finally
    FreeMem(sst);
    FreeMem(sstbuf);
    FreeMem(extsst);
  end;
  
  wbiff(Stream,b8_EOF,nil,0);
  
  //
  for i:=0 to FWorkBook.SheetsCount-1 do
    begin
      sh := FWorkBook.Sheets[i];
      FSheetsRecs[i].StreamBOFOffset := Stream.Position;
      WriteSheetToStream(Stream,sh);
    end;
  
  // updating sheets information
  for i:=0 to FWorkBook.SheetsCount-1 do
    begin
      Stream.Position := FSheetsRecs[i].StreamBOFOffsetPosition;
      Stream.Write(FSheetsRecs[i].StreamBOFOffset,4);
    end;

finally
  FUsedColors.Clear;
  FreeMem(FRangesRecs);
  FRangesRecs := nil;
  FreeMem(FSheetsRecs);
  FSheetsRecs := nil;
end;
end;

procedure TvteExcelWriter.Save(WorkBook : TvteXLSWorkbook; const FileName : string);
var
  hr : HResult;
  buf : PWideChar;
  Stream : IStream;
  OleStream : TOleStream;
  RootStorage : IStorage;
begin
GetMem(buf,Length(FileName)*sizeof(WideChar)+1);
System.StringToWideChar(FileName,buf,Length(FileName)*sizeof(WideChar));
hr := StgCreateDocFile(buf,
                       STGM_CREATE or STGM_READWRITE or STGM_DIRECT or STGM_SHARE_EXCLUSIVE,
                       0,RootStorage);
FreeMem(buf);
if hr<>S_OK then
  raise Exception.CreateFmt('StgCreateDocFile error %d',[hr]);
hr := RootStorage.CreateStream('Workbook',
                               STGM_CREATE or STGM_READWRITE or STGM_DIRECT or STGM_SHARE_EXCLUSIVE,
                               0,0,Stream);
if hr<>S_OK then
  raise Exception.CreateFmt('CreateStream error %d',[hr]);

// Create the OleStream.
OleStream := TOleStream.Create(Stream);
try
  // Save the memo's text to the OleStream.
  SaveAsBIFFToStream(WorkBook,OleStream);
finally
  // Release the OleStream stream.
  OleStream.Free;
end;
end;

end.

