
{******************************************}
{                                          }
{           vtk Export library             }
{                                          }
{      Copyright (c) 2002 by vtkTools      }
{                                          }
{******************************************}

unit vteExcel;

interface

{$include vtk.inc}                                  

uses
  Windows, SysUtils, Classes, Graphics, math,
  {$IFDEF VTK_D6} variants, {$ENDIF}

  BIFF8_Types, vteExcelTypes;

const
  sXLSWorksheetTitlePrefix = 'Sheet';
  sDefaultFontName = 'Arial';
  XLSDefaultRowHeight = $00FF;
  XLSDefaultColumnWidthInChars = $0008;
  MaxDefaultColors = 16;
  MaxBiffRecordSize = 8228;

type
TvteXLSWorkbook = class;
TvteXLSWorksheet = class;
TvteXLSRow = class;
TvteXLSCol = class;
TCardinalArray = array [0..0] of cardinal;
PCardinalArray = ^TCardinalArray;

///////////////////////////
//
// TvteXLSBorder
//
///////////////////////////
TvteXLSBorder = class(TObject)
private
  FColor : TColor;
  FLineStyle : TvteXLSLineStyleType;
  FWeight : TvteXLSWeightType;
public
  property Color : TColor read FColor write FColor;
  property LineStyle : TvteXLSLineStyleType read FLineStyle write FLineStyle;
  property Weight : TvteXLSWeightType read FWeight write FWeight;

  constructor Create;
  destructor Destroy; override; 
end;

///////////////////////////
//
// TvteXLSBorders
//
///////////////////////////
TvteXLSBorders = class(TObject)
private
  FBorders : array [TvteXLSBorderType] of TvteXLSBorder;
  function GetItem(i : TvteXLSBorderType) : TvteXLSBorder;
public
  property Borders[i : TvteXLSBorderType] : TvteXLSBorder read GetItem; default;
  constructor Create;
  destructor Destroy; override;
end;

TDynWordArray = array of word;
///////////////////////////
//
// TvteXLSRange
//
//////////////////////////
TvteXLSRange = class(TObject)
private
  FWorksheet : TvteXLSWorksheet;
  FPlace : TRect;
  FBorders : TvteXLSBorders;
  FFont : TFont;
  FHorizontalAlignment : TvteXLSHorizontalAlignmentType;
  FVerticalAlignment : TvteXLSVerticalAlignmentType;
  FWrapText : boolean;
  FRotation : byte;
  FFormat : string;
  FValue : Variant;
  FFillPattern : TvteXLSFillPattern;
  FForegroundFillPatternColor : TColor;
  FBackgroundFillPatternColor : TColor;

  FExportData : pointer;

  function GetWorkbook : TvteXLSWorkbook;
  function GetCellDataType : TvteCellDataType;
  procedure SetValue(Value : Variant);
public
  property Worksheet : TvteXLSWorksheet read FWorksheet;
  property Workbook : TvteXLSWorkbook read GetWorkbook;
  property Place : TRect read FPlace;
  property Borders : TvteXLSBorders read FBorders;
  property Font : TFont read FFont;
  property HorizontalAlignment : TvteXLSHorizontalAlignmentType read FHorizontalAlignment write FHorizontalAlignment;
  property VerticalAlignment : TvteXLSVerticalAlignmentType read FVerticalAlignment write FVerticalAlignment;
  property Value : Variant read FValue write SetValue;
  property WrapText : boolean read FWrapText write FWrapText;
  property Rotation : byte read FRotation write FRotation;
  property Format : string read FFormat write FFormat;
  property FillPattern : TvteXLSFillPattern read FFillPattern write FFillPattern;
  property ForegroundFillPatternColor : TColor read FForegroundFillPatternColor write FForegroundFillPatternColor;
  property BackgroundFillPatternColor : TColor read FBackgroundFillPatternColor write FBackgroundFillPatternColor;

  property ExportData : pointer read FExportData write FExportData;
  property CellDataType : TvteCellDataType read GetCellDataType;

  constructor Create(AWorksheet : TvteXLSWorksheet);
  destructor Destroy; override;
end;

/////////////////////////////
//
// TvteXLSRow
//
/////////////////////////////
TvteXLSRow = class(TObject)
private
  FInd : integer;
  FHeight : integer;
public
  property Ind : integer read FInd;
  property Height : integer read FHeight write FHeight;

  constructor Create;
end;

/////////////////////////////
//
// TvteXLSCol
//
/////////////////////////////
TvteXLSCol = class(TObject)
private
  FInd : integer;
  FWidth : integer;
public
  property Ind : integer read FInd write FInd;
  property Width : integer read FWidth write FWidth;

  constructor Create;
end;

//////////////////////////////
//
// TvteXLSPageSetup
//
//////////////////////////////
TvteXLSPageSetup = class(TObject)
private
  FBlackAndWhite : boolean;
  FCenterFooter : string;
  FCenterHeader : string;
  FCenterHorizontally : boolean;
  FCenterVertically : boolean;
  FDraft : boolean;
  FFirstPageNumber : integer;
  FFitToPagesTall : integer;
  FFitToPagesWide : integer;
  FLeftFooter : string;
  FLeftHeader : string;
  FOrder : TvteXLSOrderType;
  FOrientation : TvteXLSOrientationType;
  FPaperSize : TvteXLSPaperSizeType;
  FPrintGridLines : boolean;
  FPrintHeaders : boolean;
  FPrintNotes : boolean;
  FRightFooter : string;
  FRightHeader : string;
  FLeftMargin : double;
  FRightMargin : double;
  FTopMargin : double;
  FBottomMargin : double;
  FFooterMargin : double;
  FHeaderMargin : double;
  FZoom : integer;
  FCopies : integer;
public
  property LeftFooter : string read FLeftFooter write FLeftFooter;
  property LeftHeader : string read FLeftHeader write FLeftHeader;
  property CenterFooter : string read FCenterFooter write FCenterFooter;
  property CenterHeader : string read FCenterHeader write FCenterHeader;
  property RightFooter : string read FRightFooter write FRightFooter;
  property RightHeader : string read FRightHeader write FRightHeader;

  property CenterHorizontally : boolean read FCenterHorizontally write FCenterHorizontally;
  property CenterVertically : boolean read FCenterVertically write FCenterVertically;

  property LeftMargin : double read FLeftMargin write FLeftMargin;
  property RightMargin : double read FRightMargin write FRightMargin;
  property TopMargin : double read FTopMargin write FTopMargin;
  property BottomMargin : double read FBottomMargin write FBottomMargin;

  property HeaderMargin : double read FHeaderMargin write FHeaderMargin;
  property FooterMargin : double read FFooterMargin write FFooterMargin;

  property PaperSize : TvteXLSPaperSizeType read FPaperSize write FPaperSize;
  property Orientation : TvteXLSOrientationType read FOrientation write FOrientation;
  property Order : TvteXLSOrderType read FOrder write FOrder;
  property FirstPageNumber : integer read FFirstPageNumber write FFirstPageNumber;
  property FitToPagesTall : integer read FFitToPagesTall write FFitToPagesTall;
  property FitToPagesWide : integer read FFitToPagesWide write FFitToPagesWide;
  property Copies : integer read FCopies write FCopies;
  property Zoom : integer read FZoom write FZoom;

  property BlackAndWhite : boolean read FBlackAndWhite write FBlackAndWhite;
  property Draft : boolean read FDraft write FDraft;
  property PrintNotes : boolean read FPrintNotes write FPrintNotes;

  property PrintGridLines : boolean read FPrintGridLines write FPrintGridLines;
  property PrintHeaders : boolean read FPrintHeaders write FPrintHeaders;
  
  constructor Create;
end;

//////////////////////////////
//
// TvteXLSWorksheet
//
//////////////////////////////
TvteXLSWorksheet = class(TObject)
private
  FWorkbook : TvteXLSWorkbook;
  FTitle : string;
  FPageSetup : TvteXLSPageSetup;

  FRanges : TList;
  FCols : TList;
  FRows : TList;

  FDimensions : TRect; // sizes of worksheet in cells

  procedure SetTitle(Value : string);
  function GetRange(xl,yt,xr,yb : integer) : TvteXLSRange;
  function GetCol(ColIndex : integer) : TvteXLSCol;
  function GetRow(RowIndex : integer) : TvteXLSRow;
  function GetRangesCount : integer;
  function GetXLSRange(RangeIndex : integer) : TvteXLSRange;
  function GetColsCount : integer;
  function GetRowsCount : integer;
  function GetIndexInWorkBook : integer;
  function GetColByIndex(i : integer) : TvteXLSCol;
  function GetRowByIndex(i : integer) : TvteXLSRow;

  function AddRange(xl,yt,xr,yb : integer) : TvteXLSRange;
  function AddRow(RowIndex : integer) : TvteXLSRow;
  function AddCol(ColIndex : integer) : TvteXLSCol;
public
  property Title : string read FTitle write SetTitle;
  property PageSetup : TvteXLSPageSetup read FPageSetup;
  property Ranges[xl,yt,xr,yb : integer] : TvteXLSRange read GetRange; default;
  property Cols[ColIndex : integer] : TvteXLSCol read GetCol;
  property Rows[RowIndex : integer] : TvteXLSRow read GetRow;
  property RangeByIndex[RangeIndex : integer] : TvteXLSRange read GetXLSRange;
  property RangesCount : integer read GetRangesCount;
  property ColByIndex[ColIndex : integer] : TvteXLSCol read GetColByIndex;
  property ColsCount : integer read GetColsCount;
  property RowByIndex[RowIndex : integer] : TvteXLSRow read GetRowByIndex;
  property RowsCount : integer read GetRowsCount;
  property IndexInWorkBook : integer read GetIndexInWorkBook;

  property Workbook : TvteXLSWorkbook read FWorkbook;
  property Dimensions : TRect read FDimensions;

  function FindRange(xl,yt,xr,yb : integer) : TvteXLSRange;
  function FindRow(RowIndex : integer) : TvteXLSRow;
  function FindCol(ColIndex : integer) : TvteXLSCol;

  constructor Create(AWorkbook : TvteXLSWorkbook);
  destructor Destroy; override;
end;

////////////////////////////
//
// TvteXLSWorkbook
//
////////////////////////////
TvteXLSWorkbook = class(TObject)
private
  FUserNameOfExcel : string;
  FSheets : TList;

  procedure SetUserNameOfExcel(Value : string);
  procedure ClearSheets;
  function GetSheetsCount : integer;
  function GetXLSWorkSheet(i : integer) : TvteXLSWorkSheet;
public
  property UserNameOfExcel : string read FUserNameOfExcel write SetUserNameOfExcel;
  property SheetsCount : integer read GetSheetsCount;
  property Sheets[i : integer] : TvteXLSWorkSheet read GetXLSWorkSheet;

  procedure SaveAsXLSToFile(const FileName : string);
  procedure SaveAsHTMLToFile(const FileName : string);

  function AddSheet : TvteXLSWorksheet;
  procedure Clear;
  constructor Create;
  destructor Destroy; override;
end;

///////////////////////////
//
// TvteCustomWriter
//
///////////////////////////
TvteCustomWriter = class(TObject)
public
  procedure Save(WorkBook : TvteXLSWorkbook; const FileName : string);  virtual;
end;

function PointInRect(X,Y : integer; const R : TRect) : boolean;
function RectOverRect(const r1 : TRect; const r2 : TRect) : boolean;

implementation

uses
  vteWriters;

function PointInRect(X,Y : integer; const R : TRect) : boolean;
begin
Result:=((X>=R.Left) and (X<=R.Right)) and ((Y>=R.Top) and (Y<=R.Bottom))
end;

function RectOverRect(const r1 : TRect; const r2 : TRect) : boolean;
begin
Result:=PointInRect(r1.Left,r1.Top,r2) or
        PointInRect(r1.Right,r1.Top,r2) or
        PointInRect(r1.Right,r1.Bottom,r2) or
        PointInRect(r1.Left,r1.Bottom,r2) or

        PointInRect(r2.Left,r2.Top,r1) or
        PointInRect(r2.Right,r2.Top,r1) or
        PointInRect(r2.Right,r2.Bottom,r1) or
        PointInRect(r2.Left,r2.Bottom,r1) or

        ((r1.Left>r2.Left) and (r1.Right<r2.Right) and (r1.Top<r2.Top) and (r1.Bottom>r2.Bottom)) or
        ((r1.Left<r2.Left) and (r1.Right>r2.Right) and (r1.Top>r2.Top) and (r1.Bottom<r2.Bottom)) or

        ((r2.Left>r1.Left) and (r2.Right<r1.Right) and (r2.Top<r1.Top) and (r2.Bottom>r1.Bottom)) or
        ((r2.Left<r1.Left) and (r2.Right>r1.Right) and (r2.Top>r1.Top) and (r2.Bottom<r1.Bottom));
end;

///////////////////////////
//
// TvteXLSRow
//
///////////////////////////
constructor TvteXLSRow.Create;
begin
inherited Create;
Height := XLSDefaultRowHeight;
end;

////////////////////////////
//
// TvteXLSCol
//
////////////////////////////
constructor TvteXLSCol.Create;
begin
inherited Create;
Width := XLSDefaultColumnWidthInChars*256;
end;

///////////////////////////
//
// TvteXLSBorder
//
///////////////////////////
constructor TvteXLSBorder.Create;
begin
inherited;
// Init to default values
FLineStyle := vtelsNone;
FWeight := vtexlHairline;
FColor := clBlack;
end;

destructor TvteXLSBorder.Destroy;
begin
inherited;
end;

///////////////////////////
//
// TvteXLSBorders
//
///////////////////////////
constructor TvteXLSBorders.Create;
var
  i : TvteXLSBorderType;
begin
inherited;
for i:=Low(TvteXLSBorderType) to High(TvteXLSBorderType) do
  FBorders[i] := TvteXLSBorder.Create;
end;

destructor TvteXLSBorders.Destroy;
var
  i : TvteXLSBorderType;
begin
for i:=Low(TvteXLSBorderType) to High(TvteXLSBorderType) do
  FBorders[i].Free;
inherited;
end;

function TvteXLSBorders.GetItem;
begin
Result := FBorders[i];
end;

///////////////////////////
//
// TvteXLSRange
//
//////////////////////////
constructor TvteXLSRange.Create;
begin
inherited Create;
FVerticalAlignment := vtexlVAlignBottom;
FHorizontalAlignment := vtexlHAlignGeneral;
FWorksheet := AWorksheet;
FBorders := TvteXLSBorders.Create;
FFont := TFont.Create;
FFont.Name := sDefaultFontName;
FFont.Size := 10;
FFont.Color := clBlack;
end;

destructor TvteXLSRange.Destroy;
begin
inherited;
FBorders.Free;
FFont.Free;
end;

function TvteXLSRange.GetWorkbook;
begin
Result := nil;
if FWorksheet<>nil then
  Result := FWorksheet.Workbook;
end;

procedure TvteXLSRange.SetValue;
begin
if (VarType(Value)=varOleStr) or (VarType(Value)=varString) then
  FValue := StringReplace(VarToStr(Value),#13#10,#10,[rfReplaceAll])
else
  FValue := Value;
end;

function TvteXLSRange.GetCellDataType;
var
  vt : integer;
begin
vt := VarType(FValue);
if (vt=varSmallint) or
   (vt=varInteger) or
   (vt=varSingle) or
   (vt=varDouble) or
   (vt=varCurrency) or
   (vt=varByte) then
  Result := vtecdtNumber
else
  Result := vtecdtString;
end;

//////////////////////////////
//
// TvteXLSPageSetup
//
//////////////////////////////
constructor TvteXLSPageSetup.Create;
begin
inherited;
FLeftMargin := 2;
FRightMargin := 2;
FTopMargin := 2.5;
FBottomMargin := 2.5;
FPaperSize := vtexlPaperA4;
FZoom := 100;
FitToPagesTall := 1;
FitToPagesWide := 1;
FirstPageNumber := 1;
end;

//////////////////////////////
//
// TvteXLSWorksheet
//
//////////////////////////////
constructor TvteXLSWorksheet.Create;
var
  i,j : integer;
begin
inherited Create;
FDimensions := Rect(-1,-1,-1,-1);
FWorkbook := AWorkbook;
FRanges := TList.Create;
FCols := TList.Create;
FRows := TList.Create;
FPageSetup := TvteXLSPageSetup.Create;

i := Workbook.FSheets.Count+1;
while true do
  begin
    j := 0;
    while (j<FWorkbook.FSheets.Count) and
          (AnsiCompareText(TvteXLSWorksheet(FWorkbook.FSheets[j]).Title,
                           sXLSWorksheetTitlePrefix+IntToStr(i))=0) do Inc(j);
    if (j>=FWorkbook.FSheets.Count) or
       (AnsiCompareText(TvteXLSWorksheet(FWorkbook.FSheets[j]).Title,
                        sXLSWorksheetTitlePrefix+IntToStr(i))<>0) then
      break;
    Inc(i);
  end;
Title := sXLSWorksheetTitlePrefix+IntToStr(i);
end;

destructor TvteXLSWorksheet.Destroy;
var
  i : integer;
begin
for i:=0 to FRanges.Count-1 do
  TvteXLSRange(FRanges[i]).Free;
for i:=0 to FCols.Count-1 do
  TvteXLSCol(FCols[i]).Free;
for i:=0 to FRows.Count-1 do
  TvteXLSRow(FRows[i]).Free;
FRanges.Free;
FCols.Free;
FRows.Free;
FPageSetup.Free;
inherited;
end;

function TvteXLSWorksheet.GetIndexInWorkBook;
begin
if WorkBook=nil then
  Result := -1
else
  Result := WorkBook.FSheets.IndexOf(Self);
end;

procedure TvteXLSWorksheet.SetTitle;
begin
FTitle := Trim(Copy(Value,1,31));
end;

function TvteXLSWorksheet.GetColByIndex(i : integer) : TvteXLSCol;
begin
Result := TvteXLSCol(FCols[i]);
end;

function TvteXLSWorksheet.GetRowByIndex(i : integer) : TvteXLSRow;
begin
Result := TvteXLSRow(FRows[i]);
end;

function TvteXLSWorksheet.GetColsCount : integer;
begin
Result := FCols.Count;
end;

function TvteXLSWorksheet.GetRowsCount : integer;
begin
Result := FRows.Count;
end;

function TvteXLSWorksheet.GetRangesCount : integer;
begin
Result := FRanges.Count;
end;

function TvteXLSWorksheet.GetXLSRange(RangeIndex : integer) : TvteXLSRange;
begin
Result := TvteXLSRange(FRanges[RangeIndex]);
end;

function TvteXLSWorksheet.GetCol;
begin
Result := FindCol(ColIndex);
if Result=nil then
  Result := AddCol(ColIndex);
end;

function TvteXLSWorksheet.GetRow;
begin
Result := FindRow(RowIndex);
if Result=nil then
  Result := AddRow(RowIndex);
end;

function TvteXLSWorksheet.FindRow;
var
  i : integer;
begin
Result := nil;
for i:=0 to FRows.Count-1 do
  if TvteXLSRow(FRows[i]).Ind=RowIndex then
    begin
      Result := TvteXLSRow(FRows[i]);
      break;
    end;
end;

function TvteXLSWorksheet.AddRow;
begin
Result := TvteXLSRow.Create;
Result.FInd := RowIndex;
FRows.Add(Result);
// change FDimensions
if (FDimensions.Top=-1) or (RowIndex<FDimensions.Top) then
  FDimensions.Top := RowIndex;
if (FDimensions.Bottom=-1) or (RowIndex>FDimensions.Bottom) then
  FDimensions.Bottom := RowIndex;
end;

function TvteXLSWorksheet.FindCol;
var
  i : integer;
begin
Result := nil;
for i:=0 to FCols.Count-1 do
  if TvteXLSCol(FCols[i]).Ind=ColIndex then
    begin
      Result := TvteXLSCol(FCols[i]);
      break;
    end;
end;

function TvteXLSWorksheet.AddCol;
begin
Result := TvteXLSCol.Create;
Result.Ind := ColIndex;
FCols.Add(Result);
// change FDimensions
if (FDimensions.Left=-1) or (ColIndex<FDimensions.Left) then
  FDimensions.Left := ColIndex;
if (FDimensions.Right=-1) or (ColIndex>FDimensions.Right) then
  FDimensions.Right := ColIndex;
end;

function TvteXLSWorksheet.GetRange;
begin
Result := FindRange(xl,yt,xr,yb);
if Result=nil then
  Result := AddRange(xl,yt,xr,yb); // create range
end;

function TvteXLSWorksheet.FindRange(xl,yt,xr,yb : integer): TvteXLSRange;
var
  i : integer;
begin
Result := nil;
i := 0;
while i<FRanges.Count do
  begin
    Result := TvteXLSRange(FRanges[i]);
    if (Result.Place.Left=xl) and
       (Result.Place.Top=yt) and
       (Result.Place.Right=xr) and
       (Result.Place.Bottom=yb) then break;
    Inc(i);
  end;
if i>=FRanges.Count then
  Result := nil;
end;

function TvteXLSWorksheet.AddRange;
var
  i : integer;
  r : TRect;
  ran : TvteXLSRange;
begin
r := Rect(xl,yt,xr,yb);
i := 0;
while i<FRanges.Count do
  begin
    ran := TvteXLSRange(FRanges[i]);
    if RectOverRect(r,ran.Place) then
      begin
        ran.Free;
        FRanges.Delete(i)
      end
    else
      Inc(i);
  end;
// create range
Result := TvteXLSRange.Create(Self);
Result.FPlace := r;
FRanges.Add(Result);
if (FDimensions.Left=-1) or (r.Left<FDimensions.Left) then
  FDimensions.Left := r.Left;
if (FDimensions.Top=-1) or (r.Top<FDimensions.Top) then
  FDimensions.Top := r.Top;
if (FDimensions.Right=-1) or (r.Right>FDimensions.Right) then
  FDimensions.Right := r.Right;
if (FDimensions.Bottom=-1) or (r.Bottom>FDimensions.Bottom) then
  FDimensions.Bottom := r.Bottom;
end;

////////////////////////////////
//
// TvteXLSWorkbook
//
////////////////////////////////
constructor TvteXLSWorkbook.Create;
begin
inherited;
UserNameOfExcel := 'PReport';
FSheets := TList.Create;
end;

destructor TvteXLSWorkbook.Destroy;
begin
ClearSheets;
FSheets.Free;
inherited;
end;

procedure TvteXLSWorkbook.ClearSheets;
var
  i : integer;
begin
for i:=0 to FSheets.Count-1 do
  TvteXLSWorkSheet(FSheets[i]).Free;
FSheets.Clear;
end;

procedure TvteXLSWorkbook.SetUserNameOfExcel;
begin
FUserNameOfExcel := Trim(Copy(Value,1,66));
end;

function TvteXLSWorkbook.GetSheetsCount : integer;
begin
Result := FSheets.Count;
end;

function TvteXLSWorkbook.GetXLSWorkSheet(i : integer) : TvteXLSWorkSheet;
begin
Result := TvteXLSWorkSheet(FSheets[i]);
end;

procedure TvteXLSWorkbook.SaveAsXLSToFile;
var
  Writer : TvteExcelWriter;
begin
  Writer := TvteExcelWriter.Create;
  try
    Writer.Save(Self,FileName);
  finally
    Writer.Free;
  end;
end;

procedure TvteXLSWorkbook.SaveAsHTMLToFile(const FileName : string);
var
  Writer : TvteHTMLWriter;
begin
  Writer := TvteHTMLWriter.Create;
  try
    Writer.Save(Self, FileName);
  finally
    Writer.Free;
  end;
end;

function TvteXLSWorkbook.AddSheet;
begin
Result := TvteXLSWorkSheet.Create(Self);
FSheets.Add(Result);
end;

procedure TvteXLSWorkbook.Clear;
begin
ClearSheets;
end;

///////////////////////////
//
// TvteCustomWriter
//
///////////////////////////
procedure TvteCustomWriter.Save(WorkBook : TvteXLSWorkbook; const FileName : string);
begin
end;

end.

