unit udcVersionInfo;
////////////////////////////////////////////////////////////////////////////////
// PROJECT: Various
// AUTHOR: Donovan J. Edye (DJE)
// DATE CREATED: 04 July 1999
// PURPOSE: Version Info component.
// VERSION: 1.00
// MODIFICATION HISTORY
// -----------------------------------------------------------------------------

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, DsgnIntf,
  {$IFDEF VER100}uudcHelpAboutPropertyEditor3;{$ENDIF}
  {$IFDEF VER120}uudcHelpAboutPropertyEditor4;{$ENDIF}

//UDC Exception handler
type TudcException = class(Exception);

//File version information
type TFileVersionInfo = record
        MajorVersion,
        MinorVersion,
        Release,
        Build : SmallInt;
        CompanyName,
        FileDescription,
        FileVersion,
        InternalName,
        LegalCopyRight,
        LegalTradeMark,
        OriginalFileName,
        ProductName,
        ProductVersion,
        Comments: string;
end;

type
  TudcVersionInfo = class(TComponent)
  private
    FAbout : string;
    FFileName : string;
    FVersionInfo : TFileVersionInfo;
    FLocale : string;
    function GetAllFileVersionInfo(FileName : string) : TFileVersionInfo;
    function GetLocale : string;
  protected
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure GetVersionInfo;
  published
    property About : string read FAbout write FAbout;
    property FileName : string read FFileName write FFileName;
    property LocaleID : string read FLocale write FLocale;
    property MajorVersion : SmallInt read FVersionInfo.MajorVersion ;
    property MinorVersion : SmallInt read FVersionInfo.MinorVersion;
    property Release : SmallInt read FVersionInfo.Release;
    property Build : Smallint read FVersionInfo.Build;
    property CompanyName : string read FVersionInfo.CompanyName;
    property FileDescription : string read FVersionInfo.FileDescription;
    property FileVersion : string read FVersionInfo.FileVersion;
    property InternalName : string read FVersionInfo.InternalName;
    property LegalCopyRight : string read FVersionInfo.LegalCopyRight;
    property LegalTradeMark : string read FVersionInfo.LegalTradeMark;
    property OriginalFileName : string read FVersionInfo.OriginalFileName;
    property ProductName : string read FVersionInfo.ProductName;
    property ProductVersion : string read FVersionInfo.ProductVersion;
    property Comments: string read FVersionInfo.Comments;
  end;

procedure Register;

implementation

//-- START: UTILITY String Functions -------------------------------------------

function Before(const SubStr : string;const s : string) : string;
var p : Integer;
begin
  p:=pos(SubStr,s);
  if p=0
  then result:=s
  else result:=copy(s,1,p-1);
end;

function After(const SubStr : string;const s : string) : string;
var p : Integer;
begin
  p:=pos(SubStr,s);
  if p=0
  then result:=''
  else result:=copy(s,p+length(substr),length(s));
end;

function StripRange(const s : string;FromAnsi,UptoAnsi : word) : string;
var
  i : Integer;
begin
  result:='';
  for i:=1 to length(s) do
  begin
    if ((word(s[i])<FromAnsi) or (word(s[i])>UptoAnsi)) then result:=result+s[i];
  end;
end;

function PurgeCtrl(const s : string) : string;
begin
  result:=StripRange(s,1,26);
end;

function DelimToCR(s : string;delim : char) : string;
var x : string;
    p : integer;
begin
  result:='';
  p:=pos(delim,s);
  while ((p>0) or (s<>'')) do //uses p (not only: while s<>'') to differenciate between "a,b,c" and "a,b,c,"
  begin
    x:=before(delim,s);
    result:=Result+PurgeCtrl(x)+#13+#10;
    p:=pos(delim,s);
    s:=after(delim,s);
  end;
end;

function DelimToStrList(const s : string;delim : char) : TStringList;
begin
  result:=TStringList.create;
  result.text:=DelimToCR(s,delim);
end;

//-- END: UTILITY String Functions -------------------------------------------


//-- START: TudcVersionInfo ----------------------------------------------------

constructor TudcVersionInfo.Create;
////////////////////////////////////////////////////////////////////////////////
// PURPOSE: Class constructor
begin
     //Get the locale
     FLocale := GetLocale;

     //Set globals for about property
     CompName := Self.ClassName;
     CompVersion := '1.00';

     inherited Create(AOwner);
end;

destructor TudcVersionInfo.Destroy;
////////////////////////////////////////////////////////////////////////////////
// PURPOSE: Class destructor
begin
     inherited Destroy;
end;

procedure TudcVersionInfo.GetVersionInfo;
////////////////////////////////////////////////////////////////////////////////
// PURPOSE: Determines the file information
begin
     if Trim(FFileName) = '' then
        Raise TudcException.Create('No file specified.');
     if Trim(FLocale) = '' then
        Raise TudcException.Create('No locale specified.');
     FVersionInfo := GetAllFileVersionInfo(FFileName);
end;


function TudcVersionInfo.GetLocale : string;
////////////////////////////////////////////////////////////////////////////////
// PURPOSE: Determines the locale ID
begin
     Result := GetLocaleStr(GetThreadLocale, LOCALE_ILANGUAGE, '');
end;

function TudcVersionInfo.GetAllFileVersionInfo(FileName : string) : TFileVersionInfo;
////////////////////////////////////////////////////////////////////////////////
// PURPOSE: Retrieves all the version information from the supplied EXE
var
  Buf      : PChar;
  fInfoSize: Dword;
  Versions : TStringList;
  s : string;

  procedure InitVersion;
  var
    FileNamePtr: PChar;
  begin
    with Result do
      begin
        FileNamePtr := PChar(FileName);
        fInfoSize := GetFileVersionInfoSize(FileNamePtr, fInfoSize);
        if fInfoSize > 0 then
          begin
            ReAllocMem(Buf, fInfoSize);
            GetFileVersionInfo(FileNamePtr, 0, fInfoSize, Buf);
          end;
      end;
  end;

  function GetVersion(What: string): string;
  var
    tmpVersion: string;
    Len   : Dword;
    Value : PChar;
  begin
    Result := '';

    if fInfoSize > 0 then
      begin
        SetLength(tmpVersion, 200);
        Value := @tmpVersion;
        if VerQueryValue(Buf, PChar('StringFileInfo\' + FLocale +  '04E4\' + What),
                         Pointer(Value), Len) then
        Result := Value;
     end;
  end;

begin
     try
        try
           Buf := nil;

           with Result do
             begin
               InitVersion;
               CompanyName      := GetVersion('CompanyName');
               FileDescription  := GetVersion('FileDescription');
               FileVersion      := GetVersion('FileVersion');
               try
                  Versions := TStringList.Create;
                  Versions := DelimToStrList(Result.FileVersion, '.');
                  if Versions.Count = 4 then
                  begin
                       try
                          MajorVersion := StrToInt(Versions[0]);
                          MinorVersion := StrToInt(Versions[1]);
                          Release := StrToInt(Versions[2]);
                          Build := StrToInt(Versions[3]);
                       except

                       end;
                  end;
               finally
                  Versions.Free;
                  Versions := nil;
               end;
               InternalName     := GetVersion('InternalName');
               LegalCopyRight   := GetVersion('LegalCopyRight');
               LegalTradeMark   := GetVersion('LegalTradeMark');
               OriginalFileName := GetVersion('OriginalFileName');
               ProductName      := GetVersion('ProductName');
               ProductVersion   := GetVersion('ProductVersion');
               Comments         := GetVersion('Comments');
             end;
        except
          on E : Exception do
          begin
               s := 'GetAllFileVersionInfo(' + FileName + ') - ' + E.Message;
               TudcException.Create(s);
          end;
        end;
     finally
        if Buf <> nil then
           FreeMem(Buf);
     end;
end;
//-- END: TudcVersionInfo ------------------------------------------------------

procedure Register;
begin
  RegisterComponents('Upside Down Coding', [TudcVersionInfo]);
  RegisterPropertyEditor(TypeInfo(string), TudcVersionInfo, 'About', TudcHelpAboutProperty);
end;

end.

