{ SystrayApplication.pas

  Borland Delphi 4.0 Component.
  Programmed 1999 by Jan Bohlen. <jan.bohlen@tu-clausthal.de>

  Please feel free to distribute this component, but inform
  me about any source code changes that you make.

  Any comments to improve this component would be helpful.


  HOW TO USE:
  -----------

  0. All the examples below use a 'Systray: TSystrayApplication'
     field in your main form's declaration. Of course you can
     assign a name other than 'Systray'.


  1. This component is meant for applications that totally rely
     on a system tray icon and don't have any visible main form.
     This means:

     a) You design your application as usual, but your main form
        is just a container for TSystrayApplication and any other
        (non-visual) components that you would like to use. Your
        main form will only be visible during draft time.

     b) Your system tray application is controled by a user-de-
        fined TPopupMenu. You MUST assign a TPopupMenu component
        during draft time, otherwise an ESystrayApplication
        exception is raised (simply because your application
        could not be controled without a TPopupMenu).

     c) You MUST NOT (!) declare an event handler for the 'OnShow'
        event of your MAIN FORM. Any other form's 'OnShow'-
        event handler may be declared as usual. (This is because
        TSystrayApplication overrides the main form's 'OnShow'
        handler to hide the main form. For further details look
        at the source code.)


  2. To bring up a MessageBox or another form (modal or normal)
     use TSystrayApplication's methods:

     a) function TSystrayApplication.MessageBox(...)

        Parameters and return value like TApplication.MessageBox
        except that strings are used instead of PChar (makes it
        easier for you). ALWAYS use this type of message box
        instead of MessageDlg, TApplication.MessageBox or
        Windows.MessageBox. This method does essentially the
        same (and more).


     b) procedure TSystrayApplication.Show(const AForm: TForm;
         ForbidRightClick: Boolean = False);

        AForm = Form that you want to bring up;

        ForbidRightClick = Optional parameter to prevent
        the user from right mouse button clicks, while the
        form is visible; in this case you must execute the
        'AllowRightClick' method in the form's 'OnClose' event
        handler (usually you will use the following ShowModal
        method instead of this method, so don't think about
        it too much!);


     c) function TSystrayApplication.ShowModal(const AForm: TForm): Integer;

        AForm = Form that you want to bring up modal;

        Return value = Modal result of the form (if used);


     Q: WHY should I use these methods instead of
        TApplication.MessageBox, TForm.Show and TForm.ShowModal?

     A: TSystrayApplication internally counts open windows by
        these methods. While any window is open, you won't be
        able to right click the application's system tray icon
        to bring up the TPopupMenu. (This is reasonable, because
        otherwise it would for example be possible to simul-
        taniously open a result form and an options form. The
        result form works with set options, and the options
        are changed, while the result form is visible. There-
        fore the result form would not show the correct result
        depending on the options. Complex, but perhaps you see
        what I mean!?).


     Q: Examples for these methods?

     A: ModRes := Form2.ShowModal;

            ... would be...

        ModRes := Systray.ShowModal(Form2);


        Application.MessageBox('Text', 'Caption', mb_OK);

                       ... would be...

        Systray.MessageBox('Text', 'Caption', mb_OK);


     Q: Will my forms and message boxes pop up 'WS_TOPMOST'
        even if other apps are currently active?

     A: Yes. This is another advantage of TSystrayApplication's
        methods.


  3) If your application is in a 'critical section' (maybe you
     use a timer that checks something every _ minutes), you
     would also like to prevent the user from bringing up your
     application's TPopupMenu. To do so, use the following pro-
     perties:

     a) property CriticalSectionHint: string;

        A hint that is displayed during your critical section, if
        the user moves her/his mouse pointer over the system tray
        icon.


     b) property CriticalSection: Boolean;

        Set 'True', if you enter your critical section,
        set 'False', if you exit your critical section (the sys-
        tem tray icon's hint is restored to its previous hint).


     Q: Examples for these properties?

     A: procedure MyForm.TimerTimer(Sender: TObject);
        begin
          MyTimer.Enabled := False;

          Systray.CriticalSectionHint := 'Calculating...';
          Systray.CriticalSection := True;

          ... // Your calculations

          Systray.CriticalSection := False;
          MyTimer.Enabled := True;
        end;


  4) Use of the RightClickAllowed method:

     function TSystrayApplication.RightClickAllowed: Boolean;

     There is a possibility (if you use timers), that the
     user is currently browsing your TPopupMenu, while your
     timer's event handler is called. So your application is
     in a critical section, but the user nevertheless is able
     to execute a TPopupMenu's command, which would cause
     trouble in certain circumstances. To prevent the user
     from executing TPopupMenu's commands use the following
     code (example for an exit command in your TPopupMenu):

     procedure MyForm.puExit(Sender: TObject);
     begin
       Close;
     end;

                ... would be...

     procedure MyForm.puExit(Sender: TObject);
     begin
       if RightClickAllowed then
         Close;
     end;


   5) General properties:

      a) property Icon: TIcon;

         Usually TSystrayApplication uses the icon which is
         set in your project options (Application.Icon). If
         you want to change the system tray icon during run
         time, use the Icon property. Example:

         Systray.Icon := LoadIcon(0, IDI_WINLOGO);

         ... // Your code

         Systray.Icon := Application.Icon; // Restores the icon


      b) property Hint: string;

         Use this property to set your system tray icon's hint
         during draft or run time.


      c) property PopupMenu: TPopupMenu;

         Self explaining: Assign your application's TPopupMenu
         (ONLY (!) during draft time, otherwise an ESystray-
         Application exception is raised).


      d) property OnDblClick: TNotifyEvent;

         Assign your event handler for a double click on your
         system tray icon. If unassigned, a double click does
         nothing.


   6) A single left mouse button click always puts your appli-
      cation into the foreground.


   7) Now go ahead, and build your own system tray application. }


unit SystrayApplication;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Forms, ShellAPI, Menus;

const
  WM_SYSTRAY = WM_USER + $400;

type
  ESystrayApplication = class(Exception);

type
  TSystrayApplication = class(TComponent)
  private
    { Private declarations }
    FWnd: HWnd;
    FIcon: TIcon;
    FHint: string;
    FPopupMenu: TPopupMenu;
    FNotifyIconData: TNotifyIconData;
    FWindowsOpen: Integer;
    FOnDblClick: TNotifyEvent;
    FCriticalSection: Boolean;
    FCriticalSectionHint: string;
    FOldHint: string;

    procedure SetIcon(const Value: TIcon);
    procedure SetHint(const Value: string);
    procedure SetPopupMenu(const Value: TPopupMenu);
    procedure SetCriticalSection(Value: Boolean);
    procedure WndProc(var Msg: TMessage);
    procedure DoMainFormShow(Sender: TObject);
    procedure DblClick;
    procedure LeftClick;
    procedure RightClick;
    procedure UpdateSystray;
    procedure CleanUp;
  protected
    { Protected declarations }
    procedure Loaded; override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    function MessageBox(const Text, Caption: string;
     Flags: LongInt): Integer;
    procedure Show(const AForm: TForm;
     ForbidRightClick: Boolean = False);
    procedure AllowRightClick;
    function ShowModal(const AForm: TForm): Integer;
    function RightClickAllowed: Boolean;

    property Icon: TIcon read FIcon write SetIcon;
    property CriticalSection: Boolean read FCriticalSection write SetCriticalSection;
    property CriticalSectionHint: string read FCriticalSectionHint write FCriticalSectionHint;
    property Handle: HWnd read FWnd;
  published
    { Published declarations }
    property Hint: string read FHint write SetHint;
    property PopupMenu: TPopupMenu read FPopupMenu write SetPopupMenu;
    property OnDblClick: TNotifyEvent read FOnDblClick write FOnDblClick;
  end;

procedure Register;


implementation

{$R *.RES}

procedure Register;
begin
  RegisterComponents('Additional', [TSystrayApplication]);
end;


{ TSystrayApplication }

constructor TSystrayApplication.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FIcon := TIcon.Create; // Create FIcon
  FIcon.Assign(Application.Icon); // Assign the application's icon

  FWnd := AllocateHWnd(WndProc); // Get a window handle and assign WndProc

  { General initializations: }
  FHint := '';
  FPopupMenu := nil;
  FWindowsOpen := 0;
  FOnDblClick := nil;
  FCriticalSection := False;
  FCriticalSectionHint := '';

  { Create system tray icon: }
  if not (csDesigning in ComponentState) then
  begin
    FillChar(FNotifyIconData, SizeOf(TNotifyIconData), 0);

    with FNotifyIconData do
    begin
      cbSize := SizeOf(TNotifyIconData); // Set size of structure
      uID := 1; // Dummy
      Wnd := FWnd; // Set window handle for messages

      uFlags := NIF_MESSAGE or NIF_ICON; // Recieve messages and display icon

      if Length(FHint) > 0 then
      begin
        uFlags := uFlags or NIF_TIP; // Display hint
        StrPCopy(szTip, PChar(FHint));
      end;

      uCallbackMessage := WM_SYSTRAY; // Set call back message
      hIcon := FIcon.Handle; // Set icon
    end;

    Shell_NotifyIcon(NIM_ADD, @FNotifyIconData); // Finally display icon
  end;
end;

destructor TSystrayApplication.Destroy;
begin
  CleanUp; // See below...

  inherited Destroy;
end;

function TSystrayApplication.MessageBox(const Text, Caption: string;
 Flags: LongInt): Integer;

begin
  try
    Inc(FWindowsOpen);
    Result := Windows.MessageBox(0, PChar(Text), PChar(Caption),
     Flags or MB_TOPMOST);
  finally
    Dec(FWindowsOpen);
  end;
end;

procedure TSystrayApplication.Show(const AForm: TForm;
 ForbidRightClick: Boolean = False);

 begin
   if ForbidRightClick then
     Inc(FWindowsOpen);

  { Put your application into the foreground: }
  SetWindowPos(Application.Handle, HWND_TOPMOST, -1, -1, -1, -1, 0);

  AForm.Show;
end;

procedure TSystrayApplication.AllowRightClick;
begin
  Dec(FWindowsOpen);
end;

function TSystrayApplication.ShowModal(const AForm: TForm): Integer;
begin
  Inc(FWindowsOpen);
  try
    { Put your application into the foreground: }
    SetWindowPos(Application.Handle, HWND_TOPMOST, -1, -1, -1, -1, 0);

    Result := AForm.ShowModal;
  finally
    Dec(FWindowsOpen);
  end;
end;

function TSystrayApplication.RightClickAllowed: Boolean;
begin
  Result := (FWindowsOpen = 0) and (not FCriticalSection);
end;

procedure TSystrayApplication.SetIcon(const Value: TIcon);
begin
  if Value <> FIcon then
  begin
    FIcon.Assign(Value);
    UpdateSystray; // See below...
  end;
end;

procedure TSystrayApplication.SetHint(const Value: string);
begin
  if Value <> FHint then
  begin
    FHint := Value;
    UpdateSystray; // See below...
  end;
end;

procedure TSystrayApplication.SetPopupMenu(const Value: TPopupMenu);
begin
  FPopupMenu := Value;

  if (not (csDesigning in ComponentState))
   and (not (csLoading in ComponentState)) then
  begin
    CleanUp; // See below...

    raise ESystrayApplication.Create('The ''PopupMenu'' property ' +
     'MUST NOT be altered during run-time.');
  end;
end;

procedure TSystrayApplication.SetCriticalSection(Value: Boolean);
begin
  if Value <> FCriticalSection then
  begin
    FCriticalSection := Value;

    if FCriticalSectionHint <> '' then
      if FCriticalSection then
      begin
        FOldHint := FHint;
        SetHint(FCriticalSectionHint);
      end
      else
        SetHint(FOldHint);
  end;
end;

procedure TSystrayApplication.WndProc(var Msg: TMessage);
begin
  with Msg do
  begin
    if Msg = WM_SYSTRAY then
    case LParam of
      WM_LBUTTONDOWN: LeftClick;
      WM_LBUTTONDBLCLK: DblClick;
      WM_RBUTTONDOWN: RightClick;
    end
    else
      DefWindowProc(FWnd, Msg, wParam, lParam);
  end;
end;

procedure TSystrayApplication.DoMainFormShow(Sender: TObject);
begin
  ShowWindow(Application.Handle, SW_HIDE); // Hide your application's main form
end;

procedure TSystrayApplication.DblClick;
begin
  if Assigned(FOnDblClick) then
    FOnDblClick(Self);
end;

procedure TSystrayApplication.LeftClick;
begin
  { Puts your application into the foreground: }
  SetForegroundWindow(Application.Handle);
end;

procedure TSystrayApplication.RightClick;
var
  Pos: TPoint;

begin
  if (FWindowsOpen = 0) and not FCriticalSection then
  begin
    GetCursorPos(Pos);
    FPopupMenu.Popup(Pos.x, Pos.y); // Bring up your TPopupMenu
  end
  else
    Beep; // Generate system default sound
end;

procedure TSystrayApplication.UpdateSystray;
begin
  if not (csDesigning in ComponentState) then
  begin
    with FNotifyIconData do
    begin
      if Length(FHint) > 0 then
      begin
        uFlags := uFlags or NIF_TIP; // Display hint
        StrPCopy(szTip, PChar(FHint));
      end
      else
        uFlags := uFlags and not NIF_TIP; // Don't display hint

      hIcon := FIcon.Handle; // Set icon
      Shell_NotifyIcon(NIM_MODIFY, @FNotifyIconData);
    end;
  end;
end;

procedure TSystrayApplication.CleanUp;
begin
  DeallocateHWnd(FWnd); // Free window handle

  if not (csDesigning in ComponentState) then
    Shell_NotifyIcon(NIM_DELETE, @FNotifyIconData); // Finally destroy icon

  FIcon.Free; // Free icon
end;

procedure TSystrayApplication.Loaded;
begin
  inherited Loaded;

  if not (csDesigning in ComponentState) then
  begin
    { General main form initializations: }
    (Owner as TForm).BorderStyle := bsNone;
    (Owner as TForm).Width := 0;
    (Owner as TForm).Height := 0;
    (Owner as TForm).Visible := True; // IMPORTANT!!!
    (Owner as TForm).OnShow := DoMainFormShow;
  end;

  if not Assigned(FPopupMenu) and (not (csDesigning in ComponentState)) then
  begin
    CleanUp; // See above...

    raise ESystrayApplication.Create('You MUST assign a ' +
     'TPopupMenu (with an ''Exit''-Command). Otherwise your appli' +
     'cation cannot be controled or terminated(!).');
  end;
end;

end.
