unit rjMime;
{
Unit:           rjMime
Version:        1.10
Last Modified:  19. April 2000
Author:         Ralf Junker <ralfjunker@gmx.de>

Description:    Ligtening fast Mime (Base64) Encoding and Decoding routines.
                More detailed descriptions follow the declarations of the
                functions and procedures below.

Legal:          This software is provided 'as-is', without any express or
                implied warranty. In no event will the author be held liable
                for any  damages arising from the use of this software.

                Permission is granted to anyone to use this software for any
                purpose, including commercial applications, and to alter it
                and redistribute it freely, subject to the following
                restrictions:

                1. The origin of this software must not be misrepresented,
                    you must not claim that you wrote the original software.
                    If you use this software in a product, an acknowledgment
                    in the product documentation would be appreciated but is
                    not required.

                2. Altered source versions must be plainly marked as such, and
                   must not be misrepresented as being the original software.

                3. This notice may not be removed or altered from any source
                   distribution.

History:

Version 1.10
------------
19.04.2000      Fixed a small bug in MimeEncode which sometimes screwed up
                the very first bytes of the encoded output.

                Added the following wrapper functions:
                * MimeEncodeString & MimeDecodeString
                * MimeEncodeStream & MimeDecodeStream

Version 1.01
------------
09.04.2000      Fixed a bug in MimeDecodeTable which caused wrong results
                decoding binary files.

Version 1.00
------------
17.01.2000      Initial Public Release

Copyright (c) 2000 Ralf Junker
}

interface

{$DEFINE RangChecking}
{ RangeChecking defines if an EMime exception will be raised if either
  InputBuffer and OutputBuffer are nil or if InputByteCount is not greater than zero.
  If defined, unit SysUtils will be included. }

uses Classes
{$IFDEF RangeChecking}
 , SysUtils
{$ENDIF}
 ;

function MimeEncodeString (const s: AnsiString): AnsiString;
{ MimeEncodeString takes a string, encodes it, and returns the result as a string.
  To decode the result string, use MimeDecodeString. }

function MimeDecodeString (const s: AnsiString): AnsiString;
{ MimeDecodeString takes a a string, decodes it, and returns the result as a string.
  Use MimeDecodeString to decode a string previously encoded with MimeEncodeString. }

procedure MimeEncodeStream (const InputStream: TStream; const OutputStream: TStream);
{ MimeEncodeStream encodes InputStream starting at the current position
  up to the end and writes the result to OutputStream, again starting at
  the current position. When done, It leaves the stream offset at the
  very end of both the InputStream and OutputStream. To encode the entire
  InputStream from beginning to end, make sure that its offset is positioned
  at the beginning of the stream. You can force this by issuing
  Seek (0, soFromBeginning) before calling this function. }

procedure MimeDecodeStream (const InputStream: TStream; const OutputStream: TStream);
{ MimeDecodeStream decodes InputStream starting at the current position
  up to the end and writes the result to OutputStream, again starting at
  the current position. When done, It leaves the stream offset at the
  very end of both the InputStream and OutputStream. To decode the entire
  InputStream from beginning to end, make sure that its offset is positioned
  at the beginning of the stream. You can force this by issuing
  Seek (0, soFromBeginning) before calling this function. }

function MimeEncodedSize (const i: Integer): Integer;
{ Calculates the output size of i MimeEncoded bytes. Use for MimeEncode only. }

procedure MimeEncode (const InputBuffer: Pointer; const OutputBuffer: Pointer; const InputByteCount: Integer);
{ The primary Mime encoding routine.
  CAUTTION: OutputBuffer must have enough memory allocated to take all encoded output.
  MimeEncodedSize (InputByteCount) calculates this amount in bytes. MimeEncode will
  then fill all OutputBuffer, so there is no OutputBytesCount with this procedure.
  Preallocating all memory at once (as required by MimeEncode)
  avoids the time-cosuming process of reallocation. }

procedure MimeDecode (const InputBuffer: Pointer; const InputByteCount: Integer; const OutputBuffer: Pointer; out OutputBytesCount: Integer);
{ The primary Mime decoding routinge.
  CAUTTION: OutputBuffer must have enough memory allocated to take all output.
  Its size should be at least as large as InputByteCount bytes.
  There is therefore no need to calculate OutputBuffer size using MimeEncodedSize.
  OutputBytesCount then returns acutal number of bytes written to OutputBuffer.
  Preallocating all memory at once (as required by MimeDecode)
  avoids the time-cosuming process of reallocation. After calling
  MimeDecode, simply cut the allocated memory down to OutputBytesCount,
  i.e. SetLength (OutString, OutputBytesCount). }

{$IFDEF RangeChecking}
resourcestring
 SInputBufferNil              = 'MIME (Base64) Conversion: Input Buffer must not be NIL.';
 SOutputBufferNil             = 'MIME (Base64) Conversion: Output Buffer must not be NIL.';
 SInputByteCountZero          = 'MIME (Base64) Conversion: InputByteCount must be greater than zero.';
{$ENDIF}
 
const
 BufferSize                   = $2000;
 
type
{$IFDEF RangeChecking}
 EMime = class (Exception);
{$ENDIF}
 
 PByte4 = ^TByte4;
 TByte4 = packed record
  a: Byte;
  b: Byte;
  c: Byte;
  d: Byte;
 end;
 
 PByte3 = ^TByte3;
 TByte3 = packed record
  a: Byte;
  b: Byte;
  c: Byte;
 end;
 
const
 MimeEncodeTable              : array[0..63] of Byte = (
  065, 066, 067, 068, 069, 070, 071, 072, // 00 - 07
  073, 074, 075, 076, 077, 078, 079, 080, // 08 - 15
  081, 082, 083, 084, 085, 086, 087, 088, // 16 - 23
  089, 090, 097, 098, 099, 100, 101, 102, // 24 - 31
  103, 104, 105, 106, 107, 108, 109, 110, // 32 - 39
  111, 112, 113, 114, 115, 116, 117, 118, // 40 - 47
  119, 120, 121, 122, 048, 049, 050, 051, // 48 - 55
  052, 053, 054, 055, 056, 057, 043, 047); // 56 - 63
 
 MimeDecodeTable              : array[Byte] of Cardinal = (
  255, 255, 255, 255, 255, 255, 255, 255, //  00 -  07
  255, 255, 255, 255, 255, 255, 255, 255, //  08 -  15
  255, 255, 255, 255, 255, 255, 255, 255, //  16 -  23
  255, 255, 255, 255, 255, 255, 255, 255, //  24 -  31
  255, 255, 255, 255, 255, 255, 255, 255, //  32 -  39
  255, 255, 255, 062, 255, 255, 255, 063, //  40 -  47
  052, 053, 054, 055, 056, 057, 058, 059, //  48 -  55
  060, 061, 255, 255, 255, 255, 255, 255, //  56 -  63
  255, 000, 001, 002, 003, 004, 005, 006, //  64 -  71
  007, 008, 009, 010, 011, 012, 013, 014, //  72 -  79
  015, 016, 017, 018, 019, 020, 021, 022, //  80 -  87
  023, 024, 025, 255, 255, 255, 255, 255, //  88 -  95
  255, 026, 027, 028, 029, 030, 031, 032, //  96 - 103
  033, 034, 035, 036, 037, 038, 039, 040, // 104 - 111
  041, 042, 043, 044, 045, 046, 047, 048, // 112 - 119
  049, 050, 051, 255, 255, 255, 255, 255, // 120 - 127
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255);
 
implementation

{ **************************************************************************** }
{ Wrapper functions & procedures
{ **************************************************************************** }

function MimeEncodeString (const s: AnsiString): AnsiString;
var
 l                            : Integer;
begin
 if Pointer (s) = nil then
  begin
   Result := '';
   Exit;
  end;
 { l := length (s); }
 l := Integer (Pointer (Integer (Pointer (s)) - 4)^);
 SetString (Result, nil, MimeEncodedSize (l));
 if Pointer (Result) = nil then
  Result := ''
 else
  MimeEncode (Pointer (s), Pointer (Result), l);
end;

{ ********** }

function MimeDecodeString (const s: AnsiString): AnsiString;
var
 lIn, lOut                    : Integer;
begin
 if Pointer (s) = nil then
  begin
   Result := '';
   Exit;
  end;
 { lIn := length (s); }
 lIn := Integer (Pointer (Integer (Pointer (s)) - 4)^);
 SetString (Result, nil, lIn);
 if Pointer (Result) = nil then
  Result := ''
 else
  begin
   MimeDecode (Pointer (s), lIn, Pointer (Result), lOut);
   SetLength (Result, lOut);
  end;
end;

{ **************************************************************************** }

procedure MimeEncodeStream (const InputStream: TStream; const OutputStream: TStream);
var
 InputBuffer                  : array[0..BufferSize - 1] of Byte;
 OutputBuffer                 : array[0.. ((BufferSize + 2) div 3) * 4 - 1] of Byte;
 BytesRead                    : Integer;
begin
 BytesRead := InputStream.Read (InputBuffer, SizeOf (InputBuffer));
 while BytesRead = SizeOf (InputBuffer) do
  begin
   MimeEncode (@InputBuffer, @OutputBuffer, SizeOf (InputBuffer));
   OutputStream.Write (OutputBuffer, SizeOf (OutputBuffer));
   BytesRead := InputStream.Read (InputBuffer, SizeOf (InputBuffer));
  end;
 if BytesRead > 0 then
  begin
   MimeEncode (@InputBuffer, @OutputBuffer, BytesRead);
   OutputStream.Write (OutputBuffer, MimeEncodedSize (BytesRead));
  end;
end;

{ ********** }

procedure MimeDecodeStream (const InputStream: TStream; const OutputStream: TStream);
var
 InputBuffer, OutputBuffer    : array[0..BufferSize - 1] of Byte;
 BytesRead, OutputBytesCount  : Integer;
begin
 BytesRead := InputStream.Read (InputBuffer, SizeOf (InputBuffer));
 while BytesRead > 0 do
  begin
   MimeDecode (@InputBuffer, BytesRead, @OutputBuffer, OutputBytesCount);
   OutputStream.Write (OutputBuffer, OutputBytesCount);
   BytesRead := InputStream.Read (InputBuffer, SizeOf (InputBuffer));
  end;
end;

{ **************************************************************************** }
{ Primary functions & procedures
{ **************************************************************************** }

function MimeEncodedSize (const i: Integer): Integer;
begin
 Result := ((i + 2) div 3) * 4;
end;

{ ********** }

procedure MimeEncode (const InputBuffer: Pointer; const OutputBuffer: Pointer; const InputByteCount: Integer);
var
 b                            : Cardinal;
 InMax3                       : Integer;
 pIn, PInLimit                : ^Byte;
 POut                         : PByte4;
begin
{$IFDEF RangeChecking}
 if InputBuffer = nil then raise EMime.Create (SInputBufferNil);
 if OutputBuffer = nil then raise EMime.Create (SOutputBufferNil);
 if InputByteCount = 0 then raise EMime.Create (SInputByteCountZero);
{$ENDIF}
 InMax3 := InputByteCount div 3 * 3;
 Integer (PInLimit) := Integer (InputBuffer) + InMax3;
 
 pIn := InputBuffer;
 POut := OutputBuffer;
 
 b := 0;
 while pIn <> PInLimit do
  begin
   b := b or pIn^; // Read 3 bytes from InputBuffer.
   Inc (pIn);
   b := b shl 8;
   b := b or pIn^;
   Inc (pIn);
   b := b shl 8;
   b := b or pIn^;
   Inc (pIn);

   // Write 4 bytes to OutputBuffer (in reverse order).
   POut.d := MimeEncodeTable[b and $3F];
   b := b shr 6;
   POut.c := MimeEncodeTable[b and $3F];
   b := b shr 6;
   POut.b := MimeEncodeTable[b and $3F];
   b := b shr 6;
   POut.a := MimeEncodeTable[b and $3F];
   b := b shr 6;

   Inc (POut);
  end;
 
 case InputByteCount - InMax3 of
  1:
   begin
    b := b or pIn^;

    b := b shl 4;

    POut.b := MimeEncodeTable[b and $3F];
    b := b shr 6;
    POut.a := MimeEncodeTable[b and $3F];

    POut.c := Ord ('='); // Fill remaining 2 bytes.
    POut.d := Ord ('=');
   end;
  2:
   begin
    b := b or pIn^;
    Inc (pIn);
    b := b shl 8;
    b := b or pIn^;

    b := b shl 2;

    POut.c := MimeEncodeTable[b and $3F];
    b := b shr 6;
    POut.b := MimeEncodeTable[b and $3F];
    b := b shr 6;
    POut.a := MimeEncodeTable[b and $3F];

    POut.d := Ord ('='); // Fill remaining byte.
   end;
 end;
end;

{ ********** }

procedure MimeDecode (const InputBuffer: Pointer; const InputByteCount: Integer; const OutputBuffer: Pointer; out OutputBytesCount: Integer);
var
 b, i, j                      : Cardinal;
 pIn, PInLimit                : ^Byte;
 POut                         : PByte3;
begin
{$IFDEF RangeChecking}
 if InputBuffer = nil then raise EMime.Create (SInputBufferNil);
 if OutputBuffer = nil then raise EMime.Create (SOutputBufferNil);
 if InputByteCount = 0 then raise EMime.Create (SInputByteCountZero);
{$ENDIF}
 Integer (PInLimit) := Integer (InputBuffer) + InputByteCount;

 pIn := InputBuffer;
 POut := OutputBuffer;

 j := 4;
 b := 0;
 while pIn <> PInLimit do
  begin
   i := MimeDecodeTable[pIn^]; // Read from InputBuffer.
   if i <> $FF then
    begin
     b := b shl 6;
     b := b or i;
     Dec (j);
     if j = 0 then // 4 bytes read from InputBuffer?
      begin
       POut.c := Byte (b); // Write 3 bytes to OutputBuffer (in reverse order).
       b := b shr 8;
       POut.b := Byte (b);
       b := b shr 8;
       POut.a := Byte (b);
       b := b shr 8;
       Inc (POut);

       j := 4;
      end;
    end;
   Inc (pIn);
  end;

 OutputBytesCount := Integer (POut) - Integer (OutputBuffer);

 case j of
  2:
   begin // Write 2 Bytes.
    b := b shl 4;
    POut.b := Byte (b);
    b := b shr 8;
    POut.a := Byte (b);
    Inc (OutputBytesCount);
   end;
  1:
   begin // Write 3 Bytes.
    b := b shl 6;
    POut.c := Byte (b);
    b := b shr 8;
    POut.b := Byte (b);
    b := b shr 8;
    POut.a := Byte (b);
    Inc (OutputBytesCount, 2);
   end;
 end;

end;

end.
