{***************************************************************
 *
 * Unit Name: StatsThread
 * Purpose  : This is a Thread that reads the Connection-Statistics
 *            from the Registry(!). Every 500 ms, this thread will
 *            read the new Values. The Program has to check if this.
 *            Thread should be 'suspended' or active.
 * Author   : Pilif <pilit@dataway.ch> with the help of Angus Robertson <email here>.
 * History  : none. This is the first Release...
 *
 ****************************************************************}

unit StatsThread;

interface

uses
  Classes, SysUTILS, Windows, forms, extctrls, winperf;

type
TRasErrors = Record
  AllErrors    : Integer;
  Alignment    : Integer;
  BufferOverrun: Integer;
  CRC          : Integer; //CheckSum-Error
  Framing      : Integer; //Framing-Erros
  Overrun      : Integer;
  Runouts      : Integer;
  Timeout      : Integer;
end;

//The following is only used under NT since the values
//are not stored by Windows 9x
TRasCompression = Record
 Sent     : Integer;
 Received : Integer;
end;

TRasBytesStats = Record
 BytesReceived    : Integer;
 BytesTransmitted : Integer;
 FramesReceived   : Integer; //NT only
 FramesTransmitted: Integer; //NT only
end;

TRasSpeedStats = Record
 ErrorsPerSecond      : Integer;
 BytesRecPerSecond    : Integer;
 BytesTransPerSecond  : Integer;
 FramesRecPerSecond   : Integer;   //NT
 FramesTransPerSecond : Integer;   //NT
end;

  TStatsThread = class(TThread)
  private
    FNT             : Boolean; //Windows NT?
    FSpeedStats     : TRasSpeedStats;
    FRasErrors      : TRasErrors;
    FRasCompression : TRasCompression; //NT only
    FRasBytesStats  : TRasBytesStats;
    FConnectionSpeed: Integer; //Win9x only. Not stored under NT
    FHRec           : Integer;
    FHTrans         : Integer;

    function  EnablePerfStats (start: boolean): boolean;
    function  GetPerfStats: boolean;
  protected
    procedure Execute; override;
  public
    Property SpeedStats      : TRasSpeedStats  read FSpeedStats;
    property RasErrors       : TRasErrors      read FRasErrors;
    property RasCompression  : TRasCompression read FRasCompression;
    property RasBytesStats   : TRasBytesStats  read FRasBytesStats;
    property ConnectionSpeed : Integer         read FConnectionSpeed;
    constructor Create(CreateSuspended: Boolean);
    destructor Destroy; override;
    procedure Terminate;
    procedure ResetPerfStats;
  end;

const
//Used for the Perfmon-API under NT
  TOTALBYTES =  8192;
  BYTEINCREMENT = 1024;

 // NT performance counter identifiers, assume they are fixed data
 // you can find them under
 //
 // HKEY_LOCAL_MACHINE\Microsoft\WindowsNT\CurrentVersion\PerfLib\00x\Counters
 //
 // For x you can choose a key. They are used to store the descriptions
 // in different languages.
 //
 // NOTE: You have to use regedt32.exe. Regedit does not support the
 //       REG_MULTI_SZ datatype (unfortunally the VCL does not support
 //       it either).

// Now the Windows9x registry Keys. You will find them under
//
// HKEY_DYN_DATA\PerfStats\*
//
// Before you can begin to read the statistics from \PerfStats\StatData,
// you will have to read them once from \PerfStats\StartStat and if
// you have finished reading values, you will have to close the session with
// reading the stats from \PerfStats\StopStat.
//
// Please nothe that this is not documented anywhere!

  Reg_PerfStatStart = 'PerfStats\StartStat';
  Reg_PerfStatData  = 'PerfStats\StatData';
  Reg_PerfStatStop  = 'PerfStats\StopStat';
  Reg_PerfXmit 	    = 'Dial-Up Adapter\TotalBytesXmit';
  Reg_PerfRecv 	    = 'Dial-Up Adapter\TotalBytesRecvd';
  Reg_PerfConn 	    = 'Dial-Up Adapter\ConnectSpeed';
  Reg_PerfBuffer    = 'Dial-Up Adapter\Buffer';
  Reg_PerfFraming   = 'Dial-Up Adapter\Framing';
  Reg_PerfOverrun   = 'Dial-Up Adapter\Overrun ';
  Reg_PerfAlignment = 'Dial-Up Adapter\Alignment';
  Reg_PerfTimeout   = 'Dial-Up Adapter\Timeout';
  Reg_PerfCRC       = 'Dial-Up Adapter\CRC';
  Reg_PerfRunts     = 'Dial-Up Adapter\Runts';

implementation

uses Main, RasExt, Perf;

{ I removed the warning from Borland about synchronizing, because
  I am using a german version of delphi. So ou probably could not do
  much with the big comment.... }

{ TStatsThread }

constructor TStatsThread.Create(CreateSuspended: Boolean);
begin
  inherited Create(CreateSuspended);
  FNT := (Win32Platform = VER_PLATFORM_WIN32_NT); //Win32Platform is defined in SysUtils
  FreeOnTerminate := True;
  Priority:=tpLower;
  If not EnablePerfStats(true) then
   begin
   Raise Exception.Create('Could not initialize the PerfStats. Quitting Thread!');
   Free;
//   exit;
   end;
end;

destructor TStatsThread.Destroy;
begin
  EnablePerfStats(false);
  inherited Destroy;
end;

procedure TStatsThread.Terminate;
begin
  inherited Terminate;
  EnablePerfStats(false);
  if FNT then
   RegCloseKey (HKEY_PERFORMANCE_DATA);
end;

procedure TStatsThread.ResetPerfStats;
begin
FHTrans       := FRasBytesStats.BytesTransmitted;
fHRec         := FRasBytesStats.BytesReceived;
FillChar(FRasBytesStats, SizeOf(TRasBytesStats), 0);
FillChar(FRasErrors, SizeOf(TRasErrors), 0);
FillChar(FRasCompression, SizeOf(TRasCompression), 0);
FillChar(FSpeedStats, SizeOf(TRasSpeedStats), 0);
end;

function TStatsThread.EnablePerfStats (start: boolean): boolean;
var
TempKey: HKey;
keyname: string;
dwType, dwSize: DWORD;
TempData: Pointer;

function InitData (const ValueName: string): boolean;
var ret: Dword;
begin
 result := false;
 ret     := RegQueryValueEx (TempKey, PChar(ValueName), nil,
							@dwType, nil, @dwSize);
 if ret = ERROR_SUCCESS then
 begin
  try		// read data but ignore it
  GetMem (TempData, dwSize);
  Result := RegQueryValueEx (TempKey, PChar(ValueName), nil,
        	@dwType, TempData, @dwSize) = ERROR_SUCCESS;
  finally
   FreeMem (TempData);
  end;
 end;
end;

begin
//result := false;
if Win32Platform = VER_PLATFORM_WIN32_WINDOWS then
begin
 TempKey := 0;
 if start then
  keyname := Reg_PerfStatStart
 else
  keyname := Reg_PerfStatStop;

 result := RegOpenKeyEx (HKEY_DYN_DATA, PChar(keyname), 0,
                         KEY_ALL_ACCESS, TempKey) = ERROR_SUCCESS;
 if result then
  begin
   result := InitData (Reg_PerfXmit);
   if result then result := InitData(Reg_PerfRecv);
   if result then result := InitData(Reg_PerfConn);
   if result then result := InitData(Reg_PerfBuffer);
   if result then result := InitData(Reg_PerfFraming);
   if result then result := InitData(Reg_PerfOverrun);
   if result then result := InitData(Reg_PerfAlignment);
   if result then result := InitData(Reg_PerfTimeout);
   if result then result := InitData(Reg_PerfCRC);
   if result then result := InitData(Reg_PerfRunts);
   if result then result := InitData(Reg_PerfConn);
   RegCloseKey (TempKey);
  end; //if result
end //if win32_windows
else
 Result:=true;

if result then
  begin
   if start then result := GetPerfStats;	// get counters
   ResetPerfStats;			// set current
  end;
end;




function TStatsThread.GetPerfStats: boolean;
var tempkey: Hkey;

function GetData (const ValueName: string; var Info: Integer): boolean;
var dwSize, dwtype: dword;
begin
 dwSize := 4;	// data is four bytes of binary, aka a DWORD
 Result := RegQueryValueEx (TempKey, PChar(ValueName), nil,
                            @dwType, @Info, @dwSize) = ERROR_SUCCESS;
end;

begin
result := false;
if Win32Platform = VER_PLATFORM_WIN32s then exit;
if Win32Platform = VER_PLATFORM_WIN32_WINDOWS then  // Win95/98
begin
 TempKey := 0;
 result := RegOpenKeyEx (HKEY_DYN_DATA, PChar(Reg_PerfStatData), 0,  KEY_READ, TempKey) = ERROR_SUCCESS;
 if result then result := GetData(Reg_PerfXmit, FRasBytesStats.BytesTransmitted);
 if result then result := GetData(Reg_PerfRecv, FRasBytesStats.BytesReceived);
 if result then result := GetData(Reg_PerfBuffer, FRasErrors.BufferOverrun);
 if result then result := GetData(Reg_PerfFraming, FRasErrors.Framing);
 if result then result := GetData(Reg_PerfOverrun, FRasErrors.Overrun);
 if result then result := GetData(Reg_PerfAlignment, FRasErrors.Alignment);
 if result then result := GetData(Reg_PerfTimeout, FRasErrors.Timeout);
 if result then result := GetData(Reg_PerfCRC, FRasErrors.CRC);
 if result then result := GetData(Reg_PerfRunts, FRasErrors.Runouts);
 if result then result := GetData(Reg_PerfConn, FConnectionSpeed);
 if result then
  begin
   if FRasBytesStats.BytesTransmitted < FHTrans then ResetPerfStats;
   if FRasBytesStats.BytesReceived    < FHrec   then ResetPerfStats;
   RegCloseKey (TempKey);
  end;
end
else
begin  //Windows NT (I hate the Perfmon-API, so I use components)
 Result:=true;
 if not assigned(PerfModule) then exit;
 PerfModule.pf.Collect; //in BETAs of Rit, I used my own implementeation, but it was buggy
 try
 FRasBytesStats.BytesReceived     := PerfModule.pf.Items[0].CtrAsInteger;
 FRasBytesStats.BytesTransmitted  := PerfModule.pf.Items[3].CtrAsInteger;
 FRasBytesStats.FramesReceived    := PerfModule.pf.Items[1].CtrAsInteger;
 FRasBytesStats.FramesTransmitted := PerfModule.pf.Items[4].CtrAsInteger;

 FRasCompression.Sent             := PerfModule.pf.Items[5].CtrAsInteger;
 FRasCompression.Received         := PerfModule.pf.Items[2].CtrAsInteger;

 FRasErrors.AllErrors             := PerfModule.pf.Items[6].CtrAsInteger;
 FRAsErrors.Alignment             := PerfModule.pf.Items[7].CtrAsInteger;
 FRasErrors.BufferOverrun         := PerfModule.pf.Items[9].CtrAsInteger;
 FRasErrors.CRC                   := PerfModule.pf.Items[8].CtrAsInteger;
 FRasErrors.Overrun               := PerfModule.pf.Items[10].CtrAsInteger;
 FRasErrors.Timeout               := PerfModule.pf.Items[11].CtrAsInteger;

 FSpeedStats.ErrorsPerSecond      := PerfModule.pf.Items[12].CtrAsInteger;
 except

 end;
end;
end;


procedure TStatsThread.Execute;
var
oldRasBytesStats : TRasBytesStats;
oldAllErrors : Integer;
begin
  while not Terminated do
  begin         //Read the Stats every 500 msecs
    oldRasBytesStats  := FRasBytesStats;
    oldAllErrors:=FRasErrors.Alignment + FRasErrors.BufferOverrun + FRasErrors.CRC +
                  FRasErrors.Framing   + FRasErrors.Overrun       + FRasErrors.Runouts +
                  FRasErrors.Timeout;
   GetPerfStats;
   if Win32Platform = VER_PLATFORM_WIN32_WINDOWS then  // Win95/98
    begin //NT does this for me
     FSpeedStats.BytesRecPerSecond  :=(FRasBytesStats.BytesReceived    - oldRasBytesStats.BytesReceived) * 2; //*2 because this function is called every 500 msecs
     FSpeedStats.BytesTransPerSecond:=(FRasBytesStats.BytesTransmitted - oldRasBytesStats.BytesTransmitted) * 2; //*2 because this function is called every 500 msecs
     FRasErrors.AllErrors := FRasErrors.Alignment + FRasErrors.BufferOverrun + FRasErrors.CRC +
                             FRasErrors.Framing   + FRasErrors.Overrun       + FRasErrors.Runouts +
                             FRasErrors.Timeout;
     FSpeedStats.ErrorsPerSecond    :=(FRasErrors.AllErrors - oldAllErrors) * 2; //500msecs....
    end
   else
     begin
     FSpeedStats.BytesRecPerSecond  :=(FRasBytesStats.BytesReceived    - oldRasBytesStats.BytesReceived) * 2; //*2 because this function is called every 500 msecs
     FSpeedStats.BytesTransPerSecond:=(FRasBytesStats.BytesTransmitted - oldRasBytesStats.BytesTransmitted) * 2; //*2 because this function is called every 500 msecs
     end;
   Sleep(500);
  end;
end;

end.
