unit dcUnits;

interface

uses
  Classes, PEFile, Procs, dcDecomps, dcDFMs, MethodLists;

type
  { TUnit }

  TUnits = class;

  TUnitType = (utNormal, utSystem, utProgram);

  TUnit = class(TCollectionItem)
  private
    FAddress: PChar;
    FSize: Integer;
    FAInit: TProc;
    FFInit: TProc;
    FName: string;
    FUnitType: TUnitType;
    FDFM: TdcDFM;
    FPEFileClass: TPEFile;

    FDecompItems: TList;
    FImplUnits: TList;
    FIntfUnits: TList;
    FUnitSrc: TStrings;
    FImportedUnit: Boolean;

    FComments: TComments;

    procedure SetName(Value: string);
    function GetImplUnitCount: Integer;
    function GetImplUnit(Index: Integer): TUnit;
    function GetIntfUnitCount: Integer;
    function GetIntfUnit(Index: Integer): TUnit;
    procedure SetAInit(AInit: TProc);
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    function FindProcByName(Name: string): TProc;
    function FindClassByName(Name: string): TClassInfo;
    procedure InsertImplUnit(Index: Integer; AUnit: TUnit);
    procedure AddImplUnit(AUnit: TUnit);
    procedure InsertIntfUnit(Index: Integer; AUnit: TUnit);
    procedure AddIntfUnit(AUnit: TUnit);
    procedure GenUnitSrc;
    procedure DeterIntfImpl;

    property Address: PChar read FAddress write FAddress;
    property Size: Integer read FSize write FSize;
    property Init: TProc read FAInit write SetAInit;
    property FInit: TProc read FFInit write FFInit;
    property Name: string read FName write SetName;
    property UnitSrc: TStrings read FUnitSrc;
    property UnitType: TUnitType read FUnitType;
    property DFM: TdcDFM read FDFM write FDFM;
    property PEFileClass: TPEFile read FPEFileClass;

    property DecompItems: TList read FDecompItems;
    property ImplUnitCount: Integer read GetImplUnitCount;
    property ImplUnits[Index: Integer]: TUnit read GetImplUnit;
    property IntfUnitCount: Integer read GetIntfUnitCount;
    property IntfUnits[Index: Integer]: TUnit read GetIntfUnit;
    property ImportedUnit: Boolean read FImportedUnit;

    property Comments: TComments read FComments;
  end;

  { TUnits }

  TUnits = class(TCollection)
  private
    FPEFileClass: TPEFile;
    FSysInitUnit: TUnit;
    FSystemUnit: TUnit;
    FProgramUnit: TUnit;
    FFirstNormalUnit: TUnit;
    
    FOnAssignUnits: TmlneMethodList;
    function GetItem(Index: Integer): TUnit;
    procedure SetItem(Index: Integer; Value: TUnit);
  public
    constructor Create(PEFileClass: TPEFile); reintroduce; overload;
    destructor Destroy; override;
    function FindInUnitUsingFInit(Address: PChar): TUnit;
    function FindInUnit(Address: PChar): TUnit;
    function FindByName(const Name: string): Integer;
    procedure GenerateReqUnits;
    procedure GenerateNames;
    procedure GenUnitSrcs;
    procedure DeterIntfImpls;
    procedure AssignUnits;
    procedure LoadInitFInit;

    property Items[Index: Integer]: TUnit read GetItem write SetItem; default;

    property OnAssignUnits: TmlneMethodList read FOnAssignUnits;
    property PEFileClass: TPEFile read FPEFileClass;
    property SysInitUnit: TUnit read FSysInitUnit;
    property SystemUnit: TUnit read FSystemUnit;
    property ProgramUnit: TUnit read FProgramUnit;
    property FirstNormalUnit: TUnit read FFirstNormalUnit;
  end;

implementation

uses
  {$IFOPT D+} dcDebug, {$ENDIF}
  SysUtils, PEFileClass, Vars, TypInfo, dcNTInfoTypes, dcThrVar, dcTypeIntf,
  DisAsm, Dialogs, Controls;

{ TUnit }

constructor TUnit.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FDecompItems := TList.Create;
  FImplUnits := TList.Create;
  FIntfUnits := TList.Create;
  FUnitSrc := TStringList.Create;
  FPEFileClass := (Collection as TUnits).FPEFileClass;
  FComments := TComments.Create;
end;

destructor TUnit.Destroy;
begin
  FComments.Free;
  FUnitSrc.Free;
  FIntfUnits.Free;
  FImplUnits.Free;
  FDecompItems.Free;
  inherited Destroy;
end;

procedure TUnit.SetName(Value: string);
var
  I: Integer;
resourcestring
  SUnitNameAlreadyExists = 'Unit named %s already exists.';
  SUnitAlreadyHasAName = 'Cann''t change that name to %s, because it is already set to %s.';
begin
  if AnsiCompareText(Value, FName) = 0 then Exit;

  if FName <> '' then
    raise EDecompilerError.CreateFmt(SUnitAlreadyHasAName, [FName, Value]);

  for I := 0 to TPEFileClass(PEFileClass).Units.Count -1 do
    if TPEFileClass(PEFileClass).Units[I].Name = Value then
      raise EDecompilerError.CreateFmt(SUnitNameAlreadyExists, [Value]);

  FName := Value;
end;

function TUnit.GetImplUnitCount: Integer;
begin
  Result := FImplUnits.Count;
end;

procedure TUnit.InsertImplUnit(Index: Integer; AUnit: TUnit);
var
  I: Integer;
begin
  // exit when the unit is not in one list already, or it is a system unit.
  if (FIntfUnits.IndexOf(AUnit) <> -1) or
     (AUnit.Index < 2) or
     (AUnit = Self) then
    exit;
  I := FImplUnits.IndexOf(AUnit);
  if I = -1 then
    FImplUnits.Insert(Index, AUnit)
  else
    if I >= Index then
      FImplUnits.Move(I, Index)
    else
      FImplUnits.Move(I, Index -1);
end;

procedure TUnit.AddImplUnit(AUnit: TUnit);
begin
  InsertImplUnit(ImplUnitCount, AUnit);
end;

function TUnit.GetImplUnit(Index: Integer): TUnit;
begin
  Result := TUnit(FImplUnits[Index]);
end;

function TUnit.GetIntfUnitCount: Integer;
begin
  Result := FIntfUnits.Count;
end;

procedure TUnit.InsertIntfUnit(Index: Integer; AUnit: TUnit);
var
  I: integer;
begin
  // exit when the unit is not in the list already, or it is a system unit.
  if (AUnit.Index < 2) or (AUnit = Self) then
    exit;
  // If this is the program unit only add it to the impl unit.
  if UnitType = utProgram then
  begin
    AddImplUnit(AUnit);
    Exit;
  end;
  // If the unit is in the Impl Unit list remove it from there
  FImplUnits.Remove(AUnit);
  I := FIntfUnits.IndexOf(AUnit);
  if I = -1 then
    FIntfUnits.Insert(Index, AUnit)
  else
    if I >= Index then
      FIntfUnits.Move(I, Index)
    else
      FIntfUnits.Move(I, Index -1);
end;

procedure TUnit.AddIntfUnit(AUnit: TUnit);
begin
  InsertIntfUnit(IntfUnitCount, AUnit);
end;

function TUnit.GetIntfUnit(Index: Integer): TUnit;
begin
  Result := TUnit(FIntfUnits[Index]);
end;

procedure TUnit.SetAInit(AInit: TProc);
begin
  if AInit.Address[0] = #$FF then
    FImportedUnit := True;
  FAInit := AInit;
end;

function TUnit.FindProcByName(Name: string): TProc;
var
  I: Integer;
begin
  for I := 0 to FDecompItems.Count -1 do
  begin
    Result := TProc(FDecompItems[I]);
    if (TDecompItem(Result) is TProc) and (Result.Name = Name) then
      exit;
  end;
  Result := nil;
end;

function TUnit.FindClassByName(Name: string): TClassInfo;
var
  I: Integer;
begin
  for I := 0 to FDecompItems.Count -1 do
  begin
    Result := TClassInfo(FDecompItems[I]);
    if (TDecompItem(Result) is TClassInfo) and (Result.AClass.ClassName = Name) then
      exit;
  end;
  Result := nil;
end;

function DecompItemSortBssBeforeData(Item1, Item2: Pointer): Integer;
begin
  Result := TDecompItem(Item1).Address - TDecompItem(Item2).Address;
  // if both decomp items are vars and one is in the BSS section and the other not,
  // put the one in the bss section before the other.
  if (TDecompItem(Item1) is TVar) and (TDecompItem(Item2) is TVar) then
  begin
    if (TDecompItem(Item1).Address >= TVar(Item1).PEFileClass.BSS) and
       (TDecompItem(Item2).Address < TVar(Item1).PEFileClass.BSS) then
      Result := -1;
    if (TDecompItem(Item1).Address < TVar(Item1).PEFileClass.BSS) and
       (TDecompItem(Item2).Address >= TVar(Item1).PEFileClass.BSS) then
      Result := 1;
  end;
end;

procedure TUnit.GenUnitSrc;
type
  TSectionType = (stConst, stType, stVar, stProc, stLabel, stResourceString, stThreadVar);
var
  SectionType: TSectionType;
  Vars: TStringList;
  Consts: TStringList;

  procedure SetSectionType(ASectionType: TSectionType);
  const
    SectionTypeDecl: array[TSectionType] of string = ('const', 'type',
      'var', '', 'label', 'resourcestring', 'threadvar');
  var
    I: Integer;
  begin
    if ASectionType = SectionType then exit;
    // Add the vars if they exits.
    if Vars.Count > 0 then
    begin
      if SectionType <> stVar then
        UnitSrc.Add('var');
      for I := 0 to Vars.Count -1 do
        UnitSrc.Add(Vars[I]);
      Vars.Clear;
      SectionType := stVar;
    end;
    // Add the Consts if thet exits.
    if Consts.Count > 0 then
    begin
      if SectionType <> stConst then
        UnitSrc.Add('const');
      for I := 0 to Consts.Count -1 do
        UnitSrc.Add(Consts[I]);
      Consts.Clear;
      SectionType := stConst;
    end;
    if ASectionType = SectionType then exit;
    UnitSrc.Add(SectionTypeDecl[ASectionType]);
    SectionType := ASectionType;
  end;

  procedure AddComments(Strings: TStrings);
  begin
    if Strings.Count <> 0 then
    begin
      UnitSrc.Add('{');
      UnitSrc.AddStrings(Strings);
      UnitSrc.Add('}');
    end;
  end;

const
  BeginUnit = 'unit %s;' + #13#10#13#10 + 'interface';
  BeginProgram: array[TProjectType] of string = ('program %s;', 'library ^s;', 'package %s;');
  UsesClause = #13#10'uses';
  ContainsClause = #13#10'contains';
  ImplUnit = #13#10'implementation';
  EndUnit = #13#10'end.';
  DFMInclude = '{$R *.DFM}'#13#10;
  SConsoleApp = #13#10'{$APPTYPE CONSOLE}'#13#10;
var
  I, J, K, L, M: Integer;
  Changed: Boolean;
  Str: string;
begin
  // Add unit comments.
  AddComments(Comments);

  Vars := TStringList.Create;
  try
   Consts := TStringList.Create;
   try
    // Don't generate unit source if this is a system unit.
    if UnitType = utSystem then exit;
    if UnitType <> utProgram then
    begin
      // Start with the unit name and interface.
      UnitSrc.Add(Format(BeginUnit, [Name]));
    end
    else
    begin
      // It is the program "unit"
      UnitSrc.Add(Format(BeginProgram[TPEFileClass(FPEFileClass).ProjectType], [Name]));
      // Psossible Add {$APPTYPE CONSOLE}
      if PEFileClass.IsConsole then
        UnitSrc.Add(SConsoleApp);
    end;

    // Sort the decompItems (In this following they were also declared).
    DecompItems.Sort(DecompItemSortBssBeforeData);
    // Set all the req items before the items which requires them (possible endless loop).
    repeat
      Changed := False;
      for I := 0 to DecompItems.Count -1 do
      begin
        for J := 0 to TDecompItem(DecompItems[I]).ReqDecompCount -1 do
        begin
          with TDecompItem(DecompItems[I]) do
          begin

            // Don't Move if it is a proc or a type info which doesn't have a
            // type def or a ClassInfo which is requires by TypeInfo.
            if not ((ReqDecomps[J] is TProc) or
                   ((ReqDecomps[J] is TTypeInfoInfo) and
                    (not TTypeInfoInfo(ReqDecomps[J]).HasTypeDef))) then
            begin
              // Move the req item before the other.
              K := Self.DecompItems.IndexOf(ReqDecomps[J]);
              if K > I then
              begin
                // if there is a type and req item is a class, then it must only
                // be before if there is no type between them.
                if (ReqDecomps[J] is TClassInfo) and
                   IsTypeDecomp(TDecompItem(DecompItems[I])) then
                begin
                  // Move the class after the last typeinfo decomp after the first item.
                  for L := I +1 to K -1 do
                    if not IsTypeDecomp(TDecompItem(Self.DecompItems[L])) then
                    begin
                      Self.DecompItems.Move(K, L);
                      Changed := True;
                      Break;
                    end;
                end
                else
                begin
                  if (TDecompItem(Self.DecompItems[K]) is TVar) and
                     (not (TVar(Self.DecompItems[K]).AType.TypeKind in [etkUTInteger, etkUTString])) then
                  begin
                    // Vars may not require a var after it.
                    if TDecompItem(Self.DecompItems[I]) is TVar then
                      TDecompItem(Self.DecompItems[I]).Comments.AddComment(
                          Format('var requiring a var after it %p',
                          [Pointer(TDecompItem(Self.DecompItems[K]).Address)]), ctWarning)
                    else
                    begin
                      Self.DecompItems.Move(K, I);
                      // Move all the vars which were before this var before the new place.
                      M := I;
                      for L := I + 1 to K do
                        if TDecompItem(Self.DecompItems[L]) is TVar then
                        begin
                          // Move the var.
                          Self.DecompItems.Move(L, M);
                          Inc(M);
                        end;
                      Changed := True;
                    end;
                  end
                  else
                  begin
                    if IsTypeDecomp(TDecompItem(Self.DecompItems[I])) and
                       (TDecompItem(Self.DecompItems[K]) is TVar) then
                    begin
                      M := 0;
                      for L := I -1 downto 0 do
                        if not IsTypeDecomp(Self.DecompItems[L]) then
                        begin
                          M := L + 1;
                          break;
                        end;
                    end
                    else
                      M := I;
                    Self.DecompItems.Move(K, M);
                    Changed := True;
                  end;
                  Break;
                end;
              end;
            end;
          end;
        end;
      end;
    until not Changed;

    // Don't add an interface section if this is the program unit.
    if UnitType <> utProgram then
    begin
      if IntfUnitCount > 0 then
      begin
        // Add the interface uses clause.
        UnitSrc.Add(UsesClause);
        Str := '  ';
        for I := 0 to IntfUnitCount -2 do
          Str := Str + IntfUnits[I].Name + ', ';
        Str := Str + IntfUnits[IntfUnitCount -1].Name + ';';
        UnitSrc.Add(Str);
      end;

      // Add the interface declarations.
      SectionType := stProc;
      for I := 0 to DecompItems.Count -1 do
      begin
        if TDecompItem(DecompItems[I]).IntfImpl = iiInterface then
        begin
          // Add the comments
          AddComments(TDecompItem(DecompItems[I]).Comments);
          // It this decomp item requires a class form the same unit which isn't
          // declared yet make a forward declaration
          for J := 0 to TDecompItem(DecompItems[I]).ReqDecompCount -1 do
          begin
            if (not (TDecompItem(DecompItems[I]) is TProc)) and
               (TDecompItem(TDecompItem(DecompItems[I]).ReqDecomps[J]) is TClassInfo) and
               (not TClassInfo(TDecompItem(DecompItems[I]).ReqDecomps[J]).ForwardDecl) and
               (TDecompItem(DecompItems[I]).ReqDecomps[J].AUnit = Self) then
            begin
              SetSectionType(stType);
              UnitSrc.Add(TClassInfo(TDecompItem(DecompItems[I]).ReqDecomps[J]).AClass.ClassName + ' = class;');
              TClassInfo(TDecompItem(DecompItems[I]).ReqDecomps[J]).ForwardDecl := True;
            end;
          end;
          if TDecompItem(DecompItems[I]) is TProc then
          begin
            TProc(DecompItems[I]).ForwardDecl := True;
            // Proc.
            if not (TProc(DecompItems[I]).ProcType in ptMethods) then
            begin
              // Add a proc definition.
              SetSectionType(stProc);
              UnitSrc.Add(TProc(DecompItems[I]).DefSrc);
            end;
          end
          else if TDecompItem(DecompItems[I]) is TClassInfo then
          begin
            // Add the class declaration.
            SetSectionType(stType);
            UnitSrc.Add(TClassInfo(DecompItems[I]).ClassDef.Text);
            TClassInfo(DecompItems[I]).ForwardDecl := True;
            with TClassInfo(DecompItems[I]) do
            begin
              Str := AClass.ClassName;
              Delete(Str, 22, Length(Str) - 22);
              Consts.Add('  ' + Str + 'ClassConst: TClass = ' + AClass.ClassName + ';');
            end;
          end
          else if TDecompItem(DecompItems[I]) is TVar then
          begin
            if TVar(DecompItems[I]).Name[1] <> '!' then
            begin
              // Var/const.
              if vtVar in TVar(DecompItems[I]).VarConst then
              begin
                // Add a var
                SetSectionType(stVar);
                UnitSrc.Add('  ' + TVar(DecompItems[I]).VarDecl);
              end
              else
              begin
                // Add a const
                SetSectionType(stConst);
                UnitSrc.Add('  ' + TVar(DecompItems[I]).VarDecl);
              end;
            end;
          end
          else if TDecompItem(DecompItems[I]) is TStringInfo then
          begin
            // String
            if TStringInfo(DecompItems[I]).StringType = Procs.stResourceString then
            begin
              SetSectionType(stResourceString);
              UnitSrc.Add(Format('  %s = %s;', [TStringInfo(DecompItems[I]).Name,
                EnhQuotedStr(TStringInfo(DecompItems[I]).Value)]));
              Consts.Add(Format('  %0:sRec: Pointer = @%0:s;', [TStringInfo(DecompItems[I]).Name]));
            end
            else
            begin
              SetSectionType(stConst);
              UnitSrc.Add(Format('  %s: %s = %s;', [TStringInfo(DecompItems[I]).Name,
                TStringInfo(DecompItems[I]).StringTypeName,
                EnhQuotedStr(TStringInfo(DecompItems[I]).Value)]));
            end;
          end
          else if TDecompItem(DecompItems[I]) is TTypeInfoInfo then
          begin
            // Type Info
            with TTypeInfoInfo(DecompItems[I]) do
              if HasTypeDef then
              begin
                SetSectionType(stType);
                UnitSrc.Add('  ' + TypeDef);
                Vars.Add('  ' + TypeInfoVarName + ': Pointer absolute TypeInfo(' + Name + ');');
              end;
          end
          else if TDecompItem(DecompItems[I]) is TNoTInfoType then
          begin
            with TNoTInfoType(DecompItems[I]) do
            begin
              SetSectionType(stType);
              UnitSrc.Add('  ' + Defenition);
            end;
          end
          else if TDecompItem(DecompItems[I]) is TThreadVar then
          begin
            SetSectionType(stThreadvar);
            UnitSrc.Add('  ' + TThreadVar(DecompItems[I]).GetDeclaration);
          end;
        end;
      end;
      SetSectionType(stProc);

      // Implemenation part. --------------------------------
      UnitSrc.Add(ImplUnit);
    end;

    // Add the form DFM
    if DFM <> nil then
      UnitSrc.Add(DFMInclude);

    // Add the implementation uses clause.
    if ImplUnitCount > 0 then
    begin
      if (UnitType = utProgram) and (TPEFileClass(PEFileClass).ProjectType = ptPackage) then
        UnitSrc.Add(ContainsClause)
      else
        UnitSrc.Add(UsesClause);
      Str := '  ';
      for I := 0 to ImplUnitCount -2 do
        Str := Str + ImplUnits[I].Name + ', ';
      Str := Str + ImplUnits[ImplUnitCount -1].Name + ';';
      UnitSrc.Add(Str);
    end;

    // Don't add any declaration to a package main unit.
    if (UnitType <> utProgram) or (TPEFileClass(PEFileClass).ProjectType <> ptPackage) then
    begin
      // Add the some addional system declarations.
      SectionType := stProc;
      for J := 0 to 1 do
        for I := 0 to TUnit(Collection.Items[J]).DecompItems.Count -1 do
          begin
            if (TDecompItem(TUnit(Collection.Items[J]).DecompItems[I]) is TTypeInfoInfo) and
               TTypeInfoInfo(TUnit(Collection.Items[J]).DecompItems[I]).HasTypeDef then
            begin
              with TTypeInfoInfo(TUnit(Collection.Items[J]).DecompItems[I]) do
                for K := 0 to ReqByDecompCount -1 do
                  if ReqByDecomps[K].AUnit = Self then
                  begin
                    SetSectionType(stVar);
                    UnitSrc.Add(Format('  %s: Pointer absolute TypeInfo(%s);', [TypeInfoVarName, TypeInfo.Name]));
                    break;
                  end;
            end
            else if (TDecompItem(TUnit(Collection.Items[J]).DecompItems[I]) is TClassInfo) then
            begin
              with TClassInfo(TUnit(Collection.Items[J]).DecompItems[I]) do
                for K := 0 to ReqByDecompCount -1 do
                  if ReqByDecomps[K].AUnit = Self then
                  begin
                    SetSectionType(stConst);
                    UnitSrc.Add(Format('  %0:sClassConst: TClass = %0:s;', [AClass.ClassName]));
                    break;
                  end;
            end;
          end;

      // Add the implementation declarations.
      for I := 0 to DecompItems.Count -1 do
      begin
        // If this decomp item requires a proc or class without a forward declaration make
        // one now.
        for J := 0 to TDecompItem(DecompItems[I]).ReqDecompCount -1 do
        begin
          if (TDecompItem(DecompItems[I]).ReqDecomps[J] is TProc) and
             (not TProc(TDecompItem(DecompItems[I]).ReqDecomps[J]).ForwardDecl) and
             (not (TProc(TDecompItem(DecompItems[I]).ReqDecomps[J]).ProcType in ptMethods)) and
             (not (TProc(TDecompItem(DecompItems[I]).ReqDecomps[J]).ProcType in [ptInitialization, ptFinalization, ptEntryPointProc])) and
             (TDecompItem(DecompItems[I]).ReqDecomps[J].AUnit = Self) then
          begin
            SetSectionType(stProc);
            UnitSrc.Add(TProc(TDecompItem(DecompItems[I]).ReqDecomps[J]).DefSrc + ' forward;');
            TProc(TDecompItem(DecompItems[I]).ReqDecomps[J]).ForwardDecl := True;
          end;
          if (not (TDecompItem(DecompItems[I]) is TProc)) and
             (TDecompItem(DecompItems[I]).ReqDecomps[J] is TClassInfo) and
             (not TClassInfo(TDecompItem(DecompItems[I]).ReqDecomps[J]).ForwardDecl) and
             (TDecompItem(DecompItems[I]).ReqDecomps[J].AUnit = Self) then
          begin
            SetSectionType(stType);
            UnitSrc.Add(TClassInfo(TDecompItem(DecompItems[I]).ReqDecomps[J]).AClass.ClassName + ' = class;');
            TClassInfo(TDecompItem(DecompItems[I]).ReqDecomps[J]).ForwardDecl := True;
          end;
        end;

        if TDecompItem(DecompItems[I]) is TProc then
        begin
          // Don't add it if it is the initlization of finalization proce
          if (TProc(DecompItems[I]) <> Init) and (TProc(DecompItems[I]) <> FInit) and
             (TProc(DecompItems[I]).ProcType <> ptEntryPointProc) then
          begin
            // Add comments.
            AddComments(TDecompItem(DecompItems[I]).Comments);
            TProc(DecompItems[I]).ForwardDecl := True;
            SetSectionType(stProc);
            if TProc(DecompItems[I]).Instr.Text.Text = '' then
            begin
              if TProc(DecompItems[I]).IntfImpl = iiImplementation then
                UnitSrc.Add(TProc(DecompItems[I]).DefSrc);
            end
            else
              UnitSrc.Add(TProc(DecompItems[I]).Instr.Text.Text);
          end;
        end
        else
        begin
          if TDecompItem(DecompItems[I]).IntfImpl = iiImplementation then
          begin
            // Add the comments
            AddComments(TDecompItem(DecompItems[I]).Comments);

            if TDecompItem(DecompItems[I]) is TClassInfo then
            begin
              SetSectionType(stType);
              UnitSrc.Add(TClassInfo(DecompItems[I]).ClassDef.Text);
              TClassInfo(DecompItems[I]).ForwardDecl := True;
              with TClassInfo(DecompItems[I]) do
              begin
                Str := AClass.ClassName;
                Delete(Str, 22, Length(Str) - 22);
                Consts.Add('  ' + Str + 'ClassConst: TClass = ' + AClass.ClassName + ';');
              end;
            end
            else if TDecompItem(DecompItems[I]) is TVar then
            begin
              if TVar(DecompItems[I]).Name[1] <> '!' then
              begin
                // Var/const.
                if vtVar in TVar(DecompItems[I]).VarConst then
                begin
                  // Add a var
                  SetSectionType(stVar);
                  UnitSrc.Add('  ' + TVar(DecompItems[I]).VarDecl);
                end
                else
                begin
                  // Add a const
                  SetSectionType(stConst);
                  UnitSrc.Add('  ' + TVar(DecompItems[I]).VarDecl);
                end;
              end;
            end
            else if TDecompItem(DecompItems[I]) is TStringInfo then
            begin
              // String
              if TStringInfo(DecompItems[I]).StringType = Procs.stResourceString then
              begin
                SetSectionType(stResourceString);
                UnitSrc.Add(Format('  %s = %s;', [TStringInfo(DecompItems[I]).Name,
                  EnhQuotedStr(TStringInfo(DecompItems[I]).Value)]));
                Consts.Add(Format('  %0:sRec: Pointer = @%0:s;', [TStringInfo(DecompItems[I]).Name]));
              end
              else
              begin
                SetSectionType(stConst);
                UnitSrc.Add(Format('  %s: %s = %s;', [TStringInfo(DecompItems[I]).Name,
                  TStringInfo(DecompItems[I]).StringTypeName,
                  EnhQuotedStr(TStringInfo(DecompItems[I]).Value)]));
              end;
            end
            else if TDecompItem(DecompItems[I]) is TTypeInfoInfo then
            begin
              with TTypeInfoInfo(DecompItems[I]) do
                if HasTypeDef then
                begin
                  SetSectionType(stType);
                  UnitSrc.Add('  ' + TypeDef);
                  Vars.Add(Format('  %s: Pointer absolute TypeInfo(%s);', [TypeInfoVarName, Name]));
                end;
            end
            else if TDecompItem(DecompItems[I]) is TNoTInfoType then
            begin
              with TNoTInfoType(DecompItems[I]) do
              begin
                SetSectionType(stType);
                UnitSrc.Add('  ' + Defenition);
              end;
            end
            else if TDecompItem(DecompItems[I]) is TThreadVar then
            begin
              SetSectionType(stThreadvar);
              UnitSrc.Add('  ' + TThreadVar(DecompItems[I]).GetDeclaration);
            end;
          end;
        end;
      end;
    end;

    SetSectionType(stProc);
    if UnitType = utNormal then
    begin
      // Add the inilization and finalization sections.
      if (Init <> nil) then
      begin
        AddComments(Init.Comments);
        UnitSrc.Add(Init.Instr.Text.Text);
      end;
      if FInit <> nil then
      begin
        AddComments(FInit.Comments);
        UnitSrc.Add(FInit.Instr.Text.Text);
      end;
      UnitSrc.Add(EndUnit);
    end
    else
    begin
      if TPEFileClass(PEFileClass).ProjectType = ptPackage then
        // Add "end." if this is an package
        UnitSrc.Add(EndUnit)
      else
        // Add the entrypoint proc (ignore the finalization section)
        UnitSrc.Add(TPEFileClass(TProcs(Collection).PEFileClass).EntryPointProc.Instr.Text.Text);
    end;
   finally
     Consts.Free;
   end;
  finally
    Vars.Free;
  end;
end;

procedure TUnit.DeterIntfImpl;
var
  I, J: Integer;
label
  Next;
begin
  for I := 0 to DecompItems.Count -1 do
  begin
    with TDecompItem(DecompItems[I]) do
    begin
      // Don't check for program source objects
      if TUnit(TDecompItem(DecompItems[I]).AUnit).UnitType = utProgram then
        goto Next;  
      // Add a var to the interface section to maintain the order of the vars.
      if (TDecompItem(DecompItems[I]) is TVar) and (Size <> 0) then
        IntfImpl := iiInterface;
      // If it already is in the interface section leave it there.
      if IntfImpl = iiInterface then
        goto Next;
      // If it is a Initialization or finalization proc it is in implementation.
      if (TDecompItem(DecompItems[I]) is TProc) and
         (TProc(DecompItems[I]).ProcType in [ptInitialization, ptFinalization]) then
        goto Next;
      // If this item is required by an item in another unit it must be in the interface section.
      for J := 0 to ReqByDecompCount -1 do
        if (TDecompItem(ReqByDecomps[J]).AUnit <> Self) and
           (TDecompItem(ReqByDecomps[J]).AUnit <> nil) then
        begin
          IntfImpl := iiInterface;
          goto Next;
        end;
      Next:
    end;
  end;
end;

{ TUnits }

constructor TUnits.Create(PEFileClass: TPEFile);
begin
  inherited Create(TUnit);
  FPEFileClass := PEFileClass as TPEFileClass;
  FOnAssignUnits := TmlneMethodList.Create;
end;

destructor TUnits.Destroy;
begin
  FOnAssignUnits.Free;
  inherited Destroy;
end;

function TUnits.FindInUnitUsingFInit(Address: PChar): TUnit;
var
  J: Integer;
begin
  // The unit in which Address is, is the unit which Finalization address is
  // the closest after the address.
  Result := nil;
  for J := 0 to Count -1 do
    if (Items[J].FInit.Address >= Address) and
       ((Result = nil) or (Result.FInit.Address > Items[J].FInit.Address)) then
      Result := Items[J];
  // The addres must be inside an Unit.
  if Result = nil then
    raise EDecompilerError.CreateFmt('Address not in an unit %p', [Pointer(Address)]);
end;

function TUnits.FindInUnit(Address: PChar): TUnit;
var
  J: Integer;
begin
  for J := 0 to Count -1 do
  begin
    Result := Items[J];
    if (Result.Address <= Address) and (Result.Address + Result.Size > Address) then
      exit;
  end;
  // If the address isn't in a unit it might be directly after the project unit.
  if (Address > ProgramUnit.Address) and
     (Address < TPEFileClass(PEFileClass).Code + TPEFileClass(PEFileClass).CodeSize) then
  begin
    Result := ProgramUnit;
    Exit;
  end;
  // The addres must be inside an Unit.
  raise EDecompilerError.CreateFmt('Address not in an unit %p,', [Pointer(Address)]);
end;

function TUnits.FindByName(const Name: string): Integer;
begin
  for Result := 0 to Count -1 do
    if AnsiCompareText(Items[Result].Name, Name) = 0 then
      Exit;
  Result := -1;
end;

procedure TUnits.GenerateReqUnits;
var
  I, J, K: Integer;

  procedure SortInitList;

    procedure CorrectI(StartI, EndI: Integer; Intf: Boolean; AUnit: TUnit);

      procedure AddAtIndex(Index: Integer; XAUnit: TUnit);
      begin
        if Intf then
          AUnit.InsertIntfUnit(Index, XAUnit)
        else
          AUnit.InsertImplUnit(Index, XAUnit);
      end;

    var
      StartBlock, EndBlock: Integer;
      ProjectIndex: Integer;
      X: Boolean;
      A, B: Integer;
    begin
      StartBlock := StartI;
      ProjectIndex := 0;

      while StartBlock <= EndI do
      begin
        A := StartBlock;
        EndBlock := StartBlock;
        repeat
          for B := 0 to Items[A].IntfUnitCount -1 do
            if (Items[A].IntfUnits[B].Index > EndBlock) and
               (Items[A].IntfUnits[B].Index <= EndI) then
              EndBlock := Items[A].IntfUnits[B].Index;
          for B := 0 to Items[A].ImplUnitCount -1 do
            if (Items[A].ImplUnits[B].Index > EndBlock) and
               (Items[A].ImplUnits[B].Index <= EndI) then
              EndBlock := Items[A].ImplUnits[B].Index;
          Inc(A);
        until A > EndBlock;

        // If this is only one unit add it.
        if StartBlock = EndBlock then
        begin
          AddAtIndex(ProjectIndex, Items[StartBlock]);
          Inc(ProjectIndex);
        end
        else if Items[EndBlock].FIntfUnits.Count = 0 then
        begin
          // If the last units hasn't interface units, let is handle
          // it in the implementation uses.
          AddAtIndex(ProjectIndex, Items[EndBlock]);
          CorrectI(StartBlock, EndBlock -1, False, Items[EndBlock]);
          Inc(ProjectIndex);
        end
        else
        begin
          // If the last unit isn't required in a in interface section of the other
          // units.
          X := True;
          for B := StartBlock to EndBlock -1 do
            X := X and (Items[B].FIntfUnits.IndexOf(Items[EndBlock]) = -1);
          if X then
          begin
            AddAtIndex(ProjectIndex, Items[EndBlock]);
            CorrectI(StartBlock, EndBlock -1, True, Items[EndBlock]);
            Inc(ProjectIndex);
          end
          else
          begin
            // If the last unit doens't require an of the other units in its
            // interface units.
            X := True;
            for B := StartBlock to EndBlock -1 do
              X := X and (Items[EndBlock].FIntfUnits.IndexOf(Items[B]) = -1);
            if X then
            begin
              AddAtIndex(ProjectIndex, Items[EndBlock]);
              CorrectI(StartBlock, EndBlock -1, False, Items[EndBlock]);
              Inc(ProjectIndex);
            end
            else
            begin
              // If the last unit isn't required by the unit before it,
              // Add the last unit and transfer the problem to the one therefor.
              if Items[EndBlock -1].FIntfUnits.IndexOf(Items[EndBlock]) = -1 then
              begin
                AddAtIndex(ProjectIndex, Items[EndBlock]);
                Inc(ProjectIndex);
                CorrectI(StartBlock, EndBlock -1, True, Items[EndBlock]);
              end;
            end;
          end;
        end;

        // Next block.
        StartBlock := EndBlock + 1;
      end;
    end;

  begin
    CorrectI(2, Count -2, False, Items[Count -1]);
  end;

begin
    for I := 2 to Count -1 do
    begin
      for J := 0 to Items[I].DecompItems.Count -1 do
      begin
        for K := 0 to TDecompItem(Items[I].DecompItems[J]).ReqDecompCount -1 do
        begin
          if TDecompItem(TDecompItem(Items[I].DecompItems[J]).ReqDecomps[K]).AUnit <> nil then
          begin
            if not (TDecompItem(TDecompItem(Items[I].DecompItems[J]).ReqDecomps[K]).AUnit is TUnit) then
              raise EDecompilerError.Create('Unit not an unit');
            if (not (TDecompItem(Items[I].DecompItems[J]) is TProc)) and
               (TDecompItem(Items[I].DecompItems[J]).IntfImpl = iiInterface) then
              Items[I].AddIntfUnit(TDecompItem(TDecompItem(Items[I].DecompItems[J]).ReqDecomps[K]).AUnit as TUnit)
            else
              Items[I].AddImplUnit(TDecompItem(TDecompItem(Items[I].DecompItems[J]).ReqDecomps[K]).AUnit as TUnit);
          end;
        end;
      end;
    end;

    SortInitList;
end;

procedure TUnits.GenerateNames;
var
  I, J: Integer;
begin
  J := 1;
  for I := 0 to Count -2 do
  begin
    if Items[I].Name = '' then
    begin
      // There is no check for a already existing UnknownX unit.
      Items[I].Name := Format('Unknown%d', [J]);
      Inc(J);
    end;
  end;
  if Items[Count -1].Name = '' then
  begin
    // There is no check for a already existing Project1 unit.
    Items[Count -1].Name := 'Project1';
  end;
end;

function TUnits.GetItem(Index: Integer): TUnit;
begin
  Result := TUnit(inherited GetItem(Index));
end;

procedure TUnits.SetItem(Index: Integer; Value: TUnit);
begin
  inherited SetItem(Index, Value);
end;

procedure TUnits.GenUnitSrcs;
var
  I: Integer;
begin
  for I := 0 to Count -1 do
    Items[I].GenUnitSrc;
end;

procedure TUnits.DeterIntfImpls;
var
  I: Integer;
begin
  for I := 0 to Count -1 do
    Items[I].DeterIntfImpl;
end;

procedure TUnits.AssignUnits;
var
  I: Integer;
begin
  with TPEFileClass(FPEFileClass) do
    for I := 0 to Decomps.Count -1 do
      if (TDecompItem(Decomps[I]).Address < Data) and
         (TDecompItem(Decomps[I]).Address <> nil) and
         (TDecompItem(Decomps[I]).AUnit = nil) then
        TDecompItem(Decomps[I]).AUnit := Units.FindInUnit(TDecompItem(Decomps[I]).Address);
  // Call the event handler
  OnAssignUnits.CallFirst;
end;

procedure TUnits.LoadInitFInit;
var
  I: Integer;
  XUnit: TUnit;
begin
  with TPEFileClass(PEFileClass) do
  begin
    // Loop though the init table.
    with TDisAsm.Create do
    try
      for I := 0 to InitTable^.UnitCount -1 do
      begin
        // Must not be an imported unit.
        XUnit := TUnit.Create(Self);
        with XUnit do
        begin
          // Set the unit type.
          if I = 0 then
          begin
            FUnitType := utSystem;
            FSysInitUnit := XUnit;
          end
          else if (I = 1) then
          begin
            FUnitType := utSystem;
            FSystemUnit := XUnit;
          end
          else if (I = InitTable^.UnitCount -1) and (ProjectType <> ptPackage) then
          begin
            FUnitType := utProgram;
            FProgramUnit := XUnit;
          end
          else
          begin
            FUnitType := utNormal;
            if FirstNormalUnit = nil then
              FFirstNormalUnit := XUnit;
          end;
          // Create the init proc.
          if Assigned(InitTable^.UnitInfo^[I].Init) then
          begin
            Init := TInitProc.CreateInit(Procs, @InitTable^.UnitInfo^[I].Init);
            Init.AUnit := XUnit;
          end;
          // Units must have a finalization section (required for unit size etc).
          if @InitTable^.UnitInfo^[I].FInit = nil then
            raise EDecompilerError.Create('Every Unit must have an Finalization section');
          // Create the FInit proc.
          FInit := TInitProc.CreateFInit(Procs, @InitTable^.UnitInfo^[I].FInit);
          FInit.AUnit := XUnit;
        end
      end;
    finally
      Free;
    end;
    if ProjectType = ptPackage then
    begin
      // Load the package unit.
      XUnit := TUnit.Create(Self);
      FProgramUnit := XUnit;
      with XUnit do
      begin
        // Set the unit type
        FUnitType := utProgram;
        // Set PackageUnload as the finalization unit.
        FInit := TProc.Create(Procs, PEExports[PEExports.FindCaseInSens('@' + ProjectName + '@@PackageUnload$qqrv')].Address);
        with FInit do
        begin
          AUnit := XUnit;
          AppendBefore := atMayNot;
        end;
        // Create the initilaization unit
        Init := TProc.Create(Procs, PEExports[PEExports.FindCaseInSens('@' + ProjectName + '@initialization$qqrv')].Address);
        with Init do
        begin
          AUnit := XUnit;
          AppendBefore := atMayNot;
          AppendAfter := atMayNot;
        end;
      end;
    end;
  end;
end;

end.


