unit dcType;

interface

uses
  TypInfo, Classes, Procs, dcTypeIntf;

type
  TdcType = class;

  { TdcTypeImplement }

  TdcTypeImplement = class(TInterfacedObject, IdcType, IdcArrayType, IdcClassType,
                                    IdcDynArrayType, IdcEnumerationType,
                                    IdcFloatType, IdcInt64Type, IdcInterfaceType,
                                    IdcNonSetOrdType, IdcOrdType, IdcRecordType,
                                    IdcStringType, IdcSetOrdType)
  private
    FdcType: TdcType;
    function GetType: IdcType;
    function GetArrayType: IdcArrayType;
    function GetClassType: IdcClassType;
    function GetDynArrayType: IdcDynArrayType;
    function GetEnumerationType: IdcEnumerationType;
    function GetFloatType: IdcFloatType;
    function GetInt64Type: IdcInt64Type;
    function GetInterfaceType: IdcInterfaceType;
    function GetNonSetOrdType: IdcNonSetOrdType;
    function GetOrdType: IdcOrdType;
    function GetRecordType: IdcRecordType;
    function GetSetOrdType: IdcSetOrdType;
    function GetStringType: IdcStringType;
  public
    constructor CreateX(dcType: TdcType); virtual;
    property dcType: IdcType read GetType implements IdcType;
    property dcArrayType: IdcArrayType read GetArrayType implements IdcArrayType;
    property dcClassType: IdcClassType read GetClassType implements IdcClassType;
    property dcDynArrayType: IdcDynArrayType read GetDynArrayType implements IdcDynArrayType;
    property dcEnumerationType: IdcEnumerationType read GetEnumerationType implements IdcEnumerationType;
    property dcFloatType: IdcFloatType read GetFloatType implements IdcFloatType;
    property dcInt64Type: IdcInt64Type read GetInt64Type implements IdcInt64Type;
    property dcInterfaceType: IdcInterfaceType read GetInterfaceType implements IdcInterfaceType;
    property dcNonSetOrdType: IdcNonSetOrdType read GetNonSetOrdType implements IdcNonSetOrdType;
    property dcOrdType: IdcOrdType read GetOrdType implements IdcOrdType;
    property dcRecordType: IdcRecordType read GetRecordType implements IdcRecordType;
    property dcSetOrdType: IdcSetOrdType read GetSetOrdType implements IdcSetOrdType;
    property dcStringType: IdcStringType read GetStringType implements IdcStringType;
  end;

  TdcTypeImplementClass = class of TdcTypeImplement;
  
  { TdcArrayType }
  TdcArrayType = class(TdcTypeImplement, IdcArrayType)
  private
    FCount: Integer;
    FElementType: IdcType;
  public
    function GetCount: Integer;
    procedure SetCount(Value: Integer);
    function GetElementType: IdcType;
    procedure SetElementType(Value: IdcType);
    property Count: Integer read GetCount write SetCount;
    property ElementType: IdcType read GetElementType write SetElementType;
  end;

  { TdcClassType }
  TdcClassType = class(TdcTypeImplement, IdcClassType)
  private
    FAClass: TClassInfo;
  public
    function GetAClass: TClassInfo;
    procedure SetAClass(Value: TClassInfo);
    property AClass: TClassInfo read GetAClass write SetAClass;
  end;

  { TdcDynArrayType }
  TdcDynArrayType = class(TdcTypeImplement, IdcDynArrayType)
  private
    FElementType: IdcType;
  public
    function GetElementType: IdcType;
    procedure SetElementType(Value: IdcType);
    property ElementType: IdcType read GetElementType write SetElementType;
  end;

  { TdcEnumerationType }
  TdcEnumerationType = class(TdcTypeImplement, IdcEnumerationType)
  private
    FBaseType: IdcType;
    FNameList: TStrings;
  public
    constructor CreateX(dcType: TdcType); override;
    destructor Destroy; override;
    function GetBaseType: IdcType;
    procedure SetBaseType(Value: IdcType);
    function GetNameList: TStrings;
    procedure SetNameList(Value: TStrings);
    property BaseType: IdcType read GetBaseType write SetBaseType;
    property NameList: TStrings read GetNameList write SetNameList;
  end;

  { TdcFloatType }
  TdcFloatType = class(TdcTypeImplement, IdcFloatType)
  private
    FFloatType: TFloatType;
  public
    function GetFloatType: TFloatType;
    procedure SetFloatType(Value: TFloatType);
    property FloatType: TFloatType read GetFloatType write SetFloatType;
  end;

  { TdcInt64Type }
  TdcInt64Type = class(TdcTypeImplement, IdcInt64Type)
  private
    FMinValue: Int64;
    FMaxValue: Int64;
  public
    function GetMinValue: Int64;
    procedure SetMinValue(Value: Int64);
    function GetMaxValue: Int64;
    procedure SetMaxValue(Value: Int64);
    property MinValue: Int64 read GetMinValue write SetMinValue;
    property MaxValue: Int64 read GetMaxValue write SetMaxValue;
  end;

  { TdcInterfaceType }
  TdcInterfaceType = class(TdcTypeImplement, IdcInterfaceType)
  private
    FAInterface: TInterface;
  public
    function GetInterface: TInterface;
    procedure SetInterface(Value: TInterface);
    property AInterface: TInterface read GetInterface write SetInterface;
  end;

  { TdcNonSetOrdType }
  TdcNonSetOrdType = class(TdcTypeImplement, IdcNonSetOrdType)
  private
    FMinValue: Integer;
    FMaxValue: Integer;
  public
    function GetMinValue: Integer;
    procedure SetMinValue(Value: Integer);
    function GetMaxValue: Integer;
    procedure SetMaxValue(Value: Integer);
    property MinValue: Integer read GetMinValue write SetMinValue;
    property MaxValue: Integer read GetMaxValue write SetMaxValue;
  end;

  { TdcOrdType }
  TdcOrdType = class(TdcTypeImplement, IdcOrdType)
  private
    FOrdType: TOrdType;
  public
    function GetOrdType: TOrdType;
    procedure SetOrdType(Value: TOrdType);
    property OrdType: TOrdType read GetOrdType write SetOrdType;
  end;

  { TdcRecordType }
  TdcRecordType = class(TdcTypeImplement, IdcRecordType)
  private
    FCount: Integer;
    FFields: TList;
  public
    constructor CreateX(dcType: TdcType); override;
    destructor Destroy; override;
    function GetCount: Integer;
    procedure SetCount(Value: Integer);
    function GetField(Index: Integer): TdcRecordField;
    procedure SetField(Index: Integer; Value: TdcRecordField);
    property Count: Integer read GetCount write SetCount;
    property Fields[Index: Integer]: TdcRecordField read GetField write SetField;
  end;

  { TdcSetOrdType }
  TdcSetOrdType = class(TdcTypeImplement, IdcSetOrdType)
  private
    FCompType: IdcType;
  public
    function GetCompType: IdcType;
    procedure SetCompType(Value: IdcType);
    property CompType: IdcType read GetCompType write SetCompType;
  end;

  { TdcStringType }
  TdcStringType = class(TdcTypeImplement, IdcStringType)
  private
    FMaxLength: Integer;
  public
    function GetMaxLength: Integer;
    procedure SetMaxLength(Value: Integer);
    property MaxLength: Integer read GetMaxLength write SetMaxLength;
  end;

  { TdcType }

  TdcType = class(TInterfacedObject, IdcType, IdcArrayType, IdcClassType,
                                     IdcDynArrayType, IdcEnumerationType,
                                     IdcFloatType, IdcInt64Type, IdcInterfaceType,
                                     IdcNonSetOrdType, IdcOrdType, IdcRecordType,
                                     IdcStringType, IdcSetOrdType)
  private
    FPossTypeKinds: TEnhTypeKinds;
    FPossSizes: TTypeSizes;
    // interface implementation vars
    FTypeData: array[1..3] of IUnknown;
    // properties setters.
    function GetPossTypeKinds: TEnhTypeKinds;
    procedure SetPossTypeKinds(Value: TEnhTypeKinds);
    function GetTypeKind: TEnhTypeKind;
    function GetPossSizes: TTypeSizes;
    procedure SetPossSizes(Value: TTypeSizes);
    // Implementations getters.
    function GetArrayType: IdcArrayType;
    function GetClassType: IdcClassType;
    function GetDynArrayType: IdcDynArrayType;
    function GetEnumerationType: IdcEnumerationType;
    function GetFloatType: IdcFloatType;
    function GetInt64Type: IdcInt64Type;
    function GetInterfaceType: IdcInterfaceType;
    function GetNonSetOrdType: IdcNonSetOrdType;
    function GetOrdType: IdcOrdType;
    function GetRecordType: IdcRecordType;
    function GetSetOrdType: IdcSetOrdType;
    function GetStringType: IdcStringType;
  public
    constructor Create; virtual;
    // Properties.
    property PossTypeKinds: TEnhTypeKinds read GetPossTypeKinds write SetPossTypeKinds;
    property TypeKind: TEnhTypeKind read GetTypeKind;
    property PossSizes: TTypeSizes read GetPossSizes write SetPossSizes;
    // Interface implementations.
    property ArrayType: IdcArrayType read GetArrayType implements IdcArrayType;
    property AClassType: IdcClassType read GetClassType implements IdcClassType;
    property DynArrayType: IdcDynArrayType read GetDynArrayType implements IdcDynArrayType;
    property EnumerationType: IdcEnumerationType read GetEnumerationType implements IdcEnumerationType;
    property FloatType: IdcFloatType read GetFloatType implements IdcFloatType;
    property Int64Type: IdcInt64Type read GetInt64Type implements IdcInt64Type;
    property InterfaceType: IdcInterfaceType read GetInterfaceType implements IdcInterfaceType;
    property NonSetOrdType: IdcNonSetOrdType read GetNonSetOrdType implements IdcNonSetOrdType;
    property OrdType: IdcOrdType read GetOrdType implements IdcOrdType;
    property RecordType: IdcRecordType read GetRecordType implements IdcRecordType;
    property SetOrdType: IdcSetOrdType read GetSetOrdType implements IdcSetOrdType;
    property StringType: IdcStringType read GetStringType implements IdcStringType;
  end;

implementation

uses
  SysUtils, PEFileClass;

{ TdcTypeImplement }

function TdcTypeImplement.GetType: IdcType;
begin
  Result := FdcType as IdcType;
end;

function TdcTypeImplement.GetArrayType: IdcArrayType;
begin
  Result := FdcType as IdcArrayType;
end;

function TdcTypeImplement.GetClassType: IdcClassType;
begin
  Result := FdcType as IdcClassType;
end;

function TdcTypeImplement.GetDynArrayType: IdcDynArrayType;
begin
  Result := FdcType as IdcDynArrayType;
end;

function TdcTypeImplement.GetEnumerationType: IdcEnumerationType;
begin
  Result := FdcType as IdcEnumerationType;
end;

function TdcTypeImplement.GetFloatType: IdcFloatType;
begin
  Result := FdcType as IdcFloatType;
end;

function TdcTypeImplement.GetInt64Type: IdcInt64Type;
begin
  Result := FdcType as IdcInt64Type;
end;

function TdcTypeImplement.GetInterfaceType: IdcInterfaceType;
begin
  Result := FdcType as IdcInterfaceType;
end;

function TdcTypeImplement.GetNonSetOrdType: IdcNonSetOrdType;
begin
  Result := FdcType as IdcNonSetOrdType;
end;

function TdcTypeImplement.GetOrdType: IdcOrdType;
begin
  Result := FdcType as IdcOrdType;
end;

function TdcTypeImplement.GetRecordType: IdcRecordType;
begin
  Result := FdcType as IdcRecordType;
end;

function TdcTypeImplement.GetSetOrdType: IdcSetOrdType;
begin
  Result := FdcType as IdcSetOrdType;
end;

function TdcTypeImplement.GetStringType: IdcStringType;
begin
  Result := FdcType as IdcStringType;
end;

constructor TdcTypeImplement.CreateX(dcType: TdcType);
begin
  FdcType := dcType;
end;

{ TdcArrayType }

function TdcArrayType.GetCount: Integer;
begin
  Result := FCount;
end;

procedure TdcArrayType.SetCount(Value: Integer);
begin
  FCount := Value;
end;

function TdcArrayType.GetElementType: IdcType;
begin
  Result := FElementType;
end;

procedure TdcArrayType.SetElementType(Value: IdcType);
begin
  FElementType := Value;
end;

{ TdcClassType }

function TdcClassType.GetAClass: TClassInfo;
begin
  Result := FAClass;
end;

procedure TdcClassType.SetAClass(Value: TClassInfo);
begin
  FAClass := Value;
end;

{ TdcDynArrayType }

function TdcDynArrayType.GetElementType: IdcType;
begin
  Result := FElementType;
end;

procedure TdcDynArrayType.SetElementType(Value: IdcType);
begin
  FElementType := Value;
end;

{ TdcEnumerationType }

constructor TdcEnumerationType.CreateX(dcType: TdcType);
begin
  inherited CreateX(dcType);
  NameList := TStringList.Create;
end;

destructor TdcEnumerationType.Destroy;
begin
  NameList.Free;
  inherited Destroy;
end;

function TdcEnumerationType.GetBaseType: IdcType;
begin
  Result := FBaseType;
end;

procedure TdcEnumerationType.SetBaseType(Value: IdcType);
begin
  FBaseType := Value;
end;

function TdcEnumerationType.GetNameList: TStrings;
begin
  Result := FNameList;
end;

procedure TdcEnumerationType.SetNameList(Value: TStrings);
begin
  FNameList.Assign(Value);
end;

{ TdcFloatType }

function TdcFloatType.GetFloatType: TFloatType;
begin
  Result := FFloatType;
end;

procedure TdcFloatType.SetFloatType(Value: TFloatType);
begin
  FFloatType := Value;
end;

{ TdcInt64Type }

function TdcInt64Type.GetMaxValue: Int64;
begin
  Result := FMaxValue;
end;

procedure TdcInt64Type.SetMaxValue(Value: Int64);
begin
  FMaxValue := Value;
end;

function TdcInt64Type.GetMinValue: Int64;
begin
  Result := FMinValue;
end;

procedure TdcInt64Type.SetMinValue(Value: Int64);
begin
  FMinValue := Value;
end;

{ TdcInterfaceType }

function TdcInterfaceType.GetInterface: TInterface;
begin
  Result := FAInterface;
end;

procedure TdcInterfaceType.SetInterface(Value: TInterface);
begin
  FAInterface := Value;
end;

{ TdcNonSetOrdType }

function TdcNonSetOrdType.GetMaxValue: Integer;
begin
  Result := FMaxValue;
end;

procedure TdcNonSetOrdType.SetMaxValue(Value: Integer);
begin
  FMaxValue := Value;
end;

function TdcNonSetOrdType.GetMinValue: Integer;
begin
  Result := FMinValue;
end;

procedure TdcNonSetOrdType.SetMinValue(Value: Integer);
begin
  FMinValue := Value;
end;

{ TdcOrdType }

function TdcOrdType.GetOrdType: TOrdType;
begin
  Result := FOrdType;
end;

procedure TdcOrdType.SetOrdType(Value: TOrdType);
begin
  FOrdType := Value;
end;

{ TdcRecordType }

constructor TdcRecordType.CreateX(dcType: TdcType);
begin
  inherited CreateX(dcType);
  FFields := TList.Create;
end;

destructor TdcRecordType.Destroy;
begin
  FFields.Free;
  inherited Destroy;
end;

function TdcRecordType.GetCount: Integer;
begin
  Result := FCount;
end;

procedure TdcRecordType.SetCount(Value: Integer);
begin
  FCount := Value;
end;

function TdcRecordType.GetField(Index: Integer): TdcRecordField;
begin
  Result := TdcRecordField(FFields[Index]^);
end;

procedure TdcRecordType.SetField(Index: Integer; Value: TdcRecordField);
begin
  TdcRecordField(FFields[Index]^) := Value;
end;

{ TdcSetOrdType }

function TdcSetOrdType.GetCompType: IdcType;
begin
  Result := FCompType;
end;

procedure TdcSetOrdType.SetCompType(Value: IdcType);
begin
  FCompType := Value;
end;

{ TdcString}

function TdcStringType.GetMaxLength: Integer;
begin
  Result := FMaxLength;
end;

procedure TdcStringType.SetMaxLength(Value: Integer);
begin
  FMaxLength := Value;
end;

{ TdcType }

constructor TdcType.Create;
begin
  inherited Create;
  FPossSizes := [Low(TTypeSize)..High(TTypeSize)];
  FPossTypeKinds := [Low(TEnhTypeKind).. High(TEnhTypeKind)];
end;

function TdcType.GetPossTypeKinds: TEnhTypeKinds;
begin
  Result := FPossTypeKinds;
end;

procedure TdcType.SetPossTypeKinds(Value: TEnhTypeKinds);
const
  TypeDataType: array[TEnhTypeKind] of array[1..3] of TdcTypeImplementClass =
    ((nil, nil, nil), (TdcOrdType, TdcNonSetOrdType, nil), (TdcOrdType, TdcNonSetOrdType, nil),
    (TdcOrdType, TdcNonSetOrdType, TdcEnumerationType), (TdcFloatType, nil, nil),
    (TdcStringType, nil, nil), (TdcOrdType, TdcSetOrdType, nil), (TdcClassType, nil, nil),
    (nil, nil, nil), (TdcOrdType, TdcNonSetOrdType, nil),
    (nil, nil, nil), (nil, nil, nil), (nil, nil, nil), (TdcArrayType, nil, nil),
    (TdcRecordType, nil, nil), (TdcInterfaceType, nil, nil),
    (TdcInt64Type, nil, nil), (TdcDynArrayType, nil, nil), (nil, nil, nil),
    (nil, nil, nil), (nil, nil, nil));
var
  NewTypeKind: TEnhTypeKind;
  OldTypeKind: TEnhTypeKind;
  I: Integer;
begin
  if Value = [] then
    raise EDecompilerError.Create('Empty PossTypeKinds');
  if not (Value <= FPossTypeKinds) then
    raise EDecompilerError.Create('A new PossTypeKind is introduced.');
  // Save the old type kind.
  OldTypeKind := TypeKind;
  // set the private var.
  FPossTypeKinds := Value;
  // Change the TypeData object.
  NewTypeKind := TypeKind;
  for I := 1 to 3 do
    if TypeDataType[OldTypeKind, I] <> TypeDataType[NewTypeKind, I] then
    begin
      FTypeData[I] := nil;
      FTypeData[I] := TypeDataType[NewTypeKind, I].CreateX(Self);
    end;
end;

function TdcType.GetPossSizes: TTypeSizes;
begin
  Result := FPossSizes;
end;

const
  CPossTypeKinds: array[TTypeSize] of TEnhTypeKinds =
    ({ts0}  [etkUnknown, etkUTInteger, etkUTString],
     {ts1}  [etkInteger, etkChar],
     {ts2}  [etkInteger],
     {ts4}  [etkInteger, etkChar, etkEnumeration, etkFloat,
      etkString, etkSet, etkClass, etkWChar, etkLString, etkWString,
      etkInterface, etkDynArray, etkPointer],
     {ts6}  [etkFloat],
     {ts8}  [etkFloat, etkMethod, etkInt64],
     {ts12} [etkFloat],
     {ts16} [etkVariant, etkRecord, etkArray],
     {tsComplex} [etkArray, etkRecord]);

procedure TdcType.SetPossSizes(Value: TTypeSizes);
var
  IncludedSize: TTypeSize;
begin
  if Value = [] then
    raise EDecompilerError.Create('Empty PossTypeKinds');
  if not (Value <= FPossSizes) then
    raise EDecompilerError.Create('A new PossTypeKind is introduced.');
  // Set the var.
  FPossSizes := Value;
  for IncludedSize := Low(TTypeSize) to High(TTypeSize) do
    if IncludedSize in PossSizes then
      PossTypeKinds := PossTypeKinds * CPossTypeKinds[IncludedSize];
end;

function TdcType.GetTypeKind: TEnhTypeKind;
var
  I, J: Integer;
begin
  // Convert the poss type kind to a type kind, take the first set item as the result.
  I := Integer(FPossTypeKinds);
  Assert(I <> 0, 'Empty possible type kind');
  J := 0;
  while I mod 2 = 0 do
  begin
    I := I div 2;
    Inc(J);
  end;
  Result := TEnhTypeKind(J);
end;

function TdcType.GetArrayType: IdcArrayType;
begin
  if (FTypeData[1] = nil) or
     (FTypeData[1].QueryInterface(IdcArrayType, Result) <> 0) then
    raise EDecompilerError.Create('Invalid typecast to type type');
end;

function TdcType.GetClassType: IdcClassType;
begin
  if (FTypeData[1] = nil) or
     (FTypeData[1].QueryInterface(IdcClassType, Result) <> 0) then
    raise EDecompilerError.Create('Invalid typecast to type type');
end;

function TdcType.GetDynArrayType: IdcDynArrayType;
begin
  if (FTypeData[1] = nil) or
     (FTypeData[1].QueryInterface(IdcDynArrayType, Result) <> 0) then
    raise EDecompilerError.Create('Invalid typecast to type type');
end;

function TdcType.GetEnumerationType: IdcEnumerationType;
begin
  if (FTypeData[3] = nil) or
     (FTypeData[3].QueryInterface(IdcEnumerationType, Result) <> 0) then
    raise EDecompilerError.Create('Invalid typecast to type type');
end;

function TdcType.GetFloatType: IdcFloatType;
begin
  if (FTypeData[1] = nil) or
     (FTypeData[1].QueryInterface(IdcFloatType, Result) <> 0) then
    raise EDecompilerError.Create('Invalid typecast to type type');
end;

function TdcType.GetInt64Type: IdcInt64Type;
begin
  if (FTypeData[1] = nil) or
     (FTypeData[1].QueryInterface(IdcInt64Type, Result) <> 0) then
    raise EDecompilerError.Create('Invalid typecast to type type');
end;

function TdcType.GetInterfaceType: IdcInterfaceType;
begin
  if (FTypeData[1] = nil) or
     (FTypeData[1].QueryInterface(IdcInterfaceType, Result)<> 0) then
    raise EDecompilerError.Create('Invalid typecast to type type');
end;

function TdcType.GetNonSetOrdType: IdcNonSetOrdType;
begin
  if (FTypeData[2] = nil) or
     (FTypeData[2].QueryInterface(IdcNonSetOrdType, Result) <> 0) then
    raise EDecompilerError.Create('Invalid typecast to type type');
end;

function TdcType.GetOrdType: IdcOrdType;
begin
  if (FTypeData[1] = nil) or
     (FTypeData[1].QueryInterface(IdcOrdType, Result) <> 0) then
    raise EDecompilerError.Create('Invalid typecast to type type');
end;

function TdcType.GetRecordType: IdcRecordType;
begin
  if (FTypeData[1] = nil) or
     (FTypeData[1].QueryInterface(IdcRecordType, Result) <> 0) then
    raise EDecompilerError.Create('Invalid typecast to type type');
end;

function TdcType.GetSetOrdType: IdcSetOrdType;
begin
  if (FTypeData[2] = nil) or
     (FTypeData[2].QueryInterface(IdcSetOrdType, Result) <> 0) then
    raise EDecompilerError.Create('Invalid typecast to type type');
end;

function TdcType.GetStringType: IdcStringType;
begin
  if (FTypeData[1] = nil) or
     (FTypeData[1].QueryInterface(IdcStringType, Result) <> 0) then
    raise EDecompilerError.Create('Invalid typecast to type type');
end;

end.
