unit dcProcInstr;

interface

uses
  dcDecomps, DisAsmX, Classes, SysUtils, dcOpcToInstr;

type
  { TdcInstructions }

  TdcInstructions = class(TInterfacedObject, IInstrOwner)
  private
    FText: TStrings;
    FProc: TDecompItem;
    FItems: TList;
    function GetText: TStrings;
    function GetAddress: TAddress;
    function GetSize: Integer;
    function GetCount: Integer;
    function GetItem(Index: Integer): TInstruction;
    function GetItemIndex(Item: TInstruction): Integer;
    function InstrToPascal(Strings: TStrings): Boolean;
  public
    constructor Create(AProc: TDecompItem);
    destructor Destroy; override;
    procedure ReplaceAssign;

    property Text: TStrings read GetText;
    property Proc: TDecompItem read FProc;
    property Address: TAddress read GetAddress;
    property Size: Integer read GetSize;
    property Count: Integer read GetCount;
    property Items[Index: Integer]: TInstruction read GetItem;
  end;

  EdcInstructionError = class(Exception);

implementation

uses
  Procs, dcInstrSource, ActiveX, dcAsmInstr, TypInfo, PEFileClass;

{ TdcInstructions }

type
  TInstructionList = array[0..MaxListSize] of TInstruction;

  TIsAddressCallbackObj = class(TObject)
  private
    PEFileClass: TPEFileClass;
    procedure Callback(Imm: TAddress; var IsAddress: Boolean);
  end;

procedure TIsAddressCallbackObj.Callback(Imm: TAddress; var IsAddress: Boolean);
begin
  IsAddress := PEFileClass.Fixups.FindFixupTo(Imm) <> -1;
end;

constructor TdcInstructions.Create(AProc: TDecompItem);
var
  AAddress: TAddress;
  IsAddressCallbackObj: TIsAddressCallbackObj;
begin
  inherited Create;
  FItems := TList.Create;

  FProc := AProc;
  with Proc as TProc do
  begin
    // Prevent further appending.
    AppendBefore := atMayNot;
    AppendAfter := atMayNot;
  end;

  IsAddressCallbackObj := TIsAddressCallbackObj.Create;
  try
    IsAddressCallbackObj.PEFileClass := TPEFileClass(TProc(Proc).PEFileClass);
    IsImmAddressCallback := IsAddressCallbackObj.Callback;
    // Create the instructions
    AAddress := TProc(Proc).Address;
    while AAddress < TProc(Proc).Address + TProc(Proc).ProcSize do
    begin
      FItems.Add(GetInstrClass(AAddress).Create(AAddress));
      // Create an Asm instruction if the original create instruction is to big.
      if Items[Count -1].Address + Items[Count -1].Size > TProc(Proc).Address + TProc(Proc).ProcSize then
      begin
        TInstruction(FItems[Count -1]).Free;
        FItems[Count -1] := TAsmInstr.CreateOpc(AAddress,
                TProc(Proc).Address + TProc(Proc).ProcSize - AAddress);
      end;
      AAddress := AAddress + Items[Count -1].Size;
    end;
  finally
    IsAddressCallbackObj.Free;
    IsImmAddressCallback := nil;
  end;

  // Try Replace some assign things.
  ReplaceAssign;
end;

destructor TdcInstructions.Destroy;
var
  I: Integer;
begin
  inherited Destroy;
  for I := 0 to FItems.Count -1 do
    TInstruction(FItems[I]).Free;
  FItems.Free;
end;

type
  TProcInstrReplAssCallback = class(TObject)
  private
    Reg: TRegister;
    Used: Boolean;
    procedure Callback(Source: IInstrSource);
  end;

procedure TProcInstrReplAssCallback.Callback(Source: IInstrSource);
var
  RegSource: IRegSource;
begin
  if not Used then
    if Succeeded(Source.QueryInterface(IRegSource, RegSource)) then
      if RegSource.Reg = Reg then
        Used := True;
end;

procedure TdcInstructions.ReplaceAssign;
// Try replacing two succeeding assignment instructions with one.
var
  I: Integer;
  ASource: IInstrSource;
  ATarget: IInstrSource;
  ARegTarget: IRegSource;
  Instr: TInstruction;
  ReplCallback: TProcInstrReplAssCallback;

  function IsRegSetAgain: Boolean;
  var
    J: Integer;
  begin
    // The Target is used as source only in the second instruction, or already
    // been set before if is used as source, because if it were used after the second
    // instruction it might be changed
    ReplCallback.Used := False;
    ReplCallback.Reg := ARegTarget.Reg;
    for J := I + 1 to Count -1 do
    begin
      if Items[J] is TAssignInstr then
      begin
        ReplCallback.Callback(TAssignInstr(Items[J]).Source);
        if ReplCallback.Used then
        begin
          Result := False;
          Exit;
        end;
        ReplCallback.Callback(TAssignInstr(Items[J]).Target);
        if ReplCallback.Used then
        begin
          Result := True;
          Exit;
        end;
      end
      else if Items[J] is TRetInstr then
      begin
        // Reg is only used if it is eax (return value).
        Result := ARegTarget.Reg <> rEax;
        Exit;
      end
      else
        raise EdcInstructionError.Create('Unknown Instruction');
    end;
    Result := False;
  end;

begin
  // All instructions must be a assign or Ret.
  for I := 0 to Count -1 do
    if not ((Items[I] is TAssignInstr) or (Items[I] is TRetInstr)) then
      Exit;
  // Last instruction must be a Ret instruction of course.
  if not ((Count > 0) and (Items[Count -1] is TRetInstr)) then
    Exit;
  { TODO -cAdditions : Add checking for call instructions, not yet needed, because there is now call suport yet. }

  ReplCallback := TProcInstrReplAssCallback.Create;
  try
    // Replace two successive Assign instructions, with the first having
    // the target in the second source.
    for I := Count -1 downto 1 do
    begin
      if (Items[I] is TAssignInstr) and
          // The target of the first instruction is a register.
         (Items[I -1] is TAssignInstr) and
         Succeeded(TAssignInstr(Items[I -1]).Target.QueryInterface(IRegSource, ARegTarget)) and
         // The Target is set again and not used before it is set again
         IsRegSetAgain then
      begin
        ASource := TAssignInstr(Items[I]).Source.Replace(
            TAssignInstr(Items[I -1]).Target, TAssignInstr(Items[I -1]).Source);
        ATarget := TassignInstr(Items[I]).Target;
        Instr := TAssignInstr.CreateAssign(Items[I -1].Address, Items[I].Size + Items[I -1].Size, ATarget, ASource);
        TInstruction(FItems[I -1]).Free;
        TInstruction(FItems[I]).Free;
        FItems[I -1] := Instr;
        FItems.Delete(I);
      end;
    end;
  finally
    ReplCallback.Free;
  end;
end;

function TdcInstructions.GetAddress: TAddress;
begin
  Result := FProc.Address;
end;

function TdcInstructions.GetSize: Integer;
begin
  Result := TProc(FProc).ProcSize;
end;

function TdcInstructions.GetCount: Integer;
begin
  Result := FItems.Count;
end;

function TdcInstructions.GetItem(Index: Integer): TInstruction;
begin
  Result := TInstruction(FItems[Index]);
end;

function TdcInstructions.GetItemIndex(Item: TInstruction): Integer;
resourcestring
  EItemNotFound = 'Item not found in instruction list.';
begin
  Result := FItems.IndexOf(Item);
  if Result = -1 then
    raise EDecompilerError.Create(EItemNotFound);
end;

function TdcInstructions.GetText: TStrings;
begin
  Result := FText;
  if Result = nil then
  begin
    FText := TStringList.Create;
    Result := FText;
    // Get the instructions in pascal, or if failed in Assembler.
    if not InstrToPascal(Result) then
      GetAsmInstr(TProc(Proc), Result);
  end;
end;

type
  TUsedSourceCallback = class(TObject)
  private
    Used: set of TRegister;
    UsedBeforeWrite: set of TRegister;
    procedure TargetCallback(Source: IInstrSource);
    procedure SourceCallback(Source: IInstrSource);
  end;

procedure TUsedSourceCallback.TargetCallback(Source: IInstrSource);
var
  RegSource: IRegSource;
begin
  // Is is a a reg source and the register is not already used, mark it as used.`
  if Succeeded(Source.QueryInterface(IRegSource, RegSource)) and
     (not (RegSource.Reg in Used)) then
    Used := Used + [RegSource.Reg];
end;

procedure TUsedSourceCallback.SourceCallback(Source: IInstrSource);
var
  RegSource: IRegSource;
begin
  // Is is a a reg source and the register is not already used, mark it
  // as used before write and used.
  if Succeeded(Source.QueryInterface(IRegSource, RegSource)) and
     (not (RegSource.Reg in Used)) then
  begin
    UsedBeforeWrite := UsedBeforeWrite + [RegSource.Reg];
    Used := Used + [RegSource.Reg];
  end;
end;

type
  TRegNames = array[TRegister] of string;
  TPascalCallback = class(TObject)
  private
    PascalRegNames: TRegNames;
    PascalProc: TProc;
    procedure Callback(Source: IInstrSource; var AsPascal: string);
  end;

procedure TPascalCallback.Callback(Source: IInstrSource; var AsPascal: string);
var
  RegSource: IRegSource;
  AddressSource: IAddressSource;
resourcestring
  ERegWithoutAName = 'Reg doesn''t have a name (Bug in code)';
  EFixupConstError = 'A Const value which is a fixup. Not yet supported.';
begin
  // If it is a RegSource replace the AsPascal with the RegName
  if Succeeded(Source.QueryInterface(IRegSource, RegSource)) then
  begin
    AsPascal := PascalRegNames[RegSource.Reg];
    if AsPascal = '' then
      raise EDecompilerError.Create(ERegWithoutAName);
  end;
  if Succeeded(Source.QueryInterface(IAddressSource, AddressSource)) then
    raise EdcInstructionError.Create(EFixupConstError);
end;

function TdcInstructions.InstrToPascal(Strings: TStrings): Boolean;
const
  ProcDef: array[TProcType, Boolean] of string =
    (('procedure  %s(%s);', 'function %s(%s): %s;'), ('class procedure %s.%s(%s);', 'class function %s.%s(%s): %s;'),
     ('procedure %s.%s(%s);', 'function %s.%s(%s): %s;'), ('constructor %s.%s(%s);', 'constructor %s.%s(%s);'),
     ('destructor %s.%s(%s);', 'destructor %s.%s(%s);'), ('', ''), ('', ''), ('', ''));
var
  I: Integer;
  UsedSourceCallback: TUsedSourceCallback;
  Vars: set of TRegister;
  Params: set of TRegister;
  RegNames: TRegNames;
  Reg: TRegister;
begin
  Result := False;
  try
    // Must be a Strings object passed.
    if Strings = nil then
      Exit;

    // The proc must be only assign instructions.
    for I := 0 to Count -2 do
      if not (Items[I] is TAssignInstr) then
        Exit;
    // Except the last one which must be a ret instruction.
    if not ((Count > 0) and (Items[count -1] is TRetInstr)) then
      Exit;

    // The proc may not already have parameters or a function result,
    // The proc may also not be a initialization or finalization method.
    // The proc must also be a static proc.
    with TProc(Proc) do
      if (Parameters.Parameters <> '') or (Parameters.FuncResult <> '') or
         (ProcType in [ptInitialization, ptFinalization]) or
         (MethodBindingType <> mbtStatic) then
        Exit;

    // Generate a list of register which is read before they are set.
    UsedSourceCallback := TUsedSourceCallback.Create;
    try
      for I := 0 to Count -2 do
      begin
        TAssignInstr(Items[I]).Source.GetSources(UsedSourceCallback.SourceCallback);
        TAssignInstr(Items[I]).Target.GetSources(UsedSourceCallback.TargetCallback);
      end;
      Vars := UsedSourceCallback.Used;
      Params := UsedSourceCallback.UsedBeforeWrite;
    finally
      UsedSourceCallback.Free;
    end;

    // If this list contains other registers then eax, edx and ecx, we don't do this,
    // because there is now way to get access to these registers except assembler.
    if (Params >= [rEax, rEdx, rEcx]) or (Vars >= [rEax, rEdx, rEcx]) then
      Exit;

    if rEcx in Params then
      Params := Params + [rEax, rEdx]
    else if rEdx in Params then
      Params := Params + [rEax, rEdx];
    Vars := Vars - Params;

    // If eax is totally not used, it might simply be re return value.
    if not ((rEax in Params) and (rEax in Vars)) then
      Params := Params + [rEax];

    for Reg := Low(TRegister) to High(TRegister) do
    begin
      if Reg in Params then
        RegNames[Reg] := 'Param' + GetEnumName(TypeInfo(TRegister), Integer(Reg));
      if Reg in Vars then
        RegNames[Reg] := 'Var' + GetEnumName(TypeInfo(TRegister), Integer(Reg));
    end;

    // There must be return value.
    TProc(Proc).Parameters.FuncResult := 'Integer';

    // If eax is a var name it result.
    if rEax in Vars then
      RegNames[rEax] := 'Result';

    // Set the parameters.
    if rEcx in Params then
      TProc(Proc).Parameters.Parameters := 'ParamrEax, ParamrEdx, ParamrEcx: Integer'
    else if rEdx in Params then
      TProc(Proc).Parameters.Parameters := 'ParamrEax, ParamrEdx: Integer'
    else if rEax in Params then
      TProc(Proc).Parameters.Parameters := 'ParamrEax: Integer';

    // Create the proc header
    with TProc(Proc) do
      if ProcType in ptMethods then
        Strings.Add(Format(ProcDef[ProcType, Parameters.FuncResult <> ''],
            [AClass.AClass.ClassName, Name, Parameters.Parameters, Parameters.FuncResult]))
      else
        Strings.Add(Format(ProcDef[ProcType, Parameters.FuncResult <> ''], [Name, Parameters.Parameters, Parameters.FuncResult]));

    // create the vars.
    if Vars <> [] then
    begin
      Strings.Add('var');
      for Reg := Low(TRegister) to High(TRegister) do
        if Reg in Vars then
          Strings.Add(Format('  Var%s: Integer;', [GetEnumName(TypeInfo(TRegister), Integer(Reg))]));
    end;

    Strings.Add('begin');
    with TPascalCallback.Create do
    try
      PascalRegNames := RegNames;
      PascalProc := TProc(Proc);
      // Decompile the assignment instructions.
      for I := 0 to Count -2 do
        Strings.Add(Format('  %s := %s;', [
           TAssignInstr(Items[I]).Target.AsPascal(Callback),
           TAssignInstr(Items[I]).Source.AsPascal(Callback)]));
    finally
      Free;
    end;
    // If eax is not already result, set result to eax.
    if RegNames[rEax] <> 'Result' then
      Strings.Add(Format('  Result := %s;', [RegNames[rEax]]));
    Strings.Add('end;');
    Result := True;
  except
    // Swallow EdcInstructionErrors and EDisAsmErrors, There is just no code generated.
    on EdcInstructionError do Strings.Clear;
    on EDisAsmError do Strings.Clear;
  end;
end;

end.
