unit dcParams;

interface

type
  { TdcParameters }

  TCallingConv = (ccRegister, ccStdcall, ccSafecall); // No support for pascal and cdecl.
  TCallingConvs = set of TCallingConv;

const
  CallingConvsAll = [ccRegister .. ccSafeCall];

type
  TdcParameters = class(TObject)
  private
    FPossCallingConv: TCallingConvs;
    FParameters: string;
    FFuncResult: string;
    procedure SetParameters(const Value: string);
    procedure SetFuncResult(const Value: string);
    function GetCallingConv: TCallingConv;
    procedure SetPossCallingConv(Value: TCallingConvs);
  public
    constructor Create; virtual;
    property Parameters: string read FParameters write SetParameters;
    property FuncResult: string read FFuncResult write SetFuncResult;
    property CallingConv: TCallingConv read GetCallingConv;
    property PossCallingConv: TCallingConvs read FPossCallingConv write SetPossCallingConv;
  end;

implementation

uses
  PEFileClass;

{ TdcParameters }

constructor TdcParameters.Create;
begin
  inherited Create;
  FPossCallingConv := CallingConvsAll;
end;

procedure TdcParameters.SetParameters(const Value: string);
begin
  // Parameters may only be set once.
  if (FParameters <> '') and (FParameters <> Value) then
    raise EDecompilerError.Create('Can''t reset parameters');
  // Set the private field.
  FParameters := Value;
end;

procedure TdcParameters.SetFuncResult(const Value: string);
begin
  // FuncResult may only be set once.
  if (FFuncResult <> '') and (FFuncResult <> Value) then
    raise EDecompilerError.Create('Can''t reset funcion result');
  // Set the private field.
  FFuncResult := Value;
end;

function TdcParameters.GetCallingConv: TCallingConv;
begin
  // The calling conv is the first from the posscalling conv
  if ccRegister in PossCallingConv then
    Result := ccRegister
  else if ccStdcall in PossCallingConv then
    Result := ccStdcall
  else if ccSafecall in PossCallingConv then
    Result := ccSafecall
  else
    raise EDecompilerError.Create('Error in PossCallingConv');
  // Lock the calling conv to this one.
  FPossCallingConv := [Result];
end;

procedure TdcParameters.SetPossCallingConv(Value: TCallingConvs);
begin
  // Don' do anything if the new value is the old one.
  if Value = FPossCallingConv then
    Exit;
  // There must be at least one calling convetion possible.
  if Value = [] then
    raise EDecompilerError.Create('Poss calling conv may not be empty');
  // There may not be introduced a new calling convention
  if not (Value <= FPossCallingConv) then
    raise EDecompilerError.Create('Can''t introduce new poss calling convention.');
  // Set the private field
  FPossCallingConv := Value;
end;

end.
