unit dcInstrSource;

interface

uses
  Classes, DisAsmX;

type
  { IInstrSource }

  IInstrSource = interface;

  TGetSourcesCallback = procedure (Source: IInstrSource) of object;
  TAsPascalCallback = procedure (Source: IInstrSource; var AsPascal: string) of object;

  IInstrSource = interface(IUnknown)
    ['{9E19A141-FE8D-11D3-9895-AF349D573647}']
    function Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource;
    function Compare(ASource: IInstrSource): Boolean;
    procedure GetSources(Callback: TGetSourcesCallback);
    function AsPascal(Callback: TAsPascalCallback): string;
  end;

  { IMonoSource }

  TMonoSourceType = (mostRef);

  IMonoSource = interface(IInstrSource)
    ['{9E19A140-FE8D-11D3-9895-AF349D573647}']
    function GetSource: IInstrSource;
    function GetSourceType: TMonoSourceType;

    property Source: IInstrSource read GetSource;
    property SourceType: TMonoSourceType read GetSourceType;
  end;

  { IRefSource }

  IRefSource = interface(IMonoSource)
    ['{9E19A142-FE8D-11D3-9895-AF349D573647}']
    function GetSize: Integer;

    property Size: Integer read GetSize;
  end;

  { IDuoSource }

  TDuoSourceType = (dstMultiply, dstDivide, dstAdd, dstSub);

  IDuoSource = interface(IInstrSource)
    ['{9E19A143-FE8D-11D3-9895-AF349D573647}']
    function GetSource1: IInstrSource;
    function GetSource2: IInstrSource;
    function GetSourceType: TDuoSourceType;

    property Source1: IInstrSource read GetSource1;
    property Source2: IInstrSource read GetSource2;
    property SourceType: TDuoSourceType read GetSourceType;
  end;

(*  { IMultiSource }

  TMultiSourceType = (mustFuncCall);

  IMultiSource = interface(IInstrSource)
    ['{9E19A144-FE8D-11D3-9895-AF349D573647}']
    function GetItem(Index: Integer): IInstrSource;
    function GetCount: Integer;
    function GetSourceType: TMultiSourceType;

    property Count: Integer read GetCount;
    property Items[Index: Integer]: IInstrSource read GetItem;
    property SourceType: TMultiSourceType read GetSourceType;
  end;
*)
  { IRegSource }

  IRegSource = interface(IInstrSource)
    ['{9E19A145-FE8D-11D3-9895-AF349D573647}']
    function GetReg: TRegister;
    function GetRegType: TRegisterType;

    property RegType: TRegisterType read GetRegType;
    property Reg: TRegister read GetReg;
  end;

  { IConstSource }

  IConstSource = interface(IInstrSource)
    ['{9E19A146-FE8D-11D3-9895-AF349D573647}']
    function GetValue: Cardinal;

    property Value: Cardinal read GetValue;
  end;

  { IAddressSource }

  IAddressSource = interface(IInstrSource)
    ['{9E19A147-FE8D-11D3-9895-AF349D573647}']
    function GetAddress: TAddress;

    property Address: TAddress read GetAddress;
  end;

// Interface creation functions.

// function CreateMonoSource(ASource: IInstrSource; ASourceType: TMonoSourceType): IMonoSource;
function CreateDuoSource(ASource1, ASource2: IInstrSource; ASourceType: TDuoSourceType): IDuoSource;
// function CreateMultiSource(const AItems: array of IInstrSource; ASourceType: TMultiSourceType): IMultiSource;
function CreateRegSource(AReg: TRegister; ARegType: TRegisterType): IRegSource;
function CreateConstSource(AValue: Cardinal): IConstSource;
function CreateAddressSource(AAddress: TAddress): IAddressSource;
function CreateRefSource(ASource: IInstrSource; ASize: Integer): IRefSource;

// Creates a IIinstrSource from an TOpcInstr

function ConvArgToIInstr(const Arg: TArgument): IInstrSource;

type
  TIsAddressCallback = procedure (Imm: TAddress; var IsAddress: Boolean) of object;

var
  IsImmAddressCallback: TIsAddressCallback;

implementation

uses
  ActiveX, SysUtils, dcProcInstr;

type
  { TInstrSource }

  TInstrSource = class(TInterfacedObject, IInstrSource)
  public
    function Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource; virtual; abstract;
    function Compare(ASource: IInstrSource): Boolean; virtual; abstract;
    procedure GetSources(Callback: TGetSourcesCallback); virtual;
    function AsPascal(Callback: TAsPascalCallback): string; virtual; abstract;
  end;

  { TODO -cAdditions : Add GetSources and AsPascal for the TMultipleSource.}

  { TMonoSource }

  TMonoSource = class(TInstrSource, IMonoSource)
  private
    FSource: IInstrSource;
    FSourceType: TMonoSourceType;
    function GetSource: IInstrSource;
    function GetSourceType: TMonoSourceType;
  public
    constructor Create(ASource: IInstrSource; ASourceType: TMonoSourceType);
//    function Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource; o
    function Compare(ASource: IInstrSource): Boolean; override;
    procedure GetSources(Callback: TGetSourcesCallback); override;

    property Source: IInstrSource read GetSource;
    property SourceType: TMonoSourceType read GetSourceType;
  end;

  { TRefSoucre }

  TRefSource = class(TMonoSource, IRefSource)
  private
    FSize: Integer;
    function GetSize: Integer;
  public
    constructor Create(ASource: IInstrSource; ASize: Integer);
    function Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource; override;
    function Compare(ASource: IInstrSource): Boolean; override;
    function AsPascal(Callback: TAsPascalCallback): string; override;

    property Size: Integer read GetSize;
  end;

  { TDuoSource }

  TDuoSource = class(TInstrSource, IDuoSource)
  private
    FSource1: IInstrSource;
    FSource2: IInstrSource;
    FSourceType: TDuoSourceType;
    function GetSource1: IInstrSource;
    function GetSource2: IInstrSource;
    function GetSourceType: TDuoSourceType;
  public
    constructor Create(ASource1, ASource2: IInstrSource; ASourceType: TDuoSourceType);
    function Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource; override;
    function Compare(ASource: IInstrSource): Boolean; override;
    procedure GetSources(Callback: TGetSourcesCallback); override;
    function AsPascal(Callback: TAsPascalCallback): string; override;

    property Source1: IInstrSource read GetSource1;
    property Source2: IInstrSource read GetSource2;
    property SourceType: TDuoSourceType read GetSourceType;
  end;

(*  { TMultiSource }

  TSourceItems = array[0..MaxListSize] of IInstrSource;
  PSourceItems = ^TSourceItems;

  TMultiSource = class(TInstrSource, IMultiSource)
  private
    FCount: Integer;
    FItems: PSourceItems;
    FSourceType: TMultiSourceType;
    function GetCount: Integer;
    function GetSourceType: TMultiSourceType;
    function GetItem(Index: Integer): IInstrSource;
  public
    constructor Create(const AItems: array of IInstrSource; ASourceType: TMultiSourceType);
    destructor Destroy; override;
    function Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource; override;
    function Compare(ASource: IInstrSource): Boolean; override;

    property Count: Integer read GetCount;
    property Items[Index: Integer]: IInstrSource read GetItem;
    property SourceType: TMultiSourceType read GetSourceType;
  end;
*)
  { TRegSource }

  TRegSource = class(TInstrSource, IRegSource)
  private
    FReg: TRegister;
    FRegType: TRegisterType;
    function GetReg: TRegister;
    function GetRegType: TRegisterType;
  public
    constructor Create(AReg: TRegister; ARegType: TRegisterType);
    function Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource; override;
    function Compare(ASource: IInstrSource): Boolean; override;
    function AsPascal(Callback: TAsPascalCallback): string; override;

    property RegType: TRegisterType read GetRegType;
    property Reg: TRegister read GetReg;
  end;

  { TConstSource }

  TConstSource = class(TInstrSource, IConstSource)
  private
    FValue: Cardinal;
    function GetValue: Cardinal;
  public
    constructor Create(AValue: Cardinal);
    function Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource; override;
    function Compare(ASource: IInstrSource): Boolean; override;
    function AsPascal(Callback: TAsPascalCallback): string; override;

    property Value: Cardinal read GetValue;
  end;

  { TAddressSource }

  TAddressSource = class(TInstrSource, IAddressSource)
  private
    FAddress: TAddress;
    function GetAddress: TAddress;
  public
    constructor Create(AAddress: TAddress);
    function Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource; override;
    function Compare(ASource: IInstrSource): Boolean; override;
    function AsPascal(Callback: TAsPascalCallback): string; override;

    property Address: TAddress read GetAddress;
  end;

{ TInstrSource }

procedure TInstrSource.GetSources(Callback: TGetSourcesCallback);
begin
  Callback(Self);
end;

{ TMonoSource }

constructor TMonoSource.Create(ASource: IInstrSource; ASourceType: TMonoSourceType);
begin
  inherited Create;
  FSource := ASource;
  FSourceType := ASourceType;
end;
(*
function TMonoSource.Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource;
begin
  if Compare(FindSource) then
    Result := ReplaceSource
  else
    Result := TMonoSource.Create(Source.Replace(FindSource, ReplaceSource), SourceType);
end;
*)
function TMonoSource.Compare(ASource: IInstrSource): Boolean;
var
  AMonoSource: IMonoSource;
begin
  Result := Succeeded(ASource.QueryInterface(IMonoSource, AMonoSource)) and
   (AMonoSource.SourceType = SourceType) and Source.Compare(AMonoSource.Source);
end;

function TMonoSource.GetSource: IInstrSource;
begin
  Result := FSource;
end;

function TMonoSource.GetSourceType: TMonoSourceType;
begin
  Result := FSourceType;
end;

procedure TMonoSource.GetSources(Callback: TGetSourcesCallback);
begin
  inherited GetSources(Callback);
  Source.GetSources(Callback);
end;

{ TRefSource }

constructor TRefSource.Create(ASource: IInstrSource; ASize: Integer);
begin
  inherited Create(ASource, mostRef);
  FSize := ASize;
end;

function TRefSource.GetSize: Integer;
begin
  Result := FSize;
end;

function TRefSource.Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource;
begin
  if Compare(FindSource) then
    Result := ReplaceSource
  else
    Result := TRefSource.Create(Source.Replace(FindSource, ReplaceSource), Size);
end;

function TRefSource.Compare(ASource: IInstrSource): Boolean;
var
  ARefSource: IRefSource;
begin
  Result := Succeeded(ASource.QueryInterface(IRefSource, ARefSource)) and
     (ARefSource.Size = Size) and Source.Compare(ARefSource.Source);
end;

function TRefSource.AsPascal(Callback: TAsPascalCallback): string;
begin
  case Size of
    1: Result := Format('ShortInt(Pointer(%s)^)', [Source.AsPascal(Callback)]);
    2: Result := Format('SmallInt(Pointer(%s)^)', [Source.AsPascal(Callback)]);
    4: Result := Format('Integer(Pointer(%s)^)', [Source.AsPascal(Callback)]);
  end;
  Callback(Self, Result);
end;

{ TDuoSource }

constructor TDuoSource.Create(ASource1, ASource2: IInstrSource; ASourceType: TDuoSourceType);
begin
  inherited Create;
  FSource1 := ASource1;
  FSource2 := ASource2;
  FSourceType := ASourceType;
end;

function TDuoSource.Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource;
begin
  if Compare(FindSource) then
    Result := ReplaceSource
  else
    Result := TDuoSource.Create(Source1.Replace(FindSource, ReplaceSource),
       Source2.Replace(FindSource, ReplaceSource), SourceType);
end;

function TDuoSource.Compare(ASource: IInstrSource): Boolean;
var
  ADuoSource: IDuoSource;
begin
  Result := Succeeded(ASource.QueryInterface(IDuoSource, ADuoSource)) and
   (ADuoSource.SourceType = SourceType) and Source1.Compare(ADuoSource.Source1) and
   Source2.Compare(ADuoSource.Source2);
end;

function TDuoSource.GetSource1: IInstrSource;
begin
  Result := FSource1;
end;

function TDuoSource.GetSource2: IInstrSource;
begin
  Result := FSource2;
end;

function TDuoSource.GetSourceType: TDuoSourceType;
begin
  Result := FSourceType;
end;

procedure TDuoSource.GetSources(Callback:  TGetSourcesCallback);
begin
  inherited GetSources(Callback);
  Source1.GetSources(Callback);
  Source2.GetSources(Callback);
end;

function TDuoSource.AsPascal(Callback: TAsPascalCallback): string;
const
  SourceTypeNames: array[TDuoSourceType] of string =
    ('*', 'div', '+', '-');
begin
  Result := Format('%s %s %s', [Source1.AsPascal(Callback), SourceTypeNames[SourceType],
     Source1.AsPascal(Callback)]);
  Callback(Self, Result);
end;
(*
{ TMultiSource }

constructor TMultiSource.Create(const AItems: array of IInstrSource; ASourceType: TMultiSourceType);
var
  I: Integer;
begin
  FCount := Length(AItems);
  GetMem(FItems, FCount * SizeOf(TInstrSource));
  for I := 0 to FCount -1 do
    FItems^[I] := AItems[I + Low(AItems)];
end;

destructor TMultiSource.Destroy;
var
  I: Integer;
begin
  for I := 0 to FCount -1 do
    FItems^[I] := nil;
  FreeMem(FItems, FCount * SizeOf(TInstrSource));
  inherited Destroy;
end;

function TMultiSource.Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource;
var
  I: Integer;
  AItems: array of IInstrSource;
begin
  if Compare(FindSource) then
    Result := ReplaceSource
  else
  begin
    SetLength(AItems, Count);
    for I := 0 to Count -1 do
      AItems[I] := Items[I].Replace(FindSource, ReplaceSource);
    Result := TMultiSource.Create(AItems, SourceType);
  end;
end;

function TMultiSource.Compare(ASource: IInstrSource): Boolean;
var
  AMultiSource: IMultiSource;
  I: Integer;
begin
  Result := Succeeded(ASource.QueryInterface(IMultiSource, AMultiSource)) and
     (AMultiSource.SourceType = SourceType);
  for I := 0 to Count -1 do
  begin
    if not Result then Exit;
    Result := Items[I].Compare(AMultiSource.Items[I]);
  end;
end;

function TMultiSource.GetItem(Index: Integer): IInstrSource;
begin
  Result := FItems^[Index];
end;

function TMultiSource.GetCount: Integer;
begin
  Result := FCount;
end;

function TMultiSource.GetSourceType: TMultiSourceType;
begin
  Result := FSourceType;
end;
*)
{ TRegSource }

constructor TRegSource.Create(AReg: TRegister; ARegType: TRegisterType);
begin
  inherited Create;
  FReg := AReg;
  FRegType := ARegType;
end;

function TRegSource.Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource;
begin
  if Compare(FindSource) then
    Result := ReplaceSource
  else
    Result := Self;
end;

function TRegSource.Compare(ASource: IInstrSource): Boolean;
var
  ARegSource: IRegSource;
begin
  Result := Succeeded(ASource.QueryInterface(IRegSource, ARegSource)) and
     (ARegSource.Reg = Reg) and (ARegSource.RegType = RegType);
end;

function TRegSource.GetReg: TRegister;
begin
  Result := FReg;
end;

function TRegSource.GetRegType: TRegisterType;
begin
  Result := FRegType;
end;

function TRegSource.AsPascal(Callback: TAsPascalCallback): string;
const
  RegNames: array[TRegister, TRegisterType] of string =
    (('ah', 'al', 'ax', 'eax'), ('ch', 'cl', 'cx', 'ecx'), ('dh', 'dl', 'dx', 'edx'),
     ('bh', 'bl', 'bx', 'ebx'), ('', '', 'sp', 'esp'), ('', '', '', 'ebp'),
     ('', '', 'si', 'esi'), ('', '', '', ''));
begin
  Result := RegNames[Reg, RegType];
  Callback(Self, Result);
end;

{ TConstSource }

constructor TConstSource.Create(AValue: Cardinal);
begin
  inherited Create;
  FValue := AValue;
end;

function TConstSource.Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource;
begin
  if Compare(FindSource) then
    Result := ReplaceSource
  else
    Result := Self;
end;

function TConstSource.Compare(ASource: IInstrSource): Boolean;
var
  AConstSource: IConstSource;
begin
  Result := Succeeded(ASource.QueryInterface(IConstSource, AConstSource)) and
     (AConstSource.Value = Value);
end;

function TConstSource.GetValue: Cardinal;
begin
  Result := FValue;
end;

function TConstSource.AsPascal(Callback: TAsPascalCallback): string;
begin
  Result := IntToStr(Value);
  Callback(Self, Result);
end;

{ TAddressSource }

constructor TAddressSource.Create(AAddress: TAddress);
begin
  inherited Create;
  FAddress := AAddress;
end;

function TAddressSource.Replace(FindSource, ReplaceSource: IInstrSource): IInstrSource;
begin
  if Compare(FindSource) then
    Result := ReplaceSource
  else
    Result := Self;
end;

function TAddressSource.Compare(ASource: IInstrSource): Boolean;
var
  AAddressSource: IAddressSource;
begin
  Result := Succeeded(ASource.QueryInterface(IAddressSource, AAddressSource)) and
     (AAddressSource.Address = Address);
end;

function TAddressSource.GetAddress: TAddress;
begin
  Result := FAddress;
end;

function TAddressSource.AsPascal(Callback: TAsPascalCallback): string;
begin
  Result := '$' + IntToHex(Integer(Address), 8);
  Callback(Self, Result);
end;

// Interface creation functions implementations.
(*
function CreateMonoSource(ASource: IInstrSource; ASourceType: TMonoSourceType): IMonoSource;
begin
  Result := TMonoSource.Create(ASource, ASourceType);
end;
*)
function CreateDuoSource(ASource1, ASource2: IInstrSource; ASourceType: TDuoSourceType): IDuoSource;
begin
  Result := TDuoSource.Create(ASource1, ASource2, ASourceType);
end;
(*
function CreateMultiSource(const AItems: array of IInstrSource; ASourceType: TMultiSourceType): IMultiSource;
begin
  Result := TMultiSource.Create(AItems, ASourceType);
end;
*)
function CreateRegSource(AReg: TRegister; ARegType: TRegisterType): IRegSource;
begin
  Result := TRegSource.Create(AReg, ARegType);
end;

function CreateConstSource(AValue: Cardinal): IConstSource;
begin
  Result := TConstSource.Create(AValue);
end;

function CreateAddressSource(AAddress: TAddress): IAddressSource;
begin
  Result := TAddressSource.Create(AAddress);
end;

function CreateRefSource(ASource: IInstrSource; ASize: Integer): IRefSource;
begin
  Result := TRefSource.Create(ASource, ASize);
end;

// Creates a IIinstrSource from an TOpcInstr

function ConvArgToIInstr(const Arg: TArgument): IInstrSource;

  function RefToSource(const Ref: TdaRef): IInstrSource;

    function GetMultiplyRef(Multiply: Integer; Reg: TRegister): IInstrSource;
    var
      Tmp: IInstrSource;
    begin
      Result := nil;
      if Multiply > 0 then
        Result := CreateRegSource(Ref.ARegister1, rtDWord);
      if Multiply > 1 then
      begin
        Tmp := CreateConstSource(Multiply);
        Result := CreateDuoSource(Result, Tmp, dstMultiply);
      end;
    end;

  var
    Tmp: IInstrSource;
    IsAddress: Boolean;
  begin
    // Create the multiply reg instructions.
    Result := GetMultiplyRef(Ref.MultiplyReg1, Ref.ARegister1);
    Tmp := GetMultiplyRef(Ref.MultiplyReg2, Ref.ARegister2);
    // Combine the two Instructions.
    if Result = nil then
      Result := Tmp
    else
      Result := CreateDuoSource(Result, Tmp, dstAdd);
    // Create the Imm instruction.
    if Ref.Immidiate <> nil then
    begin
      IsAddress := False;
      if Assigned(IsImmAddressCallback) then
        IsImmAddressCallback(Ref.Immidiate, IsAddress);
      if IsAddress then
        Tmp := CreateConstSource(Cardinal(Ref.Immidiate))
      else
        Tmp := CreateConstSource(Cardinal(Ref.Immidiate));
      Result := CreateDuoSource(Result, Tmp, dstAdd);
    end;
  end;

resourcestring
  ENotSupportedArgType = 'Not supported instruction source argument type.';
begin
  case Arg.ArgumentType of
    atNone: Result := nil;
    atRegv: Result := CreateRegSource(Arg.Reg, rtDWord);
    atRefv:
      begin
        Result := CreateRefSource(RefToSource(Arg.Ref), 4);
      end;
    else
      raise EdcInstructionError.Create(ENotSupportedArgType);
  end;
end;

end.
