unit Vars;

interface

uses
  Classes, Procs, PEFile, dcDecomps, dcUnits, dcTypeIntf;

type
  TVarConst  = set of (vtVar, vtConst);
  TDecompType = (dtNormal, dtResString);

  { TVar }

  TVar = class(TDecompItem)
  private
    FInitValue: Pointer;
    FVarConst: TVarConst;
    FDecomps: TList;
    FDecompTypes: TList;
    FOffsets: TList;
    FName: string;
    FRefVar: Boolean;
    FVarSize: Integer;
    FContainType: Boolean;
    FType: IdcType;
    FAppendBefore: TAppendType;
    FAppendAfter: TAppendType;
    procedure SetInitValue(Value: Pointer);
    function GetDecompCount: Integer;
    function GetDecompItem(Index: Integer): TDecompItem;
    function GetDecompType(Index: Integer): TDecompType;
    function GetOffset(Index: Integer): Integer;
    function GetDeclaration: string;
    procedure SetVarSize(Value: Integer);
    procedure SetAppendBefore(Value: TAppendType);
    procedure SetAppendAfter(Value: TAppendType);
  protected
    procedure SetSize(Value: Integer); override;
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure AddDecomp(Decomp: TDecompItem; Offset: Integer; DecompType: TDecompType);
    function IsRefAddress(AAddress: PChar): Boolean; override;

    property InitValue: Pointer read FInitValue write SetInitValue;
    property VarConst: TVarConst read FVarConst write FVarConst;
    property DecompCount: Integer read GetDecompCount;
    property DecompItems[Index: Integer]: TDecompItem read GetDecompItem;
    property DecompItemTypes[Index: Integer]: TDecompType read GetDecompType;
    property OffSet[Index: Integer]: Integer read GetOffset;
    property Name: string read FName write FName;
    property ContainType: Boolean read FContainType write FContainType;
    property VarDecl: string read GetDeclaration;
    property VarSize: Integer read FVarSize write SetVarSize;
    property AType: IdcType read FType;
    property AppendBefore: TAppendType read FAppendBefore write SetAppendBefore;
    property AppendAfter: TAppendType read FAppendAfter write SetAppendAfter;
    // If RefVar is true a reference to the value of the var must be replaced by
    // a reference to the address and there shouldn't by a ref direct to the var
    // and the size must be 4, exactly on the decompitem.
    property RefVar: Boolean read FRefVar write FRefVar;
  end;

  { TVarInfos }

  TVarInfos = class(TDecompCollection)
  private
    function GetItem(Index: Integer): TVar;
    procedure SetItem(Index: Integer; Value: TVar);
  public
    procedure LoadVars;
    procedure LoadFixups;
    procedure LoadInitVars;
    procedure DeterUnits;
    procedure GenerateNames;
    function IndexOfName(Name: string): Integer;
    function IndexOfAddress(AAddress: PChar): Integer;
    procedure LoadVar(Address: PChar; Name: string; AUnit: TUnit);
    property Items[Index: Integer]: TVar read GetItem write SetItem; default;
  end;

implementation

uses
  {$IFOPT D+}dcDebug, Dialogs,{$ENDIF}
  PEFileClass, SysUtils, SysVars, NameMangling;

type
  PWord = ^Word;
  PCardinal = ^Cardinal;

{ TVar }

constructor TVar.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FVarConst := [vtVar, vtConst];
  FContainType := True;
  FType := CreateType;
end;

destructor TVar.Destroy;
begin
  FreeMem(FInitValue);
  FDecomps.Free;
  FDecompTypes.Free;
  FOffsets.Free;
  inherited Destroy;
end;

procedure TVar.SetInitValue(Value: Pointer);
begin
  Move(Value^, FInitValue^, VarSize);
end;

function TVar.GetDecompCount: Integer;
begin
  if FDecomps = nil then
    Result := 0
  else
    Result := FDecomps.Count;
end;

function TVar.GetDecompItem(Index: Integer): TDecompItem;
begin
  if FDecomps = nil then
    raise EDecompilerError.Create('Var has no decomps');
  Result := TDecompItem(FDecomps[Index]);
end;

function TVar.GetDecompType(Index: Integer): TDecompType;
begin
  if FDecompTypes = nil then
    raise EDecompilerError.Create('Var has no decomps');
  Result := TDecompType(FDecompTypes[Index]);
end;

function TVar.GetOffset(Index: Integer): Integer;
begin
  if FOffsets = nil then
    raise EDecompilerError.Create('Var has no decomps');
  Result := Integer(FOffsets[Index]);
end;

procedure TVar.SetSize(Value: Integer);
begin
  inherited SetSize(Value);
  VarSize := Value;
end;

procedure TVar.AddDecomp(Decomp: TDecompItem; Offset: Integer; DecompType: TDecompType);
begin
  if FDecomps = nil then
  begin
    FDecomps := TList.Create;
    FOffsets := TList.Create;
    FDecompTypes := TList.Create;
  end;
  FDecomps.Add(Decomp);
  FDecompTypes.Add(Pointer(DecompType));
  AddReq(Decomp, nil);
  FOffsets.Add(Pointer(Offset));
end;

function TVar.GetDeclaration: string;

  function GetIntValue(Address: PChar; Size: Integer): Cardinal;
  begin
    case Size of
      1: Result := PByte(Address)^;
      2: Result := PWord(Address)^;
      4: Result := PCardinal(Address)^;
      else
        raise EDecompilerError.CreateFmt('Unknown integer value %d', [Size]);
    end;
  end;

  function GetTypeName: string;
  const
    TypeName: array[1..4] of string = ('Byte', 'Word', '', 'Cardinal');

    function GetArrayName(ArraySize: Integer = 4): string;
    var
      I: Integer;
    begin
      if Size mod ArraySize = 0 then
      begin
        Result := Format('array[0..%d] of %s', [Size div ArraySize -1, TypeName[ArraySize]]);
        if Address < PEFileClass.BSS then
        begin
          Result := Format('%s = (%u', [Result, GetIntValue(InitValue, ArraySize)]);
          for I := 1 to VarSize div ArraySize -1 do
            Result := Format('%s, %u', [Result, GetIntValue(PChar(InitValue) + I * ArraySize, ArraySize)]);
          Result := Result + ')';
        end;
      end
      else
        Result := GetArrayName(ArraySize div 2);
    end;

  type
    TXType = (xtByte, xtWord, xtCardinal, xtString, xtPointer, xtArray);
    // if XType is xtArray then the XType after it is the the xType of the array and
    // the Value after that is the number of elements.
  var
    Types: TList;

    procedure AddVarTypes(ASize: Integer);
    var
      I: Integer;
   begin
      if ASize mod 4 = 0 then
        for I := 0 to ASize div 4 -1 do
          Types.Add(Pointer(xtCardinal))
      else if ASize mod 2 = 0 then
        for I := 0 to ASize div 2 -1 do
          Types.Add(Pointer(xtWord))
      else
        for I := 0 to ASize -1 do
          Types.Add(Pointer(xtByte))
    end;

    function GetXTypeSize(Index: Integer): Integer;
    const
      XTypeSize: array[TXType] of Integer =
        (1, 2, 4, 4, 4, 33);
    begin
      if Types[Index] = Pointer(xtArray) then
        Result := Integer(Types[Index +1]) * GetXTypeSize(Index + 1)
      else
        Result := XTypeSize[TXType(Types[Index])];
    end;

    function GetXTypeName(Index: Integer): string;
    const
      XTypeName: array[TXType] of string =
        ('Byte', 'Word', 'Cardinal', 'string', 'Pointer', 'Error');
    begin
      if Types[Index] = Pointer(xtArray) then
        Result := Format('array[0..%d] of %s', [Integer(Types[Index + 2])-1,
          GetXTypeName(Index + 1)])
      else
        Result := XTypeName[TXType(Types[Index])];
    end;

    function GetXTypeInitName(Index: Integer; Offset: Integer): string;
    var
      I: Integer;
    begin
      case TXType(Types[Index]) of
        xtByte: Result := IntToStr(PByte(PChar(InitValue) + Offset)^);
        xtWord: Result := IntToStr(PWord(PChar(InitValue) + Offset)^);
        xtCardinal: Result := IntToStr(PWord(PChar(InitValue) + Offset)^);
        xtPointer:
          begin
            for I := 0 to DecompCount -1 do
              if Self.Offset[I] = Offset then
              begin
                if DecompItems[I] is TProc then
                  Result := '@' + TProc(DecompItems[I]).Name
                else if DecompItems[I] is TVar then
                  Result := '@' + TVar(DecompItems[I]).Name
                else if DecompItems[I] is TStringInfo then
                  Result := '@' + TStringInfo(DecompItems[I]).Name
                else if DecompItems[I] is TClassInfo then
                  Result := TClassInfo(DecompItems[I]).AClass.ClassName
                else
                  raise EDecompilerError.CreateFmt('Undefined const, %s', [DecompItems[I].ClassName]);
              end;
          end;
        xtString:
          begin
            for I := 0 to DecompCount -1 do
              if Self.Offset[I] = Offset then
              begin
                Result := EnhQuotedStr((DecompItems[I] as TStringInfo).Value);
                Break;
              end;
          end;
        xtArray:
          begin
            Result := '(';
            for I := 0 to Integer(Types[Index + 2]) -1 do
              Result := Result + GetXTypeInitName(Index + 1, Offset + I * GetXTypeSize(Index +1)) + ', ';
            SetLength(Result, Length(Result) -1);
            Result[Length(Result)] := ')';
          end;
      end;
    end;

  var
    I, J: Integer;
  begin
    if DecompCount = 0 then
    begin
      // If there are known decomp items referenced use a array of a byte, Word or dword.
      if Address < PEFileClass.BSS then
        Result := '%s = %u'
      else
        Result := '%s';
      if VarSize in [1, 2, 4] then
        Result := Format(Result, [TypeName[VarSize], GetIntValue(InitValue, VarSize)])
      else
        Result := GetArrayName;
    end
    else
    begin
      Types := TList.Create;
      try
        // Create a list of the var Types.
        I := 0;
        J := 0;
        while J <> VarSize do
        begin
          if I >= DecompCount then
          begin
            // No decomp item after J
            AddVarTypes(VarSize - J);
            J := Size;
          end
          else if Offset[I] <> J then
          begin
            // No decomp item untile Offset[I]
            AddVarTypes(Offset[I] - J);
            J := OffSet[I];
          end
          else
          begin
            // Decomp item at J.
            if DecompItemTypes[I] = dtResString then
              Types.Add(Pointer(xtString))
            else
            begin
              if (DecompItems[I] is TStringInfo) and
                 (TStringInfo(DecompItems[I]).StringType in [stAnsiString]) then
                Types.Add(Pointer(xtString))
              else
                Types.Add(Pointer(xtPointer));
            end;
            Inc(J, 4);
            Inc(I);
          end;
        end;

        // Replace a row of Types with xtArray
        I := 0;
        while I < Types.Count -2 do
        begin
          if Types[I] = Types[I+1] then
          begin
            J := I+2;
            while (J < Types.Count) and (Types[J] = Types[I])  do
              Inc(J);
            Types[I] := Pointer(xtArray);
            Types.Insert(I + 2, Pointer(J - I));
            I := I + 3;
            while (I < Types.Count) and (Types[I] = Types[I-2]) do
              Types.Delete(I);
          end
          else
            Inc(I);
        end;

        // Create a record using the Types.
        if (Types.Count = 1) or
           ((Types.Count = 3) and (Types[0] = Pointer(xtArray))) then
          Result := GetXTypeName(0) + ' = ' + GetXTypeInitName(0, 0)
        else
        begin
          I := 0;
          Result := 'record';
          while I < Types.Count do
          begin
            Result := Format('%s F%d: %s;', [Result, I, GetXTypeName(I)]);
            if Types[I] = Pointer(xtArray) then
              Inc(I, 3)
            else
              Inc(I);
          end;
          Result := Result + ' end = (';
          I := 0;
          J := 0;
          while I <= Types.Count -1 do
          begin
            Result := Format('%sF%d: %s; ', [Result, I, GetXTypeInitName(I, J)]);
            Inc(J, GetXTypeSize(I));
            if Types[I] = Pointer(xtArray) then
              Inc(I, 3)
            else
              Inc(I);
          end;
          Result[Length(Result) -1] := ')';
        end
      finally
        Types.Free;
      end;
    end;
  end;

begin
  case AType.TypeKind of
    etkUnknown: // Add the var and its type.
               Result := Format('  %s: %s;', [Name, GetTypeName]);
    etkUTInteger: Result := Format('  %s = %d;', [Name, PInteger(InitValue)^]);
    else
      raise EDecompilerError.Create('Unsupported Type');
  end;
end;

procedure TVar.SetVarSize(Value: Integer);
begin
  FVarSize := Value;
  ReAllocMem(FInitValue, Value);
  FillChar(FInitValue^, Value, 0);
end;

function TVar.IsRefAddress(AAddress: PChar): Boolean;
begin
  Result := (AAddress >= Address) and (AAddress < Address + Size);
end;

procedure TVar.SetAppendBefore(Value: TAppendType);
begin
  if Value <> FAppendBefore then
  begin
    FAppendBefore := Value;
  end;
end;

procedure TVar.SetAppendAfter(Value: TAppendType);
begin
  if Value <> FAppendAfter then
  begin
    FAppendAfter := Value;
  end;
end;

{ TVarInfos }

procedure TVarInfos.LoadVars;
var
  I, J: Integer;
  VarList: TList;
  DC: TDecompItem;
  NMInfo: TNameManglingInfo;
  VarInfo: TVar;
resourcestring
  SUnitNotFound = 'Unit name %s not found';
  SVarAlreadyHasAName = 'Var already has the name %s, so it can''t be set to %s.';
  SVarAlreadyHasAUnit = 'Var already has a unit %s, so it can''t be set to %s.';
  SAddressAlreadySet = 'Var address already set to %p, so it can''t be set to %p.';
begin
  with TPEFileClass(PEFileClass) do
  begin
    varList := TList.Create;
    try
      if ProjectType = ptPackage then
        // Load vars from the peexports.
        for I := 0 to PEExports.Count -1 do
        begin
          NMInfo := GetNameManglingInfo(PEExports[I].Name);
          // Exported item must be a var and must be in the data section.
          if (PEExports[I].Address >= Data) and (PEExports[I].Address < Data + DataSize) and
             (NMInfo.NMType = eitVar) and (VarList.IndexOf(PEExports[I].Address) = -1) and
             (IndexOfAddress(PEExports[I].Address) = -1) then
          begin
            VarList.Add(PEExports[I].Address);
          end;
        end;
      // Load the vars, the start at the first
      // Search all items in the BSS section.
      for I := 0 to Fixups.Count -1 do
      begin
        if (Fixups[I].FixupType = 3) and (PPChar(Fixups[I].Address)^ >= Data) and
           (PPChar(Fixups[I].Address)^ < BSS + BSSSize) and
           (varList.IndexOf(PPChar(Fixups[I].Address)^) = -1) and
           (IndexOfAddress(PPChar(Fixups[I].Address)^) = -1) then
        begin
          VarList.Add(PPChar(Fixups[I].Address)^);
        end;
      end;

      VarList.Sort(ListSimpleSort);

      // Create the var.
      for I := 0 to VarList.Count -1 do
      begin
        J := IndexOfAddress(VarList[I]);
        if J = -1 then
          VarInfo := TVar.Create(Self)
        else
          VarInfo := Items[J];

        with VarInfo do
        begin
          // Set the addresses.
          if (Address <> nil) and (Address <> VarList[I]) then
            raise EDecompilerError.CreateFmt(SAddressAlreadySet, [Pointer(Address), Pointer(VarList[I])]);
          Address := VarList[I];
          RefAddress := VarList[I];
          // calculate the Size.
          if Size = 0 then
          begin
            DC := TPEFileClass(PEFileClass).FindDecompItemAfter(VarList[I]);
            if I = VarList.Count -1 then
            begin
              if (DC <> nil) and (DC.Address < BSS + BSSSize) then
                Size := DC.Address - Address
              else
                Size := BSS + BSSSize - Address;
            end
            else
            begin
              if (DC <> nil) and (DC.Address < VarList[I+1]) then
                Size := DC.Address - VarList[I]
              else
                Size := Integer(VarList[I+1]) - Integer(VarList[I]);
            end;
          end;
          
          if VarList[I] < Data + DataSize then
            InitValue := VarList[I]
          else
            VarConst := [vtVar];

          // Set the properties, using the export info.
          if ProjectType = ptPackage then
          begin
            J := PEExports.FindByAddress(VarList[I]);
            if J <> -1 then
            begin
              Comments.AddComment('Exported var: ' + PEExports[J].Name, ctDebug);
              NMInfo := GetNameManglingInfo(PEExports[J].Name);
              // This var may not appendbefore
              AppendBefore := atMayNot;
              // Set the name
              if (Name <> '') and (AnsiCompareText(NMInfo.ItemName, Name) = 0) then
                raise EDecompilerError.CreateFmt(SVarAlreadyHasAName, [Name, NMInfo.ItemName]);
              Name := NMInfo.ItemName;
              // Set the unit.
              J := Units.FindByName(NMInfo.UnitName);
              if J = -1 then
                raise EDecompilerError.CreateFmt(SUnitNotFound, [NMInfo.UnitName]);
              if (AUnit <> nil) and (AUnit <> Units[J]) then
                raise EDecompilerError.CreateFmt(SVarAlreadyHasAUnit, [TUnit(AUnit).Name, Units[J].Name]);
              AUnit := Units[J];
            end;
          end;
        end;
      end;
    finally
      varList.Free;
    end;
  end;
end;

procedure TVarInfos.LoadFixups;
var
  J, I, K: Integer;
  Decomp: TDecompItem;
begin
  with TPEFileClass(PEFileClass) do
  begin
    // Add the fixups to the vars.
    for I := 0 to Fixups.Count -1 do
    begin
      if (Fixups[I].Address >= Data) and
         (Fixups[I].FixupType <> 0) then
      begin
        if Fixups[I].FixupType <> 3 then
          raise EDecompilerError.Create('Unexpected fixup type');
        K := -1;
        for J := 0 to Count -1 do
          if (Fixups[I].Address >= Items[J].Address) and
             (Fixups[I].Address < Items[J].Address + Items[J].Size) then
          begin
            K := J;
            Break;
          end;
        if K <> -1 then
        begin
          Decomp := FindDecompItemByRef(PPChar(Fixups[I].Address)^);
          if Decomp <> nil then
            Items[K].AddDecomp(Decomp, Fixups[I].Address - Items[K].Address, dtNormal);
          with TPEFileClass(PEFileClass).Units[TPEFileClass(PEFileClass).Units.Count - 1] do
            if (PPChar(Fixups[I].Address)^ >= Address) and
               (PPChar(Fixups[I].Address)^ < Address + Size) then
              Items[K].AUnit := TPEFileClass(PEFileClass).Units[TPEFileClass(PEFileClass).Units.Count - 1];
        end;
      end;
    end;
  end;
end;

procedure TVarInfos.LoadInitVars;
var
  I: Integer;
begin
  with TPEFileClass(PEFileClass) do
    for I := 0 to Units.Count -1 do
    begin
      if Units[I].UnitType <> utProgram then
        with TVar.Create(Self) do
        begin
          Comments.AddComment('Init var', ctDebug);
          Size := 4;
          Address := PPChar(Units[I].FInit.Address + $13)^;
          Name := Format('!InitCount%x', [I]);
          AUnit := Units[I];
        end
    end;
end;

function TVarInfos.GetItem(Index: Integer): TVar;
begin
  Result := TVar(inherited GetItem(Index));
end;

procedure TVarInfos.SetItem(Index: Integer; Value: TVar);
begin
  inherited SetItem(Index, Value);
end;

procedure TVarInfos.GenerateNames;
var
  I: Integer;
begin
  for I := 0 to Count -1 do
    if Items[I].Name = '' then
    begin
      if vtVar in Items[I].VarConst then
        Items[I].Name := Format('Var%p', [Pointer(Items[I].Address)])
      else
        Items[I].Name := Format('Const%p', [Pointer(Items[I].Address)]);
    end;
end;

function TVarInfos.IndexOfName(Name: string): Integer;
begin
  for Result := 0 to Count -1 do
    if Items[Result].Name = Name then
      exit;
  Result := -1;
end;

function TVarInfos.IndexOfAddress(AAddress: PChar): Integer;
begin
  for Result := 0 to Count -1 do
    with Items[Result] do
      if (Address <= AAddress) and (Address + Size > AAddress) then
        Exit;
  Result := -1;
end;

procedure TVarInfos.LoadVar(Address: PChar; Name: string; AUnit: TUnit);
var
  V: TVar;
  I: Integer;
begin
  V := TPEFileClass(PEFileClass).FindDecompItemByRef(Address) as TVar;
  if V = nil then
  begin
    V := TVar.Create(Self);
    V.Address := Address;
    V.Comments.AddComment('LoadVar: ' + Name, ctDebug);
  end
  else
    if (V.Name <> '') and (CompareText(V.Name, Name) <> 0) then
      raise EDecompilerError.Create('Var already exists ' + V.Name + ' ' + Name);

  // Create the var and set the name and the address.
  for I := 0 to High(SysVarList) do
    if CompareText(SysVarList[I]^.Name, Name) = 0 then
    begin
      V.Name := Name;
      V.Size := SysVarList[I]^.Size;
      V.AUnit := AUnit;
      V.AppendBefore := atMayNot;
      V.AppendAfter := atMayNot;
      exit;
    end;
  raise EDecompilerError.Create('Hard coded var not found. ' + Name);
end;

procedure TVarInfos.DeterUnits;
var
  I, J: Integer;
  UIndex: Integer;
  ReqUnit: TUnit;
begin
  // Sort the vars on address.
  for I := 0 to Count -1 do
    for J := I +1 to Count -1 do
      if Items[J].Address < Items[I].Address then
        Items[J].Index := I;
  // The unit index can't have a lower index than a previous unit.
  UIndex := 2;
  for I := 0 to Count -1 do
  begin
    // Reset the minimal unit index if we are starting with the BSS section or
    // if this var may not append before.
    if (Items[I].Address = PEFileClass.BSS) or (Items[I].AppendBefore = atMayNot) then
      UIndex := 2;

    if Items[I].AUnit = nil then
    begin
      ReqUnit := nil;
      // If the item is requires in two different units it must be in the interface section.
      for J := 0 to Items[I].ReqByDecompCount -1 do
        if (Items[I].ReqByDecomps[J].AUnit <> nil) then
        begin
          if ReqUnit <> nil then
            // This item is required in two different units
            Items[I].IntfImpl := iiInterface
          else
            ReqUnit := Items[I].ReqByDecomps[J].AUnit as TUnit;
        end;
      if Items[I].IntfImpl = iiInterface then
      begin
        // If it is in the interface section it can't require a decomp after it.
        for J := 0 to Items[I].ReqDecompCount -1 do
          if (Items[I].ReqDecomps[J].AUnit <> nil) and
             (Items[I].ReqDecomps[J].AUnit.Index > UIndex) and
             (Items[I].ReqDecomps[J].PEFileClass = PEFileClass) then
            UIndex := Items[I].ReqDecomps[J].AUnit.Index;
      end
      else
      begin
        for J := 0 to Items[I].ReqDecompCount -1 do
          if (Items[I].ReqDecomps[J].AUnit <> nil) and
             (Items[I].ReqDecomps[J].AUnit.Index > UIndex) and
             (Items[I].ReqDecomps[J].IntfImpl = iiImplementation) and
             (Items[I].ReqDecomps[J].PEFileClass = PEFileClass) then
            UIndex := Items[I].ReqDecomps[J].AUnit.Index;
      end;
      Items[I].AUnit := TPEFileClass(PEFileClass).Units[UIndex];
    end
    else
      if Items[I].AUnit.Index > UIndex then
        UIndex := Items[I].AUnit.Index;

    // If this var may not append after, reset the unit index
    if Items[I].AppendAfter = atMayNot then
      UIndex := 2;
  end;
end;

end.
