unit TypeInfoUtils;

interface

uses
  TypInfo, SysUtils;

type
{ record type data }
  TRecordField = record
    TypeInfo: PPTypeInfo;
    Offset: Integer;
  end;

  PRecordFields = ^TRecordFields;
  TRecordFields = array[0..30000] of TRecordField;

  PRecordTypeData = ^TRecordTypeData;
  TRecordTypeData = packed record
    Size: Integer;
    FieldCount: Integer;
    Fields: TRecordFields;
  end;

  PArrayTypeData = ^TArrayTypeData;
  TArrayTypeData = packed record
    ArraySize: Integer;
    ItemSize: Integer;
    ItemType: ^PTypeInfo;
  end;

  { See system }
  PDynArrayTypeData = ^TDynArrayTypeData;
  TDynArrayTypeData = packed record
    elSize: Longint;
    elType: ^PTypeInfo;
    varType: Integer;
  end;

  PParamRecord = ^TParamRecord;
  TParamRecord = record
    Flags: TParamFlags;
    ParamName: ShortString;
    TypeName: ShortString;
  end;

  PPropData = ^TPropData;

  ETypeInfoError = class(Exception);

function GetPropData(TypeData: PTypeData): PPropData;
procedure NextPropInfo(var PropInfo: PPropInfo);

function GetSizeName(Size: Integer): string;
function GetTypeInfoName(TypeInfo: PTypeInfo): string;
function GetTypeDef(TypeInfo: PTypeInfo): string;
function GetTypeInfoSize(TypeInfo: PTypeInfo): Integer;
function GetVarSize(TypeInfo: PTypeInfo): Integer;

{ Method types }
function GetMethodTypeParameters(TypeInfo: PTypeInfo): string;
function GetMethodTypeResult(TypeInfo: PTypeInfo): string;

implementation

uses
  ComObj;

function GetPropData(TypeData: PTypeData): PPropData;
begin
  Result := Pointer(Integer(TypeData) + SizeOf(TClass) + SizeOf(PPTypeInfo) +
    SizeOf(Smallint) + Length(TypeData^.UnitName) + 1);
end;

procedure NextPropInfo(var PropInfo: PPropInfo);
begin
  PropInfo := Pointer(Integer(PropInfo) + SizeOf(TPropInfo) - 255 + Length(PropInfo^.Name));
end;

type
  PWord = ^Word;
  PInteger = ^Integer;

function GetSizeName(Size: Integer): string;
const
  TypeName: array[1..4] of string = ('shortint', 'smallint', '', 'integer');
var
  ArraySize: Integer;
begin
  if Size <= 0 then
    raise ETypeInfoError.Create('Negative Size');
  if (Size = 1) or (Size = 2) or (Size = 4) then
    Result := TypeName[Size]
  else
  begin
    ArraySize := 4;
    while Size mod ArraySize <> 0 do
      ArraySize := ArraySize div 2;
    Result := Format('array[0..%d] of %s', [Size div ArraySize -1, TypeName[ArraySize]])
  end;
end;
    
function GetTypeInfoName(TypeInfo: PTypeInfo): string;
begin
  Result := TypeInfo^.Name;
  if Result[1] = '.' then
    Result := GetTypeDef(TypeInfo);
end;

function GetTypeDef(TypeInfo: PTypeInfo): string;
var
  TypeData: PTypeData;
  DynArrayTypeData: PDynArrayTypeData absolute TypeData;
  ArrayTypeData: PArrayTypeData absolute TypeData;
  RecordTypeData: PRecordTypeData absolute TypeData;
  ParamRecord: PParamRecord;
  BaseTypeData: PTypeData;
  I, J: Integer;
  X: PShortString;
const
{$IFDEF VER120}
  OrdTypeName: array[TOrdType] of string = ('ShortInt', 'Byte', 'Smallint',
    'Word', 'Integer');
{$ELSE}
  OrdTypeName: array[TOrdType] of string = ('ShortInt', 'Byte', 'Smallint',
    'Word', 'Integer', 'Cardinal');
{$ENDIF}
  FloatTypeName: array [TFloatType] of string = ('Single', 'Double', 'Extended',
     'Comp', 'Currency');
  ProcName: array[TMethodKind] of string = ('procedure', 'function',
     'constructor', 'destructor', 'class procedure', 'class function',
     'what', 'what what');
resourcestring
  SInvalidArraySize = 'Array size (%d) of typeinfo at %p should be a multiple of the item size (%d)';
begin
  typeData := GetTypeData(TypeInfo);
  case TypeInfo^.Kind of
    tkLString: Result := 'string';
    tkWString: Result := 'WideString';
    tkVariant: Result := 'Variant';
    tkInteger, tkInt64: begin
                          if (TypeData.OrdType = otSLong) and
                             (Low(Integer) = TypeData.MinValue) then
                            Result := Format('Low(Integer)..%d', [TypeData.MaxValue])
                          else
                            Result := Format('%d..%d', [TypeData.MinValue, TypeData.MaxValue]);
                        end;
    tkChar, tkWChar: Result := Format('#%d..#%d', [TypeData.MinValue, TypeData.MaxValue]);
    tkEnumeration:
      if TypeData^.BaseType^ = TypeInfo then
      begin
        X := @TypeData^.NameList;
        Result := '(';
        for I := TypeData^.MinValue to TypeData^.MaxValue -1 do
        begin
          Result := Result + X^ + ', ';
          Inc(Integer(X), Length(X^) +1);
        end;
        Result := Result + X^ + ')';
      end
      else
      begin
        if TypeData^.BaseType^^.Kind = tkEnumeration then
        begin
          BaseTypeData := GetTypeData(TypeData^.BaseType^);
          X := @BaseTypeData^.NameList;
          for I := BaseTypeData^.MinValue to BaseTypeData^.MaxValue do
          begin
            // Add the name with the name of the min value.
            if TypeData^.MinValue = I then
              Result := Result + X^ + ' .. ';
            // Add the name with the name of the max value.
            if TypeData^.MaxValue = I then
            begin
              Result := Result + X^;
              Break;
            end;
            Inc(Integer(X), Length(X^) +1);
          end;
        end
        else
          raise ETypeInfoError.Create('Unsupported type');
      end;
    tkSet: Result := 'set of ' + GetTypeInfoName(TypeData^.CompType^);
    tkFloat: Result := FloatTypeName[TypeData^.FloatType];
    tkString: Result := Format('string[%d]', [TypeData^.MaxLength]);
    tkMethod:
      begin
        // set the name of the method type
        Result := ProcName[TypeData.MethodKind];
        Result := Result + '(';
        // Add all the params
        ParamRecord := @TypeData.ParamList;
        for I := 0 to TypeData.ParamCount -1 do
        begin
          X := Pointer(Integer(@ParamRecord^.ParamName) + Length(ParamRecord^.ParamName) +1);
          // classifie the parameter.
          if pfVar in ParamRecord.Flags then
            Result := Result + 'var ';
          if pfConst in ParamRecord.Flags then
            Result := Result + 'const ';
          if pfOut in ParamRecord.Flags then
            Result := Result + 'out ';
          // Add the param name.
          Result := Result + ParamRecord^.ParamName + ': ';
          // add the param type;
          if pfArray in ParamRecord.Flags then
            Result := Result + 'array of ';
          Result := Result + X^;
          // add a semicolon and space if this isn't the last parameter.
          if I < TypeData.ParamCount -1 then
            Result := Result + '; ';
          // go to the next param record.
          ParamRecord := PParamRecord(Integer(ParamRecord) + SizeOf(TParamFlags) +
              (Length(ParamRecord^.Paramname) +1) + (Length(X^) + 1));
        end;
        Result := Result + ')';
        // Add the function Result if Exit.
        if TypeData.MethodKind in [mkFunction, mkClassFunction] then
          Result := Result + ': ' + PShortString(ParamRecord)^;
        Result := Result + ' of object';
      end;
    tkDynArray:
      begin
        if DynArrayTypeData^.elType = nil then
        begin
          if DynArrayTypeData^.elSize mod 4 = 0 then
            Result := 'array of ' + GetSizeName(DynArrayTypeData^.elSize)
          else
            Result := 'packed array of ' + GetSizeName(DynArrayTypeData^.elSize);
        end
        else
          Result := Format('array of %s', [GetTypeInfoName(DynArrayTypeData^.elType^)]);
      end;
    tkRecord:
      begin
        Result := 'packed record ';
        I := 0;
        for J := 0 to RecordTypeData^.FieldCount -1 do
        begin
          if I <> RecordTypeData^.Fields[J].Offset then
            Result := Result + Format('f%d: %s; ', [I, GetSizeName(RecordTypeData^.Fields[J].Offset - I)]);
          I := RecordTypeData^.Fields[J].Offset;
          Result := Result + Format('f%d: %s; ', [I, GetTypeInfoName(RecordTypeData^.Fields[J].TypeInfo^)]);
          I := I + GetVarSize(RecordTypeData^.Fields[J].TypeInfo^);
        end;
        if I <> RecordTypeData.Size then
          Result := Result + Format('f%d: %s; ', [I, GetSizeName(RecordTypeData.Size - I)]);
        Result := Result + 'end';
      end;
    tkInterface:
      Result := 'interface(' + TypeData.IntfParent^^.Name + ') [''' + GUIDToString(TypeData.Guid) + '''] end';
    tkArray:
      begin
        I := GetVarSize(ArrayTypeData.ItemType^);
        if ArrayTypeData.ArraySize mod I <> 0 then
          raise ETypeInfoError.CreateFmt(SInvalidArraySize, [ArrayTypeData.ArraySize, Pointer(TypeInfo), I]);
        Result := Format('array[0..%d] of %s', [ArrayTypeData.ArraySize div I -1, GetTypeInfoName(ArrayTypeData.ItemType^)]);
      end;
    else
      raise ETypeInfoError.Create('type Info defenition not yet supported');
  end;
end;

function GetTypeInfoSize(TypeInfo: PTypeInfo): Integer;
var
  TypeData: PTypeData;
  PropInfo: PPropInfo;
  J: PChar;
  I: Integer;
begin
  Result := SizeOf(TTypeKind) + Length(TypeInfo.Name) + 1;
  TypeData := GetTypeData(TypeInfo);
  case TypeInfo.Kind of
    tkUnknown, tkLString, tkWString, tkVariant: ;
    tkInteger, tkChar, tkWChar: Result := Result + SizeOf(TOrdType) +
       2 * SizeOf(Longint);
    tkEnumeration:
      begin         
        Result := Result + SizeOf(TOrdType) + 2 * SizeOf(Longint) +
           SizeOf(Pointer);
        if TypeData.BaseType^ = TypeInfo then
        begin
          J := @TypeData.NameList;
          // following from the typeinfo from byteBool MaxValue, the numbers of
          // items in the namelist should be the differnce between MinValue and MaxValue
          // As Cardinals.
          for I := 0 to abs(Cardinal(TypeData.MinValue) - Cardinal(TypeData.MaxValue)) do
          begin
            Result := Result + Byte(J[0]) +1;
            J := J + Byte(J[0]) +1;
          end;
        end;
      end;
    tkSet: Result := Result + SizeOf(Pointer);
    tkFloat: Result := Result + SizeOf(TFloatType);
    tkString: Result := Result + SizeOf(Byte);
    tkClass:
      begin
         Result := Result + SizeOf(TClass) + SizeOf(Pointer) +
           SizeOf(Smallint) + Length(TypeData.UnitName) + 1 + SizeOf(Word);
         PropInfo := PPropInfo(Integer(TypeData) + SizeOf(TClass) + SizeOf(Pointer) +
           SizeOf(Smallint) + Length(TypeData.UnitName) + 1 + SizeOf(Word));
         for I := 0 to PWord(Integer(PropInfo) - SizeOf(Word))^ -1 do
         begin
           Result := Result + 4 * SizeOf(Pointer) + SizeOf(Integer) + SizeOf(longint) +
              SizeOf(Smallint) + Length(PropInfo.Name) + 1;
           PropInfo := PPropInfo(Integer(PropInfo) + 4 * SizeOf(Pointer) + SizeOf(Integer) + SizeOf(longint) +
              SizeOf(Smallint) + Length(PropInfo.Name) + 1);
         end;
      end;
    tkMethod:
      begin
        J := PChar(TypeData) + SizeOf(TMethodKind) + SizeOf(Byte);
        for I := 0 to TypeData.ParamCount -1 do
        begin
          J := J + SizeOf(TParamFlags);
          J := J + Byte(J[0]) + 1;
          J := J + Byte(J[0]) + 1;
        end;
        if TypeData.MethodKind in [mkFunction, mkClassFunction, mkSafeFunction] then
          J := J + Byte(J[0]) + 1;
        Result := Result + J - PChar(TypeData);
      end;
    tkInterface:
      begin
        Result := Result + SizeOf(Pointer) + SizeOf(TIntfFlagsBase) +
           SizeOf(TGUID) + Length(TypeData.IntfUnit) + 1 + SizeOf(Word);
        PropInfo := PPropInfo(Integer(TypeData) + SizeOf(Pointer) +
          SizeOf(TIntfFlagsBase) + SizeOf(TGUID) + Length(TypeData.IntfUnit) + 1 + SizeOf(Word));
        for I := 0 to PWord(Integer(PropInfo) - SizeOf(Word))^ -1 do
        begin
          Result := Result + 4 * SizeOf(Pointer) + SizeOf(Integer) + SizeOf(longint) +
             SizeOf(Smallint) + Length(PropInfo.Name) + 1;
          PropInfo := PPropInfo(Integer(PropInfo) + 4 * SizeOf(Pointer) + SizeOf(Integer) + SizeOf(longint) +
             SizeOf(Smallint) + Length(PropInfo.Name) + 1);
        end;
      end;
    tkInt64: Result := Result + 2 * SizeOf(Int64);
    tkDynArray: Result := Result + SizeOf(TDynArrayTypeData);
    tkRecord:
      begin
        Result := Result + 2 * SizeOf(Integer) +
          PInteger(Integer(TypeData) + SizeOf(Integer))^ * SizeOf(TRecordField);
      end;
    tkArray: Result := Result + 2* SizeOf(Integer) + SizeOf(Pointer);
    else
      raise ETypeInfoError.Create('Unkwno Type Info Size');
  end;
end;

function GetVarSize(TypeInfo: PTypeInfo): Integer;
const
{$IFDEF VER120}
  OrdTypeSize: array[TOrdType] of Integer = (1, 1, 2, 2, 4);
{$ELSE}
  OrdTypeSize: array[TOrdType] of Integer = (1, 1, 2, 2, 4, 4);
{$ENDIF}
  FloatTypeSize: array[TFloatType] of Integer = (4, 8, 10, 8, 8);
begin
  case TypeInfo^.Kind of
    tkLString, tkWString, tkString, tkClass, tkInterface, tkDynArray:
        Result := 4;
    tkInteger, tkChar, tkEnumeration, tkSet, tkWChar:
        Result := OrdTypeSize[GetTypeData(TypeInfo).OrdType];
    tkFloat:
        Result := FloatTypeSize[GetTypeData(TypeInfo).FloatType];
    tkMethod, tkInt64: Result := 8;
    tkRecord: Result := PRecordTypeData(GetTypeData(TypeInfo)).Size;
    tkArray: Result := PArrayTypeData(GetTypeData(TypeInfo)).ArraySize;
    tkVariant: Result := SizeOf(TVarData);
    else
      raise ETypeInfoError.Create('Unknown Variant type');
  end;
end;

function GetMethodTypeParameters(TypeInfo: PTypeInfo): string;
var
  TypeData: PTypeData;
  ParamRecord: PParamRecord;
  I: Integer;
  X: PShortString;
begin
  TypeData := GetTypeData(TypeInfo);
  ParamRecord := @TypeData.ParamList;
  for I := 0 to TypeData.ParamCount -1 do
  begin
    X := Pointer(Integer(@ParamRecord^.ParamName) + Length(ParamRecord^.ParamName) +1);
    // classifie the parameter.
    if pfVar in ParamRecord.Flags then
      Result := Result + 'var ';
    if pfConst in ParamRecord.Flags then
      Result := Result + 'const ';
    if pfOut in ParamRecord.Flags then
      Result := Result + 'out ';
    // Add the param name.
    Result := Result + ParamRecord^.ParamName + ': ';
    // add the param type;
    if pfArray in ParamRecord.Flags then
      Result := Result + 'array of ';
    Result := Result + X^;
    // add a semicolon and space if this isn't the last parameter.
    if I < TypeData.ParamCount -1 then
      Result := Result + '; ';
    // go to the next param record.
    ParamRecord := PParamRecord(Integer(ParamRecord) + SizeOf(TParamFlags) +
        (Length(ParamRecord^.Paramname) +1) + (Length(X^) + 1));
  end;
end;

function GetMethodTypeResult(TypeInfo: PTypeInfo): string;
var
  I: Integer;
  J: PChar;
  TypeData: PTypeData;
begin
  TypeData := GetTypeData(TypeInfo);
  if TypeData.MethodKind in [mkFunction, mkClassFunction] then
  begin
    J := PChar(TypeData) + SizeOf(TMethodKind) + SizeOf(Byte);
    for I := 0 to TypeData.ParamCount -1 do
    begin
      J := J + SizeOf(TParamFlags);
      J := J + Byte(J[0]) + 1;
      J := J + Byte(J[0]) + 1;
    end;
    Result := J + 1;
  end;
end;

end.
