unit Procs;

interface

uses
  PEFile, Classes, TypInfo,
  {$IFOPT D+} dcDebug, dialogs, {$ENDIF}
  dcDecomps, dcFields, MethodLists, dcParams, peExports, dcProcInstr;

type
  { TMiscs }

  TMiscs = class(TDecompCollection)
  private
    function GetItem(Index: Integer): TDecompItem;
    procedure SetItem(Index: Integer; Value: TDecompItem);
  public
    property Items[Index: Integer]: TDecompItem read GetItem write SetItem; default;
  end;

  { TGUIDConst }

  TGUIDConst = class(TDecompItem)
  public
    function IsRefAddress(AAddress: PChar): Boolean; override;
  end;
  
  { TTypeInfoInfo }

  TTypeInfoInfo = class(TDecompItem)
  private
    FTypeInfo: PTypeInfo;
    FName: string;
    function GetTypeDef: string;
    function GetTypeInfoVarName: string;
    function GetName: string;
  public
    function IsRefAddress(AAddress: PChar): Boolean; override;
    function HasTypeDef: Boolean;
    procedure LoadMethodRefs;

    property TypeInfo: PTypeInfo read FTypeInfo;
    property TypeDef: string read GetTypeDef;
    property TypeInfoVarName: string read GetTypeInfoVarName;
    property Name: string read GetName;
  end;

  { TTypeInfoInfos }

  TTypeInfoInfos = class(TDecompCollection)
  private
    function GetItem(Index: Integer): TTypeInfoInfo;
    procedure SetItem(Index: Integer; Value: TTypeInfoInfo);
  public
    procedure LoadTypeInfos;
    function FindTypeInfo(TypeInfo: PTypeInfo): TTypeInfoInfo;
    function IndexOfName(Name: string): Integer;
    property Items[Index: Integer]: TTypeInfoInfo read GetItem write SetItem; default;
  end;

  { TInterface }

  TInterface = class(TCollectionItem)
  private
    FGuid: TGUID;
    FMethodCount: Integer;
  public
    property GUID: TGUID read FGUID;
    property MethodCount: Integer read FMethodCount;
  end;

  { TInterfaces }

  TInterfaces = class(TCollection)
  private
    function GetItem(Index: Integer): TInterface;
    procedure SetItem(Index: Integer; Value: TInterface);
  public
    function Add(GUID: TGUID; MethodCount: Integer): TInterface;
    function FindInterface(GUID: TGUID): TInterface;
    property Items[Index: Integer]: TInterface read GetItem write SetItem; default;
  end;

const
  AppendBeforeIndex = 0;
  AppendAfterIndex = 1;

type
  TClassInfo = class;
  TClassInfos = class;
  TProcs = class;

  { TProc }

  TMethodBindingType = (mbtVirtual, mbtDynamic, mbtStatic);
  TProcType = (ptProcedure, ptClassProcedure, ptMethodProcedure, ptConstructor,
     ptDestructor, ptInitialization, ptFinalization, ptEntryPointProc);
  TProcTypes = set of TProcType;

  TProc = class;

  TImportInfo = record
    Imported: Boolean;
    DLLName: string;
    Entry: TPEImport;
    ImportedProc: TProc;
  end;

  TAppendType = (atMay, atMayNot, atMust);

  TProc = class(TDecompItem)
  private
    FOnSizeChange: TmlneMethodList;
    FOnInitSizeChange: TmlneMethodList;
    FOnPossProcTypesChange: TmlneMethodList;

    FInitSize: Integer;
    FBeforeInitSize: Integer;
    FFinaSize: Integer;
    FAfterFinaSize: Integer;
    FName: string;
    FImportInfo: TImportInfo;
    FPossProcTypes: TProcTypes;
    FMethodBindingType: TMethodBindingType;
    FPublished: Boolean;
    FMethodIndex: Integer;
    FClass: TClassInfo;
    FInstr: TdcInstructions;
    FOverrides: Boolean;
    FForwardDecl: Boolean;
    FParameters: TdcParameters;
    FProcEnh: TObject;
    FProcSize: Integer;
    FAppend: array[AppendBeforeIndex..AppendAfterIndex] of TAppendType;
    procedure SetPossProcTypes(Value: TProcTypes);
    function GetProcType: TProcType;
    procedure SetMethodBindingType(Value: TMethodBindingType);
    procedure SetClass(Value: TClassInfo);
    function GetDefSrc: string;
    function GetAncestorMethod: TProc;
    function GetIncName: string;
    procedure SetInitSize(Value: Integer);
    procedure ProcSizeChange(Sender: TmlneMethodList);
    procedure SetName(Value: string);
    function GetName: string;
    function GetPossProcTypes: TProcTypes;
    procedure SetOverrides(Value: Boolean);
    function GetAppend(Index: Integer): TAppendType;
    procedure SetAppend(Index: Integer; Value: TAppendType);
    function GetInstr: TdcInstructions;
  protected
    procedure PossSetToIntf(DecompItem: TDecompItem); override;
    procedure SetSize(Value: Integer); override;
    procedure CheckAppend;
  public
    constructor Create(Procs: TProcs; Address: PChar); reintroduce; overload;
    destructor Destroy; override;
    function IsRefAddress(AAddress: PChar): Boolean; override;
    procedure Append(Proc: TProc);
    procedure AddReqBy(Decomp: TDecompItem; AAddress: PChar); override;

    // InitSize is the size of the initilization code automaticly generated.
    property InitSize: Integer read FInitSize write SetInitSize;
    // Beofer init size is the size of the code before the auto generated code
    property BeforeInitSize: Integer read FBeforeInitSize write FBeforeInitSize;
    // FinaSize if the size of the finalization code automaticly generated.
    property FinaSize: Integer read FFinaSize write FFinaSize;
    // After init size is the size of the code after the auto generated code, to the end
    // of the proc indicated by procsize (not Size).
    property AfterFinaSize: Integer read FAfterFinaSize write FAfterFinaSize;

    property Name: string read GetName write SetName;
    // Inc name is the name including the object name (and Unit).
    property IncName: string read GetIncName;
    property ImportInfo: TImportInfo read FImportInfo write FImportInfo;
    property ProcType: TProcType read GetProcType;
    property PossProcTypes: TProcTypes read GetPossProcTypes write SetPossProcTypes;   // All posible proc types are set.
    property MethodBindingType: TMethodBindingType read FMethodBindingType write SetMethodBindingType;
    property APublished: Boolean read FPublished write FPublished;
    property MethodIndex: Integer read FMethodIndex write FMethodIndex; // VMT or Dynamic index
    property AClass: TClassInfo read FClass write SetClass;
    property Overrides: Boolean read FOverrides write SetOverrides;
    property Instr: TdcInstructions read GetInstr;
    property DefSrc: string read GetDefSrc;
    property ForwardDecl: Boolean read FForwardDecl write FForwardDecl;
    property AncestorMethod: TProc read GetAncestorMethod;
    property ProcSize: Integer read FProcSize write FProcSize;
    property Parameters: TdcParameters read FParameters;

    property AppendBefore: TAppendType index AppendBeforeIndex read GetAppend write SetAppend;
    property AppendAfter: TAppendType index AppendAfterIndex read GetAppend write SetAppend;

    property OnSizeChange: TmlneMethodList read FOnSizeChange;
    property OnInitSizeChange: TmlneMethodList read FOnInitSizeChange;
    property OnPossProcTypesChange: TmlneMethodList read FOnPossProcTypesChange;
  end;

  { TProcEnh }

  TProcEnh = class(TObject)
  private
    FProc: TProc;
  public
    constructor CreateEnh(Proc: TProc); virtual;
    property Proc: TProc read FProc;
  end;

  { TDestructorProcEnh }

  TDestructorProcEnh = class(TProcEnh)
  private
    FHasBeforeDestruction: Boolean;
    FHasClassDestroy: Boolean;
  public
    constructor CreateEnh(Proc: TProc); override;
    destructor Destroy; override;
    procedure ProcSizeChange(Sender: TmlneMethodList);
    property HasBeforeDestruction: Boolean read FHasBeforeDestruction;
    property HasClassDestroy: Boolean read FHasClassDestroy;
  end;

  { TInitProcEnh }

  TInitProcEnh = class(TProcEnh)
  private
    FHasInitResStringImport: Boolean;
    FHasInitImport: Boolean;
  public
    constructor CreateEnh(Proc: TProc); override;
    destructor Destroy; override;
    procedure ProcInitSizeChange(Sender: TmlneMethodList);
    property HasInitResStringImport: Boolean read FHasInitResStringImport;
    property HasInitImport: Boolean read FHasInitImport;
  end;

  { TInitProc }

  TInitProc = class(TProc)
  public
    constructor CreateInit(Procs: TProcs; Address: PChar);
    constructor CreateFInit(Procs: TProcs; Address: PChar);
    destructor Destroy; override;
  end;

  { TProcs }

  TProcs = class(TDecompCollection)
  private
    FOnLoadPublishedMethods: TmlneMethodList;
    FPublishedMethodsLoaded: Boolean;
    function GetItem(Index: Integer): TProc;
    procedure SetItem(Index: Integer; Value: TProc);
  public
    constructor CreateDecomp(PEFileClass: TPEFile); override;
    destructor Destroy; override;
    procedure LoadPublishedMethods;
    procedure LoadExportedProcs;
    function AnalyzeProc(Proc: TProc): Boolean;
    function Add(Address: PChar): TProc;
    function FindProc(Address: PChar): TProc;
    function FindProcByName(const Name: string): Integer;
    function FindProcIndex(Address: PChar; var Index: Integer): Boolean;

    property Items[Index: Integer]: TProc read GetItem write SetItem; default;
    property OnLoadPublishedMethods: TmlneMethodList read FOnLoadPublishedMethods;
    property PublishedMethodsLoaded: Boolean read FPublishedMethodsLoaded;
  end;

  { TClassInfo }

  TClassInfo = class(TDecompItem)
  private
    FClass: TClass;
    FMethods: TList;
    FInterfaces: TList;
    FClassDef: TStrings;
    FFields: TdcFieldList;
    FAncestorClass: TClassInfo;
    FForwardDecl: Boolean;
    function GetMethod(Index: Integer): TProc;
    function GetMethodCount: Integer;
    function GetInterfaceCount: Integer;
    function GetInterface(Index: Integer): TInterface;
  public
    constructor Create(ClassInfos: TClassInfos; AClass: TClass); reintroduce; overload;
    destructor Destroy; override;
    procedure GenerateClassDef;
    procedure AnaClass;
    function IsRefAddress(AAddress: PChar): Boolean; override;
    function GetVirtualMethod(Index: Integer): TProc;
    function GetDynamicMethod(Index: Integer): TProc;
    function FindProc(const ProcName: string): TProc;

    property AClass: TClass read FClass write FClass;
    property MethodCount: Integer read GetMethodCount;
    property Methods[Index: Integer]: TProc read GetMethod;
    property Fields: TdcFieldList read FFields;
    property InterfaceCount: Integer read GetInterfaceCount;
    property Interfaces[Index: Integer]: TInterface read GetInterface;
    property ClassDef: TStrings read FClassDef write FClassDef;
    property AncestorClass: TClassInfo read FAncestorClass;
    property ForwardDecl: Boolean read FForwardDecl write FForwardDecl;
  end;

  { TClassInfos }

  TClassInfos = class(TDecompCollection)
  private
    FOnLoadClasses: TmlneMethodList;
    function GetItem(Index: Integer): TClassInfo;
    procedure SetItem(Index: Integer; Value: TClassInfo);
  public
    constructor CreateDecomp(PEFileClass: TPEFile); override;
    destructor Destroy; override;
    procedure LoadClassInfos;
    procedure GenerateClassDefs;
    function Add(AClass: TClass): TClassInfo;
    function FindClass(AClass: TClass): TClassInfo;
    function FindClassByName(const Name: string): TClassInfo;

    property Items[Index: Integer]: TClassInfo read GetItem write SetItem; default;
    property OnLoadClasses: TmlneMethodList read FOnLoadClasses;
  end;

  { TStringInfo }

  TStringInfos = class;

  TStringType = (stAnsiString, stWideString, stResourceString, stPAnsiChar,
       stPWideChar);

  TStringInfo = class(TDecompItem)
  private
    FValue: string;
    FStringAddress: PChar;
    FStringType: TStringType;
    FName: string;
    function GetStringTypeName: string;
  public
    constructor Create(StringInfos: TStringInfos; Address: PChar;
       StringType: TStringType; ASize: Integer = 0); reintroduce; overload;
    function IsRefAddress(AAddress: PChar): Boolean; override;

    property Value: string read FValue;
    property StringAddress: PChar read FStringAddress;
    property StringType: TStringType read FStringType;
    property StringTypeName: string read GetStringTypeName;
    property Name: string read FName write FName;
  end;

  { TStringInfos }

  TStringInfos = class(TDecompCollection)
  private
    function GetItem(Index: Integer): TStringInfo;
    procedure SetItem(Index: Integer; Value: TStringInfo);
  public
    procedure LoadStringInfos;
    function FindString(Address: PChar): TStringInfo;
    property Items[Index: Integer]: TStringInfo read GetItem write SetItem; default;
  end;

const
  ptMethods: TProcTypes = [ptClassProcedure .. ptDestructor];
  ptAll: TProcTypes = [Low(TProcType) .. High(TProcType)];
  ptAllStatic: TProcTypes = [ptProcedure .. ptDestructor];

// Returns true if the decomp item is a decomp item.
function IsTypeDecomp(Decomp: TDecompItem): Boolean;

// Functions for aligning Address and size to 4 bytes.
function Align4(Address: PChar): PChar; overload;
function Align4(Value: Integer): Integer; overload;

implementation

uses
  {$IFDEF VER120} ObjFileConsts4, {$ELSE} {$IFDEF VER130} ObjFileConsts5, {$ENDIF} {$ENDIF}
  SysUtils, ProcDecomp, DisAsm, VMTUtils, PEFileClass, Windows,
  Vars, dcUnits, TypeInfoUtils, dcNTInfoTypes, dcTypeIntf,
  NameMangling;

type
  PDWord = ^DWord;

{ TMiscs }

function TMiscs.GetItem(Index: Integer): TDecompItem;
begin
  Result := TDecompItem(inherited GetItem(Index));
end;

procedure TMiscs.SetItem(Index: Integer; Value: TDecompItem);
begin
  inherited SetItem(Index, Value);
end;

{ TGUIDConst }

function TGUIDConst.IsRefAddress(AAddress: PChar): Boolean;
begin
  Result := (AAddress = Address);
end;

{ TTypeInfoInfo }

function TTypeInfoInfo.IsRefAddress(AAddress: PChar): Boolean;
begin
  Result := (AAddress = Address) or (AAddress = Address + 4);
end;

function TTypeInfoInfo.GetTypeDef: string;
resourcestring
  STypeInfoWithoutDef = 'Error getting def from a TypeInfo without def at %p.';
begin
  if TypeInfo.Name[1] = '.' then
  begin
    if FName = '' then
      raise EDecompilerError.CreateFmt(STypeInfoWithoutDef, [Pointer(TypeInfo)]);
    Result := FName + ' = ' + TypeInfoUtils.GetTypeDef(TypeInfo) + ';';
  end
  else
    Result := TypeInfo^.Name + ' = ' + TypeInfoUtils.GetTypeDef(TypeInfo) + ';';
end;

function TTypeInfoInfo.GetTypeInfoVarName: string;
begin
  if TypeInfo.Name[1] = '.' then
  begin
    // Generate type info name.
    if FName = '' then
    begin
      // Make the name of "TypeInfo" + the old name - "."
      FName := 'TypeInfo';
      SetLength(FName, Length('TypeInfo')+ Ord(TypeInfo.Name[0]) -1);
      Move(TypeInfo.Name[2], FName[Length('TypeInfo') + 1], Ord(TypeInfo.Name[0]) -1);
    end;
    Result := FName + 'TypeInfo';
  end
  else
    Result := GetTypeInfoName(TypeInfo) + 'TypeInfo';
end;

function TTypeInfoInfo.GetName: string;
resourcestring
  SErrorTypeInfoWithoutAName = 'Can''t get name of TypeInfo at %p, because it doesn''t have one.';
begin
  if TypeInfo.Name[1] = '.' then
  begin
    if FName = '' then
      raise EDecompilerError.CreateFmt(SErrorTypeInfoWithoutAName, [Pointer(TypeInfo)]);
    Result := FName;
  end
  else
    Result := TypeInfo.Name;
end;

function TTypeInfoInfo.HasTypeDef: Boolean;
begin
  Result := ((not (TypeInfo^.Kind in [tkClass])) and (TypeInfo^.Name[1] <> '.')) or
            (FName <> '');
end;

procedure TTypeInfoInfo.LoadMethodRefs;
var
  I, J: Integer;
  ParamRecord: PParamRecord;
  TypeData: PTypeData;
  TypeName: PShortString;
  DC: TDecompItem;
  XUnit1, XUnit2: TUnit;
begin
  if TypeInfo.Kind = tkMethod then
  begin
    // Add all the type infos and classes requrired by the method type.
    TypeData := GetTypeData(TypeInfo);
    ParamRecord := @TypeData.ParamList;
    for I := 0 to TypeData.ParamCount -1 do
    begin
     TypeName := Pointer(Integer(@ParamRecord^.ParamName) + Length(ParamRecord^.ParamName) +1);
     DC := TPEFileClass(PEFileClass).Classes.FindClassByName(TypeName^);
     if DC = nil then
     begin
       J := TPEFileClass(PEFileClass).TypeInfos.IndexOfName(TypeName^);
       if J <> -1 then
         DC := TPEFileClass(PEFileClass).TypeInfos[J]
       else
       begin
         J := TPEFileClass(PEFileClass).NoTInfoTypes.IndexOfName(TypeName^);
         if J <> -1 then
         begin
           // Make the unit Index as Low as possible.
           DC := TPEFileClass(PEFileClass).NoTInfoTypes[J];
           XUnit1 := TPEFileClass(PEFileClass).Units.FindInUnitUsingFInit(DC.Address);
           XUnit2 := TPEFileClass(PEFileClass).Units.FindInUnitUsingFInit(Address);
           if XUnit2.Index < XUnit1.Index then
             DC.Address := Address
           else
             if (XUnit1.Index = XUnit1.Index) and (DC.Address > Address) then
               DC.Address := Address;
         end
         else
           if TypeName^ <> 'Pointer' then
           begin
             // If there isn't a type with th anme declare a new one.
             DC := TNoTInfoType.Create(TPEFileClass(PEFileClass).NoTInfoTypes);
             DC.Address := Address;
             TNoTInfoType(DC).Name := TypeName^;
             TNoTInfoType(DC).Defenition := TypeName^ + ' = Pointer;';
           end;
       end;
     end;
     if DC <> nil then
       Self.AddReq(DC, nil);
     ParamRecord := PParamRecord(Integer(ParamRecord) + SizeOf(TParamFlags) +
        (Length(ParamRecord^.Paramname) +1) + (Length(TypeName^) + 1));
    end;
    if TypeData.MethodKind = mkFunction then
    begin
      TypeName := PShortString(ParamRecord);
      DC := TPEFileClass(PEFileClass).Classes.FindClassByName(TypeName^);
      if DC = nil then
      begin
        J := TPEFileClass(PEFileClass).TypeInfos.IndexOfName(TypeName^);
        if J <> -1 then
          DC := TPEFileClass(PEFileClass).TypeInfos[J]
        else
        begin
          J := TPEFileClass(PEFileClass).NoTInfoTypes.IndexOfName(TypeName^);
          if J <> -1 then
          begin
            // If there is a type without type info with the name, make the address as low as possible.
            DC := TPEFileClass(PEFileClass).NoTInfoTypes[J];
            XUnit1 := TPEFileClass(PEFileClass).Units.FindInUnitUsingFInit(DC.Address);
            XUnit2 := TPEFileClass(PEFileClass).Units.FindInUnitUsingFInit(Address);
            if XUnit2.Index < XUnit1.Index then
              DC.Address := Address
            else
              if (XUnit1.Index = XUnit1.Index) and (DC.Address > Address) then
                DC.Address := Address;
          end
          else
          begin
            // If there isn't a type with th anme declare a new one.
            DC := TNoTInfoType.Create(TPEFileClass(PEFileClass).NoTInfoTypes);
            DC.Address := Address;
            TNoTInfoType(DC).Name := TypeName^;
            TNoTInfoType(DC).Defenition := TypeName^ + ' = Pointer;';
          end;
        end;
      end;
      if DC <> nil then
        Self.AddReq(DC, nil);
    end;
  end;
end;

{ TTypeInfoInfos }

procedure TTypeInfoInfos.LoadTypeInfos;
var
  I, J: Integer;
label
  NextFixup;
begin
  with TPEFileClass(PEFileClass) do
    for I := 0 to Fixups.Count -1 do
    begin
      if (Fixups[I].FixupType = 3) and
         (Fixups[I].Address >= Code) and
         (Fixups[I].Address < Code + CodeSize) and
         (PPChar(Fixups[I].Address)^ = Fixups[I].Address + 4) and
         (Fixups[I].Address[4] in [#0..#17]) and
         (Integer(Fixups[I].Address) mod 4 = 0) and
         (IsIdentifier(Fixups[I].Address + 5)) then
        begin
          // Check that the fixups isn't inside an class declaration.
          for J := 0 to Classes.Count -1 do
            if (Fixups[I].Address >= Classes[J].Address) and
               (Fixups[I].Address < Classes[J].Address + Classes[J].Size) then
              goto NextFixup;
          // Check that there is a fixup at the location.
          for J := 0 to Fixups.Count -1 do
            if Fixups[I].Address + 4 = Fixups[J].Address then
              goto NextFixup;

          // TypeInfo found.
          with TTypeInfoInfo.Create(Self) do
          begin
            FTypeInfo := PTypeInfo(Fixups[I].Address +4);
            Address := Fixups[I].Address;
            RefAddress := Fixups[I].Address;
            // include Pointer in size and align to 4 byte.
            Size := Align4(GetTypeInfoSize(FTypeInfo)) +4;
          end;
        end;
      NextFixup:
    end;
end;

function TTypeInfoInfos.FindTypeInfo(TypeInfo: PTypeInfo): TTypeInfoInfo;
var
  I: Integer;
begin
  for I := 0 to Count - 1 do
  begin
    Result := TTypeInfoInfo(inherited GetItem(I));
    if Result.TypeInfo = TypeInfo then Exit;
  end;
  Result := nil;
end;

function TTypeInfoInfos.GetItem(Index: Integer): TTypeInfoInfo;
begin
  Result := TTypeInfoInfo(inherited GetItem(Index));
end;

procedure TTypeInfoInfos.SetItem(Index: Integer; Value: TTypeInfoInfo);
begin
  inherited SetItem(Index, Value);
end;

function TTypeInfoInfos.IndexOfName(Name: string): Integer;
begin
  for Result := 0 to Count -1 do
    if TTypeInfoInfo(Items[Result]).TypeInfo.Name = Name then
      Exit;
  Result := -1;
end;

{ TInterface }

{ TInterfaces }

function TInterfaces.Add(GUID: TGUID; MethodCount: Integer): TInterface;
begin
  Result := TInterface.Create(Self);
  Result.FGUID := GUID;
  Result.FMethodCount := MethodCount;
end;

function TInterfaces.FindInterface(GUID: TGUID): TInterface;
var
  I: Integer;
begin
  for I := 0 to Count - 1 do
  begin
    Result := TInterface(inherited GetItem(I));
    if CompareMem(@Result.GUID, @GUID, SizeOf(TGUID)) then Exit;
  end;
  Result := nil;
end;

function TInterfaces.GetItem(Index: Integer): TInterface;
begin
  Result := TInterface(inherited GetItem(Index));
end;

procedure TInterfaces.SetItem(Index: Integer; Value: TInterface);
begin
  inherited SetItem(Index, Value);
end;

{ TProc }

constructor TProc.Create(Procs: TProcs; Address: PChar);
var
  I: Integer;
begin
  Procs.FindProcIndex(Address, I);
  inherited Create(Procs);
  // Create the event handlers.
  FOnSizeChange := TmlneMethodList.Create;
  FOnInitSizeChange := TmlneMethodList.Create;
  FOnPossProcTypesChange := TmlneMethodList.Create;

  Self.Address := Address;
  RefAddress := Address;
  FPossProcTypes := ptAll;
  FMethodBindingType := mbtStatic;
  FParameters := TdcParameters.Create;
  Index := I;
  OnSizeChange.Add(ProcSizeChange);
  Comments.AddComment(Format('create a proc at %p', [Pointer(Address)]), ctDebug);
end;

destructor TProc.Destroy;
begin
  if not Overrides then
    FParameters.Free;
  FInstr.Free;
  FProcEnh.Free;
  FOnInitSizeChange.Free;
  FOnSizeChange.Free;
  FOnPossProcTypesChange.Free;
  inherited Destroy;
end;

function TProc.IsRefAddress(AAddress: PChar): Boolean;
begin
  Result := (AAddress = Address);
end;

procedure TProc.Append(Proc: TProc);
var
  I, J: Integer;
  AAddress: PChar;
resourcestring
  SMayNotAppendError = 'trying appending a proc which may not append, %p, %p';
begin
  // Check that they may append.
  if (AppendAfter = atMayNot) or (Proc.AppendBefore = atMayNot) then
  begin
    FAppend[AppendAfterIndex] := atMayNot;
    Proc.FAppend[AppendBeforeIndex] := atMayNot;
    Comments.AddComment(Format(SMayNotAppendError, [Pointer(Proc.Address), Pointer(Address)]), ctError);
    Exit;
  end;
  // Add comments.
  Comments.AddComment(Format('Proc at %p append', [Pointer(Proc.Address)]), ctInfo);
  Comments.AddComment('Comments of the append proc', ctInfo);
  Comments.AddStrings(Proc.Comments);
  Comments.AddComment('Append Comments ended', ctInfo);

  FAppend[1] := Proc.AppendAfter;
  // Add the req decomps to this one.
  for I := 0 to Proc.ReqDecompCount -1 do
    if Proc.ReqDecomps[I] <> Self then
      AddReq(Proc.ReqDecomps[I], PChar(Proc.ReqDecompsAddress[I]));
  // Add the Self to the items which requires the item to Append.
  for I := 0 to Proc.ReqByDecompCount -1 do
  begin
    if Proc.ReqByDecomps[I] <> Self then
    begin
      // Find the item which requires this item.
      AAddress := nil;
      for J := 0 to Proc.ReqByDecomps[I].ReqDecompCount -1 do
        if Proc.ReqByDecomps[I].ReqDecomps[J] = Proc then
        begin
          AAddress := PChar(Proc.ReqByDecomps[I].ReqDecompsAddress[J]);
          Break;
        end;
      Proc.ReqByDecomps[I].AddReq(Self, AAddress);
      // If item which requires the Append item is a Proc, it must append.
      if Proc.ReqByDecomps[I] is TProc then
      begin
        if Proc.ReqByDecomps[I].Address < Address then
        begin
          if TProc(Proc.ReqByDecomps[I]).AppendAfter = atMay then
            TProc(Proc.ReqByDecomps[I]).AppendAfter := atMust
        end
        else
          if TProc(Proc.ReqByDecomps[I]).AppendBefore = atMay then
            TProc(Proc.ReqByDecomps[I]).AppendBefore := atMust;
      end;
    end;
  end;
  Size := Proc.Address + Proc.Size - Address;
  ProcSize := Size;
  Proc.Free;
  // Recheck the req address,
  for I := 0 to ReqDecompCount -1 do
    if (ReqDecomps[I] is TProc) and (PChar(ReqDecompsAddress[I]) <> ReqDecomps[I].Address) and
       (ReqDecompsAddress[I] <> nil) then
    begin
      if ReqDecomps[I].Address > Address then
      begin
        if AppendAfter = atMay then
          AppendAfter := atMust;
      end
      else
        if AppendBefore = atMay then
          AppendBefore := atMust;
    end;
end;

procedure TProc.AddReqBy(Decomp: TDecompItem; AAddress: PChar);
begin
  inherited AddReqBy(Decomp, AAddress);
  // if this is method the class is also req.
  if AClass <> nil then
    Decomp.AddReq(AClass, nil);
end;

procedure TProc.SetPossProcTypes(Value: TProcTypes);
begin
  // Set the ancestor proc types.
  if Overrides then
  begin
    AncestorMethod.PossProcTypes := Value;
    Exit;
  end;
  if Value = FPossProcTypes then
    Exit;
  if Value = [] then
    raise EDecompilerError.Create('Empty PossProcType');
  if not (Value <= FPossProcTypes) then
    raise EDecompilerError.Create('Project type introduces a new PossProcType.');
  // Set the private field.
  FPossProcTypes := Value;
  if FProcEnh <> nil then
    raise EDecompilerError.Create('There is already an proc enh.');
  if FPossProcTypes = [ptDestructor] then
    FProcEnh := TDestructorProcEnh.CreateEnh(Self);
  if FPossProcTypes = [ptInitialization] then
    FProcEnh := TInitProcEnh.CreateEnh(Self);
  // Call event handler.
  OnPossProcTypesChange.CallFirst;
end;

function TProc.GetProcType: TProcType;
begin
  if Overrides then
    Result := AncestorMethod.ProcType
  else
  begin
    if ptEntryPointProc in PossProcTypes then
      Result := ptEntryPointProc
    else
      if ptInitialization in PossProcTypes then
        Result := ptInitialization
      else
        if ptFinalization in PossProcTypes then
          Result := ptFinalization
        else
          if ptProcedure in PossProcTypes then
            Result := ptProcedure
          else
            if ptClassProcedure in PossProcTypes then
              Result := ptClassProcedure
            else
              if ptMethodProcedure in PossProcTypes then
                Result := ptMethodProcedure
              else
                if ptDestructor in PossProcTypes then
                  Result := ptDestructor
                else
                  if ptConstructor in PossProcTypes then
                    Result := ptConstructor
                  else
                    raise EDecompilerError.CreateFmt('Unknown proc type %d', [byte(PossProcTypes)]);
  end;
  // Set the possible proctypes only to the proc type
  PossProcTypes := [Result];
end;

procedure TProc.SetMethodBindingType(Value: TMethodBindingType);
var
  MsgVar: TVar;
begin
  Comments.AddComment(Format('Method binding type set %d', [Integer(Value)]), ctDebug);
  if not (PossProcTypes <= ptMethods) then
    raise EDecompilerError.Create('Setting binding type of a procedure.');
  FMethodBindingType := Value;
  // non-static proc may not be append before.
  if FMethodBindingType <> mbtStatic then
    AppendBefore := atMayNot;
  // Message handler are dynamic methods with index < $C000.
  if (FMethodBindingType = mbtDynamic) and (MethodIndex < $C000) then
  begin
    MsgVar := TVar.Create(TPEFileClass(PEFileClass).Miscs);
    with MsgVar do
    begin
      Address := AClass.Address - 1;
      Name := 'MI_' + AClass.AClass.ClassName + '_' + IntToStr(MethodIndex);
      VarConst := [vtConst];
      VarSize := 4;
      AUnit := AClass.AUnit;
      PInteger(InitValue )^:= MethodIndex;
      AType.PossTypeKinds := [etkUTInteger];
    end;
    AClass.AddReq(MsgVar, nil);
    PossProcTypes := [ptMethodProcedure];
    if Parameters.Parameters = '' then
      Parameters.Parameters := 'var Message: Integer';
  end;
end;

procedure TProc.SetClass(Value: TClassInfo);
var
  AClass1: TClass;
  AClass2: TClassInfo;
  I: Integer;
label
  Found;
begin
  if Value = AClass then
    exit;
  if (Value = nil) then
    raise EDecompilerError.Create('Trying to set class to nil');
  if (AClass <> nil) then
  begin
    if mbtStatic <> MethodBindingType then
      raise EDecompilerError.Create('Can''t change class again.');
    // Remove the proc from the old class.
    FClass.FMethods.Remove(Self);
    // This method has already a class, search the class with the classes have both in common.
    AClass1 := AClass.AClass;
    repeat
      AClass2 := Value;
      repeat
        // Exit the search when they are equal.
        if AClass1 = AClass2.AClass then
        begin
          Value := AClass2;
          goto Found;
        end;
        AClass2 := AClass2.AncestorClass;
      until AClass2 = nil;
      AClass1 := AClass1.ClassParent;
    until AClass1 = nil;
    Assert(False, 'Impossible to come here');
  end;
Found:
  // Make sure the method is in the same unit as the class. (except when this a imported proc).
  if (mbtStatic = MethodBindingType) and (not ImportInfo.Imported) then
    with TPEFileClass(PEFileClass).Units do
      while FindInUnitUsingFInit(PChar(Value.AClass)) <> FindInUnitUsingFInit(Self.Address) do
        Value := Value.AncestorClass;
  // Add the method to the class.
  FClass := Value;
  FClass.FMethods.Add(Self);
  // All items which require this proc also require the class.
  for I := 0 to ReqByDecompCount -1 do
    ReqByDecomps[I].AddReq(FClass, nil);
end;

function TProc.GetDefSrc: string;
const
  ProcTypeDef: array[Low(TProcType)..High(TProcType), Boolean] of string =
    (('procedure %s(%s);', 'function %s(%s): %s;'), ('class procedure %s(%s);', 'class function %s(%s): %s;'),
     ('procedure %s(%s);', 'function %s(%s): %s;'), ('constructor %s(%s);', 'constructor %s(%s);'),
     ('destructor %s(%s);', 'destructor %s(%s);'), ('', ''), ('', ''), ('', ''));
begin
  // start the proc types and the name.
  Result := Format(ProcTypeDef[ProcType, Parameters.FuncResult <> ''],
         [Name, Parameters.Parameters, Parameters.FuncResult]);
  if overrides then
    Result := Result + ' override;'
  else
    case MethodBindingType of
      mbtVirtual: Result := Result + ' virtual;' + '{' + IntToStr(MethodIndex) +'}';
      mbtDynamic: begin
                    if MethodIndex < $C000 then
                      // Message directive
                      Result := Result + ' message MI_'+ AClass.AClass.ClassName + '_' + IntToStr(MethodIndex) +';'
                    else
                      // dynamic
                      Result := Result + ' dynamic;' + '{' + IntToStr(MethodIndex) +'}';
                  end;
    end;
  if Address = nil then
    Result := Result + ' abstract;';
  if ImportInfo.Imported then
    Result := Result + ' external ' + EnhQuotedStr(ImportInfo.DLLName) + ' name ' +
      EnhQuotedStr(ImportInfo.Entry.Name);
end;

procedure TProc.PossSetToIntf(DecompItem: TDecompItem);
begin
  // Don't add it the decomp to the interface section.
end;

procedure TProc.SetSize(Value: Integer);
begin
  inherited SetSize(Value);
  Comments.AddComment(Format('Proc size changed to %d', [Value]), ctDebug);
  CheckAppend;
  OnSizeChange.CallFirst;
end;

function TProc.GetAncestorMethod: TProc;
begin
  case MethodBindingType of
    mbtVirtual: Result := AClass.AncestorClass.GetVirtualMethod(MethodIndex);
    mbtDynamic: Result := AClass.AncestorClass.GetDynamicMethod(MethodIndex);
    else
      raise EDecompilerError.Create('not a virtual or static method');
  end;
end;

function TProc.GetIncName: string;
begin
  Result := Name;
  if ProcType in ptMethods then
    Result := AClass.AClass.ClassName + '.' + Result;
  if (Length(Result) > 0) and (Result[1] = '@') then
    Result := TUnit(AUnit).Name + '.' + Result;
end;

procedure TProc.SetInitSize(Value: Integer);
begin
  FInitSize := Value;
  OnInitSizeChange.CallFirst;
end;

procedure TProc.ProcSizeChange(Sender: TmlneMethodList);
var
  AAddress: PChar;
  I: Integer;
  Proc: TProc;
begin
  // Don't check if this is a system unit).
  if ((AUnit = nil) or (TUnit(AUnit).UnitType <> utSystem)) and
     (not ImportInfo.Imported) then
  begin
    with TPEFileClass(TProcs(Collection).PEFileClass) do
    begin
      // Check to see if the proc has a fixup to the middle of a proc, if that is
      // the case it must append.
      I := Fixups.FindFixupAfter(Address);
      if I <> -1 then
      begin
        repeat
          AAddress := Fixups[I].Address;
          // Search for proc which this proc points to.
          if AAddress >= Address + Size  then Break;
          Assert(AAddress >= Address, 'Error in sorting routine');
          Proc := Procs.FindProc(PPChar(AAddress)^);
          if (Proc <> nil) and (Proc.Address <> PPChar(AAddress)^) and
             ((PPChar(AAddress)^ < Address) or (PPChar(AAddress)^ >= Address + Size)) then
          begin
            AddReq(Proc, AAddress);
            // If this proc points to the middle of a proc it must append.
            if Proc.Address < Address then
            begin
              // Ignore if this is proc may not append before.
              if (AppendBefore <> atMayNot) and (Proc.AppendAfter <> atMayNot) then
              begin
                Comments.AddComment(Format('Append before set to must because of a Fixups at %p', [Pointer(AAddress)]), ctDebug);
                Proc.Comments.AddComment(Format('Append after set to must because of a Fixups from %p', [Pointer(AAddress)]), ctDebug);
                AppendBefore := atMust;
                Proc.AppendAfter := atMust;
              end;
            end
            else
            begin
              Comments.AddComment(Format('Append after set to must because of a Fixups at %p', [Pointer(AAddress)]), ctDebug);
              Proc.Comments.AddComment(Format('Append before set to must because of a Fixups from %p', [Pointer(AAddress)]), ctDebug);
              AppendAfter := atMust;
              Proc.AppendBefore := atMust;
            end;
          end;
          Inc(I);
        until I >= TProcs(Collection).PEFileClass.Fixups.Count;
      end;
      // If there is a fixup (to) directly after the proc and the proc doesn't end on a dword boundary
      // it must be larger.
      if (Integer(Address + Size) mod 4 <> 0) and
         ((Fixups.FindFixup(Address + Size) <> -1) or
          (Fixups.FindFixupTo(Address + Size) <> -1)) then
        AppendAfter := atMust;
    end;
  end;
  // Call the next event handler.
  if Sender <> nil then
    Sender.CallNext(ProcSizeChange);
end;

procedure TProc.SetName(Value: string);
resourcestring
  SErrorProcHasAName = 'Can''t set proc name to %s, because it is already %s';
begin
  if AnsiCompareText(Value, FName) <> 0 then
  begin
    if FName <> '' then
      Comments.AddComment(Format(SErrorProcHasAName, [value, fname]), ctWarning);
    FName := Value;
    if Overrides then
      AncestorMethod.Name := Value;
  end;
end;

function TProc.GetName: string;
begin
  if Overrides then
    Result := AncestorMethod.Name
  else
    Result := FName;
end;

function TProc.GetPossProcTypes: TProcTypes;
begin
  if Overrides then
    Result := AncestorMethod.PossProcTypes
  else
    Result := FPossProcTypes;
end;

procedure TProc.SetOverrides(Value: Boolean);
begin
  if Value <> FOverrides then
  begin
    if not Value then
      raise EDecompilerError.Create('Can not set overrides to false');
    FParameters.Free;
    FOverrides := Value;
    FParameters := AncestorMethod.Parameters;
  end;
end;

function TProc.GetAppend(Index: Integer): TAppendType;
begin
  Result := FAppend[Index];
end;

procedure TProc.SetAppend(Index: Integer; Value: TAppendType);
resourcestring
  SAppendAlreadySet = 'Append already set. %p';
begin
  if Value = FAppend[Index] then Exit;
  if FAppend[Index] <> atMay then
  begin
    Comments.AddComment(Format(SAppendAlreadySet, [Pointer(Address)]), ctError);
    Exit;
  end;
  // Set the private field.
  FAppend[Index] := Value;
  CheckAppend;
end;

function TProc.GetInstr: TdcInstructions;
begin
  Result := FInstr;
  if Result = nil then
  begin
    Result := TdcInstructions.Create(Self);
    FInstr := Result;
  end;
end;

procedure TProc.CheckAppend;
var
  Proc: TProc;
resourcestring
  SConvlictingAppend = 'Must append convlicting with a not must append at %p';
begin
  // If this proc must append before/after and there is a proc before/after this one,
  // which can't append there is something wrong.
  if Size <> 0 then
  begin
    Proc := TProcs(Collection).FindProc(Address - 1);
    if Proc <> nil then
    begin
      if (Proc.AppendAfter <> atMay) and (AppendBefore <> atmay) and
         (Proc.AppendAfter <> AppendBefore) then
        Comments.AddComment(Format(SConvlictingAppend, [Pointer(Address)]), ctError);
      if Proc.AppendAfter <> atMay then
        AppendBefore := Proc.AppendAfter
      else
        Proc.AppendAfter := AppendBefore;
    end;
    Proc := TProcs(Collection).FindProc(Address + Size + 1);
    if Proc <> nil then
    begin
      if (Proc.AppendBefore <> atMay) and (AppendAfter <> atMay) and
         (Proc.AppendBefore <> AppendAfter) then
      begin
        if Proc.AppendBefore = atMust then
          Proc.FAppend[AppendBeforeIndex] := atMayNot;
        if AppendAfter = atMust then
          FAppend[AppendAfterIndex] := atMayNot;
        Comments.AddComment(Format(SConvlictingAppend, [Pointer(Address)]), ctError);
      end;
      if Proc.AppendBefore <> atMay then
        AppendAfter := Proc.AppendBefore
      else
        Proc.AppendBefore := AppendAfter;
    end;
  end;
end;

{ TProcEnh }

constructor TProcEnh.CreateEnh(Proc: TProc);
begin
  inherited Create;
  FProc := Proc;
end;

{ TDestructorProcEnh }

constructor TDestructorProcEnh.CreateEnh(Proc: TProc);
begin
  inherited CreateEnh(Proc);
  Proc.OnSizeChange.Add(ProcSizeChange);
  ProcSizeChange(nil);
end;

destructor TDestructorProcEnh.Destroy;
begin
  FProc.OnSizeChange.Remove(ProcSizeChange);
  inherited Destroy;
end;

procedure TDestructorProcEnh.ProcSizeChange(Sender: TmlneMethodList);
var
  I: Integer;
begin
  // Add the call to Beforedestruction and classdestroy to the auto generated code.
  if Proc.ImportInfo.Imported then Exit;
  with FProc do
  begin
    // BeforeDestruction.
    if not FHasBeforeDestruction then
    begin
      if InitSize <> 0 then
        raise EDecompilerError.Create('There alredy is some init code??');
      I := FindFirstSimpleCallTo(TPEFileClass(PEFileClass).Units.SystemUnit.FindProcByName('@BeforeDestruction').Address,
        Address, Size) - Address;
      if I <> - Integer(Address) then
      begin
        BeforeInitSize := I;
        InitSize := 5;
        FHasBeforeDestruction := True;
      end;
    end;
    // ClassDestroy
    if not FHasClassDestroy then
    begin
      if FinaSize <> 0 then
        raise EDecompilerError.Create('There alredy is some fina code??');
      I := FindLastSimpleCallTo(TPEFileClass(PEFileClass).Units.SystemUnit.FindProcByName('@ClassDestroy').Address,
        Address, Size) - Address;
      if I <> -Integer(Address) then
      begin
        AfterFinaSize := Size - I - 5;
        FinaSize := 5;
        FHasClassDestroy := True;
      end;
    end;
  end;
  // Call the next event handler
  if Sender <> nil then
    Sender.CallNext(ProcSizeChange);
end;

{ TInitProcEnh }

constructor TInitProcEnh.CreateEnh(Proc: TProc);
begin
  inherited CreateEnh(Proc);
  Proc.OnInitSizeChange.Add(ProcInitSizeChange);
  Proc.OnSizeChange.Add(ProcInitSizeChange);
  ProcInitSizeChange(nil);
end;

destructor TInitProcEnh.Destroy;
begin
  Proc.OnSizeChange.Remove(ProcInitSizeChange);
  Proc.OnInitSizeChange.Remove(ProcInitSizeChange);
  inherited Destroy;
end;

procedure TInitProcEnh.ProcInitSizeChange(Sender: TmlneMethodList);
var
  AfterInitAddress: PChar;
  Table: PChar;
  J, K: Integer;
  PEFileClass: TPEFileClass;
begin
  if Proc.ImportInfo.Imported then Exit;
  PEFileClass := TPEFileClass(Proc.PEFileClass);
  AfterInitAddress := Proc.Address + Proc.BeforeInitSize + Proc.InitSize;
  // Check for init res string imort
  if (not FHasInitResStringImport) and
     (PEFileClass.StringInfos.Count <> 0) and
     (PEFileClass.VarInfos.Count <> 0) and
     (AfterInitAddress + 10 <= Proc.Address + Proc.ProcSize) and
     (AfterInitAddress[0] = #$B8) and
     (AfterInitAddress[5] = #$E8) and
     (TPEFileClass(PEFileClass).FindSystemProc(InitResStringImportsProcName) <> nil) and
     (PDWord(AfterInitAddress + 6)^ + AfterInitAddress + 10 = TPEFileClass(PEFileClass).FindSystemProc(InitResStringImportsProcName).Address) then
  begin
    Table := PPChar(AfterInitAddress + 1)^;
    // Create the vars.
    for J := 0 to PDWord(Table)^ -1 do
    begin
      // The assigend var.
      K := PEFileClass.VarInfos.IndexOfAddress(PPChar(Table + J * 8 + 4)^);
      if K = -1 then
        raise EDecompilerError.Create('Res string import var isn''t a var');
      PEFileClass.VarInfos[K].AUnit := Proc.AUnit;
      PEFileClass.VarInfos[K].AddDecomp(PEFileClass.StringInfos.FindString(PPChar(PPChar(Table + J * 8 +8)^)^),
        PPChar(Table + J * 8 + 4)^ - PEFileClass.VarInfos[K].Address, dtResString);
    end;
    // Create a Decomps the fill the block
    with TDecompItem.Create(PEFileClass.miscs) do
    begin
      Address := Table;
      RefAddress := Address;
      Size := PDWord(Table)^ * 8 + 4;
      Comments.AddComment('InitResStringImports Table filler', ctDebug);
    end;
    // Add the call the InitImports to the auto generated Code.
    {$IFOPT D+}
    SendDebug(Format('Found an init res string imports %p', [Pointer(Table)]));
    {$ENDIF}
    FHasInitResStringImport := True;
    Proc.InitSize := Proc.InitSize + 10;
    // Exit the proc because we are already called by changed the initsize.
    Exit;
  end;

  // Check for init import
  if (PEFileClass.FindSystemProc(InitImportsProcName) <> nil) and
     (not HasInitImport) and
     (PEFileClass.StringInfos.Count <> 0) and
     (PEFileClass.VarInfos.Count <> 0) and
     (AfterInitAddress + 10 <= Proc.Address + Proc.ProcSize) and
     (AfterInitAddress[0] = #$B8) and
     (AfterInitAddress[5] = #$E8) and
     (PDWord(AfterInitAddress + 6)^ + AfterInitAddress + 10 =
        PEFileClass.FindSystemProc(InitImportsProcName).Address) then
  begin
    Table := PPChar(AfterInitAddress + 1)^;
    // Apply the fixup
    asm
      mov  eax, Table
      call system.@InitImports
    end;
    // Create the vars.
    for J := 0 to PDWord(Table)^ -1 do
    begin
      K := PEFileClass.VarInfos.IndexOfAddress(PPChar(Table + J * 12 + 4)^);
      if K = -1 then
        raise EDecompilerError.Create('Address not in a var');
      PEFileClass.VarInfos[K].AUnit := Proc.AUnit;
      if PPChar(Table + J * 12 + 8)^ >= PEFileClass.Data then
      begin
        K := PEFileClass.VarInfos.IndexOfAddress(PPChar(Table + J*12 + 8)^);
        if K = -1 then
          raise EDecompilerError.Create('Address not in a var *');
        with PEFileClass.VarInfos[K] do
        begin
          Size := 4;
          Name := Format('!VarCorrector%p', [Pointer(Address)]);
          AUnit := Proc.AUnit;
          RefVar := True;
        end;
      end;
    end;
    // Create a Decomps the fill the block
    with TDecompItem.Create(PEFileClass.miscs) do
    begin
      Address := Table;
      RefAddress := Address;
      Size := PDWord(Table)^ * 12 + 4;
      Comments.AddComment('InitStringImports Table filler', ctDebug);
    end;
    // Add the call the InitImports to the auto generated Code.
    {$IFOPT D+}
    SendDebug(Format('Found an init imports %p', [Pointer(Table)]));
    {$ENDIF}
    FHasInitImport := True;
    Proc.InitSize := Proc.InitSize + 10;
    Exit;
  end;
  // Call the next event handler.
  if Sender <> nil then
    Sender.CallNext(ProcInitSizeChange);
end;

{ TInitProc }

constructor TInitProc.CreateInit(Procs: TProcs; Address: PChar);
var
  ASize: Integer;
begin
  inherited Create(Procs, Address);
  PossProcTypes := [ptInitialization];
  AppendBefore := atMayNot;
  if (Address[0] = #$FF) and (Address[1] = #$25) then
  begin
    // This is an imported init proc.
    ProcSize := 0;
    Size := 8;
    Exit;
  end;
  ProcSize := GetProcSize(Address);
  Size := Align4(ProcSize);
  FinaSize := 1;
  // If Size > 8 a jnb must also be skipped.
  if Size > 8 then
  begin
    // The init code starts at sub dword Ptr [], $01; jnz ...
    while PWord(Address + FBeforeInitSize)^ <> $2D83 do
    begin
      Inc(FBeforeInitSize);
      if BeforeInitSize > Size then
        raise EDecompilerError.Create('Wrong Init section');
    end;
    with TDisAsm.Create do
    try
      GetInstruction(Address + BeforeInitSize + 7, ASize);
    finally
      Free;
    end;
    InitSize := 7 + ASize;
  end
  else
    InitSize := 7;
end;

constructor TInitProc.CreateFInit(Procs: TProcs; Address: PChar);
var
  ASize: Integer;
begin
  inherited Create(Procs, Address);
  PossProcTypes := [ptFinalization];
  AppendBefore := atMayNot;
  if (Address[0] = #$FF) and (Address[1] = #$25) then
  begin
    // This is an imported FInit proc.
    ProcSize := 0;
    Size := 8;
    Exit;
  end;
  ProcSize := GetProcSize(Address);
  Size := Align4(ProcSize);
  if Size > $30 then
  begin
    with TDisAsm.Create do
    try
      GetInstruction(Address + $17, ASize);
    finally
      Free;
    end;
    InitSize := $17 + ASize;
  end
  else
    InitSize := $17;
  FinaSize := $17;
end;

destructor TInitProc.Destroy;
resourcestring
  SDestroyInitProc = 'Can''t destroy init proc';
begin
  if not TPEFileClass(PEFileClass).Destroying then
    raise EDecompilerError.Create(SDestroyInitProc);
  inherited Destroy;
end;

{ TProcs }

constructor TProcs.CreateDecomp(PEFileClass: TPEFile);
begin
  inherited CreateDecomp(PEFileClass);
  FOnLoadPublishedMethods := TmlneMethodList.Create;
end;

destructor TProcs.Destroy; 
begin
  FOnLoadPublishedMethods.Free;
  inherited Destroy;
end;

procedure TProcs.LoadPublishedMethods;
var
  MethodTable: PMethodTable;
  MethodEntry: PMethodEntry;
  I, J: Integer;
  Proc: TProc;
begin
  with TPEFileClass(PEFileClass) do
    // Loop all the classes.
    for I := 0 to Classes.Count -1 do
      begin
        // Get the method table.
        MethodTable := GetMethodTable(Classes[I].AClass);
        if MethodTable <> nil then
          for J := 0 to MethodTable^.Count -1 do
          begin
            MethodEntry := GetMethodEntry(MethodTable, J);
            Proc := Procs.FindProc(MethodEntry^.Address);
            if Proc = nil then
              Proc := Procs.Add(MethodEntry^.Address);
            with Proc do
            begin
              Comments.AddComment('Published method', ctDebug);
              PossProcTypes := PossProcTypes * ptMethods;
              Name := MethodEntry^.Name;
              APublished := True;
              AClass := Classes[I];
            end;
          end;
      end;
  // Call the event handler
  FPublishedMethodsLoaded := True;
  FOnLoadPublishedMethods.CallFirst;
end;

procedure TProcs.LoadExportedProcs;
var
  I, J: Integer;
  Proc: TProc;
  AClassInfo: TClassInfo;
  InUnit: TUnit;
  NMInfo: TNameManglingInfo;
resourcestring
  SClassInExportNotFoundError = 'Class named %s mentioned in export not found';
begin
  with TPEFileClass(PEFileClass) do
    case ProjectType of
      ptDLL:
        // If this is a library assume all exported items are procs.
        for I := 0 to PEExports.Count -1 do
        begin
          Proc := FindProc(PEExports[I].Address);
          if Proc = nil then
            Proc := Add(PEExports[I].Address);
          with Proc do
          begin
            Comments.AddComment('Exported proc ' + PEExports[I].Name, ctInfo);
            PossProcTypes := [ptProcedure];
            IntfImpl := iiInterface;
            AppendBefore := atMayNot;
          end;
        end;
      ptPackage:
        begin
          for I := 0 to PEExports.Count -1 do
          begin
            NMInfo := GetNameManglingInfo(PEExports[I].Name);
            if NMInfo.NMType = eitClass then
            begin
              AClassInfo := TPEFileClass(PEFileClass).Classes.FindClass(TClass(PEExports[I].Address));
              if AClassInfo <> nil then
              begin
                AClassInfo.AUnit := Units.FindInUnitUsingFInit(PChar(AClassInfo.AClass));
                with TUnit(AClassInfo.AUnit) do
                begin
                  Name := NMInfo.UnitName;
                  Comments.AddComment('Unit name set because of export item ' + PEExports[I].Name, ctDebug);
                end;
              end;  
            end
            else if NMInfo.NMType = eitMethod then
            begin
              // Find the class.
              with TPEFileClass(PEFileClass).Units do
                AClassInfo := Items[FindByName(NMInfo.UnitName)].FindClassByName(NMInfo.ClassName);
              if AClassInfo = nil then
                raise EDecompilerError.CreateFmt(SClassInExportNotFoundError, [NMInfo.ClassName]);
            end
            else
              AClassInfo := nil;

            if (NMInfo.NMType in [eitProc, eitMethod]) and
               (NMInfo.ItemName <> 'initialization') then
            begin
              InUnit := Units.FindInUnitUsingFInit(PEExports[I].Address);
              InUnit.Name := NMInfo.UnitName;
              InUnit.Comments.AddComment('Unit name set because of export item ' + PEExports[I].Name, ctDebug);
              if (NMInfo.ItemName <> 'Finalization') then
              begin
                // Create the proc.
                Proc := FindProc(PEExports[I].Address);
                if Proc = nil then
                  Proc := Add(PEExports[I].Address);
                with Proc do
                begin
                  AUnit := InUnit;
                  Comments.AddComment('Exported proc ' + PEExports[I].Name, ctInfo);
                  if NMInfo.NMType = eitProc then
                  begin
                    PossProcTypes := [ptProcedure];
                    // move the proc the the interface part unless it is in the package unit
                    if TUnit(AUnit).UnitType <> utProgram then
                      IntfImpl := iiInterface;
                  end
                  else
                  begin
                    AClass := AClassInfo;
                    // there is no name it this is a constructor.
                    if NMInfo.ItemName = '' then
                      PossProcTypes := PossProcTypes * [ptConstructor, ptDestructor]
                    else
                      PossProcTypes := PossProcTypes * ptMethods;
                  end;
                  AppendBefore := atMayNot;
                  if Address <= Units.SystemUnit.FInit.Address then
                  begin
                    // Make it a proc of size 0 if it is in the system unit.
                    AppendAfter := atMayNot;
                    if NMInfo.ItemName <> '' then
                      Name := NMInfo.ItemName;
                  end
                  else
                    // there is no name it this is a constructor.
                    if (NMInfo.ItemName <> '') and
                       (AnsiCompareText(Proc.Name, NMInfo.ItemName) <> 0) then
                    begin
                      // If a proc with this name already exists add the address to the name.
                      J := FindProcByName(NMInfo.ItemName);
                      if (J = -1) or (Proc.Index = J) then
                        Name := NMInfo.ItemName
                      else
                        Name := Format('%s%p', [NMInfo.ItemName, Pointer(PEExports[I].Address)]);
                    end;
                  // Add the name including the methods as a comment.
                  Comments.AddComment(NMInfo.ItemProp, ctInfo);
                  // There are some strange procs in coride40.bpl which are in full capital
                  // Just thread them as proc markers (I don't known what else to do with them).
                  if NMInfo.ItemProp[1] = 'Q' then
                    AppendAfter := atMayNot;
                end;
              end;
            end;
          end;
        end;
    end;
end;

function TProcs.AnalyzeProc(Proc: TProc): Boolean;
var
  EndAddress: PChar;
  DisAsm: TDisAsm;
  Size: Integer;
  TempAddress: PChar;
  I: Integer;
  AAddress: PChar;
  AProc: TProc;
  Jumps: TList;
begin
  Result := False;
  DisAsm := TDisAsm.Create;
  try
    Jumps := TList.Create;
    try
      AAddress := Proc.Address;
      DisAsm.OnJumpInstr := SaveJumpAddress;
      DisAsm.OnCallInstr := SaveJumpAddress;
      DisAsm.Param := @TempAddress;

      EndAddress := AAddress + Proc.Size;
      while AAddress < EndAddress do
      begin
        // if it is a pointer to something add it to the list and skip, otherwise
        // decompile the instruction.
        if TPEFileClass(PEFileClass).Fixups.FindFixup(AAddress) <> -1 then
        begin
          Size := 4;
          if (PPChar(AAddress)^ >= PEFileClass.Code) and (PPChar(AAddress)^ < PEFileClass.Code + PEFileClass.CodeSize) then
            Jumps.Add(PPChar(AAddress)^);
        end
        else
        begin
          TempAddress := nil;
          DisAsm.GetInstruction(AAddress, Size);
          if TempAddress <> nil then
          begin
            // Check to see if it is a jump to HandleOnException.
            if (TPEFileClass(PEFileClass).FindSystemProc(HandleOnExceptionProcName) <> nil) and
               (TempAddress = TPEFileClass(PEFileClass).FindSystemProc(HandleOnExceptionProcName).Address) then
            begin
              TempAddress := PChar(High(Integer));
              for I := 1 to PDWord(AAddress + 5)^ do
              begin
                if PPChar(AAddress + 5 + I * 8)^ < TempAddress then
                  TempAddress := PPChar(AAddress + 5 + I * 8)^;
                if (PPChar(AAddress + 5 + I * 8)^ < PEFileClass.Code) or
                   (PPChar(AAddress + 5 + I * 8)^ >= PEFileClass.Code + PEFileClass.CodeSize) then
                  raise EDecompilerError.CreateFmt('Jump to outside the code section (2). %p %p',
                    [Pointer(AAddress + 5 + I * 8), Pointer(PPChar(AAddress + 5 + I * 8)^)]);
                Jumps.Add(PPChar(AAddress + 5 + I * 8)^);
              end;
              AAddress := TempAddress;
              Size := 0;
            end
            else
            begin
              // Save the address to which is jumped.
              if (TempAddress < PEFileClass.Code) or
                 (TempAddress >= PEFileClass.Code + PEFileClass.CodeSize) then
                raise EDecompilerError.CreateFmt('Jump to outside the code section. %p %p (3)',
                   [Pointer(AAddress), Pointer(TempAddress)]);
              Jumps.Add(TempAddress);
            end;
          end;
        end;
        Inc(AAddress, Size);
        if AAddress > PEFileClass.Code + PEFileClass.CodeSize then
          raise EDecompilerError.Create('Function outside Code section');
      end;
      // Check to see if the jumps are to code otherwise decompile it.
      Jumps.Sort(ListSimpleSort);
      for I := 0 to Jumps.Count -1 do
      begin
        AProc := FindProc(Jumps[I]);
        Proc.Comments.AddComment(Format('Jump to %p', [Pointer(Jumps[I])]), ctDebug);
        if AProc = nil then
        begin
          if (Jumps[I] < PEFileClass.Code) or (Jumps[I] >= PEFileClass.Code + PEFileClass.CodeSize) then
            Break;
          if TPEFileClass(PEFileClass).FindDecompItemByBlock(Jumps[I]) <> nil then
          begin
            Proc.AddReq(TPEFileClass(PEFileClass).FindDecompItemByBlock(Jumps[I]), Jumps[I]);
          end
          else
          begin
            Result := True;
            AProc := Add(Jumps[I]);
            AProc.Comments.AddComment('Jump created as an proc', ctInfo);
            AProc.PossProcTypes := ptAllStatic;
            AProc.Size := GetProcSize(AProc.Address);
            AProc.ProcSize := AProc.Size;
            AProc.AUnit := TPEFileClass(PEFileClass).Units.FindInUnitUsingFInit(AProc.Address);
            AnalyzeProc(AProc);
          end;
        end;
        if AProc <> nil then
        begin
          if AProc <> Proc then
          begin
            // If the Proc doesn't jump/call to the beginning of the Proc is must Append.
            if AProc.Address <> Jumps[I] then
            begin
              if AProc.Address > Proc.Address then
              begin
                Proc.Comments.AddComment(Format('Append after set because of analyse proc at %p', [Pointer(AProc.Address)]), ctDebug);
                Proc.AppendAfter := atMust;
                AProc.Comments.AddComment(Format('Append after set because of jump to a middle of a proc at %p', [Pointer(Proc.Address)]), ctDebug);
                AProc.AppendBefore := atMust;
              end
              else
                if Proc.AppendBefore <> atMayNot then
                begin
                  Proc.Comments.AddComment(Format('Append before set because of analyse proc at %p', [Pointer(AProc.Address)]), ctDebug);
                  Proc.AppendBefore := atMust;
                  AProc.Comments.AddComment(Format('Append before set because jump to a middle of a proc at %p', [Pointer(Proc.Address)]), ctDebug);
                  AProc.AppendAfter := atMust;
                end;
            end;
            Proc.AddReq(AProc, Jumps[I]);
          end;
        end;
      end;
    finally
      Jumps.Free;
    end;
  finally
    DisAsm.Free;
  end;
end;

function TProcs.Add(Address: PChar): TProc;
begin
  Result := TProc.Create(Self, Address);
end;

function TProcs.FindProc(Address: PChar): TProc;
var
  I, L, H, C: Integer;
begin
  L := 0;
  H := Count - 1;
  while L <= H do
  begin
    I := (L + H) shr 1;
    C := Items[I].Address - Address;
    if C <= 0 then
      L := I + 1
    else
      H := I - 1;
  end;

  for I := H downto 0 do
  begin
    Result := TProc(inherited GetItem(I));
    if (Result.Address = Address) or
       ((Result.Address <= Address) and
        (Result.Address + Result.Size > Address)) then
      Exit;
  end;
  Result := nil;
end;

function TProcs.FindProcByName(const Name: string): Integer;
begin
  for Result := 0 to Count -1 do
    if Items[Result].Name = Name then
      exit;
  Result := -1;
end;

function TProcs.FindProcIndex(Address: PChar; var Index: Integer): Boolean;
var
  L, H, I, C: Integer;
begin
  Result := False;
  L := 0;
  H := Count - 1;
  while L <= H do
  begin
    I := (L + H) shr 1;
    C := Items[I].Address - Address;
    if C < 0 then
      L := I + 1
    else
    begin
      H := I - 1;
      if C = 0 then
        Result := True;
    end;
  end;
  Index := L;
end;

function TProcs.GetItem(Index: Integer): TProc;
begin
  Result := TProc(inherited GetItem(Index));
end;

procedure TProcs.SetItem(Index: Integer; Value: TProc);
begin
  inherited SetItem(Index, Value);
end;

{ TClassInfo }

constructor TClassInfo.Create(ClassInfos: TClassInfos; AClass: TClass);
var
  InterfaceTable: PInterfaceTable;
  I, J: Integer;
  MethodCount: Integer;
resourcestring
  SErrorAncestorClassNotFound = 'Ancestor class from %s not found';
begin
  inherited Create(ClassInfos);
  FClass := AClass;
  FMethods := TList.Create;
  FInterfaces := TList.Create;
  InterfaceTable := AClass.GetInterfaceTable;
  FClassDef := TStringList.Create;
  // Find the ancestor class.
  for I := 0 to ClassInfos.Count -1 do
    if ClassInfos[I].AClass = AClass.ClassParent then
      FAncestorClass := ClassInfos[I];
  if FAncestorClass = nil then
    with TPEFileClass(PEFileClass).DecompThread do
      for J := 0 to PEFileClassCount -1 do
        for I := 0 to TPEFileClass(PEFileClasses[J]).Classes.Count -1 do
          if TPEFileClass(PEFileClasses[J]).Classes[I].AClass = AClass.ClassParent then
          begin
            FAncestorClass := TPEFileClass(PEFileClasses[J]).Classes[I];
            Break;
          end;
  if (FAncestorClass = nil) and (AClass.ClassName <> 'TObject') then
    raise EDecompilerError.CreateFmt(SErrorAncestorClassNotFound, [AClass.ClassName]);

  if InterfaceTable <> nil then
  begin
    for I := 0 to InterfaceTable.EntryCount -1 do
    begin
      MethodCount := Integer(InterfaceTable) -
         Integer(InterfaceTable.Entries[I].VTable);
      for J := I+1 to InterfaceTable.EntryCount -1 do
        if Integer(InterfaceTable.Entries[I].VTable) <
            Integer(InterfaceTable.Entries[J].VTable) then
        begin
          MethodCount := Integer(InterfaceTable.Entries[J].VTable) -
            Integer(InterfaceTable.Entries[I].VTable);
          break;
        end;
      FInterfaces.Add(TPEFileClass(ClassInfos.PEFileClass).Interfaces.Add(
         InterfaceTable.Entries[I].IID, MethodCount));
    end;
  end;

  // First address used is InterfaceVMT method or otherwise start VMT.
  Address := PChar(AClass) + vmtSelfPtr;
  if InterfaceTable <> nil then
  begin
    J := 0;
    // Get the first VTable.
    while J < InterfaceTable.EntryCount do
    begin
      I := Integer(InterfaceTable.Entries[J].VTable);
      if I <> 0 then
      begin
        while I < Integer(InterfaceTable) do
        begin
          if PInteger(I)^ < Integer(Address) then
            Address := PPChar(I)^;
          Inc(I, 4);
        end;
        break;
      end;
      Inc(J);
    end;
  end;
  // Address is 4 byte aligned
  Address := Address - Integer(Address) mod 4;

  // Size is address to end ClassName.
  Size := ((PInteger(Integer(AClass) + vmtClassName)^ + Length(AClass.ClassName)) div 4) * 4 + 4 - Integer(Address);
end;

procedure TClassInfo.AnaClass;

  procedure LoadVirtualMethods;
  var
    J: Integer;
    ParentClassCount: Integer;
    Address: PChar;
    Proc: TProc;
  const
    StdVirtualMethodNames: array[-8..-1] of string =
       ('SafeCallException', 'AfterConstruction', 'BeforeDestruction',
        'Dispatch', 'DefaultHandler', 'NewInstance', 'FreeInstance', 'Destroy');
    StdVirtualMethodParams: array[-8..-1] of string =
       ('ExceptObject: TObject; ExceptAddr: Pointer', '', '',
        'var Message', 'var Message', '', '', '');
    StdVirtualMethodResults: array[-8..-1] of string =
       ('HResult', '', '', '', '', 'TObject', '', '');
    StdVirtualMethodTypes: array[-8..-1] of TProcTypes =
      ([ptMethodProcedure], [ptMethodProcedure], [ptMethodProcedure],
       [ptMethodProcedure], [ptMethodProcedure], [ptClassProcedure],
       [ptMethodProcedure], [ptDestructor]);
  resourcestring
    SVirtualMethodInsideAnother = 'VirtualMethod at %p is inside another method at %p';
  begin
    if AClass.ClassParent <> nil then
      ParentClassCount := GetVirtualMethodCount(AClass.ClassParent)
    else
      ParentClassCount := -9;
    // J = -8 to start with the virtual Methods in the VMT.
    for J := -8 to GetVirtualMethodCount(AClass) -1 do
    begin
      Address := VMTUtils.GetVirtualMethod(AClass, J);
      // Search for an already existing proc.
      if (Address <> nil) then
        Proc := TPEFileClass(PEFileClass).Procs.FindProc(Address)
      else
        Proc := nil;
      // Don't add a virtual method if this is the same as the parents virtual method
      if (J >= ParentClassCount) or ((VMTUtils.GetVirtualMethod(AClass.ClassParent, J) <> Address) and
          ((Proc = nil) or (not Proc.ImportInfo.Imported))) then
      begin
        // Compare the Method with AbstractError procedure.
        if (TPEFileClass(PEFileClass).FindSystemProc(AbstractErrorProcName) <> nil) and
           (TPEFileClass(PEFileClass).FindSystemProc(AbstractErrorProcName).Address = Address) then
        begin
          // Set the address to nil to indicate that it is a abstract method.
          Address := nil;
          Proc := nil;
        end;
        if Proc = nil then
          Proc := TPEFileClass(PEFileClass).Procs.Add(Address);
        if Proc.Address <> Address then
          raise EDecompilerError.CreateFmt(SVirtualMethodInsideAnother, [Pointer(Address), Pointer(Proc.Address)]);
        with Proc do
        begin
          Comments.AddComment('Virtual method', ctDebug);
          PossProcTypes := PossProcTypes * ptMethods;
          MethodBindingType := mbtVirtual;
          AClass := Self;
          if Address = nil then
            AddReq(TPEFileClass(PEFileClass).FindSystemProc(AbstractErrorProcName), nil);
          MethodIndex := J;
          if J < ParentClassCount then
            Overrides := True;
          // If the proc is a standard proc, set the it values.
          if J < 0 then
          begin
            Name := StdVirtualMethodNames[J];
            Parameters.Parameters := StdVirtualMethodParams[J];
            Parameters.FuncResult := StdVirtualMethodResults[J];
            PossProcTypes := StdVirtualMethodTypes[J];
          end;
        end;
      end;
    end;
  end;

  procedure LoadFieldsProp;
  var
    PropCount: Integer;
    I: Integer;

    procedure AnaProc(Proc: PChar; ParamStr, ResultStr: string; TypeInfo: PTypeInfo);
    var
      AProc: TProc;
      CarProc: Cardinal absolute Proc;
    begin
      if Proc = nil then Exit;
      if CarProc < $FE000000 then
      begin
        // Static Proc.
        AProc := TPEFileClass(PEFileClass).Procs.FindProc(Proc);
        if AProc = nil then
          AProc := TPEFileClass(PEFileClass).Procs.Add(Proc);
        AProc.Comments.AddComment('Field property getter/setter', ctInfo);
        AProc.Parameters.Parameters := ParamStr;
        AProc.Parameters.FuncResult := ResultStr;
        AProc.AClass := Self;
        AProc.PossProcTypes := [ptMethodProcedure];
      end
      else if CarProc > $FF000000 then
      begin
        // Fields.
        CarProc := Smallint(Proc);
        if Fields.FindFieldOffset(CarProc) = nil then
          Fields.AddField(CarProc, TypeInfo).Name :=
             Format('F%s%d', [TypeInfo^.Name, CarProc]);
      end
      else
      begin
        // Virtual Methods.
        AProc := GetVirtualMethod(Smallint(CarProc) div 4);
        AProc.PossProcTypes := [ptMethodProcedure];
        AProc.Parameters.Parameters := ParamStr;
        AProc.Parameters.FuncResult := ResultStr;
      end;
    end;

  var
    PropInfo: PPropInfo;
  begin
    // Load Fields using properties.
    if AClass.ClassInfo = nil then Exit;
    PropInfo := @(GetPropData(GetTypeData(AClass.ClassInfo))^.PropList);
    PropCount := GetPropData(GetTypeData(AClass.ClassInfo)).PropCount;
    for I := 0 to PropCount -1 do
    begin
      // Add the type info the the req decomps.
      AddReq(TPEFileClass(PEFileClass).FindTypeByName(PropInfo^.PropType^^.Name), nil);
      // Analyse the GetProc
      if Cardinal(PropInfo^.Index) <>  $80000000 then
        AnaProc(PropInfo^.GetProc, 'Index: Integer', PropInfo^.PropType^^.Name, PropInfo^.PropType^)
      else
        AnaProc(PropInfo^.GetProc, '', PropInfo^.PropType^^.Name, PropInfo^.PropType^);
      // Analyse the SetProc
      if Cardinal(PropInfo^.Index) <> $80000000 then
        AnaProc(PropInfo^.SetProc, 'Index: Integer; Value: ' + PropInfo^.PropType^^.Name, '', PropInfo^.PropType^)
      else
        AnaProc(PropInfo^.SetProc, 'Value: ' + PropInfo^.PropType^^.Name, '', PropInfo^.PropType^);
      // Analyse the stored proc
      if Cardinal(PropInfo^.StoredProc) > 1 then
      begin
        if Cardinal(PropInfo^.Index) <>  $80000000 then
          AnaProc(PropInfo^.StoredProc, 'Index: Integer', 'Boolean', TypeInfo(Boolean))
        else
          AnaProc(PropInfo^.StoredProc, '', 'Boolean', TypeInfo(Boolean));
      end;
      NextPropInfo(PropInfo);
    end;
  end;

  procedure LoadFieldsInit;
  var
    InitTable: PTypeInfo;
    InitTableData: PRecordTypeData;
    I: Integer;
  begin
    // Load Fields using the init table.
    InitTable := GetInitTable(AClass);
    if InitTable <> nil then
    begin
      InitTableData := PRecordTypeData(GetTypeData(InitTable));
      for I := 0 to InitTableData.FieldCount -1 do
      begin
        with InitTableData.Fields[I] do
        begin
          // Add the type info the the req decomps.
          AddReq(TPEFileClass(PEFileClass).FindTypeByName(TypeInfo^.Name), nil);
          if Fields.FindFieldOffset(Offset) = nil then
            Fields.AddField(Offset, TypeInfo^).Name :=
               Format('F%s%d', [TypeInfo^.Name, Offset]);
        end;
      end;
    end;
  end;

  procedure LoadPublishedFields;
  var
    FieldTable: PFieldTable;
    FieldEntry: PFieldEntry;
    I: Integer;
    Field: TdcField;
  begin
    FieldTable := GetFieldTable(AClass);
    if FieldTable <> nil then
    begin
      FieldEntry := @FieldTable.FirstEntry;
      for I := 0 to FieldTable.EntryCount -1 do
      begin
        Field := Fields.FindFieldOffset(FieldEntry.Offset);
        if Field = nil then
          Field := Fields.AddField(FieldEntry.Offset);
        Field.Name := FieldEntry.Name;
        Field.FieldType := FieldTable.FieldClassTable.Classes[FieldEntry.IDX].ClassName;
        Field.Size := 4;
        Field.Visiblity := vPublished;
        FieldEntry := Pointer(Integer(FieldEntry) + SizeOf(Integer) + SizeOf(Word) + Ord(FieldEntry.Name[0]) + 1);
      end;
    end;
  end;

begin
  // Load virtual methods
  LoadVirtualMethods;
  // Create the Fields
  if FAncestorClass <> nil then
    FFields := TdcFieldList.Create(AClass.InstanceSize, FAncestorClass.Fields, Self)
  else
    FFields := TdcFieldList.Create(AClass.InstanceSize, nil, Self);
  LoadFieldsProp;
  LoadFieldsInit;
  LoadPublishedFields;
end;

destructor TClassInfo.Destroy;
begin
  FClassDef.Free;
  FInterfaces.Free;
  FMethods.Free;
  inherited Destroy;
end;

procedure TClassInfo.GenerateClassDef;

  function GetProcName(Proc: Pointer): string;
  begin
    if Cardinal(Proc) < $FE000000 then
      // Static Proc.
      Result := TPEFileClass(PEFileClass).Procs.FindProc(Proc).Name
    else if Cardinal(Proc) < $FF000000 then
      // Virtual Proc.
      Result := GetVirtualMethod((Cardinal(Proc) and $00FFFFFF) div 4).Name
    else
      // field.
      Result := Fields.FindFieldOffset(Cardinal(Proc) and $00FFFFFF).Name;
  end;

var
  FVisibility: TdcVisibility;

  procedure SetVisibility(NewVisibility: TdcVisibility);
  const
    VisibilityWord: array[TdcVisibility] of string =
      ('  private', '  protected', '  public', '  published', '  automated');
  begin
    if NewVisibility <> FVisibility then
      ClassDef.Add(VisibilityWord[NewVisibility]);
    FVisibility := NewVisibility;
  end;

var
  Field: TdcField;
  I, J: Integer;
  PropInfo: PPropInfo;
  Str: string;
begin
  // exit when class def already exists or the object is declared inside system or sysinit.
  if (ClassDef.Count <> 0) or (AUnit.Index < 2) then exit;
  // Does the class has typeinfo and the parent doesn't,
  // Add {$M+} to indicate that the class must have typeinfo
  if (AClass.ClassInfo <> nil) and (AClass.ClassParent.ClassInfo = nil) then
    ClassDef.Add('  {$M+}');
  // Start with the class(ClassName) (no interface support jet)
  ClassDef.Add(Format('  %s = class(%s)', [AClass.ClassName, AClass.ClassParent.ClassName]));

  FVisibility := vPublished;
  // Start with the field declaration at the previous instancesize.
  I := AClass.ClassParent.InstanceSize;
  // Keep determining the type of the next 4 bytes until the the InstanceSize is reached.
  while I < AClass.InstanceSize do
  begin
    Field := Fields.FindFieldOffset(I);
    if Field <> nil then
    begin
      SetVisibility(Field.Visiblity);
      ClassDef.Add('    ' + Field.Declaration + ';');
      Inc(I, Field.Size);
    end
    else
    begin
      SetVisibility(vPublic);
      ClassDef.Add(Format('    Fields%d: byte;', [I]));
      Inc(I, 1);
    end;
  end;
  // Make the Method declarations.
  for I := 0 to MethodCount -1 do
  begin
    if Methods[I].FPublished then
      SetVisibility(vPublished)
    else
      SetVisibility(vPublic);
    ClassDef.Add('    ' + Methods[I].DefSrc);
  end;
  // declare the published properties.
  if AClass.ClassInfo <> nil then
  begin
    J := GetPropData(GetTypeData(AClass.ClassInfo)).PropCount;
    PropInfo := @GetPropData(GetTypeData(AClass.ClassInfo)).PropList;
    for I := 0 to J -1 do
    begin
      SetVisibility(vPublished);
      Str := '    property ' + PropInfo^.Name + ': ' + GetTypeInfoName(PropInfo^.PropType^);
      if Cardinal(PropInfo^.Index) <>  $80000000 then
        Str := Str + ' index ' + IntToStr(PropInfo^.Index);
      if PropInfo^.GetProc <> nil then
        Str := Str + ' read ' + GetProcName(PropInfo^.GetProc);
      if PropInfo^.SetProc <> nil then
        Str := Str + ' write ' + GetProcName(PropInfo^.SetProc);
      if PropInfo^.StoredProc <> Pointer($1) then
      begin
        if PropInfo^.StoredProc = nil then
          Str := Str + ' stored False'
        else
          Str := Str + ' stored ' + GetProcName(PropInfo^.StoredProc);
      end;
      Str := Str + ';';
      ClassDef.Add(Str);
      NextPropInfo(PropInfo);
    end;
  end;
  // end the class with end;
  ClassDef.Add('  end;');
  // Does the class has typeinfo and the parent doesn't,
  // Add {$M+} to indicate that the class must have typeinfo
  if (AClass.ClassInfo <> nil) and (AClass.ClassParent.ClassInfo = nil) then
    ClassDef.Add('  {$M-}');
  ClassDef.Add('');
end;

function TClassInfo.IsRefAddress(AAddress: PChar): Boolean;
begin
  Result := (AAddress = PChar(AClass) + vmtSelfPtr) or (AAddress = PChar(AClass));
end;

function TClassInfo.GetMethod(Index: Integer): TProc;
begin
  Result := FMethods[Index];
end;

function TClassInfo.GetMethodCount: Integer;
begin
  Result := FMethods.Count;
end;

function TClassInfo.GetInterface(Index: Integer): TInterface;
begin
  Result := FInterfaces[Index];
end;

function TClassInfo.GetInterfaceCount: Integer;
begin
  Result := FInterfaces.Count;
end;

function TClassInfo.GetVirtualMethod(Index: Integer): TProc;
var
  I: Integer;
begin
  for I := 0 to MethodCount -1 do
  begin
    Result := Methods[I];
    if (Result.MethodBindingType = mbtVirtual) and
       (Result.MethodIndex = Index) then
      Exit;
  end;
  if FAncestorClass <> nil then
    Result := FAncestorClass.GetVirtualMethod(Index)
  else
    raise EDecompilerError.CreateFmt('virtual Methods with Index %d not found.', [Index]);
end;

function TClassInfo.GetDynamicMethod(Index: Integer): TProc;
var
  I: Integer;
begin
  for I := 0 to MethodCount -1 do
  begin
    Result := Methods[I];
    if (Result.MethodBindingType = mbtDynamic) and
       (Result.MethodIndex = Index) then
      Exit;
  end;
  if FAncestorClass <> nil then
    Result := FAncestorClass.GetDynamicMethod(Index)
  else
    raise EDecompilerError.CreateFmt('Dynamci Methods with Index %d not found.', [Index]);
end;

function TClassInfo.FindProc(const ProcName: string): TProc;
var
  I: Integer;
begin
  Result := nil;
  for I := 0 to FMethods.Count -1 do
    if AnsiCompareText(TProc(FMethods[I]).Name, ProcName) = 0 then
    begin
      Result := FMethods[I];
      Exit;
    end;
  if AncestorClass <> nil then
    Result := AncestorClass.FindProc(ProcName);
end;

{ TClassInfos }

constructor TClassInfos.CreateDecomp(PEFileClass: TPEFile);
begin
  inherited CreateDecomp(PEFileClass);
  FOnLoadClasses := TmlneMethodList.Create;
end;

destructor TClassInfos.Destroy;
begin
  FOnLoadClasses.Free;
  inherited Destroy;
end;

procedure TClassInfos.LoadClassInfos;
var
  I: PChar;
  J, K: integer;
  PossClasses: TList;
  Added: Boolean;
begin
  // Extract all the classes from the PEFile.
  PossClasses := TList.Create;
  try
    // Try all address in the code.
    with TPEFileClass(PEFileClass) do
    begin
      I := Code - vmtSelfPtr;
      while I < Code + CodeSize do
      begin
        // vmtSelfPtr must point to itself.
        if PPChar(I + vmtSelfPtr)^ = I then
        begin
            if PPChar(I + vmtParent)^ = nil then
            try
              // If no classParent then class can be object
              if (not UsePackages) and (TClass(I).ClassName = 'TObject') then
                // if class is object add it to classes.
                Add(TClass(I));
            except
              on EAccessViolation do
            end
            else
              // className must be in the code section.
              // classParent must be in the code section or the import section (when it is imported).
              if (PPChar(I + vmtClassName)^ <= Code + CodeSize) and
                 (PPChar(I + vmtClassName)^ >= Code) and
                 (((PPChar(I + vmtParent)^ <= Code + CodeSize) and
                   (PPChar(I + vmtParent)^ >= Code)) or
                  ((PPChar(I + vmtParent)^ <= ImportStart + ImportSize) and
                   (PPChar(I + vmtParent)^ >= ImportStart))) then
                // Add possible class to possible class list.
                PossClasses.Add(I);
        end;
        Inc(I, 4);
      end;
       // Can't be more then 1 TObject.
      if (not TPEFileClass(PEFileClass).UsePackages) and (Count > 1) then
        raise EDecompilerError.Create('There can only be one TObject.');

      // Add Classes to the list which parent is in the list or in one of the packages.
      repeat
        Added := False;
        for J := PossClasses.Count -1 downto 0 do
        begin
          // Try to find parent class in classList.
          if FindClass(TClass(PossClasses[J]).ClassParent) <> nil then
          begin
            // Class in class list
            Add(PossClasses[J]);
            PossClasses.Delete(J);
            Added := True;
            Continue;
          end;
          // Try to find parent class in a other package.
          for K := 0 to High(PEFiles) do
            if PEFiles[K].Classes.FindClass(TClass(PossClasses[J]).ClassParent) <> nil then
            begin
              // Class in class list
              Add(PossClasses[J]);
              PossClasses.Delete(J);
              Added := True;
              Break;
            end;
        end;
      until not Added;
    end;
  finally
    PossClasses.Free;
  end;
  // Call the event handler.
  OnLoadClasses.CallFirst;
end;

procedure TClassInfos.GenerateClassDefs;
var
  I: Integer;
begin
  for I := 0 to Count -1 do
    Items[I].GenerateClassDef;
end;

function TClassInfos.Add(AClass: TClass): TClassInfo;
begin
  Result := TClassInfo.Create(Self, AClass);
end;

function TClassInfos.FindClass(AClass: TClass): TClassInfo;
var
  I: Integer;
begin
  for I := 0 to Count - 1 do
  begin
    Result := TClassInfo(inherited GetItem(I));
    if Result.AClass = AClass then Exit;
  end;
  Result := nil;
end;

function TClassInfos.FindClassByName(const Name: string): TClassInfo;
var
  I: Integer;
begin
  for I := 0 to Count - 1 do
  begin
    Result := TClassInfo(inherited GetItem(I));
    if AnsiCompareText(Result.AClass.ClassName, Name) = 0 then Exit;
  end;
  Result := nil;
end;

function TClassInfos.GetItem(Index: Integer): TClassInfo;
begin
  Result := TClassInfo(inherited GetItem(Index));
end;

procedure TClassInfos.SetItem(Index: Integer; Value: TClassInfo);
begin
  inherited SetItem(Index, Value);
end;

{ TString }

constructor TStringInfo.Create(StringInfos: TStringInfos; Address: PChar; StringType: TStringType; ASize: Integer = 0);
var
  I, J: Integer;
  AAddress: PChar;
begin
  inherited Create(StringInfos);
  FStringAddress := Address;
  FStringType := StringType;
  Self.Address := Address;
  RefAddress := Address;
  case StringType of
    stAnsiString: begin
                    FValue := String(Pointer(Address + 8));
                    Size := (PInteger(Address +4)^ div 4) * 4 + 12;
                    RefAddress := Address + 8;
                  end;
    stWideString: begin
                    FValue := WideString(Pointer(Address + 8));
                    Size := (PInteger(Address +4)^ div 4) * 4 + 12;
                  end;
    stPAnsiChar: begin
                   if ASize <> 0 then
                   begin
                     SetLength(FValue, ASize);
                     Move(Address[0], FValue[1], ASize);
                     Size := ASize
                   end
                   else
                   begin
                     SetLength(FValue, StrLen(Address));
                     FValue := Address;
                     Size := StrLen(Address) div 4 * 4 + 4;
                   end;
                 end;
    stPWideChar: begin
                  FValue := PWideChar(Pointer(Address));
                  Size := (StrLen(PChar(FValue)) * 2 + 1) div 4 * 4 + 4;
                end;
    stResourceString: begin
                        Size := 8;
                        with StringInfos.PEFileClass do
                          for I := 0 to High(Resources) do
                            if (Resources[I].NameOrID = niID) and (Resources[I].ID = 6) then
                            begin
                              with Resources[I] do
                                AAddress := Entries[High(Entries) - ($FFFF - PInteger(Address + 4)^) div 16].Entries[0].Data;
                              for J := 1 to 15 - ($FFFF - PInteger(Address + 4)^) mod 16 do
                                AAddress := AAddress + PWord(AAddress)^ * 2 + 2;
                              FValue := PWideChar(Pointer(AAddress + 2));
                              SetLength(FValue, PWord(AAddress)^);
                              break;
                            end;
                        end;
  end;
end;

function TStringInfo.GetStringTypeName: string;
const
  StringTypeNames: array[TStringType] of string = ('string', 'widestring',
     '', 'PChar', 'PWideChar');
begin
  Result := StringTypeNames[StringType];
end;

function TStringInfo.IsRefAddress(AAddress: PChar): Boolean;
begin
  Result := (AAddress = FStringAddress);
end;

{ TStrings }

procedure TStringInfos.LoadStringInfos;
var
  I, J: Integer;
label
  Next;
begin
  with TPEFileClass(PEFileClass) do
  begin
    for I := 0 to Fixups.Count -1 do
      with Fixups[I] do
        if (FixupType = 3) and
           (PPChar(Address)^ < Code + CodeSize) and
           (PPChar(Address)^ >= Code +8) then
        begin
          if (PDWord(Address)^ mod 4 = 0) and
             (PDWord(PPChar(Address)^ -8)^ = $FFFFFFFF) and
             (PPChar(Address)^ + PDWord(PPChar(Address)^ - 4)^ < Code + CodeSize) and
             (PPChar(Address)^ + PDWord(PPChar(Address)^ - 4)^ >= Code) and
             ((PPChar(Address)^ + PDWord(PPChar(Address)^ -4)^)[0] = #0) and
             (TPEFileClass(PEFileClass).FindDecompItemByBlock(PPChar(Address)^ -8) = nil) then
          begin
            for J := 0 to PDWord(PPChar(Address)^ -4)^ -1 do
              if PPChar(Address)^[J] = #0 then
                goto Next;
            with TStringInfo.Create(Self, PPChar(Fixups[I].Address)^ -8, stAnsiString)do
            begin
              Name := Format('String%p', [Pointer(Address)]);
              Comments.AddComment(Format('Ansi string pointed from %p', [Pointer(Fixups[I].Address)]), ctDebug);
            end;
          Next:
          end;
        end;
  end;
end;

function TStringInfos.FindString(Address: PChar): TStringInfo;
var
  I: Integer;
begin
  for I := 0 to Count -1 do
  begin
    Result := TStringInfo(inherited GetItem(I));
    if Result.StringAddress = Address then exit;
  end;
  Result := nil;
end;

function TStringInfos.GetItem(Index: Integer): TStringInfo;
begin
  Result := TStringInfo(inherited GetItem(Index));
end;

procedure TStringInfos.SetItem(Index: Integer; Value: TStringInfo);
begin
  inherited SetItem(Index, Value);
end;

function IsTypeDecomp(Decomp: TDecompItem): Boolean;
begin
  Result := (Decomp is TTypeInfoInfo) or
            (Decomp is TClassInfo) or (Decomp is TNoTInfoType);
end;

function Align4(Address: PChar): PChar;
begin
  if Address = nil then
    Result := nil
  else
    Result := PChar(Integer(Address -1) div 4 * 4 + 4);
end;

function Align4(Value: Integer): Integer;
begin
  if Value = 0 then
    Result := 0
  else
    Result := (Value -1) div 4 * 4 + 4;
end;

end.

