unit PEFileClass;

interface

uses
  Classes, PEFile, Windows, SysUtils, TypInfo, dcDecompThread,
  {$IFOPT D+} dcdebug, dialogs, {$ENDIF}
  Vars, dcUnits, dcDecomps, Procs, dcNTInfoTypes, dcThrVar, dcDFMs;

type
  EDecompilerError = class(Exception);

  TProjectType = (ptProgram, ptDLL, ptPackage);

  { TPEFileClass }

  TPEFileClass = class(TPEFile)
  private
    FProjectType: TProjectType;
    FUsePackages: Boolean;
    FImportStart: PChar;
    FImportSize: Integer;
    FDecompThread: TdcDecompThread;
    FMainFile: Boolean;
    FDestroying: Boolean;
    function NotRecurFindTypeByName(const Name: string): TDecompItem;
    function NotRecurFindDecompItemByRef(AAddress: PChar): TDecompItem;
  public
    InitTable: PackageInfo;
    HasFFMTObj: Boolean;

    StringInfos: TStringInfos;
    Classes: TClassInfos;
    Interfaces: TInterfaces;
    TypeInfos: TTypeInfoInfos;
    NoTInfoTypes: TNoTInfoTypes;
    Procs: TProcs;
    Units: TUnits;
    Miscs: TMiscs;
    VarInfos: TVarInfos;
    EntryPointProc: TProc;
    Decomps: TDecompList;
    DFMs: TdcDFMs;
    constructor CreateDecomp(const AFileName: string; ADecompThread: TdcDecompThread; AMainFile: Boolean); virtual;
    destructor Destroy; override;
    procedure LoadPackages;

    function FindDecompItemByRef(AAddress: PChar): TDecompItem;
    function FindDecompItemByBlock(Address: PChar): TDecompItem;
    function FindDecompItemAfter(AAddress: PChar): TDecompItem;
    function FindTypeByName(const Name: string): TDecompItem;

    function FindSystemProc(const Name: string): TProc;

    property ProjectType: TProjectType read FProjectType;
    property UsePackages: Boolean read FUsePackages;
    property ImportStart: PChar read FImportStart;
    property ImportSize: Integer read FImportSize;

    property DecompThread: TdcDecompThread read FDecompThread;
    property MainFile: Boolean read FMainFile;
    property Destroying: Boolean read FDestroying;
  end;

function ListPointerToSort(Item1, Item2: Pointer): Integer;
function ListSimpleSort(Item1, Item2: Pointer): Integer;

function EnhQuotedStr(Str: string): string;

function IsIdentifier(Address: PChar): Boolean;

function IsPackage(const FileName: string): Boolean;

var
  PEFiles: array of TPEFileClass;

implementation

uses
  {$IFDEF VER120} ObjFileConsts4, {$ELSE} {$IFDEF VER130} ObjFileConsts5, {$ENDIF} {$ENDIF}
  DisAsm, VMTUtils, ProcDecomp, peExports,
  TypeInfoUtils, NameMangling;

{ TPEFileClass }

function ListPointerToSort(Item1, Item2: Pointer): Integer;
begin
  Result := PPChar(Item1)^ - PPChar(Item2)^;
end;

function ListSimpleSort(Item1, Item2: Pointer): Integer;
begin
  Result := Integer(Item1) - Integer(Item2);
end;

function DecompItemSort(Item1, Item2: Pointer): Integer;
begin
  Result := TDecompItem(Item1).Address - TDecompItem(Item2).Address;
end;

constructor TPEFileClass.CreateDecomp(const AFileName: string; ADecompThread: TdcDecompThread; AMainFile: Boolean);

  procedure LoadInitTable;
  var
    AAddress: PChar;
    I: Integer;
  resourcestring
    SErrorFindPckgInfoTable = 'Error finding the package info table';
    SErrorInInitTable = 'Error in init table';
  begin
    if ProjectType = ptPackage then
    begin
      // InitTable is pointed in the proc @PackageName@@GetPackageInfoTable$qqrv
      I := PEExports.FindCaseInSens('@' + ProjectName + '@@GetPackageInfoTable$qqrv');
      if I = -1 then
        raise EDecompilerError.Create(SErrorFindPckgInfoTable);
      InitTable := PPointer(PEExports[I].Address + 1)^;
    end
    else
    begin
      { InitTable is directly before the EntryPoint, search for the first non-fixups address
        which is not nil }
      AAddress := EntryPoint -4;
      while (PDWord(AAddress)^ = 0) or (Fixups.FindFixup(AAddress) <> -1) do
        Dec(AAddress, 4);
      InitTable := Pointer(AAddress);
    end;
    // There must be minimal 3 unit (system, SysInit, Project).
    if InitTable^.UnitCount < 3 then
      raise EDecompilerError.Create(SErrorInInitTable);
    // Create a block for the init table.
    with TDecompItem.Create(miscs) do
    begin
      Address := PChar(InitTable);
      RefAddress := PChar(InitTable);
      Size := InitTable^.UnitCount * 8 + 8;
      Comments.AddComment('InitTable', ctDebug);
    end;
  end;

  procedure RemoveProcsInAnother;
  var
    I, J: Integer;
  resourcestring
    SProcMayNotAppend = 'Proc may not append. %p';
  begin
    for I := Procs.Count -1 downto 0 do
    begin
      if Procs[I].Size <> 0 then
        for J := I +1 to Procs.Count -1 do
        begin
          if Procs[J].Address > Procs[I].Address + Procs[I].Size then
            Break;
          if Procs[J].Size <> 0 then
          begin
            if (Procs[J].Address >= Procs[I].Address) and
               (Procs[J].Address + Procs[J].Size <= Procs[I].Address + Procs[I].Size) then
            begin
              {$IFOPT D+}SendDebug(Format('Proc inside another proc %p *',
                 [Pointer(Procs[J].Address)]));{$ENDIF}
              if (Procs[J].AppendBefore = atMayNot) or (Procs[J].AppendAfter = atMayNot) then
              begin
                if Procs[I].AppendAfter = atMayNot then
                  raise EDecompilerError.CreateFmt(SProcMayNotAppend, [Pointer(Procs[J].Address)]);
                Procs[I].Size := Procs[J].Address - Procs[I].Address;
                Procs[I].ProcSize := Procs[I].Size;
                Procs[I].Comments.AddComment(Format('Proc truncated at %p', [Pointer(Procs[J].Address)]), ctDebug);
              end
              else
              begin
                Procs[I].Comments.AddComment(Format('Had a proc inside at %p', [Pointer(Procs[J].Address)]), ctDebug);
                Procs[J].Free;
              end;
              break;
            end;
          end;
        end;
    end;
  end;

  procedure AppendProcs;
  var
    Proc: TProc;
    I, J, K: Integer;
    Nothing: Boolean;
    DC: TDecompItem;
  label
    Next;
  resourcestring
    STryAppendNot4ByteProc = 'Not 4 byte aligend proc at %p can''t be appended';
  begin
    RemoveProcsInAnother;
    // Search all the not 4 byte aligned procs (not the ones in the sytem obj files).
    repeat
      Nothing := True;
    for I := Procs.Count -1 downto 1 do
    begin
      if (Procs[I].Size <> 0) and
         ((Procs[I].AUnit = nil) or (TUnit(Procs[I].AUnit).UnitType <> utSystem)) then
      begin
        // Search the first proc before the proc.
        J := I;
        repeat
          Dec(J);
          if J < 0 then
            goto Next;
        until Procs[J].Size <> 0;
        Proc := Procs[J];

        if (Integer(Procs[I].Address) mod 4 <> 0) and (Procs[I].AppendBefore <> atMayNot) then
        begin
          if Proc <> nil then
          begin
            // Don't append if there is something between the two procs.
            if (Procs[I].Address = Proc.Address + Proc.Size) or
               (FindDecompItemAfter(Proc.Address + Proc.Size) = Procs[I]) then
            begin
              if Procs[I].Address > Proc.Address + Proc.Size then
                Proc.Comments.AddComment(Format('Not filling between Proc %p %p, %p',
                  [Pointer(Proc.Address), Pointer(Proc.Address + Proc.Size),
                        Pointer(Procs[I].Address)]), ctWarning);
              // Append the proc to the found proc.
              Proc.Comments.AddComment('Proc appended because the first wasn''t 4 byte aligned', ctDebug);
              Proc.Append(Procs[I]);
              Nothing := False;
            end;
          end
          else
            raise EDecompilerError.CreateFmt(STryAppendNot4ByteProc, [Pointer(Procs[I].Address)]);
        end
        else
          if (Proc <> nil) and (Procs[I].Address = Proc.Address + Proc.Size) and
             ((Proc.AppendAfter = atMust) or (Procs[I].AppendBefore = atMust)) then
          begin
            if Proc.Size = 0 then
              goto Next;
            // Append the proc to the found proc.
            Proc.Comments.AddComment('Proc appended because one of the procs must', ctDebug);
            Proc.Append(Procs[I]);
            {$IFOPT D+}SendDebug(Format('Special Proc appended %p %p',
              [Pointer(Proc.Address), Pointer(Proc.Address + Proc.Size)]));{$ENDIF}
            Nothing := False;
          end;
        // Append the pchar before if it must append before.
        if Nothing and (Procs[I].AppendBefore = atMust) then
        begin
          DC := FindDecompItemByBlock(Procs[I].Address -1);
          if (DC <> nil) and (DC is TStringInfo) and
             (TStringInfo(DC).StringType = stPAnsiChar) then
          begin
            J := Procs[I].Address + Procs[I].Size - DC.Address;
            K := Procs[I].Address + Procs[I].ProcSize - DC.Address;
            Procs[I].Address := DC.Address;
            Procs[I].Size := J;
            Procs[I].ProcSize := K;
            DC.Free;
            Nothing := False;
          end;
        end;
      end;
      Next:
    end;
    until Nothing;
    DecompThread.CheckTerminated;
  end;

  procedure AlignProcs;
  var
    I: Integer;
  begin
    for I := 0 to Procs.Count -1 do
      if (Procs[I].AppendAfter <> atMust) and (Procs[I].Size mod 4 <> 0) then
        Procs[I].Size := Align4(Procs[I].Size);
  end;

  procedure CheckSystemCode;
  var
    LowestAddress: PChar;
    MaxAddress: PChar;
    I, J: Integer;
    CodeItems: TList;
    {$IFOPT D+}
    LastDC: TDecompItem;
    {$ENDIF}
  label
    Next;
  begin
    CodeItems := TList.Create;
    // Add all code items to the CodeItems list.
    CodeItems.Capacity := Classes.Count + TypeInfos.Count + Procs.Count + StringInfos.Count;
    for I := 0 to Classes.Count -1 do
      CodeItems.Add(Classes[I]);
    for I := 0 to TypeInfos.Count -1 do
      CodeItems.Add(TypeInfos[I]);
    for I := 0 to Procs.Count -1 do
      if (Procs[I].Size <> 0) and (Procs[I].Address <> nil) then
        CodeItems.Add(Procs[I]);
    for I := 0 to StringInfos.Count -1 do
      CodeItems.Add(StringInfos[I]);
    for I := 0 to Miscs.Count -1 do
      if Miscs[I].Address <> nil then
        CodeItems.Add(Miscs[I]);

    // Remove all the codeitem in the System unit to find overlapping code items and gaps.
    MaxAddress := Code;
    {$IFOPT D+}
    LastDC := nil;
    {$ENDIF}
    while (CodeItems.Count <> 0) and (MaxAddress <= Units.SystemUnit.FInit.Address) do
    begin
      LowestAddress := PChar(High(Integer));
      J := -1;
      for I := 0 to CodeItems.Count -1 do
      begin
        if TDecompItem(CodeItems[I]).Address < MaxAddress then
        begin
          {$IFOPT D+}
           raise EDecompilerError.CreateFmt('Overlapping code items %p %p %s' ,
             [Pointer(MaxAddress), Pointer(TDecompItem(CodeItems[I]).Address), LastDC.ClassName]);
          {$ELSE}
          raise EDecompilerError.CreateFmt('Overlapping code items %p %p' ,
             [Pointer(MaxAddress), Pointer(TDecompItem(CodeItems[I]).Address)]);
          {$ENDIF}
        end;
        if TDecompItem(CodeItems[I]).Address < LowestAddress then
        begin
          LowestAddress := TDecompItem(CodeItems[I]).Address;
          J := I;
        end;
      end;

     {$IFOPT D+}
       if LowestAddress <> MaxAddress then
         SendDebug(Format('System Gap between items %p %p',
          [Pointer(MaxAddress), Pointer(LowestAddress)]));
     {$ENDIF}
     Assert(J <> -1, 'J = -1');
      // Save the end address of the lowest code item as max address.
      MaxAddress := LowestAddress + TDecompItem(CodeItems[J]).Size;
      {$IFOPT D+}
      LastDC := TDecompItem(codeItems[J]);
      {$ENDIF}
      CodeItems.Delete(J);
    end;
    CodeItems.Free;
  end;

  procedure CheckCode;
  var
    MaxAddress: PChar;
    I: Integer;
    CodeItems: TList;
    LastDC: Integer;
  {$IFOPT D+}
    DbugCounter: Integer;
  {$ENDIF}
  label
  {$IFOPT D+}
    NextChecK,
  {$ENDIF}
    ReDo;

    function AnaUnknownBlock(var StartAddress: PChar; EndAddress: PChar): Boolean;
    var
      OrgEndAddress: PChar;
      FixupsTo: TList;
      FixupsInside: TList;
      P: Integer;
      Adjusted: Boolean;
      XAddress: PChar;
      PCharEndAddress: PChar;
      ProcSize: Integer;
      PossType: set of (ptPChar, ptProc, ptGUID);
      AProc: TProc;
      ALog: string;
    resourcestring
      SProcSize0Error = 'Proc with size 0 at %p';
    begin
      Result := False;
      OrgEndAddress := EndAddress;
      FixupsTo := TList.Create;
      try
        FixupsInside := TList.Create;
        try
          // Generate a list of Fixups To and inside the block,
          // order on the address which contains the fixup.
          for P := 0 to Fixups.Count -1 do
          begin
            if (Fixups[P].FixupType = 3) and
               (PPChar(Fixups[P].Address)^ >= StartAddress) and
               (PPChar(Fixups[P].Address)^ < EndAddress) then
            begin
              FixupsTo.Add(Fixups[P].Address);
            end;
            if (Fixups[P].Address >= StartAddress) and
               (Fixups[P].Address < EndAddress) then
              FixupsInside.Add(Fixups[P].Address);
          end;

          // if there is a fixupTo not to the start from outside the block, the blocks ends there.
          Adjusted := True;
          while Adjusted do
          begin
            Adjusted := False;
            for P := 0 to FixupsTo.Count -1 do
              if (PPChar(FixupsTo[P])^ <> StartAddress) and
                 ((PChar(FixupsTo[P]) < StartAddress) or
                  (PChar(FixupsTo[P]) >= EndAddress)) then
              begin
                {$IFOPT D+}
                  SendDebug(Format('EndAddress ajusted %p %p %p %p', [Pointer(StartAddress),
                    Pointer(EndAddress), PPointer(FixupsTo[P])^, FixupsTo[P]]));
                {$ENDIF}
                EndAddress := PPChar(FixupsTo[P])^;
                Adjusted := True;
                break;
              end;
            for P := FixupsTo.Count -1 downto 0 do
              if PPChar(FixupsTo[P])^ >= EndAddress then
                FixupsTo.Delete(P);
            for P := FixupsInside.Count -1 downto 0 do
              if PChar(FixupsInside[P]) >= EndAddress then
                FixupsInside.Delete(P);
          end;

          // Align4 the proc to 4 bytes.
          EndAddress := Align4(EndAddress);
          
          // Sort the lists of fiups.
          FixupsTo.Sort(ListPointerToSort);
          FixupsInside.Sort(ListSimpleSort);

          // At the moment all of the possebilities are possible except ptGUID (no Interfaces).
          PossType := [ptPChar, ptProc];

          // If there is no 0 on the last byte of a DWord before the first Fixup Inside or FixupTo then it is not a PChar.
          XAddress := EndAddress;

          for P := 0 to FixupsTo.Count -1 do
            if PPChar(FixupsTo[P])^ <> StartAddress then
            begin
              XAddress := PPChar(FixupsTo[P])^;
              break;
            end;

          if (FixupsInside.Count > 0) and (FixupsInside[0] < XAddress) then
            XAddress := FixupsInside[0];
          P := 3;
          PCharEndAddress := nil;
          PossType := PossType - [ptPChar];
          while P < XAddress - StartAddress do
          begin
            if StartAddress[P] = #0 then
            begin
              ALog := ALog + Format('PChar still possible due to a #0 at %p, ', [Pointer(StartAddress + P)]);
              PossType := PossType + [ptPChar];
              PCharEndAddress := StartAddress + P + 1;
              break;
            end;
            Inc(P, 4);
          end;

          // if there are fixups inside then it must be a proc.
          if (FixupsInside.Count <> 0) and
             ((PCharEndAddress = nil) or (PCharEndAddress > FixupsInside[0])) then
          begin
            ALog := ALog + Format('can only be proc due to a fixup inside at %p, ', [Pointer(FixupsInside[0])]);
            PCharEndAddress := nil;
            PossType := [ptProc];
          end;

          // If the block smaller than 16 untile the next fixup, it can't be a GUID
          if XAddress - StartAddress <> 16 then
            PossType := PossType - [ptGUID];

          // If the proc doesn't cotain a $C3 or $C2 instruction.
          PossType := PossType - [ptProc];
          for P := Align4(EndAddress-StartAddress) -4 to Align4(EndAddress-StartAddress)-1 do
            if StartAddress[P] = #$C3 then
            begin
              PossType := PossType + [ptProc];
              break;
            end;

          for P := Align4(EndAddress-StartAddress) -5 to Align4(EndAddress-StartAddress) -2 do
            if StartAddress[P] = #$C2 then
            begin
              PossType := PossType + [ptProc];
              break;
            end;

          // A PChar has a maximum Length of 255.
          if (EndAddress - StartAddress > 256) and
             ((PCharEndAddress <> nil) or (PCharEndAddress - StartAddress > 256)) then
          begin
            ALog := ALog + 'pchar not possible because the block is to long, ';
            PossType := PossType - [ptPChar];
            PCharEndAddress := nil;
          end;

          // The proc may not be greater then the empty block.
          if ptProc in PossType then
            ProcSize := GetProcSize(StartAddress)
          else
            ProcSize := 0;
            
          if (ptProc in PossType) and (EndAddress - StartAddress < ProcSize) then
            PossType := PossType - [ptProc];

          if PossType = [] then
          begin
            if (EndAddress - StartAddress > 4) and (StartAddress[0] = #$55) and
               (StartAddress[1] = #$8B) and (StartAddress[2] = #$EC) then
            begin
              PossType := [ptProc];
              {$IFOPT D+}
                SendDebug(Format('Nothing As proc gap %p %p', [Pointer(StartAddress), Pointer(EndAddress)]));
              {$ENDIF}
            end
            else
            begin
              // Search for the first Fixups to after then end, before the original end
              // it probably is a proc.
              XAddress := OrgEndAddress;
              for P := 0 to Fixups.Count -1 do
                if (PPChar(Fixups[P].Address)^ < XAddress) and
                   (PPChar(Fixups[P].Address)^ > StartAddress) then
                begin
                  XAddress := PPChar(Fixups[P].Address)^;
                end;
              if (XAddress < OrgEndAddress) and (Integer(XAddress) mod 4 <> 0) then
              begin
                PossType := [ptProc];
                {$IFOPT D+}
                  SendDebug(Format('Nothing As **proc gap %p %p', [Pointer(StartAddress), Pointer(EndAddress)]));
                {$ENDIF}
              end
              else
              begin
                PossType := [ptPChar];
                {$IFOPT D+}
                  SendDebug(Format('Nothing As PChar gap %p %p', [Pointer(StartAddress), Pointer(EndAddress)]));
                {$ENDIF}
              end;
            end;
          end;

          // If the difference between the pchar end address and the EndAddress of the proc, is
          // very big (say $40, if is probably a pchar followed by a proc.
          if (ptPChar in PossType) and (PCharEndAddress <> nil) and
             (ProcSize <> 0) and (ProcSize - (PCharEndAddress - StartAddress) > $40) then
            PossType := [ptPChar];

          // If a proc is still possible say it is a proc.
          if ptProc in PossType then
            PossType := [ptProc];
          // If a PChar is still possible say it is a pchar.
          if ptPChar in PossType then
            PossType := [ptPChar];

          // Create a TDecompItem to fill the block.
          if PossType = [ptProc] then
          begin
            Result := True;
            AProc := Procs.Add(StartAddress);
            AProc.Comments.AddComment(Format('Unknown block proc at %p to %p',
               [Pointer(StartAddress), Pointer(EndAddress)]), ctDebug);
            AProc.Comments.AddComment(ALog, ctDebug);
            AProc.PossProcTypes := ptAllStatic;
            AProc.Size := GetProcSize(StartAddress);
            AProc.ProcSize := AProc.Size;
            if AProc.Size > OrgEndAddress - StartAddress then
              {$IFOPT D+}SendDebugEx(Format('To big proc %p %x %p', [Pointer(StartAddress), AProc.Size,
                Pointer(EndAddress)]), mtError);{$ENDIF}
            Procs.AnalyzeProc(AProc);
            AppendProcs;
            AlignProcs;
            // A Proc maybe Free in Append Procs, so search for the proc at startaddress.
            AProc := Procs.FindProc(StartAddress);
            if AProc.Size = 0 then
              AProc := Procs.FindProc(StartAddress +1);
            EndAddress := AProc.Address + AProc.Size;
          end
          else
            if PossType = [ptPChar] then
            begin
              if PCharEndAddress <> nil then
                EndAddress := PCharEndAddress;
              with TStringInfo.Create(StringInfos, StartAddress, stPAnsiChar, EndAddress - StartAddress) do
              begin
                Comments.AddComment(ALog, ctDebug);
                Name := Format('PChar%p', [Pointer(StartAddress)]);
              end;
            end
            else
              with TGUIDConst.Create(miscs) do
              begin
                Comments.AddComment(ALog, ctDebug);
                Address := StartAddress;
                Size := EndAddress - StartAddress;
              end;

          StartAddress := EndAddress;
        finally
          FixupsInside.Free;
        end;
      finally
        FixupsTo.Free;
      end;
    end;

  begin
    CodeItems := TList.Create;
    try
      ReDo:
      DecompThread.CheckTerminated;
      LastDC := -1;
      {$IFOPT D+}
      SendDebug('Start Analysing unknown blocks');
      DbugCounter := 0;
      {$ENDIF}

      MaxAddress := Units.SysInitUnit.FInit.Address + Units.SysInitUnit.FInit.Size;
      // Add all code items to the CodeItems list.
      CodeItems.Clear;
      CodeItems.Capacity := Classes.Count + TypeInfos.Count + Procs.Count + StringInfos.Count;
      for I := 0 to Classes.Count -1 do
        if (Classes[I].Address >= MaxAddress) then
          CodeItems.Add(Classes[I]);
      for I := 0 to TypeInfos.Count -1 do
        if (TypeInfos[I].Address >= MaxAddress) then
          CodeItems.Add(TypeInfos[I]);
      for I := 0 to Procs.Count -1 do
        if (Procs[I].Size <> 0) and (Procs[I].Address <> nil) and
           (Procs[I].Address >= MaxAddress) then
          CodeItems.Add(Procs[I]);
      for I := 0 to StringInfos.Count -1 do
        if (StringInfos[I].Address < Data) and
           (StringInfos[I].Address >= MaxAddress) then
          CodeItems.Add(StringInfos[I]);
      for I := 0 to Miscs.Count -1 do
        if (Miscs[I].Size <> 0) and
           (Miscs[I].Address >= MaxAddress) then
          CodeItems.Add(Miscs[I]);

      CodeItems.Sort(DecompItemSort);

      for I := 0 to CodeItems.Count -1 do
      begin
        if TDecompItem(CodeItems[I]).Address < MaxAddress then
        begin
          if TDecompItem(CodeItems[I]).Address + TDecompItem(CodeItems[I]).Size <= MaxAddress then
          begin
            // There is a decomp item inside another decomp item.
            if (TDecompItem(CodeItems[I]) is TProc) or
               ((TDecompItem(CodeItems[I]) is TStringInfo) and
                (TStringInfo(CodeItems[I]).StringType = stPAnsiChar)) then
            begin
              // If it is a proc or PChar inside the other decomp item, Delete it.
              TDecompItem(CodeItems[I]).Free;
              CodeItems[I] := nil;
              Continue;
            end
            else
            begin
              {$IFOPT D+}
                if LastDC <> -1 then
                  SendDebugEx(Format('Not proc block inside another block %s %p %x, %s %p',
                     [TDecompItem(CodeItems[LastDC]).ClassName, Pointer(TDecompItem(CodeItems[LastDC]).Address), TDecompItem(CodeItems[LastDC]).Size,
                      TDecompItem(CodeItems[I]).ClassName, Pointer(TDecompItem(CodeItems[I]).Address)]), mtError)
                else
                  SendDebugEx(Format('Not proc block inside another block %s %p',
                     [TDecompItem(CodeItems[I]).ClassName, Pointer(TDecompItem(CodeItems[I]).Address)]), mtError)
              {$ELSE}
                raise EDecompilerError.CreateFmt('Not proc block inside another block %p %p',
                   [Pointer(MaxAddress), Pointer(TDecompItem(CodeItems[I]).Address)]);
              {$ENDIF}
            end;
          end
          else
          begin
            // There is a decomp item inside another decomp item.
            if (LastDC <> -1) and
               ((TDecompItem(CodeItems[LastDC]) is TProc) or
                ((TDecompItem(CodeItems[LastDC]) is TStringInfo) and
                 (TStringInfo(TDecompItem(CodeItems[LastDC])).StringType = stPAnsiChar))) then
            begin
              // If it is a proc or PChar inside the other decomp item, Delete it.
              TDecompItem(CodeItems[LastDC]).Free;
              CodeItems[LastDC] := nil;
              LastDC := I;
              Continue;
            end
            else
            begin
            {$IFOPT D+}
              if LastDC <> -1 then
                SendDebugEx(Format('Overlapping code items %s %p %x, %s %p',
                   [TDecompItem(CodeItems[LastDC]).ClassName, Pointer(TDecompItem(CodeItems[LastDC]).Address), TDecompItem(CodeItems[LastDC]).Size,
                    TDecompItem(CodeItems[I]).ClassName, Pointer(TDecompItem(CodeItems[I]).Address)]), mtError)
              else
                SendDebugEx(Format('Overlapping code items, %s %p',
                   [TDecompItem(CodeItems[I]).ClassName, Pointer(TDecompItem(CodeItems[I]).Address)]), mtError);
            {$ELSE}
              raise EDecompilerError.CreateFmt('Overlapping code items %p %p',
                [Pointer(MaxAddress), Pointer(TDecompItem(CodeItems[I]).Address)]);
            {$ENDIF}
            end;
          end;
        end;

        while MaxAddress < TDecompItem(CodeItems[I]).Address do
        begin
          {$IFOPT D+`}Inc(DbugCounter);{$ENDIF}
          if AnaUnknownBlock(MaxAddress, TDecompItem(CodeItems[I]).Address) then
          begin
            goto ReDo;
          end;
          if MaxAddress > TDecompItem(CodeItems[I]).Address then
          begin
            {$IFOPT D+}SendDebug('Max higher than lowest ' + IntToStr(DbugCounter));{$ENDIF}
            goto ReDo;
          end;
        end;

        Assert(TDecompItem(CodeItems[I]).Address = MaxAddress, 'Lowest address must be Max address');

        LastDC := I;
        // Save the end address of the lowest code item as max address.
        MaxAddress := TDecompItem(CodeItems[I]).Address + TDecompItem(CodeItems[I]).Size;
      end;
    finally
      CodeItems.Free;
    end;
  end;

  procedure LoadUnitNames;
  var
    I: Integer;
  begin
    // The first unit in the list is always SysInit and the second System
    Units.SysInitUnit.Name := 'SysInit';
    if Units.SystemUnit <> nil then
      Units.SystemUnit.Name := 'System';

    // Extract names from Classes.
    for I := 0 to Classes.Count -1 do
      if Classes[I].AClass.ClassInfo <> nil then
        (Classes[I].AUnit as TUnit).Name := GetTypeData(Classes[I].AClass.ClassInfo)^.UnitName;
    // Extract names from Interfaces.
    for I := 0 to TypeInfos.Count -1 do
      if TypeInfos[I].TypeInfo^.Kind = tkInterface then
        (TypeInfos[I].AUnit as TUnit).Name := GetTypeData(TypeInfos[I].TypeInfo)^.IntfUnit;
  end;

  procedure LoadImportedProcs;
  var
    I, J, K, L: Integer;
    Proc: TProc;
    NMInfo: TNameManglingInfo;
  label
    NextFixup;
  begin
    // Imported procs can be identefied by a hard coded pointer to a imported proc address
    // Loop though all the fixups which are pointers to somewhere inside the
    // Data Section (where the imported procs addresses are).
    for I := 0 to Fixups.Count -1 do
      if (Fixups[I].FixupType = 3) and
         (PPChar(Fixups[I].Address)^ >= Data) then
      begin
        // Search for imported proc address matching the fixup address.
        for J := 0 to High(Imports) do
          with Imports[J] do
          begin
            for K := 0 to High(Entries) do
            begin
              if Entries[K].PAddress = PPChar(Fixups[I].Address)^ then
              begin
                if IsPackage(Imports[J].DLLName) then
                begin
                  // Entry must a a proc.
                  NMInfo := GetNameManglingInfo(Entries[K].Name);
                  if not (NMInfo.NMType in [eitProc, eitMethod]) then
                    Continue;
                end;

                Proc := Procs.FindProc(Fixups[I].Address -2);
                if Proc <> nil then
                begin
                  // Ignore this imported proc if this imported unit initialization of finalization.
                  if (UsePackages) and
                     ((Proc.PossProcTypes = [ptInitialization]) or
                      (Proc.PossProcTypes = [ptFinalization])) then
                  begin
                    with Proc.ImportInfo do
                    begin
                      Imported := True;
                      DLLName := Imports[J].DLLName;
                      Entry := Entries[K];
                    end;
                    Continue;
                  end;
                  raise EDecompilerError.CreateFmt('Imported procedure already found %p', [Pointer(Fixups[I].Address -1)]);
                end;
                // Fixups matched, then this is a jmp [Entries].
                Proc := Procs.Add(Fixups[I].Address -2);
                with Proc do
                begin
                  Comments.AddComment('Imported proc', ctDebug);
                  Comments.AddComment('Imported proc name: ' + Entries[K].Name, ctDebug);
                  AUnit := Units.FindInUnitUsingFInit(Fixups[I].Address -2);
                  // Save all those imported stuff.
                  with ImportInfo do
                  begin
                    Imported := True;
                    DLLName := Imports[J].DLLName;
                    Entry := Entries[K];
                  end;
                  Size := 8;
                  AppendBefore := atMayNot;
                  AppendAfter := atMayNot;
                  if IsPackage(Imports[J].DLLName) then
                  begin
                    // Set the package import info
                    Name := NMInfo.ItemName;
                    TUnit(AUnit).Name := NMInfo.UnitName;
                    // Search for the imported proc, in other pefiles.
                    for L := 0 to DecompThread.PEFileClassCount -1 do
                      if AnsiCompareText(DecompThread.PEFileClasses[L].ProjectName + '.bpl', DLLName) = 0 then
                      begin
                        with ImportInfo do
                          ImportedProc := TPEFileClass(DecompThread.PEFileClasses[L]).Procs.FindProc(PPChar(Entries[K].PAddress)^);
                        PossProcTypes := ImportInfo.ImportedProc.PossProcTypes;
                        AClass := ImportInfo.ImportedProc.AClass;
                        Break;
                      end;
                    if ImportInfo.ImportedProc = nil then
                      raise EDecompilerError.CreateFmt('Imported package proc not found', []);
                  end
                  else
                  begin
                    PossProcTypes := [ptProcedure];
                  end;
                end;
                goto NextFixup;
              end;
            end;
          end;
        NextFixup:
      end;
  end;

{  procedure LoadStrings;
  var
    I, J: Integer;
  label
    Next;
  begin
    for I := 0 to Fixups.Count -1 do
      with Fixups[I] do
        if (FixupType = 3) and
           (PPChar(Address)^ < Code + CodeSize) and
           (PPChar(Address)^ >= Code +8) then
        begin
          if (PDWord(PPChar(Address)^ -8)^ = $FFFFFFFF) and
             (PPChar(Address)^ + PDWord(PPChar(Address)^ - 4)^ < Code + CodeSize) and
             (PPChar(Address)^ + PDWord(PPChar(Address)^ - 4)^ >= Code) and
             ((PPChar(Address)^ + PDWord(PPChar(Address)^ -4)^)[0] = #0) and
             (StringInfos.FindString(PPChar(Fixups[I].Address)^ -8) = nil) then
          begin
            for J := 0 to PDWord(PPChar(Address)^ -4)^ -1 do
              if PPChar(Address)^[J] = #0 then
                goto Next;
            with TStringInfo.Create(StringInfos, PPChar(Fixups[I].Address)^ -8, stAnsiString) do
              Name := Format('String%p', [Pointer(PPChar(Fixups[I].Address)^)]);
          end;
        Next:
        end;
  end;}

  procedure LoadResStrings;
  var
    I: Integer;
    MinIdentifier: DWord;
    ModuleAddress: PChar;
  begin
    // ResString is a block of 2 DWord, the first containing a pointer to a Module,
    // (so this must be a fixup to a BSS location) and the second containing the
    // identifier, a value equal or below $0000FFFF.

    // Get the count of the res string.
    MinIdentifier := $00010000;
    for I := 0 to High(Resources) do
      if (Resources[I].NameOrID = niID) and (Resources[I].ID = 6) then
      begin
        MinIdentifier := $00010000 - 16 - High(Resources[I].Entries) * 16;
        break;
      end;

    // The first pointer points to the var HInstance.
    I := VarInfos.IndexOfName('HInstance');
    if I <> -1 then
      ModuleAddress := VarInfos[I].Address
    else
      ModuleAddress := nil;

    for I := 0 to Fixups.Count -1 do
      with Fixups[I] do
        if (FixupType = 3) and
           ((ModuleAddress = nil) or (PPChar(Address)^ = ModuleAddress)) and
           (PDWord(Address + 4)^ <= $0000FFFF) and
           (PDWord(Address + 4)^ >= MinIdentifier) and
           (Integer(Address) mod 4 = 0) then
        begin
          // If there isn't a module address already this address must be HInstance
          if ModuleAddress = nil then
          begin
            VarInfos.LoadVar(PPChar(Address)^, 'HInstance', Units.SystemUnit);
            ModuleAddress := PPChar(Address)^;
          end;
          // Creae the res string.
          with TStringInfo.Create(StringInfos, Address, stResourceString) do
            Name := Format('ResString%p', [Pointer(Address)]);
        end;
  end;

  procedure LoadVirtualMethods;
  var
    I, J: Integer;
    ParentClassCount: Integer;
    Address: PChar;
    AAClass: TClass;
    Proc: TProc;
  const
    StdVirtualMethodNames: array[-8..-1] of string =
       ('SafeCallException', 'AfterConstruction', 'BeforeDestruction',
        'Dispatch', 'DefaultHandler', 'NewInstance', 'FreeInstance', 'Destroy');
    StdVirtualMethodParams: array[-8..-1] of string =
       ('ExceptObject: TObject; ExceptAddr: Pointer', '', '',
        'var Message', 'var Message', '', '', '');
    StdVirtualMethodResults: array[-8..-1] of string =
       ('HResult', '', '', '', '', 'TObject', '', '');
    StdVirtualMethodTypes: array[-8..-1] of TProcTypes =
      ([ptMethodProcedure], [ptMethodProcedure], [ptMethodProcedure],
       [ptMethodProcedure], [ptMethodProcedure], [ptClassProcedure],
       [ptMethodProcedure], [ptDestructor]);
  begin
      for I := 0 to Classes.Count -1 do
      begin
        AAClass := Classes[I].AClass;
        if AAClass.ClassParent <> nil then
          ParentClassCount := GetVirtualMethodCount(AAClass.ClassParent)
        else
          ParentClassCount := -9;
        // J = -8 to start with the virtual Methods in the VMT.
        for J := -8 to GetVirtualMethodCount(AAClass) -1 do
        begin
          Address := GetVirtualMethod(AAClass, J);
          if not ((J < ParentClassCount) and (GetVirtualMethod(AAClass.ClassParent, J) = Address)) then
          begin
            // Compare the Method with AbstractError procedure.
            if (Units.SystemUnit.FindProcByName(AbstractErrorProcName)<> nil) and
               (Units.SystemUnit.FindProcByName(AbstractErrorProcName).Address = Address) then
              // Set the address to nil to indicate that it is a abstract method.
              Address := nil;
            Proc := Procs.Add(Address);
            with Proc do
            begin
              Comments.AddComment('Virtual method', ctDebug);
              PossProcTypes := PossProcTypes * ptMethods;
              AClass := Classes[I];
              MethodBindingType := mbtVirtual;
              if Address = nil then
                AddReq(Units.SystemUnit.FindProcByName(AbstractErrorProcName), nil);
              MethodIndex := J;
              if J < ParentClassCount then
                Overrides := True;
              if J < 0 then
              begin
                Name := StdVirtualMethodNames[J];
                Parameters.Parameters := StdVirtualMethodParams[J];
                Parameters.FuncResult := StdVirtualMethodResults[J];
                PossProcTypes := StdVirtualMethodTypes[J];
              end;
            end;
          end;
        end;
      end;
  end;

  procedure LoadDynamicMethods;
  var
    DynamicIndexList: PDynamicIndexList;
    DynamicAddressList: PDynamicAddressList;
    I, J: Integer;
    Proc: TProc;
    AAClass: TClass;
    Address: PChar;
  resourcestring
    SDynamicMethodInsideAnother = 'DynamicMethod at %p is inside another method at %p';
  begin
      for I := 0 to Classes.Count -1 do
      begin
        AAClass := Classes[I].AClass;
        if GetDynamicMethodCount(AAClass) <> 0 then
        begin
          DynamicIndexList := GetDynamicIndexList(AAClass);
          DynamicAddressList := GetDynamicAddressList(AAClass);
          for J := 0 to GetDynamicMethodCount(AAClass) -1 do
          begin
            Address := DynamicAddressList^[J];
            // Compare the Method with AbstractError procedure.
            if (Units.SystemUnit.FindProcByName(AbstractErrorProcName) <> nil) and
               (Units.SystemUnit.FindProcByName(AbstractErrorProcName).Address = Address) then
              // Set the address to nil to indicate that it is a abstract method.
              Address := nil;
            // Search for an already existing proc.
            if (Address <> nil) then
              Proc := Procs.FindProc(Address)
            else
              Proc := nil;
            // Create a new proc if there isn't one yet.
            if Proc = nil then
              Proc := Procs.Add(Address);
            // Address of the proc and found proc must be the same.
            if Proc.Address <> Address then
              raise EDecompilerError.CreateFmt(SDynamicMethodInsideAnother, [Pointer(Address), Pointer(Proc.Address)]);

            with Proc do
            begin
              Comments.AddComment('Dynamic method', ctDebug);
              PossProcTypes := PossProcTypes * ptMethods;
              MethodIndex := DynamicIndexList^[J];
              AClass := Classes[I];
              MethodBindingType := mbtDynamic;
              Overrides := HasDynamicMethod(AAClass.ClassParent, MethodIndex);
              if Address = nil then
                AddReq(Units.SystemUnit.FindProcByName(AbstractErrorProcName), nil);
            end;
          end;
        end;
      end;
  end;

  procedure SetClassesTypeInfosUnits;
  var
    I: Integer;
  begin
    // Loop all the classes.
    for I := 0 to Classes.Count -1 do
    begin
      // Find the unit in which the class exists, and add the class to it and
      // set the class unit to the unit.
      Classes[I].AUnit := Units.FindInUnitUsingFInit(PChar(Classes[I].AClass));
    end;

    // Loop all the TypeInfos
    for I := 0 to TypeInfos.Count -1 do
    begin
      // Find the unit in which the class exists, and add the typeinfo to it and
      // set the typinfo unit to the unit.
      TypeInfos[I].AUnit := Units.FindInUnitUsingFInit(PChar(TypeInfos[I].TypeInfo));
    end;
  end;

  procedure LoadSystemImportedProcNames;
  var
    I: Integer;
    Str: string;
  begin
    // Don't do this if the system proc is imported.
    if UsePackages then
      Exit;
    with Units.SystemUnit.DecompItems do
      for I := 0 to Count -1 do
      if (TDecompItem(Items[I]) is TProc) and TProc(Items[I]).ImportInfo.Imported then
      begin
        // Found an imported proc. If it ends with a A give it the name without the A.
        Str := TProc(Items[I]).ImportInfo.Entry.Name;
        if Str[Length(Str)] = 'A' then
          SetLength(Str, Length(Str) -1);
        TProc(Items[I]).Name := Str;
      end;
    with Units.SysInitUnit.DecompItems do
      for I := 0 to Count -1 do
      if (TDecompItem(Items[I]) is TProc) and TProc(Items[I]).ImportInfo.Imported then
      begin
        // Found an imported proc. If it ends with a A give it the name without the A.
        Str := TProc(Items[I]).ImportInfo.Entry.Name;
        if Str[Length(Str)] = 'A' then
          SetLength(Str, Length(Str) -1);
        TProc(Items[I]).Name := Str;
      end;
  end;

// ----------------------------------------------------------------------------

  procedure LoadSystemObjFiles;

    procedure AnaObjFile(ObjSig: array of Byte; ObjSigO, ObjSize: Integer; HCProcs: array of THCProc); overload;
    var
      AAddress: PChar;
      I: Integer;
    begin
      AAddress := Code;
      while AAddress <= Units.SystemUnit.FInit.Address - ObjSize do
      begin
        if CompareMem(@ObjSig, AAddress + ObjSigO, SizeOf(ObjSig)) then
        begin
          // Create a block for the Obj File
          with TDecompItem.Create(Miscs) do
          begin
            Comments.AddComment('Obj file filler', ctDebug);
            Address := AAddress;
            Size := ObjSize;
            AUnit := Units.SystemUnit;
            for I := Low(HCProcs) to High(HCProcs) do
              // Create the readln procedure.
              with Procs.Add(AAddress + HCProcs[I].R) do
              begin
                Comments.AddComment('Obj file proc', ctDebug);
                AppendAfter := atMayNot;
                AppendBefore := atMayNot;
                Name := HCProcs[I].N;
                PossProcTypes := [ptProcedure];
                AUnit := Units.SystemUnit;
             end;
         end;
         // exit the search
         break;
       end;
       Inc(AAddress, 4);
      end;
    end;

    procedure AnaObjFile(ObjSigA: array of Byte; ObjSigAO, ObjSize: Integer; HCProcs: array of THCProc;
      ObjSigB: array of Byte; ObjSigBO: Integer); overload;
    var
      AAddress: PChar;
      I: Integer;
    begin
      AAddress := Code;
      while AAddress <= Units.SystemUnit.FInit.Address - ObjSize do
      begin
        if CompareMem(@ObjSigA, AAddress + ObjSigAO, SizeOf(ObjSigA)) and
           CompareMem(@ObjSigB, AAddress + ObjSigBO, SizeOf(ObjSigB)) then
        begin
          // Create a block for the Obj File
          with TDecompItem.Create(Miscs) do
          begin
            Comments.AddComment('Obj file filler', ctDebug);
            Address := AAddress;
            Size := ObjSize;
            AUnit := Units.SystemUnit;
            for I := Low(HCProcs) to High(HCProcs) do
              // Create the readln procedure.
              with Procs.Add(AAddress + HCProcs[I].R) do
              begin
                Comments.AddComment('Obj file proc', ctDebug);
                AppendAfter := atMayNot;
                AppendBefore := atMayNot;
                Name := HCProcs[I].N;
                PossProcTypes := [ptProcedure];
                AUnit := Units.SystemUnit;
             end;
         end;
         // exit the search
         break;
       end;
       Inc(AAddress, 4);
      end;
    end;

    procedure AnaConst(ConstSig: array of Byte; ConstName: string);
    var
      AAddress: PChar;
    begin
      AAddress := Code;
      while AAddress <= Units.SystemUnit.FInit.Address - SizeOf(ConstSig)do
      begin
        if CompareMem(@ConstSig, AAddress, SizeOf(ConstSig)) then
        begin
          // Create a block for the const File
          with TStringInfo.Create(StringInfos, AAddress, stPAnsiChar) do
          begin
            Name := ConstName;
            Size := SizeOf(ConstSig);
            AUnit := Units.SystemUnit;
          end;
          // exit the search
          break;
        end;
        Inc(AAddress, 4);
      end;
    end;

    procedure AnaHCBlock(HCBlock: PHCBlock);
    var
      AAddress: PChar;

      function IsBlock: Boolean;
      var
        I: Integer;
        PR: TPRoc;
      begin
        Result := False;
        // Compare the Val blocks
        for I := 0 to HCBlock^.ValC -1 do
        begin
          if not CompareMem(AAddress + PValArray(HCBlock^.Val)^[I].RA,
            PValArray(HCBlock^.Val)^[I].Val, PValArray(HCBlock^.Val)^[I].Count) then exit;
        end;
        // Compare the Internal RA.
        for I := 0 to HCBlock^.IAC -1 do
          if PPChar(AAddress + TIAArray(HCBlock^.IA^)[I].RA)^ <> AAddress + TIAArray(HCBlock^.IA^)[I].RRA then
            exit;
        // Compare the procs.
        for I := 0 to HCBlock^.PRC -1 do
        begin
          PR := FindDecompItemByRef(AAddress + TPRArray(HCBlock^.PR^)[I].RA + 4 + PInteger(AAddress + TPRArray(HCBlock^.PR^)[I].RA)^) as TPRoc;
          if (PR = nil) or (PR.Name <> TPRArray(HCBlock^.PR^)[I].PN) then exit;
          if (TPRArray(HCBlock^.PR^)[I].CN = '') and (PR.AClass <> nil) then exit;
          if (TPRArray(HCBlock^.PR^)[I].CN <> '') and
             ((PR.AClass = nil) or (PR.AClass.AClass.ClassName <> TPRArray(HCBlock^.PR^)[I].CN)) then exit;
        end;
        // If there was no exit the this is the block.
        Result := True;
      end;

    var
      I: Integer;
      AProc: TProc;
      DC: TDecompItem;
    begin
      // Search the address at which the block starts.
      AAddress := Code;
      DC := FindDecompItemByBlock(AAddress);
      while DC <> nil do
      begin
        AAddress := Align4(DC.Address + DC.Size);
        DC := FindDecompItemByBlock(AAddress);
      end;
      while IsBlock = False do
      begin
        Inc(AAddress, 4);
        DC := FindDecompItemByBlock(AAddress);
        while DC <> nil do
        begin
          AAddress := Align4(DC.Address + DC.Size);
          DC := FindDecompItemByBlock(AAddress);
        end;
        // The block must be before the sysinit finalization section.
        if AAddress + HCBlock.Size > Units.SysInitUnit.FInit.Address then
          exit;
      end;
      {$IFOPT D+}
        SendDebug(Format('Found proc %s, %p', [TRAArray(HCBlock^.RA^)[0].PN, Pointer(AAddress)]));
      {$ENDIF}
      // Create a memory filler.
      with TDecompItem.Create(miscs) do
      begin
        Comments.AddComment('HC proc filler', ctDebug);
        Address := AAddress;
        Size := HCBlock^.Size;
      end;
      // Create the procs.
      for I := 0 to HCBlock^.RAC -1 do
        with TRAArray(HCBlock^.RA^)[I] do
        begin
          AProc := Procs.Add(AAddress + RA);
          with AProc do
          begin
            Comments.AddComment('system proc', ctDebug);
            AppendAfter := atMayNot;
            AppendBefore := atMayNot;
            Name := PN;
            Size := 0;
            if (PT = ptProcedure) and (CN <> '') then
              PossProcTypes := [ptMethodProcedure]
            else
              PossProcTypes := [PT];
            AUnit := Units.FindInUnitUsingFInit(AAddress);
            if CN <> '' then
              AClass := Classes.FindClassByName(CN);
            if PV <> nil then
              PV^ := AProc;
          end;
        end;
      // Create the vars.
      for I := 0 to HCBlock^.VarsC -1 do
        with TVarArray(HCBlock^.Vars^)[I] do
          VarInfos.loadVar(PPChar(AAddress + RA)^ - VRA, VN, Units.SystemUnit);
    end;

    procedure AnaSysUtilsHCBlock(HCBlock: PHCBlock);
    var
      AAddress: PChar;

      function IsBlock: Boolean;
      var
        I: Integer;
        PR: TPRoc;
      begin
        Result := False;
        // Compare the Val blocks
        for I := 0 to HCBlock^.ValC -1 do
          if not CompareMem(AAddress + TValArray(HCBlock^.Val^)[I].RA,
            TValArray(HCBlock^.Val^)[I].Val, TValArray(HCBlock^.Val^)[I].Count) then exit;
        // Compare the Internal RA.
        for I := 0 to HCBlock^.IAC -1 do
          if PPChar(AAddress + TIAArray(HCBlock^.IA^)[I].RA)^ <> AAddress + TIAArray(HCBlock^.IA^)[I].RRA then
            exit;
        // Compare the procs.
        for I := 0 to HCBlock^.PRC -1 do
        begin
          PR := FindDecompItemByRef(AAddress + TPRArray(HCBlock^.PR^)[I].RA + 4 + PInteger(AAddress + TPRArray(HCBlock^.PR^)[I].RA)^) as TPRoc;
          if (PR = nil) or (PR.Name <> TPRArray(HCBlock^.PR^)[I].PN) then exit;
          if (TPRArray(HCBlock^.PR^)[I].CN = '') and (PR.AClass <> nil) then exit;
          if (TPRArray(HCBlock^.PR^)[I].CN <> '') and
             ((PR.AClass = nil) or (PR.AClass.AClass.ClassName <> TPRArray(HCBlock^.PR^)[I].CN)) then exit;
        end;
        // If there was no exit the this is the block.
        Result := True;
      end;

    var
      I: Integer;
      AProc: TProc;
    begin
      // Search the address at which the block starts.
      AAddress := Units.SysInitUnit.FInit.Address;
      while IsBlock = False do
      begin
        Inc(AAddress, 4);
        // The block must be before the sysinit finalization section.
        if AAddress + HCBlock.Size > Units[Units.Count -1].FInit.Address then
          exit;
      end;
      {$IFOPT D+}
        SendDebug(Format('Found proc %s, %p', [TRAArray(HCBlock^.RA^)[0].PN, Pointer(AAddress)]));
      {$ENDIF}
      HasFFMTObj := True;
      // Create a memory filler.
      with TDecompItem.Create(miscs) do
      begin
        Comments.AddComment('System log, HCBlock filler', ctDebug);
        Address := AAddress;
        Size := HCBlock^.Size;
      end;
      // Create the procs.
      for I := 0 to HCBlock^.RAC -1 do
        with TRAArray(HCBlock^.RA^)[I] do
        begin
          AProc := Procs.Add(AAddress + RA);
          with AProc do
          begin
            Comments.AddComment('System proc', ctDebug);
            AppendAfter := atMayNot;
            AppendBefore := atMayNot;
            Name := PN;
            Size := 0;
            PossProcTypes := [ptProcedure];
            AUnit := Units.FindInUnitUsingFInit(AAddress);
            Instr.Text.Text := Source;
            TUnit(AUnit).Name := 'Sysutils';
            if PV <> nil then
              PV^ := AProc;
          end;
        end;
      // Create the vars.
      for I := 0 to HCBlock^.VarsC -1 do
        with TVarArray(HCBlock^.Vars^)[I] do
          VarInfos.loadVar(PPChar(AAddress + RA)^ - VRA, VN, Units.FindInUnitUsingFInit(AAddress));
      // Change the Fixups address to prevent a empty req add error.
      AAddress[$73E] := Chr($18);
    end;

  var
    I: Integer;
  begin
    if Units.SystemUnit <> nil then
    begin
      // Increment the size of the initialization proc
      Units.SystemUnit.Init.Size := Units.SystemUnit.Init.Size + 4;

      // Load the var Output (can only happen in the system finalization proc.
      VarInfos.loadVar(PPChar(Units.SystemUnit.FInit.Address + $24)^, 'Output', Units.SystemUnit);
    end;

    for I := 0 to High(HCBlocks) do
      AnaHCBlock(HCBlocks[I]);

    if not UsePackages then
    begin
      // Search for Assign Object File (don't complain about the ugly search method).
      AnaObjFile(AssignObj, Low(AssignObj), AssignObjSize, AssignProcs);
      AnaObjFile(CloseObj, Low(CloseObj), CloseObjSize, CloseProcs);
      AnaObjFile(OpenTextObj, Low(OpenTextObj), OpenTextObjSize, OpenTextProcs);
      AnaObjFile(BlockReaObj, Low(BlockReaObj), BlockReaObjSize, BlockReaProcs);
      AnaObjFile(BlockwriObj, Low(BlockwriObj), BlockwriObjSize, BlockwriProcs);
      AnaObjFile(eoffileObj, Low(eoffileObj), eoffileObjSize, eoffileProcs);
      AnaObjFile(eoftextObj, Low(eoftextObj), eoftextObjSize, eoftextProcs);
      AnaObjFile(eolnObj, Low(eolnObj), eolnObjSize, eolnProcs);
      AnaObjFile(ereaseObj, Low(ereaseObj), ereaseObjSize, ereaseProcs);
      AnaObjFile(fdivObj, Low(fdivObj), fdivObjSize, fdivProcs);
      AnaObjFile(fileposObj, Low(fileposObj), fileposObjSize, fileposProcs);
      AnaObjFile(filesizeObj, Low(filesizeObj), filesizeObjSize, filesizeProcs);
      AnaObjFile(readrecObj, Low(readrecObj), readrecObjSize, readrecProcs);
      AnaObjFile(readcharObj, Low(readcharObj), readcharObjSize, readcharProcs);
      AnaObjFile(readlongObjA, Low(readlongObjA), readlongObjSize, readlongProcs, readlongObjB, Low(readlongObjB));
      AnaObjFile(readstrObj, Low(readstrObj), readstrObjSize, readstrProcs);
      AnaObjFile(readextObj, Low(readextObj), readextObjSize, readextProcs);
      AnaObjFile(readlnObj, Low(readlnObj), readlnObjSize, readlnProcs);
      AnaObjFile(renameObj, Low(renameObj), renameObjSize, renameProcs);
      AnaObjFile(openfileObj, Low(openfileObj), openfileObjSize, openfileProcs);
      AnaObjFile(seekObj, Low(seekObj), seekObjSize, seekProcs);
      AnaObjFile(seekeofObj, Low(seekeofObj), seekeofObjSize, seekeofProcs);
      AnaObjFile(seekeolnObj, Low(seekeolnObj), seekeolnObjSize, seekeolnProcs);
      AnaObjFile(settextbObj, Low(settextbObj), settextbObjSize, settextbProcs);
      AnaObjFile(truncateObj, Low(truncateObj), truncateObjSize, truncateProcs);
      AnaObjFile(strextObj, Low(strextObj), strextObjSize, strextProcs);
      AnaObjFile(writerecObj, Low(writerecObj), writerecObjSize, writerecProcs);
      AnaObjFile(pow10Obj, Low(pow10Obj), pow10ObjSize, pow10Procs);
      AnaObjFile(valextObj, Low(valextObj), valextObjSize, valextProcs);
      AnaObjFile(real2extObj, Low(real2extObj), real2extObjSize, real2extProcs);
      AnaObjFile(ext2realObj, Low(ext2realObj), ext2realObjSize, ext2realProcs);
      AnaObjFile(_llObj, Low(_llObj), _llObjSize, _llProcs);
      AnaObjFile(readint64ObjA, Low(readint64ObjA), readint64ObjSize, readint64Procs, readint64objB, low(readint64ObjB));

      AnaConst(OldLocaleOverrideKeyConstSig, OldLocaleOverrideKeyConstName);
      AnaConst(NewLocaleOverrideKeyConstSig, NewLocaleOverrideKeyConstName);

      VarInfos.LoadVar(PPChar(Units.SystemUnit.Init.Address + $D6)^, 'MainThreadID', Units.SystemUnit);
      VarInfos.LoadVar(PPChar(Units.SystemUnit.Init.Address + $C2)^, 'CmdLine', Units.SystemUnit);
      VarInfos.LoadVar(PPChar(Units.SystemUnit.Init.Address + $CC)^, 'CmdShow', Units.SystemUnit);
      VarInfos.LoadVar(PPChar(Units.SystemUnit.Init.Address + $2F)^, 'RandSeed', Units.SystemUnit);
      VarInfos.LoadVar(PPChar(Units.SystemUnit.Init.Address + $9F)^, 'Input', Units.SystemUnit);
      VarInfos.LoadVar(PPChar(Units.SystemUnit.Init.Address + $6F)^, 'ClearAnyProc', Units.SystemUnit);
      VarInfos.LoadVar(PPChar(Units.SystemUnit.Init.Address + $79)^, 'ChangeAnyProc', Units.SystemUnit);
      VarInfos.LoadVar(PPChar(Units.SystemUnit.Init.Address + $83)^, 'RefAnyProc', Units.SystemUnit);
      VarInfos.LoadVar(PPChar(Units.SystemUnit.Init.Address + $4B)^, 'Unassigned', Units.SystemUnit);
      VarInfos.LoadVar(PPChar(Units.SystemUnit.Init.Address + $54)^, 'Null', Units.SystemUnit);
      VarInfos.LoadVar(PPChar(Units.SystemUnit.Init.Address + $5D)^, 'EmptyParam', Units.SystemUnit);
    end;

    if Self.IsConsole then
      VarInfos.LoadVar(PPChar(EntryPoint + 7)^, 'IsConsole', Units.SystemUnit);

    // SysUtils
    AnaSysUtilsHCBlock(@SysUtilsHCBlock);
  end;

  procedure GetEntryPointProc;
  var
    AAddress: PChar;
    Halt0Address: PChar;
    InitExeAddress: PChar;
    ASize: Integer;
    Proc: TProc;
  const
    InitProcName: array[TProjectType] of string = ('@InitExe', '@InitLib', 'Error package doesn''t have init');
  resourcestring
    {$IFDEF VER120}
    SHalt0NotFound =' @Halt0 Proc doesn''t exist (this probably isn''t a Delphi 4 program.)';
    {$ELSE} {$IFDEF VER130}
    SHalt0NotFound =' @Halt0 Proc doesn''t exist (this probably isn''t a Delphi 5 program.)';
    {$ENDIF} {$ENDIF}
  begin
    // Package Entrypoint first have a jump to somewhere what does the some work,
    // but you can't add normal code to it.
    if ProjectType = ptPackage then
    begin
      // Get the target address of the jump and its size.
      with TDisAsm.Create do
      try
        Param := @AAddress;
        OnJumpInstr := SaveJumpAddress;
        GetInstruction(EntryPoint, ASize);
      finally
        Free;
      end;
      // Creaet a filler for the jump.
      with TDecompItem.Create(Miscs) do
      begin
        Address := EntryPoint;
        Size := Align4(ASize);
        Comments.AddComment('Package Entry point filler', ctDebug);
      end;

      // Create the real entry point proc.
      EntryPointProc := Procs.Add(AAddress);
      EntryPointProc.PossProcTypes := [ptEntryPointProc];
      EntryPointProc.Size := GetProcSize(AAddress);
      Exit;
    end;

    // For applications and libraries.
    EntryPointProc := Procs.Add(EntryPoint);
    // Set the unit to the last unit.
    EntryPointProc.AUnit := Units[Units.Count -1];
    EntryPointProc.PossProcTypes := [ptEntryPointProc];

    Proc := Units.SystemUnit.FindProcByName('@Halt0');
    if Proc = nil then
      raise EDecompilerError.Create(SHalt0NotFound);
    Halt0Address := Proc.Address;

    Proc := Units.SysInitUnit.FindProcByName(InitProcName[ProjectType]);
    if Proc = nil then
      raise EDecompilerError.Create(InitProcName[ProjectType] + ' Proc doesn''t exist');

    InitExeAddress := Proc.Address;
    with TDisAsm.Create do
    try
      // Add procs found be following calls.
      AAddress := EntryPoint;
      // keep looping until call to Halt0.
      repeat
        // Get the first call.
        while AAddress[0] <> #$E8 do
        begin
          GetInstruction(AAddress, ASize);
          Inc(AAddress, ASize);
          // raise an EDecompilerError if were are outside the code section.
          if AAddress > Code + CodeSize then
            raise EDecompilerError.Create('No jmp Halt found');
        end;
        Inc(AAddress, 5);
        // raise an EDecompilerError if were are outside the code section.
        if AAddress > Code + CodeSize then
          raise EDecompilerError.Create('No jmp Halt found');
        // If this is a call to @initexe set initsize to after the call.
        if AAddress + PInteger(AAddress-4)^ = InitExeAddress then
          EntryPointProc.InitSize := AAddress - EntryPoint;
      until AAddress + PInteger(AAddress -4)^ = Halt0Address;
    finally
      Free;
    end;
    EntryPointProc.ProcSize := Integer(AAddress-EntryPoint);
    EntryPointProc.Size := Align4(EntryPointProc.ProcSize);
    // At the Halt proc starts the automatic generated Code.
    EntryPointProc.FinaSize := 5;
  end;

  procedure GenerateProcNames;
  var
    Str: string;
    I: Integer;
  begin
    for I := 0 to Procs.Count -1 do
    begin
      if Procs[I].Name = '' then
      begin
        if Procs[I].AClass = nil then
        begin
          Str := Format('Proc%p', [Pointer(Procs[I].Address)]);
          Procs[I].Name := Str;
        end
        else
        begin
          if Procs[I].overrides then
          begin
            case Procs[I].MethodBindingType of
              mbtVirtual: Procs[I].Name := Procs[I].AClass.AncestorClass.GetVirtualMethod(Procs[I].MethodIndex).Name;
              mbtDynamic: Procs[I].Name := Procs[I].AClass.AncestorClass.GetDynamicMethod(Procs[I].MethodIndex).Name;
              else raise EDecompilerError.Create('not expected Method binding type');
            end;
          end
          else
          begin
            if Procs[I].Address = nil then
              Procs[I].Name := 'AbstractMethod' + IntToStr(Procs[I].MethodIndex)
            else
              Procs[I].Name := Format('Proc%p', [Pointer(Procs[I].Address)]);
          end;
        end;
      end;
    end;
  end;

  procedure AnaProcs;
  var
    I: Integer;
  begin
    I := 0;
    while (I <= Procs.Count -1) do
    begin
      // Analyse every proc, which has a name or is virtual or published (excluding the system procs) and
      // isn't imported and isn't abstract and isn't the entrypointproc.
      if ((Procs[I].Size = 0) or (Procs[I].PossProcTypes = [ptInitialization]) or (Procs[I].PossProcTypes = [ptFinalization])) and
         (not Procs[I].ImportInfo.Imported) and
         (Procs[I].Address <> nil) and
         (Procs[I].PossProcTypes <> [ptEntryPointProc]) and
         (Procs[I].AppendAfter <> atMayNot) and
         (Procs[I].Address > Units.SystemUnit.FInit.Address) then
      begin
        DecompThread.CheckTerminated;

        Procs[I].Size := GetProcSize(Procs[I].Address);
        Procs[I].ProcSize := Procs[I].Size;
        Procs.AnalyzeProc(Procs[I]);
      end;
      Inc(I);
    end;
  end;

  procedure LoadInitResStringImports;
  var
    I: Integer;
  begin
    // Reset the size to make a research for the InitResStringImports and InitImports.
    for I := 2 to Units.Count -2 do
      Units[I].Init.InitSize := Units[I].Init.InitSize;
  end;

  procedure LoadRefs;
  var
    I: Integer;
    DC: TDecompItem;
    DC2: TDecompItem;
  begin
    for I := 0 to Fixups.Count -1 do
    begin
      if (Fixups[I].FixupType = 3) and
         (((Units.FirstNormalUnit <> nil) and (Fixups[I].Address >= Units.FirstNormalUnit.Address)) or
           (Fixups[I].Address >= Units.SystemUnit.Address)) and
         (Fixups[I].Address < BSS + BSSSize) and
         (PPChar(Fixups[I].Address)^ < BSS + BSSSize) then
      begin
        DC := FindDecompItemByBlock(Fixups[I].Address);
        if (DC <> nil) and ((DC.AUnit = nil) or (TUnit(DC.AUnit).Index > 1)) and
           (DC.ClassType <> TDecompItem) then
        begin
          // Ignore the fixups to parts in the own code.
          if (PPChar(Fixups[I].Address)^ < DC.Address) or
             (PPChar(Fixups[I].Address)^ >= DC.Address + DC.Size) then
          begin
            DC2 := FindDecompItemByRef(PPChar(Fixups[I].Address)^);
            if DC2 = nil then
            begin
              DC.Comments.AddComment(Format('Empty req at $%p to $%p',
                  [Pointer(Fixups[I].Address), Pointer(PPChar(Fixups[I].Address)^)]), ctWarning);
            end
            else
              DC.AddReq(DC2, PPChar(Fixups[I].Address)^);
          end;
        end;
      end;
    end;
  end;

  procedure GenerateUnitInfos;
  var
    I: Integer;
    Next: TUnit;
    DC: TDecompItem;
    StartAddress: PChar;
    IsPart: Boolean;
  begin
    // Calculate the System and SysInit address and size with hardcode Init proc sizes.
    with Units.SystemUnit do
    begin
      Address := Code;
      Size := Init.Address + $F8 - Code;
    end;
    with Units.SysInitUnit do
    begin
      Address := Code + Units.SystemUnit.Size;
      Size := Init.Address + $8 - Address;
      StartAddress := Address + Size;
    end;

    repeat
      Next := Units.FindInUnitUsingFInit(StartAddress);
      Next.Address := StartAddress;
      // If the PChars or GUIDs following the Init proc is only used within the init proc
      // it probably is part of the Init proc.
      if Next.Init <> nil then
      begin
        Next.Size := Next.Init.Address + Next.Init.Size - StartAddress;
        // Keep looping untile a Decomp found which is not only required by Init.
        IsPart := True;
        while IsPart do
        begin
          IsPart := False;
          DC := FindDecompItemByBlock(Next.Address + Next.Size);
          // is must be a GUID or string
          if (DC is TGUIDConst) or (DC is TStringInfo) then
          begin
            for I := 0 to DC.ReqByDecompCount -1 do
              if DC.ReqByDecomps[I] <> Next.Init then
                IsPart := False;
            if IsPart then
              Next.Size := Next.Size + DC.Size;
          end;
          // Also if this is the program unit include all decomp items after it.
          if (DC <> nil) and (Next.UnitType = utProgram) then
          begin
            IsPart := True;
            Next.Size := Next.Size + DC.Size;
          end;
        end;
      end
      else
      begin
        // Init is only nil for the program unit.
        Next.Size := EntryPointProc.Address + EntryPointProc.Size - StartAddress;
      end;
      StartAddress := Next.Address + Next.Size;
    until (Next.Init = nil) or (Next.UnitType = utProgram);
  end;

  procedure FindProcsSure;
  var
    DC: TDecompItem;
    Address: PChar;
    AProc: TProc;
    ProcAdded: Boolean;
  begin
    repeat
      ProcAdded := False;
      // Search for a empty block in the Code section
      Address := Code;
      while Address + 4 < Units[Units.Count -1].FInit.Address do
      begin
        DC := FindDecompItemByBlock(Address);
        // If dc1 = nil (meaning there is no Decomps after the previous DC1 and
        if DC = nil then
        begin
          // the gap starts with push ebp ($55), mov ebp, esp ($8B, $EC) it is a proc
          if (((Address[0] = #$55) and (Address[1] = #$8B) and
               (Address[2] = #$EC)) or
              ((Address[0] = #$81) and (Address[1] = #$C4))) and
             (FindDecompItemByRef(Address) = nil) then
          begin
            AProc := Procs.Add(Address);
            with AProc do
            begin
              // It must be a static proc.
              Comments.AddComment('Find proc sure proc', ctDebug);
              PossProcTypes := ptAllStatic;
              Size := GetProcSize(Address);
              ProcSize := Size;
              AUnit := Units.FindInUnitUsingFInit(Address);
            end;
            Procs.AnalyzeProc(AProc);
            ProcAdded := True;
            Inc(Address, Align4(AProc.Size));
          end
          else
          begin
            // If it isn't a proc increment the Address with 4
            Inc(Address, 4);
          end;
        end
        else
          Address := Align4(DC.Address + DC.Size);
      end;
      // Append procs
      AppendProcs;
      // Align4 to proc on DWord
      AlignProcs;
    until not ProcAdded;
  end;

var
  I: Integer;
  S: string;
resourcestring
  SBSSNotFound = 'BSS section not found (this probably isn''t a Delphi program).';
begin
  inherited Create(AFileName);

  if BSS = nil then
    raise EDecompilerError.Create(SBSSNotFound);

  FDecompThread := ADecompThread;
  FMainFile := AMainFile;

  // Determine the project types using the files extention.
  S := UpperCase(ExtractFileExt(FileName));
  if S = '.DLL' then
    FProjectType := ptDLL
  else if S = '.BPL' then
    FProjectType := ptPackage
  else
    FProjectType := ptProgram;

  // Set the import start address and size
  for I := 0 to High(Objects) do
    if Objects[I].ObjectName = '.idata' then
    begin
      FImportStart := Objects[I].Address;
      FImportSize := Objects[I].VirtualSize;
    end;

  // Delete nothing doing Fixups.
  Fixups.Del0Fixups;

  // Create all the collections.
  Decomps := TDecompList.Create;
  Miscs := TMiscs.CreateDecomp(Self);
  Classes := TClassInfos.CreateDecomp(Self);
  TypeInfos := TTypeInfoInfos.CreateDecomp(Self);
  Procs := TProcs.CreateDecomp(Self);
  Units := TUnits.Create(Self);
  StringInfos := TStringInfos.CreateDecomp(Self);
  VarInfos := TVarInfos.CreateDecomp(Self);
  NoTInfoTypes := TNoTInfoTypes.CreateDecomp(Self);
  DFMs := TdcDFMs.CreateDFMs(Self);

  // Add Self to the list of PEFiles.
  SetLength(PEFiles, Length(PEFiles) + 1);
  PEFiles[High(PEFiles)] := Self;

  // Load the ThreadVars(requires: Nothing).
  TThreadVar.Create(Miscs).LoadThreadVar;

  // Load the used packages (requires: Nothing).
  LoadPackages;

  // Load all the InitTable (requires: Nothing).
  LoadInitTable;

  // Load all the Units Init and FInit procs (requires: InitTable).
  Units.LoadInitFInit;

  // Load all the classes (requires: Nothing).
  Classes.LoadClassInfos;

  DecompThread.CheckTerminated;

  // Get the system procs. (requires: Units FInits)
  LoadSystemObjFiles;

  DecompThread.CheckTerminated;

  // Load the imported procs (requires: Unit FInits).
  LoadImportedProcs;

  // Get System Units imported proc file names (requires: Units and imported procs)
  LoadSystemImportedProcNames;

  DecompThread.CheckTerminated;

  // Load all the TypeInfos (requires: Classes).
  TypeInfos.LoadTypeInfos;

  // Analyse the class infos (requires: Classes, AbstractErrorProc; TypeInfos, classes).
  for I := 0 to Classes.Count -1 do
    Classes[I].AnaClass;

  // Load the exported procedures (requires: units, Classes).
  Procs.LoadExportedProcs;

  // Set the Classes and TypeInfos units (requires: Unit FInit, Classes and TypeInfos).
  SetClassesTypeInfosUnits;

  DecompThread.CheckTerminated;

  // Load all the unit names (requires: Classes, TypeInfos and Unit FInits).
  LoadUnitNames;

  // Get the size of the entrypoint procedure (requires: @Halt0 and @InitExe procedures).
  GetEntryPointProc;

  // Load Init count vars. (requires: FInit Procs; EntryPointProc).
  VarInfos.LoadInitVars;

  // Load all the vars. (requires: SystemVars; Init Vars)
  VarInfos.LoadVars;

  DecompThread.CheckTerminated;

  // Load the resource strings (requires: Nothing)
  LoadResStrings;

  // Load the resource string import tables(requires: vars, Inits Procs; res strings).
  LoadInitResStringImports;

  // Load all the class procs (only dynamic and published, no static procs) (requires: Classes).
  LoadDynamicMethods;
  Procs.LoadPublishedMethods;

  DecompThread.CheckTerminated;

  if MainFile then
  begin
    // Analyze the already existing procs.
    AnaProcs;

    DecompThread.CheckTerminated;

    // Get the procs called from within the EntryPointProc. (require)
    Procs.AnalyzeProc(EntryPointProc);

    DecompThread.CheckTerminated;

    // Append none 4 byte alligned procs.
    AppendProcs;

    // Align the procs to 4 bytes.
    AlignProcs;
  end;

  // Check to see if the system code is completly filled. (requires: procs, strings, classes)
// Removed to prevent errors with packages.  CheckSystemCode;

  // Load all the string consts and initilized var (requires: Nothing)
  StringInfos.LoadStringInfos;
  DecompThread.CheckTerminated;

  if MainFile then
  begin
    // Search the empty blocks in the Code section which are absolutly a proc.
    FindProcsSure;

    DecompThread.CheckTerminated;

    // Fill the empty blocks.
    CheckCode;

    DecompThread.CheckTerminated;

    // Append Procs which must Append (as generate in AnalyzeProc), (requires: All Procs).
    AppendProcs;

    // -- All decomp Items are loaded now.

    // Load all the references (references to procs are already loaded by AnaProc).
    LoadRefs;
  end;

  // Load the references made by the method types.
  for I := 0 to TypeInfos.Count -1 do
    TypeInfos[I].LoadMethodRefs;

  DecompThread.CheckTerminated;

  // Generate the unit addresses and sizes (requires: all decomps, all refs).
  GenerateUnitInfos;

  // Load all the shortcuts to the decomps items (requires: All decomp Items);
  VarInfos.LoadFixups;

  // Assign the decomps it's units (AUnit is still nil for vars).
  Units.AssignUnits;

  // Determen vars Units.
  VarInfos.DeterUnits;

  // Determen where the items belongs in the interface or implementation
  // section (requires: all Unit items.).
  Units.DeterIntfImpls;

  // Generate the names for all the procs. (requires: all procs).
  GenerateProcNames;

  DecompThread.CheckTerminated;

  // Generate the names for all the vars (requires: all vars).
  VarInfos.GenerateNames;

  // Generate the names for all the units.
  Units.GenerateNames;

  DecompThread.CheckTerminated;

  if MainFile then
  begin
    DecompThread.CheckTerminated;

    // Generate the class definetions
    Classes.GenerateClassDefs;

    // Generate the list of req units
    Units.GenerateReqUnits;

    DecompThread.CheckTerminated;

    // Generate the source for the units
    Units.GenUnitSrcs;
  end;
  
  // Add self to the PEFile list.
  TdcDecompThread(DecompThread).AddPEFileClass(Self);
end;

destructor TPEFileClass.Destroy;
begin
  FDestroying := True;
  DFMs.Free;
  Miscs.Free;
  StringInfos.Free;
  Classes.Free;
  TypeInfos.Free;
  Procs.Free;
  Decomps.Free;
  NoTInfoTypes.Free;
  VarInfos.Free;
  Units.Free;
  inherited Destroy;
end;

procedure TPEFileClass.LoadPackages;
var
  J, K, L: Integer;
  Str: string;
  UsedPackage: TPEFileClass;
label
  AlreadyLoaded;
resourcestring
  SErrorIDImport = 'Can''t import package item using an ID';
  SErrorImportNotFount = 'Imported procedure %s not found';
begin
  for J := 0 to High(Imports) do
    with Imports[J] do
    begin
      // If this is a package and this package import dll, has a .bpl ext, analyse it.
      if AnsiCompareText(ExtractFileExt(DLLName), '.BPL') = 0 then
      begin
        // Don't load it if it is already loaded.
        for K := 0 to High(PEFiles) do
          if AnsiCompareText(PEFiles[K].ProjectName, Copy(DLLName, 1, Length(DLLName) -4)) = 0 then
          begin
            UsedPackage := PEFiles[K];
            goto AlreadyLoaded;
          end;
        // Load the package.
        if FileExists(DLLName) then
          UsedPackage := TPEFileClass.CreateDecomp(DLLName, DecompThread, False)
        else
        begin
          SetLength(Str, MAX_PATH);
          GetSystemDirectory(PChar(Str), Length(Str));
          SetLength(Str, StrLen(PChar(Str)));
          UsedPackage := TPEFileClass.CreateDecomp(Str + '\'+ DLLName, DecompThread, False);
        end;
      AlreadyLoaded:
        // this file uses packages.
        FUsePackages := True;
        // Make sure the imports points to the rights address.
        for K := 0 to High(Entries) do
        begin
          if Entries[K].NameOrID = niID then
            raise EDecompilerError.Create(SErrorIDImport);
          L := UsedPackage.PEExports.FindCaseInSens(Entries[K].Name);
          if L = -1 then
            raise EDecompilerError.CreateFmt(SErrorImportNotFount, [Entries[K].Name]);
          PPChar(Entries[K].PAddress)^ := UsedPackage.PEExports[L].Address;
        end;
      end;
    end;
end;

function TPEFileClass.NotRecurFindDecompItemByRef(AAddress: PChar): TDecompItem;
var
  I: Integer;
begin
  I := Decomps.FindByRef(AAddress);
  if I = -1 then
    Result := nil
  else
    Result := Decomps[I];
end;

function TPEFileClass.FindDecompItemByRef(AAddress: PChar): TDecompItem;
var
  I: Integer;
begin
  Result := NotRecurFindDecompItemByRef(AAddress);
  if (AAddress >= ImportStart) and (AAddress < ImportStart + ImportSize) then
    AAddress := PPChar(AAddress)^;
  for I := 0 to DecompThread.PEFileClassCount -1 do
  begin
    if Result <> nil then Exit;
    Result := TPEFileClass(DecompThread.PEFileClasses[I]).NotRecurFindDecompItemByRef(AAddress);
  end;
end;

function TPEFileClass.FindDecompItemByBlock(Address: PChar): TDecompItem;
var
  I: Integer;
begin
  I := Decomps.FindByBlock(Address);
  if I = -1 then
    Result := nil
  else
    Result := Decomps[I];
end;

function TPEFileClass.FindDecompItemAfter(AAddress: PChar): TDecompItem;
var
  I: Integer;
begin
  Result := nil;
  for I := 0 to Decomps.Count -1 do
    with TDecompItem(Decomps[I]) do
      if (Address > AAddress) and
         ((Result = nil) or (Result.Address > Address)) then
        Result := Decomps[I];
end;

function TPEFileClass.NotRecurFindTypeByName(const Name: string): TDecompItem;
var
  I: Integer;
begin
  for I := 0 to Classes.Count -1 do
    if Classes[I].AClass.ClassName = Name then
    begin
      Result := Classes[I];
      Exit;
    end;
  for I := 0 to TypeInfos.Count -1 do
    if TypeInfos[I].TypeInfo.Name = Name then
    begin
      Result := TypeInfos[I];
      Exit;
    end;
  for I := 0 to NoTInfoTypes.Count -1 do
    if NoTInfoTypes[I].Name = Name then
    begin
      Result := NoTInfoTypes[I];
      Exit;
    end;
  Result := nil;
end;

function TPEFileClass.FindTypeByName(const Name: string): TDecompItem;
var
  I: Integer;
begin
  Result := NotRecurFindTypeByName(Name);
  if Result <> nil then Exit;
  for I := 0 to DecompThread.PEFileClassCount -1 do
  begin
    Result := TPEFileClass(DecompThread.PEFileClasses[I]).NotRecurFindTypeByName(Name);
    if Result <> nil then
      Exit;
  end;
  raise EDecompilerError.Create('type with Name ' + Name + ' not found');
end;

function TPEFileClass.FindSystemProc(const Name: string): TProc;
begin
  if Units.SystemUnit <> nil then
    Result := Units.SystemUnit.FindProcByName(Name)
  else
    Result := nil;
end;

function EnhQuotedStr(Str: string): string;
var
  I: Integer;
  X: char;
  InQuotes: Boolean;
begin
  Result := Str;
  InQuotes := False;
  for I := Length(Str) downto 1 do
  begin
    X := Str[I];
    // If the string is longer then 900 character split it using a +
    if (Length(Result) - I + 1) mod 900 = 0 then
    begin
      if InQuotes then
      begin
        InQuotes := False;
        Insert('''', Result, I);
      end;
      Insert(' + ', Result, I);
    end;

    if (X in [' '..'z']) and (X <> '''') then
    begin
      if not InQuotes then
      begin
        InQuotes := True;
        Insert('''', Result, I+1);
      end
    end
    else
    begin
      Delete(Result, I, 1);
      if InQuotes then
      begin
        InQuotes := False;
        Insert('''', Result, I);
      end;
      Insert('#' + IntToStr(Ord(X)), Result, I);
    end;
  end;
  if InQuotes then
    Result := '''' + Result;
  if Length(Str) = 0 then
    Result := ''''''; 
end;

function IsIdentifier(Address: PChar): Boolean;
var
  I: Integer;
begin
  Result := False;
  if (Ord(Address[0]) <= 0) and (Address[1] in ['a'..'z', 'A'..'Z', '_']) then
    Exit;
  for I := 2 to Ord(Address[0]) do
    if not (Address[I] in ['a'..'z', 'A'..'Z', '.', '_', '0'..'9']) then
      Exit;
  Result := True;
end;

function IsPackage(const FileName: string): Boolean;
begin
  Result := AnsiCompareText(ExtractFileExt(FileName), '.bpl') = 0;
end;

end.


