
//    Author:          Python (python@softhome.net)
//    Version:         0.0.1.2
//    LastModified:    3-23-2000
//    LatestVersion:   http://thunder.prohosting.com/~pytho/
//    Copyright (c) 1999, 2000 Python. All rights reserved

unit PEFile;

interface

uses
  SysUtils, Windows, Classes, peFixups, peExports;

type
  PImageDosHeader = ^TImageDosHeader;
  _IMAGE_DOS_HEADER = packed record      { DOS .EXE header                  }
      e_magic: Word;                     { Magic number                     }
      e_cblp: Word;                      { Bytes on last page of file       }
      e_cp: Word;                        { Pages in file                    }
      e_crlc: Word;                      { Relocations                      }
      e_cparhdr: Word;                   { Size of header in paragraphs     }
      e_minalloc: Word;                  { Minimum extra paragraphs needed  }
      e_maxalloc: Word;                  { Maximum extra paragraphs needed  }
      e_ss: Word;                        { Initial (relative) SS value      }
      e_sp: Word;                        { Initial SP value                 }
      e_csum: Word;                      { Checksum                         }
      e_ip: Word;                        { Initial IP value                 }
      e_cs: Word;                        { Initial (relative) CS value      }
      e_lfarlc: Word;                    { File address of relocation table }
      e_ovno: Word;                      { Overlay number                   }
      e_res: array [0..3] of Word;       { Reserved words                   }
      e_oemid: Word;                     { OEM identifier (for e_oeminfo)   }
      e_oeminfo: Word;                   { OEM information; e_oemid specific}
      e_res2: array [0..9] of Word;      { Reserved words                   }
      _lfanew: LongInt;                  { File address of new exe header   }
  end;
  TImageDosHeader = _IMAGE_DOS_HEADER;

  PIMAGE_OPTIONAL_HEADER = ^IMAGE_OPTIONAL_HEADER;
  IMAGE_OPTIONAL_HEADER = packed record
   { Standard fields. }
    Magic           : WORD;
    MajorLinkerVersion : Byte;
    MinorLinkerVersion : Byte;
    SizeOfCode      : DWORD;
    SizeOfInitializedData : DWORD;
    SizeOfUninitializedData : DWORD;
    AddressOfEntryPoint : DWORD;
    BaseOfCode      : DWORD;
    BaseOfData      : DWORD;
   { NT additional fields. }
    ImageBase       : DWORD;
    SectionAlignment : DWORD;
    FileAlignment   : DWORD;
    MajorOperatingSystemVersion : WORD;
    MinorOperatingSystemVersion : WORD;
    MajorImageVersion : WORD;
    MinorImageVersion : WORD;
    MajorSubsystemVersion : WORD;
    MinorSubsystemVersion : WORD;
    Reserved1       : DWORD;
    SizeOfImage     : DWORD;
    SizeOfHeaders   : DWORD;
    CheckSum        : DWORD;
    Subsystem       : WORD;
    DllCharacteristics : WORD;
    SizeOfStackReserve : DWORD;
    SizeOfStackCommit : DWORD;
    SizeOfHeapReserve : DWORD;
    SizeOfHeapCommit : DWORD;
    LoaderFlags     : DWORD;
    NumberOfRvaAndSizes : DWORD;
    DataDirectory   : packed array [0..IMAGE_NUMBEROF_DIRECTORY_ENTRIES-1] of IMAGE_DATA_DIRECTORY;
    Objects: packed array [0..9999] of IMAGE_SECTION_HEADER;
  end;

  PIMAGE_NT_HEADERS = ^IMAGE_NT_HEADERS;
  IMAGE_NT_HEADERS = packed record
    Signature       : DWORD;
    FileHeader      : IMAGE_FILE_HEADER;
    OptionalHeader  : IMAGE_OPTIONAL_HEADER;
  end;

  { TPEObject }

  TPEObject = record
    ObjectName: string;
    Address: PChar;
    PhysicalSize: Integer;
    VirtualSize: Integer;
    Characteristics: Cardinal;
    PointerToRawData: Integer;
  end;

  TNameOrID = (niName, niID);

  { TPEImports }

  TPEImport = record
    NameOrID: TNameOrID;
    Name: string;
    ID: Integer;
    PAddress: PChar;  // Pointer to the imported function
                      // (used to call the function).
  end;

  TPEImports = record
    DLLName: string;
    Entries: array of TPEImport;
  end;


  { TPEFixups }

  TPEFixup = record
    FixupType: Byte;
    Address: PChar;
  end;

  TPEFixups = array of TPEFixup;

  { TPEResource }

  TDataOrEntries = (deData, deEntries);

  TPEResource = class;

  TPEResourceList = array of TPEResource;

  TPEResource = class(TObject)
  private
    FNameOrID: TNameOrID;
    FName: WideString;
    FID: Integer;
    FDataOrEntries: TDataOrEntries;
    FData: PChar;
    FDataSize: Integer;
    FEntries: TPEResourceList;
    function GetData: PChar;
    function GetDataSize: Integer;
    function GetEntries: TPEResourceList;
  public
    property NameOrID: TNameOrID read FNameOrID;
    property Name: WideString read FName;
    property ID: Integer read FID;
    property DataOrEntries: TDataOrEntries read FDataOrEntries;
    property Data: PChar read GetData;
    property DataSize: Integer read GetDataSize;
    property Entries: TPEResourceList read GetEntries;
  end;

  { TPEFile }

  EPEError = class(Exception);

  TPEFile = class(TObject)
  private
    FFileName: string;
    FProjectName: string;
  protected
    FFileStream: TFileStream;
    FFileBase: PChar;
    FHeaderSize: Integer;
    FImageSize: Integer;
    FIsConsole: Boolean;
    FNTHeader: PIMAGE_NT_HEADERS;
    procedure Error(S: string);
  public
    ImageBase: PChar;
    StackCommitSize: Cardinal;
    StackReserveSize: Cardinal;
    EntryPoint: PChar;
    Code: PChar;
    CodeSize: Cardinal;
    Data: PChar;
    DataSize: Cardinal;
    BSS: PChar;
    BSSSize: Cardinal;
    Objects: array of TPEObject;
    PEExports: TpeExportList;
    Imports: array of TPEImports;
    Fixups: TFixups;
    Resources: TPEResourceList;
    constructor Create(FileName: string); virtual;
    destructor Destroy; override;
    function PhysOffset(Address: PChar): Integer;

    property FileBase: PChar read FFileBase;
    property NTHeader: PIMAGE_NT_HEADERS read FNTHeader;
    property IsConsole: Boolean read FIsConsole;
    property FileName: string read FFileName;
    property ProjectName: string read FProjectName;
  end;

type
  PByte = ^Byte;
  PPointer = ^Pointer;
  PInteger = ^Integer;
  PPChar = ^PChar;

resourcestring
  SErrorAddressNotInFile = 'Error, Address %p not in file';
  SErrorInvalidResourceType = 'Error, can''t read this, other resource'; 

implementation

{ TPEResource }

function TPEResource.GetData: PChar;
begin
  if DataOrEntries = deEntries then
    raise EPEError.Create(SErrorInvalidResourceType);
  Result := FData;
end;

function TPEResource.GetDataSize: Integer;
begin
  if DataOrEntries = deEntries then
    raise EPEError.Create(SErrorInvalidResourceType);
  Result := FDataSize;
end;

function TPEResource.GetEntries: TPEResourceList;
begin
  if DataOrEntries = deData then
    raise EPEError.Create(SErrorInvalidResourceType);
  Result := FEntries;
end;

{ TPEFile }

constructor TPEFile.Create(FileName: string);
var
  I: Integer;

  procedure LoadResources;
  var
    BaseEntry: PChar;

    procedure FillResources(var RL: TPEResourceList;
       DirEntry: Integer);
    var
      I: Integer;
      NameCount, IDCount: Integer;

    type
      PIMAGE_RESOURCE_DIRECTORY = ^IMAGE_RESOURCE_DIRECTORY;
      IMAGE_RESOURCE_DIRECTORY = packed record
        Characteristics : DWORD;
        TimeDateStamp   : DWORD;
        MajorVersion    : WORD;
        MinorVersion    : WORD;
        NumberOfNamedEntries : WORD;
        NumberOfIdEntries : WORD;
      end;

    begin
      // Save Name and ID count.
      NameCount :=PIMAGE_RESOURCE_DIRECTORY(DirEntry)^.NumberOfNamedEntries;
      IDCount := PIMAGE_RESOURCE_DIRECTORY(DirEntry)^.NumberOfIDEntries;
      // Make resources large enough.
      SetLength(RL, NameCount + IDCount);
      // Jump over Resource table.
      Inc(DirEntry, SizeOf(Image_Resource_Directory));
      for I := 0 to NameCount -1 do
      begin
        // Create object
        RL[I] := TPEResource.Create;
        // Name or ID is Name.
        RL[I].FNameOrID := niName;
        // Copy name.
        SetLength(RL[I].FName, PWord(BaseEntry + PDWord(DirEntry)^ and $7FFFFFFF)^);
        Move(PDWord(BaseEntry + PDWord(DirEntry)^ and $7FFFFFFF + 2)^,
          RL[I].FName[1], Length(RL[I].FName) * 2);
        if PDWord(PChar(DirEntry) + 4)^ and $80000000 = 0 then
        begin
          // DataEntry
          RL[I].FDataOrEntries := deData;
          RL[I].FData := FFileBase + PDWord(BaseEntry + PDWord(DirEntry + 4)^)^;
          RL[I].FDataSize := PInteger(BaseEntry + PDWord(DirEntry +4)^ +4)^;
        end
        else
        begin
          // SubEntries
          RL[I].FDataOrEntries := deEntries;
          FillResources(RL[I].FEntries,
            Integer(BaseEntry + PDWord(DirEntry + 4)^ and $7FFFFFFF));
        end;
        Inc(DirEntry, 8);
      end;
      for I := NameCount to NameCount + IDCount -1 do
      begin
        // Create object
        RL[I] := TPEResource.Create;
        // Name or ID is Name.
        RL[I].FNameOrID := niID;
        // Copy name.
        RL[I].FID := PDWord(DirEntry)^;
        if PDWord(DirEntry+ 4)^ and $80000000 = 0 then
        begin
          // DataEntry
          RL[I].FDataOrEntries := deData;
          RL[I].FData := FFileBase + PDWord(BaseEntry + PDWord(DirEntry + 4)^)^;
          RL[I].FDataSize := PInteger(BaseEntry + PDWord(DirEntry +4)^ +4)^;
        end
        else
        begin
          // SubEntries
          RL[I].FDataOrEntries := deEntries;
          FillResources(RL[I].FEntries,
            Integer(BaseEntry + PDWord(DirEntry + 4)^ and $7FFFFFFF));
        end;
        Inc(DirEntry, 8);
      end;
    end;

  begin
    if NTHeader^.OptionalHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_RESOURCE].Size <> 0 then
    begin
      BaseEntry := FFileBase +
        NTHeader^.OptionalHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_RESOURCE].VirtualAddress;
      FillResources(Resources, Integer(BaseEntry));
    end;
  end;

  procedure LoadImports;

  type
    PImage_Import_Entry = ^Image_Import_Entry;
    Image_Import_Entry = record
      Characteristics: Integer;
      TimeDateStamp: Integer;
      MajorVersion: WORD;
      MinorVersion: WORD;
      Name: Integer;
      LookupTable: Integer;
    end;

  var
    ImportEntry: PImage_Import_Entry;
    LookupEntry: PDWord;
  begin
    ImportEntry := PImage_Import_Entry(FFileBase +
      NTHeader^.OptionalHeader.DataDirectory[IMAGE_DIRECTORY_ENTRY_IMPORT].VirtualAddress);
    // Keep reading import entry until empty entry.
    while ImportEntry^.Name <> 0 do
    begin
      // New Import entry.
      SetLength(Imports, Length(Imports) +1);
      with Imports[High(Imports)] do
      begin
        DLLName := FFileBase + ImportEntry.Name;
        LookupEntry := PDWord(FFileBase + ImportEntry.LookupTable);
        // Keep reading loolup table until empty lookup
        while LookupEntry^ <> 0 do
        begin
          SetLength(Entries, Length(Entries) +1);
          with Entries[High(Entries)] do
          begin
            if (LookupEntry^ and $80000000) <> 0 then
            begin
              NameOrID := niID;
              ID := LookupEntry^ and $7FFFFFFF;
            end
            else
            begin
              NameOrID := niName;
              Name := PChar(FFileBase + LookupEntry^ + 2);
            end;
            PAddress := PChar(LookupEntry);
          end;
          Inc(LookupEntry);
        end;
        Inc(ImportEntry);
      end;
    end;
  end;

var
  J: Integer;
begin
  inherited Create;
  FFileName := FileName;
  FProjectName := ExtractFileName(FileName);
  SetLength(FProjectName, Length(FProjectName) - Length(ExtractFileExt(FProjectName)));

  FFileStream := TFileStream.Create(FileName, fmOpenRead or fmShareDenyWrite);
  with FFileStream do
  begin
    // Read new header address.
    Position := 60;
    ReadBuffer(I, 4);
    Position := I + 52;
    ReadBuffer(ImageBase, 4);
    Seek(24, soFromCurrent);
    // Read Image and Header size.
    ReadBuffer(FImageSize, 4);
    ReadBuffer(FHeaderSize, 4);

    // Allocate memory at ImageAddress.
    FFileBase := VirtualAlloc(ImageBase, FImageSize, MEM_RESERVE or MEM_COMMIT, PAGE_READWRITE);
    if FFileBase = nil then
    begin
      FFileBase := VirtualAlloc(nil, FImageSize, MEM_RESERVE or MEM_COMMIT, PAGE_READWRITE);
      if FFileBase = nil then
        Error('Couldn''t allocate memory.');
    end;

    // Read Header
    Position := 0;
    ReadBuffer(PPointer(FFileBase)^, FHeaderSize);

    with PImageDosHeader(FFileBase)^ do
    begin
      // Check magic number.
      if not e_magic = IMAGE_DOS_SIGNATURE then Error('unrecognized file format');
      // Load NT Header.
      FNTHeader := PIMAGE_NT_HEADERS(FFileBase + _lfanew);
    end;

    with NTHeader^ do
    begin
      // Check NT Header.
      if Signature <> IMAGE_NT_SIGNATURE then
        Error('Not a PE (WIN32 Executable) file');

      // Save StackCommitSize
      StackCommitSize := OptionalHeader.SizeOfStackCommit;
      // Save StackReserveSize
      StackReserveSize := OptionalHeader.SizeOfStackReserve;
      // Save EntryPoint
      EntryPoint := FFileBase + OptionalHeader.AddressOfEntryPoint;
      // Save Code Size
      CodeSize := OptionalHeader.SizeOfCode;
      // Save Code Address
      Code := FFileBase + OptionalHeader.BaseOfCode;
      // Save Data Size
      DataSize := OptionalHeader.SizeOfInitializedData;
      // Save Data Address
      Data := FFileBase + OptionalHeader.BaseOfData;

      { Read the object from file and save the information }

      // Set the number of Objects
      SetLength(Objects, FileHeader.NumberOfSections);
      // Save the Objects
      for I := 0 to High(Objects) do
      begin
        // Copy the name.
        SetLength(Objects[I].ObjectName, 8);
        Move(OptionalHeader.Objects[I].Name, Objects[I].ObjectName[1], 8);
        SetLength(Objects[I].ObjectName, StrLen(PChar(Objects[I].ObjectName)));
        // Save the physical size.
        Objects[I].PhysicalSize := OptionalHeader.Objects[I].SizeOfRawData;
        Objects[I].VirtualSize := OptionalHeader.Objects[I].Misc.VirtualSize;
        // Save the address (virtual).
        Objects[I].Address := FFileBase + OptionalHeader.Objects[I].VirtualAddress;
        // Read the data to the Address
        Objects[I].PointerToRawData := OptionalHeader.Objects[I].PointerToRawData;
        Position := OptionalHeader.Objects[I].PointerToRawData;
        ReadBuffer(PPointer(Objects[I].Address)^, Objects[I].PhysicalSize);
        // Save the Characteristics.
        Objects[I].Characteristics := OptionalHeader.Objects[I].Characteristics;
      end;

      // Load the BBS
      for J := 0 to High(Objects) -1 do
        if Objects[J].ObjectName = 'BSS' then
        begin
          BSS := Objects[J].Address;
          BSSSize := Objects[J +1].Address - Objects[J].Address;
          Break;
        end;

      // Set IsConsole
      FIsConsole := OptionalHeader.SubSystem = 3;

      Fixups := TFixups.Create(Self);
      Fixups.ApplyFixups;

      LoadResources;
      LoadImports;
    end;
  end;

  PEExports := TpeExportList.Create(Self);
end;

destructor TPEFile.Destroy;

  procedure FreeResources(const AResources: TPEResourceList);
  var
    I: Integer;
  begin
    for I := 0 to High(AResources) do
    begin
      if AResources[I].DataOrEntries = deEntries then
        FreeResources(AResources[I].Entries);
      AResources[I].Free;
    end;
  end;

begin
  FreeResources(Resources);
  PEExports.Free;
  Fixups.Free;
  if FFileBase <> nil then
    VirtualFree(FFileBase, 0, MEM_RELEASE);
  FFileStream.Free;
  inherited Destroy;
end;

procedure TPEFile.Error(S: string);
begin
  raise EPEError.Create(S);
end;

function TPEFile.PhysOffset(Address: PChar): Integer;
var
  I: Integer;
begin
  // Return a address is physical offset in the file, for modifying it.
  if (Address < FileBase) or
     (Address >= Objects[High(Objects)].Address + Objects[High(Objects)].VirtualSize) then
    raise EPEError.CreateFmt(SErrorAddressNotInFile, [Pointer(Address)]);
  // If the address in one of the objects.
  for I := 0 to High(Objects) do
    if (Address >= Objects[I].Address) and (Address < Objects[I].Address + Objects[I].VirtualSize) then
    begin
      // Address must be inside the physical declared part.
      if Address >= Objects[I].Address + Objects[I].PhysicalSize then
        raise EPEError.CreateFmt(SErrorAddressNotInFile, [Pointer(Address)]);
      // Calculate the address.
      Result := Address - Objects[I].Address + Objects[I].PointerToRawData;
      Exit;
    end;
  if Address < PChar(NTHeader) + SizeOf(Windows.IMAGE_NT_HEADERS) +
      Length(Objects) * SizeOf(TImageSectionHeader) then
  begin
    Result := Address - FFileBase;
    Exit;
  end;
  raise EPEError.CreateFmt(SErrorAddressNotInFile, [Pointer(Address)]);
end;

end.
