unit NameMangling;

interface

uses
  SysUtils;

type
  TNameManglingType = (eitType, eitClass, eitProc, eitMethod, eitVar, eitStrangeType, eitSpecial);
  TNameManglingInfo = record
    NMType: TNameManglingType;
    UnitName: string;
    ClassName: string;
    ItemName: string;
    ItemProp: string;
  end;

  ENameManglingError = class(Exception);

function GetNameManglingInfo(Str: string): TNameManglingInfo;

implementation

function GetNameManglingInfo(Str: string): TNameManglingInfo;
var
  I, J: Integer;
  HasClass: Boolean;
resourcestring
  SInvalidStrangType = 'Unrecognized strang type name %s';
  SUnitNameExpected = 'Unit name followed by a @, but %s found.';
begin
  if (Length(Str) = 0) or (Str[1] <> '@') then
  begin
    // If the first item isn't a '@' this is a special type.
    Result.NMType := eitSpecial;
    Result.ItemProp := Str;
  end
  else
  begin
    if (Length(Str) >= 2) and (Str[2] = '$') then
    begin
      // If the second character is a '$' than this is StrangeType (sorry, I couldn't think of a better name).
      Result.NMType := eitStrangeType;
      // string should start with '@$xp$'
      if (Length(Str) < 5) or (Copy(Str, 1, 5) <> '@$xp$') then
        raise ENameManglingError.CreateFmt(SInvalidStrangType, [Str]);
      // Remove '@$xp$'
      Delete(str, 1, 5);
      // there will now be a number
      if not (Str[1] in ['1'..'9']) then
        raise ENameManglingError.CreateFmt(SInvalidStrangType, [Str]);
      I := 1;
      while Str[I + 1] in ['0'..'9'] do
        Inc(I);
      // Delete the number.
      Delete(Str, 1, I);
      // If there is a @ in Str, there is first a unit name.
      I := Pos('@', Str);
      if I <> 0 then
      begin
        Result.UnitName := Copy(Str, 1, I -1);
        Result.ItemName := Copy(Str, I +1, Length(Str) - I);
      end
      else
        Result.ItemName := Str;
    end
    else
    begin
      // Delete the first '@'
      Delete(Str, 1, 1);
      // There is now a unit name until a new @
      I := Pos('@', Str);
      if I = 0 then
      begin
        // If there is no @ this is a special item.
        Result.ItemName := Str;
        Result.NMType := eitSpecial;
        Exit;
      end;

      if I = 1 then
        raise ENameManglingError.CreateFmt(SUnitNameExpected, [Str]);
        
      Result.UnitName := Copy(Str, 1, I -1);
      // Delete the unit name and the following '@'
      Delete(Str, 1, I);

      I := Pos('@', Str);
      J := Pos('$', Str);
      if (I > 1) and ((J = 0) or (J > I)) then
      begin
        // If there is a '@' withoud a '$' it there is first a class defenition.
        HasClass := True;
        Result.ClassName := Copy(Str, 1, I -1);
        Delete(Str, 1, I);
        J := Pos('$', Str);
        // If there is nothing after the @ this is a class defenition.
        if Str = '' then
        begin
          Result.NMType := eitClass;
          Exit;
        end;
      end
      else
      begin
        HasClass := False;
        Result.ClassName := '';
      end;

      if J = 0 then
      begin
        // If there is no '$' this is a var.
        Result.ItemName := Str;
        Result.NMType := eitVar;
      end
      else
      begin
        // This is a proc
        // First there is the proc name until the $.
        Result.ItemName := Copy(Str, 1, J -1);
        // Save the rest as item prop
        Result.ItemProp := Copy(Str, J + 1, Length(Str) - J);
        // Set the type.
        if HasClass then
          Result.NMType := eitMethod
        else
          Result.NMType := eitProc;
      end;
    end;
  end;
end;

end.
