unit MethodLists;

interface

uses
  Classes, SysUtils;

type
  EMethodListError = class(Exception);
  
  { TMethodList }

  TConstMethodArray = array[0..MaxListSize] of TMethod;
  PConstMethodArray = ^TConstMethodArray;

  TMethodList = class(TObject)
  private
    FCount: Integer;
    ItemList: PConstMethodArray;
  protected
    procedure AddMethod(const Method: TMethod);
    procedure RemoveMethod(const Method: TMethod);
    function FirstMethod(var Method: TMethod): Boolean;
    function NextMethod(const Method: TMethod; var NextMethod: TMethod): Boolean;
  public
    destructor Destroy; override;
  end;

  { TmlneMethodList }

  TmlneMethodList = class;

  TMLNotifyEvent = procedure (Sender: TmlneMethodList) of object;

  TmlneMethodList = class(TMethodList)
  public
    procedure Add(const Method: TMLNotifyEvent);
    procedure Remove(const Method: TMLNotifyEvent);
    procedure CallFirst;
    procedure CallNext(const Method: TMLNotifyEvent);
  end;

resourcestring
  SMethodNotFound = 'Method not found in the list.';

implementation

{ TMethodList }

destructor TMethodList.Destroy;
begin
  inherited Destroy;
  FreeMem(ItemList, FCount * SizeOf(TMethod));
end;

procedure TMethodList.AddMethod(const Method: TMethod);
var
  I: Integer;
begin
  // Exit when the method is already in the list.
  for I := 0 to FCount -1 do
    if (ItemList^[I].Code = Method.Code) and (ItemList^[I].Data = Method.Data) then
      Exit;
  Inc(FCount);
  ReallocMem(ItemList, FCount * SizeOf(TMethod));
  ItemList[FCount -1] := Method;
end;

procedure TMethodList.RemoveMethod(const Method: TMethod);
var
  I: Integer;
begin
  Dec(FCount);
  for I := 0 to FCount do
    if (ItemList[I].Code = Method.Code) and (ItemList[I].Data = Method.Data) then
    begin
      Move(ItemList^[I +1], ItemList^[I], FCount - I);
      ReallocMem(ItemList, FCount * SizeOf(TMethod));
      Exit;
    end;
  raise EMethodListError.Create(SMethodNotFound);
end;

function TMethodList.FirstMethod(var Method: TMethod): Boolean;
begin
  Result := FCount <> 0;
  if Result then
    Method := ItemList^[0];
end;

function TMethodList.NextMethod(const Method: TMethod; var NextMethod: TMethod): Boolean;
var
  I: Integer;
begin
  for I := 0 to FCount -1 do
    if (ItemList[I].Code = Method.Code) and (ItemList[I].Data = Method.Data) then
    begin
      Result := I <> FCount -1;
      if Result then
        NextMethod := ItemList[I + 1];
      Exit;
    end;
  raise EMethodListError.Create(SMethodNotFound);
end;

{ TmlneMethodList }

procedure TmlneMethodList.Add(const Method: TMLNotifyEvent);
begin
  AddMethod(TMethod(Method));
end;

procedure TmlneMethodList.Remove(const Method: TMLNotifyEvent);
begin
  RemoveMethod(TMethod(Method));
end;

procedure TmlneMethodList.CallFirst;
var
  Method: TMLNotifyEvent;
begin
  if FirstMethod(TMethod(Method)) then
    Method(Self);
end;

procedure TmlneMethodList.CallNext(const Method: TMLNotifyEvent);
var
  NextM: TMLNotifyEvent;
begin
  if NextMethod(TMethod(Method), TMethod(NextM)) then
    NextM(Self);
end;

end.
