
unit Mover;

interface

uses
  Windows, Controls, Classes;

type
  NN = 1..High(Integer);

  TMover = class(TGraphicControl)
  private
    FLineDC: HDC;
    FDownPos: TPoint;
    FSplit: Integer;
    FMinSize: NN;
    FMaxSize: Integer;
    FControl: TControl;
    FNewSize: Integer;
    FActiveControl: TWinControl;
    FOldKeyDown: TKeyEvent;
    FBeveled: Boolean;
    FLineVisible: Boolean;
    FOnMoved: TNotifyEvent;
    procedure AllocateLineDC;
    procedure DrawLine;
    procedure ReleaseLineDC;
    procedure UpdateSize(X, Y: Integer);
    procedure FocusKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure SetBeveled(Value: Boolean);
  protected
    procedure Paint; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure StopSizing;
  public
    constructor Create(AOwner: TComponent); override;
    procedure Loaded; override;
  published
    property Align default alLeft;
    property Beveled: Boolean read FBeveled write SetBeveled default True;
    property Color;
    property MinSize: NN read FMinSize write FMinSize default 30;
    property ParentColor;
    property OnMoved: TNotifyEvent read FOnMoved write FOnMoved;
  end;

procedure Register;

implementation

uses
  Forms, Graphics;

type
  THack = class(TWinControl);

{ tMover }

constructor tMover.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Align := alLeft;
  Width := 3;
  Cursor := crHSplit;
  FMinSize := 30;
  FBeveled := True;
end;

procedure tMover.Loaded;
begin
  if (Align = alLeft) or (Align = alRight) then Cursor := crHSplit
                                           else Cursor := crVSplit;
end;

procedure tMover.AllocateLineDC;
begin
  FLineDC := GetDCEx(Parent.Handle, 0, DCX_CACHE or DCX_CLIPSIBLINGS
    or DCX_LOCKWINDOWUPDATE);
end;

procedure tMover.DrawLine;
var
  P: TPoint;
begin
  FLineVisible := not FLineVisible;
  P := Point(Left, Top);
  if Align in [alLeft, alRight] then
    P.X := Left + FSplit else
    P.Y := Top + FSplit;
  with P do PatBlt(FLineDC, X, Y, Width, Height, PATINVERT);
end;

procedure tMover.ReleaseLineDC;
begin
  ReleaseDC(Parent.Handle, FLineDC);
end;

procedure tMover.Paint;
var
  FrameBrush: HBRUSH;
  R: TRect;
begin
  R := ClientRect;
  Canvas.Brush.Color := Color;
  Canvas.FillRect(ClientRect);
  if Beveled then
  begin
    if Align in [alLeft, alRight] then
      InflateRect(R, -1, 2) else
      InflateRect(R, 2, -1);
    OffsetRect(R, 1, 1);
    FrameBrush := CreateSolidBrush(ColorToRGB(clBtnHighlight));
    FrameRect(Canvas.Handle, R, FrameBrush);
    DeleteObject(FrameBrush);
    OffsetRect(R, -2, -2);
    FrameBrush := CreateSolidBrush(ColorToRGB(clBtnShadow));
    FrameRect(Canvas.Handle, R, FrameBrush);
    DeleteObject(FrameBrush);
  end;
end;

procedure tMover.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);

  function FindControl: TControl;
  var
    P: TPoint;
    I: Integer;
  begin
    Result := nil;
    P := Point(Left, Top);
    case Align of
      alLeft: Dec(P.X);
      alRight: Inc(P.X, Width);
      alTop: Dec(P.Y);
      alBottom: Inc(P.Y, Height);
    else
      Exit;
    end;
    for I := 0 to Parent.ControlCount - 1 do
    begin
      Result := Parent.Controls[I];
      if PtInRect(Result.BoundsRect, P) then Exit;
    end;
    Result := nil;
  end;

var
  I: Integer;
begin
  inherited;
  if Button = mbLeft then
  begin
    FControl := FindControl;
    FDownPos := Point(X, Y);
    if Assigned(FControl) then
    begin
      if Align in [alLeft, alRight] then
      begin
	FMaxSize := Parent.ClientWidth - FMinSize;
	for I := 0 to Parent.ControlCount - 1 do
	  with Parent.Controls[I] do
	    if Align in [alLeft, alRight] then Dec(FMaxSize, Width);
	Inc(FMaxSize, FControl.Width);
      end
      else
      begin
	FMaxSize := Parent.ClientHeight - FMinSize;
	for I := 0 to Parent.ControlCount - 1 do
	  with Parent.Controls[I] do
	    if Align in [alTop, alBottom] then Dec(FMaxSize, Height);
	Inc(FMaxSize, FControl.Height);
      end;
      UpdateSize(X, Y);
      AllocateLineDC;
      with ValidParentForm(Self) do
	if ActiveControl <> nil then
	begin
	  FActiveControl := ActiveControl;
	  FOldKeyDown := THack(FActiveControl).OnKeyDown;
	  THack(FActiveControl).OnKeyDown := FocusKeyDown;
	end;
      DrawLine;
    end;
  end;
end;

procedure tMover.UpdateSize(X, Y: Integer);
var
  S: Integer;
begin
  if Align in [alLeft, alRight] then
    FSplit := X - FDownPos.X
  else
    FSplit := Y - FDownPos.Y;
  S := 0;
  case Align of
    alLeft: S := FControl.Width + FSplit;
    alRight: S := FControl.Width - FSplit;
    alTop: S := FControl.Height + FSplit;
    alBottom: S := FControl.Height - FSplit;
  end;
  FNewSize := S;
  if S < FMinSize then
    FNewSize := FMinSize
  else if S > FMaxSize then
    FNewSize := FMaxSize;
  if S <> FNewSize then
  begin
    if Align in [alRight, alBottom] then
      S := S - FNewSize else
      S := FNewSize - S;
    Inc(FSplit, S);
  end;
end;

procedure tMover.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
  inherited;
  if Assigned(FControl) then
  begin
    DrawLine;
    UpdateSize(X, Y);
    DrawLine;
  end;
end;

procedure tMover.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited;
  if Assigned(FControl) then
  begin
    DrawLine;
    case Align of
      alLeft: FControl.Width := FNewSize;
      alTop: FControl.Height := FNewSize;
      alRight:
	begin
	  Parent.DisableAlign;
	  try
	    FControl.Left := FControl.Left + (FControl.Width - FNewSize);
	    FControl.Width := FNewSize;
	  finally
	    Parent.EnableAlign;
	  end;
	end;
      alBottom:
	begin
	  Parent.DisableAlign;
	  try
	    FControl.Top := FControl.Top + (FControl.Height - FNewSize);
	    FControl.Height := FNewSize;
	  finally
	    Parent.EnableAlign;
	  end;
	end;
    end;
    StopSizing;
  end;
end;

procedure tMover.FocusKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if Key = VK_ESCAPE then
    StopSizing
  else if Assigned(FOldKeyDown) then
    FOldKeyDown(Sender, Key, Shift);
end;

procedure tMover.SetBeveled(Value: Boolean);
begin
  FBeveled := Value;
  Repaint;
end;

procedure tMover.StopSizing;
begin
  if Assigned(FControl) then
  begin
    if FLineVisible then DrawLine;
    FControl := nil;
    ReleaseLineDC;
    if Assigned(FActiveControl) then
    begin
      THack(FActiveControl).OnKeyDown := FOldKeyDown;
      FActiveControl := nil;
    end;
  end;
  if Assigned(FOnMoved) then
    FOnMoved(Self);
end;

// Registration

procedure Register;
begin
  RegisterComponents('AlexSh', [tMover]);
end;

end.
