unit RegClass;  { v.1.1 beta }

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ntreg ;

type
  TRootRegistryKey = (rgk_HKEY_CLASSES_ROOT,rgk_HKEY_CURRENT_USER,
                      rgk_HKEY_LOCAL_MACHINE,rgk_HKEY_CURRENT_CONFIG);
  TPropFieldType = (pfInteger,pfWord,pfFloat,pfDateTime,pfString,pfBoolean) ;
  TKeyFieldType = (kfString,kfDWord,kfBinary) ;

  TRegistryClass = class ;
  TRegistryValues = class ;           

  TRegistryValue = class (TCollectionItem)
  private
    FPropertyName: string;
    FKeyName: string;
    FDefaultValue: variant;
    FKeyType: TKeyFieldType;
    FPropertyType: TPropFieldType;
    procedure SetPropertyName(const Value: string);
    procedure SetDefaultValue(const Value: variant);
    procedure SetKeyName(const Value: string);
    procedure SetKeyType(const Value: TKeyFieldType);
    procedure SetPropertyType(const Value: TPropFieldType);
  protected
    procedure DefineProperties (Filer : TFiler) ; override ;
    function GetDisplayName : string ; override ;
  public
    constructor Create (Collection : TCollection) ; override ;
    destructor Destroy ; override ;
    procedure Assign (Source : TPersistent) ; override ;
  published
    property PropertyName : string  read FPropertyName write SetPropertyName;
    property KeyName : string  read FKeyName write SetKeyName;
    property DefaultValue : variant  read FDefaultValue write SetDefaultValue;
    property PropertyType : TPropFieldType  read FPropertyType write SetPropertyType;
    property KeyType : TKeyFieldType  read FKeyType write SetKeyType;
  end ;

  TRegistryCollection = class (TCollection)
  private
    FOwner : TRegistryClass ;
    function GetItem(Index: integer): TRegistryValue;
    procedure SetItem(Index: integer; const Value: TRegistryValue);
  protected
    function GetOwner : TPersistent ; override ;
    procedure Update (Item : TCollectionItem) ; override ;
  public
    constructor Create (AOwner : TRegistryClass) ;
    function Add : TRegistryValue ;
    property RegistryClass : TRegistryClass read FOwner ;
    property Items[Index : integer]: TRegistryValue read GetItem write SetItem ; default ;
  end ;

  TRegistryClass = class(TComponent)
  private
    FStorage: TRegistryValues;
    FOwner : TComponent ;
    FLoadOnCreate: boolean;
    FKeys: TStringList;
    FRootKey: TRootRegistryKey;
    FValues: TRegistryCollection;
    FSaveOnExit: boolean;
    procedure SetStorage(const Value: TRegistryValues);
    procedure SetLoadOnCreate(const Value: boolean);
    procedure SetKeys(const Value: TStringList);
    function GetKeyFromList: string;
    procedure SetRootKey(const Value: TRootRegistryKey);
    function GetRootKey: HKEY;
    procedure SetValues(const Value: TRegistryCollection);
    function GetRegWordValue(reg: TNTRegistry; V: TRegistryValue): boolean;
    function GetRegStrValue(reg: TNTRegistry; V: TRegistryValue): boolean;
    function GetRegFloatValue(reg: TNTRegistry;
      V: TRegistryValue): boolean;
    function GetRegDTValue(reg: TNTRegistry; V: TRegistryValue): boolean;
    function GetRegIntValue(reg: TNTRegistry; V: TRegistryValue): boolean;
    procedure SetSaveOnExit(const Value: boolean);
    { Private-Deklarationen}
  protected
    { Protected-Deklarationen}
  public
    { Public-Deklarationen}
    property Storage : TRegistryValues  read FStorage write SetStorage;
    constructor Create (AOwner : TComponent) ; override ;
    procedure Load ;
    procedure Save ;
  published
    { Published-Deklarationen }
    property LoadOnCreate : boolean  read FLoadOnCreate write SetLoadOnCreate;
    property SaveOnExit : boolean  read FSaveOnExit write SetSaveOnExit;
    property Keys : TStringList  read FKeys write SetKeys;
    property RootKey : TRootRegistryKey  read FRootKey write SetRootKey;
    property Values : TRegistryCollection  read FValues write SetValues;
    { Events }
  end;


  { Von dieser Klasse aus muss eigene Registry Klasse abgeleitet werden }
  { welche die Schlsselwerte enthlt }
  TRegistryValues = class (TComponent)
  private
    FRegClass : TRegistryClass ;
    procedure SetRegClass(const Value: TRegistryClass);
  public
    property RegClass : TRegistryClass  read FRegClass write SetRegClass;
    constructor Create (AOwner : TComponent ; RegistryClass : TRegistryClass) ; overload ;
    destructor Destroy ; override ;
    procedure Load ;
    procedure Save ;
  end ;


procedure Register;

implementation
uses TypInfo ;

procedure Register;
begin
  RegisterComponents('Tom', [TRegistryClass]);
end;


{ TRegistryClass }

constructor TRegistryClass.Create(AOwner : TComponent);
begin
     inherited create (AOwner) ;
     FOwner := AOwner ;
     FLoadOnCreate := false ;
     FSaveOnExit := false ;
     FRootKey := rgk_HKEY_CURRENT_USER ;
     FValues := TRegistryCollection.create (Self) ;

     FKeys := TStringList.create ;
end;

function TRegistryClass.GetKeyFromList : string ;
var
 i : integer ;
 k : string ;
begin
     Result := '' ;
     if FKeys.count > 0 then begin
         while FKeys[FKeys.count-1] = '' do FKeys.Delete (FKeys.count-1) ;
         for i := 0 to FKeys.count-1 do begin
            k := FKeys[i] ;
            if k <> '' then begin
               Result := Result + k ;
               if i <> FKeys.count-1 then Result := Result + '\' ;
            end ;
         end ;
     end ;
end ;

function TRegistryClass.GetRootKey : HKEY ;
begin
     Result := 0 ;
     case FRootKey of
       rgk_HKEY_LOCAL_MACHINE : Result := HKEY_LOCAL_MACHINE ;
       rgk_HKEY_CLASSES_ROOT : Result := HKEY_CLASSES_ROOT ;
       rgk_HKEY_CURRENT_CONFIG : Result := HKEY_CURRENT_CONFIG ;
       rgk_HKEY_CURRENT_USER : Result := HKEY_CURRENT_USER ;
     end ;
end ;

function TRegistryClass.GetRegWordValue (reg : TNTRegistry ;
                                        V : TRegistryValue) : boolean ;
var
 PInfo: PPropInfo;
 int_Value : integer ;
begin
     if reg.ValueExists (V.KeyName)
     then int_Value := reg.ReadInteger (V.KeyName)
     else int_Value := V.DefaultValue ;

     PInfo := GetPropInfo(FStorage.ClassInfo, V.PropertyName);
     if PInfo <> nil then begin
       if V.PropertyType = pfWord
       then SetOrdProp(FStorage, PInfo,int_Value) ;
       if V.PropertyType = pfBoolean then begin
          if abs(int_value) = 1
          then SetOrdProp(FStorage,PInfo,1)
          else SetOrdProp(FStorage,PInfo,0) ;
       end ;
       Result := true ;
     end else begin
       Result := false ;
     end ;
end ;

function TRegistryClass.GetRegIntValue (reg : TNTRegistry ;
                                        V : TRegistryValue) : boolean ;
var
 PInfo: PPropInfo;
 int_Value : integer ;
begin
     if reg.ValueExists (V.KeyName)
     then int_Value := reg.ReadInteger (V.KeyName)
     else int_Value := V.DefaultValue ;

     PInfo := GetPropInfo(FStorage.ClassInfo, V.PropertyName);
     if PInfo <> nil then begin
       SetOrdProp(FStorage, PInfo,int_Value) ;
       Result := true ;
     end else begin
       Result := false ;
     end ;
end ;

function TRegistryClass.GetRegStrValue (reg : TNTRegistry ;
                                        V : TRegistryValue) : boolean ;
var
 PInfo: PPropInfo;
 str_Value : string ;
begin
     if reg.ValueExists (V.KeyName)
     then str_Value := reg.ReadString (V.KeyName)
     else str_Value := V.DefaultValue ;

     PInfo := GetPropInfo(FStorage.ClassInfo, V.PropertyName);
     if PInfo <> nil then begin
       SetStrProp(FStorage, PInfo,str_Value);
       Result := true ;
     end else begin
       Result := false ;
     end ;
end ;

function TRegistryClass.GetRegFloatValue (reg : TNTRegistry ;
                                        V : TRegistryValue) : boolean ;
var
 PInfo: PPropInfo;
 fl_Value : extended ;
 i : integer ;
begin
     if reg.ValueExists (V.KeyName)
     then fl_Value := reg.ReadFloat (V.KeyName)
     else fl_Value := V.DefaultValue ;

     PInfo := GetPropInfo(FStorage.ClassInfo, V.PropertyName);
     if PInfo <> nil then begin
       if V.PropertyType = pfFloat
       then SetFloatProp(FStorage, PInfo,fl_Value) ;
       if V.PropertyType = pfInteger
       then SetOrdProp (FStorage,PInfo,Trunc(fl_Value)) ;
       Result := true ;
     end else begin
       Result := false ;
     end ;
end ;

function TRegistryClass.GetRegDTValue (reg : TNTRegistry ;
                                        V : TRegistryValue) : boolean ;
var
 PInfo: PPropInfo;
 dt_Value : TDateTime ;
begin
     if reg.ValueExists (V.KeyName)
     then dt_Value := reg.ReadDateTime (V.KeyName)
     else dt_Value := V.DefaultValue ;

     PInfo := GetPropInfo(FStorage.ClassInfo, V.PropertyName);
     if PInfo <> nil then begin
       SetFloatProp(FStorage, PInfo,dt_Value);
       Result := true ;
     end else begin
       Result := false ;
     end ;
end ;

procedure TRegistryClass.Load;
var
 Key : String ;
 reg : TNTRegistry ;
 i : integer ;
 int_Value : integer ;
 V : TRegistryValue ;
begin
     reg := TNTRegistry.create ;
     reg.RootKey := GetRootKey ;

     if reg.RootKey <> 0 then begin
         Key := GetKeyFromList ;
         reg.OpenKey (Key,true) ;

         for i := 0 to values.count-1 do begin
             V := values[i] ;
             try
               case V.PropertyType of
                  pfWord,
                  pfBoolean     : GetRegWordValue (reg,V) ;
                  pfString      : GetRegStrValue (reg,V) ;
                  pfInteger     : GetRegIntValue (reg,V) ;
                  pfFloat       : GetRegFloatValue (reg,V) ;
                  pfDateTime    : GetRegDTValue (reg,V) ;
               end ;
             except
             end ;
         end ;

         reg.CloseKey ;
     end ;

     reg.free ;
end;

procedure TRegistryClass.Save;
var
 Key : String ;
 reg : TNTRegistry ;
 i : integer ;
 int_Value : integer ;
 V : TRegistryValue ;
 PInfo: PPropInfo;
 fl_Value : extended ;
 str_Value : string ;
begin
     reg := TNTRegistry.create ;
     reg.RootKey := GetRootKey ;

     if reg.RootKey <> 0 then begin
         Key := GetKeyFromList ;
         reg.OpenKey (Key,true) ;

         for i := 0 to values.count-1 do begin
             V := values[i] ;
             PInfo := GetPropInfo(FStorage.ClassInfo,V.PropertyName);
             try
               if PInfo <> nil then begin
                 case V.PropertyType of
                   pfInteger   : int_Value := GetOrdProp (FStorage, PInfo) ;
                   pfFloat,
                   pfDateTime  : fl_Value := GetFloatProp(FStorage, PInfo) ;
                   pfString    : str_Value := GetStrProp (FStorage,PInfo) ;
                   pfWord,
                   pfBoolean   : int_Value := GetOrdProp (FStorage,PInfo) ;
                 end ;

                 case V.PropertyType of
                   pfInteger : reg.WriteInteger (V.KeyName,int_Value) ;
                   pfWord : reg.WriteInteger (V.KeyName,int_value) ;
                   pfFloat : reg.WriteFloat (V.KeyName,fl_Value) ;
                   pfDateTime : reg.WriteDateTime (V.KeyName,fl_Value) ;
                   pfString : reg.WriteString (V.KeyName,str_Value) ;
                   pfBoolean : if abs(int_value) = 1
                               then reg.WriteBool (V.KeyName,true)
                               else reg.WriteBool (V.KeyName,false) ;
                 end ;
               end ;
             except
             end ;
         end ;

         reg.CloseKey ;
     end ;

     reg.free ;
end;

procedure TRegistryClass.SetLoadOnCreate(const Value: boolean);
begin
  FLoadOnCreate := Value;
end;

procedure TRegistryClass.SetKeys(const Value: TStringList);
begin
  FKeys.assign (Value) ;
end;

procedure TRegistryClass.SetRootKey(const Value: TRootRegistryKey);
begin
  FRootKey := Value;
end;

procedure TRegistryClass.SetStorage(const Value: TRegistryValues);
begin
  FStorage := Value;
end;




procedure TRegistryClass.SetValues(const Value: TRegistryCollection);
begin
  FValues.assign (Value) ;
end;

procedure TRegistryClass.SetSaveOnExit(const Value: boolean);
begin
  FSaveOnExit := Value;
end;


{ TRegistryValues }

constructor TRegistryValues.Create(AOwner: TComponent;
  RegistryClass: TRegistryClass);
begin
     inherited Create (Aowner) ;
     RegistryClass.Storage := Self ;
     FRegClass := RegistryClass ;

     if FRegClass.LoadOnCreate then FRegClass.Load ;
end;

destructor TRegistryValues.Destroy;
begin
     if FRegClass.SaveOnExit
     then FRegClass.Save ;
     inherited Destroy ;
end;

procedure TRegistryValues.Load;
begin
     FRegClass.Load ;
end;

procedure TRegistryValues.Save;
begin
     FRegClass.Save ;
end;

procedure TRegistryValues.SetRegClass(const Value: TRegistryClass);
begin
  FRegClass := Value;
end;

{ TRegistryValue }

procedure TRegistryValue.Assign(Source: TPersistent);
begin
     inherited Assign (Source) ;
end;

constructor TRegistryValue.Create(Collection: TCollection);
begin
     inherited Create (Collection) ;
     FPropertyName := '' ;
     PropertyType := pfString ;
end;

procedure TRegistryValue.DefineProperties(Filer: TFiler);
begin
     inherited DefineProperties (Filer) ;
end;

destructor TRegistryValue.Destroy;
begin
     inherited Destroy ;
end;

function TRegistryValue.GetDisplayName: string;
begin
     if (FPropertyName = '') and (FKeyName = '')
     then GetDisplayName := inherited GetDisplayName
     else GetDisplayName := FPropertyName + ' [' + FKeyName + ']' ;
end;

procedure TRegistryValue.SetDefaultValue(const Value: variant);
begin
  FDefaultValue := Value;
end;

procedure TRegistryValue.SetKeyName(const Value: string);
begin
  FKeyName := Value;
end;

procedure TRegistryValue.SetKeyType(const Value: TKeyFieldType);
begin
  //FKeyType := Value;
end;

procedure TRegistryValue.SetPropertyName(const Value: string);
begin
  FPropertyName := Value;
end;

procedure TRegistryValue.SetPropertyType(const Value: TPropFieldType);
begin
  FPropertyType := Value;
  case Value of
    pfWord,
    pfInteger,
    pfBoolean        : FKeyType := kfDWord ;

    pfFloat,
    pfDateTime       : FKeyType := kfBinary ;

    pfString         : FKeyType := kfString ;
  end ;
end;

{ TRegistryCollection }

function TRegistryCollection.Add: TRegistryValue;
begin
     Result := TRegistryValue(inherited Add) ;
end;

constructor TRegistryCollection.Create(AOwner: TRegistryClass);
begin
     inherited Create (TRegistryValue) ;
     FOwner := AOwner ;
end;

function TRegistryCollection.GetItem(Index: integer): TRegistryValue;
begin
     Result := TRegistryValue(inherited GetItem(index)) ;
end;

function TRegistryCollection.GetOwner: TPersistent;
begin
     Result := FOwner ;
end;

procedure TRegistryCollection.SetItem(Index: integer;
  const Value: TRegistryValue);
begin
     inherited SetItem (Index,Value) ;
end;

procedure TRegistryCollection.Update(Item: TCollectionItem);
begin
     //
end;

end.
