{
  RASManager.pas

  Written by Frank Plagge
  Copyright (c) 1998-2002 by Frank Plagge, Elsterweg 39, 38446 Wolfsburg, Germany
  All rights reserved

  *****************************************************************************
  Remote Access external API
  Public header for external API clients
  ras.h   - Copyright (c) 1992-1996, Microsoft Corporation, all rights reserved
  *****************************************************************************

  Please send comments to plagge@wolfsburg.de

  V 1.01 - Aug 7th, 1998
           first implementation, never trust a version 1.00 :-)
  V 1.02 - Aug 23th, 1998
           + added real connected state
             the property Connected represents the connection state from
             RAS services. if the connection state for a service is needed
             simply set the property RASName and next check the poperty
             Connected.
  V 1.03 - Sep 29th, 1998
           + fixed some uninititialized function results
  V 1.04 - Oct 17th, 1998
           + minor changes
           + documentation completed - it's never complete :-)
           + first public release
  V 1.05 - Apr 1st 2002
           + added getting the phone number of a RAS service

  V 1.10 - July 26th 2002
           + complete new interface and internal design. now all available
             ras service can be managed by one instance

  V 1.11 - November 25th 2002
           + little bugfix within TRASManager.GetProperties
             many thanks to Kees Helminck, cfh@lansurvey.com

  *****************************************************************************
  NO REPRESENTATIONS ARE MADE ABOUT THE SUITABILITY OF THIS SOFTWARE FOR ANY
  PURPOSE. IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
  NEITHER FRANK PLAGGE OR ANY OTHER PERSON SHALL BE LIABLE FOR ANY DAMAGES
  SUFFERED BY THE USE OF THIS SOFTWARE.
  *****************************************************************************

  description:
   This is a component for an easy access to ras services. It includes
   connecting and disconnecting services as well as an easy access to
   important properties like connection state, IP numbers of client and server,
   phone number, aerea code, country code, device name, device type and a time
   stamp of the last connection state change. The systems dialogs for

   I know there are much more possibilities with RAS, but there is no time to
   implement all of them.

  *****************************************************************************

  properties and events at design time:

    property OnConnect: TOnConnectionEvent read FOnConnect write FOnConnect;           // connect event
    property OnConnecting: TOnConnectingEvent read FOnConnecting write FOnConnecting;  // disconnect event
    property OnDisconnect: TOnConnectionEvent read FOnDisconnect write FOnDisconnect;  // state change event during connection phase

  *****************************************************************************
   properties and events at runtime:

    constructor Create(AOwner: TComponent); override;  // create a new instance
    destructor Destroy; override;                      // destroy an existing instance
    procedure Connect( Index: Integer );               // connect a ras service
    procedure DisConnect(Index: Integer);              // disconnect a ras service
    function  IndexOf( Name: string ) : Integer;       // get the index of a ras service by name
    procedure Start;                                   // start the autorefresh mechanism
    procedure Stop;                                    // stop the autorefresh mechanism
    property Count: Integer read GetCount;             // get the number of available ras connections
    property AreaCode[Index: Integer]: string read GetAreaCode;        // get area code by index
    property Connected[Index: Integer]: Boolean read GetConnected;     // get connection state by index
    property CountryCode[Index: Integer]: Integer read GetCountryCode; // get countryCode by index
    property DeviceName[Index: Integer]: string read GetDeviceName;    // get device name by index
    property DeviceType[Index: Integer]: string read GetDeviceType;    // get device type by index
    property Handle[Index: Integer]: THRasConn read GetHandle;         // get ras service handle by index
    property IPClient[Index: Integer]: string read GetIPClient;        // get client ip number by index
    property IPServer[Index: Integer]: string read GetIPServer;        // get server ip number by index
    property Name[Index: Integer]: string read GetName;                // get connection name by index
    property PhoneNumber[Index: Integer]: string read GetPhoneNumber;  // get phone number by index
    property TimeStamp[Index: Integer]: TDateTime read GetTimeStamp;   // get state toggle timestamp by index
    property Version: string read GetVersion;

  *****************************************************************************
  other very useful procedures

    function  GetStatusString(State: Integer): string;  // convert state to string message

}
unit RasManager;

interface

uses
  Windows, Messages, SysUtils, Classes, ExtCtrls,
  Ras;

type
  TRasEntry = class
              public
                Handle:      THRasConn;               // connection handle
                DeviceName:  string;                  // name of connection device
                DeviceType:  string;                  // type of conection device
                Name:        string;                  // name of the connection
                IPClient:    string;                  // own IP number if connected
                IPServer:    string;                  // server IP number if connected
                Connected:   Boolean;                 // connection state
                TimeStamp:   TDateTime;               // timestamp connection toggled
                PhoneNumber: string;                  // local phone number
                AreaCode:    string;                  // area code for phone number
                CountryCode: Integer;                 // country code for phone number
                constructor Create( aName: string );  // create a new entry
              end;

  // list of all available ras connections
  TRasList = class (TList )
             public
               destructor Destroy; override;
               procedure  AddName( Name: string );                           // add a new conneection name
               function   GetItem( Name: string ) : TRasEntry;               // get the ras entry
             end;

  // event type for connect and disconnect events
  TOnConnectionEvent = procedure (Sender: TObject; Connection: TRasEntry) of object;

  // asynchronous dialing event
  TOnConnectingEvent = procedure ( Sender: TObject; Index: Integer;
                                   Msg: Integer; State: Integer; Error: Longint) of object;

  // main RASManager component
  TRasManager = class(TComponent)
  private
    FOnConnect: TOnConnectionEvent;     // local connect event
    FOnConnecting: TOnConnectingEvent;  // local event for connecting events (state is changing)
    FOnDisconnect: TOnConnectionEvent;  // local disconnect event
    RasList: TRasList;                  // list of available RAS connections
    Timer: TTimer;                      // autorefresh timer
    function  GetParams(Server: string; var DialParams: TRasDialParams): Boolean;
    function  GetAreaCode(Index: Integer) : string;     // get area code
    function  GetCount: Integer;                        // get number of ras services
    function  GetConnected(Index: Integer) : Boolean;   // get connection state
    function  GetCountryCode(Index: Integer) : Integer; // get connection state
    function  GetDeviceName(Index: Integer) : string;   // get device name
    function  GetDevicetype(Index: Integer) : string;   // get device type
    function  GetHandle(Index: Integer) : THRasConn;    // get ras handle
    function  GetIPClient(Index: Integer) : string;     // get client IP address
    function  GetIPServer(Index: Integer) : string;     // get server IP address
    function  GetName(Index: Integer) : string;         // get connection name
    function  GetPhoneNumber(Index: Integer): string;   // get phone number
    procedure GetProperties;                            // get device name, device type, area code phone number
    function  GetTimeStamp(Index: Integer) : TDateTime; // get connection timestamp
    function  GetVersion: string;                       // get internal version number
    procedure OnTimer( Sender: TObject );               // local timer method for auto refresh
    procedure RefreshConnections;                       // refresh state of ras services
  protected
  public
    constructor Create(AOwner: TComponent); override;  // create a new instance
    destructor Destroy; override;                      // destroy an existing instance
    procedure Connect( Index: Integer );               // connect a ras service
    procedure DisConnect(Index: Integer);              // disconnect a ras service
    function  IndexOf( Name: string ) : Integer;       // get the index of a ras service by name
    procedure Start;                                   // start the autorefresh mechanism
    procedure Stop;                                    // stop the autorefresh mechanism
    property Count: Integer read GetCount;             // get the number of available ras connections
    property AreaCode[Index: Integer]: string read GetAreaCode;        // get area code by index
    property Connected[Index: Integer]: Boolean read GetConnected;     // get connection state by index
    property CountryCode[Index: Integer]: Integer read GetCountryCode; // get countryCode by index
    property DeviceName[Index: Integer]: string read GetDeviceName;    // get device name by index
    property DeviceType[Index: Integer]: string read GetDeviceType;    // get device type by index
    property Handle[Index: Integer]: THRasConn read GetHandle;         // get ras service handle by index
    property IPClient[Index: Integer]: string read GetIPClient;        // get client ip number by index
    property IPServer[Index: Integer]: string read GetIPServer;        // get server ip number by index
    property Name[Index: Integer]: string read GetName;                // get connection name by index
    property PhoneNumber[Index: Integer]: string read GetPhoneNumber;  // get phone number by index
    property TimeStamp[Index: Integer]: TDateTime read GetTimeStamp;   // get state toggle timestamp by index
    property Version: string read GetVersion;
  published
    property OnConnect: TOnConnectionEvent read FOnConnect write FOnConnect;           // connect event
    property OnConnecting: TOnConnectingEvent read FOnConnecting write FOnConnecting;  // disconnect event
    property OnDisconnect: TOnConnectionEvent read FOnDisconnect write FOnDisconnect;  // state change event during connection phase
  end;

procedure Register;                                 // component registration
function  GetStatusString(State: Integer): string;  // convert state to string message

implementation

const
  cIPEmpty = '0.0.0.0';   // default empty IP number
  cRegisterName = 'FPL';
  cVersion = '1.11';

var
  RasSave:  TRasManager;  // class var for callback function at asynch dial
  RASIndex: Integer;      // class var for ras service index at asynch dial

// ***************************************************************************
// register the component
procedure Register;
begin
  RegisterComponents( cRegisterName, [TRasManager]);
end;

// ***************************************************************************
// convert a ras state to a text message
function GetStatusString(State: Integer): string;
begin
  case State of
      RASCS_OpenPort:            Result := 'Opening port';
      RASCS_PortOpened:          Result := 'Port opened';
      RASCS_ConnectDevice:       Result := 'Connecting device';
      RASCS_DeviceConnected:     Result := 'Device connected';
      RASCS_AllDevicesConnected: Result := 'All devices connected';
      RASCS_Authenticate:        Result := 'Start authenticating';
      RASCS_AuthNotify:          Result := 'Authentication: notify';
      RASCS_AuthRetry:           Result := 'Authentication: retry';
      RASCS_AuthCallback:        Result := 'Authentication: callback';
      RASCS_AuthChangePassword:  Result := 'Authentication: change password';
      RASCS_AuthProject:         Result := 'Authentication: projecting';
      RASCS_AuthLinkSpeed:       Result := 'Authentication: link speed';
      RASCS_AuthAck:             Result := 'Authentication: acknowledge';
      RASCS_ReAuthenticate:      Result := 'Authentication: reauthenticate';
      RASCS_Authenticated:       Result := 'Authenticated';
      RASCS_PrepareForCallback:  Result := 'Preparing for callback';
      RASCS_WaitForModemReset:   Result := 'Waiting for modem reset';
      RASCS_WaitForCallback:     Result := 'Waiting for callback';
      RASCS_Projected:           Result := 'Projected';
      RASCS_StartAuthentication: Result := 'Start authentication';
      RASCS_CallbackComplete:    Result := 'Callback complete';
      RASCS_LogonNetwork:        Result := 'Logging on network';
      RASCS_Interactive:         Result := 'Interactive';
      RASCS_RetryAuthentication: Result := 'Retry Authentication';
      RASCS_CallbackSetByCaller: Result := 'Callback set by caller';
      RASCS_PasswordExpired:     Result := 'Password expired';
      RASCS_Connected:           Result := 'Connected';
      RASCS_Disconnected:        Result := 'Disconnected';
    else                         Result := 'Unknown state';
  end;
end;

// ***************************************************************************
// callback function if asynchronous dialing is enabled
procedure RASCallback(Msg: Integer; State: TRasConnState; Error: Longint); stdcall
begin
  // if a connecting event is defined the event is called
  if Assigned(RASSave.FOnConnecting) then begin
    RASSave.FOnConnecting( RASSave, RASIndex, Msg, State, Error );
  end;
end;

// create a new ras service entry and initialize it
constructor TRasEntry.Create( aName: string );
begin
  inherited Create;    // create to object itself
  Handle := 0;         // do the initialization
  DeviceName:= '';
  DeviceType := '';
  Name := aName;
  IPClient := cIPEmpty;
  IPServer := cIPEmpty;
  Connected := false;
  TimeStamp := now;    // get the actual time and date
  Phonenumber := '';
  AreaCode := '';
  CountryCode := 0;
end;

// ***************************************************************************
// destroy the instance of a ras service list
destructor TRasList.Destroy;
var thisItem: TRasEntry;    // single ras service entry
    i: Integer;             // loop counter
begin
  for i :=1 to Count do begin // for all entries
    thisItem := Items[i-1];   // get the entry
    thisItem.Free;            // destroy the entry
  end;
  inherited Destroy;          // destroy the instance itself
end;

// ***************************************************************************
// add a new ras service name
procedure TRasList.AddName( Name: string );
var thisItem: TRasEntry;
begin
  if GetItem( Name ) = nil then begin       // if the name is really new
    thisItem := TRasEntry.Create( Name );   // create a new ras list entry
    Add( thisItem );                        // add this entry the the list
  end;
end;

// ***************************************************************************
// get a list item by name
function TRasList.GetItem( Name: string ) : TRasEntry;
var thisItem: TRasEntry;
    i: Integer;
begin
  Result := nil;                          // the default result is 'not found'
  for i :=1 to Count do begin             // for all entries
    thisItem := Items[i-1];               // the the entry
    if thisItem.Name = Name then begin    // if the search name was found
      Result := thisItem;                 // this entry is the result
      Break;                              // leave the loop
    end;
  end;
end;

// ***************************************************************************
// create a new ras manager instance
constructor TRasManager.Create(AOwner: TComponent);
const cMaxRas = 100;                            // maximum number of ras services
var BufferSize: LongInt;                        // used for size of result buffer
    RASNames:   array[1..cMaxRas] of TRasEntryName; // the API result buffer itself
    RASCount:   LongInt;                         // number of found ras services
    i:           Integer;                        // loop counter
begin
  inherited Create( AOwner );      // create the instance
  RasList := TRasList.Create;      // create an empty ras service list

  // get all available RAS names
  FillChar( RASNames, SizeOf(RASNames), 0 );    // clear the API Buffer
  RASNames[1].dwSize := SizeOf(TRasEntryName);  // set the API buffer size for a single record
  BufferSize := SizeOf(TRasEntryName) * cMaxRas;// calc complete buffer size
  // if the API call causes no error
  if RasEnumEntries(nil, nil, @RASNames[1], BufferSize, RASCount) = 0 then begin
    for i := 1 to RASCount do begin                 // for all found ras services
      RasList.AddName( RasNames[i].szEntryName );   // save the name of the ras service
    end;
  end;

  RefreshConnections;                               // refresh the ras connection states
                                                    // this call will generate the connect events for
                                                    // already connected services

  GetProperties;                   // get device name, device type, phone number, area code

  Timer := TTimer.Create( self );  // create the autorefresh timer
  Timer.Enabled := false;          // it is not active
  Timer.Interval := 1000;          // the timer interval is 1 second
  Timer.OnTimer := OnTimer;        // bind the timer event method
end;

// ***************************************************************************
// destroy an instance of a RASManager
destructor TRasManager.Destroy;
begin
  RasList.Free;        // destroy the list of ras services
  Timer.Free;          // destroy the autorefresh timer
  inherited Destroy;   // destroy the component itself
end;

// ***************************************************************************
// connect a not connected ras service
procedure TRASManager.Connect(Index: Integer);
var DialParams: TRasDialParams;  // local dial parameters
    RASResult:  LongInt;         // used result for RASDial call, not used
    thisEntry:  TRasEntry;       // the ras service entry
begin
  thisEntry := RasList.Items[Index];                      // get the ras service entry
  if not thisEntry.Connected then begin                   // only if the service is not connected
    if GetParams( thisEntry.Name, DialParams ) then begin // get actual dial parameters
      RASSave := self;                                    // save the object itself for callback function
      RASIndex := Index;                                  // save the ras service index for callback function
      RASResult := RasDial(nil, nil, DialParams, 0, @RASCallback, thisEntry.Handle ); // call with a callback function
    end;
  end;
end;

// ***************************************************************************
// disconnect the already connected ras service
procedure TRASManager.DisConnect(Index: Integer);
var thisEntry:  TRasEntry;
begin
  thisEntry := RasList.Items[Index];      // get the ras entry from internal list
  if thisEntry.Connected then begin       // only if a connection is available
    if thisEntry.Handle<>0 then begin     // only if a vaild handle is available
      RasHangup( thisEntry.Handle );      // hangup the ras service
    end;
  end;
end;

// ***************************************************************************
// get the dial parameters for a ras server
function TRASManager.GetParams(Server: string; var DialParams: TRasDialParams): Boolean;
var DialPassword: LongBool;
    RASResult:    LongInt;
begin
  Result := true;                                     // result is first vaild
  FillChar( DialParams, SizeOf(TRasDialParams), 0);   // clear the result record
  DialParams.dwSize := Sizeof(TRasDialParams);        // set the result array size
  StrPCopy(DialParams.szEntryName, Server);           // set the ras service name
  DialPassword := true;                               // get the dial password
  RASResult := RasGetEntryDialParams(nil, DialParams, DialPassword); // read the ras parameters
  if (RASResult<>0) then begin                        // if the API call was not successful
    Result := false;                                  // result is not vaild
  end;
end;

// ***************************************************************************
// get the index of a ras service name
function TRasManager.IndexOf( Name: string ) : Integer;
var i: Integer;
begin
  Result := -1;                                             // default result if 'not found'
  for i := 1 to RasList.Count do begin                      // for all available ras services
    if TRasEntry(RasList.Items[i-1]).Name = Name then begin // if the name was found
      Result := i-1;                                        // save the index
      Break;                                                // leave the loop
    end;
  end;
end;

// ***************************************************************************
// index based wrapper to RasList entries
function TRasManager.GetAreaCode(Index: Integer) : string;
begin
  Result := TRasEntry(RasList.Items[Index]).AreaCode;
end;

// ***************************************************************************
// get the number of available ras services
function TRasManager.GetCount: Integer;
begin
  Result := RasList.Count;  // wrapper to ras service list counter
end;

// ***************************************************************************
// index based wrapper to RasList entries
function TRasManager.GetConnected(Index: Integer) : Boolean;
begin
  Result := TRasEntry(RasList.Items[Index]).Connected;
end;

// ***************************************************************************
// index based wrapper to RasList entries
function TRasManager.GetCountryCode(Index: Integer) : Integer;
begin
  Result := TRasEntry(RasList.Items[Index]).CountryCode;
end;

// ***************************************************************************
// index based wrapper to RasList entries
function TRasManager.GetDeviceName(Index: Integer) : string;
begin
  Result := TRasEntry(RasList.Items[Index]).DeviceName;
end;

// ***************************************************************************
// index based wrapper to RasList entries
function TRasManager.GetDeviceType(Index: Integer) : string;
begin
  Result := TRasEntry(RasList.Items[Index]).DeviceType;
end;

// ***************************************************************************
// index based wrapper to RasList entries
function TRasManager.GetHandle(Index: Integer) : THRasConn;
begin
  Result := TRasEntry(RasList.Items[Index]).Handle;
end;

// ***************************************************************************
// index based wrapper to RasList entries
function TRasManager.GetIPClient(Index: Integer) : string;
begin
  Result := TRasEntry(RasList.Items[Index]).IPClient;
end;

// ***************************************************************************
function TRasManager.GetIPServer(Index: Integer) : string;
// index based wrapper to RasList entries
begin
  Result := TRasEntry(RasList.Items[Index]).IPServer;
end;

// ***************************************************************************
// index based wrapper to RasList entries
function TRasManager.GetName(Index: Integer) : string;
begin
  Result := TRasEntry(RasList.Items[Index]).Name;
end;

// ***************************************************************************
// index based wrapper to RasList entries
function TRasManager.GetPhoneNumber(Index: Integer): string;
begin
  Result := TRasEntry(RasList.Items[Index]).PhoneNumber;
end;


// ***************************************************************************
// get device name, device type, phone number, aera code, country code
procedure TRASManager.GetProperties;
var thisEntry: TRasEntry;       // entry from internal list
    Entry: LPRasEntry;          // single ras entry
    EntrySize: Integer;         // size of a single ras service entry
    DeviceInfoSize: Integer;    // used for api call
    i: Integer;                 // loop counter
    RasResult: LongInt;         // result of ras api call
begin
  for i := 1 to RasList.Count do begin    // for all available ras entries do
    thisEntry := RasList.Items[i-1];      // get the ras entry from internal list
    EntrySize := 0;                       // init the entry size
    DeviceInfoSize := 0;                  // init the devive info size
    // first get the real needed buffer sizes into EntrySize and DeviceInfoSize
    if RasGetEntryProperties(nil, PChar(thisEntry.Name), nil, EntrySize, nil, DeviceInfoSize) <> ERROR_BUFFER_TOO_SMALL then begin
      raise Exception.Create( 'Call of RasGetEntryProperties failed for "' + thisEntry.Name+ '"' );
      Exit;
    end;

    if EntrySize < SizeOf(Ras.TRasEntry) then begin  // if the entry size doesn't fit (possibly Windows ME)
      EntrySize := SizeOf(Ras.TRasEntry);            // set the needed size for memory allocation
    end;

    Entry := AllocMem(EntrySize);              // allocate the needed buffer size
    try
      Entry^.dwSize := SizeOf(Ras.TRasEntry); // set the size of an entry
     // Get the properties for the needed RAS entry
      RasResult := RasGetEntryProperties(nil, PChar(thisEntry.Name), Entry, EntrySize, nil, DeviceInfoSize);
      if RasResult = 0 then begin                            // if the api call was successful
        thisEntry.Phonenumber := Entry^.szLocalPhoneNumber;  // save ras service informations
        thisEntry.AreaCode := Entry^.szAreaCode;
        thisEntry.DeviceName := Entry^.szDeviceName;
        thisEntry.DeviceType := Entry^.szDeviceType;
        thisEntry.CountryCode := Entry^.dwCountryCode;
      end;
    finally
      FreeMem(Entry);  // free the formerly allocated memory
    end;
  end;
end;

// ***************************************************************************
// index based wrapper to RasList entries
function TRasManager.GetTimeStamp(Index: Integer) : TDateTime;
begin
  Result := TRasEntry(RasList.Items[Index]).TimeStamp;
end;

// ***************************************************************************
// autorefresh mechanism
function TRasManager.GetVersion: string;                       // get internal version number
begin
  Result := cVersion;
end;

// ***************************************************************************
// autorefresh mechanism
procedure TRasManager.OnTimer( Sender: TObject );
begin
  RefreshConnections; // every timer event the connection state is refreshed
end;

// ***************************************************************************
// get all available ras conections and save them within RASList
procedure TRasManager.RefreshConnections;
const cMaxRas = 100;                             // maximum number of ras services
var BufferSize: LongInt;                         // used for size of result buffer
    RASConnect: array[1..cMaxRas] of TRasConn;   // the API result buffer itself
    RASCount:   LongInt;                         // number of found ras services
    i:          Integer;                         // loop counter
    thisItem:   TRasEntry;                       // single ras service
    ConnList:   TStringList;                     // list of available connections
    RASInfo:  DWord;                             // result of gettimg IP addresses
    RASPppIp: TRASPppIp;                         // result record of getting IP addresses
    lpcp: LongInt;                               // IP address result size

begin
  ConnList := TStringList.Create;                    // create list of found connections
  FillChar( RASConnect, SizeOf(RASConnect), 0 );     // clear the API result buffer
  RASConnect[1].dwSize := SizeOf(TRasConn);          // set the API buffer size for a single result record
  BufferSize := SizeOf(TRasConn) * cMaxRas;          // calc complete buffer size
  // if the API call causes no error
  if RasEnumConnections(@RASConnect[1], BufferSize, RASCount) = 0 then begin
    for i := 1 to RASCount do begin                  // for all found ras services
      ConnList.Add(RasConnect[i].szEntryName);       // save RAS name to the list of established connections
      thisItem := RASList.GetItem( RasConnect[i].szEntryName );
      thisItem.Handle := RASConnect[i].hrasconn;                 // save the belonging ras handle
//      RasList.SetDeviceName( RASConnect[i].szEntryName, StrPas(RasConnect[i].szDeviceName ) ); // save the belonging device name
//      RasList.SetDeviceType( RASConnect[i].szEntryName, StrPas(RasConnect[i].szDeviceType) ); // save the belonging device type

      // get IP Numbers of connection
      FillChar(RASPppIp,SizeOf(TRASPppIp), 0);   // clear API result buffer
      RASPppIp.dwSize:=SizeOf(TRASPppIp);        // set the API buffer size for a single result record
      lpcp := RASPppIp.dwSize;                   // same again
      RASInfo := RASGetProjectionInfo(RASConnect[i].hrasconn,RASP_PppIp,@RASPppIp,lpcp);  // get the IP addresses for this ras handle
      if RASInfo = 0 then begin                  // if the API call was successful
        thisItem.IPClient := StrPas(RASPppIp.szIpAddress);       // save the client address
        thisItem.IPServer := StrPas(RASPppIp.szServerIpAddress); // save the server address
      end else begin                             // if the API call wa snot succesful
        thisItem.IPClient := cIPEmpty;           // save an empty default address
        thisItem.IPServer := cIPEmpty;           // save an empty default address
      end;
    end;
  end;

  for i := 1 to RasList.Count do begin                     // for all available ras services do
    thisItem := RasList.Items[i-1];                        // get the ras service entry
    if thisItem.Connected then begin                       // is fromerly the service was sconnected
      if ConnList.IndexOf(thisItem.Name) < 0 then begin    // and now the service is not within the list of established connections
        thisItem.Connected := false;                       // it is no longer connected
        thisItem.DeviceName := '';                         // clear the device name
        thisItem.DeviceType := '';                         // clear the device type
        thisItem.Handle := 0;                              // clear the service handle
        thisItem.IPClient := cIPEmpty;                     // set the client IP address to a default empty address
        thisItem.IPServer := cIPEmpty;                     // set the server IP address to a default empty address
        thisItem.TimeStamp := now;                         // reinit the time stamp
        if Assigned(FOnDisConnect) then begin              // if a disconnect event is assigned
          FOnDisConnect( self, thisItem );                 // call the assigned event
        end;
      end;
    end else begin                                         // if fromerly the service was not connected
      if ConnList.IndexOf(thisItem.Name) >= 0 then begin   // and now it is connected
        thisItem.Connected := true;                        // set the new connection state
        thisItem.TimeStamp := now;                         // reinit the time stamp because state is toggled
        if Assigned(FOnConnect) then begin                 // if a connect event is assiged
          FOnConnect( self, thisItem );                    // call the assigned event
        end;
      end;
    end;
  end;

  ConnList.Free;  // destroy local list of connections
end;

// ***************************************************************************
// start the autorefresh mechanism
procedure TRasManager.Start;
var thisItem: TRasEntry;  // a single ras service entry
    i: Integer;
begin
  for i := 1 to RasList.Count do begin   // for all available ras service entries
    thisItem := RasList.Items[i-1];      // get the ras service entry
    if thisItem.Connected then begin     // if the ras service is connected
      if Assigned(FOnConnect) then begin // if a connect event is assiged
        FOnConnect( self, thisItem );    // call the assigned event
      end;
    end;
  end;
  Timer.Enabled := true;                  // enable the autorefersh timer
end;

// ***************************************************************************
// stop the autorefresh mechanism
procedure TRasManager.Stop;
begin
  Timer.Enabled := false;     // disable the autorefresh timer
end;

end.
