{******************************************}
{                                          }
{                 PReport v1.5             }
{                                          }
{ Copyright (c) 1999-2002 by Manuzin A.    }
{                                          }
{******************************************}

unit pr_TxPreview;

{$i pr.inc}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  RichEdit, StdCtrls, ImgList, ActnList, ComCtrls, math, ToolWin, ExtCtrls,
  IniFiles, {$ifdef PR_D6} types, {$endif}

  pr_Common, pr_TxClasses, pr_MultiLang;

type
  ///////////////////////////////
  //
  // TprTxPreviewBox
  //
  ///////////////////////////////
  TprTxPreviewBox = class(TScrollBox)
  private
    FCharHeight : integer;
    FCharWidth : integer;
    FMaxLineLength : integer;
    FText : TStrings;

    FTopLine : integer;
    FLeftCol : integer;

    FFullLinesOnPage : integer;
    FFullColsOnPage : integer;

    FCurLine : integer;
    FCurCol : integer;

    FApiFont : HFONT;
    FPagePen : HPEN;
    FCaretCreated : boolean;

    FPagesList : TList;

    FLastFoundCol : integer;
    FLastFoundLine : integer;

    FOnCursorPosChanged : TNotifyEvent;

    FLastHighlightedRec : pTextDeviceRec;

    function GetCurPage : integer;
    function GetPagesCount : integer;
    function GetLinesCount : integer;

    procedure DoCursorPosChanged;
    procedure MakeCursorVisible;
    procedure mEnter(Sender : TObject);
    function FindText(FindText : string; IsCase,IsDown : boolean) : boolean;
    procedure GetPointInfoAt(x,y : integer; var rec : pTextDeviceRec);
    procedure InvertRec(DC : HDC; rec : pTextDeviceRec);
  protected
    procedure UpdateCursor;

    procedure Resize; override;
    procedure mKeyDown(var Message : TWMKeyDown); message WM_KEYDOWN;
    procedure mPaint(var message : TWMPaint); message WM_PAINT;
    procedure mGetDlgCode(var Message : TWMGetDlgCode); message WM_GETDLGCODE;
    procedure mVScroll(var Message: TWMVScroll); message WM_VSCROLL;
    procedure mHScroll(var Message: TWMHScroll); message WM_HSCROLL;
    procedure mLButtonDown(var Message: TWMLButtonDown); message WM_LBUTTONDOWN;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure DblClick; override;
  public
    Report : TprTxReport;

    procedure UpdateSettings;
    procedure SetToNewReport;
    procedure SetCaretPos(x,y : integer);
    procedure GotoPage(p : integer);

    property  CurLine : integer read FCurLine;
    property  CurCol : integer read FCurCol;
    property  CurPage : integer read GetCurPage;
    property  PagesCount : integer read GetPagesCount;
    property  LinesCount : integer read GetLinesCount;
    property  OnCursorPosChanged : TNotifyEvent read FOnCursorPosChanged write FOnCursorPosChanged;

    constructor CreatePreview(AOwner : TComponent; _Report : TprTxReport);
    destructor Destroy; override;
  end;

  ///////////////////////////////
  //
  // TprTxPreviewForm
  //
  ///////////////////////////////
  TprTxPreviewForm = class(TprPreview)
    ActionList: TActionList;
    aPrint: TAction;
    aFind: TAction;
    aClose: TAction;
    aSave: TAction;
    aOpen: TAction;
    ImageList: TImageList;
    SaveDialog: TSaveDialog;
    OpenDialog: TOpenDialog;
    SB: TStatusBar;
    FindDialog: TFindDialog;
    aFontOptions: TAction;
    FontDialog: TFontDialog;
    aCustomAction: TAction;
    prMLRes1: TprMLRes;
    ToolBar1: TToolBar;
    ToolButton1: TToolButton;
    ToolButton12: TToolButton;
    ToolButton2: TToolButton;
    ToolButton3: TToolButton;
    ToolButton4: TToolButton;
    ToolButton5: TToolButton;
    Label1: TLabel;
    EDLine: TComboBox;
    Label2: TLabel;
    EDPage: TComboBox;
    ToolButton6: TToolButton;
    ToolButton7: TToolButton;
    ToolButton8: TToolButton;
    ToolButton10: TToolButton;
    ToolButton11: TToolButton;
    ToolButton9: TToolButton;
    ToolButton13: TToolButton;
    ToolButton14: TToolButton;
    aExportToTXT: TAction;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure aPrintExecute(Sender: TObject);
    procedure aOpenExecute(Sender: TObject);
    procedure aSaveExecute(Sender: TObject);
    procedure EDLineKeyPress(Sender: TObject; var Key: Char);
    procedure EDPageKeyPress(Sender: TObject; var Key: Char);
    procedure EDPageClick(Sender: TObject);
    procedure EDLineClick(Sender: TObject);
    procedure aFindExecute(Sender: TObject);
    procedure FindDialogFind(Sender: TObject);
    procedure aCloseExecute(Sender: TObject);
    procedure aFontOptionsExecute(Sender: TObject);
    procedure aCustomActionExecute(Sender: TObject);
    procedure aCustomActionUpdate(Sender: TObject);
    procedure aExportToTXTExecute(Sender: TObject);
  private
    { Private declarations }
    Preview   : TprTxPreviewBox;

    procedure UpdateCaretPos;

    procedure CursorPosChanged(Sender : TObject);
  protected
    procedure prRestoreProperties(Ini : TIniFile; sn : string); override;
    procedure prSaveProperties(Ini : TIniFile; sn : string); override;
  public
    { Public declarations }
  end;

implementation

uses pr_Strings, pr_TxConsts, pr_Utils, pr_TxUtils;

{$R *.DFM}

/////////////////////////////////
//
// TprTxPreviewBox
//
/////////////////////////////////
constructor TprTxPreviewBox.CreatePreview;
begin
Report        :=_Report;
FApiFont      :=0;
FPagePen      :=0;
FCaretCreated :=false;
FText         :=Report.TextDevice.SList;
FPagesList    :=TList.Create;

FLastFoundCol :=-1;
FLastFoundLine:=-1;

inherited Create(AOwner);

TabStop:=true;
OnEnter:=mEnter;
end;

destructor TprTxPreviewBox.Destroy;
begin
if FCaretCreated then
  DestroyCaret;
if FApiFont<>0 then
  DeleteObject(FApiFont);
if FPagePen<>0 then
  DeleteObject(FPagePen);
FPagesList.Free;

inherited;
end;

procedure TprTxPreviewBox.DoCursorPosChanged;
begin
if Assigned(OnCursorPosChanged) then
  OnCursorPosChanged(Self);
end;

procedure TprTxPreviewBox.Resize;
begin
inherited;
if FCharHeight=0 then
  FFullLinesOnPage:=0
else
  FFullLinesOnPage:=ClientHeight div FCharHeight;
if FCharWidth=0 then
  FFullColsOnPage:=0
else
  FFullColsOnPage:=ClientWidth div FCharWidth;
end;

procedure TprTxPreviewBox.UpdateCursor;
begin
FCaretCreated:=CreateCaret(Handle,0,FCharWidth,FCharHeight);

if ShowCaret(Handle) then
  begin
    Windows.SetCaretPos(FCurCol*FCharWidth-HorzScrollBar.Position,
                        FCurLine*FCharHeight-VertScrollBar.Position);
  end;
end;

procedure TprTxPreviewBox.UpdateSettings;
var
  DC : HDC;
  tb : tagLOGBRUSH;
  i,l : integer;
  OldFont : HFONT;
  SymbolSize : tagSize;
begin
FPagesList.Clear;
ParsePages(FText,false,0,FPagesList);

// ApiFont
if FApiFont<>0 then DeleteObject(FApiFont);
FApiFont           :=CreateAPIFont(Font);

// PagePen
if FPagePen=0 then
  begin
    tb.lbStyle := BS_SOLID;
    tb.lbColor := clBlue;
    FPagePen := ExtCreatePen(PS_GEOMETRIC+PS_ENDCAP_SQUARE+PS_SOLID,2,tb,0,nil);
  end;

//  
DC     :=GetDC(Handle);
OldFont:=SelectObject(DC,FApiFont);
GetTextExtentPoint32(DC,'1',1,SymbolSize);
SelectObject(DC,OldFont);

FCharWidth   :=SymbolSize.cx;
FCharHeight  :=SymbolSize.cy-1;

//   
FMaxLineLength:=0;
for i:=0 to FText.Count-1 do
  begin
    l:=Length(FText[i]);
    if l>FMaxLineLength then
      FMaxLineLength:=l;
  end;

// 
VertScrollBar.Position :=0;
HorzScrollBar.Position :=0;
VertScrollBar.Range    :=FCharHeight*FText.Count;
HorzScrollBar.Range    :=FCharWidth*FMaxLineLength;
VertScrollBar.Increment:=FCharHeight;
HorzScrollBar.Increment:=FCharWidth;

//  
FCurLine:=0;
FCurCol :=0;

FTopLine:=0;
FLeftCol:=0;

FFullLinesOnPage:=ClientHeight div FCharHeight;
FFullColsOnPage :=ClientWidth div FCharWidth;

UpdateCursor;
end;

procedure TprTxPreviewBox.SetToNewReport;
begin
UpdateSettings;
Repaint;
end;

procedure TprTxPreviewBox.mEnter;
begin
UpdateCursor;
end;

procedure TprTxPreviewBox.mGetDlgCode;
begin
inherited;
Message.Result:=Message.Result or DLGC_WANTARROWS;
end;

procedure TprTxPreviewBox.InvertRec;
var
  r : TRect;
begin
r.Left := rec.rPlace.Left*FCharWidth-HorzScrollBar.ScrollPos;
//r.Top := (rec.rPlace.Top-Max(VertScrollBar.ScrollPos div FCharHeight,0))*FCharHeight-VertScrollBar.ScrollPos;
r.Top := rec.rPlace.Top*FCharHeight-VertScrollBar.ScrollPos;
r.Right := r.Left+(rec.rPlace.Right-rec.rPlace.Left+1)*FCharWidth;
r.Bottom := r.Top+(rec.rPlace.Bottom-rec.rPlace.Top+1)*FCharHeight;
InvertRect(DC,r);
end;

procedure TprTxPreviewBox.mPaint;
var
  DC : hDC;
  PS : TPaintStruct;
  Buf : string;
  rgn : HRGN;
  OldPen : HPEN;
  OldFont : HFONT;
  fInvert : boolean;
  i,p,j,si,ei,li,fLine,tLine,CurY : integer;
begin
DC := BeginPaint(Handle,PS);

rgn := CreateRectRgnIndirect(ps.rcPaint);
OldFont := SelectObject(DC,FApiFont);
OldPen := SelectObject(DC,FPagePen);
fInvert := true;
SelectClipRgn(DC,rgn);
try
  fLine := Max(VertScrollBar.ScrollPos div FCharHeight,0);
  tLine := Min(fLine+FFullLinesOnPage+1,FText.Count-1);
  CurY := fLine*FCharHeight-VertScrollBar.ScrollPos;

  //       , 
  //     fLine  tLine, ..    
  //       fLine   fLine
  si := 1;
  ei := Max(1,FPagesList.Count-1);
  p := -1;
  while si<>ei do
    begin
      p := (ei+si) div 2;
      li := integer(FPagesList[p]);
      if (li>=fLine) and (li<=tLine) then
        begin
          //    FPagesList      li<fLine
          while (p>0) and (integer(FPagesList[p])>=fLine) do Dec(p);
          Inc(p);
          break;
        end
      else
        if li<fLine then
          begin
            si := Min(ei,p+1);
          end
        else
          if li>tLine then
            begin
              ei := Max(si,p-1);
            end;
    end;
  if si=ei then
    p := si;

  SetBkMode(DC,TRANSPARENT);
  for i:=fLine to tLine do
    begin
      Buf := RemoveESCFromString(FText[i]);

      TextOut(DC,-HorzScrollBar.ScrollPos,CurY,PChar(Buf),Length(Buf));

      if p<>-1 then
        begin
          j := p;
          while (j<FPagesList.Count) and
                (integer(FPagesList[j])<=tLine) and
                (integer(FPagesList[j])<>i) do Inc(j);
          if (j<FPagesList.Count) and (integer(FPagesList[j])=i) then
            begin
              // j -  
              MoveToEx(DC,0,CurY-1,nil);
              LineTo(DC,ClientWidth,CurY-1);
            end;
        end;
      if fInvert and (FLastHighlightedRec<>nil) and (i>=FLastHighlightedRec.rPlace.Top) and (i>=FLastHighlightedRec.rPlace.Bottom) then
        begin
          InvertRec(DC,FLastHighlightedRec);
          fInvert := false;
        end;

      CurY := CurY+FCharHeight;
    end;

  SetBkMode(DC,OPAQUE);
finally
  SelectClipRgn(DC,0);
  DeleteObject(rgn);
  SelectObject(DC,OldFont);
  SelectObject(DC,OldPen);
  EndPaint(Handle,PS);
end;
end;

procedure TprTxPreviewBox.mKeyDown;
var
  ss : TShiftState;
  OldCol,OldLine,OldTopLine,OldLeftCol : integer;
begin
ss := KeyDataToShiftState(Message.KeyData);
OldCol := FCurCol;
OldLine := FCurLine;
OldTopLine := FTopLine;
OldLeftCol := FLeftCol;

if ss=[] then
  case Message.CharCode of
    VK_HOME:
      begin
        FCurCol :=0;
        FLeftCol:=0;
      end;
    VK_END:
      begin
        FCurCol:=Max(0,Length(FText[FCurLine])-1);
        if FCurCol>=FFullColsOnPage then
          FLeftCol:=FCurCol-FFullColsOnPage+1;
      end;
    VK_DOWN:
      begin
        if FCurLine<FText.Count-1 then
          Inc(FCurLine);
        FTopLine:=FTopLine+integer(not ((FCurLine>=FTopLine) and (FCurLine<FTopLine+FFullLinesOnPage)));
      end;
    VK_UP:
      begin
        if FCurLine>0 then
          Dec(FCurLine);
        FTopLine:=FTopLine-integer(not ((FCurLine>=FTopLine) and (FCurLine<=FTopLine+FFullLinesOnPage)));
      end;
    VK_RIGHT:
      begin
        if FCurCol<FMaxLineLength-1 then
          Inc(FCurCol);
        FLeftCol:=FLeftCol+integer(not ((FCurCol>=FLeftCol) and (FCurCol<FLeftCol+FFullColsOnPage)));
      end;
    VK_LEFT:
      begin
        if FCurCol>0 then
          Dec(FCurCol);
        FLeftCol:=FLeftCol-integer(not ((FCurCol>=FLeftCol) and (FCurCol<FLeftCol+FFullColsOnPage)));
      end;
    VK_NEXT:
      begin
        if FCurLine+FFullLinesOnPage<=FText.Count-1 then
          begin
            Inc(FCurLine,FFullLinesOnPage);
            Inc(FTopLine,FFullLinesOnPage);
          end
        else
          begin
            FCurLine:=FText.Count-1;
            FTopLine:=FText.Count-FFullLinesOnPage;
          end;
      end;
    VK_PRIOR:
      begin
        if FCurLine-FFullLinesOnPage>=0 then
          begin
            Dec(FCurLine,FFullLinesOnPage);
            Dec(FTopLine,FFullLinesOnPage);
          end
        else
          begin
            FCurLine:=0;
            FTopLine:=0;
          end;
      end;
  end
else
  if ss=[ssCtrl] then
    case Message.CharCode of
      VK_NEXT:
        begin
          FCurLine:=FText.Count-1;
          FTopLine:=FText.Count-FFullLinesOnPage;
        end;
      VK_PRIOR:
        begin
          FCurLine:=0;
          FTopLine:=0;
        end;
    end;

if (OldCol=FCurCol) and (OldLine=FCurLine) then exit;

UpdateCursor;

if not (((FCurLine>=FTopLine) and (FCurLine<FTopLine+FFullLinesOnPage)) and
        ((FCurCol>=FLeftCol) and (FCurCol<FLeftCol+FFullColsOnPage))) then
  begin
    //   ,   
    FTopLine:=Max(0,FCurLine-FFullLinesOnPage+1);
    FLeftCol:=Max(0,FCurCol-FFullColsOnPage+1);
  end;

//      
if (OldTopLine<>FTopLine) or (OldLeftCol<>FLeftCol) then
  begin
    VertScrollBar.Position:=FTopLine*FCharHeight;
    HorzScrollBar.Position:=FLeftCol*FCharWidth;
  end;

DoCursorPosChanged;
end;

procedure TprTxPreviewBox.mVScroll;
begin
inherited;
FTopLine:=VertScrollBar.Position div FCharHeight;
end;

procedure TprTxPreviewBox.mHScroll;
begin
inherited;
FLeftCol:=HorzScrollBar.Position div FCharWidth;
end;

procedure TprTxPreviewBox.mLButtonDown;
var
  x,y : integer;
begin
x := (HorzScrollBar.Position+Message.XPos) div FCharWidth;
y := (VertScrollBar.Position+Message.YPos) div FCharHeight;

if y<FText.Count then
  begin
    if x<FMaxLineLength then
      begin
        FCurCol := x;
        FCurLine := y;
        UpdateCursor;
        DoCursorPosChanged;
      end;
  end;
inherited;
end;

procedure TprTxPreviewBox.GetPointInfoAt;
var
  i,charx,chary : integer;
begin
rec := nil;
charx := (HorzScrollBar.Position+X) div FCharWidth;
chary := (VertScrollBar.Position+Y) div FCharHeight;
if (chary<FText.Count) and (charx<FMaxLineLength) then
  begin
    for i:=0 to Report.TextDevice.Recs.Count-1 do
      if PointInRect(charx,chary,pTextDeviceRec(Report.TextDevice.Recs[i]).rPlace) then
        begin
          rec := pTextDeviceRec(Report.TextDevice.Recs[i]); 
          break;
        end;
  end;
end;

procedure TprTxPreviewBox.MouseDown;
var
  rec : pTextDeviceRec;
  PreviewUserData : TprPreviewUserData;
begin
inherited;
GetPointInfoAt(x,y,rec);
PreviewUserData := nil;
if rec<>nil then
  PreviewUserData := rec.UserData;
Report.DoOnPreviewMouseDown(PreviewUserData,Shift);
end;

procedure TprTxPreviewBox.DblClick;
var
  p : TPoint;
  rec : pTextDeviceRec;
  PreviewUserData : TprPreviewUserData;
begin
inherited;
GetCursorPos(p);
p := ScreenToClient(p);
GetPointInfoAt(p.x,p.y,rec);
PreviewUserData := nil;
if rec<>nil then
  PreviewUserData := rec.UserData;
Report.DoOnPreviewDblClick(PreviewUserData);
end;

procedure TprTxPreviewBox.MouseUp;
begin
inherited;
end;

procedure TprTxPreviewBox.MouseMove;
var
  DC : HDC;
  cur : TCursor;
  rec : pTextDeviceRec;
  PreviewUserData : TprPreviewUserData;
  HighlightObject : boolean;
begin
inherited;
GetPointInfoAt(x,y,rec);
cur := crArrow;
HighlightObject := false;
PreviewUserData := nil;
if rec<>nil then
  PreviewUserData := rec.UserData;
Report.DoOnPreviewMouseMove(PreviewUserData,cur,HighlightObject);
if cur<>Cursor then
  Cursor := cur;

if rec<>FLastHighlightedRec then
  begin
    //  
    DC := GetDC(Handle);
    if FLastHighlightedRec<>nil then
      InvertRec(DC,FLastHighlightedRec);
    if (rec<>nil) and HighlightObject then
      begin
        InvertRec(DC,rec);
        FLastHighlightedRec := rec;
      end
    else
      FLastHighlightedRec := nil;
    ReleaseDC(Handle,DC);
  end;
end;

function TprTxPreviewBox.GetCurPage;
var
  y : integer;
begin
Result:=0;
y     :=FCurLine;
while (Result<FPagesList.Count) and (y>=integer(FPagesList[Result])) do Inc(Result);
if Result>=FPagesList.Count then
  Result:=PagesCount-1
else
  Result:=Result-1
end;

function TprTxPreviewBox.GetPagesCount;
begin
Result:=FPagesList.Count;
end;

function TprTxPreviewBox.GetLinesCount;
begin
Result:=FText.Count;
end;

procedure TprTxPreviewBox.MakeCursorVisible;
begin
if not (((FCurLine>=FTopLine) and (FCurLine<FTopLine+FFullLinesOnPage)) and
        ((FCurCol>=FLeftCol) and (FCurCol<FLeftCol+FFullColsOnPage))) then
  begin
    //   ,   
    FTopLine:=Max(0,FCurLine-FFullLinesOnPage+1);
    FLeftCol:=Max(0,FCurCol-FFullColsOnPage+1);

    VertScrollBar.Position:=FTopLine*FCharHeight;
    HorzScrollBar.Position:=FLeftCol*FCharWidth;
  end;
end;

procedure TprTxPreviewBox.SetCaretPos;
begin
if y<0 then y:=0;
if y>=FText.Count then y:=FText.Count-1;
if x<0 then x:=0;
if x>=FMaxLineLength then x:=FMaxLineLength-1;

FCurCol :=x;
FCurLine:=y;

MakeCursorVisible;
UpdateCursor;
DoCursorPosChanged;
end;

procedure TprTxPreviewBox.GotoPage;
begin
if p>=PagesCount then exit;

SetCaretPos(0,integer(FPagesList[p]));
end;

function TprTxPreviewBox.FindText;
label
  ExitLoop;
var
  i,j,ls,lfText : integer;
  fndText,s,Buf : string;
begin
Result:=false;
fndText :=FindText;
if not IsCase then
  fndText:=AnsiLowerCase(fndText);
prWINtoOEM(PChar(fndText),PChar(fndText));

lfText:=Length(fndText);

if IsDown then
  begin
    i:=FCurLine;
    if (FLastFoundCol=FCurCol) and (FLastFoundLine=FCurLine) then
      j:=FCurCol+2
    else
      j:=FCurCol+1;

    while i<FText.Count do
      begin
        s :=FText[i];
        ls:=Length(s);
        while j<=ls do
          begin
            Buf:=Copy(s,j,lfText);
            if not IsCase then
              begin
                prOEMtoWIN(PChar(Buf),PChar(Buf));
                Buf:=AnsiLowerCase(Buf);
                prWINtoOEM(PChar(Buf),PChar(Buf));
              end;
            if Buf=fndText then
              begin
                // 
                Result:=true;
                goto ExitLoop;
              end;
            Inc(j);
          end;
        j:=1;
        Inc(i);
      end;
  end
else
  begin
    i:=FCurLine;
    if (FLastFoundCol=FCurCol) and (FLastFoundLine=FCurLine) then
      j:=FCurCol
    else
      j:=FCurCol+1;

    while i>=0 do
      begin
        s :=FText[i];
        if i<>CurLine then
          j:=Length(s)-lfText;
        while j>=1 do
          begin
            Buf:=Copy(s,j,lfText);
            if not IsCase then
              Buf:=AnsiLowerCase(Buf);
            if Buf=fndText then
              begin
                // 
                Result:=true;
                goto ExitLoop;
              end;
            Dec(j);
          end;
        Dec(i);
      end;
  end;

ExitLoop:
if Result then
  begin
    FCurCol       :=j-1;
    FCurLine      :=i;
    FLastFoundCol :=FCurCol;
    FLastFoundLine:=FCurLine;

    MakeCursorVisible;
    UpdateCursor;
    DoCursorPosChanged;
  end;
end;

/////////////////////////////////
//
// TprTxPreviewForm
//
/////////////////////////////////
procedure TprTxPreviewForm.FormCreate(Sender: TObject);
begin
prLoadResImages(Self,ImageList);

if Report.Title='' then
  Caption:=Format(prLoadStr(sPreviewCaption),[prLoadStr(sNoReportName)])
else
  Caption:=Format(prLoadStr(sPreviewCaption),[Report.Title]);
SaveDialog.Filter := prLoadStr(sPreviewFileMask);
OpenDialog.Filter := prLoadStr(sPreviewFileMask);

Preview                   :=TprTxPreviewBox.CreatePreview(Self,TprTxReport(Report));
Preview.OnCursorPosChanged:=CursorPosChanged;
Preview.Parent            :=Self;
Preview.Align             :=alClient;
Preview.Color             :=clWindow;
Preview.Font.Name         :='Courier New';
Preview.Font.Size         :=14;
Preview.Font.CharSet      :=OEM_CHARSET;

Preview.SetToNewReport;
UpdateCaretPos;

Preview.HorzScrollBar.Tracking:=true;
Preview.VertScrollBar.Tracking:=true;

aCustomAction.Hint:=Report.CustomActionInPreviewCaption;
ActiveControl     :=Preview;
end;

procedure TprTxPreviewForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
Action:=caFree;
end;

procedure TprTxPreviewForm.prRestoreProperties;
begin
inherited;
Preview.Font.Name:=ini.ReadString(sn,'Preview.Font.Name',Preview.Font.Name);
Preview.Font.Size:=ini.ReadInteger(sn,'Preview.Font.Size',Preview.Font.Size);
Preview.UpdateSettings;
end;

procedure TprTxPreviewForm.prSaveProperties;
begin
inherited;
ini.WriteString(sn,'Preview.Font.Name',Preview.Font.Name);
ini.WriteInteger(sn,'Preview.Font.Size',Preview.Font.Size);
end;

procedure TprTxPreviewForm.aPrintExecute(Sender: TObject);
begin
if Report.SetupPrintParams then
  Report.PrintPreparedReport;
end;

procedure TprTxPreviewForm.aOpenExecute(Sender: TObject);
begin
if OpenDialog.Execute then
  begin
    Report.LoadPreparedReportFromFile(OpenDialog.FileName);

    Preview.SetToNewReport;
  end;
end;

procedure TprTxPreviewForm.aSaveExecute(Sender: TObject);
begin
if SaveDialog.Execute then
  Report.SavePreparedReportToFile(SaveDialog.FileName);
end;

procedure TprTxPreviewForm.CursorPosChanged;
begin
UpdateCaretPos;
end;

procedure TprTxPreviewForm.UpdateCaretPos;
var
  i,y,x : integer;
begin
x :=Preview.CurCol;
y :=Preview.CurLine;

SB.Panels[0].Text:=Format('%d : %d',[y+1,x+1]);
EDLine.Text      :=IntToStr(y+1);

i :=Preview.CurPage;
if i=-1 then
  begin
    SB.Panels[1].Text:='';
    EDPage.Text      :='';
  end
else
  begin
    SB.Panels[1].Text:=Format(prLoadStr(sTxPreviewPages),[i+1,Preview.PagesCount]);
    EDPage.Text:=IntToStr(i+1);
  end;
end;

procedure TprTxPreviewForm.EDLineKeyPress(Sender: TObject; var Key: Char);
var
  l : integer;
begin
if Key=#13 then
  begin
    Key:=#0;
    l  :=StrToIntDef(EDLine.Text,-1);
    if (l>0) and (l<=Preview.LinesCount) then
      begin
        ActiveControl:=Preview;
        Preview.SetCaretPos(0,l-1);
      end;
  end;
end;

procedure TprTxPreviewForm.EDPageKeyPress(Sender: TObject; var Key: Char);
var
  p : integer;
begin
if Key=#13 then
  begin
    Key:=#0;
    p  :=StrToIntDef(EDPage.Text,-1);
    if p>0 then
      begin
        if p>Preview.PagesCount then
          MBError(Format(prLoadStr(sErrorTxPreviewPageRange),[Preview.PagesCount]))
        else
          begin
            ActiveControl:=Preview;
            Preview.GotoPage(p-1);
          end;
      end;
  end;
end;

procedure TprTxPreviewForm.EDPageClick(Sender: TObject);
var
  Key : char;
begin
Key:=#13;
EDPageKeyPress(nil,Key);
end;

procedure TprTxPreviewForm.EDLineClick(Sender: TObject);
var
  Key : char;
begin
Key:=#13;
EDLineKeyPress(nil,Key);
end;

procedure TprTxPreviewForm.aFindExecute(Sender: TObject);
begin
FindDialog.Execute;
end;

procedure TprTxPreviewForm.FindDialogFind(Sender: TObject);
begin
with FindDialog do
  if not Preview.FindText(FindText,frMatchCase in Options,frDown in Options) then
    MBox(Format(prLoadStr(sTextNotFound),[FindText]),prLoadStr(sAttention),MB_OK+MB_ICONEXCLAMATION);
end;

procedure TprTxPreviewForm.aCloseExecute(Sender: TObject);
begin
Self.Close;
end;

procedure TprTxPreviewForm.aFontOptionsExecute(Sender: TObject);
begin
FontDialog.Font.Name:=Preview.Font.Name;
FontDialog.Font.Size:=Preview.Font.Size;
if FontDialog.Execute then
  begin
    Preview.Font.Name:=FontDialog.Font.Name;
    Preview.Font.Size:=FontDialog.Font.Size;
    Preview.UpdateSettings;
  end;
end;

procedure TprTxPreviewForm.aCustomActionExecute(Sender: TObject);
begin
DoOnCustomAction;
end;

procedure TprTxPreviewForm.aCustomActionUpdate(Sender: TObject);
begin
TAction(Sender).Enabled:=Assigned(Report.OnCustomActionInPreview);
end;

procedure TprTxPreviewForm.aExportToTXTExecute(Sender: TObject);
begin
TprTxReport(Report).ExportToTXT;
end;

initialization

  RegisterClass(TprTxPreviewForm);
  
end.
