{******************************************}
{                                          }
{                 PReport v1.5             }
{                                          }
{ Copyright (c) 1999-2002 by Manuzin A.    }
{                                          }
{******************************************}

unit pr_Parser;

{$i pr.inc}
interface

uses
  Classes, SysUtils, Pr_Utils, Db, Mask, Forms, {$ifdef PR_D6} variants, maskutils, {$endif}

  pr_Common, pr_Dataset;

const
  prMAX_CALCOPERATORS = 13;
  prMAX_FUNCTIONS  = 36;
  prMAX_OBJFUNCTIONS = 7;
  prMAX_OBJPROPS = 3;
  prMAX_QUERYES = 4;
  prDefaultStackSize = 10;
  REPORT_PROPS_INFO_INDEX = 2;

var
  CurrencyFormat : string = '#,0.00''.'';-#,0.00''.''';
  ShortCurrencyFormat : string = '0.00''.'';-0.00''.''';
  SimpleCurrencyFormat : string = '0.00;-0.00';
  PercentFormat : string = '0.00%;-0.00%';
  SpacedCurrencyFormat : string = '#,0.00;-#,0.00';
  BankCurrencyFormat : string = '\ds-0.00;-0.00';

type

  TprParser = class;
  TprExprItemType = (preiOperator,preiValue,preiVar,preiFunction,preiObjectProperty,preiObjectFunction,preiReportVariable,preiReportSystemVariable);
  TprTypeBracket = (prtbFunction,prtbExpression);
  TprIdentType = (pritVar,pritObject,pritObjectProperty,pritObjectFunction,pritReportVariable);

  TprCalcOperatorProc = procedure(ML : TprParser; var Res : TprVarValue);
  TprCalcOperatorInfo = record
    Name : string;
    Proc : TprCalcOperatorProc;
  end;

  TprFuncProc = procedure(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  TprFuncInfo = record
    Name : string;
    Func : TprFuncProc;
    min  : integer;
    max  : integer;
  end;

  TprObjFuncProc = procedure(ML : TprParser; C : TComponent; Parameters : TprVarsArray; var Res : TprVarValue);
  TprObjFuncInfo = record
    ObjClass    : string;
    FuncName    : string;
    Func        : TprObjFuncProc;
    ObjClassRef : TPersistentClass;
    min         : integer;
    max         : integer;
  end;
  PprObjFuncInfo = ^TprObjFuncInfo;

  TprObjPropFunc = procedure(C : TComponent; PropName : string; var Res : TprVarValue);
  TprGetPropsNamesFunc = procedure(C : TComponent; L : TStringList);
  TprObjPropInfo = record
    ObjClass          : string;
    ObjClassRef       : TPersistentClass;
    Func              : TprObjPropFunc;
    GetPropsNamesFunc : TprGetPropsNamesFunc;
    min               : integer;
    max               : integer;
  end;
  PprObjPropInfo = ^TprObjPropInfo;
  




  ////////////////////////////////////////
  //
  //   
  //
  ////////////////////////////////////////
  TprExprItem = record
    ObjType     : TprExprItemType; //  - .  , ,   
    ObjIndex    : integer;         //   (  )   . ,     
    ObjName     : string;          //   ( )
    ObjValue    : TprVarValue;     // 
    ObjParCount : integer;         //    
  end;

  //////////////////////////////////
  //
  //  
  //
  //////////////////////////////////
  TprExpr = array of TprExprItem;


  TprFormatReplaceCallBackOptions = (prfrcRTF);
  TprFormatReplaceCallBackOptionsSet = set of TprFormatReplaceCallBackOptions;
  TprFormatReplaceCallBackFunc = procedure (FromPos,Count : integer; const Buf : PChar; BufSize : integer; Flags : TprFormatReplaceCallBackOptionsSet; CallBackData : pointer);
  /////////////////////////////
  //
  // TprParser
  //
  /////////////////////////////
  TprParser = class(TObject)
  private
    Stack : TprVarsArray;  //      Calc
    pStack : integer;       //   
    sStack : integer;       //   
    FExprs : array of TprExpr;
    LReportProps : TStringList;

    //   
    procedure ResetStack;
    function rStack : PprVarValue;                        //   Stack
    procedure wStack(const Value : TprVarValue); overload; //   Stack Value
    function wStack : integer; overload;
    function CompileExpression(var Expr : string; var cExpr : TprExpr) : boolean;
    procedure CalcExpression(var Expr : TprExpr; var Res : TprVarValue);
    function FindObjPropInfo(Component : TComponent) : PprObjPropInfo;
    function FindObjFuncInfo(Component : TComponent; const FuncName : string) : PprObjFuncInfo;
  public
    Report : TprCustomReport;
    function Calc(var Expr : string; var Res : TprVarValue) : boolean;
    function FormatTemplate(Mask : string; var Res : string) : boolean;
    function FormatTemplateEx(const Mask : string; ReplaceCallBack : TprFormatReplaceCallBackFunc; CallBackData : pointer; var Res : string) : boolean;
    procedure ClearInternalStructs;

    constructor Create(_Report : TprCustomReport);
    destructor Destroy; override;
  end;

  ////////////////////////////
  //
  // 
  //
  ////////////////////////////
  procedure _vCopy(const vSource : TprVarValue; var vDest : TprVarValue);
  function  _vAsString(const v : TprVarValue) : string;
  function  _vAsBoolean(const v : TprVarValue) : boolean;
  function  _vAsInteger(const v : TprVarValue) : integer;
  function  _vAsDouble(const v : TprVarValue) : double;
  function  _vAsObject(const v : TprVarValue) : TObject;
  function  _vAsDateTime(const v : TprVarValue) : TDateTime;
  function  _vAsVariant(const v : TprVarValue) : Variant;
  function  _vIsNull(const v : TprVarValue) : boolean;
  
  procedure _vSetNull(var v : TprVarValue);
  procedure _vSetAsObject(var v : TprVarValue; Value : TObject);
  procedure _vSetAsString(var v : TprVarValue; Value : string);
  procedure _vSetAsDateTime(var v : TprVarValue; Value : TDateTime);
  procedure _vSetAsDouble(var v : TprVarValue; Value : double);
  procedure _vSetAsInteger(var v : TprVarValue; Value : integer);
  procedure _vSetAsBoolean(var v : TprVarValue; Value : boolean);
  procedure _vSetAsVariant(var v : TprVarValue; Value : Variant);
  procedure _vSetAsType(var v : TprVarValue; Value : Variant; VarType : TprVarValueType);

  procedure VarFromField(Field : TField; var Res : TprVarValue);

  function pFormat(const aDisplayFormat: String; const aValue: TprVarValue): String;

  ////////////////////////////////////////
  //
  //   
  //
  ////////////////////////////////////////
  procedure _OpDiv(ML : TprParser; var Res : TprVarValue);
  procedure _OpMul(ML : TprParser; var Res : TprVarValue);
  procedure _OpPlus(ML : TprParser; var Res : TprVarValue);
  procedure _OpMinus(ML : TprParser; var Res : TprVarValue);
  procedure _OpNot(ML : TprParser; var Res : TprVarValue);
  procedure _OpNEQ(ML : TprParser; var Res : TprVarValue);
  procedure _OpGEQ(ML : TprParser; var Res : TprVarValue);
  procedure _OpLEQ(ML : TprParser; var Res : TprVarValue);
  procedure _OpG(ML : TprParser; var Res : TprVarValue);
  procedure _OpL(ML : TprParser; var Res : TprVarValue);
  procedure _OpEQ(ML : TprParser; var Res : TprVarValue);
  procedure _OpAnd(ML : TprParser; var Res : TprVarValue);
  procedure _OpOr(ML : TprParser; var Res : TprVarValue);

  ////////////////////////////////
  // 
  ////////////////////////////////
  procedure _DateTime(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _Time(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _GetMonth(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _GetYear(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _GetDay(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _GetLastDayMonth(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _GetFirstDayMonth(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _GetMonthName(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _HourBetween(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _MonthsBetween(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _DaysBetween(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _IncMonth(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _IncDay(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _GetYM(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _IncYM(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _GetDateFromYM(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  
  procedure _Round(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _Trunc(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _Min(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _Max(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _In(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);

  procedure _GSN(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _IsZero(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _IsNotZero(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _IIF(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);

  procedure _Copy(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _MakeStr(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);

  procedure _AnsiUpperCase(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _AnsiLowerCase(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _Trim(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _AddLeft(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _AddRight(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _Length(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);

  procedure _UID(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);

  procedure _Null(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _Abs(ML : TprParser; Parameters : TprVarsArray; var Res : TprVarValue);

  //////////////////////////////////////////
  //  
  //////////////////////////////////////////
  // TDataSet
  procedure _TDataSet_Eof(ML : TprParser; C : TComponent; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _TDataSet_Locate(ML : TprParser; C : TComponent; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _TDataSet_IsNullField(ML : TprParser; C : TComponent; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _TDataSet_IsZeroField(ML : TprParser; C : TComponent; Parameters : TprVarsArray; var Res : TprVarValue);
  procedure _TDataSet_LineNo(ML : TprParser; C : TComponent; Parameters : TprVarsArray; var Res : TprVarValue);

  procedure _TprDataSet_LineNo(ML : TprParser; C : TComponent; Parameters : TprVarsArray; var Res : TprVarValue);

  procedure _TprGroup_LineNo(ML : TprParser; C : TComponent; Parameters : TprVarsArray; var Res : TprVarValue);

  //////////////////////////////
  //  
  //////////////////////////////
  procedure _TDataSet_rProp(C : TComponent; PropName : string; var Res : TprVarValue);
  procedure _TprCustomReport_rProp(C : TComponent; PropName : string; var Res : TprVarValue);
  procedure _TDataSet_PropsList(C : TComponent; L : TStringList);
  procedure _TprCustomReport_PropsList(C : TComponent; L : TStringList);
  procedure _TprDataSet_rProp(C : TComponent; PropName : string; var Res : TprVarValue);
  procedure _TprDataSet_PropsList(C : TComponent; L : TStringList);

var
  CO : array [1..prMAX_CALCOPERATORS] of TprCalcOperatorInfo =
  ((Name : '/';  Proc : _OpDiv),
   (Name : '*';  Proc : _OpMul),
   (Name : '-';  Proc : _OpMinus),
   (Name : '+';  Proc : _OpPlus),
   (Name : '!';  Proc : _OpNot),

   (Name : '<>'; Proc : _OpNEQ),
   (Name : '>='; Proc : _OpGEQ),
   (Name : '<='; Proc : _OpLEQ),
   (Name : '>';  Proc : _OpG),
   (Name : '<';  Proc : _OpL),
   (Name : '=';  Proc : _OpEQ),

   (Name : '&';  Proc : _OpAnd),
   (Name : '|';  Proc : _OpOr));

  FuncInfo : array [1..prMAX_FUNCTIONS] of TprFuncInfo=
  ((Name : 'DateTime';            Func : _DateTime; min : 0; max : 0),
   (Name : 'Time';                Func : _Time; min : 0; max : 0),
   (Name : 'GetFirstDayMonth';    Func : _GetFirstDayMonth; min : 1; max : 2),
   (Name : 'GetLastDayMonth';     Func : _GetLastDayMonth; min : 1; max : 2),
   (Name : 'GetYear';             Func : _GetYear; min : 1; max : 1),
   (Name : 'GetMonth';            Func : _GetMonth; min : 1; max : 1),
   (Name : 'HourBetween';         Func : _HourBetween; min : 2; max : 4),
   (Name : 'MonthsBetween';       Func : _MonthsBetween; min : 2; max : 2),
   (Name : 'DaysBetween';         Func : _DaysBetween; min : 2; max : 2),
   (Name : 'IncMonth';            Func : _IncMonth; min : 2; max : 2),
   (Name : 'IncDay';              Func : _IncDay; min : 2; max : 2),
   (Name : 'GetYM';               Func : _GetYM; min : 1; max : 2),
   (Name : 'IncYM';               Func : _IncYM; min : 2; max : 2),
   (Name : 'GetMonthName';        Func : _GetMonthName; min : 1; max : 1),
   (Name : 'GetDateFromYM';       Func : _GetDateFromYM; min : 2; max : 2),

   (Name : 'Round';               Func : _Round; min : 1; max : 2),
   (Name : 'Trunc';               Func : _Trunc; min : 1; max : 1),
   (Name : 'Min';                 Func : _Min; min : 1; max : -1),
   (Name : 'Max';                 Func : _Max; min : 1; max : -1),

   (Name : 'Copy';                Func : _Copy; min : 2; max : 3),
   (Name : 'MakeStr';             Func : _MakeStr; min : 1; max : 2),

   (Name : 'GSN';                 Func : _GSN; min : 1; max : 14),
   (Name : 'IsZero';              Func : _IsZero; min : 1; max : 1),
   (Name : 'IIF';                 Func : _IIF; min : 3; max : 3),
   (Name : 'IsNotZero';           Func : _IsNotZero; min : 1; max : 1),

   (Name : 'AnsiUpperCase';       Func : _AnsiUpperCase; min : 1; max : 1),
   (Name : 'AnsiLowerCase';       Func : _AnsiLowerCase; min : 1; max : 1),
   (Name : 'Trim';                Func : _Trim; min : 1; max : 1),

   (Name : 'UID';                 Func : _UID; min : 1; max : -1),

   (Name : 'In';                  Func : _In; min : 2; max : -1),
   (Name : 'AddLeft';             Func : _AddLeft; min : 3; max : 3),
   (Name : 'AddRight';            Func : _AddRight; min : 3; max : 3),
   (Name : 'Length';              Func : _Length; min : 1; max : 1),
   (Name : 'Null';                Func : _Null; min : 0; max : 0),

   (Name : 'Abs';                 Func : _Abs; min : 1; max :1),
   (Name : 'GetDay';              Func : _GetDay; min : 1; max :1));

  ObjFuncInfo : array [1..prMAX_OBJFUNCTIONS] of TprObjFuncInfo=
  ((ObjClass : 'TDataSet';   FuncName : 'IsNullField'; Func : _TDataSet_IsNullField; min : 1; max : 1),
   (ObjClass : 'TDataSet';   FuncName : 'Locate';      Func : _TDataSet_Locate; min : 2; max : -1),
   (ObjClass : 'TDataSet';   FuncName : 'Eof';         Func : _TDataSet_Eof; min : 0; max : 0),
   (ObjClass : 'TDataSet';   FuncName : 'IsZeroField'; Func : _TDataSet_IsZeroField; min : 1; max : 1),
   (ObjClass : 'TDataSet';   FuncName : 'LineNo';      Func : _TDataSet_LineNo; min : 0; max : 0),
   (ObjClass : 'TprGroup';   FuncName : 'LineNo';      Func : _TprGroup_LineNo; min : 0; max : 0),
   (ObjClass : 'TprDataSet'; FuncName : 'LineNo';      Func : _TprDataSet_LineNo; min : 0; max : 0)
   );


  ObjPropInfo : array [1..prMAX_OBJPROPS] of TprObjPropInfo=
  ((ObjClass : 'TDataSet';        Func : _TDataSet_rProp; GetPropsNamesFunc : _TDataSet_PropsList),
   (ObjClass : 'TprCustomReport'; Func : _TprCustomReport_rProp; GetPropsNamesFunc : _TprCustomReport_PropsList),
   (ObjClass : 'TprDataSet';      Func : _TprDataSet_rProp; GetPropsNamesFunc : _TprDataSet_PropsList));



implementation

uses
  pr_MultiLang, pr_Strings;

var
  i : integer;
  S_NumberDateChars : set of char = ['0'..'9'];

const
  S_StringChar      : char = '"';
  S_StartBracketChar: char = '(';
  S_EndBracketChar  : char = ')';
  S_StartIdentChars : set of char = ['a'..'z',''..'','A'..'Z',''..''];
  S_IdentChars      : set of char = ['a'..'z',''..'','A'..'Z',''..'','_','0'..'9','.','$','@'];
  S_SpecialChars    : set of char = ['!','>','<','=','(',')','+','-','*','/','&','|',','];
  S_OperatorChars   : set of char = ['!','>','<','=','+','-','*','/','&','|',','];



  procedure _CenterStr(var b : string; c : integer);
  begin
  if Length(b)>c then
    b:=Copy(b,1,c)
  else
    b:=CenterStr(b,c);
  end;

  procedure _LeftStr(var b : string; c : integer);
  begin
  if Length(b)>c then
    b:=Copy(b,1,c)
  else
    b:=AddCharR(' ',b,c);
  end;

  procedure _RightStr(var b : string; c : integer);
  begin
  if Length(b)>c then
    b:=Copy(b,Length(b)-c+1,Length(b))
  else
    b:=AddChar(' ',b,c);
  end;

  function pFormat;
  var
    lfFormat : TFloatFormat;
    liDigits : integer;
    liPrecision : Integer;
    liPos : Integer;
    vDouble : double;
    i,p,pLMN : integer;
    OldDecSep : char;
    OldThSep : char;
    lsFormatStr,_aDisplayFormat,lsString : String;
    OldLongMonthNames : array[1..12] of string;
  begin
  _aDisplayFormat := aDisplayFormat;
  OldDecSep := DecimalSeparator;
  OldThSep := ThousandSeparator;
  for i:=1 to 12 do
    OldLongMonthNames[i] := LongMonthNames[i];

  p := pos('\ds',_aDisplayFormat);
  if (p<>0) and (p+3<=Length(_aDisplayFormat)) then
    begin
      DecimalSeparator := _aDisplayFormat[p+3];
      Delete(_aDisplayFormat,p,4);
    end;

  p := pos('\ts',_aDisplayFormat);
  if (p<>0) and (p+3<=Length(_aDisplayFormat)) then
    begin
      ThousandSeparator := _aDisplayFormat[p+3];
      Delete(_aDisplayFormat,p,4);
    end;

  pLMN := pos('\lmn',_aDisplayFormat);
  if pLMN<>0 then
    begin
      for i:=1 to 12 do
        LongMonthNames[i] := MonthsArray[2,i];
      Delete(_aDisplayFormat,pLMN,4);
    end;

  case aValue.vType of
    prvvtInteger,prvvtDouble:
      begin
        lsFormatStr := _aDisplayFormat;
        vDouble := _vAsDouble(aValue);

        if Pos('$', _aDisplayFormat) <> 0 then
          begin
            if Pos('.', _aDisplayFormat) = 0 then
              liDigits := 0
            else
              begin
                liPos :=  Pos(';', _aDisplayFormat);
                if liPos > 0 then
                  lsString := Copy(_aDisplayFormat, 1, liPos-1)
                else
                  lsString := _aDisplayFormat;
                liDigits := Length(lsString) - Pos('.', lsString);
              end;
            lfFormat := ffCurrency;
            liPrecision := 15;
            lsString := FloatToStrF(vDouble, lfFormat, liPrecision, liDigits);
          end
        else
          if (_aDisplayFormat <> '') then
            lsString := FormatFloat(lsFormatStr, vDouble)
          else
            begin
              lfFormat := ffGeneral;
              liDigits := 0;
              liPrecision := 15;
              lsString := FloatToStrF(vDouble,lfFormat,liPrecision,liDigits);
            end;
        Result := lsString;
      end;

    prvvtDateTime:
      if Length(_aDisplayFormat) > 0 then
        Result := FormatDateTime(_aDisplayFormat, _vAsDateTime(aValue))
      else
        Result := FormatDateTime(ShortDateFormat, _vAsDateTime(aValue));

    prvvtString:
      if Length(_aDisplayFormat) > 0 then
         Result := FormatMaskText(_aDisplayFormat, _vAsString(aValue))
      else
         Result := _vAsString(aValue);

    prvvtBoolean:
      Result := _vAsString(aValue);

    else
      Result := '';
  end;

  if OldDecSep<>DecimalSeparator then
    DecimalSeparator:=OldDecSep;
  if OldThSep<>ThousandSeparator then
    ThousandSeparator:=OldThSep;
  if pLMN<>0 then
    for i:=1 to 12 do
      LongMonthNames[i]:=OldLongMonthNames[i];
  end;

//////////////////////////////////////
//
// 
//
//////////////////////////////////////
procedure VarFromField;
begin
if Field.IsNull then
  Res.vType := prvvtNull
else
  case Field.DataType of
    ftString,ftFixedChar,ftWideString :
      begin
        Res.vType  :=prvvtString;
        Res.vString:=Field.AsString;
      end;
    ftSmallInt,ftInteger,ftWord,ftLargeInt,ftAutoinc:
      begin
        Res.vType   :=prvvtInteger;
        Res.vInteger:=Field.AsInteger;
      end;
    ftFloat,ftBCD,ftCurrency :
      begin
        Res.vType  :=prvvtDouble;
        Res.vDouble:=Field.AsFloat;
      end;
    ftBoolean:
      begin
        Res.vType   :=prvvtBoolean;
        Res.vBoolean:=Field.AsBoolean;
      end;
    ftDate,ftDateTime:
      begin
        Res.vType    :=prvvtDateTime;
        Res.vDateTime:=Field.AsDateTime;
      end;
    ftMemo,ftfmtMemo,ftBlob:
      begin
        Res.vType  :=prvvtString;
        Res.vString:=Field.AsString;
      end;
    else
      raise Exception.Create(prLoadStr(sParserErrorUnknownFieldType));
  end;
end;

procedure _vCopy;
begin
vDest.vType := vSource.vType;
vDest.vString := vSource.vString;
vDest.vBoolean := vSource.vBoolean;
vDest.vInteger := vSource.vInteger;
vDest.vDateTime := vSource.vDateTime;
vDest.vDouble := vSource.vDouble;
vDest.vObject := vSource.vObject;
end;

function _vIsNull;
begin
Result := v.vType=prvvtNull;
end;

function _vAsString;
begin
try
  case v.vType of
    prvvtString : Result:=v.vString;
    prvvtInteger : Result:=IntToStr(v.vInteger);
    prvvtDateTime : Result:=DateTimeToStr(v.vDateTime);
    prvvtDouble : Result:=FloatToStr(v.vDouble);
    prvvtBoolean : Result:=BoolToStr(v.vBoolean);
    prvvtNull : Result:='';
    else raise Exception.Create(prLoadStr(sParserErrorErrorValueType));
  end;
except
  on E : Exception do
    raise Exception.CreateFmt(prLoadStr(sParserErrorCalcExpression),[E.Message]);
end
end;

function _vAsBoolean;
begin
try
  case v.vType of
    prvvtString : Result:=StrToBool(v.vString);
    prvvtInteger : Result:=v.vInteger<>0;
    prvvtBoolean : Result:=v.vBoolean;
    prvvtNull : Result:=false;
    else raise Exception.Create(prLoadStr(sParserErrorErrorValueType));
  end;
except
  on E : Exception do
    raise Exception.CreateFmt(prLoadStr(sParserErrorCalcExpression),[E.Message]);
end
end;

function _vAsInteger;
begin
try
  case v.vType of
    prvvtString : Result := StrToInt(v.vString);
    prvvtInteger : Result := v.vInteger;
    prvvtDouble : Result := ExtRound(v.vDouble);
    prvvtBoolean : Result := integer(v.vBoolean);
    prvvtNull : Result := 0;
    else raise Exception.Create(prLoadStr(sParserErrorErrorValueType));
  end;
except
  on E : Exception do
    raise Exception.CreateFmt(prLoadStr(sParserErrorCalcExpression),[E.Message]);
end
end;

function _vAsDouble;
begin
try
  case v.vType of
    prvvtString : Result:=StrToFloat(v.vString);
    prvvtInteger : Result:=v.vInteger;
    prvvtDouble : Result:=v.vDouble;
    prvvtNull : Result:=0;
    else raise Exception.Create(prLoadStr(sParserErrorErrorValueType));
  end;
except
  on E : Exception do
    raise Exception.CreateFmt(prLoadStr(sParserErrorCalcExpression),[E.Message]);
end
end;

function _vAsDateTime;
begin
try
  case v.vType of
    prvvtString : Result:=StrToDateTime(v.vString);
    prvvtDateTime : Result:=v.vDateTime;
    prvvtNull : Result:=0;
    else raise Exception.Create(prLoadStr(sParserErrorErrorValueType));
  end;
except
  on E : Exception do
    raise Exception.CreateFmt(prLoadStr(sParserErrorCalcExpression),[E.Message]);
end
end;

function _vAsObject;
begin
try
  case v.vType of
    prvvtObject : Result:=v.vObject;
    prvvtNull : Result:=nil;
    else raise Exception.Create(prLoadStr(sParserErrorErrorValueType));
  end;
except
  on E : Exception do
    raise Exception.CreateFmt(prLoadStr(sParserErrorCalcExpression),[E.Message]);
end
end;

function _vAsVariant;
begin
try
  case v.vType of
    prvvtString : Result:=v.vString;
    prvvtInteger : Result:=v.vInteger;
    prvvtDateTime : Result:=v.vDateTime;
    prvvtDouble : Result:=v.vDouble;
    prvvtBoolean : Result:=v.vBoolean;
    prvvtNull : Result:=Null;
    else raise Exception.Create(prLoadStr(sParserErrorErrorValueType));
  end;
except
  on E : Exception do
    raise Exception.CreateFmt(prLoadStr(sParserErrorCalcExpression),[E.Message]);
end
end;

procedure _vSetNull;
begin
  v.vType := prvvtNull;
end;

procedure _vSetAsObject;
begin
  v.vType := prvvtObject;
  v.vObject := Value;
end;

procedure _vSetAsString;
begin
  v.vType := prvvtString;
  v.vString := Value;
end;

procedure _vSetAsDateTime;
begin
  v.vType := prvvtDateTime;
  v.vDateTime := Value;
end;

procedure _vSetAsDouble;
begin
  v.vType := prvvtDouble;
  v.vDouble := Value;
end;

procedure _vSetAsInteger;
begin
v.vType := prvvtInteger;
v.vInteger := Value;
end;

procedure _vSetAsBoolean;
begin
v.vType := prvvtBoolean;
v.vBoolean := Value;
end;

procedure _vSetAsType;
begin
case VarType of
  prvvtString  : v.vString  :=Value;
  prvvtDateTime: v.vDateTime:=Value;
  prvvtInteger : v.vInteger :=Value;
  prvvtBoolean : v.vBoolean :=Value;
  prvvtDouble  : v.vDouble  :=Value
  else           raise Exception.Create(prLoadStr(sParserErrorErrorValueType));
end;
v.vType:=VarType;
end;

procedure _vSetAsVariant;
begin
case VarType(Value) of
  varNull,
  varEmpty : _vSetNull(v);

  varSmallInt,
  varInteger,
  varByte: _vSetAsInteger(v,Value);

  varSingle,
  varDouble,
  varCurrency : _vSetAsDouble(v,Value);

  varDate : _vSetAsDateTime(v,Value);

  varString,
  varOleStr : _vSetAsString(v,Value);

  varBoolean : _vSetAsBoolean(v,Value);
  else raise Exception.Create(prLoadStr(sParserErrorErrorValueType));
end
end;


//////////////////////
//
// TprParser
//
//////////////////////
constructor TprParser.Create;
begin
inherited Create;
Report := _Report;
LReportProps := TStringList.Create;
ObjPropInfo[REPORT_PROPS_INFO_INDEX].GetPropsNamesFunc(Report,LReportProps);
SetLength(Stack,prDefaultStackSize);
sStack := prDefaultStackSize;
pStack := -1;
end;

destructor TprParser.Destroy;
var
  i : integer;
begin
LReportProps.Free;
Finalize(Stack);
for i:=0 to High(FExprs) do
  Finalize(FExprs[i]);
Finalize(FExprs);
inherited;
end;

procedure TprParser.ClearInternalStructs;
var
  i : integer;
begin
for i:=0 to High(FExprs) do
  Finalize(FExprs[i]);
Finalize(FExprs);
end;

procedure TprParser.ResetStack;
begin
pStack:=-1;
end;

//
//   
//
function TprParser.rStack;
begin
if pStack<0 then
  raise Exception.Create(prLoadStr(sParserErrorStack));
Result:=@(Stack[pStack]);
Dec(pStack);
end;

//
//   
//
procedure TprParser.wStack(const Value : TprVarValue);
begin
inc(pStack);
if pStack>=sStack then
  begin
    sStack:=sStack+prDefaultStackSize;
    SetLength(Stack,sStack);
  end;
_vCopy(Value,Stack[pStack]);
end;

function TprParser.wStack : integer;
begin
Inc(pStack);
if pStack>=sStack then
  begin
    sStack:=sStack+prDefaultStackSize;
    SetLength(Stack,sStack);
  end;
Result:=pStack;
end;

function TprParser.FindObjPropInfo;
var
  i : integer;
begin
i:=1;
while (i<=prMAX_OBJPROPS) and not (Component is ObjPropInfo[i].ObjClassRef) do Inc(i);
if i<=prMAX_OBJPROPS then
  Result:=@(ObjPropInfo[i])
else
  Result:=nil;
end;

function TprParser.FindObjFuncInfo;
var
  i : integer;
begin
i:=1;
while (i<=prMAX_OBJFUNCTIONS) and
      not((Component is ObjFuncInfo[i].ObjClassRef) and
          (CompText(ObjFuncInfo[i].FuncName,FuncName)=0)) do Inc(i);
if i<=prMAX_OBJFUNCTIONS then
  Result:=@(ObjFuncInfo[i])
else
  Result:=nil;
end;

//
//   
//  true,      
//   ,     
//   ,    .
//     Expr = $ID
//        
//      ,
//     .
//
//  :
//  -  ,   PagesCount+2
//  -  false, Expr = $0,    
//          0,    
//    PagesCount   $PagesCount0,  0  ,   
//    Fversions
//  -  ,   $0,    
//        
//
function TprParser.CompileExpression;
type
  //   
  //   toValue    -  
  //   toObjName  -    
  //   toFunction - 
  TTypeObj = (toValue,toObjName,toStack,toOperator,toFunction);
  TObj = record
    oType : TTypeObj; //  
    oIndex : integer;  //  (  toStack)
    oName : string;   // 
    oValue : TprVarValue;// 
    oParCount : integer;  //    
    oTemp : integer;  // ,    
  end;

  TObjs  = array of TObj;
  TStorageObjs = array of TObjs;

var
  f : boolean;
  i,j,l1,l2,bCount : integer;
  sObjs : TStorageObjs; //        ProcessNonBracketExpression
  LastName,b1,ExprTemp : string;
  tempVars : TprVarsArray;
  e : TprExprItem;
  v : TprVarValue;
  Component : TComponent;
  pofi : PprObjFuncInfo;
  popi : PprObjPropInfo;
  IsProcessed : boolean;

  //
  //       
  //      ,  
  //    TcExpr
  //
  procedure ProcessExpression(var ss : string);
  var
    vd : extended;
    vi,valCode,i,NumArgs,sp,ep : integer;       //       
    b1,BracketExpr : string;         //    
    EndBracketFlag : boolean;        // true -   
    BracketExprType : TprTypeBracket; //     -    

    //    
    procedure ProcessBracket;
    var
      fl : boolean;
    begin
    sp:=1;
    ep:=1;
    fl:=true;
    while (ep<=Length(ss)) and ((not fl) or (ss[ep]<>S_EndBracketChar)) do
      begin
        if fl and (ss[ep]=S_StartBracketChar) then
          begin
            sp:=ep;
            if (sp>1) and not(ss[sp-1] in S_SpecialChars) then
              BracketExprType:=prtbFunction
            else
              BracketExprType:=prtbExpression;
          end;
        fl:=fl xor (ss[ep]=S_StringChar);
        Inc(ep);
      end;

    EndBracketFlag:=(sp=1) and (ep>Length(ss)); // True    
    if EndBracketFlag then
      begin
        BracketExpr:=ss;
        BracketExprType:=prtbExpression;
      end
    else
      begin
        BracketExpr:=Copy(ss,sp+1,ep-sp-1)
      end;
    end;

    //
    //  ,     
    //    sObjs    ,    
    //
    procedure ProcessNonBracketExpression(NBE : string);
    type
      TOperandType = (otValue,otName,otStack);
      TOperand = record
        oType        : TOperandType;
        oName        : string;
        oIndex       : integer;
        oValue       : TprVarValue;
        UseOperation : integer;
      end;

      TOperator = record
        name   : string;
        index  : integer;
        oLeft  : integer;  //    ,    oLeft+1
      end;
    var
      i,j,k,hi,opi,lNBE,lOperands,lOperators : integer;
      b1 : string;
      Value : TprVarValue;
      Operands : array of TOperand;
      Operators : array of TOperator;

      //   sObjs[hi]      
      //  n
      procedure InsertAfter(n : integer);
      var
        i : integer;
      begin
      SetLength(sObjs[hi],Length(sObjs[hi])+1);
      for i:=High(sObjs[hi]) downto n+1 do
        with sObjs[hi][i] do
          begin
            oType :=sObjs[hi][i-1].otype;
            oIndex:=sObjs[hi][i-1].oIndex;
            oName :=sObjs[hi][i-1].oName;
            oValue:=sObjs[hi][i-1].oValue;
            oTemp :=sObjs[hi][i-1].oTemp
          end
      end;

    begin
    //   
    try
      lOperands :=0;
      lOperators:=0;
      lNBE      :=Length(NBE);
      i         :=1;
      while i<=lNBE do
        begin
          if NBE[i] in S_StartIdentChars then
            begin
              // 
              b1:='';
              while (i<=lNBE) and (NBE[i] in S_IdentChars) do
                begin
                  b1:=b1+NBE[i];
                  Inc(i);
                end;
              Inc(lOperands);
              SetLength(Operands,lOperands);
              with Operands[lOperands-1] do
                begin
                  oType       :=otName;
                  oName       :=b1;
                  UseOperation:=-1;
                end;
            end
          else
          if NBE[i] = S_StringChar then
            begin
              //  
              Inc(i);
              b1:='';
              while (i<=lNBE) and (NBE[i]<>S_StringChar) do
                begin
                  b1:=b1+NBE[i];
                  Inc(i);
                end;
              if i>lNBE then
                raise Exception.Create(prLoadStr(sParserErrorString));
              Inc(i);
              Inc(lOperands);
              SetLength(Operands,lOperands);
              with Operands[lOperands-1] do
                begin
                  oType       :=otValue;
                  UseOperation:=-1;
                  _vSetAsString(oValue,b1);
                end;
            end
          else
          if (NBE[i] in S_NumberDateChars) then
            begin
              //    
              b1 := '';
              while (i<=lNBE) and (NBE[i] in S_NumberDateChars) do
                begin
                  b1 := b1+NBE[i];
                  Inc(i);
                end;

              val(b1,vi,valCode);
              if valCode=0 then
                _vSetAsInteger(Value,vi)
              else
                if TextToFloat(PChar(b1),vd,fvExtended) then
                  _vSetAsDouble(Value,vd)
                else
                  try
                    _vSetAsDateTime(Value,StrToDateTime(b1));
                  except
                    // ,    
                    raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownOperandType),[b1]);
                  end;

              Inc(lOperands);
              SetLength(Operands,lOperands);
              with Operands[lOperands-1] do
                begin
                  oType       :=otValue;
                  UseOperation:=-1;
                  _vCopy(Value,oValue);
                end;
            end
          else
          if NBE[i]='_' then
            begin
              //   otStack
              Inc(i);
              b1:='';
              while (i<=lNBE) and (NBE[i] in ['0'..'9']) do
                begin
                  b1:=b1+NBE[i];
                  Inc(i);
                end;
              Inc(lOperands);
              SetLength(Operands,lOperands);
              with Operands[lOperands-1] do
                begin
                  oType       :=otStack;
                  oIndex      :=StrToInt(b1);
                  UseOperation:=-1;
                end;
            end
          else
            begin
              //      
              b1:='';
              while (i<=lNBE) and (NBE[i] in S_OperatorChars) do
                begin
                  b1:=b1+NBE[i];
                  Inc(i);
                end;
              //     
              j:=1;
              while (j<=prMAX_CALCOPERATORS) and (CompText(CO[j].Name,b1)<>0) do Inc(j);
              if j>prMAX_CALCOPERATORS then
                raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownOperator),[b1]);
              Inc(lOperators);
              SetLength(Operators,lOperators);
              with Operators[lOperators-1] do
                begin
                  index:=j;
                  name :=b1;
                  oLeft:=lOperands-1;
                end;
            end;
        end;

      //  
      //   sObjs
      SetLength(sObjs,Length(sObjs)+1);
      hi:=High(sObjs);
      SetLength(sObjs[hi],lOperands);
      //   sObjs[hi]  
      for i:=0 to lOperands-1 do
        with sObjs[hi][i] do
          begin
            oType :=TTypeObj(Operands[i].oType);
            oName :=Operands[i].oName;
            oIndex:=Operands[i].oIndex;
            oValue:=Operands[i].oValue;
            oTemp :=i;
          end;


      //    
      for i:=1 to prMAX_CALCOPERATORS do
        for j:=0 to lOperators-1 do
          if Operators[j].index=i then
            begin
              //     ,   , 
              //     - ,     
              //  
              if Operators[j].oLeft+1>=lOperands then
                Exception.Create(prLoadStr(sParserErrorCommonError));

              if Operands[Operators[j].oLeft+1].UseOperation>=0 then
                begin
                  //     UseOperation
                  k  :=Operators[j].oLeft+1;
                  opi:=Operands[k].UseOperation;
                  while (opi>=0) and
                        (Operators[opi].index<i) and
                        (Operands[Operators[opi].Oleft].UseOperation<>Operands[Operators[opi].Oleft+1].UseOperation) do
                    begin
                      k  :=Operators[opi].OLeft+1;
                      opi:=Operands[k].UseOperation;
                    end;

                  //    (opi)
                  k:=0;
                  while (k<=High(sObjs[hi])) and
                        not ((sObjs[hi][k].oType=toOperator) and
                             (sObjs[hi][k].oTemp=opi)) do Inc(k);
                  if k>High(sObjs[hi]) then
                    Exception.Create(prLoadStr(sParserErrorCommonError));
                  //          opi
                  while (k<=High(sObjs[hi])) and
                        (sObjs[hi][k].oType=toOperator) do Inc(k);

                  InsertAfter(k-1);
                  with sObjs[hi][k] do
                    begin
                      oType:=toOperator;
                      oName:=Operators[j].name;
                      oTemp:=j;
                    end;
                  if (Operators[j].oLeft>=0) and (High(Operands)>=Operators[j].oLeft) then
                    Operands[Operators[j].oLeft].UseOperation:=j;
                end
              else
                begin
                  //    Operands[Operators[j].oLeft+1]
                  //     sObjs
                  k:=0;
                  while (k<=High(sObjs[hi])) and
                        not ((sObjs[hi][k].oType in [toValue,toObjName,toStack]) and
                             (sObjs[hi][k].oTemp=Operators[j].oLeft+1)) do Inc(k);
                  if k>High(sObjs[hi]) then
                    Exception.Create(prLoadStr(sParserErrorCommonError));
                  InsertAfter(k);
                  with sObjs[hi][k+1] do
                    begin
                      oType:=toOperator;
                      oName:=Operators[j].name;
                      oTemp:=j;
                    end;
                  if (Operators[j].oLeft>=0) and (High(Operands)>=Operators[j].oLeft) then
                    Operands[Operators[j].oLeft].UseOperation:=j;
                  if (Operators[j].oLeft+1>=0) and (High(Operands)>=Operators[j].oLeft+1) then
                    Operands[Operators[j].oLeft+1].UseOperation:=j;
                end;

            end;
    finally
      Finalize(Operands);
      Finalize(Operators);
    end
    end;

  begin
  ProcessBracket;

  //  BracketExpr
  //  
  if BracketExprType=prtbFunction then
    begin
      //  
      //  
      i:=1;
      NumArgs:=0;
      while True do
        begin
          b1:=PExtractSubStr(BracketExpr,i);
          if b1='' then break;
          ProcessNonBracketExpression(b1);
          Inc(NumArgs);
        end;

      //  
      Dec(sp);
      b1:='';
      while (sp>0) and not (ss[sp] in S_SpecialChars) do
        begin
          b1:=ss[sp]+b1;
          Dec(sp);
        end;

      // b1 -  ,    ,     
      //     TObjs  sObjs
      SetLength(sObjs,Length(sObjs)+1);
      SetLength(sObjs[High(sObjs)],NumArgs+1);
      for i:=0 to NumArgs-1 do
        with sObjs[High(sObjs)][i] do
          begin
            oType :=toStack;
            oIndex:=High(sObjs)-i-1;
          end;
      with sObjs[High(sObjs)][NumArgs] do
        begin
          oType    :=toFunction;
          oName    :=b1;
          oParCount:=NumArgs;
        end;

      //  ss
      ss:=Copy(ss,1,sp)+'_'+IntToStr(High(sObjs))+Copy(ss,ep+1,Length(ss));
    end
  else
    begin
      //     
      ProcessNonBracketExpression(BracketExpr);

      //  ss
      ss:=Copy(ss,1,sp-1)+'_'+IntToStr(High(sObjs))+Copy(ss,ep+1,Length(ss));
    end;

  if not EndBracketFlag then
    ProcessExpression(ss);
  end;

  //
  //  cExpr   sObj 
  //  ,  toObjName,    
  //   Values
  //  true,      , 
  //  
  //
  function AddFromsObj(objs : TObjs) : boolean;
  var
    i,j,hi : integer;
  begin
  Result:=true;
  for i:=0 to High(Objs) do
    begin
      if Objs[i].oType=toStack then
        Result:=AddFromsObj(sObjs[Objs[i].oIndex]) and Result
      else
        begin
          SetLength(cExpr,Length(cExpr)+1);
          hi:=High(cExpr);
          cExpr[hi].ObjIndex:=-1;
          case Objs[i].oType of
            toValue   :
              begin
                with cExpr[hi] do
                  begin
                    ObjType :=preiValue;
                    objValue:=Objs[i].oValue;
                  end;
              end;
            toOperator:
              begin
                j:=1;
                while (j<=prMAX_CALCOPERATORS) and (CompText(CO[j].name,Objs[i].oName)<>0) do Inc(j);
                if j>prMAX_CALCOPERATORS then
                  raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownOperator),[Objs[i].oName]);
                with cExpr[hi] do
                  begin
                    ObjType :=preiOperator;
                    ObjName :=Objs[i].oName;
                    ObjIndex:=j;
                  end;
              end;
            toObjName :
              begin
                with cExpr[hi] do
                  begin
                    j:=Report.SystemValues.IndexByName(Objs[i].oName);
                    if j<>-1 then
                      begin
                        ObjType:=preiReportSystemVariable;
                        ObjName:=Report.SystemValues[j].GetCurrentVersionID;
                        Result :=false;
                      end
                    else
                      begin
                        j:=Report.Values.IndexByName(Objs[i].oName);
                        if j<>-1 then
                          begin
                            ObjType:=preiReportVariable;
                            ObjName:=Report.Values[j].GetCurrentVersionID;
                            Result :=false;
                          end
                        else
                          begin
                            if pos('.',Objs[i].oName)<>0 then
                              ObjType:=preiObjectProperty
                            else
                              ObjType:=preiVar;
                            ObjName:=Objs[i].oName;
                          end;
                      end;
                end;
              end;
            toFunction:
              begin
                with cExpr[hi] do
                  begin
                    if pos('.',Objs[i].oName)<>0 then
                      ObjType:=preiObjectFunction
                    else
                      begin
                        ObjType:=preiFunction;
                        j:=1;
                        while (j<=prMAX_FUNCTIONS) and (CompText(FuncInfo[j].name,Objs[i].oName)<>0) do Inc(j);
                        if j>prMAX_FUNCTIONS then ObjIndex:=-1
                                             else ObjIndex:=j;
                      end;
                    ObjName:=Objs[i].oName;
                    ObjParCount:=Objs[i].oParCount;
                end;
              end;
          end;
        end;
    end;
  end;

begin
Result := false;
SetLength(cExpr,0);
try
  try
    //         
    ExprTemp := '';
    f := false;
    for i:=1 to Length(Expr) do
      begin
        b1 := Expr[i];
        f := f xor (b1=S_StringChar);
        if (b1<>' ') or f then
          ExprTemp := ExprTemp+b1;
      end;
    if ExprTemp='' then
      begin
        Result := true;
        exit; //  
      end;

    b1 := Copy(ExprTemp,2,Length(ExprTemp));
    if (ExprTemp[1]='$') and
       (b1<>'') and
       IsThisCharsOnly(b1,['0'..'9']) then
      begin
        //     
        i := StrToInt(b1);
        if i<=High(FExprs) then
          begin
            l1 := Length(FExprs[i]);
            SetLength(cExpr,l1);
            for j:=0 to l1-1 do
              cExpr[j] := FExprs[i][j];
            Result := true;
          end
      end
    else
      begin
        //     
        f := true;
        bCount := 0;
        for i:=1 to Length(ExprTemp) do
          begin
            if f then
              begin
                if ExprTemp[i]=S_StartBracketChar then
                  Inc(bCount)
                else
                  if ExprTemp[i]=S_EndBracketChar then
                    Dec(bCount);
              end;
            f := f xor (ExprTemp[i]=S_StringChar);
          end;
        if bCount<>0 then
          raise Exception.Create(prLoadStr(sParserErrorInvalidBrackets));
    
        ProcessExpression(ExprTemp);
        //   sObjs -    
        //      cExpr
        Result := AddFromsObj(sObjs[High(sObjs)]);
        if not Result then
          begin
            //    
            //
            // !!!     : !!!
            // !!!  ,   !!!
            // !!!         !!!

            l1 := Length(FExprs);
            l2 := Length(cExpr);
            SetLength(FExprs,l1+1);
            SetLength(FExprs[l1],l2);

            for i:=0 to l2-1 do
              begin
                e:=cExpr[i];
                if e.ObjType in [preiObjectProperty,preiObjectFunction] then
                  begin
                    Report.TranslateObjectName(e.ObjName,Component,LastName);

                    if Component=nil then
                      raise Exception.CreateFmt(prLoadStr(sParserErrorObjectNotFound),[e.ObjName]);

                    //   ,  
                    case e.ObjType of
                      preiObjectFunction:
                        begin
                          //      
                          j:=1;
                          while (j<=e.ObjParCount) and (cExpr[i-j].ObjType=preiValue) do Inc(j);

                          if j>e.ObjParCount then
                            begin
                              //   
                              j:=1;
                              SetLength(tempVars,e.ObjParCount);
                              while (j<=e.ObjParCount) do
                                begin
                                  _vCopy(cExpr[i-j].ObjValue,TempVars[j-1]);
                                  Inc(j);
                                end;

                              pofi:=FindObjFuncInfo(Component,LastName);
                              if pofi=nil then
                                begin
                                  if not Assigned(Report.OnUnknownObjFunction) then
                                    raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownFunc),[e.ObjName]);

                                  IsProcessed:=false;
                                  Report.OnUnknownObjFunction(Report,Component,e.ObjName,TempVars,e.ObjParCount,v,IsProcessed);
                                  if not IsProcessed then
                                    raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownFunc),[e.ObjName]);
                                end
                              else
                                begin
                                  if not (((pofi.min=-1) or (e.ObjParCount>=pofi.min)) and
                                          ((pofi.max=-1) or (e.ObjParCount<=pofi.max))) then
                                    raise Exception.CreateFmt(prLoadStr(sParserErrorInvalidFuncParametersCount),[LastName,e.ObjParCount,pofi.min,pofi.max]);
                                  pofi.Func(Self,Component,tempVars,v)
                                end;
                            end;
                        end;

                      preiObjectProperty:
                        begin
                          //   ,  
                          popi:=FindObjPropInfo(Component);
                          if popi=nil then
                            raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownProp),[LastName,Component.className]);
                          popi.Func(Component,LastName,v)
                        end;
                    end;

                    // v -   
                    with FExprs[l1][i] do
                      begin
                        ObjType     :=preiValue;
                        _vCopy(v,ObjValue);
                      end;
                  end
                else
                  FExprs[l1][i]:=cExpr[i];
              end;

            Expr:=Format('$%d',[l1]);
          end;
      end;

  except
    on E : Exception do
      raise Exception.CreateFmt(prLoadStr(sParserErrorCompile),[Expr,E.Message])
  end
finally
  for i:=0 to High(sObjs) do
    Finalize(sObjs[i]);
  Finalize(sObjs);
end;
end;

//
//     Expr
//
procedure TprParser.CalcExpression;
var
  e : TprExprItem;
  v : TprVarValue;
  i,j : integer;
  ver : TprValueVersion;
  Vars : TprVarsArray;
  pofi : PprObjFuncInfo;
  popi : PprObjPropInfo;
  LastName : string;
  Component : TComponent;
  IsProcessed : boolean;

  procedure ReadParams(n : integer);
  var
    i : integer;
  begin
  SetLength(Vars,n);
  for i:=0 to n-1 do
    _vCopy(rStack^,Vars[i]);
  end;

  procedure CheckParams(min,max,v : integer; FuncName : string);
  begin
  if not (((min=-1) or (v>=min)) and ((max=-1) or (v<=max))) then
    raise Exception.CreateFmt(prLoadStr(sParserErrorInvalidFuncParametersCount),[FuncName,v,min,max]);
  end;

begin
try
  if Length(Expr)=0 then
    begin
      _vSetAsBoolean(Res,false);
      exit;
    end;
  for i:=0 to High(Expr) do
    begin
      e:=Expr[i];
      case e.ObjType of
        preiValue:
          begin
            wStack(e.ObjValue);
          end;
        preiOperator:
          begin
            co[e.ObjIndex].Proc(Self,v);
            _vCopy(v,Stack[wStack]);
          end;
        preiFunction:
          begin
            //     FuncInfo,  
            ReadParams(e.ObjParCount);
            if e.ObjIndex<>-1 then
              begin
                CheckParams(FuncInfo[e.ObjIndex].min,
                            FuncInfo[e.ObjIndex].max,
                            e.ObjParCount,
                            e.ObjName);
                FuncInfo[e.ObjIndex].Func(Self,Vars,Stack[wStack])
              end
            else
              begin
                //  
                IsProcessed:=false;
                if Assigned(Report.OnUnknownFunction) then
                  begin
                    Report.OnUnknownFunction(Report,e.ObjName,Vars,e.ObjParCount,v,IsProcessed);
                    if IsProcessed then
                      wStack(v)
                  end;
                if not IsProcessed then
                  raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownFunc),[e.ObjName]);
              end;
          end;
        preiReportVariable:
          begin
            // e.ObjName -   
            ver := Report.Values.VersionByVersionID(e.ObjName);
            if ver<>nil then
              begin
                _vSetAsVariant(v,ver.V);
                wStack(v);
              end
            else
              raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownReportVariableVersion),[e.ObjName]);
          end;
        preiReportSystemVariable:
          begin
            // e.ObjName -   
            ver:=Report.SystemValues.VersionByVersionID(e.ObjName);
            if ver<>nil then
              begin
                _vSetAsVariant(v,ver.V);
                wStack(v);
              end
            else
              raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownReportVariableVersion),[e.ObjName]);
          end;
        preiVar:
          begin
            //        
            //   Report,     Report
            j:=0;
            while (j<LReportProps.Count) and (CompText(LReportProps[j],e.ObjName)<>0) do Inc(j);
            if j>=LReportProps.Count then
              begin
                //   ,   
                IsProcessed:=false;
                if Assigned(Report.OnUnknownVariable) then
                  begin
                    Report.OnUnknownVariable(Report,e.ObjName,v,IsProcessed);
                    if IsProcessed then
                      begin
                        wStack(v);
                      end;
                  end;
                if not IsProcessed then
                  raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownVariable),[e.ObjName]);
              end
            else
              begin
                ObjPropInfo[REPORT_PROPS_INFO_INDEX].Func(Report,e.ObjName,v);
                wStack(v);
              end;
          end;
        preiObjectProperty,preiObjectFunction:
          begin
            Report.TranslateObjectName(e.ObjName,Component,LastName);

            if Component=nil then
              raise Exception.CreateFmt(prLoadStr(sParserErrorObjectNotFound),[e.ObjName]);

            //   ,  
            if e.ObjType=preiObjectFunction then
              begin
                ReadParams(e.ObjParCount);
                pofi := FindObjFuncInfo(Component,LastName);

                if pofi<>nil then
                  begin
                    CheckParams(pofi.min,
                                pofi.max,
                                e.ObjParCount,
                                e.ObjName);
                    pofi.Func(Self,Component,Vars,v)
                  end
                else
                  begin
                    //  
                    IsProcessed:=false;
                    if Assigned(Report.OnUnknownObjFunction) then
                      begin
                        Report.OnUnknownObjFunction(Report,Component,e.ObjName,Vars,e.ObjParCount,v,IsProcessed);
                        if IsProcessed then
                          wStack(v)
                      end;
                    if not IsProcessed then
                      raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownFunc),[e.ObjName]);
                  end;
              end
            else
              begin
                //   ,  
                popi:=FindObjPropInfo(Component);
                if popi=nil then
                  begin
                    IsProcessed:=false;
                    if Assigned(Report.OnUnknownObjProp) then
                      begin
                        Report.OnUnknownObjProp(Report,Component,LastName,e.ObjName,v,IsProcessed);
                        if IsProcessed then
                          wStack(v)
                      end;
                    if not IsProcessed then
                      raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownProp),[LastName,Component.ClassName]);
                  end
                else
                  popi.Func(Component,LastName,v)
              end;

            _vCopy(v,Stack[wStack]);
          end;
      end;
    end;
  _vCopy(rStack^,Res);
except
  on E : Exception do
    begin
      ResetStack;
      _vSetNull(Res);
      raise Exception.CreateFmt(prLoadStr(sParserErrorCalcExpression),[E.Message]);
    end;
end
end;

//
//   
// :[N Null ][ ][<>]
//  Null :
//       Null ,     Null
//      ( ),   
//   S -     Null,    
//         
//   D -     Null,    
//          ''
//   N -     Null,    
//        0
//
// :
//           
//   ,     
// :
//   r -   
//   l -   
//   c -  
//  
//        "<>"
//
// :
// :NN<0.00>
//
function TprParser.FormatTemplateEx;
var
  v : TprVarValue;
  nv : Variant;
  Align : char;
  fRTF,fFormatNull,f : boolean;
  FullMask,FormatMask,b2 : string;
  iStartBrackets,i,j,k,Width,LenMask,Lenb2 : integer;

  procedure CheckChar;
  begin
  if i>LenMask then
    raise Exception.Create(prLoadStr(sParserErrorInvalidFormatString));
  end;

  procedure NextChar;
  begin
  Inc(i);
  CheckChar;
  end;

  procedure CheckCharb2;
  begin
  if j>Lenb2 then
    raise Exception.Create(prLoadStr(sParserErrorInvalidFormatString));
  end;

  procedure NextCharb2;
  begin
  Inc(j);
  CheckCharb2;
  end;

begin
Result := true;
try
  Res := '';
  i := 1;
  LenMask := Length(Mask);
  while i<=LenMask do
    begin
      if (Mask[i] = '[') and (i+1<=LenMask) then
        begin
          //     
          iStartBrackets := i;
          f := false;
          b2 := '';
          NextChar;
          while (i<=LenMask) and (f or (Mask[i]<>']')) do
            begin
              f := f xor (Mask[i]='"');
              b2 := b2+Mask[i];
              Inc(i);
            end;
          if i>LenMask then
            break;
          Inc(i);

          // b2 -    , i    
          //     [:]    
          fRTF := false;
          Lenb2 := Length(b2);
          if Lenb2>0 then
            begin
              j := 1;
              FormatMask := '';
              FullMask := '';
              Width := 0;
              Align := 'r';
              fFormatNull := false;
              nv := 0;
              if b2[j]=':' then
                begin
                  //  
                  FullMask := ':';
                  NextCharb2;
                  // Null 
                  fFormatNull := b2[j] in ['n','N'];
                  if fFormatNull then
                    begin
                      FullMask := FullMask+'N';
                      NextCharb2;
                      case b2[j] of
                        'd','D': begin FullMask := FullMask+'D'; NextCharb2; nv := VarAsType(0,varDate); end;
                        's','S': begin FullMask := FullMask+'S'; NextCharb2; nv := ''; end;
                        'n','N': begin FullMask := FullMask+'N'; NextCharb2; nv := 0; end;
                        else begin FullMask := FullMask+'N'; nv := 0; end;
                      end;
                    end;
                  //   
                  k := j;
                  while (j<=Lenb2) and (b2[j] in ['0'..'9']) do Inc(j);
                  if k<>j then
                    begin
                      val(Copy(b2,k,j-k),Width,k);
                      FullMask := FullMask+IntToStr(Width);
                      CheckCharb2;
                      Align := b2[j];
                      FullMask := FullMask+Align;
                      NextCharb2;
                    end;
                  //   FormatMask
                  if b2[j]='<' then
                    begin
                      FullMask := FullMask+'<';
                      NextCharb2;
                      while (j<=Lenb2) and (b2[j]<>'>') do
                        begin
                          FormatMask := FormatMask+b2[j];
                          Inc(j);
                        end;
                      FullMask := FullMask+FormatMask+'>';
                      NextCharb2;
                      if Length(FormatMask)=1 then
                        case FormatMask[1] of
                          'r','R' : FormatMask := SimpleCurrencyFormat;
                          's','S' : FormatMask := ShortCurrencyFormat;
                          'c','C' : FormatMask := CurrencyFormat;
                          'p','P' : FormatMask := PercentFormat;
                          'd','D' : FormatMask := ShortDateFormat;
                          'q','Q' : FormatMask := SpacedCurrencyFormat;
                          'b','B' : FormatMask := BankCurrencyFormat;
                        end
                      else
                        fRTF := AnsiCompareText(FormatMask,'rtf')=0;
                    end;
                end;

              //     j
              b2 := Copy(b2,j,Lenb2);
              if Calc(b2,v) then
                begin
                  if fRTF then
                    b2 := _vAsString(v)
                  else
                    begin
                      //      FormatMask, Width, Align
                      if (fFormatNull or (v.vType<>prvvtNull)) then
                        begin
                          if v.vType=prvvtNull then
                            _vSetAsVariant(v,nv);
                          if FormatMask='' then b2 := _vAsString(v)
                                           else b2 := pFormat(FormatMask,v);

                          if Width<>0 then
                            begin
                              case Align of
                                'r','R': _RightStr(b2,Width);
                                'c','C': _CenterStr(b2,Width);
                                else _LeftStr(b2,Width);
                              end;
                            end;
                        end
                      else
                        b2 := '';
                    end;
                end
              else
                begin
                  b2 := '['+FullMask+b2+']';
                  Result := false;
                end;
            end;
          // b2 - ,   
          // i-iStartBrackets - ,   
          if Assigned(ReplaceCallBack) then
            begin
              if fRTF then
                ReplaceCallBack(iStartBrackets,i-iStartBrackets,PChar(b2),Length(b2),[prfrcRTF],CallBackData)
              else
                ReplaceCallBack(iStartBrackets,i-iStartBrackets,PChar(b2),Length(b2),[],CallBackData)
            end
          else
            Res := Res+b2
        end
      else
        if Mask[i] in ['\'] then
          begin
            iStartBrackets := i;
            NextChar;
            if Mask[i] in ['0'..'9'] then
              begin
                j := i;
                while (i<=Length(Mask)) and (Mask[i] in ['0'..'9']) do Inc(i);
                b2 := chr(StrToInt(Copy(Mask,j,i-j)));
              end
            else
              begin
                b2 := Mask[i];
                Inc(i);
              end;
            if Assigned(ReplaceCallBack) then
              ReplaceCallBack(iStartBrackets,i-iStartBrackets,PChar(b2),Length(b2),[],CallBackData)
            else
              Res := Res+b2
          end
        else
          begin
            if not Assigned(ReplaceCallBack) then
              Res := Res+Mask[i];
            Inc(i);
          end;
    end;
except
  on E : Exception do
    raise Exception.CreateFmt(prLoadStr(sParserErrorInFormatString),[i,E.Message]);
end;
end;

function TprParser.FormatTemplate;
begin
Result := FormatTemplateEx(Mask,nil,nil,Res);
end;

function TprParser.Calc;
var
  cExpr : TprExpr;
begin
try
  Result := CompileExpression(Expr,cExpr);
  if Result then
    CalcExpression(cExpr,Res);
finally
  Finalize(cExpr);
end;
end;

/////////////////////////////////////////////////////
//
//
//  (  )
//
//
/////////////////////////////////////////////////////
function IsEq(v1,v2 : double) : boolean;
begin
Result := Abs(v1-v2)<prDoublePrec;
end;

function NotIsEQ(v1,v2 : double) : boolean;
begin
Result:= Abs(v1-v2)>prDoublePrec;
end;

procedure _OpPlus;
var
  v1,v2 : PprVarValue;
begin
v2 := ML.rStack;
v1 := ML.rStack;
if (v1^.vType in [prvvtString,prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) and
   (v2^.vType in [prvvtString,prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) then
  begin
    if (v1^.vType=prvvtString) or (v2^.vType=prvvtString) then
      _vSetAsString(Res,_vAsString(v1^)+_vAsString(v2^))
    else
      if (v1^.vType=prvvtDouble) or (v2^.vType=prvvtDouble) then
        _vSetAsDouble(Res,_vAsDouble(v1^)+_vAsDouble(v2^))
      else
        if (v1^.vType=prvvtDateTime) and (v2^.vType=prvvtInteger) then
          _vSetAsDateTime(Res,IncDay(_vAsDateTime(v1^),_vAsInteger(v2^)))
        else
          if (v1^.vType=prvvtInteger) and (v2^.vType=prvvtDateTime) then
            _vSetAsDateTime(Res,IncDay(_vAsDateTime(v2^),_vAsInteger(v1^)))
          else
            if (v1^.vType=prvvtInteger) or (v2^.vType=prvvtInteger) then
              _vSetAsInteger(Res,_vAsInteger(v1^)+_vAsInteger(v2^))
            else
              if (v1^.vType=prvvtNull) and (v2^.vType=prvvtNull) then
                _vSetNull(Res)
              else
                raise Exception.Create(prLoadStr(sParserErrorIncompatibleTypes));
  end
else
  raise Exception.Create(prLoadStr(sParserErrorInvalidTypes));
end;

procedure _OpMinus;
var
  v1,v2 : PprVarValue;
begin
v2:=ML.rStack;
v1:=ML.rStack;
if (v1^.vType in [prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) and
   (v2^.vType in [prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) then
  begin
    if (v1^.vType=prvvtDateTime) and (v2^.vType=prvvtInteger) then
      _vSetAsDateTime(Res,IncDay(_vAsDateTime(v1^),_vAsInteger(v2^)))
    else
      if (v1^.vType=prvvtDateTime) and (v2^.vType=prvvtDateTime) then
        _vSetAsInteger(Res,DaysBetween(_vAsDateTime(v2^),_vAsDateTime(v1^)))
      else
        if (v1^.vType=prvvtDouble) or (v2^.vType=prvvtDouble) then
          _vSetAsDouble(Res,_vAsDouble(v1^)-_vAsDouble(v2^))
        else
          if (v1^.vType=prvvtInteger) or (v2^.vType=prvvtInteger) then
            _vSetAsInteger(Res,_vAsInteger(v1^)-_vAsInteger(v2^))
          else
            if (v1^.vType=prvvtNull) and (v2^.vType=prvvtNull) then
              _vSetNull(Res)
            else
              raise Exception.Create(prLoadStr(sParserErrorIncompatibleTypes));
  end
else
  raise Exception.Create(prLoadStr(sParserErrorInvalidTypes));
end;

procedure _OpMul;
var
  v1,v2 : PprVarValue;
begin
v2:=ML.rStack;
v1:=ML.rStack;
if (v1^.vType in [prvvtDouble,prvvtInteger,prvvtNull]) and
   (v2^.vType in [prvvtDouble,prvvtInteger,prvvtNull]) then
  begin
    if (v1^.vType=prvvtDouble) or (v2^.vType=prvvtDouble) then
      _vSetAsDouble(Res,_vAsDouble(v1^)*_vAsDouble(v2^))
    else
      if (v1^.vType=prvvtInteger) or (v2^.vType=prvvtInteger) then
        _vSetAsInteger(Res,_vAsInteger(v1^)*_vAsInteger(v2^))
      else
        if (v1^.vType=prvvtNull) and (v2^.vType=prvvtNull) then
          _vSetNull(Res)
        else
          raise Exception.Create(prLoadStr(sParserErrorIncompatibleTypes));
  end
else
  raise Exception.Create(prLoadStr(sParserErrorInvalidTypes));
end;

procedure _OpDiv;
var
  v1,v2 : PprVarValue;
begin
v2:=ML.rStack;
v1:=ML.rStack;
if (v1^.vType in [prvvtDouble,prvvtInteger,prvvtNull]) and
   (v2^.vType in [prvvtDouble,prvvtInteger,prvvtNull]) then
  begin
    if v2^.vType=prvvtNull then
      _vSetNull(Res)
    else
      if (v1^.vType=prvvtDouble) or (v2^.vType=prvvtDouble) then
        _vSetAsDouble(Res,_vAsDouble(v1^)/_vAsDouble(v2^))
      else
        if (v1^.vType=prvvtInteger) or (v2^.vType=prvvtInteger) then
          _vSetAsDouble(Res,_vAsInteger(v1^)/_vAsInteger(v2^))
        else
          raise Exception.Create(prLoadStr(sParserErrorIncompatibleTypes));
  end
else
  raise Exception.Create(prLoadStr(sParserErrorInvalidTypes));
end;

function __OpL(ML : TprParser; v1,v2 : PprVarValue) : boolean;
var
  vd1,vd2 : double;
begin
if (v1^.vType in [prvvtString,prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) and
   (v2^.vType in [prvvtString,prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) then
  begin
    if (v1^.vType=prvvtString) or (v2^.vType=prvvtString) then
      Result := _vAsString(v1^)<_vAsString(v2^)
    else
      if (v1^.vType=prvvtDouble) or (v2^.vType=prvvtDouble) then
        begin
          vd1 := _vAsDouble(v1^);
          vd2 := _vAsDouble(v2^);
          Result := NotIsEQ(vd1,vd2) and (vd1<vd2)
        end
      else
        if (v1^.vType=prvvtInteger) or (v2^.vType=prvvtInteger) then
          Result := _vAsInteger(v1^)<_vAsInteger(v2^)
        else
          if (v1^.vType=prvvtDateTime) or (v2^.vType=prvvtDateTime) then
            Result := _vAsDateTime(v1^)<_vAsDateTime(v2^)
          else
            if (v1^.vType=prvvtNull) and (v2^.vType=prvvtNull) then
              Result := false
            else
              raise Exception.Create(prLoadStr(sParserErrorIncompatibleTypes));
  end
else
  raise Exception.Create(prLoadStr(sParserErrorInvalidTypes));
end;

procedure _OpL;
var
  v1,v2 : PprVarValue;
begin
v2 := ML.rStack;
v1 := ML.rStack;
_vSetAsBoolean(Res,__OpL(ML,v1,v2));
end;

function __OpG(ML : TprParser; v1,v2 : PprVarValue) : boolean;
var
  vd1,vd2 : double;
begin
if (v1^.vType in [prvvtString,prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) and
   (v2^.vType in [prvvtString,prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) then
  begin
    if (v1^.vType=prvvtString) or (v2^.vType=prvvtString) then
      Result := _vAsString(v1^)>_vAsString(v2^)
    else
      if (v1^.vType=prvvtDouble) or (v2^.vType=prvvtDouble) then
        begin
          vd1 := _vAsDouble(v1^);
          vd2 := _vAsDouble(v2^);
          Result := NotIsEQ(vd1,vd2) and (vd1>vd2)
        end
      else
        if (v1^.vType=prvvtInteger) or (v2^.vType=prvvtInteger) then
          Result := _vAsInteger(v1^)>_vAsInteger(v2^)
        else
          if (v1^.vType=prvvtDateTime) or (v2^.vType=prvvtDateTime) then
            Result := _vAsDateTime(v1^)>_vAsDateTime(v2^)
          else
            if (v1^.vType=prvvtNull) and (v2^.vType=prvvtNull) then
              Result := false
            else
              raise Exception.Create(prLoadStr(sParserErrorIncompatibleTypes));
  end
else
  raise Exception.Create(prLoadStr(sParserErrorInvalidTypes));
end;

procedure _OpG;
var
  v1,v2 : PprVarValue;
begin
v2 := ML.rStack;
v1 := ML.rStack;
_vSetAsBoolean(Res,__OpG(ML,v1,v2));
end;

function __OpEQ(ML : TprParser; v1,v2 : PprVarValue) : boolean;
var
  vd1,vd2 : double;
begin
if (v1^.vType in [prvvtString,prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) and
   (v2^.vType in [prvvtString,prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) then
  begin
    if (v1^.vType=prvvtString) or (v2^.vType=prvvtString) then
      Result := _vAsString(v1^)=_vAsString(v2^)
    else
      if (v1^.vType=prvvtDouble) or (v2^.vType=prvvtDouble) then
        begin
          vd1 := _vAsDouble(v1^);
          vd2 := _vAsDouble(v2^);
          Result := IsEQ(vd1,vd2)
        end
      else
        if (v1^.vType=prvvtInteger) or (v2^.vType=prvvtInteger) then
          Result := _vAsInteger(v1^)=_vAsInteger(v2^)
        else
          if (v1^.vType=prvvtDateTime) or (v2^.vType=prvvtDateTime) then
            Result := _vAsDateTime(v1^)=_vAsDateTime(v2^)
          else
            if (v1^.vType=prvvtNull) and (v2^.vType=prvvtNull) then
              Result := true
            else
              raise Exception.Create(prLoadStr(sParserErrorIncompatibleTypes));
  end
else
  raise Exception.Create(prLoadStr(sParserErrorInvalidTypes));
end;

procedure _OpEQ;
var
  v1,v2 : PprVarValue;
begin
v2 := ML.rStack;
v1 := ML.rStack;
_vSetAsBoolean(Res,__OpEQ(ML,v1,v2));
end;

function __OpNEQ(ML : TprParser; v1,v2 : PprVarValue) : boolean;
var
  vd1,vd2 : double;
begin
if (v1^.vType in [prvvtString,prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) and
   (v2^.vType in [prvvtString,prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) then
  begin
    if (v1^.vType=prvvtString) or (v2^.vType=prvvtString) then
      Result := _vAsString(v1^)<>_vAsString(v2^)
    else
      if (v1^.vType=prvvtDouble) or (v2^.vType=prvvtDouble) then
        begin
          vd1 := _vAsDouble(v1^);
          vd2 := _vAsDouble(v2^);
          Result := NotIsEQ(vd1,vd2)
        end
      else
        if (v1^.vType=prvvtInteger) or (v2^.vType=prvvtInteger) then
          Result := _vAsInteger(v1^)<>_vAsInteger(v2^)
        else
          if (v1^.vType=prvvtDateTime) or (v2^.vType=prvvtDateTime) then
            Result := _vAsDateTime(v1^)<>_vAsDateTime(v2^)
          else
            if (v1^.vType=prvvtNull) and (v2^.vType=prvvtNull) then
              Result := false
            else
              raise Exception.Create(prLoadStr(sParserErrorIncompatibleTypes));
  end
else
  raise Exception.Create(prLoadStr(sParserErrorInvalidTypes));
end;

procedure _OpNEQ;
var
  v1,v2 : PprVarValue;
begin
v2 := ML.rStack;
v1 := ML.rStack;
_vSetAsBoolean(Res,__OpNEQ(ML,v1,v2));
end;

function __OpLEQ(ML : TprParser; v1,v2 : PprVarValue) : boolean;
var
  vd1,vd2 : double;
begin
if (v1^.vType in [prvvtString,prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) and
   (v2^.vType in [prvvtString,prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) then
  begin
    if (v1^.vType=prvvtString) or (v2^.vType=prvvtString) then
      Result := _vAsString(v1^)<=_vAsString(v2^)
    else
      if (v1^.vType=prvvtDouble) or (v2^.vType=prvvtDouble) then
        begin
          vd1 := _vAsDouble(v1^);
          vd2 := _vAsDouble(v2^);
          Result := IsEQ(vd1,vd2) or (vd1<vd2)
        end
      else
        if (v1^.vType=prvvtInteger) or (v2^.vType=prvvtInteger) then
          Result := _vAsInteger(v1^)<=_vAsInteger(v2^)
        else
          if (v1^.vType=prvvtDateTime) or (v2^.vType=prvvtDateTime) then
            Result := _vAsDateTime(v1^)<=_vAsDateTime(v2^)
          else
            if (v1^.vType=prvvtNull) and (v2^.vType=prvvtNull) then
              Result := true
            else
              raise Exception.Create(prLoadStr(sParserErrorIncompatibleTypes));
  end
else
  raise Exception.Create(prLoadStr(sParserErrorInvalidTypes));
end;

procedure _OpLEQ;
var
  v1,v2 : PprVarValue;
begin
v2 := ML.rStack;
v1 := ML.rStack;
_vSetAsBoolean(Res,__OpLEQ(ML,v1,v2));
end;

function __OpGEQ(ML : TprParser; v1,v2 : PprVarValue) : boolean;
var
  vd1,vd2 : double;
begin
if (v1^.vType in [prvvtString,prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) and
   (v2^.vType in [prvvtString,prvvtDouble,prvvtDateTime,prvvtInteger,prvvtNull]) then
  begin
    if (v1^.vType=prvvtString) or (v2^.vType=prvvtString) then
      Result := _vAsString(v1^)>=_vAsString(v2^)
    else
      if (v1^.vType=prvvtDouble) or (v2^.vType=prvvtDouble) then
        begin
          vd1 := _vAsDouble(v1^);
          vd2 := _vAsDouble(v2^);
          Result := IsEQ(vd1,vd2) or (vd1>vd2)
        end
      else
        if (v1^.vType=prvvtInteger) or (v2^.vType=prvvtInteger) then
          Result := _vAsInteger(v1^)>=_vAsInteger(v2^)
        else
          if (v1^.vType=prvvtDateTime) or (v2^.vType=prvvtDateTime) then
            Result := _vAsDateTime(v1^)>=_vAsDateTime(v2^)
          else
            if (v1^.vType=prvvtNull) and (v2^.vType=prvvtNull) then
              Result := true
            else
              raise Exception.Create(prLoadStr(sParserErrorIncompatibleTypes));
  end
else
  raise Exception.Create(prLoadStr(sParserErrorInvalidTypes));
end;

procedure _OpGEQ;
var
  v1,v2 : PprVarValue;
begin
v2 := ML.rStack;
v1 := ML.rStack;
_vSetAsBoolean(Res,__OpGEQ(ML,v1,v2));
end;

procedure _OpAnd;
var
  v1,v2 : PprVarValue;
begin
v2:=ML.rStack;
v1:=ML.rStack;
if (v1^.vType in [prvvtBoolean,prvvtInteger]) and
   (v2^.vType in [prvvtBoolean,prvvtInteger]) then
  begin
    if (v1^.vType=prvvtBoolean) and (v2^.vType=prvvtBoolean) then
      _vSetAsBoolean(Res,v1^.vBoolean and v2^.vBoolean)
    else
      if (v1^.vType=prvvtInteger) and (v2^.vType=prvvtInteger) then
        _vSetAsInteger(Res,v1^.vInteger and v2^.vInteger)
      else
        raise Exception.Create(prLoadStr(sParserErrorIncompatibleTypes));
  end
else
  raise Exception.Create(prLoadStr(sParserErrorInvalidTypes));
end;

procedure _OpOr;
var
  v1,v2 : PprVarValue;
begin
v2:=ML.rStack;
v1:=ML.rStack;
if (v1^.vType in [prvvtBoolean,prvvtInteger]) and
   (v2^.vType in [prvvtBoolean,prvvtInteger]) then
  begin
    if (v1^.vType=prvvtBoolean) and (v2^.vType=prvvtBoolean) then
      _vSetAsBoolean(Res,v1^.vBoolean or v2^.vBoolean)
    else
      if (v1^.vType=prvvtInteger) and (v2^.vType=prvvtInteger) then
        _vSetAsInteger(Res,v1^.vInteger or v2^.vInteger)
      else
        raise Exception.Create(prLoadStr(sParserErrorIncompatibleTypes));
  end
else
  raise Exception.Create(prLoadStr(sParserErrorInvalidTypes));
end;

procedure _OpNot;
var
  v1 : PprVarValue;
begin
v1:=ML.rStack;
if (v1^.vType in [prvvtBoolean,prvvtInteger]) then
  begin
    if (v1^.vType=prvvtBoolean) then
      _vSetAsBoolean(Res,not v1^.vBoolean)
    else
      if (v1^.vType=prvvtInteger) then
        _vSetAsInteger(Res,not v1^.vInteger)
      else
        raise Exception.Create(prLoadStr(sParserErrorIncompatibleTypes));
  end
else
  raise Exception.Create(prLoadStr(sParserErrorInvalidTypes));
end;

///////////////////////////////////////////
//
//
// 
//
//
////////////////////////////////////////////
procedure _DateTime;
begin
_vSetAsDateTime(Res,Now);
end;

procedure _Time;
begin
_vSetAsDateTime(Res,Time);
end;

procedure _AnsiUpperCase;
begin
_vSetAsString(Res,AnsiUpperCase(_vAsString(Parameters[0])));
end;

procedure _AnsiLowerCase;
begin
_vSetAsString(Res,AnsiLowerCase(_vAsString(Parameters[0])));
end;

procedure _Trim;
begin
_vSetAsString(Res,Trim(_vAsString(Parameters[0])));
end;

//
//         
// 0 - 
// 1 - 
// 2 - 
//
procedure _AddLeft;
begin
_vSetAsString(Res,AddChar(_vAsString(Parameters[1])[1],_vAsString(Parameters[0]),_vAsInteger(Parameters[2])));
end;

//
//         
// 0 - 
// 1 - 
// 2 - 
//
procedure _AddRight;
begin
_vSetAsString(Res,AddCharR(_vAsString(Parameters[1])[1],_vAsString(Parameters[0]),_vAsInteger(Parameters[2])));
end;

procedure _Length;
begin
_vSetAsInteger(Res,Length(_vAsString(Parameters[0])));
end;

//
//     ,
//    ,   
//
procedure _UID;
var
  s : string;
  i : integer;
begin
s := _vAsString(Parameters[0]);
for i:=1 to High(Parameters) do
  s := s+#01+_vAsString(Parameters[i]);
_vSetAsString(Res,s);
end;

//
// 0 - 
//    ,       
//  True
//
procedure _In;
var
  i,hi : integer;
  sBuf : string;
  bBuf : boolean;
  iBuf : integer;
  dtBuf : TDateTime;
  dBuf : double;
  oBuf : TObject;
begin
hi := High(Parameters);
i := 1;
case Parameters[0].vType of
  prvvtString   :
    begin
      sBuf:=_vAsString(Parameters[0]);
      while (i<=hi) and (sBuf<>_vAsString(Parameters[i])) do Inc(i);
    end;
  prvvtBoolean  :
    begin
      bBuf:=_vAsBoolean(Parameters[0]);
      while (i<=hi) and (bBuf=_vAsBoolean(Parameters[i])) do Inc(i);
    end;
  prvvtInteger  :
    begin
      iBuf:=_vAsInteger(Parameters[0]);
      while (i<=hi) and (iBuf<>_vAsInteger(Parameters[i])) do Inc(i);
    end;
  prvvtDateTime :
    begin
      dtBuf:=_vAsDateTime(Parameters[0]);
      while (i<=hi) and (dtBuf<>_vAsDateTime(Parameters[i])) do Inc(i);
    end;
  prvvtDouble   :
    begin
      dBuf:=_vAsDouble(Parameters[0]);
      while (i<=hi) and (not IsEQ(dBuf,_vAsDouble(Parameters[i]))) do Inc(i);
    end;
  prvvtObject   :
    begin
      oBuf:=_vAsObject(Parameters[0]);
      while (i<=hi) and (oBuf<>_vAsObject(Parameters[i])) do Inc(i);
    end;
end;
_vSetAsBoolean(Res,i<=hi);
end;

procedure _Null;
begin
_vSetNull(Res);
end;

procedure _Abs;
begin
if Parameters[0].vType=prvvtInteger then
  _vSetAsInteger(Res,Abs(_vAsInteger(Parameters[0])))
else
  _vSetAsDouble(Res,Abs(_vAsDouble(Parameters[0])));
end;

//
// 0 - YM
// 1 - 
//
procedure _IncYM;
begin
_vSetAsInteger(Res,IncYM(_vAsInteger(Parameters[0]),_vAsInteger(Parameters[1])));
end;

//
// 0 - 
// 1 - 
// 
// 0 - 
//
procedure _GetYM;
var
  y,m,d : word;
begin
if Parameters[0].vType=prvvtDateTime then
  begin
    DecodeDate(_vAsDateTime(Parameters[0]),y,m,d);
    _vSetAsInteger(Res,m+y*100);
  end
else
  _vSetAsInteger(Res,_vAsInteger(Parameters[0])+_vAsInteger(Parameters[1])*100);
end;

//
// 0 - YM
// 1 -   
//
procedure _GetDateFromYM;
begin
_vSetAsDateTime(Res,StrToDate(Format('%d.%d.%d',[_vAsInteger(Parameters[1]),
                                                 _vAsInteger(Parameters[0]) mod 100,
                                                 _vAsInteger(Parameters[0]) div 100])));
end;

//
// 0 -  
// 1 -   ()
// 2 -  
// 3 -   ()
//
procedure _HourBetween;
var
  dd1,dd2 : TDateTime;
  tt1,tt2 : string;
  d1,d2   : TDateTime;
begin
dd1 := _vAsDateTime(Parameters[0]);
dd2 := _vAsDateTime(Parameters[1]);
if High(Parameters)>=2 then tt1 := _vAsString(Parameters[2])
                       else tt1 := '00:00';
if High(Parameters)>=3 then tt2 := _vAsString(Parameters[3])
                       else tt2 := '00:00';

d1 := StrToDateTime(Format('%s %s:00',[DateToStr(CutTime(dd1)),tt1]));
d2 := StrToDateTime(Format('%s %s:00',[DateToStr(CutTime(dd2)),tt2]));
_vSetAsInteger(Res,integer(ExtRound((d2-d1)*24)));
end;

//
// 0 -  
// 1 -   (   )
//
procedure _MonthsBetween;
begin
_vSetAsInteger(Res,integer(Trunc(MonthsBetween(_vAsDateTime(Parameters[0]),_vAsDateTime(Parameters[1])))));
end;

//
// 0 -   
// 1 -   
//
procedure _DaysBetween;
begin
_vSetAsInteger(Res,DaysBetween(_vAsDateTime(Parameters[0]),_vAsDateTime(Parameters[1])));
end;

//
// 0 - 
// 1 -     
//
procedure _Round;
begin
if High(Parameters)>=1 then
  _vSetAsDouble(Res,RoundEx(_vAsDouble(Parameters[0]),_vAsInteger(Parameters[1])))
else
  _vSetAsDouble(Res,RoundEx(_vAsDouble(Parameters[0])));
end;

//
// 0 - 
//
procedure _Trunc;
begin
_vSetAsInteger(Res,Trunc(_vAsDouble(Parameters[0])))
end;

//
// 0 - 
//
procedure _GetMonth;
var
  d,m,y : Word;
begin
if Parameters[0].vType in [prvvtInteger] then
  begin
    _vSetAsInteger(Res,_vAsInteger(Parameters[0]) mod 100);
  end
else
  begin
    DecodeDate(_vAsDateTime(Parameters[0]),y,m,d);
    _vSetAsInteger(Res,m);
  end;
end;

procedure _GetDay;
var
  d,m,y : Word;
begin
DecodeDate(_vAsDateTime(Parameters[0]),y,m,d);
_vSetAsInteger(Res,d);
end;

//
// 0 - 
//
procedure _GetYear;
var
  d,m,y : Word;
begin
if Parameters[0].vType in [prvvtInteger] then
  begin
    _vSetAsInteger(Res,_vAsInteger(Parameters[0]) div 100);
  end
else
  begin
    DecodeDate(_vAsDateTime(Parameters[0]),y,m,d);
    _vSetAsInteger(Res,y);
  end;
end;

//
// 0 - 
// 1 - 
// 
// 0 - 
// 
// 0 - YM
//
procedure _GetFirstDayMonth;
var
  y,m,d : word;
begin
if High(Parameters)=0 then
  begin
    if Parameters[0].vType in [prvvtInteger] then
      begin
        y:=_vAsInteger(Parameters[0]) div 100;
        m:=_vAsInteger(Parameters[0]) mod 100;
      end
    else
      begin
        DecodeDate(_vAsDateTime(Parameters[0]),y,m,d);
      end;
  end
else
  if High(Parameters)=1 then
    begin
      m:=_vAsInteger(Parameters[0]);
      y:=_vAsInteger(Parameters[1]);
    end;

_vSetAsDateTime(Res,GetFirstDayMonth(m,y));
end;

//
// 0 - 
// 1 - 
// 
// 0 - 
// 
// 0 - YM
//
procedure _GetLastDayMonth;
var
  y,m,d : word;
begin
if High(Parameters)=0 then
  begin
    if Parameters[0].vType in [prvvtInteger] then
      begin
        y:=_vAsInteger(Parameters[0]) div 100;
        m:=_vAsInteger(Parameters[0]) mod 100;
      end
    else
      begin
        DecodeDate(_vAsDateTime(Parameters[0]),y,m,d);
      end;
  end
else
  if High(Parameters)=1 then
    begin
      m:=_vAsInteger(Parameters[0]);
      y:=_vAsInteger(Parameters[1]);
    end;

_vSetAsDateTime(Res,GetLastDayMonth(m,y));
end;

//
// 0 -   [1..12]
//
procedure _GetMonthName;
begin
_vSetAsString(Res,GetMonthName(_vAsInteger(Parameters[0])));
end;

//
// 0 - ,    ,   
// 1 - 
//
procedure _IncMonth;
begin
_vSetAsDateTime(Res,IncMonth(_vAsDateTime(Parameters[0]),_vAsInteger(Parameters[1])))
end;

//
// 0 - ,    ,   
// 1 - 
//
procedure _IncDay;
begin
_vSetAsDateTime(Res,IncDay(_vAsDateTime(Parameters[0]),_vAsInteger(Parameters[1])))
end;

procedure _Min;
var
  i : integer;
begin
_vCopy(Parameters[0],Res);
for i:=1 to High(Parameters) do
  if __OpL(ML,@(Parameters[i]),@Res) then
    _vCopy(Parameters[i],Res);
end;

procedure _Max;
var
  i : integer;
begin
_vCopy(Parameters[0],Res);
for i:=1 to High(Parameters) do
  if __OpG(ML,@(Parameters[i]),@Res) then
    _vCopy(Parameters[i],Res);
end;

//
// 0 - 1
// 1 - 2
// 2 - 
//
procedure _IIF;
begin
if _vAsBoolean(Parameters[2]) then
  _vCopy(Parameters[0],Res)
else
  _vCopy(Parameters[1],Res)
end;

//
// 0 - 
//
procedure _IsZero;
begin
_vSetAsBoolean(Res,(Parameters[0].vType=prvvtNull) or (_vAsInteger(Parameters[0])=0));
end;

//
// 0 -
//
procedure _IsNotZero;
begin
_vSetAsBoolean(Res,(Parameters[0].vType<>prvvtNull) and (_vAsInteger(Parameters[0])<>0));
end;

//
// 0 -  
// 1 -   (ntCurrency =0, ntSimple=1)
// 2 -    (cfNone, cfTwoDigit)
// 3 -    
// 4 -  gMale, gFemale
// 5 -   1
// 6 -   2..4
// 7 -   >4
// 8..10 -    - , , ,   :
//         , ,   ..
// 11..13 -    
//
procedure _GSN;
var
  Number : extended;
  NumberType : TNumberType;
  CentsFormat : TCentsFormat;
  CentsText : string;
  Gender : TGender;
  sOne : string;
  sTwoToFour : string;
  sOverFour : string;
  sOneRUB,sTwoToFourRUB,sOverFourRUB,sOneKOP,sTwoToFourKOP,sOverFourKOP : string;
begin
Number:=_vAsDouble(Parameters[0]);
if High(Parameters)>=1 then NumberType:=TNumberType(_vAsInteger(Parameters[1]))
                       else NumberType:=ntCurrency;
if High(Parameters)>=2 then CentsFormat:=TCentsFormat(_vAsInteger(Parameters[2]))
                       else CentsFormat:=cfTwoDigit;
if High(Parameters)>=3 then CentsText:=_vAsString(Parameters[3])
                       else CentsText:='.';
if High(Parameters)>=4 then Gender:=TGender(_vAsInteger(Parameters[4]))
                       else Gender:=gMale;
if High(Parameters)>=5 then sOne:=_vAsString(Parameters[5])
                       else sOne:='';
if High(Parameters)>=6 then sTwoToFour:=_vAsString(Parameters[6])
                       else sTwoToFour:='';
if High(Parameters)>=7 then sOverFour:=_vAsString(Parameters[7])
                       else sOverFour:='';
if High(Parameters)>=8 then sOneRUB:=_vAsString(Parameters[8])
                       else sOneRUB:='';
if High(Parameters)>=9 then sTwoToFourRUB:=_vAsString(Parameters[9])
                       else sTwoToFourRUB:='';
if High(Parameters)>=10 then sOverFourRUB:=_vAsString(Parameters[10])
                        else sOverFourRUB:='';
if High(Parameters)>=11 then sOneKOP:=_vAsString(Parameters[11])
                        else sOneKOP:='';
if High(Parameters)>=12 then sTwoToFourKOP:=_vAsString(Parameters[12])
                        else sTwoToFourKOP:='';
if High(Parameters)>=13 then sOverFourKOP:=_vAsString(Parameters[13])
                        else sOverFourKOP:='';
_vSetAsString(Res,GSN(Number,NumberType,CentsFormat,CentsText,Gender,sOne,sTwoToFour,sOverFour,sOneRUB,sTwoToFourRUB,sOverFourRUB,sOneKOP,sTwoToFourKOP,sOverFourKOP));
end;

//
//    
//   0 - 
//   1 -   
//   2 -   (  ,     )
//
procedure _Copy;
begin
if High(Parameters)>=2 then
  _vSetAsString(Res,Copy(_vAsString(Parameters[0]),_vAsInteger(Parameters[1]),_vAsInteger(Parameters[2])))
else
  _vSetAsString(Res,Copy(_vAsString(Parameters[0]),_vAsInteger(Parameters[1]),Length(_vAsString(Parameters[0]))))
end;

//
//     ,    
//   0 -  
//   1 -  ,   
//
procedure _MakeStr;
var
  c : char;
begin
if (High(Parameters)>=1) and (Length(_vAsString(Parameters[1]))>=1) then
  c:=_vAsString(Parameters[1])[1]
else
  c:=' ';

_vSetAsString(Res,MakeStr(c,_vAsInteger(Parameters[0])));
end;

/////////////////////////////////////////////
//
//
//  
//
//
/////////////////////////////////////////////
procedure _TDataSet_PropsList(C : TComponent; L : TStringList);
begin
L.Clear;
L.Add('RecordCount');
L.Add('RecNo');
end;

procedure _TprCustomReport_PropsList(C : TComponent; L : TStringList);
begin
L.Clear;
L.Add('StartDateTime');
end;

procedure _TDataSet_rProp;
var
  f : TField;
begin
with TDataSet(c) do
  begin
    if CompText(PropName,'RecordCount')=0 then
      begin
        _vSetAsInteger(Res,RecordCount);
      end
    else
      if CompText(PropName,'RecNo')=0 then
        begin
          _vSetAsInteger(Res,RecNo);
        end
      else
        begin
          f:=FindField(PropName);
          if f<>nil then
            begin
              VarFromField(f,Res);
            end
          else
            raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownProp),[PropName,c.ClassName]);
        end
  end;
end;

procedure _TprCustomReport_rProp;
begin
with TprCustomReport(c) do
  begin
    if CompText(PropName,'StartDateTime')=0 then
      _vSetAsDateTime(res,StartDateTime)
    else
      raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownProp),[PropName,c.ClassName]);
  end;
end;

procedure _TprDataSet_rProp;
var
  v : Variant;
begin
with TprDataSet(c) do
  begin
    if CompText(PropName,'RecordCount')=0 then
      begin
        _vSetAsInteger(Res,RecordCount);
      end
    else
      begin
        v:=GetFieldValue(PropName);
        if VarIsEmpty(v) then
          raise Exception.CreateFmt(prLoadStr(sParserErrorUnknownProp),[PropName,c.ClassName])
        else
          _vSetAsVariant(Res,v);
      end
  end;
end;

procedure _TprDataSet_PropsList(C : TComponent; L : TStringList);
begin
L.Clear;
L.Add('RecordCount');
end;

//////////////////////////////////////
//
//  
//
//////////////////////////////////////
//
//    DataSet, :
//   -   Dataset
//   -  
//
procedure _TDataSet_Locate;
var
  i : integer;
  s : string;
  va : array of Variant;
begin
if Length(Parameters) mod 2 <>0 then
  raise Exception.Create(prLoadStr(sParserErrorParamsCountForLocate));

SetLength(va,Length(Parameters) div 2);
s    :=_vAsString(Parameters[0]);
va[0]:=_vAsVariant(Parameters[1]);

i:=2;
while i<Length(Parameters) do
  begin
    s          :=s+';'+_vAsString(Parameters[i]);
    Inc(i);
    va[i div 2]:=_vAsVariant(Parameters[i]);
    Inc(i);
  end;

_vSetAsBoolean(Res,TDataSet(c).Locate(s,VarArrayOf(va),[]));
end;

procedure _TDataSet_Eof;
begin
_vSetAsBoolean(Res,TDataSet(C).Eof);
end;

//
// 0 -  
//
procedure _TDataSet_IsNullField;
begin
_vSetAsBoolean(Res,TDataSet(C).FieldByName(_vAsString(Parameters[0])).IsNull);
end;

//
// 0 -  
//
procedure _TDataSet_IsZeroField;
var
  f : TField;
begin
f:=TDataSet(C).FieldByName(_vAsString(Parameters[0]));
if f.IsNull then
  _vSetAsBoolean(Res,true)
else
  begin
    if f.DataType in [ftFloat,ftCurrency,ftBCD] then
      _vSetAsBoolean(Res,Abs(f.AsFloat)<prDoublePrec)
    else
      _vSetAsBoolean(Res,f.AsVariant=0);
  end;
end;

//
// 0 -  
//
procedure _TDataSet_LineNo;
begin
_vSetAsInteger(Res,ML.Report.GetDataSetRecNo(C));
end;

procedure _TprDataset_LineNo;
begin
_vSetAsInteger(Res,ML.Report.GetDataSetRecNo(C));
end;

procedure _TprGroup_LineNo;
begin
_vSetAsInteger(Res,TprGroup(C).LineNo);
end;

initialization

S_NumberDateChars := S_NumberDateChars+[DecimalSeparator];
S_NumberDateChars := S_NumberDateChars+[DateSeparator];

RegisterClass(TDataSet);
RegisterClass(TprGroup);
RegisterClass(TprCustomReport);
for i:=1 to prMAX_OBJPROPS do
  begin
    ObjPropInfo[i].ObjClassRef := GetClass(ObjPropInfo[i].ObjClass);
    if ObjPropInfo[i].ObjClassRef=nil then
      MBError(Format(prLoadStr(sParserErrorGetReferenceToClass),[ObjPropInfo[i].ObjClass]));
  end;

for i:=1 to prMAX_OBJFUNCTIONS do
  begin
    ObjFuncInfo[i].ObjClassRef := GetClass(ObjFuncInfo[i].ObjClass);
    if ObjFuncInfo[i].ObjClassRef=nil then
      MBError(Format(prLoadStr(sParserErrorGetReferenceToClass),[ObjFuncInfo[i].ObjClass]));
  end;

CurrencyFormat := prLoadStr(sCurrencyFormat);
ShortCurrencyFormat := prLoadStr(sShortCurrencyFormat);
SimpleCurrencyFormat := prLoadStr(sSimpleCurrencyFormat);
PercentFormat := prLoadStr(sPercentFormat);
SpacedCurrencyFormat := prLoadStr(sSpacedCurrencyFormat);
BankCurrencyFormat := prLoadStr(sBankCurrencyFormat); 

end.

