unit PolyList; {TPolymorphicList- Polymorphic Persistent List of Objects}
//---------------------------------------------------------------------
// (c) Copr 98 Omar Reis <omar@tecepe.com.br>
// Defines, Implements and Register TPolymorphicList = class(TComponent)
// See explanations after the end of the source code.
// version 1.0 - 98/11/01
// Free component
//---------------------------------------------------------------------
// history:
// 98/11/01 - ver 1.0 - omar reis
//---------------------------------------------------------------------

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DsgnIntf, TypInfo;

type
  TCreatePersistentEvent=Procedure(Sender:TObject;
    C:TPersistentClass;var aObj:TPersistent) of object;

  TPolymorphicList = class(TComponent)
  private
    fOwnObjects:boolean;
    fList:TList;
    fOnCreatePersistent:TCreatePersistentEvent;

    Function GetCount:integer;
    Function GetItem(index:integer):TPersistent;
  protected
    procedure DefineProperties(Filer: TFiler); override;
    Procedure ReadObjects(Reader:TReader);
    Procedure WriteObjects(Writer:TWriter);
  public
    Constructor Create(aOwner:TComponent); override;
    Destructor  Destroy;                   override;
    Procedure   Add(aObj:TPersistent);
    Procedure   Remove(aObj:TPersistent);
    procedure   Clear;
    Function    Edit:boolean; {dialog to edit this thing}
    Procedure   PickObjects(C:TPersistentClass;aList:TList);
    Procedure   SaveToStream(St:TStream);
    Procedure   ReadFromStream(St:TStream);

    Property    Count:integer read GetCount;
    Property    Items[index:integer]:TPersistent read GetItem;
  published
    Property  OwnObjects:boolean read fOwnObjects write fOwnObjects default FALSE;
    Property  OnCreatePersistent:TCreatePersistentEvent read fOnCreatePersistent write fOnCreatePersistent;
  end;

  TPolymorphicListEditor=class(TComponentEditor)
      function  GetVerbCount                     : Integer; override;
      function  GetVerb     ( nIndex : Integer ) : String;  override;
      procedure ExecuteVerb ( nIndex : Integer );           override;
  end;


procedure Register;

implementation

uses
  fPolyList;

type
  //TOmWriter and TOmReader are defined here to give access
  //to protected methods of TReader and TWriter ,like ReadProperty()
  TOmWriter=Class(TWriter);
  TOmReader=Class(TReader);

{ TPolymorphicList }

constructor TPolymorphicList.Create(aOwner: TComponent);
begin
  inherited;
  fList:=TList.Create;
  fOwnObjects:=FALSE;
end;

destructor TPolymorphicList.Destroy;
begin
  Clear;
  fList.Free; //we don't own the objects, so we don't dispose them.
  inherited;
end;

procedure TPolymorphicList.Clear;
var i:integer;
begin
  if fOwnObjects then for i:=0 to fList.Count-1 do
    TPersistent(fList.Items[i]).Free;
  fList.Clear;
end;

procedure TPolymorphicList.Add(aObj: TPersistent);
begin
  fList.Add(aObj);
end;

function TPolymorphicList.GetCount: integer;
begin
  Result:=fList.Count;
end;

function TPolymorphicList.GetItem(index: integer): TPersistent;
begin
  Result:=TPersistent(fList.Items[index]);
end;

procedure TPolymorphicList.Remove(aObj: TPersistent);
begin
  fList.Remove(aObj);
end;

procedure TPolymorphicList.DefineProperties(Filer:TFiler);
begin
  inherited DefineProperties(Filer);
  Filer.DefineProperty('ObjectData',ReadObjects,WriteObjects,(fList.Count>0));
end;

procedure TPolymorphicList.ReadObjects(Reader: TReader);
var aReader:TOmReader; aClassName:String;
    C:TPersistentClass; aObj:TPersistent;
begin
  aReader:=TOmReader(Reader); //get access to protected Reader methods
  while not aReader.EndOfList do
    begin
      aClassName:=aReader.ReadString;
      C:=FindClass(aClassName);
      // Components descendants override the TComponent.Create(),
      // so the correct component Create() is called :-)
      // but TPersistentClass.Create calls TPersistent.Create
      // and the appropriate descendant Create is not called :-(
      // OnCreatePersistent gives the user the chance to
      // call the correct Constructor
      // Normal handling would be like this:

      // procedure TForm1.PolyList1CreatePersistent(Sender:TObject;C:TPersistentClass;var aObj:TPersistent);
      // begin
      //   if C=TMyPersistent1 then aObj:=TMyPersistent1.Create
      //   else if C=TMyPersistent2 then aObj:=TMyPersistent2.Create;
      // end;
       if C.InheritsFrom(TComponent) then        //check if this is a component
        aObj:=TComponentClass(C).Create(nil)    //create it
        else begin  //nop. It's a TPersistent
          aObj:=nil;
          if Assigned(fOnCreatePersistent) then
            fOnCreatePersistent(Self,C,aObj);        //allow user to call the correct constructor
          if not Assigned(aObj) then aObj:=C.Create; //Not interested? Create it anyway..
        end;
      aReader.ReadListBegin; //read obj properties
      while not aReader.EndOfList do aReader.ReadProperty(aObj);
      aReader.ReadListEnd;
      fList.Add(aObj);        //add Obj to the List
    end;
  aReader.ReadListEnd;
end;

procedure TPolymorphicList.WriteObjects(Writer: TWriter);
var i,j:integer; aObj:TPersistent;
  PropInfo:PPropInfo;
  PropList:PPropList;
  ClassTypeInfo:PTypeInfo;
  ClassTypeData:PTypeData;
  aWriter:TOmWriter;
  Sz:integer;
begin
  aWriter:=TOmWriter(Writer); //get access to protected Writer methods
  for i:=0 to fList.Count-1 do
    begin
      aObj:=fList.Items[i];
      Writer.WriteString(aObj.ClassName);
      ClassTypeInfo:=aObj.ClassInfo;
      ClassTypeData:=GetTypeData(ClassTypeInfo);
      if ClassTypeData.PropCount<>0 then
        begin
          Sz:=SizeOf(PropInfo)*ClassTypeData.PropCount; //calc the size of the property list
          GetMem(PropList,Sz);
          try
            GetPropInfos(aObj.ClassInfo,PropList);
            aWriter.WriteListBegin;
            for j:=0 to ClassTypeData.PropCount-1 do
              begin
                PropInfo:=PropList[j];
                if IsStoredProp(aObj,PropInfo) then
                  aWriter.WriteProperty(aObj,PropInfo);
              end;
            aWriter.WriteListEnd;
          finally
            FreeMem(PropList,Sz);
          end;
        end;
    end;
  aWriter.WriteListEnd;
end;

function TPolymorphicList.Edit: boolean;
var aTempDialog:TFormEditPolymorphicList;
begin
  aTempDialog:=TFormEditPolymorphicList.Create(Application);
  try
    aTempDialog.PolyList:=Self;
    aTempDialog.ShowModal;
    if (aTempDialog.ModalResult = mrOK) then
      begin
      end;
  finally
    aTempDialog.Free;
  end;
end;

//Pick all objects of type C to aList. if C=nil, pick all objects
procedure TPolymorphicList.PickObjects(C:TPersistentClass;aList: TList);
var i:integer; aObj:TPersistent;
begin
  for i:=fList.Count-1 downto 0 do
    begin
      aObj:=fList.Items[i];
      if (not Assigned(C)) or (aObj.InheritsFrom(C)) then
        begin
          aList.Add(aObj);
          fList.Delete(i);
        end;
    end;
end;

procedure TPolymorphicList.ReadFromStream(St: TStream);
begin
  St.ReadComponent(self);
end;

procedure TPolymorphicList.SaveToStream(St: TStream);
begin
  St.WriteComponent(Self);
end;

{ TPolymorphicListEditor }

procedure TPolymorphicListEditor.ExecuteVerb(nIndex: Integer);
var
  iIndex          :Integer;
  aTempPolymorphicList:TPolymorphicList;
  aTempDialog     :TFormEditPolymorphicList;
  aPolymorphicList:TPolymorphicList;
begin
    case nIndex of
        0 :
        begin
            aTempDialog:=TFormEditPolymorphicList.Create(Application);
            aPolymorphicList:=Component as TPolymorphicList;
            aTempDialog.PolyList:=aPolymorphicList;
            try
                {Seta o TempDialog}
                {aTempDialog.Software     := aPolymorphicList.fSoftwareName;
                aTempDialog.Version      := aPolymorphicList.fSoftwareVersion;
                ...}
                aTempDialog.ShowModal;
                if ( aTempDialog.ModalResult = mrOK ) then
                begin
                  { Copia pro componente}
                  {aPolymorphicList.fMRUMaxItemWidth := aTempDialog.MRUMenuWidth;
                  aPolymorphicList.fMRULength       := aTempDialog.MRULength;
                  ...}
                  Designer.Modified;
                end;
            finally
              aTempDialog.Free;
            end;
        end;
    end;
end;

function TPolymorphicListEditor.GetVerb(nIndex: Integer): String;
begin
  case nIndex of
    0: Result:='Show Editor';
  end;
end;

function TPolymorphicListEditor.GetVerbCount: Integer;
begin
  Result:=1;
end;

{-----------------------}
procedure Register;
begin
  RegisterComponents('Omar', [TPolymorphicList]);
  RegisterComponentEditor(TPolymorphicList,TPolymorphicListEditor);
end;

end.




//---------------------------------------------------------------------

