{
  TObjectDataList

  Copyright  Paul Johnson 2000
  All rights reserved.


  Internal container for objects for the TObjectDataSet component.
}
unit ObjectDataList;

interface

uses
  Classes;

type

  PObjectRecord = ^TObjectRecord;
  TObjectRecord = record
    ID: Integer;
    AnObject: TObject;
  end;

  TObjectDataList = class
  private
    FObjects: TList;
    FLastID: Integer;

    function GetCount: Integer;
    function GetID(Index: Integer): Integer;
    function GetObject(Index: Integer): TObject;
    function NewRecord( AnObject: TObject ): PObjectRecord;
  public
    constructor Create;
    destructor Destroy; override;

    function Add( AnObject: TObject ): Integer;
    procedure Clear( Release: Boolean );
    procedure Delete( Index: Integer; Release: Boolean );
    function IndexOf( AnObject: TObject ): Integer;
    function IndexOfID( ItemID: Integer; OriginalIndex: Integer ): Integer; virtual;
    procedure Insert(Index: Integer; AnObject: TObject);
    function Remove( AnObject: TObject; Release: Boolean ): Integer;

    property Count: Integer read GetCount;
    property Objects[ Index: Integer ]: TObject read GetObject; default;
    property IDs[ Index: Integer ]: Integer read GetID;
  end;

implementation

{ TObjectDataList }

function TObjectDataList.Add(AnObject: TObject): Integer;
begin
  Result := FObjects.Add( NewRecord( AnObject ) );
end;

procedure TObjectDataList.Clear( Release: Boolean );
{ empty the list (freeing objects according to release) }
var
  Index: Integer;
begin
  for Index := 0 to (Count-1) do begin
    if Release then begin
      Objects[ Index ].Free();
    end;

    // free TObjectRecord
    Dispose( FObjects[ Index ] );
  end;

  FObjects.Clear()
end;

constructor TObjectDataList.Create;
begin
  inherited;

  // initialise unique object identifier
  FLastID := 0;

  FObjects := TList.Create();
end;

procedure TObjectDataList.Delete(Index: Integer; Release: Boolean);
begin
  if Release then begin
    Objects[ Index ].Free();
  end;

  // free TObjectRecord
  Dispose( FObjects.Items[ Index ] );

  FObjects.Delete( Index );
end;

destructor TObjectDataList.Destroy;
begin
  FObjects.Free();

  inherited;
end;

function TObjectDataList.GetCount: Integer;
begin
  Result := FObjects.Count;
end;

function TObjectDataList.GetID(Index: Integer): Integer;
begin
  Result := PObjectRecord( FObjects.Items[ Index ] ).ID;
end;

function TObjectDataList.GetObject(Index: Integer): TObject;
begin
  Result := PObjectRecord( FObjects.Items[ Index ] ).AnObject;
end;

function TObjectDataList.IndexOf(AnObject: TObject): Integer;
var
  Index: Integer;
begin
  Result := -1;

  for Index := 0 to (Count-1) do begin
    if Objects[ Index ] = AnObject then begin
      Result := Index;
      Break;
    end;
  end;

end;

function TObjectDataList.IndexOfID(ItemID,
  OriginalIndex: Integer): Integer;
{ return index of object with given ID, 'OriginalIndex' is the previously known
  index of the object. return -1 if the object is not found }
var
  ObjectIndex: Integer;
begin
  // check original index for ID
  if (OriginalIndex < Count) and
     (IDs[ OriginalIndex ] = ItemID)
  then begin
    Result := OriginalIndex;
  end
  else begin
    Result := -1;

    // search entire list for ID
    for ObjectIndex := 0 to (Count-1) do begin
      if IDs[ ObjectIndex ] = ItemID then begin
        Result := ObjectIndex;
        Break;
      end;
    end;
  end;
end;

procedure TObjectDataList.Insert(Index: Integer; AnObject: TObject);
begin
  FObjects.Insert( Index, NewRecord( AnObject ) );
end;

function TObjectDataList.NewRecord(AnObject: TObject): PObjectRecord;
{ Create and initialise a new object record }
begin

  New(Result);
  Result^.AnObject := AnObject;
  Result^.ID := FLastID;

  // maintain unique object identifier
  Inc(FLastID);
end;

function TObjectDataList.Remove(AnObject: TObject;
  Release: Boolean): Integer;
begin
  Result := IndexOf( AnObject );

  if Result <> -1 then begin
    Delete( Result, Release );
  end;
end;

end.
