//****************************************************************************/
//* MIDI device classes by Adrian Meyer
//****************************************************************************/
//* V1.0 Delphi 3 Windows 2000
//****************************************************************************/
//* If you get a hold of this source you may use it upon your own risk. Please
//* let me know if you have any questions: adrian.meyer@rocketmail.com.
//****************************************************************************/
unit Midi;

interface

uses
  classes, SysUtils, mmsystem, Math, Windows;

type
  // event if data is received
  TOnMidiInData = procedure ( const aDeviceIndex: integer; const aStatus, aData1, aData2: byte ) of object;

  // base class for MIDI devices
  TMidiDevices = class
  private
    fDevices: TStringList;
    procedure CheckRange( const aDeviceIndex: integer );
  public
    constructor Create; virtual;
    destructor Destroy; override;

    // open a specific device
    procedure Open( const aDeviceIndex: integer ); virtual;
    // close a specific device
    procedure Close( const aDeviceIndex: integer ); virtual;
    // close all devices
    procedure CloseAll;
    // THE devices
    property Devices: TStringList read fDevices;
  end;

  // MIDI input devices
  TMidiInput = class( TMidiDevices )
  private
    fOnMidiData: TOnMidiInData;
  public
    constructor Create; override;
    // open a specific input device
    procedure Open( const aDeviceIndex: integer ); override;
    // close a specific device
    procedure Close( const aDeviceIndex: integer ); override;

    // midi data event
    property OnMidiData: TOnMidiInData read fOnMidiData write fOnMidiData;
  end;

  // MIDI output devices
  TMidiOutput = class( TMidiDevices )
    constructor Create; override;
    // open a specific input device
    procedure Open( const aDeviceIndex: integer ); override;
    // close a specific device
    procedure Close( const aDeviceIndex: integer ); override;
    // send some midi data to the indexed device
    procedure Send( const aDeviceINdex: integer; const aStatus, aData1, aData2: byte ); 
  end;

  // MIDI input devices
  function MidiInput: TMidiInput;
  // MIDI output Devices
  function MidiOutput: TMidiOutput;

implementation

{ TMidiBase }

function GetLastErrorText: string;
begin
  SetLength( Result, 2000 );
  SetLength( Result, FormatMessage( FORMAT_MESSAGE_FROM_SYSTEM,
      nil, GetLastError(), GetUserDefaultLangID, pchar( Result ), 2000, nil ));
end;

constructor TMidiDevices.Create;
begin
  fDevices := TStringLIst.create;
end;

destructor TMidiDevices.Destroy;
begin
  fDevices.Free;
  inherited;
end;

var
  gMidiInput: TMidiInput;
  gMidiOutput: TMidiOutput;

function MidiInput: TMidiInput;
begin
  if not assigned(gMidiInput) then
    gMidiInput := TMidiInput.Create;
  Result := gMidiInput;
end;

procedure midiCallback( aMidiInHandle: PHMIDIIN; aMsg: UInt; aData, aMidiData, aTimeStamp: integer ); stdcall;
begin
  if ( aMsg = MIM_DATA ) then
    if assigned( MidiInput.OnMidiData ) then
       MidiInput.OnMidiData( aData, aMidiData and $000000FF,
       ( aMidiData and $0000FF00 ) shr 8, ( aMidiData and $00FF0000 ) shr 16 );
end;

function MidiOutput: TMidiOutput;
begin
  if not assigned(gMidiOutput) then
    gMidiOutput := TMidiOutput.Create;
  Result := gMidiOutput;
end;

{ TMidiInput }

procedure TMidiInput.Close(const aDeviceIndex: integer);
var
  lHandle: THandle;
begin
  inherited;
  if assigned( fDevices.Objects[aDeviceIndex] ) then
  begin
    lHandle := THandle( fDevices.Objects[aDeviceIndex] );
  	midiInStop(lHandle);
    midiInClose(lHandle);
    fDevices.Objects[aDeviceIndex] := nil;
  end;
end;

procedure TMidiDevices.CloseAll;
var
  i: integer;
begin
  for i:=0 to fDevices.Count - 1 do
    Close( i );
end;

constructor TMidiInput.Create;
var
  i: integer;
  lInCaps: TMidiInCaps;
  lResult: MMRESULT;
begin
  inherited;
  for i:=0 to midiInGetNumDevs - 1 do
  begin
    lResult := midiInGetDevCaps( i, @lInCaps, SizeOf( TMidiInCaps ));

    if lResult = MMSYSERR_NOERROR then
      fDevices.Add( StrPas( lInCaps.szPname ));
  end;
end;

procedure TMidiInput.Open(const aDeviceIndex: integer);
var
  lHandle: THandle;
  lResult: MMResult;
begin
  inherited;
  if assigned( fDevices.Objects[ aDeviceIndex ] ) then Exit;

  lResult := midiInOpen( @lHandle, aDeviceIndex, cardinal( @midiCallback ), aDeviceIndex, CALLBACK_FUNCTION );
  if lResult <> MMSYSERR_NOERROR then
    raise exception.Create( GetLastErrorText );
	midiInStart(lHandle);

  fDevices.Objects[ aDeviceIndex ] := TObject( lHandle );
end;

{ TMidiOutput }

procedure TMidiOutput.Close(const aDeviceIndex: integer);
var
  lHandle: THandle;
begin
  inherited;
  lHandle := THandle( fDevices.Objects[ aDeviceIndex ] );
  midiOutClose( lHandle );
  fDevices.Objects[ aDeviceIndex ] := nil;
end;

constructor TMidiOutput.Create;
var
  i: integer;
  lOutCaps: TMidiOutCaps;
  lResult: MMRESULT;
begin
  inherited;
  for i:=0 to midiOutGetNumDevs - 1 do
  begin
    lResult := midiOutGetDevCaps( i, @lOutCaps, SizeOf( TMidiOutCaps ));

    if lResult = MMSYSERR_NOERROR then
      fDevices.Add( lOutCaps.szPname );
  end;
end;

procedure TMidiDevices.Open(const aDeviceIndex: integer);
begin
  CheckRange( aDeviceIndex );
end;

procedure TMidiDevices.Close(const aDeviceIndex: integer);
begin
  CheckRange( aDeviceIndex );
end;

procedure TMidiOutput.Open(const aDeviceIndex: integer);
var
  lResult: MMRESULT;
  lHandle: THandle;
begin
  inherited;
  // device already open;
  if assigned( fDevices.Objects[ aDeviceIndex ] ) then Exit;

  lResult := midiOutOpen( @lHandle, aDeviceIndex, 0, 0, CALLBACK_NULL );
  fDevices.Objects[ aDeviceIndex ] := TObject( lHandle );

  if lResult <> MMSYSERR_NOERROR then
    raise exception.Create( GetLastErrorText );
end;

procedure TMidiOutput.Send(const aDeviceINdex: integer; const aStatus,
  aData1, aData2: byte);
var
  lMsg: cardinal;
  lResult: MMResult;
  lHandle: THandle;
begin
  CheckRange( aDeviceIndex );

  // open the device is not open
  if not assigned( fDevices.Objects[ aDeviceIndex ] ) then
    Open( aDeviceIndex );

  lHandle := THandle( fDevices.Objects[ aDeviceIndex ] );
  lMsg := aStatus + ( aData1 * $100 ) + ( aData2 * $10000 );
  lResult := midiOutShortMsg( lHandle, lMSG );

  if lResult <> MMSYSERR_NOERROR then
    raise Exception.Create( GetLastErrorText );
end;

procedure TMidiDevices.CheckRange( const aDeviceIndex: integer );
begin
  if ( aDeviceIndex < 0 ) or ( aDeviceIndex > fDevices.Count - 1 ) then
    raise Exception.CreateFmt( '%s: Device index out of bounds! (%d)', [ClassName,aDeviceIndex] );
end;

initialization
  gMidiInput := nil;
  gMidiOutput := nil;

finalization
  gMidiInput.Free;
  gMidiOutput.Free;

end.
