{+--------------------------------------------------------------------------+
 | Class:       TmTypeList
 | Created:     8.98
 | Author:      Martin Waldenburg
 | Copyright    1998 all rights reserved.
 | Description: Fills a stringlist with descriptions of published properties
 |              of a given class.
 | Version:     1.0
 | Status:      FreeWare
 | Disclaimer:
 | This is provided as is, expressly without a warranty of any kind.
 | You use it at your own risc.
 +--------------------------------------------------------------------------+}
unit mwTypeList;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, TypInfo;

type
  TmTypeList = class(Tobject)
  private
    Class procedure procTkArray(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkChar(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkClass(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkDynArray(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkEnumeration(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkFloat(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkInt64(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkInteger(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkInterface(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkMethod(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkLString(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkRecord(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkSet(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkString(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkUnknown(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkVariant(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkWChar(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
    Class procedure procTkWString(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
  protected
    { Protected-Deklarationen}
  public
    Class procedure GetInfo(anObject: TObject; List: TStringList);
  published
    { Published-Deklarationen }
  end;

implementation

{ TmTypeList }

{ TmTypeList }

Class procedure TmTypeList.GetInfo(anObject: TObject; List: TStringList);
var
  I, Count: Integer;
  PropList:PPropList;
  PKinds   : TTypeKinds;
begin
  List.Add('Class: ' + anObject.ClassName);
  List.Add('  ClassParent: ' + anObject.ClassParent.ClassName);
  List.Add('');
  PKinds:= [tkUnknown, tkInteger, tkChar, tkEnumeration, tkFloat,
    tkString, tkSet, tkClass, tkMethod, tkWChar, tkLString, tkWString,
    tkVariant, tkArray, tkRecord, tkInterface, tkInt64, tkDynArray];

  Count:= GetPropList(anObject.ClassInfo, PKinds, Nil);
  GetMem(PropList, Count *SizeOf(Pointer));
  GetPropList(anObject.ClassInfo, PKinds, PropList);
  for I:=0 to Count-1 do
  begin
    case PropList^[i].PropType^.Kind of
      tkArray: procTkArray(anObject, PropList^[i], List);
      tkChar: procTkChar(anObject, PropList^[i], List);
      tkClass: procTkClass(anObject, PropList^[i], List);
      tkDynArray: procTkDynArray(anObject, PropList^[i], List);
      tkEnumeration: procTkEnumeration(anObject, PropList^[i], List);
      tkFloat: procTkFloat(anObject, PropList^[i], List);
      tkInt64: procTkInt64(anObject, PropList^[i], List);
      tkInteger: procTkInteger(anObject, PropList^[i], List);
      tkInterface: procTkInterface(anObject, PropList^[i], List);
      tkMethod: procTkMethod(anObject, PropList^[i], List);
      tkLString: procTkLString(anObject, PropList^[i], List);
      tkRecord: procTkRecord(anObject, PropList^[i], List);
      tkSet: procTkSet(anObject, PropList^[i], List);
      tkString: procTkString(anObject, PropList^[i], List);
      tkUnknown: procTkUnknown(anObject, PropList^[i], List);
      tkVariant: procTkVariant(anObject, PropList^[i], List);
      tkWChar: procTkWChar(anObject, PropList^[i], List);
      tkWString: procTkWString(anObject, PropList^[i], List);
    end;
  List.Add('');
  end;
    FreeMem(PropList, Count *SizeOf(Pointer));
end;

Class procedure TmTypeList.procTkArray(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
  end else
  begin
    List.Add('  property: ');
  end;
  List.Add('    TypeKind: Array');
end;

Class procedure TmTypeList.procTkChar(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
var
  aLongInt: LongInt;
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
    List.Add('    TypeKind: Char');
    aLongInt:= GetOrdProp(anObject, PropInfo);
    List.Add('    Value: '+IntToStr(aLongInt));
  end else
  begin
    List.Add('  property: ');
    List.Add('    TypeKind: Char');
  end;
end;

Class procedure TmTypeList.procTkClass(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
var
  TypeData: PTypeData;
begin
  if PropInfo <> nil then
  begin
    TypeData:= GetTypeData(PropInfo.PropType^);
    List.Add('  property: ' + PropInfo.Name);
    List.Add('    TypeKind: Class');
    List.Add('    ClassName: '+TypeData^.ClassType.ClassName + ' UnitName: '+TypeData^.UnitName);
    List.Add('    ClassParent: ' + TypeData^.ClassType.ClassParent.ClassName);
  end else
  begin
    List.Add('  property: ');
    List.Add('    TypeKind: Class');
  end;
  TypeData:= nil;
end;

Class procedure TmTypeList.procTkDynArray(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
  end else
  begin
    List.Add('  property: ');
  end;
  List.Add('    TypeKind: DynArray');
end;

Class procedure TmTypeList.procTkEnumeration(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
var
  I, aLongInt: LongInt;
  TypeData: PTypeData;
  aString: String;
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
    aLongInt:= GetOrdProp(anObject, PropInfo);
    List.Add('    TypeKind: Enumeration');
    List.Add('    Value: '+ GetEnumName(PropInfo.PropType^, aLongInt));
    TypeData:= GetTypeData(PropInfo.PropType^);
    if (TypeData.BaseType<>nil) and (TypeData.BaseType^<>nil) and (TypeData.BaseType^^.Kind=tkEnumeration) then
    begin
      for i:=0 to GetTypeData(TypeData.BaseType^)^.MaxValue-GetTypeData(TypeData.BaseType^)^.MinValue do
      begin
        aString:= GetEnumName(TypeData.BaseType^,GetTypeData(TypeData.BaseType^)^.MinValue+i);
        List.Add('    Enums: '+IntToStr(i)+' '+ aString);
      end;
    end;
  end else
  begin
    List.Add('  property: ');
    List.Add('    TypeKind: Class');
  end;
  TypeData:= nil;
end;

Class procedure TmTypeList.procTkFloat(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
var
  aExtended: Extended;
  TypeData: PTypeData;
begin
  aExtended:= 0;
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
    aExtended:= GetFloatProp(anObject, PropInfo);
    TypeData:= GetTypeData(PropInfo.PropType^);
  end else
  begin
    List.Add('  property: ');
  end;
  List.Add('    TypeKind: Float');
  List.Add('    Value: '+ FloatToStr(aExtended));
  TypeData:= nil;
end;

Class procedure TmTypeList.procTkInt64(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
var
  i64:Int64;
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
    List.Add('    TypeKind: Int64');
    i64:= GetInt64Prop(anObject, PropInfo);
    List.Add('    MinValue: '+IntToStr(i64));
  end else
  begin
    List.Add('  property: ');
    List.Add('    TypeKind: Int64');
  end;
end;

Class procedure TmTypeList.procTkInteger(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
var
  aLongInt: LongInt;
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
    aLongInt:= GetOrdProp(anObject, PropInfo);
    List.Add('    TypeKind: Integer');
    List.Add('    Value: '+IntToStr(aLongInt));
  end else
  begin
    List.Add('  property: ');
    List.Add('    TypeKind: Integer');
  end;
end;

Class procedure TmTypeList.procTkInterface(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
  end else
  begin
    List.Add('  property: ');
  end;
  List.Add('    TypeKind: Interface');
end;

Class procedure TmTypeList.procTkLString(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
var
  aString: String;
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
    aString:=GetStrProp(anObject, PropInfo);
  end else
  begin
    List.Add('  property: ');
    aString:=PString(anObject)^;
  end;
  List.Add('    TypeKind: String');
  List.Add('    Value: '+ aString);
end;

Class procedure TmTypeList.procTkMethod(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
var
  aMethod: TMethod;
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
    aMethod:=GetMethodProp(anObject, PropInfo);
  end else
  begin
    List.Add('  property: ');
    List.Add('    TypeKind: Method');
  end;
  List.Add('    TypeKind: Method');
  if Assigned(aMethod.Code)then
  begin
    if Assigned(aMethod.Data) then
    List.Add('    CodeValue: '+ TObject(aMethod.Data).MethodName(aMethod.Code));
  end;
  if Assigned(aMethod.Data) then
  begin
    List.Add('    DataValue: ' + TObject(aMethod.Data).ClassName);
  end;
  if not(Assigned(aMethod.Code)) and not(Assigned(aMethod.Data)) then List.Add('    Value: nil');
end;

Class procedure TmTypeList.procTkRecord(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
  end else
  begin
    List.Add('  property: ');
  end;
  List.Add('  TypeKind: Record');
end;

Class procedure TmTypeList.procTkSet(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
var
  I, aLongInt: LongInt;
  TypeData: PTypeData;
  aString: String;
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
    aLongInt:= GetOrdProp(anObject, PropInfo);
    TypeData:= GetTypeData(PropInfo.PropType^);
    List.Add('    TypeKind: Set');
    List.Add('    Value: '+IntToStr(aLongInt));
    if (TypeData.CompType<>nil) and (TypeData.CompType^<>nil) and (TypeData.CompType^^.Kind = tkEnumeration) then
    begin
      for I:= 0 to GetTypeData(TypeData.CompType^)^.MaxValue-GetTypeData(TypeData.CompType^)^.MinValue do
      begin
        aString:= GetEnumName(TypeData.CompType^, GetTypeData(TypeData.CompType^)^.MinValue+I);
        if ((1 shl i) and aLongInt)> 0 then
        begin
          List.Add('    In Set: ' + IntToStr(I)+' ' + aString);
        end else
        begin
        List.Add('    Not In Set: ' + IntToStr(I)+' ' + aString);
        end;
      end;
    end else
    begin
      for i:=0 to 31 do
      if ((1 shl i) and aLongInt)>0 then List.Add('    InSet: '+IntToStr(I));
    end;
  end else
  begin
    List.Add('  property: ');
    List.Add('    TypeKind: Set');
  end;
end;

Class procedure TmTypeList.procTkString(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
var
  TypeData: PTypeData;
  aString: String;
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
    aString:= GetStrProp(anObject, PropInfo);
  end else
  begin
    List.Add('  property: ');
    aString:= PShortString(anObject)^;
  end;
  TypeData:= GetTypeData(PropInfo.PropType^);
  List.Add('    TypeKind: String');
  List.Add('    Value: '+aString);
  List.Add('    MaxSize: '+ IntToStr(TypeData.MaxLength));
end;

Class procedure TmTypeList.procTkUnknown(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
  end else
  begin
    List.Add('  property: ');
  end;
  List.Add('  TypeKind: Unknown');
end;

Class procedure TmTypeList.procTkVariant(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
var
  aString: String;
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
    aString:= GetVariantProp(anObject, PropInfo);
  end else
  begin
    List.Add('  property: ');
    aString:= PString(anObject)^;
  end;
  List.Add('    TypeKind: Variant');
  List.Add('  Value: '+aString);
end;

Class procedure TmTypeList.procTkWChar(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
var
  aLongInt: LongInt;
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
    aLongInt:= GetOrdProp(anObject, PropInfo);
    List.Add('    TypeKind: WChar');
    List.Add('    Value: '+IntToStr(aLongInt));
  end else
  begin
    List.Add('  property: ');
    List.Add('    TypeKind: WChar');
  end;
end;

Class procedure TmTypeList.procTkWString(anObject: TObject; PropInfo: PPropInfo; List: TStringList);
var
  aString: String;
begin
  if PropInfo <> nil then
  begin
    List.Add('  property: ' + PropInfo.Name);
    aString:=GetStrProp(anObject, PropInfo);
    List.Add('    TypeKind: WString');
    List.Add('    Value: '+ aString);
  end else
  begin
    List.Add('  property: ');
    List.Add('    TypeKind: WString');
  end;
end;

end.

