unit LiveGUID;

{
LiveGUID 1.1

LiveGUID is a class that implements decomposing/assembling
of TGUID's 16-byte structure to/from four 4-byte tokens,
which can then be manipulated as cardinal integers or strings.

By taking advantage of a GUID uniqueness, LiveGUID's tokens
can be useful to create unique integer or string id's such
as indexes for database tables.

Comments: Tested only in Delphi 4 under Windows 98

Geraldo Nascimento (geraldo@centroin.com.br)
October 1999
}

interface

uses
   Windows, SysUtils, ActiveX, Classes, ComObj, Dialogs;

type
   TLGIntTokens = array[1..4] of cardinal;

   TLGStrTokens = array[1..4] of string;

   TLGHexTokens = array[1..4] of string;

   TLiveGUID = class(TPersistent)
      private
         FGUID : TGUID;

         procedure SetFGUID(const g : TGUID);

         procedure SetLiveGUID(const g : TLiveGUID);

         procedure SetGUIDstring(const g : string);
         function  GetGUIDstring : string;

         function  GetGUIDstringClean : string;

         procedure SetGUIDintegerTokens(const tokens : TLGIntTokens);
         function  GetGUIDintegerTokens : TLGIntTokens;

         procedure SetGUIDstringTokens(const tokens : TLGStrTokens);
         function  GetGUIDstringTokens : TLGStrTokens;

         procedure SetGUIDhexTokens(const tokens : TLGHexTokens);
         function  GetGUIDhexTokens : TLGHexTokens;

         function  GetNICMACaddress : string;

         function  BuildMessage(msg : string) : string;
         procedure ShowWarning(msg : string);

      public
         property GUID : TGUID read FGUID write SetFGUID;

         property GUIDstring : string read GetGUIDstring write SetGUIDstring;

         property GUIDstringClean : string read GetGUIDstringClean;

         property IntTokens : TLGIntTokens read  GetGUIDintegerTokens
                                           write SetGUIDintegerTokens;

         property StrTokens : TLGStrTokens read  GetGUIDstringTokens
                                           write SetGUIDstringTokens;

         property HexTokens : TLGHexTokens read  GetGUIDhexTokens
                                           write SetGUIDhexTokens;

         property IntToken : TLGIntTokens read GetGUIDintegerTokens;
         property StrToken : TLGStrTokens read GetGUIDstringTokens;
         property HexToken : TLGHexTokens read GetGUIDhexTokens;

         property NICMACaddress : string read GetNICMACaddress;

         procedure SetIntTokens(const token1, token2, token3, token4 : cardinal);
         procedure SetStrTokens(const token1, token2, token3, token4 : string);
         procedure SetHexTokens(const token1, token2, token3, token4 : string);

         procedure Assign(Source : TPersistent); override;
         procedure AssignTo(Dest : TPersistent); override;

         constructor Create; overload; virtual;
         constructor Create(const g : TLiveGUID); overload; virtual;
         constructor Create(const g : TGUID); overload; virtual;
         constructor Create(const g : string); overload; virtual;
         constructor Create(const tokens : TLGIntTokens); overload; virtual;
         constructor Create(const tokens : TLGStrTokens); overload; virtual;
         constructor Create(const tokens : TLGHexTokens); overload; virtual;
         constructor Create(const token1, token2, token3, token4 : cardinal); overload; virtual;
         constructor Create(const token1, token2, token3, token4 : string); overload; virtual;
         constructor CreateHex; overload; virtual;
         constructor CreateHex(const tokens : TLGHexTokens); overload; virtual;
         constructor CreateHex(const token1, token2, token3, token4 : string); overload; virtual;

         destructor Destroy; override;
      end;

const

   lgNull      = '';
   lgBlank     = ' ';
   lgLBrace    = '{';
   lgRBrace    = '}';
   lgDash      = '-';
   lgHexPrefix = '$';
   lgCollon    = ':';
   lgZero      = '0';
   lgZeroGUID  = '{00000000-0000-0000-0000-000000000000}';

implementation

const

   lgCreateError = 'Error creating object';

// Constructors

constructor TLiveGUID.Create;
begin
   inherited Create;
   if CoCreateGUID(FGUID) <> S_OK then
      raise Exception.Create(BuildMessage(lgCreateError));
end;

constructor TLiveGUID.Create(const g : TLiveGUID);
begin
   inherited Create;
   SetLiveGUID(g);
end;

constructor TLiveGUID.Create(const g : TGUID);
begin
   inherited Create;
   SetFGUID(g);
end;

constructor TLiveGUID.Create(const g : string);
begin
   inherited Create;
   SetGUIDstring(g);
end;

constructor TLiveGUID.Create(const tokens : TLGIntTokens);
begin
   inherited Create;
   SetGUIDintegerTokens(tokens);
end;

constructor TLiveGUID.Create(const tokens : TLGStrTokens);
begin
   inherited Create;
   SetGUIDstringTokens(tokens);
end;

constructor TLiveGUID.Create(const tokens : TLGHexTokens);
begin
   inherited Create;
   SetGUIDhexTokens(tokens);
end;

constructor TLiveGUID.Create(const token1, token2, token3, token4 : cardinal);
begin
   inherited Create;
   SetIntTokens(token1, token2, token3, token4);
end;

constructor TLiveGUID.Create(const token1, token2, token3, token4 : string);
begin
   inherited Create;
   SetStrTokens(token1, token2, token3, token4);
end;

constructor TLiveGUID.CreateHex;
begin
   Create;
end;

constructor TLiveGUID.CreateHex(const tokens : TLGHexTokens);
begin
   inherited Create;
   SetGUIDhexTokens(tokens);
end;

constructor TLiveGUID.CreateHex(const token1, token2, token3, token4 : string);
begin
   inherited Create;
   SetHexTokens(token1, token2, token3, token4);
end;

// Destructor

destructor TLiveGUID.Destroy;
begin
   inherited Destroy;
end;

// LiveGUID object assignment

procedure TLiveGUID.Assign(Source : TPersistent);
begin
   try
      if Source is TLiveGUID then
         SetLiveGUID(TLiveGUID(Source))
      else
         inherited Assign(Source);
   except
      on E : Exception do
         raise Exception.Create(BuildMessage(E.Message));
   end;
end;

procedure TLiveGUID.AssignTo(Dest : TPersistent);
begin
   try
      if Dest is TLiveGUID then
         TLiveGUID(Dest).Assign(Self)
      else
         inherited AssignTo(Dest);
   except
      on E : Exception do
         raise Exception.Create(BuildMessage(E.Message));
   end;
end;

// Set FGUID from a GUID

procedure TLiveGUID.SetFGUID(const g : TGUID);
begin
   try
      FGUID := g;
   except
      on E : Exception do
         raise Exception.Create(BuildMessage(E.Message));
   end;
end;

// Set FGUID from another LiveGUID object

procedure TLiveGUID.SetLiveGUID(const g : TLiveGUID);
begin
   try
      FGUID := g.GUID;
   except
      on E : Exception do
         raise Exception.Create(BuildMessage(E.Message));
   end;
end;

// Set FGUID from a GUID string

procedure TLiveGUID.SetGUIDstring(const g : string);
begin
   try
      FGUID := StringToGUID(Trim(g));
   except
      on E : Exception do
         raise Exception.Create(BuildMessage(E.Message));
   end;
end;

// Set FGUID from integer token array

procedure TLiveGUID.SetGUIDintegerTokens(const tokens : TLGIntTokens);
var
   g : string;
   i : integer;

begin
   try
      g := lgNull;

      for i := 1 to 4 do g := g + IntToHex(tokens[i], 8);

      Insert(lgDash, g,  9);
      Insert(lgDash, g, 14);
      Insert(lgDash, g, 19);
      Insert(lgDash, g, 24);

      FGUID := StringToGUID(lgLBrace + g + lgRBrace);
   except
      on E : Exception do
         raise Exception.Create(BuildMessage(E.Message));
   end;
end;

// Set FGUID from integer tokens

procedure TLiveGUID.SetIntTokens(const token1, token2, token3, token4 : cardinal);
var
   t : TLGIntTokens;

begin
   t[1] := token1;
   t[2] := token2;
   t[3] := token3;
   t[4] := token4;

   SetGUIDintegerTokens(t);
end;

// Set FGUID from string token array

procedure TLiveGUID.SetGUIDstringTokens(const tokens : TLGStrTokens);
var
   t : TLGIntTokens;
   i : integer;

begin
   try
      for i := 1 to 4 do t[i] := StrToInt64(Trim(tokens[i]));
   except
      on E : Exception do
         raise Exception.Create(BuildMessage(E.Message));
   end;

   SetGUIDintegerTokens(t);
end;

// Set FGUID from string tokens

procedure TLiveGUID.SetStrTokens(const token1, token2, token3, token4 : string);
var
   t : TLGStrTokens;

begin
   t[1] := token1;
   t[2] := token2;
   t[3] := token3;
   t[4] := token4;

   SetGUIDstringTokens(t);
end;

// Set FGUID from hex token array

procedure TLiveGUID.SetGUIDhexTokens(const tokens : TLGHexTokens);
var
   t : TLGIntTokens;
   i : integer;
   s : string;

begin
   try
      for i := 1 to 4 do
         begin
            s := Trim(tokens[i]);
            if Copy(s, 1, 1) <> lgHexPrefix then s := lgHexPrefix + s;
            t[i] := StrToInt64(s);
         end;
   except
      on E : Exception do
         raise Exception.Create(BuildMessage(E.Message));
   end;

   SetGUIDintegerTokens(t);
end;

// Set FGUID from hex tokens

procedure TLiveGUID.SetHexTokens(const token1, token2, token3, token4 : string);
var
   t : TLGHexTokens;

begin
   t[1] := token1;
   t[2] := token2;
   t[3] := token3;
   t[4] := token4;

   SetGUIDhexTokens(t);
end;

// Get GUID string

function TLiveGUID.GetGUIDstring : string;
begin
   try
      Result := GUIDtoString(FGUID);
   except
      on E : Exception do
         begin
            Result := lgZeroGUID;
            ShowWarning(E.Message);
         end;
   end;
end;

// Get GUID string without the delimiters

function TLiveGUID.GetGUIDstringClean : string;
begin
   try
      Result := GUIDtoString(FGUID);
      Result := StringReplace(Result, lgLBrace, lgNull, [rfReplaceAll]);
      Result := StringReplace(Result, lgDash,   lgNull, [rfReplaceAll]);
      Result := StringReplace(Result, lgRBrace, lgNull, [rfReplaceAll]);
   except
      on E : Exception do
         begin
            Result := StringOfChar(lgZero, 32);
            ShowWarning(E.Message);
         end;
   end;
end;

// Get all tokens as integers

function TLiveGUID.GetGUIDintegerTokens : TLGIntTokens;
var
   t : TLGHexTokens;
   i : integer;

begin
   t := GetGUIDhexTokens;

   try
      for i := 1 to 4 do Result[i] := StrToInt64(lgHexPrefix + t[i]);
   except
      on E : Exception do
         ShowWarning(E.Message);
   end;
end;

// Get all tokens as integer strings

function TLiveGUID.GetGUIDstringTokens : TLGStrTokens;
var
   t : TLGIntTokens;
   i : integer;

begin
   t := GetGUIDintegerTokens;

   try
      for i := 1 to 4 do Result[i] := IntToStr(t[i]);
   except
      on E : Exception do
         begin
            for i := 1 to 4 do Result[i] := lgZero;
            ShowWarning(E.Message);
         end;
   end;
end;

// Get all tokens as hexadecimal strings

function TLiveGUID.GetGUIDhexTokens : TLGHexTokens;
var
   s : string;
   i : integer;

begin
   s := GetGUIDstringClean;

   try
      for i := 1 to 4 do Result[i] := Copy(s, ((i - 1) * 8) + 1, 8);
   except
      on E : Exception do
         begin
            for i := 1 to 4 do Result[i] := StringOfChar(lgZero, 8);
            ShowWarning(E.Message);
         end;
   end;
end;

// Get the MAC address of the first NIC (Network Interface Card) installed
// Note: If the machine has no NIC installed, this value is meaningless

function TLiveGUID.GetNICMACaddress : string;
begin
   try
      Result := GetGUIDstringClean;
      Result := Copy(Result, Length(Result) - 11, 12);
   except
      on E : Exception do
         begin
            Result := StringOfChar(lgZero, 12);
            ShowWarning(E.Message);
         end;
   end;
   Result := Result + lgCollon;
end;

// Prepends the class name to a message string

function TLiveGUID.BuildMessage(msg : string) : string;
begin
   Result := Trim(msg);

   if Copy(msg, 1, Length(Self.ClassName)) <> Self.ClassName then
      Result := Self.ClassName + lgCollon + lgBlank + Result;
end;

// Show a warinig message

procedure TLiveGUID.ShowWarning(msg : string);
begin
   MessageDlg(BuildMessage(msg), mtWarning, [mbOK], 0);
end;

end.
